/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { BehaviorSubject } from 'rxjs';
/** Subject used to dispatch and listen for changes to the auto change detection status . */
const autoChangeDetectionSubject = new BehaviorSubject({
    isDisabled: false,
});
/** The current subscription to `autoChangeDetectionSubject`. */
let autoChangeDetectionSubscription;
/**
 * The default handler for auto change detection status changes. This handler will be used if the
 * specific environment does not install its own.
 * @param status The new auto change detection status.
 */
function defaultAutoChangeDetectionHandler(status) {
    status.onDetectChangesNow?.();
}
/**
 * Allows a test `HarnessEnvironment` to install its own handler for auto change detection status
 * changes.
 * @param handler The handler for the auto change detection status.
 */
export function handleAutoChangeDetectionStatus(handler) {
    stopHandlingAutoChangeDetectionStatus();
    autoChangeDetectionSubscription = autoChangeDetectionSubject.subscribe(handler);
}
/** Allows a `HarnessEnvironment` to stop handling auto change detection status changes. */
export function stopHandlingAutoChangeDetectionStatus() {
    autoChangeDetectionSubscription?.unsubscribe();
    autoChangeDetectionSubscription = null;
}
/**
 * Batches together triggering of change detection over the duration of the given function.
 * @param fn The function to call with batched change detection.
 * @param triggerBeforeAndAfter Optionally trigger change detection once before and after the batch
 *   operation. If false, change detection will not be triggered.
 * @return The result of the given function.
 */
async function batchChangeDetection(fn, triggerBeforeAndAfter) {
    // If change detection batching is already in progress, just run the function.
    if (autoChangeDetectionSubject.getValue().isDisabled) {
        return await fn();
    }
    // If nothing is handling change detection batching, install the default handler.
    if (!autoChangeDetectionSubscription) {
        handleAutoChangeDetectionStatus(defaultAutoChangeDetectionHandler);
    }
    if (triggerBeforeAndAfter) {
        await new Promise(resolve => autoChangeDetectionSubject.next({
            isDisabled: true,
            onDetectChangesNow: resolve,
        }));
        // The function passed in may throw (e.g. if the user wants to make an expectation of an error
        // being thrown. If this happens, we need to make sure we still re-enable change detection, so
        // we wrap it in a `finally` block.
        try {
            return await fn();
        }
        finally {
            await new Promise(resolve => autoChangeDetectionSubject.next({
                isDisabled: false,
                onDetectChangesNow: resolve,
            }));
        }
    }
    else {
        autoChangeDetectionSubject.next({ isDisabled: true });
        // The function passed in may throw (e.g. if the user wants to make an expectation of an error
        // being thrown. If this happens, we need to make sure we still re-enable change detection, so
        // we wrap it in a `finally` block.
        try {
            return await fn();
        }
        finally {
            autoChangeDetectionSubject.next({ isDisabled: false });
        }
    }
}
/**
 * Disables the harness system's auto change detection for the duration of the given function.
 * @param fn The function to disable auto change detection for.
 * @return The result of the given function.
 */
export async function manualChangeDetection(fn) {
    return batchChangeDetection(fn, false);
}
/**
 * Resolves the given list of async values in parallel (i.e. via Promise.all) while batching change
 * detection over the entire operation such that change detection occurs exactly once before
 * resolving the values and once after.
 * @param values A getter for the async values to resolve in parallel with batched change detection.
 * @return The resolved values.
 */
export async function parallel(values) {
    return batchChangeDetection(() => Promise.all(values()), true);
}
//# sourceMappingURL=data:application/json;base64,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