/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { assertDefined, assertEqual, assertNumber, throwError } from '../util/assert';
import { getComponentDef, getNgModuleDef } from './definition';
import { isLContainer, isLView } from './interfaces/type_checks';
import { DECLARATION_COMPONENT_VIEW, HEADER_OFFSET, T_HOST, TVIEW } from './interfaces/view';
// [Assert functions do not constraint type when they are guarded by a truthy
// expression.](https://github.com/microsoft/TypeScript/issues/37295)
export function assertTNodeForLView(tNode, lView) {
    assertTNodeForTView(tNode, lView[TVIEW]);
}
export function assertTNodeForTView(tNode, tView) {
    assertTNode(tNode);
    tNode.hasOwnProperty('tView_') &&
        assertEqual(tNode.tView_, tView, 'This TNode does not belong to this TView.');
}
export function assertTNode(tNode) {
    assertDefined(tNode, 'TNode must be defined');
    if (!(tNode && typeof tNode === 'object' && tNode.hasOwnProperty('directiveStylingLast'))) {
        throwError('Not of type TNode, got: ' + tNode);
    }
}
export function assertTIcu(tIcu) {
    assertDefined(tIcu, 'Expected TIcu to be defined');
    if (!(typeof tIcu.currentCaseLViewIndex === 'number')) {
        throwError('Object is not of TIcu type.');
    }
}
export function assertComponentType(actual, msg = 'Type passed in is not ComponentType, it does not have \'ɵcmp\' property.') {
    if (!getComponentDef(actual)) {
        throwError(msg);
    }
}
export function assertNgModuleType(actual, msg = 'Type passed in is not NgModuleType, it does not have \'ɵmod\' property.') {
    if (!getNgModuleDef(actual)) {
        throwError(msg);
    }
}
export function assertCurrentTNodeIsParent(isParent) {
    assertEqual(isParent, true, 'currentTNode should be a parent');
}
export function assertHasParent(tNode) {
    assertDefined(tNode, 'currentTNode should exist!');
    assertDefined(tNode.parent, 'currentTNode should have a parent');
}
export function assertLContainer(value) {
    assertDefined(value, 'LContainer must be defined');
    assertEqual(isLContainer(value), true, 'Expecting LContainer');
}
export function assertLViewOrUndefined(value) {
    value && assertEqual(isLView(value), true, 'Expecting LView or undefined or null');
}
export function assertLView(value) {
    assertDefined(value, 'LView must be defined');
    assertEqual(isLView(value), true, 'Expecting LView');
}
export function assertFirstCreatePass(tView, errMessage) {
    assertEqual(tView.firstCreatePass, true, errMessage || 'Should only be called in first create pass.');
}
export function assertFirstUpdatePass(tView, errMessage) {
    assertEqual(tView.firstUpdatePass, true, errMessage || 'Should only be called in first update pass.');
}
/**
 * This is a basic sanity check that an object is probably a directive def. DirectiveDef is
 * an interface, so we can't do a direct instanceof check.
 */
export function assertDirectiveDef(obj) {
    if (obj.type === undefined || obj.selectors == undefined || obj.inputs === undefined) {
        throwError(`Expected a DirectiveDef/ComponentDef and this object does not seem to have the expected shape.`);
    }
}
export function assertIndexInDeclRange(lView, index) {
    const tView = lView[1];
    assertBetween(HEADER_OFFSET, tView.bindingStartIndex, index);
}
export function assertIndexInExpandoRange(lView, index) {
    const tView = lView[1];
    assertBetween(tView.expandoStartIndex, lView.length, index);
}
export function assertBetween(lower, upper, index) {
    if (!(lower <= index && index < upper)) {
        throwError(`Index out of range (expecting ${lower} <= ${index} < ${upper})`);
    }
}
export function assertProjectionSlots(lView, errMessage) {
    assertDefined(lView[DECLARATION_COMPONENT_VIEW], 'Component views should exist.');
    assertDefined(lView[DECLARATION_COMPONENT_VIEW][T_HOST].projection, errMessage ||
        'Components with projection nodes (<ng-content>) must have projection slots defined.');
}
export function assertParentView(lView, errMessage) {
    assertDefined(lView, errMessage || 'Component views should always have a parent view (component\'s host view)');
}
/**
 * This is a basic sanity check that the `injectorIndex` seems to point to what looks like a
 * NodeInjector data structure.
 *
 * @param lView `LView` which should be checked.
 * @param injectorIndex index into the `LView` where the `NodeInjector` is expected.
 */
export function assertNodeInjector(lView, injectorIndex) {
    assertIndexInExpandoRange(lView, injectorIndex);
    assertIndexInExpandoRange(lView, injectorIndex + 8 /* NodeInjectorOffset.PARENT */);
    assertNumber(lView[injectorIndex + 0], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 1], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 2], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 3], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 4], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 5], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 6], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 7], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 8 /* NodeInjectorOffset.PARENT */], 'injectorIndex should point to parent injector');
}
//# sourceMappingURL=data:application/json;base64,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