import { Component, ElementRef, EventEmitter, HostBinding, HostListener, Inject, Input, Output, ViewChild } from '@angular/core';
import { DOCUMENT } from '@angular/common';
import { animate, state, style, transition, trigger } from '@angular/animations';
import { coerceBooleanProperty } from '@angular/cdk/coercion';
import { A11yModule } from '@angular/cdk/a11y';
import { ModalContentComponent } from '../modal-content/modal-content.component';
import { ModalDialogComponent } from '../modal-dialog/modal-dialog.component';
import * as i0 from "@angular/core";
import * as i1 from "../modal.service";
import * as i2 from "../../backdrop/backdrop.service";
import * as i3 from "@angular/cdk/a11y";
export class ModalComponent {
    constructor(document, renderer, hostElement, modalService, backdropService) {
        this.document = document;
        this.renderer = renderer;
        this.hostElement = hostElement;
        this.modalService = modalService;
        this.backdropService = backdropService;
        /**
         * Align the modal in the center or top of the screen.
         * @type {'top' | 'center'}
         * @default 'top'
         */
        this.alignment = 'top';
        /**
         * Apply a backdrop on body while modal is open.
         * @type boolean | 'static'
         * @default true
         */
        this.backdrop = true;
        /**
         * Closes the modal when escape key is pressed.
         * @type boolean
         * @default true
         */
        this.keyboard = true;
        /**
         * Remove animation to create modal that simply appear rather than fade in to view.
         */
        this.transition = true;
        /**
         * Default role for modal. [docs]
         * @type string
         * @default 'dialog'
         */
        this.role = 'dialog';
        /**
         * Set aria-modal html attr for modal. [docs]
         * @type boolean
         * @default true
         */
        this.ariaModal = true;
        this._scrollable = false;
        /**
         * Event triggered on modal dismiss.
         */
        this.visibleChange = new EventEmitter();
        this._show = true;
        this.mouseDownTarget = null;
    }
    /**
     * Create a scrollable modal that allows scrolling the modal body.
     * @type boolean
     */
    set scrollable(value) {
        this._scrollable = coerceBooleanProperty(value);
    }
    get scrollable() {
        return this._scrollable;
    }
    /**
     * Toggle the visibility of modal component.
     * @type boolean
     */
    set visible(value) {
        const newValue = coerceBooleanProperty(value);
        if (this._visible !== newValue) {
            this._visible = newValue;
            this.setBackdrop(this.backdrop !== false && newValue);
            this.setBodyStyles(newValue);
            this.visibleChange.emit(newValue);
        }
    }
    get visible() {
        return this._visible;
    }
    // private inBoundingClientRect!: boolean;
    get hostClasses() {
        return {
            modal: true,
            fade: this.transition,
            show: this.show
        };
    }
    get ariaHidden() {
        return this.visible ? null : true;
    }
    ;
    get tabIndex() {
        return '-1';
    }
    get animateTrigger() {
        return this.visible ? 'visible' : 'hidden';
    }
    get show() {
        return this.visible && this._show;
    }
    set show(value) {
        this._show = value;
    }
    animateStart(event) {
        const scrollbarWidth = this.backdropService.scrollbarWidth;
        if (event.toState === 'visible') {
            this.renderer.setStyle(this.document.body, 'overflow', 'hidden');
            this.renderer.setStyle(this.document.body, 'padding-right', scrollbarWidth);
            this.renderer.setStyle(this.hostElement.nativeElement, 'display', 'block');
        }
        else {
            if (!this.transition) {
                this.renderer.setStyle(this.hostElement.nativeElement, 'display', 'none');
            }
        }
    }
    animateDone(event) {
        setTimeout(() => {
            if (event.toState === 'hidden') {
                this.renderer.setStyle(this.hostElement.nativeElement, 'display', 'none');
                this.renderer.removeStyle(this.document.body, 'overflow');
                this.renderer.removeStyle(this.document.body, 'padding-right');
            }
        });
        this.show = this.visible;
    }
    onKeyDownHandler(event) {
        if (event.key === 'Escape' && this.keyboard && this.visible) {
            if (this.backdrop === 'static') {
                this.setStaticBackdrop();
            }
            else {
                this.modalService.toggle({ show: false, modal: this });
            }
        }
    }
    onMouseDownHandler($event) {
        this.mouseDownTarget = $event.target;
    }
    onClickHandler($event) {
        if (this.mouseDownTarget !== $event.target) {
            this.mouseDownTarget = null;
            return;
        }
        const targetElement = $event.target;
        if (targetElement === this.hostElement.nativeElement) {
            if (this.backdrop === 'static') {
                this.setStaticBackdrop();
                return;
            }
            this.modalService.toggle({ show: false, modal: this });
        }
    }
    ngOnInit() {
        this.stateToggleSubscribe();
    }
    ngOnDestroy() {
        this.modalService.toggle({ show: false, modal: this });
        this.stateToggleSubscribe(false);
    }
    stateToggleSubscribe(subscribe = true) {
        if (subscribe) {
            this.stateToggleSubscription = this.modalService.modalState$.subscribe((action) => {
                if (this === action.modal || this.id === action.id) {
                    if ('show' in action) {
                        this.visible = action?.show === 'toggle' ? !this.visible : action.show;
                    }
                }
                else {
                    if (this.visible) {
                        this.visible = false;
                    }
                }
            });
        }
        else {
            this.stateToggleSubscription?.unsubscribe();
        }
    }
    setBackdrop(setBackdrop) {
        if (setBackdrop) {
            this.activeBackdrop = this.backdropService.setBackdrop('modal');
        }
        else {
            this.activeBackdrop = this.backdropService.clearBackdrop(this.activeBackdrop);
        }
    }
    setBodyStyles(open) {
        if (open) {
            if (this.backdrop === true) {
                this.renderer.addClass(this.document.body, 'modal-open');
            }
        }
        else {
            this.renderer.removeClass(this.document.body, 'modal-open');
        }
    }
    setStaticBackdrop() {
        if (this.transition) {
            this.renderer.addClass(this.hostElement.nativeElement, 'modal-static');
            this.renderer.setStyle(this.hostElement.nativeElement, 'overflow-y', 'hidden');
            setTimeout(() => {
                this.renderer.removeClass(this.hostElement.nativeElement, 'modal-static');
                this.renderer.removeStyle(this.hostElement.nativeElement, 'overflow-y');
            }, 300);
        }
    }
}
ModalComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "15.2.8", ngImport: i0, type: ModalComponent, deps: [{ token: DOCUMENT }, { token: i0.Renderer2 }, { token: i0.ElementRef }, { token: i1.ModalService }, { token: i2.BackdropService }], target: i0.ɵɵFactoryTarget.Component });
ModalComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "15.2.8", type: ModalComponent, isStandalone: true, selector: "c-modal", inputs: { alignment: "alignment", backdrop: "backdrop", fullscreen: "fullscreen", keyboard: "keyboard", id: "id", size: "size", transition: "transition", role: "role", ariaModal: "ariaModal", scrollable: "scrollable", visible: "visible" }, outputs: { visibleChange: "visibleChange" }, host: { listeners: { "@showHide.start": "animateStart($event)", "@showHide.done": "animateDone($event)", "document:keyup": "onKeyDownHandler($event)", "mousedown": "onMouseDownHandler($event)", "click": "onClickHandler($event)" }, properties: { "attr.role": "this.role", "attr.aria-modal": "this.ariaModal", "class": "this.hostClasses", "attr.aria-hidden": "this.ariaHidden", "attr.tabindex": "this.tabIndex", "@showHide": "this.animateTrigger" } }, viewQueries: [{ propertyName: "modalContent", first: true, predicate: ModalContentComponent, descendants: true, read: ElementRef }], exportAs: ["cModal"], ngImport: i0, template: "<c-modal-dialog\n  [alignment]=\"alignment\"\n  [fullscreen]=\"fullscreen\"\n  [scrollable]=\"scrollable\"\n  [size]=\"size\">\n  <c-modal-content>\n    <div [cdkTrapFocus]=\"visible\" [cdkTrapFocusAutoCapture]=\"visible\" style=\"display: contents;\">\n      <ng-content></ng-content>\n    </div>\n  </c-modal-content>\n</c-modal-dialog>\n", dependencies: [{ kind: "component", type: ModalDialogComponent, selector: "c-modal-dialog", inputs: ["alignment", "fullscreen", "scrollable", "size"] }, { kind: "component", type: ModalContentComponent, selector: "c-modal-content" }, { kind: "ngmodule", type: A11yModule }, { kind: "directive", type: i3.CdkTrapFocus, selector: "[cdkTrapFocus]", inputs: ["cdkTrapFocus", "cdkTrapFocusAutoCapture"], exportAs: ["cdkTrapFocus"] }], animations: [
        trigger('showHide', [
            state('visible', style({
            // display: 'block'
            })),
            state('hidden', style({
            // display: 'none'
            })),
            transition('visible <=> *', [animate('300ms')])
        ])
    ] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "15.2.8", ngImport: i0, type: ModalComponent, decorators: [{
            type: Component,
            args: [{ selector: 'c-modal', animations: [
                        trigger('showHide', [
                            state('visible', style({
                            // display: 'block'
                            })),
                            state('hidden', style({
                            // display: 'none'
                            })),
                            transition('visible <=> *', [animate('300ms')])
                        ])
                    ], exportAs: 'cModal', standalone: true, imports: [ModalDialogComponent, ModalContentComponent, A11yModule], template: "<c-modal-dialog\n  [alignment]=\"alignment\"\n  [fullscreen]=\"fullscreen\"\n  [scrollable]=\"scrollable\"\n  [size]=\"size\">\n  <c-modal-content>\n    <div [cdkTrapFocus]=\"visible\" [cdkTrapFocusAutoCapture]=\"visible\" style=\"display: contents;\">\n      <ng-content></ng-content>\n    </div>\n  </c-modal-content>\n</c-modal-dialog>\n" }]
        }], ctorParameters: function () { return [{ type: Document, decorators: [{
                    type: Inject,
                    args: [DOCUMENT]
                }] }, { type: i0.Renderer2 }, { type: i0.ElementRef }, { type: i1.ModalService }, { type: i2.BackdropService }]; }, propDecorators: { alignment: [{
                type: Input
            }], backdrop: [{
                type: Input
            }], fullscreen: [{
                type: Input
            }], keyboard: [{
                type: Input
            }], id: [{
                type: Input
            }], size: [{
                type: Input
            }], transition: [{
                type: Input
            }], role: [{
                type: Input
            }, {
                type: HostBinding,
                args: ['attr.role']
            }], ariaModal: [{
                type: Input
            }, {
                type: HostBinding,
                args: ['attr.aria-modal']
            }], scrollable: [{
                type: Input
            }], visible: [{
                type: Input
            }], visibleChange: [{
                type: Output
            }], modalContent: [{
                type: ViewChild,
                args: [ModalContentComponent, { read: ElementRef }]
            }], hostClasses: [{
                type: HostBinding,
                args: ['class']
            }], ariaHidden: [{
                type: HostBinding,
                args: ['attr.aria-hidden']
            }], tabIndex: [{
                type: HostBinding,
                args: ['attr.tabindex']
            }], animateTrigger: [{
                type: HostBinding,
                args: ['@showHide']
            }], animateStart: [{
                type: HostListener,
                args: ['@showHide.start', ['$event']]
            }], animateDone: [{
                type: HostListener,
                args: ['@showHide.done', ['$event']]
            }], onKeyDownHandler: [{
                type: HostListener,
                args: ['document:keyup', ['$event']]
            }], onMouseDownHandler: [{
                type: HostListener,
                args: ['mousedown', ['$event']]
            }], onClickHandler: [{
                type: HostListener,
                args: ['click', ['$event']]
            }] } });
//# sourceMappingURL=data:application/json;base64,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