import { EventEmitter, Injectable } from '@angular/core';
import { FormControl, FormGroup } from '@angular/forms';
import { filter } from 'rxjs/operators';
import { Subject } from 'rxjs';
import { isNullOrUndefined, isUndefined } from '@magic-xpa/mscorelib';
import { StylesMapManager } from './StylesMapManager';
import { EngineMagicService } from './engine.magic.service';
import { Records } from '../controls.metadata.model';
import { CommandType, GuiConstants, HtmlProperties } from '@magic-xpa/gui';
import { NString, StringBuilder } from '@magic-xpa/mscorelib';
import { StorageAttribute } from '@magic-xpa/utils';
import { getGuiEventObj } from '@magic-xpa/engine';
import { GuiInteractiveExecutor } from '../ui/GuiInteractiveExecutor';
import { OverlayWindowService } from './OverlayWindowService';
import * as i0 from "@angular/core";
import * as i1 from "./engine.magic.service";
import * as i2 from "./OverlayWindowService";
export class TaskMagicService {
    constructor(magic, overlayWindowService) {
        this.magic = magic;
        this.overlayWindowService = overlayWindowService;
        this.Records = new Records();
        this.formGroups = [];
        this.refreshDom = new Subject();
        this.detectChanges = new Subject();
        this.interactiveCommands = new Subject();
        this.OnSelectedRowChanged = new EventEmitter();
        this.customPropertiesSubject = new Subject();
        this.recordsCountChangeSubject = new Subject();
        this.mgLoadSubject = new Subject();
        this.oldPageSize = 0;
        this.Records.setGuiTopIndex(0);
        this.mgInputDateFormat = null;
    }
    getIsLoggenIn() {
        return this.magic.getIsLoggedIn();
    }
    get taskId() {
        return this._taskId;
    }
    set taskId(value) {
        this._taskId = value;
    }
    get ScreenControlsData() {
        return this.Records.list['0'];
    }
    settemplate(value) {
        this.template = value;
    }
    buildScreenModeControls() {
        const group = new FormGroup({});
        for (const key in this.template) {
            if (this.template[key] == '0') {
                let fc = new FormControl('');
                this.setStubValue(0, fc, key);
                group.addControl(key, fc);
            }
        }
        this.ScreenModeControls = group;
    }
    isTableControl(id) {
        return this.template[id] == '1';
    }
    getMgInputDateFormat() { return this.mgInputDateFormat; }
    getFormControl(guiRowid, id) {
        let c;
        let group = this.isTableControl(id) ? this.formGroups[guiRowid] : this.ScreenModeControls;
        if (!isNullOrUndefined(group)) {
            group.get(id);
            c = group.controls[id];
        }
        return c;
    }
    setInputTextValue(controlId, guiRowid, val) {
        this.setValue(controlId, guiRowid, this.ConvertValToNative(controlId, 0, val));
        const fc = this.getFormControl(guiRowid, controlId);
        if (!isNullOrUndefined(val))
            fc.setValue(val);
    }
    buildTableRowControls(guiRowId) {
        const group = new FormGroup({});
        for (const key in this.template) {
            if (this.template[key] == '1') {
                let fc = new FormControl('');
                this.setStubValue(guiRowId, fc, key);
                group.addControl(key, fc);
            }
        }
        this.formGroups[guiRowId] = group;
    }
    updateRecordsBeforeCurrentView(value) {
        this.Records.recordsBeforeCurrentView = value;
    }
    setIncludesFirst(value) {
        this.Records.includesFirst = value;
    }
    setIncludesLast(value) {
        this.Records.includesLast = value;
    }
    markRowAsCreated(guiRowId) {
        this.Records.markRowAsCreated(guiRowId);
    }
    markRowAsNotCreated(guiRowId) {
        this.Records.markRowAsNotCreated(guiRowId);
    }
    startRowEditing(guiRowId) {
        this.Records.startRowEditing(guiRowId);
    }
    stopRowEditing(guiRowId) {
        this.Records.stopRowEditing(guiRowId);
    }
    isRowInRowEditing(guiRowId) {
        return this.Records.isRowInRowEditing(guiRowId);
    }
    setIsEmptyDataView(isEmpty) {
        if (this.Records.isEmptyDataView !== isEmpty) {
            if (isEmpty)
                this.recordsCountChangeSubject.next(0);
            this.Records.isEmptyDataView = isEmpty;
            this.tableService.refreshDataSource();
            this.refreshView();
        }
    }
    initTask(taskId, taskDescription) {
        this.Records.createFirst();
        this.oldPageSize = 0;
        if (this.IsStub()) {
            this.loadData();
        }
        else {
            this.taskId = taskId;
            this.settemplate(taskDescription);
        }
        this.buildScreenModeControls();
        if (!isNullOrUndefined(this.subscribeRefreshDom))
            this.subscribeRefreshDom.unsubscribe();
        if (!isNullOrUndefined(this.subscribeInteractiveCommands))
            this.subscribeInteractiveCommands.unsubscribe();
        this.subscribeRefreshDom = this.magic.refreshDom.pipe(filter(command => command.TaskTag === this.taskId))
            .subscribe(command => {
            this.executeCommand(command);
        });
        this.subscribeInteractiveCommands = this.magic.interactiveCommands
            .pipe(filter(command => command.TaskTag === this.taskId))
            .subscribe(command => {
            this.executeInteractiveCommand(command);
        });
    }
    refreshView() {
        this.detectChanges.next('');
    }
    insertEvent(guiEvent) {
        guiEvent.TaskID = this.taskId;
        this.magic.insertEvent(guiEvent);
    }
    GetControlPictureMask(controlName) {
        return this.magic.GetControlPictureMask(this.taskId, controlName);
    }
    ValidateControlValue(controlName, value) {
        return this.magic.ValidateControlValue(this.taskId, controlName, value);
    }
    GetRangedValue(controlName, value) {
        return this.magic.GetRangedValue(this.taskId, controlName, value);
    }
    GetFldRanges(controlName) {
        return this.magic.GetFldRanges(this.taskId, controlName);
    }
    getProperty(controlId, prop, guiRowId) {
        if (isNullOrUndefined(guiRowId))
            guiRowId = '0';
        if (this.IsStub())
            return this.getPropertyStub(this.Records.list[guiRowId], controlId, prop);
        else {
            let rec = this.Records.list[guiRowId];
            if (isNullOrUndefined(rec))
                debugger;
            else
                return this.Records.list[guiRowId].getProperty(controlId, prop);
        }
    }
    getPropertyStub(ControlsProperties, controlId, prop) {
        ControlsProperties = ControlsProperties.ControlsProperties;
        if (controlId in ControlsProperties) {
            if (prop in ControlsProperties[controlId].properties) {
                return ControlsProperties[controlId].properties[prop];
            }
        }
        return '';
    }
    getStyleStub(ControlsProperties, controlId, styleName) {
        ControlsProperties = ControlsProperties.ControlsProperties;
        return ControlsProperties[controlId].stylesMap.get(styleName);
    }
    getClasses(controlId, guiRowId) {
        if (isNullOrUndefined(guiRowId))
            guiRowId = '0';
        return this.Records.list[guiRowId].getControlMetadata(controlId).classes;
    }
    getStyle(controlId, styleName, guiRowId) {
        if (isNullOrUndefined(guiRowId))
            guiRowId = '0';
        if (this.IsStub())
            return this.getStyleStub(this.Records.list[guiRowId], controlId, styleName);
        else {
            let magicValue = this.Records.list[guiRowId].getControlMetadata(controlId).stylesMap.get(styleName);
            return StylesMapManager.magicValueGetStyle(styleName, magicValue);
        }
    }
    getValue(controlId, guiRowId) {
        if (isNullOrUndefined(guiRowId))
            guiRowId = '0';
        return this.Records.list.length > +guiRowId ? this.Records.list[guiRowId].values[controlId] : '';
    }
    getFormattedValue(controlName, val, rowId) {
        return this.magic.GetFormattedValue(this.taskId, controlName, val, rowId);
    }
    setValue(controlId, guiRowId, value) {
        return this.Records.list[guiRowId].setValue(controlId, value);
    }
    executeInteractiveCommand(guiInteractiveCommand) {
        GuiInteractiveExecutor.executeInteractiveCommand(this, guiInteractiveCommand, this.overlayWindowService);
        this.interactiveCommands.next(guiInteractiveCommand);
    }
    handleSetProperty(command, isTableChild) {
        let controlId = command.CtrlName;
        let dvRowId = (command.line || 0);
        let guiRowId = this.getGuiRowId(dvRowId, isTableChild);
        let properties;
        if (command.Operation === HtmlProperties.SelectedRow) {
            if (command.obj1 !== GuiConstants.NO_ROW_SELECTED) {
                guiRowId = this.getGuiRowId(command.obj1, true);
                properties = this.Records.list[guiRowId].getControlMetadata(controlId);
                properties.properties.set(command.Operation, command.line);
                this.tableService.selectRow(guiRowId.toString());
                this.OnSelectedRowChanged.emit(guiRowId.toString());
            }
        }
        else if (command.Operation === HtmlProperties.Enabled) {
            let c = this.getFormControl(guiRowId.toString(), controlId);
            if (!isNullOrUndefined(c)) {
                if (command.obj1)
                    c.enable();
                else
                    c.disable();
            }
            properties = this.Records.list[guiRowId].getControlMetadata(controlId);
            properties.properties.set(command.Operation, command.obj1);
        }
        else {
            properties = this.Records.list[guiRowId].getControlMetadata(controlId);
            properties.properties.set(command.Operation, command.obj1);
        }
    }
    executeCommand(command) {
        let dvRowId = (command.line || 0);
        let guiRowId = 0;
        let controlId = command.CtrlName;
        let properties;
        let isTableChild = this.isTableControl(controlId);
        if (command.obj !== null)
            isTableChild = isTableChild || command.obj.IsTableChild;
        guiRowId = this.getGuiRowId(dvRowId, isTableChild);
        switch (command.CommandType) {
            case CommandType.SET_IS_LOGGED_IN:
                break;
            case CommandType.SET_TABLE_TOP_INDEX:
                this.tableService.setTableTopIndex(command.number);
                break;
            case CommandType.REFRESH_TASK:
                this.tableService.refreshDataSource();
                this.refreshView();
                break;
            case CommandType.SET_TABLE_ITEMS_COUNT:
                if (!isUndefined(command.number)) {
                    this.tableService.updateTableSize(command.number);
                    if (command.number > 0 && this.Records.sizeUpdated &&
                        !this.Records.isEmptyDataView) {
                        this.recordsCountChangeSubject.next(command.number);
                        this.Records.sizeUpdated = false;
                    }
                }
                break;
            case CommandType.SET_CHUNK_SIZE:
                if (!isUndefined(command.number)) {
                    this.tableService.setChunkSize(command.number);
                }
                break;
            case CommandType.SET_RECORDS_BEFORE_CURRENT_VIEW:
                this.updateRecordsBeforeCurrentView(command.number);
                break;
            case CommandType.CREATE_TABLE_ROW:
                guiRowId = this.getGuiRowId(command.number, true);
                this.buildTableRowControls(guiRowId);
                this.markRowAsCreated(guiRowId);
                break;
            case CommandType.SET_IS_ROW_EDITING:
                guiRowId = this.getGuiRowId(command.line, command.value);
                this.startRowEditing(guiRowId);
                break;
            case CommandType.SET_NOT_IS_ROW_EDITING:
                guiRowId = this.getGuiRowId(command.line, command.value);
                this.stopRowEditing(guiRowId);
                break;
            case CommandType.UNDO_CREATE_TABLE_ROW:
                this.markRowAsNotCreated(this.getGuiRowId(command.number, true));
                break;
            case CommandType.SET_TABLE_INCLUDES_FIRST:
                this.setIncludesFirst(command.Bool1);
                break;
            case CommandType.SET_TABLE_INCLUDES_LAST:
                this.setIncludesLast(command.Bool1);
                break;
            case CommandType.SET_PROPERTY:
                this.handleSetProperty(command, isTableChild);
                if (command.Operation == HtmlProperties.ReadOnly ||
                    command.Operation == HtmlProperties.ItemsList)
                    this.refreshDom.next(command);
                break;
            case CommandType.PROP_SET_USER_PROPERTY:
                properties = this.Records.list[guiRowId].getControlMetadata(controlId);
                if (properties.userProperties.get(command.Operation) !== command.obj1) {
                    properties.userProperties.set(command.Operation, command.obj1);
                    this.customPropertiesSubject.next({ propertyName: controlId + '~' + command.Operation, value: command.obj1, rowId: guiRowId });
                }
                break;
            case CommandType.PROP_SET_CUSTOM_VALIDATOR:
                {
                    properties = this.Records.list[guiRowId].getControlMetadata(controlId);
                    let control = this.getFormControl(guiRowId.toString(), controlId);
                    if (control != null) {
                        properties.customValidators.set(command.Operation, command.obj1);
                        if (control.validator != null)
                            control.setValidators([control.validator, this.customValidator(guiRowId.toString(), controlId)]);
                        else
                            control.setValidators(this.customValidator(guiRowId.toString(), controlId));
                        control.updateValueAndValidity();
                    }
                }
                break;
            case CommandType.SET_CLASS:
                properties = this.Records.list[guiRowId].getControlMetadata(controlId);
                properties.setClass(command.Operation, command.obj1);
                break;
            case CommandType.SET_STYLE:
                properties = this.Records.list[guiRowId].getControlMetadata(controlId);
                properties.setStyle(command.Operation, command.obj1);
                break;
            case CommandType.SET_ATTRIBUTE:
                properties = this.Records.list[0].getControlMetadata(controlId);
                properties.dataType = String.fromCharCode(command.number);
                break;
            case CommandType.SET_VALUE:
                this.Records.list[guiRowId].values[controlId] = command.value;
                let c = this.getFormControl(guiRowId.toString(), controlId);
                if (!isNullOrUndefined(c)) {
                    c.setValue(this.ConvertValFromNative(controlId, command.line, command.value));
                }
                break;
            case CommandType.CREATE_SUB_FORM:
                this.mgSubformService.addSubformComp(command.CtrlName, command.userDropFormat.toString(), command.str, command.fileName, command.contextID, command.params, command.Bool1);
                break;
            case CommandType.CLOSE_SUB_FORM:
                this.mgSubformService.deleteSubformComp(command.CtrlName, command.userDropFormat.toString());
                break;
            case CommandType.SET_EMPTY_DATAVIEW:
                this.setIsEmptyDataView(command.Bool1);
                break;
            case CommandType.SET_FOCUS:
                this.refreshDom.next(command);
                break;
            case CommandType.SET_WC_IDLE:
                this.mgLoadSubject.next();
                break;
        }
    }
    customValidator(rowid, id) {
        return (control) => {
            let controlMetadata;
            controlMetadata = this.Records.list[rowid].getControlMetadata(id);
            for (let key of controlMetadata.customValidators.keys()) {
                let validator = controlMetadata.customValidators.get(key);
                if (!validator.isValid()) {
                    return { [key]: true };
                }
            }
            return null;
        };
    }
    ConvertValToNative(controlId, rowId, val) {
        let properties;
        properties = this.Records.list[0].getControlMetadata(controlId);
        switch (properties.dataType) {
            case StorageAttribute.BOOLEAN: {
                if (typeof val === 'string') {
                    let ranges = this.GetFldRanges(controlId);
                    let found = ranges.indexOf(',');
                    let trueOption = ranges.substr(0, found);
                    let falseOption = ranges.substr(found + 1);
                    if (NString.Equals(val, trueOption, true))
                        return true;
                    else if (NString.Equals(val, falseOption, true))
                        return false;
                    else
                        return val;
                }
                else
                    return val;
            }
            case StorageAttribute.TIME: {
                if (typeof val === 'string') {
                    let idx = val.indexOf(':', 0);
                    let hours = +val.substr(0, idx);
                    let minutes = +val.substr(idx + 1, 2);
                    let seconds = 0;
                    if (val.length > 5) {
                        idx = val.indexOf(':', idx + 1);
                        if (idx >= 0)
                            seconds = +val.substr(idx + 1, 2);
                    }
                    if (val.indexOf('PM') > 0)
                        hours += 12;
                    let dt = new Date();
                    dt.setHours(hours);
                    dt.setMinutes(minutes);
                    dt.setSeconds(seconds);
                    return dt;
                }
                else
                    return val;
            }
            case StorageAttribute.NUMERIC:
                if (isUndefined(val))
                    return val;
                else
                    return (val != null) ? +val : null;
            default:
                return val;
        }
    }
    ConvertValFromNative(controlId, rowId, val) {
        let properties;
        properties = this.Records.list[0].getControlMetadata(controlId);
        switch (properties.dataType) {
            case StorageAttribute.TIME: {
                if (val instanceof Date) {
                    let str = new StringBuilder();
                    let format = val.getHours() > 9 ? '{0}' : '0{0}';
                    str.AppendFormat(format, val.getHours().toString());
                    str.Append(':');
                    format = val.getMinutes() > 9 ? '{0}' : '0{0}';
                    str.AppendFormat(format, val.getMinutes().toString());
                    if (val.getSeconds() > 0) {
                        str.Append(':');
                        format = val.getSeconds() > 9 ? '{0}' : '0{0}';
                        str.AppendFormat(format, val.getSeconds().toString());
                    }
                    return str.ToString();
                }
                else
                    return val;
            }
            case StorageAttribute.BOOLEAN: {
                let properties;
                properties = this.Records.list['0'].getControlMetadata(controlId);
                if (typeof val === 'boolean' && properties.rangeValidator) {
                    let ranges = this.GetFldRanges(controlId);
                    let found = ranges.indexOf(',');
                    let trueOption = ranges.substr(0, found);
                    let falseOption = ranges.substr(found + 1);
                    if (val)
                        return trueOption;
                    else
                        return falseOption;
                }
                else
                    return val;
            }
            default:
                return val;
        }
    }
    resize(pageSize, topGuiRowId) {
        let guiEvent = getGuiEventObj('resize', 'table', 0);
        guiEvent.PageSize = pageSize;
        guiEvent.Line = topGuiRowId;
        this.insertEvent(guiEvent);
    }
    onScrollDown() {
        if (!this.Records.includesLast) {
            let guiEvent = getGuiEventObj('getRows', 'table', 0);
            guiEvent.Line = this.formGroups.length;
            this.insertEvent(guiEvent);
        }
    }
    setTitle(newTitle) {
        this.mgTitleService.setTitle(newTitle);
    }
    getGuiRowId(dvRowId, isTableControl) {
        return (isTableControl ? dvRowId + this.Records.recordsBeforeCurrentView : 0);
    }
    getDvRowId(guiRowId) {
        return (guiRowId - this.Records.recordsBeforeCurrentView);
    }
    dispose() {
        this.refreshDom.complete();
        this.detectChanges.complete();
        this.subscribeInteractiveCommands.unsubscribe();
        this.subscribeRefreshDom.unsubscribe();
        this.interactiveCommands.complete();
        this.customPropertiesSubject.complete();
        this.OnSelectedRowChanged.complete();
    }
    onComboboxSelectionChanged(event, idx, line) {
        let guiEvent = getGuiEventObj('selectionchanged', idx, line);
        if (typeof (event) == 'number') {
            guiEvent.Value = event;
        }
        else {
            if (!isNullOrUndefined(event.target)) {
                let indexes = new Array(event.target.selectedOptions.length);
                for (let i = 0; i < event.target.selectedOptions.length; i++) {
                    indexes[i] = event.target.selectedOptions[i].index;
                }
                guiEvent.Value = indexes.join(',');
            }
            else
                guiEvent.Value = event.value;
        }
        this.insertEvent(guiEvent);
    }
    onListBoxSelectionChanged(event, idx) {
        let guiEvent = getGuiEventObj('selectionchanged', idx, 0);
        let selectedOptions;
        if (!isNullOrUndefined(event.target))
            selectedOptions = event.target.selectedOptions;
        else
            selectedOptions = event.source.selectedOptions.selected;
        let length = selectedOptions.length;
        let indexes = new Array(length);
        for (let i = 0; i < length; i++) {
            if (!isNullOrUndefined(event.target))
                indexes[i] = (selectedOptions[i]).index;
            else
                indexes[i] = (selectedOptions[i]).value;
        }
        guiEvent.Value = indexes;
        this.insertEvent(guiEvent);
    }
    onCheckChanged(event, idx, rowId) {
        if (typeof rowId === 'undefined')
            rowId = 0;
        let guiEvent = getGuiEventObj('selectionchanged', idx, rowId);
        if (typeof (event) == 'boolean') {
            guiEvent.Value = event;
        }
        else {
            if (typeof event.target === 'undefined')
                guiEvent.Value = (event).checked;
            else
                guiEvent.Value = (event.target).checked;
        }
        this.insertEvent(guiEvent);
    }
    mgOnTabSelectionChanged(idx, layer) {
        let guiEvent = getGuiEventObj('selectionchanged', idx, 0);
        guiEvent.Value = layer.toString();
        this.insertEvent(guiEvent);
    }
    mgOnRadioSelectionChanged(idx) {
        let result = this.getFormControl('0', idx);
        let guiEvent = getGuiEventObj('selectionchanged', idx, 0);
        if (typeof result.value !== 'number')
            guiEvent.Value = result.value.index;
        else
            guiEvent.Value = result.value.toString();
        this.insertEvent(guiEvent);
    }
    close() {
        this.insertEvent(getGuiEventObj('close', null, 0));
    }
    IsStub() {
        return this.magic.isStub;
    }
    saveData(data) {
        this.magic.saveData(data);
    }
    createData() {
        let myData = {
            records: this.Records,
            template: this.template
        };
        let text = 'loadData():any   {\n' +
            '    let stubData = ' + JSON.stringify(myData) + ';\n' +
            '    this.loadStubData(stubData);}';
        console.log(text);
        this.saveData(text);
    }
    loadStubData(stubData) {
        this.Records = stubData.records;
        this.settemplate(stubData.template);
        this.taskId = '1';
        for (let i = 0; i < this.Records.list.length; i++)
            this.buildTableRowControls(i);
    }
    loadData() {
        alert('Please, overwrite method loadData');
    }
    setStubValue(guiRowId, fc, name) {
        if (this.IsStub()) {
            try {
                let val = this.Records.list[guiRowId].values[name];
                fc.setValue(val);
            }
            catch (e) {
            }
        }
    }
}
TaskMagicService.ɵfac = function TaskMagicService_Factory(t) { return new (t || TaskMagicService)(i0.ɵɵinject(i1.EngineMagicService), i0.ɵɵinject(i2.OverlayWindowService)); };
TaskMagicService.ɵprov = i0.ɵɵdefineInjectable({ token: TaskMagicService, factory: TaskMagicService.ɵfac });
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(TaskMagicService, [{
        type: Injectable
    }], function () { return [{ type: i1.EngineMagicService }, { type: i2.OverlayWindowService }]; }, null); })();
//# sourceMappingURL=data:application/json;base64,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