import * as ts from 'typescript';
import * as EJS from "ejs";
import {chain, Rule, SchematicContext, SchematicsException, Tree} from "@angular-devkit/schematics";
import {env, GEN_COMPONENTS_ARRAY, GEN_COMPONENTS_HASH, GEN_LAZY_LOAD_MODULES_ARRAY} from "../env";
import {MagicOptionScheme} from "./magic-option.scheme";
import {GeneratedFileTypes, TemplateConfig, View, WindowType} from "../../../types";
import {generate} from "./generate.rule";
import {getSourceFile} from "../../utils/ast";
import {insertImport, insertImport2} from "../../utils/devkit-utils/route-utils";
import {InsertChange} from "../../utils/devkit-utils/change";
import {findNodes} from "../../utils/devkit-utils/ast-utils";
import {LogLn} from "../Util";
import {isNullOrUndefined} from "util";


let existCmpListSet: Set<string>;
let newCmpListSet = new Set();
let toAddImports: boolean = true;
let orgPreventLog = env.prevent_log;


export function componentlistGen(options: MagicOptionScheme, module_name: string, loadOnDemand: boolean): Rule {
  return (host: Tree, context: SchematicContext) => {
    LogLn(`[>] Step 4: components-list.`);
    return chain([
      addNewCmpsToCmpListOrGenAll(options, module_name),
      addCmpsImportsToCmpList(options, module_name),
      createLazyModuleMap(options, module_name),
      addLazyLoadComponentList(options, module_name, loadOnDemand),
      addLazyModuleToMagicGen(options),
      addImportForLazyModule(options)
    ])(host, context);
  }
}

// create the lazy load map if it does not exist
function createLazyModuleMap(options: MagicOptionScheme, module_name: string) {
  return (host: Tree, context: SchematicContext) => {
    const cmpListPath = env.metadata.paths.componentListPath("");
    env.prevent_log = false;
    const source = host.read(cmpListPath).toString("utf-8")

      const LazyModuleArrayStart = 'export const ' + GEN_LAZY_LOAD_MODULES_ARRAY + ' = {';
      const LazyMOduleArrayEnd = '};';
      // create lazy load component array if it does not exist
      if (source.indexOf(LazyModuleArrayStart) < 0) {
        LogLn(`[>] Creating LazyLoadModulesMap `);
        // add the code of creating map in file
        host.overwrite(cmpListPath, source + "\r\n" + LazyModuleArrayStart + LazyMOduleArrayEnd);
      }

  }
}

// add the lazy load components
function addLazyLoadComponentList(options: MagicOptionScheme, module_name: string, loadOnDemand: boolean): Rule {
  return (host: Tree, context: SchematicContext) => {
    if (loadOnDemand) {
      let lazyLoadedComponents: Array<View> = new Array<View>();
      const cmpListPath = env.metadata.paths.componentListPath("");

      const source = host.read(cmpListPath);
      if (source) {
        let cmpListSource = getSourceFile(host, cmpListPath);

        let cmpListTs = getLazyLoadCompoenentsInfo(cmpListSource, GEN_LAZY_LOAD_MODULES_ARRAY);
        let sourceStep1 = host.read(cmpListPath).toString("utf-8");

        env.app.views.forEach(view => {
            if( isNullOrUndefined(cmpListTs.components) || !(cmpListTs.components.includes(view.props.component_uniquename))) {
              lazyLoadedComponents.push(view); // if component does not exist add it to list
              LogLn(`[>] Add  ${view.props.component_uniquename} component to LazyLoadModulesMap`, );
            }
        });

        // add components to map
        if (lazyLoadedComponents.length > 0) {
          let sourceStep2 = addComponentsToLazyLoad(sourceStep1, cmpListTs, lazyLoadedComponents, module_name);
          host.overwrite(cmpListPath, sourceStep2);
        }
      }
      env.prevent_log = orgPreventLog;
    }
    return host;
  }
}

function addImportForLazyModule(options) {
  return (tree: Tree, context: SchematicContext) => {

    let genLibPath=  env.metadata.paths.rootMagicGenFolder + '/magic/magic.gen.lib.module.ts';
    const recorder = tree.beginUpdate(genLibPath);
    let genLibSource = getSourceFile(tree, genLibPath);

    const importChange = insertImport(genLibSource, genLibPath, GEN_LAZY_LOAD_MODULES_ARRAY, './component-list.g') as InsertChange;
    if (importChange.toAdd) {
      recorder.insertLeft(importChange.pos, importChange.toAdd);
    }
    tree.commitUpdate(recorder);
  }
}

// add the lazy load module to magic.gen.lib
function addLazyModuleToMagicGen(options) {
  return (tree: Tree, context: SchematicContext) => {

    let origModuleFileName=  env.metadata.paths.rootMagicGenFolder + '/magic/magic.gen.lib.module.ts';
    if(!tree.exists(origModuleFileName)){
      LogLn(`      [>Error] File cannot be overwrite, The file is not exist  !!! : ${origModuleFileName}`);
    }
    const text = tree.read(origModuleFileName);

    if (text === null) {
      throw new SchematicsException(`File ${origModuleFileName} does not exist.`);
    }
    let sourceText = text.toString('utf-8');
    const lazyModuleStatement = '\r\n    componentList.lazyLoadModulesMap = '+ GEN_LAZY_LOAD_MODULES_ARRAY+';';
    if (sourceText !== "" && sourceText.indexOf(lazyModuleStatement)== -1) {
      let genLibSource = getSourceFile(tree, origModuleFileName);
      let newLazyModuleText: string = sourceText.replace(lazyModuleStatement, '').trim();
      const constNode: ts.FunctionLikeDeclarationBase = findNodes(genLibSource, ts.SyntaxKind.Constructor)[0] as ts.FunctionLikeDeclarationBase;

      let lastStatementEnd = (<ts.Block>constNode.body).statements.end;

      const prefix = sourceText.substring(0, lastStatementEnd);
      const suffix = sourceText.substring(lastStatementEnd );
      tree.overwrite(origModuleFileName,  `${prefix} ${lazyModuleStatement} ${suffix}` );

    }
    return tree;
  }
}

/*
* Add new component to component-list.g file (Array and Hash).
* If the file not exist new file will create by componentListGenIfNotExist function.
* */
function addNewCmpsToCmpListOrGenAll(options: MagicOptionScheme, module_name: string): Rule {
  return (host: Tree, context: SchematicContext) => {
    const metadata = env.metadata;
    const cmpsToGen = env.app.views;
    const cmpListPath = env.metadata.paths.componentListPath(module_name);
    const source = host.read(cmpListPath);
    if (source) {
      let cmpToGenNum = 0;
      let sourceStep1 = source.toString("utf-8");
      let cmpListSource = getSourceFile(host, cmpListPath);

      let cmpListTs = getComponentsArrayInfo(cmpListSource, GEN_COMPONENTS_ARRAY);
      existCmpListSet = new Set(cmpListTs.components);
      newCmpListSet = new Set<string>();

      // merge between components in the file and new component to gen
      cmpsToGen.forEach(newCmpGen => {
        if (!existCmpListSet.has(newCmpGen.props.component_uniquename)) {
          newCmpListSet.add(newCmpGen.props.component_uniquename);
        }
      });
      const newCmpListArray: string[] = <string[]>Array.from(newCmpListSet);
      if (newCmpListArray && newCmpListArray.length > 0) {
        let sourceStep2 = addComponentsToArray(sourceStep1, cmpListTs, newCmpListArray);

        let cmpListHashTs = getComponentsHashInfo(cmpListSource, GEN_COMPONENTS_HASH);
        let sourceStep3 = addComponentsToHash(sourceStep2, cmpListHashTs, newCmpListArray);

        LogLn(` Total old components: ${existCmpListSet.size}, added components: ${newCmpListSet.size}. `);

        host.overwrite(cmpListPath, sourceStep3);
      }
    } else {
      // If component-list.g.ts doesn't exist.
      toAddImports = false; // We generate the entire file with imports.
      return componentListGenIfNotExist(options)(host, context);
    }

    return host;
  }
}

/*
* Create new component-list.g file.
* */
function componentListGenIfNotExist(options: MagicOptionScheme): Rule {
  return (host: Tree, context: SchematicContext) => {

   console.log("creating component-list");
    const metadata = env.metadata;
    const cmpList = new Set<string>();
    const data: any = {
      app: env.app,
      cmpList: env.app.views
    };

    const componentFile: TemplateConfig = {
      template: `./templates/angular/src/app/component-list.g.ts`,
      name: 'component-list.g.ts',
      destination: metadata.paths.magicGenFolderPath,
      type: GeneratedFileTypes.TS,
      data: data
    };

    return chain([
      generate(componentFile, options)
    ])(host, context);
  }
}

/*
* Add new imports to component-list.g file.
* */
function addCmpsImportsToCmpList(options: MagicOptionScheme, module_name: string): Rule {
  return (host: Tree, context: SchematicContext) => {
    if (toAddImports) {
      const cmpsToGen = env.app.views;
      let cmpListSource = getSourceFile(host, env.metadata.paths.componentListPath(module_name));

      const recorder = host.beginUpdate(env.metadata.paths.componentListPath(module_name));

      cmpsToGen.forEach(cmp => {
        let name = cmp.props.id;
        let uniqueName = cmp.props.component_uniquename;
        let cmpPath = cmp.props.component_path;

        if (newCmpListSet.has(uniqueName)) {
          const importsChange =
            insertImport2(
              cmpListSource,
              env.metadata.paths.componentListPath(module_name),
              name,
              uniqueName,
              `./${cmpPath}${name}.component`
            ) as InsertChange;

          if (importsChange.toAdd) {
            recorder.insertLeft(importsChange.pos, importsChange.toAdd);
          }
        }
      });

      host.commitUpdate(recorder);
    }
    return host;
  }
}


// //-------------------------------------------------------------------------------------------------
// //
// //-------------------------------------------------------------------------------------------------
// function checkIfTheComponentNeedToBeRemoveFromGeneralMagicModule
//        (options: MagicOptionScheme, modulePath:string, moduleName:string, srcModuleName:string) : Rule {
//   return (tree: Tree , context : SchematicContext)=>{
//     const project   = env.project;
//
//     if (env.metadata.paths.generateCompForModuleName != "") {
//       LogLn(`      [>] checking component name [${moduleName}]
//                             from                      ${srcModuleName}]
//                             To App Module           : [${modulePath}]`);
//       const moduleSourceFile = getSourceFile(tree, modulePath);
//       if (isImported(moduleSourceFile, moduleName, srcModuleName)) {
//
//       }
//     }
//
//     return tree;
//   }
// }

/**************************
 * Helper Functions
 ***************************/

interface IComponentListParser {
  components: string[];
  endPos: number;
  isEmpty: boolean;
}

function getComponentsHashInfo(source: ts.SourceFile, variable: string): IComponentListParser {

  const rootNode = source;
  const cmpListVars: ts.Node[] = findNodes(rootNode, ts.SyntaxKind.VariableDeclaration);
  let endPos = -1;
  let isEmpty = true;

  let magicGenCmpsInfo = cmpListVars.filter(
    (node: any) => node.name.escapedText === variable);

  if (magicGenCmpsInfo && magicGenCmpsInfo.length > 0) {
    const firstPunctuation: ts.Node[] = findNodes(magicGenCmpsInfo[0], ts.SyntaxKind.FirstPunctuation);
    if (!(firstPunctuation && firstPunctuation.length > 0)) {
      throw new Error('magicGenCmpsHash variable has no open bracket ( { )');
    }
    endPos = firstPunctuation[0].end;
    const propertyAssignment = findNodes(firstPunctuation[0].parent, ts.SyntaxKind.PropertyAssignment);
    if (propertyAssignment && propertyAssignment.length > 0) {
      isEmpty = false;
    }
  }
  return {
    components: null,
    endPos,
    isEmpty
  };
}

function getComponentsArrayInfo(source: ts.SourceFile, variable: string): IComponentListParser {

  const rootNode = source;
  const cmpListVars: ts.Node[] = findNodes(rootNode, ts.SyntaxKind.VariableDeclaration);
  let endPos = -1;
  let cmpListArray;
  let isEmpty = true;
  let magicGenCmpsInfo = cmpListVars.filter(
    (node: any) => node.name.escapedText === variable);

  if (magicGenCmpsInfo && magicGenCmpsInfo.length > 0) {
    const openBracketToken: ts.Node[] = findNodes(magicGenCmpsInfo[0], ts.SyntaxKind.OpenBracketToken);
    if (!(openBracketToken && openBracketToken.length > 0)) {
      throw new Error('magicGenComponents variable has no open bracket ([)');
    }
    endPos = openBracketToken[0].end;
    const identifiers = findNodes(openBracketToken[0].parent, ts.SyntaxKind.Identifier);


    // Array with items
    if (identifiers && identifiers.length > 0) {
      isEmpty = false;
      //endPos = identifiers[identifiers.length - 1].end;
      cmpListArray = identifiers.map((i: any) => i.escapedText) as string[];
    }
    // Array with no items
    // else{
    //   endPos = openBracketToken[0].end;
    // }
  }
  return {
    components: cmpListArray,
    endPos,
    isEmpty
  };
}


function getLazyLoadCompoenentsInfo(source: ts.SourceFile, variable: string): IComponentListParser {

  const cmpListVars: ts.Node[] = findNodes(source, ts.SyntaxKind.VariableDeclaration);
  let endPos = -1;
  let cmpListArray;
  let isEmpty = true;
  let magicGenCmpsInfo = cmpListVars.filter(
    (node: any) => node.name.escapedText === variable);

  if (magicGenCmpsInfo && magicGenCmpsInfo.length > 0) {
    const openBracketToken: ts.Node[] = findNodes(magicGenCmpsInfo[0], ts.SyntaxKind.FirstPunctuation);
    if (!(openBracketToken && openBracketToken.length > 0)) {
      throw new Error('FirstPunctuation variable has no open bracket ([)');
    }
    endPos = openBracketToken[0].end;
    const identifiers = findNodes(openBracketToken[0].parent, ts.SyntaxKind.PropertyAssignment);
    const identifiers1 = findNodes(openBracketToken[0].parent, ts.SyntaxKind.Identifier);

    // Array with items
    if (identifiers && identifiers.length > 0) {
      isEmpty = false;
      cmpListArray = identifiers.map((i: any) => i.name.escapedText ).filter(id => id !== "moduleName" && id !== "modulePath");
    }
  }
  return {
    components: cmpListArray,
    endPos,
    isEmpty
  };
}

function addComponentsToArray(source: string, info: IComponentListParser, cmpsToAdd: string[]): string {
  const prefix = source.substring(0, info.endPos);
  const suffix = source.substring(info.endPos);
  let toAdd = `${cmpsToAdd}`;
  toAdd = `${toAdd.replace(/,/g, ",\n\t")}`;

  return info.isEmpty ?
`${prefix}
\t${toAdd} ${suffix}` :
`${prefix}
\t${toAdd},${suffix}` ;
}

function addComponentsToHash(source: string, info: IComponentListParser, cmpsToAdd: string[]): string {
  const prefix = source.substring(0, info.endPos);
  const suffix = source.substring(info.endPos);
  const cmpHashStr = EJS.render(`
      <% components.forEach(componentUniqueName => { %>
        <%= componentUniqueName %>:<%- componentUniqueName %>,
      <%})%>`, {components: cmpsToAdd});
  let toAdd = `${cmpHashStr.replace(/\n/g, '').replace(/,/g, ",\n")}`;
  // update source
  return `${prefix} ${toAdd} ${suffix}`;
  /*return info.isEmpty ?
    `${prefix} ${delLastComma(cmpHashStr)} ${suffix}` :
    `${prefix} ${delLastComma(cmpHashStr)}, ${suffix}` ;*/
}

// add the lazy loaded components to map
function addComponentsToLazyLoad(source: string, info: IComponentListParser, cmpsToAdd: View[], module_name: string): string {
  const prefix = source.substring(0, info.endPos);
  const suffix = source.substring(info.endPos);

  // get the module path
  let currentModulePath = env.metadata.paths.magicGenLibModulePath(module_name);

  const cmpHashStr = EJS.render(`
<%{components.forEach(view =>{-%>
<%=view.props.component_uniquename %> : { moduleName : 'Magic<%=moduleName%>Module'},
<%})}-%>`, {components: cmpsToAdd, moduleName: module_name, modulePath: currentModulePath});

  let toAdd = `${cmpHashStr.replace(/\n/g, '').replace(/,/g, ",\n")}`;
  // update source
  return `${prefix} ${toAdd} ${suffix}`;
}

function delLastComma(source: string) {
  let index = source.lastIndexOf(',');
  return source.substring(0, index);
}



