import {
  apply, branchAndMerge,
  chain,
  MergeStrategy, mergeWith,
  move,
  Rule,
  SchematicContext, SchematicsException,
  Tree,
  url
} from "@angular-devkit/schematics";
import { MagicOptionScheme } from "./magic-option.scheme";
import {env, GEN_FOLDER, GEN_LAZY_LOAD_MODULES_ARRAY} from "../env";
import { LogLn } from "../Util";
import { Route, RouteTable, webModuleItem } from "../../../types/interfaces/app";
import {addModuleImportToModule, getSourceFile} from "../../utils/ast";
import { ModuleInfo } from "../../../types/interfaces/env.interface";

import {findNodes} from "../../utils/devkit-utils/ast-utils";
import * as ts from "typescript";
//-------------------------------------------------------------------------------------------------
// init generate module into env.modulesGen
//-------------------------------------------------------------------------------------------------
export function initGenModules(options: MagicOptionScheme, generateViaCLI:boolean = false): Rule {
  return (tree: Tree, context: SchematicContext) => {
    let metadata = env.metadata;
    let programList = env.programList;
    env.modulesGen = [];
    let web_modules:webModuleItem [] = metadata.config.web_modules;

    if(generateViaCLI){
      let programListString = JSON.stringify(programList.programListContents);

      // Validate whether the component is valid
      if(programListString.includes(options.component)){
        let moduleInfo: ModuleInfo = new ModuleInfo(options.module, options.loadOnDemand);
        env.modulesGen.push(moduleInfo)
      }else{
        LogLn(`Invalid component ${options.component} to generate.`);
      }
    }else{
      // Generate only one module from --module=xxxx
      if (options.module) {
        let moduleInfo : ModuleInfo  = new ModuleInfo(options.module, options.loadOnDemand);
        env.modulesGen.push(moduleInfo)
      } else {

        // Generate list of components from config.json, properties web_modules.module_name
        for (let modulePath of web_modules) {
          //LogLn(`   Web Module name:   [${modulePath.module_name}]   IsLoadOnDemand?   [${modulePath.load_on_demand}]`);
          let moduleInfo : ModuleInfo  = new ModuleInfo(modulePath.module_name, modulePath.load_on_demand);
          env.modulesGen.push(moduleInfo);
        }
      }
    }
    if(!generateViaCLI)
      LogLn(`   Number of modules to be check for generate: ${web_modules.length}`);
    return tree;
  };
}

//-------------------------------------------------------------------------------------------------
// overwrite the web module file name with the new content
//-------------------------------------------------------------------------------------------------
export function overwriteWebModuleFileName(options: any, moduleNameFileName:string, content:string): Rule {
  return (tree: Tree, _context: SchematicContext) => {

    if(!tree.exists(moduleNameFileName)){
      LogLn(`      [>Error] File cannot be overwrite, The file is not exist  !!! : ${moduleNameFileName}`);
    }

    tree.overwrite(moduleNameFileName, content);
    return tree;
  };
}

//-------------------------------------------------------------------------------------------------
// update the  the web module file name with the new class name
//-------------------------------------------------------------------------------------------------
export function updateWebModuleFileWithNewClassName(options: any, webModuleName:string): Rule {
  return (tree: Tree, context: SchematicContext) => {
    let rules:Rule[] = [];

    const origModuleFileName =  getWebModuleFullFileName(webModuleName)

    // create new class name for "web Module Name"
    let newClassModuleName = getWebModuleClass(webModuleName );

    //read the exist module file name
    const text = tree.read(origModuleFileName);

    if (text === null) {
      throw new SchematicsException(`File ${origModuleFileName} does not exist.`);
    }
    let sourceText = text.toString('utf-8');

    if (sourceText !== "") {
      let newsourceText: string = sourceText.replace(`MagicGenLibModule`, newClassModuleName);
      rules.push(overwriteWebModuleFileName(options, origModuleFileName, newsourceText));

    }
    return chain(rules)(tree, context);
  };
}

//-------------------------------------------------------------------------------------------------
// copy the two default files into the web module path
//-------------------------------------------------------------------------------------------------
function copyDefaultMagicFilesToWebModule(options:MagicOptionScheme, webModuleName:string): Rule {
  return (tree: Tree,context:SchematicContext) => {
    const webModulePath = getDestinationModulePathByWebModuleName(webModuleName);

    return chain([
      branchAndMerge(chain([
        mergeWith(
          apply(url(`./../mg-add/files/src/app/magic/`), [

            move(webModulePath)
          ]), MergeStrategy.AllowCreationConflict)
      ]))
    ])(tree,context);
  }
}


//-------------------------------------------------------------------------------------------------
// delete LazyLoadService for non root modules
//-------------------------------------------------------------------------------------------------
function deleteLazyLoadService( webModuleName:string): Rule {
  return (tree: Tree) => {
    const lazyLoaderServicePath = getDestinationModulePathByWebModuleName(webModuleName) + '/lazy-loader.service.ts';

    if(webModuleName != '') {
      if (!tree.exists(lazyLoaderServicePath)) {
        LogLn(`      [>Error] File cannot be overwrite, The file is not exist  !!! :lazyLoaderServicePath`);
      }

      tree.delete(lazyLoaderServicePath);
      return tree;
    }
  }
}


//-------------------------------------------------------------------------------------------------
// get destination  path according to the send webModuleName
//-------------------------------------------------------------------------------------------------
export function getDestinationModulePathByWebModuleName(webModuleName:string):string {
  const newDestModulePath = `${env.metadata.paths.rootMagicGenFolder}/${GEN_FOLDER}/${webModuleName}`;
  return newDestModulePath;
}

//-------------------------------------------------------------------------------------------------
// get web module full file name according to the send webModuleName
//-------------------------------------------------------------------------------------------------
function getWebModuleFullFileName(webModuleName:string):string {
// calc the path of new the module
  const newDestModulePath = getDestinationModulePathByWebModuleName(webModuleName);
  return `${newDestModulePath}/magic.gen.lib.module.ts`;
}

//-------------------------------------------------------------------------------------------------
// get web module class according to the send webModuleName
//-------------------------------------------------------------------------------------------------
function getWebModuleClass(webModuleName:string):string {
  return `Magic` + webModuleName + `Module`;
}


//-------------------------------------------------------------------------------------------------
// add magic module to app module
//-------------------------------------------------------------------------------------------------
function addMagicModuleToAppModule(options: MagicOptionScheme, modulePath:string, moduleName:string, srcModuleName:string) : Rule {
  return (tree: Tree , context : SchematicContext)=>{
    const project   = env.project;

    LogLn(`         [>] add New Web Module    :[${moduleName}] `);
    LogLn(`         [>]         from          :[${srcModuleName}]`);
    LogLn(`         [>]         To App Module :[${modulePath}]`);
    addModuleImportToModule(
      tree,
      modulePath,     // for now we will update env.metadata.paths.magicGenLibModule
      moduleName,     // 'MagicXXXModule',
      srcModuleName,  // `./XXX/magic.gen.lib.module`
    );
    return tree;
  }
}

//-------------------------------------------------------------------------------------------------
// create module
//-------------------------------------------------------------------------------------------------
export function checkAndCreateModule(options: MagicOptionScheme, webModuleName:string, loadOnDemand: boolean ): Rule {
  return (tree: Tree, context: SchematicContext) => {

    let rules: Rule[] = [];
    const webModuleFullFileName = getWebModuleFullFileName(webModuleName);
    // only if not exist create the new 2 files magic.gen.lib.module.ts & component-list.g.ts
    if (!tree.exists(webModuleFullFileName)) {
      rules.push(copyDefaultMagicFilesToWebModule(options, webModuleName));
      rules.push(updateWebModuleFileWithNewClassName(tree, webModuleName));
      rules.push(deleteLazyLoadService(webModuleName));
      // for NON load on demand we need to update magic root module
      if (!loadOnDemand) {
        const newDestModulePath = `./${webModuleName}/magic.gen.lib.module`;
        const webModuleClass = getWebModuleClass(webModuleName);
        rules.push(addMagicModuleToAppModule(options, env.metadata.paths.magicGenLibModule, webModuleClass, newDestModulePath));
      }
      return chain(rules)(tree, context);
    }
    ;
  }
}

//-------------------------------------------------------------------------------------------------
// check if module is exist, only if not exist create the web module
//-------------------------------------------------------------------------------------------------
export function generateWebModule(options: MagicOptionScheme, webModuleName:string, loadOnDemand:boolean ): Rule {
  return (tree: Tree, context: SchematicContext) => {

    let rules:Rule[] = [];

    LogLn(`  [>]Handling Web Module named:   [${webModuleName}]   isLoadOnDemand:   [${loadOnDemand}]`);
    rules.push(checkAndCreateModule(options, webModuleName, loadOnDemand));
    rules.push(checkAndCreateRouteForModule(options, webModuleName, loadOnDemand));

    return chain(rules)(tree,context);
  };
}


//-------------------------------------------------------------------------------------------------
// generate all the modules in  env.modulesGen
//-------------------------------------------------------------------------------------------------
export function generateWebModules(options: MagicOptionScheme, generateViaCLI:boolean = false): Rule {
  return (tree: Tree, context: SchematicContext) => {

    let rules:Rule[] = [];

    for (let webModuleInfo of env.modulesGen){
      if (webModuleInfo.moduleName !== undefined && webModuleInfo.moduleName !== "") {
        if (generateViaCLI) {
          if (webModuleInfo.moduleName === options.module) {
            rules.push(generateWebModule(options, webModuleInfo.moduleName, webModuleInfo.loadOnDemand));
          }
        }else{
          rules.push(generateWebModule(options, webModuleInfo.moduleName, webModuleInfo.loadOnDemand));
        }
      }
    }

    return chain(rules)(tree,context);
  };
}



//-------------------------------------------------------------------------------------------------
// check if module is exist, only if not exist create the web module
//-------------------------------------------------------------------------------------------------
export function checkAndCreateRouteForModule(options: MagicOptionScheme, webModuleName:string, loadOnDemand :boolean): Rule {
  return (tree: Tree, context: SchematicContext) => {

    let rules:Rule[] = [];

    if (loadOnDemand) {
      const newDestRoutePathForModule = env.metadata.paths.getRoutePathForWebModule(webModuleName);

      // only if file not exist then copy the default route
      if (!tree.exists(newDestRoutePathForModule)) {
        rules.push(createRoute(options, webModuleName));
      }
      else{
        // if file exist then only update the magic.gen.lib.module.ts
        rules.push(addRouteFileToModuleFile(options, webModuleName));
      }
    }
    return chain(rules)(tree,context);
  };
}



//-------------------------------------------------------------------------------------------------
// create route for module
//-------------------------------------------------------------------------------------------------
export function createRoute(options: MagicOptionScheme, webModuleName:string): Rule {
  return (tree: Tree, context: SchematicContext) => {

    let rules:Rule[] = [];

    rules.push(copyDefaultRouteMagicFilesToWebModule(options, webModuleName));
    // --------------------------------------------------------
    // // generate the route for module /////
    // let routeItem : RouteTable = new RouteTable();
    // routeItem.module_name = webModuleName;
    //
    // let route:Route = new Route();
    // route.ModuleName = webModuleName;
    // route.OutletName="";
    // route.RouteName="";
    // route.children=null;
    //
    // routeItem.routesArray = [route];
    // rules.push(genRouteFile(options, routeItem ));
    // --------------------------------------------------------
    // update the new MagicRoutingModule in file magic.gen.lib.module.ts of the module


    rules.push(addRouteFileToModuleFile(options, webModuleName));

    return chain(rules)(tree,context);
  };
}

//-------------------------------------------------------------------------------------------------
// create route for module
//-------------------------------------------------------------------------------------------------
export function addRouteFileToModuleFile(options: MagicOptionScheme, webModuleName:string): Rule {
  return (tree: Tree, context: SchematicContext) => {

    let rules:Rule[] = [];

    const modulePath =  getWebModuleFullFileName(webModuleName); // get the magic.gen.lib.module.ts according to the module name
    const moduleName = `MagicRoutingModule`;//for now it will be the same for all modules ... getWebModuleClass(webModuleName);
    const srcModuleName = `./app.routes`;

    rules.push(addMagicModuleToAppModule(options, modulePath, moduleName, srcModuleName));

    return chain(rules)(tree,context);
  };
}
//-------------------------------------------------------------------------------------------------
// copy the two default files into the web module path
//-------------------------------------------------------------------------------------------------
function copyDefaultRouteMagicFilesToWebModule(options:MagicOptionScheme, webModuleName:string): Rule {
  return (tree: Tree,context:SchematicContext) => {
    const routeFileForWebModulePath = env.metadata.paths.getRoutePathForWebModule(webModuleName);


    if(tree.exists(routeFileForWebModulePath))
    {
      let rules:Rule[] = [];
      return chain(rules)(tree,context);
    }
    else
    {
      const webModulePath = getDestinationModulePathByWebModuleName(webModuleName);
      LogLn(`[ >] copy default Route file to webModule [${webModuleName}]`)
      return chain([
        branchAndMerge(chain([
          mergeWith(
            apply(url(`./../mg-add/webModule/`), [

              move(webModulePath)
            ]), MergeStrategy.AllowCreationConflict)
        ]))
      ])(tree,context);
    }
  }
}
