import {
  Rule,
  chain,
  Tree,
  SchematicContext
} from "@angular-devkit/schematics";
import {NodePackageInstallTask} from "@angular-devkit/schematics/tasks";

import {addPackageToPackageJson} from './../utils/package';
import { dependencies, devDependencies, env} from "../magic-utils/env";
import { addImportTSModule, copyMagicFiles} from "../magic-utils/utils";
import { initMagicMetadata} from "../magic-utils/rules/init-magic-metadata.rule";
import { addModuleImportToRootModule } from "../utils/ast";
import { LogLn } from "../magic-utils/Util";
import { generate } from "../magic-utils/rules/generate.rule";
import { GeneratedFileTypes } from "../../types/enums/generated-file-types.enum";
import { MagicOptionScheme } from "../magic-utils/rules/magic-option.scheme";
import {addProviderToModule, insertImport} from "../utils/devkit-utils/ast-utils";
import * as ts from 'typescript';
import {InsertChange} from "../utils/devkit-utils/change";
export function mgAdd(options:MagicOptionScheme): Rule {
  return chain([
    initMagicMetadata                 (options),
    copyMagicFiles                    (options),

    // handel app.modules.ts
    addMagicModuleToAppModule         (options),

    // handel Angular.json
    updateAngularJsonWithMagicStyle   (options),

    updateTsConfigAppJson  (options),

    addLazyLoadProvider(),
    // handel app.component.html
    //updateAppComponentHtmlFiles       (options),

    //handel package.Json
    //addMagicDependenciesToPackageJson (options),
  ]);
}

function updateAppComponentHtmlFiles(options:MagicOptionScheme): Rule {
  return generate({
    template    : "./templates/angular/src/app/app.component.html.ejs",
    name        : "app.component.html",
    destination : env.metadata.paths.rootMagicGenFolder,
    type        :GeneratedFileTypes.HTML,
    data        : {
      skipHelp : options.skipHelp
    }
  },options)
}
function updateAngularJsonWithMagicStyle(options:MagicOptionScheme): Rule {
  return (host: Tree, context: SchematicContext) => {
    const project   = env.project;
    let   isDirty   = false;

    let fullPathmagicStylePath = env.metadata.paths.magicStylePath;
    if (!project.architect.test.options.styles.includes(fullPathmagicStylePath)) {
      context.logger.info(`  [>]add file ${fullPathmagicStylePath} to test`);
      project.architect.test.options.styles.push(fullPathmagicStylePath);
      isDirty = true;
    }

    if (!project.architect.build.options.styles.includes(fullPathmagicStylePath)) {
      context.logger.info(`  [>]add file ${fullPathmagicStylePath} to build`);
      project.architect.build.options.styles.push(fullPathmagicStylePath);
      isDirty = true;
    }

    if(isDirty){
      let fullPath = env.metadata.paths.angularJsonPath;
      host.overwrite(fullPath, JSON.stringify(env.workspace, null, 2));
    }

    return host;
  };
}

// Angular 10 , does not include .ts file for compilation
// So to avoid compilation error , add *.module.ts in includes section
function updateTsConfigAppJson (option: MagicOptionScheme): Rule{
  return (host: Tree, context: SchematicContext) => {

    const tsConfigPath = env.metadata.paths.tsConfigAppJsonPath;
    const tsConfig = host.read(tsConfigPath)!.toString("utf-8");

    // extract the comment
    let comment = tsConfig.substring(tsConfig.indexOf('/*'), tsConfig.indexOf('*/') + 2);

    // replace the comment as parsing json with comment throws exception
    var config = JSON.parse(tsConfig.replace(comment, ''));

    //Add module files for compilation
    if(!config['include'].includes('src/**/*module.ts'))
      config['include'].push('src/**/*module.ts');

    // append the comment again and write the file
    host.overwrite(tsConfigPath, comment + "\n" + JSON.stringify(config, null , 2));
    return host;
  }
}

function addMagicDependenciesToPackageJson(option:MagicOptionScheme): Rule {
	return (host: Tree, context: SchematicContext) => {
	  if(!option.skipPackageJson){
      for (let dep of dependencies){
        LogLn(`[>] Add dependencies ${dep.name}`);
        addPackageToPackageJson(host, 'dependencies', dep.name, dep.version);
      }
     /*
     // done in pre-install step
     for (let dep of devDependencies){
        LogLn(`[>] Add devDependencies ${dep.name}`);
        addPackageToPackageJson(host, 'devDependencies', dep.name, dep.version);
      }
      */
    }

		if(!option.skipInstall){
      LogLn(`[>] Install dependencies.`);
      context.addTask(new NodePackageInstallTask());
    }else{
      LogLn(`skipIstalls ${dependencies.length} dependencies`);
      LogLn(`skipIstalls ${devDependencies.length} devDependencies`);
    }

		return host;
	};
}
function addMagicModuleToAppModule(options: MagicOptionScheme) : Rule {
  return (host: Tree , context : SchematicContext)=>{
    const workspace = env.workspace;
    const project   = env.project;

    addModuleImportToRootModule(
      host,
      'BrowserAnimationsModule',
      '@angular/platform-browser/animations',
      project);

    addModuleImportToRootModule(
      host,
      'ReactiveFormsModule',
      '@angular/forms',
      project);


    addModuleImportToRootModule(
      host,
      'MagicModule',
      "@magic-xpa/angular",
      project);

    addModuleImportToRootModule(
      host,
      'MagicAngularMaterialModule',
      "@magic-xpa/angular-material-core",
      project);

    addModuleImportToRootModule(
      host,
      'MagicGenLibModule',
      "./magic/magic.gen.lib.module",
      project);

    addModuleImportToRootModule(
      host,
      'MagicRoutingModule',
      "./app.routes",
      project);


  }
}

function addLazyLoadProvider(): Rule {
  return (host: Tree) => {
    const appmodule = env.metadata.paths.rootMagicGenFolder + '/app.module.ts';

    const lazyLoaderService = 'LazyLoaderService';
    const lazyLoaderServicePath = './magic/lazy-loader.service';

    let text = host.read(appmodule);
    let sourceText = text.toString('utf-8');
    let source = ts.createSourceFile(appmodule, sourceText, ts.ScriptTarget.Latest, true);
    let providerChanges = addProviderToModule(source, appmodule, '{ provide: MagicLazyLoaderService, useClass: LazyLoaderService }', lazyLoaderServicePath)

    const providerRecorder = host.beginUpdate(appmodule);
    for (const change of providerChanges) {
      if (change instanceof InsertChange) {
        if (!change.toAdd.includes('import'))  // skip import change
        providerRecorder.insertLeft(change.pos, change.toAdd);
      }
    }
    host.commitUpdate(providerRecorder);

    const lazyLoaderServiceImport = insertImport(source, appmodule, lazyLoaderService, lazyLoaderServicePath, false);
    const lazyLoaderServiceChangesRecorder = host.beginUpdate(appmodule);
    if (lazyLoaderServiceImport instanceof InsertChange)
      lazyLoaderServiceChangesRecorder.insertLeft(lazyLoaderServiceImport.pos, lazyLoaderServiceImport.toAdd);
    host.commitUpdate(lazyLoaderServiceChangesRecorder);

    // add import for MagicLazyLoaderService
    const magicLazyLoaderServiceImportChanges: InsertChange = <InsertChange>insertImport(source, appmodule, 'MagicLazyLoaderService', '@magic-xpa/angular', false);
    const magicImportChangesRecorder = host.beginUpdate(appmodule);
    magicImportChangesRecorder.insertLeft(magicLazyLoaderServiceImportChanges.pos, magicLazyLoaderServiceImportChanges.toAdd)
    host.commitUpdate(magicImportChangesRecorder);
  }
}

