import { ApplicationException, DateTime, NString, RefParam } from '@magic-xpa/mscorelib';
import { DateTimeUtils, InternalInterface, Logger, Misc, OSEnvironment, StorageAttribute, XMLConstants } from '@magic-xpa/utils';
import { Commands, CommandType, DisplayConvertor, Events, GuiDataCollection, GuiEnvironment, Manager, PIC, PropInterface, RuntimeContextBase, Styles } from '@magic-xpa/gui';
import { EventsManager } from './event/EventsManager';
import { GuiEventsProcessor } from './GuiEventsProcessor';
import { Environment } from './env/Environment';
import { LanguageData } from './env/LanguageData';
import { TasksTable } from './tasks/TasksTable';
import { Task } from './tasks/Task';
import { MGDataCollection } from './tasks/MGDataCollection';
import { UniqueIDUtils } from './util/UniqueIDUtils';
import { HttpManager } from './http/HttpManager';
import { CommandsTable } from './CommandsTable';
import { RemoteCommandsProcessor } from './remote/RemoteCommandsProcessor';
import { CommandsProcessorBase_SendingInstruction } from './CommandsProcessorBase';
import { CommandsProcessorManager } from './CommandsProcessorManager';
import { MGData } from './tasks/MGData';
import { ServerError } from './remote/ServerError';
import { HttpClientEvents } from './http/client/HttpClientEvents';
import { CommandFactory } from './commands/ClientToServer/CommandFactory';
import { CurrentClientVersion } from './CurrentClientVersion';
import { LastFocusedManager } from './rt/LastFocusedManager';
import { EnvParamsTable } from './env/EnvVariablesTable';
import { GlobalParams } from './exp/GlobalParamsTable';
import { OpenURLCommand } from './commands/ServerToClient/OpenURLCommand';
import { ServerConfig } from './ServerConfig';
import { AccessHelper } from './AccessHelper';
import { GUIManager } from "./GUIManager";
export class ClientManager {
    static get Instance() {
        if (ClientManager._instance === null)
            ClientManager._instance = new ClientManager();
        return ClientManager._instance;
    }
    AddEvent(guiEvent) {
        let task = MGDataCollection.Instance.GetTaskByID(guiEvent.TaskID);
        let lineIdx = +guiEvent.Line;
        let dvRowIdx = 0;
        if (task != null) {
            let control = null;
            if (guiEvent.ControlName !== null)
                control = task.getForm().GetCtrl(guiEvent.ControlName);
            if (control != null) {
                let dv = task.DataView;
                if (control.IsTableChild) {
                    lineIdx -= dv.RecordsBeforeCurrentView;
                }
                switch (guiEvent.EventType) {
                    case "columnSort":
                        if (control.isColumnSortable()) {
                            Events.OnColumnClick(control, guiEvent.Line, guiEvent.ControlName);
                        }
                        break;
                    case "mousedown":
                        if (control.isButton()) {
                            this._buttonIsClicked = true;
                        }
                        else if (!control.isSubform())
                            Events.GetRuntimeContext(Manager.GetCurrentContextID()).LastClickedCtrlName = control.getName();
                        break;
                    case "click":
                        if (control.isButton() || (control.ConnectedControl && guiEvent.fromButton)) {
                            Events.OnSelection("", control, lineIdx, true);
                            Events.GetRuntimeContext(Manager.GetCurrentContextID()).LastClickedCtrlName = control.getName();
                            this._buttonIsClicked = false;
                        }
                        else {
                            Events.OnFocus(control, lineIdx, true, false);
                            let produceClick = control.isButton() || control.isTextControl() || control.isLabel() || control.isTabControl() || control.isTableControl() || control.isComboBox() || control.isImageControl();
                            Events.OnMouseDown(control.getForm(), control, null, true, lineIdx, false, produceClick);
                        }
                        break;
                    case "focus":
                        if (this._buttonIsClicked)
                            return;
                        Events.OnFocus(control, lineIdx, true, false);
                        break;
                    case "dblclick":
                        Events.OnDblClick(control, lineIdx);
                        break;
                    case "selectionchanged":
                        Events.OnSelection(guiEvent.Value, control, lineIdx, true);
                        break;
                    case "keydown":
                        Events.OnKeyDown(task.getForm(), control, guiEvent.modifiers, guiEvent.keyCode, 0, 0, '', false, '', true);
                        break;
                    case "setexternalvalue":
                        EventsManager.Instance.AddExternalValueEvent(control, guiEvent.externalValue, guiEvent.param);
                        break;
                    default:
                        break;
                }
            }
            else {
                switch (guiEvent.EventType) {
                    case "focusOnNonMagicControl":
                        control = LastFocusedManager.getLastFocusedControl();
                        if (control !== null) {
                            Events.OnFocusOnNonMagicControl(control);
                        }
                        break;
                    case "click":
                        if (guiEvent.ControlName === "magicRow") {
                            dvRowIdx = guiEvent.Line - task.DataView.RecordsBeforeCurrentView;
                            EventsManager.Instance.addGuiTriggeredEventWithTaskAndCodeAndLine(task, InternalInterface.MG_ACT_REC_PREFIX, dvRowIdx);
                        }
                        break;
                    case "navigateInTable":
                        if (guiEvent.ControlName === "magicRow")
                            this.handleNavigateInRowEvent(task, guiEvent.Value);
                        break;
                    case "close":
                        if (task.IsRoute())
                            EventsManager.Instance.AddRouterClosedEvent(task);
                        else
                            Events.CloseFormEvent(task.getForm());
                        break;
                    case "dispose":
                        MGDataCollection.Instance.deleteMGDataTree(task.getMgdID());
                        break;
                    case "resize":
                        Events.TableResizeEvent(task.getForm().getTableCtrl(), guiEvent.PageSize);
                    case "getRows":
                        dvRowIdx = lineIdx - task.DataView.RecordsBeforeCurrentView;
                        let lastFocusedVal = { guiMgControl: task.getForm().getTableCtrl(), Line: dvRowIdx, Val: "" };
                        Events.OnGetRowsData(task.getForm().getTableCtrl(), dvRowIdx, true, lastFocusedVal);
                        break;
                    case "RouterNavigate":
                        EventsManager.Instance.AddRouteEvent(task, InternalInterface.MG_ACT_WEBCLIENT_ROUTE, guiEvent.RouterPath, guiEvent.RouterOutletName, guiEvent.RouterParams);
                        break;
                    case "setexternalvalue":
                        Logger.Instance.WriteErrorToLog(NString.Format("Unable to find {0}: Trying to set value for a non-existing control.", guiEvent.ControlName));
                        break;
                    default:
                        break;
                }
            }
        }
        else
            switch (guiEvent.EventType) {
                case "confirmationClosed":
                    this.ConfirmationClosed(guiEvent.keyCode);
                    break;
            }
    }
    handleNavigateInRowEvent(task, evt) {
        let internalEvt = InternalInterface.MG_ACT_NONE;
        evt = evt.toLowerCase();
        switch (evt) {
            case "begin-page":
                internalEvt = InternalInterface.MG_ACT_TBL_BEGPAGE;
                break;
            case "previous-page":
                internalEvt = InternalInterface.MG_ACT_TBL_PRVPAGE;
                break;
            case "end-page":
                internalEvt = InternalInterface.MG_ACT_TBL_ENDPAGE;
                break;
            case "next-page":
                internalEvt = InternalInterface.MG_ACT_TBL_NXTPAGE;
                break;
            case "previous-line":
                internalEvt = InternalInterface.MG_ACT_TBL_PRVLINE;
                break;
            case "next-line":
                internalEvt = InternalInterface.MG_ACT_TBL_NXTLINE;
                break;
            case "begin-table":
                internalEvt = InternalInterface.MG_ACT_TBL_BEGTBL;
                break;
            case "end-table":
                internalEvt = InternalInterface.MG_ACT_TBL_ENDTBL;
                break;
        }
        if (internalEvt != InternalInterface.MG_ACT_NONE)
            EventsManager.Instance.addGuiTriggeredEventWithTaskAndCode(task, internalEvt);
    }
    ConfirmationClosed(res) {
        Commands.ResolvePromise(res);
    }
    async WorkThreadExecution() {
        if (Logger.Instance.ShouldLog()) {
            Logger.Instance.WriteToLog("", true);
            Logger.Instance.WriteToLog("Started on " + DateTimeUtils.ToString(DateTime.Now, XMLConstants.ERROR_LOG_DATE_FORMAT, Logger.Instance), true);
            this.WriteExecutionPropertiesToLog();
        }
        try {
            if (await CommandsProcessorManager.StartSession()) {
                let mgDataTab = MGDataCollection.Instance;
                let startupMgData = mgDataTab.StartupMgData;
                mgDataTab.currMgdID = startupMgData.GetId();
                let nonInteractiveTask = null;
                if (!startupMgData.IsAborting)
                    nonInteractiveTask = await Task.StartProgram(false, false);
                if (nonInteractiveTask === null)
                    await EventsManager.Instance.EventsLoop(startupMgData);
                else
                    await EventsManager.Instance.NonInteractiveEventsLoop(startupMgData, nonInteractiveTask);
                EventsManager.Instance.setEndOfWork(true);
            }
        }
        catch (ex) {
            if (ex instanceof ApplicationException) {
                let isNoResultError = false;
                if (ex instanceof ServerError && ex.GetCode() === ServerError.INF_NO_RESULT)
                    isNoResultError = true;
                if (!isNoResultError)
                    await EventsManager.Instance.ProcessAbortingError(ex);
                EventsManager.Instance.setEndOfWork(true);
            }
            else
                throw ex;
        }
        finally {
            await EventsManager.Instance.CheckAndShowSpinner(false);
            Commands.addGlobalCommand(CommandType.EXIT_MAGIC);
            Commands.beginInvoke();
        }
    }
    WriteExecutionPropertiesToLog() {
        if (Logger.Instance.ShouldLogServerRelatedMessages()) {
            Logger.Instance.WriteToLog("-----------------------------------------------------------------------------", true);
            Logger.Instance.WriteToLog("Execution Properties:", true);
            let propsString = JSON.stringify(ServerConfig.Instance);
            Logger.Instance.WriteToLog(propsString, true);
            Logger.Instance.WriteToLog("-----------------------------------------------------------------------------", true);
        }
    }
    constructor() {
        this._globalUniqueSessionId = null;
        this._buttonIsClicked = false;
        RuntimeContextBase.Instance.Init(RemoteCommandsProcessor.RC_NO_CONTEXT_ID);
        LastFocusedManager.Instance.Init(MGDataCollection.Instance);
        LastFocusedManager.Instance.LastActionTime = Misc.getSystemMilliseconds();
        RemoteCommandsProcessor.ShouldScrambleAndUnscrambleMessages = true;
        this._globalUniqueSessionId = UniqueIDUtils.GetUniqueMachineID() + "_";
        this.RegisterDelegates();
    }
    InitGuiManager() {
        Manager.EventsManager = EventsManager.Instance;
        Manager.DefaultServerName = ServerConfig.Instance.getServer();
        Manager.DefaultProtocol = ServerConfig.Instance.getProtocol();
        GuiEnvironment.Environment = Environment.Instance;
        GuiDataCollection.MGDataTable = MGDataCollection.Instance;
        RuntimeContextBase.noContextId = RemoteCommandsProcessor.RC_NO_CONTEXT_ID;
    }
    InitCircularReferencesBypasses() {
        TasksTable.createNewtask = Task.CreateTask;
        OpenURLCommand.createNewCommandsTable = CommandsTable.createCommandsTable;
        AccessHelper.eventsManager = EventsManager.Instance;
        AccessHelper.mgDataTable = MGDataCollection.Instance;
        AccessHelper.envParamsTable = EnvParamsTable.Instance;
        AccessHelper.globalParams = GlobalParams.Instance;
        AccessHelper.languageData = LanguageData.Instance;
        AccessHelper.environment = Environment.Instance;
    }
    static Main(httpClient) {
        ClientManager.StartExecution(httpClient);
        Logger.Instance.Flush();
    }
    static async StopExecution() {
        try {
            HttpManager.GetInstance().IsAbortingMagicEngine = true;
            MGDataCollection.Instance.StartupMgData.getFirstTask().ClosingFormUI = true;
            await MGDataCollection.Instance.StartupMgData.getFirstTask().endTask(true, false, false);
        }
        catch (ex3) {
            if (ex3 instanceof ApplicationException) {
                Logger.Instance.WriteExceptionToLog(ex3);
            }
            else
                throw ex3;
        }
    }
    static async StartExecution(httpClient) {
        HttpManager.setNativeHttpClient(httpClient);
        let loaded = await ClientManager.Instance.LoadExecutionProps();
        if (loaded) {
            new GuiEventsProcessor();
            ClientManager.Instance.InitGuiManager();
            ClientManager.Instance.InitCircularReferencesBypasses();
            let versionString = CurrentClientVersion === '0.0.0.0' ? "Development mode" : "Version : " + CurrentClientVersion;
            console.log("Magic Web Client. " + versionString);
            HttpManager.GetInstance().HttpCommunicationTimeoutMS = ServerConfig.Instance.GetFirstHttpRequestTimeout();
            EventsManager.Instance.Init();
            Logger.Instance.Initialize(RemoteCommandsProcessor.GetInstance().parseLogLevel(ServerConfig.Instance.getInternalLogLevel()), false);
            let mgd = new MGData(new CommandsTable(), new CommandsTable(), 0, null, false);
            await MGDataCollection.Instance.addMGData(mgd, 0, true);
            await ClientManager.Instance.WorkThreadExecution();
            if (Logger.Instance.ShouldLog()) {
                Logger.Instance.WriteToLog(OSEnvironment.EolSeq + "Ended on " + DateTimeUtils.ToString(DateTime.Now, XMLConstants.ERROR_LOG_DATE_FORMAT, Logger.Instance) + OSEnvironment.EolSeq + OSEnvironment.EolSeq, false);
            }
        }
        else {
            await GUIManager.Instance.MessageBox("Error", "Failed to load server configuration file (" + this._executionPropertiesFileName + ")", Styles.MSGBOX_ICON_ERROR | Styles.MSGBOX_BUTTON_OK);
        }
    }
    async LoadExecutionProps() {
        let response = "";
        let succeeded = true;
        try {
            let isError = new RefParam(false);
            response = await HttpManager.GetInstance().GetContent(ClientManager._executionPropertiesFileName, null, false, isError);
            ServerConfig.Instance.Init(JSON.parse(response));
        }
        catch (ex) {
            succeeded = false;
        }
        return succeeded;
    }
    GetGlobalUniqueSessionID() {
        return this._globalUniqueSessionId;
    }
    static GetControlPictureMask(taskId, controlName) {
        let pic = null;
        let task = MGDataCollection.Instance.GetTaskByID(taskId);
        let control = null;
        if (controlName !== null && task != null) {
            control = task.getForm().GetCtrl(controlName);
            pic = control.getPIC();
        }
        return pic;
    }
    static GetFormattedValue(taskId, controlName, value, rowId) {
        let pic = null;
        let task = MGDataCollection.Instance.GetTaskByID(taskId);
        let control = null;
        let newValue = value;
        if (controlName !== null && task !== null)
            control = task.getForm().GetCtrl(controlName);
        if (control != null && control.DataType == StorageAttribute.NUMERIC) {
            if (control.isRepeatable() && control.getProp(PropInterface.PROP_TYPE_FORMAT).isExpression()) {
                let picval = control.getProp(PropInterface.PROP_TYPE_FORMAT).getPrevValue(+rowId);
                pic = new PIC(picval, StorageAttribute.NUMERIC, control.getForm().getTask().getCompIdx());
            }
            else
                pic = control.getPIC();
            let dispValue = String(value);
            let newPic = pic.getUnformattedNumericPic();
            let num = DisplayConvertor.Instance.toNum(dispValue, newPic, 0);
            newValue = DisplayConvertor.Instance.fromNum(num, pic);
        }
        return newValue;
    }
    static GetRangedValue(taskId, controlName, value) {
        let task = MGDataCollection.Instance.GetTaskByID(taskId);
        let control = null;
        if (controlName !== null) {
            control = task.getForm().GetCtrl(controlName);
            return control.getRangedValue(value);
        }
        return null;
    }
    static ValidateControlValue(taskId, controlName, value) {
        let task = MGDataCollection.Instance.GetTaskByID(taskId);
        let control = null;
        if (controlName !== null) {
            control = task.getForm().GetCtrl(controlName);
            return control.validate(value);
        }
        return null;
    }
    static GetFldRanges(taskId, controlName) {
        let task = MGDataCollection.Instance.GetTaskByID(taskId);
        let control = null;
        if (controlName !== null) {
            control = task.getForm().GetCtrl(controlName);
            return control.getRanges();
        }
        return null;
    }
    static TransCacheExists() {
        let clientTransCacheExists = AccessHelper.mgDataTable.getCurrMGData().clientTransactionCacheExists();
        let serverTransactionCacheExists = AccessHelper.mgDataTable.getCurrMGData().serverTransactionCacheExists();
        return clientTransCacheExists || serverTransactionCacheExists;
    }
    RegisterDelegates() {
        HttpClientEvents.GetExecutionProperty_Event = ServerConfig.Instance.GetExecutionProperty.bind(ServerConfig.Instance);
        HttpClientEvents.GetRuntimeCtxID_Event = this.GetRuntimeCtxID.bind(this);
        HttpClientEvents.GetGlobalUniqueSessionID_Event = this.GetGlobalUniqueSessionID.bind(this);
        HttpClientEvents.ShouldDisplayGenericError_Event = ServerConfig.Instance.ShouldDisplayGenericError.bind(ServerConfig.Instance);
    }
    GetRuntimeCtxID() {
        return RuntimeContextBase.Instance.ContextID;
    }
    static TerminateContextUsingFetchAPI() {
        if (RemoteCommandsProcessor.IsSessionReInitializing)
            return;
        HttpManager.GetInstance().IsAbortingMagicEngine = true;
        MGDataCollection.Instance.getCurrMGData().CmdsToServer.Clear();
        let contextTerminationEventCommand = CommandFactory.
            CreateContextTerminationEventCommand(MGDataCollection.Instance.StartupMgData.getFirstTask().getTaskTag());
        MGDataCollection.Instance.getCurrMGData().CmdsToServer.Add(contextTerminationEventCommand);
        RemoteCommandsProcessor.GetInstance().Execute(CommandsProcessorBase_SendingInstruction.ONLY_COMMANDS);
    }
}
ClientManager._instance = null;
ClientManager._executionPropertiesFileName = "./assets/server-config.json";
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiQ2xpZW50TWFuYWdlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uL3Byb2plY3RzL2VuZ2luZS9zcmMvQ2xpZW50TWFuYWdlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQSxPQUFPLEVBQUMsb0JBQW9CLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxRQUFRLEVBQUMsTUFBTSxzQkFBc0IsQ0FBQztBQUV2RixPQUFPLEVBQ0wsYUFBYSxFQUNiLGlCQUFpQixFQUNqQixNQUFNLEVBQ04sSUFBSSxFQUNKLGFBQWEsRUFDYixnQkFBZ0IsRUFDaEIsWUFBWSxFQUNiLE1BQU0sa0JBQWtCLENBQUM7QUFDMUIsT0FBTyxFQUNMLFFBQVEsRUFDUixXQUFXLEVBQ1gsZ0JBQWdCLEVBQ2hCLE1BQU0sRUFDTixpQkFBaUIsRUFDakIsY0FBYyxFQUVkLE9BQU8sRUFFUCxHQUFHLEVBQUUsYUFBYSxFQUNsQixrQkFBa0IsRUFDbEIsTUFBTSxFQUNQLE1BQU0sZ0JBQWdCLENBQUM7QUFDeEIsT0FBTyxFQUFDLGFBQWEsRUFBQyxNQUFNLHVCQUF1QixDQUFDO0FBQ3BELE9BQU8sRUFBQyxrQkFBa0IsRUFBQyxNQUFNLHNCQUFzQixDQUFDO0FBQ3hELE9BQU8sRUFBQyxXQUFXLEVBQUMsTUFBTSxtQkFBbUIsQ0FBQztBQUM5QyxPQUFPLEVBQUMsWUFBWSxFQUFDLE1BQU0sb0JBQW9CLENBQUM7QUFDaEQsT0FBTyxFQUFDLFVBQVUsRUFBQyxNQUFNLG9CQUFvQixDQUFDO0FBRzlDLE9BQU8sRUFBQyxJQUFJLEVBQUMsTUFBTSxjQUFjLENBQUM7QUFDbEMsT0FBTyxFQUFDLGdCQUFnQixFQUFDLE1BQU0sMEJBQTBCLENBQUM7QUFDMUQsT0FBTyxFQUFDLGFBQWEsRUFBQyxNQUFNLHNCQUFzQixDQUFDO0FBQ25ELE9BQU8sRUFBQyxXQUFXLEVBQUMsTUFBTSxvQkFBb0IsQ0FBQztBQUMvQyxPQUFPLEVBQUMsYUFBYSxFQUFDLE1BQU0saUJBQWlCLENBQUM7QUFDOUMsT0FBTyxFQUFDLHVCQUF1QixFQUFDLE1BQU0sa0NBQWtDLENBQUM7QUFDekUsT0FBTyxFQUFDLHdDQUF3QyxFQUFDLE1BQU0seUJBQXlCLENBQUM7QUFDakYsT0FBTyxFQUFDLHdCQUF3QixFQUFDLE1BQU0sNEJBQTRCLENBQUM7QUFDcEUsT0FBTyxFQUFDLE1BQU0sRUFBQyxNQUFNLGdCQUFnQixDQUFDO0FBQ3RDLE9BQU8sRUFBQyxXQUFXLEVBQUMsTUFBTSxzQkFBc0IsQ0FBQztBQUNqRCxPQUFPLEVBQUMsZ0JBQWdCLEVBQUMsTUFBTSxnQ0FBZ0MsQ0FBQztBQUVoRSxPQUFPLEVBQUMsY0FBYyxFQUFDLE1BQU0sMENBQTBDLENBQUM7QUFDeEUsT0FBTyxFQUFDLG9CQUFvQixFQUFDLE1BQU0sd0JBQXdCLENBQUM7QUFFNUQsT0FBTyxFQUFDLGtCQUFrQixFQUFDLE1BQU0seUJBQXlCLENBQUM7QUFDM0QsT0FBTyxFQUFDLGNBQWMsRUFBQyxNQUFNLHlCQUF5QixDQUFDO0FBQ3ZELE9BQU8sRUFBQyxZQUFZLEVBQUMsTUFBTSx5QkFBeUIsQ0FBQztBQUNyRCxPQUFPLEVBQUMsY0FBYyxFQUFDLE1BQU0sMENBQTBDLENBQUM7QUFDeEUsT0FBTyxFQUFDLFlBQVksRUFBQyxNQUFNLGdCQUFnQixDQUFDO0FBQzVDLE9BQU8sRUFBQyxZQUFZLEVBQUMsTUFBTSxnQkFBZ0IsQ0FBQztBQUc1QyxPQUFPLEVBQUMsVUFBVSxFQUFDLE1BQU0sY0FBYyxDQUFDO0FBR3hDLE1BQU0sT0FBTyxhQUFhO0lBVXhCLE1BQU0sS0FBSyxRQUFRO1FBQ2pCLElBQUksYUFBYSxDQUFDLFNBQVMsS0FBSyxJQUFJO1lBQ2xDLGFBQWEsQ0FBQyxTQUFTLEdBQUcsSUFBSSxhQUFhLEVBQUUsQ0FBQztRQUVoRCxPQUFPLGFBQWEsQ0FBQyxTQUFTLENBQUM7SUFDakMsQ0FBQztJQUVNLFFBQVEsQ0FBQyxRQUFtQjtRQUNqQyxJQUFJLElBQUksR0FBZSxnQkFBZ0IsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM5RSxJQUFJLE9BQU8sR0FBVyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUM7UUFDckMsSUFBSSxRQUFRLEdBQVcsQ0FBQyxDQUFDO1FBRXpCLElBQUksSUFBSSxJQUFJLElBQUksRUFBRTtZQUNoQixJQUFJLE9BQU8sR0FBa0IsSUFBSSxDQUFDO1lBRWxDLElBQUksUUFBUSxDQUFDLFdBQVcsS0FBSyxJQUFJO2dCQUMvQixPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLENBQUM7WUFFekQsSUFBSSxPQUFPLElBQUksSUFBSSxFQUFFO2dCQUNuQixJQUFJLEVBQUUsR0FBdUIsSUFBSSxDQUFDLFFBQVEsQ0FBQztnQkFDM0MsSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFO29CQUN4QixPQUFPLElBQUksRUFBRSxDQUFDLHdCQUF3QixDQUFDO2lCQUN4QztnQkFDRCxRQUFRLFFBQVEsQ0FBQyxTQUFTLEVBQUU7b0JBRTFCLEtBQUssWUFBWTt3QkFDZixJQUFJLE9BQU8sQ0FBQyxnQkFBZ0IsRUFBRSxFQUFFOzRCQUM5QixNQUFNLENBQUMsYUFBYSxDQUFDLE9BQU8sRUFBRSxRQUFRLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxXQUFXLENBQUMsQ0FBQzt5QkFDcEU7d0JBQ0QsTUFBTTtvQkFDUixLQUFLLFdBQVc7d0JBTWQsSUFBSSxPQUFPLENBQUMsUUFBUSxFQUFFLEVBQUU7NEJBQ3RCLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLENBQUM7eUJBQzlCOzZCQUNJLElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFOzRCQUMzQixNQUFNLENBQUMsaUJBQWlCLENBQUMsT0FBTyxDQUFDLG1CQUFtQixFQUFFLENBQUMsQ0FBQyxtQkFBbUIsR0FBRyxPQUFPLENBQUMsT0FBTyxFQUFFLENBQUM7d0JBQ2xHLE1BQU07b0JBRVIsS0FBSyxPQUFPO3dCQUNWLElBQUksT0FBTyxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLGdCQUFnQixJQUFJLFFBQVEsQ0FBQyxVQUFVLENBQUMsRUFBQzs0QkFDMUUsTUFBTSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQzs0QkFDL0MsTUFBTSxDQUFDLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLENBQUMsbUJBQW1CLEdBQUcsT0FBTyxDQUFDLE9BQU8sRUFBRSxDQUFDOzRCQUNoRyxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsS0FBSyxDQUFDO3lCQUMvQjs2QkFLSTs0QkFDSCxNQUFNLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDOzRCQUM5QyxJQUFJLFlBQVksR0FBWSxPQUFPLENBQUMsUUFBUSxFQUFFLElBQUksT0FBTyxDQUFDLGFBQWEsRUFBRSxJQUFJLE9BQU8sQ0FBQyxPQUFPLEVBQUUsSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFLElBQUksT0FBTyxDQUFDLGNBQWMsRUFBRSxJQUFJLE9BQU8sQ0FBQyxVQUFVLEVBQUUsSUFBSSxPQUFPLENBQUMsY0FBYyxFQUFFLENBQUM7NEJBQ3pNLE1BQU0sQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsWUFBWSxDQUFDLENBQUM7eUJBQzFGO3dCQUNELE1BQU07b0JBQ1IsS0FBSyxPQUFPO3dCQUVWLElBQUksSUFBSSxDQUFDLGdCQUFnQjs0QkFDdkIsT0FBTzt3QkFtQlQsTUFBTSxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQzt3QkFFOUMsTUFBTTtvQkFDUixLQUFLLFVBQVU7d0JBQ2IsTUFBTSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsT0FBTyxDQUFDLENBQUM7d0JBQ3BDLE1BQU07b0JBRVIsS0FBSyxrQkFBa0I7d0JBQ3JCLE1BQU0sQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxDQUFDO3dCQUMzRCxNQUFNO29CQUNSLEtBQUssU0FBUzt3QkFDWixNQUFNLENBQUMsU0FBUyxDQUFFLElBQUksQ0FBQyxPQUFPLEVBQUUsRUFBRSxPQUFPLEVBQUUsUUFBUSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsT0FBTyxFQUFFLENBQUMsRUFBQyxDQUFDLEVBQUUsRUFBRSxFQUFDLEtBQUssRUFBQyxFQUFFLEVBQUMsSUFBSSxDQUFFLENBQUM7d0JBQ3pHLE1BQU07b0JBQ1IsS0FBSyxrQkFBa0I7d0JBQ3JCLGFBQWEsQ0FBQyxRQUFRLENBQUMscUJBQXFCLENBQWEsT0FBUSxFQUFFLFFBQVEsQ0FBQyxhQUFhLEVBQUUsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO3dCQUMzRyxNQUFNO29CQUNSO3dCQUVFLE1BQU07aUJBQ1Q7YUFDRjtpQkFDSTtnQkFDSCxRQUFRLFFBQVEsQ0FBQyxTQUFTLEVBQUU7b0JBQzFCLEtBQUssd0JBQXdCO3dCQUMzQixPQUFPLEdBQUcsa0JBQWtCLENBQUMscUJBQXFCLEVBQUUsQ0FBQzt3QkFDckQsSUFBSSxPQUFPLEtBQUssSUFBSSxFQUFFOzRCQUNwQixNQUFNLENBQUMsd0JBQXdCLENBQUMsT0FBTyxDQUFDLENBQUM7eUJBQzFDO3dCQUNELE1BQU07b0JBQ1IsS0FBSyxPQUFPO3dCQUNWLElBQUcsUUFBUSxDQUFDLFdBQVcsS0FBSyxVQUFVLEVBQUU7NEJBQ3RDLFFBQVEsR0FBRyxRQUFRLENBQUMsSUFBSSxHQUFjLElBQUksQ0FBQyxRQUFTLENBQUMsd0JBQXdCLENBQUM7NEJBQzlFLGFBQWEsQ0FBQyxRQUFRLENBQUMsMENBQTBDLENBQUMsSUFBSSxFQUFFLGlCQUFpQixDQUFDLGlCQUFpQixFQUFFLFFBQVEsQ0FBQyxDQUFDO3lCQUN4SDt3QkFDRCxNQUFNO29CQUNSLEtBQUssaUJBQWlCO3dCQUNwQixJQUFJLFFBQVEsQ0FBQyxXQUFXLEtBQUssVUFBVTs0QkFDckMsSUFBSSxDQUFDLHdCQUF3QixDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7d0JBQ3RELE1BQU07b0JBQ1IsS0FBSyxPQUFPO3dCQUNWLElBQUksSUFBSSxDQUFDLE9BQU8sRUFBRTs0QkFDaEIsYUFBYSxDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsQ0FBQzs7NEJBRWxELE1BQU0sQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7d0JBQ3hDLE1BQU07b0JBQ1IsS0FBSyxTQUFTO3dCQUNaLGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQzt3QkFDNUQsTUFBTTtvQkFDUixLQUFLLFFBQVE7d0JBQ1gsTUFBTSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQyxZQUFZLEVBQUUsRUFBRSxRQUFRLENBQUMsUUFBUSxDQUFDLENBQUM7b0JBRTVFLEtBQUssU0FBUzt3QkFDWixRQUFRLEdBQUcsT0FBTyxHQUFjLElBQUksQ0FBQyxRQUFTLENBQUMsd0JBQXdCLENBQUM7d0JBQ3hFLElBQUksY0FBYyxHQUFtQixFQUFDLFlBQVksRUFBRSxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUMsWUFBWSxFQUFFLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxHQUFHLEVBQUUsRUFBRSxFQUFDLENBQUM7d0JBQzVHLE1BQU0sQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDLFlBQVksRUFBRSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsY0FBYyxDQUFDLENBQUM7d0JBQ3BGLE1BQU07b0JBQ1IsS0FBSyxnQkFBZ0I7d0JBQ25CLGFBQWEsQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxpQkFBaUIsQ0FBQyxzQkFBc0IsRUFBRSxRQUFRLENBQUMsVUFBVSxFQUFFLFFBQVEsQ0FBQyxnQkFBZ0IsRUFDakksUUFBUSxDQUFDLFlBQVksQ0FBQyxDQUFDO3dCQUN6QixNQUFNO29CQUNSLEtBQUssa0JBQWtCO3dCQUNyQixNQUFNLENBQUMsUUFBUSxDQUFDLGVBQWUsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLHFFQUFxRSxFQUFFLFFBQVEsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO3dCQUM3SSxNQUFNO29CQUNSO3dCQUVFLE1BQU07aUJBQ1Q7YUFDRjtTQUNGOztZQUVDLFFBQVEsUUFBUSxDQUFDLFNBQVMsRUFBRTtnQkFDMUIsS0FBSyxvQkFBb0I7b0JBQ3ZCLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUM7b0JBQzFDLE1BQU07YUFDVDtJQUNMLENBQUM7SUFFTyx3QkFBd0IsQ0FBQyxJQUFVLEVBQUUsR0FBVztRQUN0RCxJQUFJLFdBQVcsR0FBRyxpQkFBaUIsQ0FBQyxXQUFXLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQUV4QixRQUFRLEdBQUcsRUFBRTtZQUNYLEtBQUssWUFBWTtnQkFDZixXQUFXLEdBQUcsaUJBQWlCLENBQUMsa0JBQWtCLENBQUM7Z0JBQ25ELE1BQU07WUFFUixLQUFLLGVBQWU7Z0JBQ2xCLFdBQVcsR0FBRyxpQkFBaUIsQ0FBQyxrQkFBa0IsQ0FBQztnQkFDbkQsTUFBTTtZQUVSLEtBQUssVUFBVTtnQkFDYixXQUFXLEdBQUcsaUJBQWlCLENBQUMsa0JBQWtCLENBQUM7Z0JBQ25ELE1BQU07WUFFUixLQUFLLFdBQVc7Z0JBQ2QsV0FBVyxHQUFHLGlCQUFpQixDQUFDLGtCQUFrQixDQUFDO2dCQUNuRCxNQUFNO1lBRVIsS0FBSyxlQUFlO2dCQUNsQixXQUFXLEdBQUcsaUJBQWlCLENBQUMsa0JBQWtCLENBQUM7Z0JBQ25ELE1BQU07WUFFUixLQUFLLFdBQVc7Z0JBQ2QsV0FBVyxHQUFHLGlCQUFpQixDQUFDLGtCQUFrQixDQUFDO2dCQUNuRCxNQUFNO1lBRVIsS0FBSyxhQUFhO2dCQUNoQixXQUFXLEdBQUcsaUJBQWlCLENBQUMsaUJBQWlCLENBQUM7Z0JBQ2xELE1BQU07WUFFUixLQUFLLFdBQVc7Z0JBQ2QsV0FBVyxHQUFHLGlCQUFpQixDQUFDLGlCQUFpQixDQUFDO2dCQUNsRCxNQUFNO1NBQ1Q7UUFFRCxJQUFJLFdBQVcsSUFBSSxpQkFBaUIsQ0FBQyxXQUFXO1lBQzlDLGFBQWEsQ0FBQyxRQUFRLENBQUMsbUNBQW1DLENBQUMsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUFDO0lBQ2xGLENBQUM7SUFLRCxrQkFBa0IsQ0FBQyxHQUFXO1FBRTVCLFFBQVEsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUtELEtBQUssQ0FBQyxtQkFBbUI7UUFDdkIsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxFQUFFO1lBQy9CLE1BQU0sQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLEVBQUUsRUFBRSxJQUFJLENBQUMsQ0FBQztZQUNyQyxNQUFNLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxhQUFhLEdBQUcsYUFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsR0FBRyxFQUFFLFlBQVksQ0FBQyxxQkFBcUIsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFDNUksSUFBSSxDQUFDLDZCQUE2QixFQUFFLENBQUM7U0FDdEM7UUFFRCxJQUFJO1lBQ0YsSUFBSSxNQUFNLHdCQUF3QixDQUFDLFlBQVksRUFBRSxFQUFFO2dCQUNqRCxJQUFJLFNBQVMsR0FBcUIsZ0JBQWdCLENBQUMsUUFBUSxDQUFDO2dCQUM1RCxJQUFJLGFBQWEsR0FBVyxTQUFTLENBQUMsYUFBYSxDQUFDO2dCQUNwRCxTQUFTLENBQUMsU0FBUyxHQUFHLGFBQWEsQ0FBQyxLQUFLLEVBQUUsQ0FBQztnQkFFNUMsSUFBSSxrQkFBa0IsR0FBUyxJQUFJLENBQUM7Z0JBQ3BDLElBQUksQ0FBQyxhQUFhLENBQUMsVUFBVTtvQkFDM0Isa0JBQWtCLEdBQUcsTUFBTSxJQUFJLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztnQkFFN0QsSUFBSSxrQkFBa0IsS0FBSyxJQUFJO29CQUM3QixNQUFNLGFBQWEsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDOztvQkFFdkQsTUFBTSxhQUFhLENBQUMsUUFBUSxDQUFDLHdCQUF3QixDQUFDLGFBQWEsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO2dCQUczRixhQUFhLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUMzQztTQUNGO1FBQ0QsT0FBTyxFQUFFLEVBQUU7WUFDVCxJQUFJLEVBQUUsWUFBWSxvQkFBb0IsRUFBRTtnQkFDdEMsSUFBSSxlQUFlLEdBQVksS0FBSyxDQUFDO2dCQUNyQyxJQUFJLEVBQUUsWUFBWSxXQUFXLElBQWtCLEVBQUcsQ0FBQyxPQUFPLEVBQUUsS0FBSyxXQUFXLENBQUMsYUFBYTtvQkFDeEYsZUFBZSxHQUFHLElBQUksQ0FBQztnQkFFekIsSUFBSSxDQUFDLGVBQWU7b0JBQ2xCLE1BQU0sYUFBYSxDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsQ0FBQztnQkFFeEQsYUFBYSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDM0M7O2dCQUVDLE1BQU0sRUFBRSxDQUFDO1NBQ1o7Z0JBQ087WUFDTixNQUFNLGFBQWEsQ0FBQyxRQUFRLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLENBQUM7WUFFeEQsUUFBUSxDQUFDLGdCQUFnQixDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUNsRCxRQUFRLENBQUMsV0FBVyxFQUFFLENBQUM7U0FDeEI7SUFDSCxDQUFDO0lBS0QsNkJBQTZCO1FBQzNCLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyw4QkFBOEIsRUFBRSxFQUFFO1lBQ3BELE1BQU0sQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLCtFQUErRSxFQUFFLElBQUksQ0FBQyxDQUFDO1lBRWxILE1BQU0sQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLHVCQUF1QixFQUFFLElBQUksQ0FBQyxDQUFDO1lBRTFELElBQUksV0FBVyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ3hELE1BQU0sQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsQ0FBQztZQUU5QyxNQUFNLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQywrRUFBK0UsRUFBRSxJQUFJLENBQUMsQ0FBQztTQUNuSDtJQUNILENBQUM7SUFHRDtRQTNSUSwyQkFBc0IsR0FBVyxJQUFJLENBQUM7UUFFdEMscUJBQWdCLEdBQUcsS0FBSyxDQUFDO1FBMFIvQixrQkFBa0IsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLHVCQUF1QixDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDM0Usa0JBQWtCLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUM1RCxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxxQkFBcUIsRUFBRSxDQUFDO1FBRzFFLHVCQUF1QixDQUFDLG1DQUFtQyxHQUFHLElBQUksQ0FBQztRQUluRSxJQUFJLENBQUMsc0JBQXNCLEdBQUcsYUFBYSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsR0FBRyxDQUFDO1FBRXZFLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO0lBQzNCLENBQUM7SUFLTyxjQUFjO1FBQ3BCLE9BQU8sQ0FBQyxhQUFhLEdBQUcsYUFBYSxDQUFDLFFBQVEsQ0FBQztRQUMvQyxPQUFPLENBQUMsaUJBQWlCLEdBQUcsWUFBWSxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsQ0FBQztRQUM5RCxPQUFPLENBQUMsZUFBZSxHQUFHLFlBQVksQ0FBQyxRQUFRLENBQUMsV0FBVyxFQUFFLENBQUM7UUFDOUQsY0FBYyxDQUFDLFdBQVcsR0FBRyxXQUFXLENBQUMsUUFBUSxDQUFDO1FBQ2xELGlCQUFpQixDQUFDLFdBQVcsR0FBRyxnQkFBZ0IsQ0FBQyxRQUFRLENBQUM7UUFDMUQsa0JBQWtCLENBQUMsV0FBVyxHQUFHLHVCQUF1QixDQUFDLGdCQUFnQixDQUFDO0lBQzVFLENBQUM7SUFHTyw4QkFBOEI7UUFDcEMsVUFBVSxDQUFDLGFBQWEsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDO1FBQzNDLGNBQWMsQ0FBQyxzQkFBc0IsR0FBRyxhQUFhLENBQUMsbUJBQW1CLENBQUM7UUFFMUUsWUFBWSxDQUFDLGFBQWEsR0FBRyxhQUFhLENBQUMsUUFBUSxDQUFDO1FBQ3BELFlBQVksQ0FBQyxXQUFXLEdBQUcsZ0JBQWdCLENBQUMsUUFBUSxDQUFDO1FBQ3JELFlBQVksQ0FBQyxjQUFjLEdBQUcsY0FBYyxDQUFDLFFBQVEsQ0FBQztRQUN0RCxZQUFZLENBQUMsWUFBWSxHQUFHLFlBQVksQ0FBQyxRQUFRLENBQUM7UUFDbEQsWUFBWSxDQUFDLFlBQVksR0FBRyxZQUFZLENBQUMsUUFBUSxDQUFDO1FBQ2xELFlBQVksQ0FBQyxXQUFXLEdBQUcsV0FBVyxDQUFDLFFBQVEsQ0FBQztJQUNsRCxDQUFDO0lBSUQsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFzQjtRQU9oQyxhQUFhLENBQUMsY0FBYyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBRXpDLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDMUIsQ0FBQztJQUdELE1BQU0sQ0FBQyxLQUFLLENBQUMsYUFBYTtRQUN4QixJQUFJO1lBQ0YsV0FBVyxDQUFDLFdBQVcsRUFBRSxDQUFDLHFCQUFxQixHQUFHLElBQUksQ0FBQztZQUN2RCxnQkFBZ0IsQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLFlBQVksRUFBRSxDQUFDLGFBQWEsR0FBRyxJQUFJLENBQUM7WUFFNUUsTUFBTSxnQkFBZ0IsQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLFlBQVksRUFBRSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUM5RSxLQUFLLENBQUMsQ0FBQztTQUNWO1FBQ0QsT0FBTyxHQUFHLEVBQUU7WUFDVixJQUFJLEdBQUcsWUFBWSxvQkFBb0IsRUFBRTtnQkFDdkMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxHQUFHLENBQUMsQ0FBQzthQUMxQzs7Z0JBRUMsTUFBTSxHQUFHLENBQUM7U0FDYjtJQUNILENBQUM7SUFFRCxNQUFNLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxVQUFxQjtRQUMvQyxXQUFXLENBQUMsbUJBQW1CLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDNUMsSUFBSSxNQUFNLEdBQVksTUFBTSxhQUFhLENBQUMsUUFBUSxDQUFDLGtCQUFrQixFQUFFLENBQUM7UUFFeEUsSUFBSSxNQUFNLEVBQUU7WUFDVixJQUFJLGtCQUFrQixFQUFFLENBQUM7WUFDekIsYUFBYSxDQUFDLFFBQVEsQ0FBQyxjQUFjLEVBQUUsQ0FBQztZQUN4QyxhQUFhLENBQUMsUUFBUSxDQUFDLDhCQUE4QixFQUFFLENBQUM7WUFDeEQsSUFBSSxhQUFhLEdBQUcsb0JBQW9CLEtBQU0sU0FBUyxDQUFDLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsWUFBWSxHQUFHLG9CQUFvQixDQUFFO1lBRXBILE9BQU8sQ0FBQyxHQUFHLENBQUMsb0JBQW9CLEdBQUcsYUFBYSxDQUFDLENBQUM7WUFFbEQsV0FBVyxDQUFDLFdBQVcsRUFBRSxDQUFDLDBCQUEwQixHQUFHLFlBQVksQ0FBQyxRQUFRLENBQUMsMEJBQTBCLEVBQUUsQ0FBQztZQUUxRyxhQUFhLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxDQUFDO1lBRTlCLE1BQU0sQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLHVCQUF1QixDQUFDLFdBQVcsRUFBRSxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLG1CQUFtQixFQUFFLENBQUMsRUFBQyxLQUFLLENBQUMsQ0FBQztZQUVuSSxJQUFJLEdBQUcsR0FBVyxJQUFJLE1BQU0sQ0FDMUIsSUFBSSxhQUFhLEVBQUUsRUFDbkIsSUFBSSxhQUFhLEVBQUUsRUFDbkIsQ0FBQyxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztZQUNsQixNQUFNLGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsR0FBRyxFQUFFLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztZQUV4RCxNQUFNLGFBQWEsQ0FBQyxRQUFRLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztZQUVuRCxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLEVBQUU7Z0JBQy9CLE1BQU0sQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEdBQUcsV0FBVyxHQUFHLGFBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLEdBQUcsRUFBRSxZQUFZLENBQUMscUJBQXFCLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxHQUFHLGFBQWEsQ0FBQyxNQUFNLEdBQUcsYUFBYSxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQzthQUNqTjtTQUNGO2FBRUQ7WUFDRSxNQUFNLFVBQVUsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSw0Q0FBNEMsR0FBRyxJQUFJLENBQUMsNEJBQTRCLEdBQUcsR0FBRyxFQUFFLE1BQU0sQ0FBQyxpQkFBaUIsR0FBRyxNQUFNLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztTQUMzTDtJQUNILENBQUM7SUFLRCxLQUFLLENBQUMsa0JBQWtCO1FBRXRCLElBQUksUUFBUSxHQUFXLEVBQUUsQ0FBQztRQUMxQixJQUFJLFNBQVMsR0FBWSxJQUFJLENBQUM7UUFFOUIsSUFBSTtZQUVGLElBQUksT0FBTyxHQUFzQixJQUFJLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUNyRCxRQUFRLEdBQUcsTUFBTSxXQUFXLENBQUMsV0FBVyxFQUFFLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyw0QkFBNEIsRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3hILFlBQVksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztTQUNsRDtRQUNELE9BQU8sRUFBRSxFQUFFO1lBQ1QsU0FBUyxHQUFHLEtBQUssQ0FBQztTQUduQjtRQUVELE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7SUFNRCx3QkFBd0I7UUFDdEIsT0FBTyxJQUFJLENBQUMsc0JBQXNCLENBQUM7SUFDckMsQ0FBQztJQU9ELE1BQU0sQ0FBRSxxQkFBcUIsQ0FBQyxNQUFjLEVBQUUsV0FBbUI7UUFFL0QsSUFBSSxHQUFHLEdBQVEsSUFBSSxDQUFDO1FBQ3BCLElBQUksSUFBSSxHQUFlLGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFckUsSUFBSSxPQUFPLEdBQWtCLElBQUksQ0FBQztRQUVsQyxJQUFJLFdBQVcsS0FBSyxJQUFJLElBQUksSUFBSSxJQUFJLElBQUksRUFBRTtZQUN0QyxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUM5QyxHQUFHLEdBQUcsT0FBTyxDQUFDLE1BQU0sRUFBRSxDQUFDO1NBQzFCO1FBRUQsT0FBTyxHQUFHLENBQUM7SUFDYixDQUFDO0lBTUQsTUFBTSxDQUFFLGlCQUFpQixDQUFDLE1BQWMsRUFBRSxXQUFtQixFQUFFLEtBQVUsRUFBRSxLQUFjO1FBRXZGLElBQUksR0FBRyxHQUFRLElBQUksQ0FBQztRQUNwQixJQUFJLElBQUksR0FBZSxnQkFBZ0IsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRXJFLElBQUksT0FBTyxHQUFrQixJQUFJLENBQUM7UUFDbEMsSUFBSSxRQUFRLEdBQVEsS0FBSyxDQUFDO1FBRTFCLElBQUksV0FBVyxLQUFLLElBQUksSUFBSyxJQUFJLEtBQUssSUFBSTtZQUN4QyxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUNoRCxJQUFJLE9BQU8sSUFBSSxJQUFJLElBQUksT0FBTyxDQUFDLFFBQVEsSUFBSSxnQkFBZ0IsQ0FBQyxPQUFPLEVBQUU7WUFDbkUsSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFLElBQUssT0FBTyxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxZQUFZLEVBQUUsRUFDN0Y7Z0JBQ0ksSUFBSSxNQUFNLEdBQUcsT0FBTyxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDbEYsR0FBRyxHQUFHLElBQUksR0FBRyxDQUFFLE1BQU0sRUFBRSxnQkFBZ0IsQ0FBQyxPQUFPLEVBQUcsT0FBTyxDQUFDLE9BQU8sRUFBRSxDQUFDLE9BQU8sRUFBRSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUE7YUFDOUY7O2dCQUVDLEdBQUcsR0FBRyxPQUFPLENBQUMsTUFBTSxFQUFFLENBQUM7WUFFekIsSUFBSSxTQUFTLEdBQUcsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQzlCLElBQUssTUFBTSxHQUFHLEdBQUcsQ0FBQyx3QkFBd0IsRUFBRSxDQUFDO1lBRTdDLElBQUksR0FBRyxHQUFHLGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQztZQUNoRSxRQUFRLEdBQUcsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUM7U0FDeEQ7UUFDRCxPQUFPLFFBQVEsQ0FBQztJQUNsQixDQUFDO0lBS0QsTUFBTSxDQUFFLGNBQWMsQ0FBQyxNQUFjLEVBQUUsV0FBbUIsRUFBRSxLQUFhO1FBRXZFLElBQUksSUFBSSxHQUFlLGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFckUsSUFBSSxPQUFPLEdBQWtCLElBQUksQ0FBQztRQUVsQyxJQUFJLFdBQVcsS0FBSyxJQUFJLEVBQ3hCO1lBQ0UsT0FBTyxHQUFJLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLENBQUM7WUFDL0MsT0FBTyxPQUFPLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQ3RDO1FBQ0QsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBS0QsTUFBTSxDQUFFLG9CQUFvQixDQUFDLE1BQWMsRUFBRSxXQUFtQixFQUFFLEtBQVU7UUFFMUUsSUFBSSxJQUFJLEdBQWUsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUVyRSxJQUFJLE9BQU8sR0FBYyxJQUFJLENBQUM7UUFFOUIsSUFBSSxXQUFXLEtBQUssSUFBSSxFQUN4QjtZQUNFLE9BQU8sR0FBSSxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBYyxDQUFDO1lBQzVELE9BQU8sT0FBTyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUNoQztRQUNELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUtELE1BQU0sQ0FBQyxZQUFZLENBQUMsTUFBYyxFQUFFLFdBQW1CO1FBQ3JELElBQUksSUFBSSxHQUFlLGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFckUsSUFBSSxPQUFPLEdBQWMsSUFBSSxDQUFDO1FBRTlCLElBQUksV0FBVyxLQUFLLElBQUksRUFDeEI7WUFDRSxPQUFPLEdBQUksSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQWMsQ0FBQztZQUM1RCxPQUFPLE9BQU8sQ0FBQyxTQUFTLEVBQUUsQ0FBQztTQUM1QjtRQUNELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUtELE1BQU0sQ0FBQyxnQkFBZ0I7UUFDckIsSUFBSSxzQkFBc0IsR0FBWSxZQUFZLENBQUMsV0FBVyxDQUFDLGFBQWEsRUFBRSxDQUFDLDRCQUE0QixFQUFFLENBQUM7UUFDOUcsSUFBSSw0QkFBNEIsR0FBWSxZQUFZLENBQUMsV0FBVyxDQUFDLGFBQWEsRUFBRSxDQUFDLDRCQUE0QixFQUFFLENBQUM7UUFFcEgsT0FBTyxzQkFBc0IsSUFBSSw0QkFBNEIsQ0FBQztJQUNoRSxDQUFDO0lBT0QsaUJBQWlCO1FBQ2YsZ0JBQWdCLENBQUMsMEJBQTBCLEdBQUcsWUFBWSxDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3JILGdCQUFnQixDQUFDLHFCQUFxQixHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3pFLGdCQUFnQixDQUFDLDhCQUE4QixHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDM0YsZ0JBQWdCLENBQUMsK0JBQStCLEdBQUcsWUFBWSxDQUFDLFFBQVEsQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ2pJLENBQUM7SUFNRCxlQUFlO1FBQ2IsT0FBTyxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDO0lBQy9DLENBQUM7SUFLRCxNQUFNLENBQUMsNkJBQTZCO1FBSWxDLElBQUksdUJBQXVCLENBQUMsdUJBQXVCO1lBQ2pELE9BQU87UUFDVCxXQUFXLENBQUMsV0FBVyxFQUFFLENBQUMscUJBQXFCLEdBQUcsSUFBSSxDQUFDO1FBR3ZELGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxhQUFhLEVBQUUsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFL0QsSUFBSSw4QkFBOEIsR0FBbUMsY0FBYztZQUNuRixvQ0FBb0MsQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLFlBQVksRUFBRSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7UUFFMUcsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLGFBQWEsRUFBRSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQWlCLDhCQUE4QixDQUFDLENBQUM7UUFHM0csdUJBQXVCLENBQUMsV0FBVyxFQUFFLENBQUMsT0FBTyxDQUFDLHdDQUF3QyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ3hHLENBQUM7O0FBaGtCYyx1QkFBUyxHQUFrQixJQUFJLENBQUM7QUFNdkIsMENBQTRCLEdBQVcsNkJBQTZCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge0FwcGxpY2F0aW9uRXhjZXB0aW9uLCBEYXRlVGltZSwgTlN0cmluZywgUmVmUGFyYW19IGZyb20gJ0BtYWdpYy14cGEvbXNjb3JlbGliJztcclxuXHJcbmltcG9ydCB7XHJcbiAgRGF0ZVRpbWVVdGlscyxcclxuICBJbnRlcm5hbEludGVyZmFjZSxcclxuICBMb2dnZXIsXHJcbiAgTWlzYyxcclxuICBPU0Vudmlyb25tZW50LFxyXG4gIFN0b3JhZ2VBdHRyaWJ1dGUsXHJcbiAgWE1MQ29uc3RhbnRzXHJcbn0gZnJvbSAnQG1hZ2ljLXhwYS91dGlscyc7XHJcbmltcG9ydCB7XHJcbiAgQ29tbWFuZHMsXHJcbiAgQ29tbWFuZFR5cGUsXHJcbiAgRGlzcGxheUNvbnZlcnRvcixcclxuICBFdmVudHMsXHJcbiAgR3VpRGF0YUNvbGxlY3Rpb24sXHJcbiAgR3VpRW52aXJvbm1lbnQsXHJcbiAgTGFzdEZvY3VzZWRWYWwsXHJcbiAgTWFuYWdlcixcclxuICBNZ0NvbnRyb2xCYXNlLFxyXG4gIFBJQywgUHJvcEludGVyZmFjZSxcclxuICBSdW50aW1lQ29udGV4dEJhc2UsXHJcbiAgU3R5bGVzXHJcbn0gZnJvbSAnQG1hZ2ljLXhwYS9ndWknO1xyXG5pbXBvcnQge0V2ZW50c01hbmFnZXJ9IGZyb20gJy4vZXZlbnQvRXZlbnRzTWFuYWdlcic7XHJcbmltcG9ydCB7R3VpRXZlbnRzUHJvY2Vzc29yfSBmcm9tICcuL0d1aUV2ZW50c1Byb2Nlc3Nvcic7XHJcbmltcG9ydCB7RW52aXJvbm1lbnR9IGZyb20gJy4vZW52L0Vudmlyb25tZW50JztcclxuaW1wb3J0IHtMYW5ndWFnZURhdGF9IGZyb20gJy4vZW52L0xhbmd1YWdlRGF0YSc7XHJcbmltcG9ydCB7VGFza3NUYWJsZX0gZnJvbSAnLi90YXNrcy9UYXNrc1RhYmxlJztcclxuaW1wb3J0IHtNZ0NvbnRyb2x9IGZyb20gJy4vZ3VpL01nQ29udHJvbCc7XHJcbmltcG9ydCB7SUNsaWVudENvbW1hbmR9IGZyb20gJy4vY29tbWFuZHMvSUNsaWVudENvbW1hbmQnO1xyXG5pbXBvcnQge1Rhc2t9IGZyb20gJy4vdGFza3MvVGFzayc7XHJcbmltcG9ydCB7TUdEYXRhQ29sbGVjdGlvbn0gZnJvbSAnLi90YXNrcy9NR0RhdGFDb2xsZWN0aW9uJztcclxuaW1wb3J0IHtVbmlxdWVJRFV0aWxzfSBmcm9tICcuL3V0aWwvVW5pcXVlSURVdGlscyc7XHJcbmltcG9ydCB7SHR0cE1hbmFnZXJ9IGZyb20gJy4vaHR0cC9IdHRwTWFuYWdlcic7XHJcbmltcG9ydCB7Q29tbWFuZHNUYWJsZX0gZnJvbSAnLi9Db21tYW5kc1RhYmxlJztcclxuaW1wb3J0IHtSZW1vdGVDb21tYW5kc1Byb2Nlc3Nvcn0gZnJvbSAnLi9yZW1vdGUvUmVtb3RlQ29tbWFuZHNQcm9jZXNzb3InO1xyXG5pbXBvcnQge0NvbW1hbmRzUHJvY2Vzc29yQmFzZV9TZW5kaW5nSW5zdHJ1Y3Rpb259IGZyb20gJy4vQ29tbWFuZHNQcm9jZXNzb3JCYXNlJztcclxuaW1wb3J0IHtDb21tYW5kc1Byb2Nlc3Nvck1hbmFnZXJ9IGZyb20gJy4vQ29tbWFuZHNQcm9jZXNzb3JNYW5hZ2VyJztcclxuaW1wb3J0IHtNR0RhdGF9IGZyb20gJy4vdGFza3MvTUdEYXRhJztcclxuaW1wb3J0IHtTZXJ2ZXJFcnJvcn0gZnJvbSAnLi9yZW1vdGUvU2VydmVyRXJyb3InO1xyXG5pbXBvcnQge0h0dHBDbGllbnRFdmVudHN9IGZyb20gJy4vaHR0cC9jbGllbnQvSHR0cENsaWVudEV2ZW50cyc7XHJcbmltcG9ydCB7SUd1aUV2ZW50fSBmcm9tICcuL2JyaWRnZS9JR3VpRXZlbnQnO1xyXG5pbXBvcnQge0NvbW1hbmRGYWN0b3J5fSBmcm9tICcuL2NvbW1hbmRzL0NsaWVudFRvU2VydmVyL0NvbW1hbmRGYWN0b3J5JztcclxuaW1wb3J0IHtDdXJyZW50Q2xpZW50VmVyc2lvbn0gZnJvbSAnLi9DdXJyZW50Q2xpZW50VmVyc2lvbic7XHJcbmltcG9ydCB7RGF0YVZpZXd9IGZyb20gJy4vZGF0YS9EYXRhVmlldyc7XHJcbmltcG9ydCB7TGFzdEZvY3VzZWRNYW5hZ2VyfSBmcm9tICcuL3J0L0xhc3RGb2N1c2VkTWFuYWdlcic7XHJcbmltcG9ydCB7RW52UGFyYW1zVGFibGV9IGZyb20gJy4vZW52L0VudlZhcmlhYmxlc1RhYmxlJztcclxuaW1wb3J0IHtHbG9iYWxQYXJhbXN9IGZyb20gJy4vZXhwL0dsb2JhbFBhcmFtc1RhYmxlJztcclxuaW1wb3J0IHtPcGVuVVJMQ29tbWFuZH0gZnJvbSAnLi9jb21tYW5kcy9TZXJ2ZXJUb0NsaWVudC9PcGVuVVJMQ29tbWFuZCc7XHJcbmltcG9ydCB7U2VydmVyQ29uZmlnfSBmcm9tICcuL1NlcnZlckNvbmZpZyc7XHJcbmltcG9ydCB7QWNjZXNzSGVscGVyfSBmcm9tICcuL0FjY2Vzc0hlbHBlcic7XHJcbmltcG9ydCB7SHR0cENsaWVudH0gZnJvbSAnQGFuZ3VsYXIvY29tbW9uL2h0dHAnO1xyXG5pbXBvcnQge0NvbnRleHRUZXJtaW5hdGlvbkV2ZW50Q29tbWFuZH0gZnJvbSAnLi9jb21tYW5kcy9DbGllbnRUb1NlcnZlci9Db250ZXh0VGVybWluYXRpb25FdmVudENvbW1hbmQnO1xyXG5pbXBvcnQge0dVSU1hbmFnZXJ9IGZyb20gXCIuL0dVSU1hbmFnZXJcIjtcclxuXHJcbi8vIEBkeW5hbWljXHJcbmV4cG9ydCBjbGFzcyBDbGllbnRNYW5hZ2VyIHtcclxuXHJcbiAgcHJpdmF0ZSBzdGF0aWMgX2luc3RhbmNlOiBDbGllbnRNYW5hZ2VyID0gbnVsbDtcclxuICBwcml2YXRlIF9nbG9iYWxVbmlxdWVTZXNzaW9uSWQ6IHN0cmluZyA9IG51bGw7XHJcbi8vICBwcml2YXRlIF9sYXN0RmFpbGVkRm9jdXNlZENvbnRyb2w6IE1nQ29udHJvbEJhc2UgPSBudWxsOyAvLyBjb250cm9sIHdoZXJlIGxhc3QgcGFya2luZyB3YXMgYXR0ZW1wdGVkIGJ1dCB3ZSBkaWQgbm90IHBhcmsgb24gaXRcclxuICBwcml2YXRlIF9idXR0b25Jc0NsaWNrZWQgPSBmYWxzZTtcclxuXHJcblxyXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IF9leGVjdXRpb25Qcm9wZXJ0aWVzRmlsZU5hbWU6IHN0cmluZyA9IFwiLi9hc3NldHMvc2VydmVyLWNvbmZpZy5qc29uXCI7IC8vIHBhdGggb2YgdGhlIGV4ZWN1dGlvbiBwcm9wZXJ0aWVzIGZpbGUgZnJvbSBhc3NldHMgZm9sZGVyXHJcblxyXG4gIHN0YXRpYyBnZXQgSW5zdGFuY2UoKTogQ2xpZW50TWFuYWdlciB7XHJcbiAgICBpZiAoQ2xpZW50TWFuYWdlci5faW5zdGFuY2UgPT09IG51bGwpXHJcbiAgICAgIENsaWVudE1hbmFnZXIuX2luc3RhbmNlID0gbmV3IENsaWVudE1hbmFnZXIoKTtcclxuXHJcbiAgICByZXR1cm4gQ2xpZW50TWFuYWdlci5faW5zdGFuY2U7XHJcbiAgfVxyXG5cclxuICBwdWJsaWMgQWRkRXZlbnQoZ3VpRXZlbnQ6IElHdWlFdmVudCk6IHZvaWQge1xyXG4gICAgbGV0IHRhc2s6IFRhc2sgPSA8VGFzaz5NR0RhdGFDb2xsZWN0aW9uLkluc3RhbmNlLkdldFRhc2tCeUlEKGd1aUV2ZW50LlRhc2tJRCk7XHJcbiAgICBsZXQgbGluZUlkeDogbnVtYmVyID0gK2d1aUV2ZW50LkxpbmU7XHJcbiAgICBsZXQgZHZSb3dJZHg6IG51bWJlciA9IDA7XHJcblxyXG4gICAgaWYgKHRhc2sgIT0gbnVsbCkge1xyXG4gICAgICBsZXQgY29udHJvbDogTWdDb250cm9sQmFzZSA9IG51bGw7XHJcblxyXG4gICAgICBpZiAoZ3VpRXZlbnQuQ29udHJvbE5hbWUgIT09IG51bGwpXHJcbiAgICAgICAgY29udHJvbCA9IHRhc2suZ2V0Rm9ybSgpLkdldEN0cmwoZ3VpRXZlbnQuQ29udHJvbE5hbWUpO1xyXG5cclxuICAgICAgaWYgKGNvbnRyb2wgIT0gbnVsbCkge1xyXG4gICAgICAgIGxldCBkdjogRGF0YVZpZXcgPSA8RGF0YVZpZXc+dGFzay5EYXRhVmlldztcclxuICAgICAgICBpZiAoY29udHJvbC5Jc1RhYmxlQ2hpbGQpIHtcclxuICAgICAgICAgIGxpbmVJZHggLT0gZHYuUmVjb3Jkc0JlZm9yZUN1cnJlbnRWaWV3O1xyXG4gICAgICAgIH1cclxuICAgICAgICBzd2l0Y2ggKGd1aUV2ZW50LkV2ZW50VHlwZSkge1xyXG5cclxuICAgICAgICAgIGNhc2UgXCJjb2x1bW5Tb3J0XCI6XHJcbiAgICAgICAgICAgIGlmIChjb250cm9sLmlzQ29sdW1uU29ydGFibGUoKSkge1xyXG4gICAgICAgICAgICAgIEV2ZW50cy5PbkNvbHVtbkNsaWNrKGNvbnRyb2wsIGd1aUV2ZW50LkxpbmUsIGd1aUV2ZW50LkNvbnRyb2xOYW1lKTtcclxuICAgICAgICAgICAgfVxyXG4gICAgICAgICAgICBicmVhaztcclxuICAgICAgICAgIGNhc2UgXCJtb3VzZWRvd25cIjpcclxuICAgICAgICAgICAgLy8gRGVmZWN0IDE1MjI4OCAmIERlZmVjdCAxNTIyNjg6IFByb2JsZW06IFdoZW4gYnV0dG9uIGlzIGNsaWNrZWQsIHdlIGdldCBmb2N1cyBldmVudCBmaXJzdCBhbmQgaWYgbWVzc2FnZUJveCBpcyBvcGVuZWQgaGVyZSBsaWtlIHZlcmlmeSwgZXRjLlxyXG4gICAgICAgICAgICAvLyB3ZSBkbyBub3QgcmVjZWl2ZSBjbGljayBldmVudC4gSGVuY2UgZXZlbnRzIGFyZSBub3QgcHJvY2Vzc2VkLiBTb2x1dGlvbjogV2UgZG8gbmVlZCBjbGljayBoYW5kbGluZyBmb3IgYnV0dG9uIGFzIGl0IGlzIGNhbGxlZCB3aGVuIHdlIHBhcmtcclxuICAgICAgICAgICAgLy8gb24gYnV0dG9uIGFuZCBwcmVzcyBzcGFjZSBrZXkuIFdlIG5lZWQgZm9jdXMgZXZlbnQgaGFuZGxpbmcgZm9yIGV4ZWN1dGluZyBDUCAmIENTIHdoZW4gcGFya2VkIG9uIGJ1dHRvbiBieSB0YWIga2V5LiBTbyB3ZSBuZWVkIHRvIG1hbmFnZVxyXG4gICAgICAgICAgICAvLyBob3cgY2xpY2sgaXMgaGFuZGxlZC4gV2UgZG8gbm90IG5lZWQgZm9jdXMgZXZlbnQgaGFuZGxpbmcgd2hlbiB3ZSBjbGljayBvbiBidXR0b24sIHNvIG9uIG1vdXNlZG93biwgc2V0IGEgZmxhZyB3aGljaCB3aWxsIGJlIHVzZWQgaW4gZm9jdXNcclxuICAgICAgICAgICAgLy8gZXZlbnQgaGFuZGxpbmcgdG8gc2tpcCBpdHMgaGFuZGxpbmcuIFRoZW4gd2Ugd2lsbCByZWNlaXZlIGNsaWNrIGV2ZW50IHdoZXJlIHRoaXMgd2lsbCBiZSBhY3R1YWxseSBoYW5kbGVkIGFuZCBoZW5jZSB3b3JrLlxyXG4gICAgICAgICAgICBpZiAoY29udHJvbC5pc0J1dHRvbigpKSB7XHJcbiAgICAgICAgICAgICAgdGhpcy5fYnV0dG9uSXNDbGlja2VkID0gdHJ1ZTtcclxuICAgICAgICAgICAgfVxyXG4gICAgICAgICAgICBlbHNlIGlmICghY29udHJvbC5pc1N1YmZvcm0oKSlcclxuICAgICAgICAgICAgICBFdmVudHMuR2V0UnVudGltZUNvbnRleHQoTWFuYWdlci5HZXRDdXJyZW50Q29udGV4dElEKCkpLkxhc3RDbGlja2VkQ3RybE5hbWUgPSBjb250cm9sLmdldE5hbWUoKTtcclxuICAgICAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICAgICAgY2FzZSBcImNsaWNrXCI6XHJcbiAgICAgICAgICAgIGlmIChjb250cm9sLmlzQnV0dG9uKCkgfHwgKGNvbnRyb2wuQ29ubmVjdGVkQ29udHJvbCAmJiBndWlFdmVudC5mcm9tQnV0dG9uKSl7XHJcbiAgICAgICAgICAgICAgRXZlbnRzLk9uU2VsZWN0aW9uKFwiXCIsIGNvbnRyb2wsIGxpbmVJZHgsIHRydWUpO1xyXG4gICAgICAgICAgICAgIEV2ZW50cy5HZXRSdW50aW1lQ29udGV4dChNYW5hZ2VyLkdldEN1cnJlbnRDb250ZXh0SUQoKSkuTGFzdENsaWNrZWRDdHJsTmFtZSA9IGNvbnRyb2wuZ2V0TmFtZSgpO1xyXG4gICAgICAgICAgICAgIHRoaXMuX2J1dHRvbklzQ2xpY2tlZCA9IGZhbHNlO1xyXG4gICAgICAgICAgICB9XHJcbiAgICAgICAgICAgIC8vIHNvcnRpbmcgc2hvdWxkIGJlIGhhbmRsZWQgb25seSBvbiBjb2x1bW4gY2xpY2sgYW5kIG5vdCBvbiByb3dEYXRhXHJcbiAgICAgICAgICAgIC8vIGVsc2UgaWYgKGNvbnRyb2wuVHlwZSA9PT0gTWdDb250cm9sVHlwZS5DVFJMX1RZUEVfQ09MVU1OKSB7XHJcbiAgICAgICAgICAgIC8vICAgRXZlbnRzLk9uQ29sdW1uQ2xpY2soY29udHJvbCwgZ3VpRXZlbnQuTGluZSwgZ3VpRXZlbnQuQ29udHJvbE5hbWUpO1xyXG4gICAgICAgICAgICAvLyB9XHJcbiAgICAgICAgICAgIGVsc2Uge1xyXG4gICAgICAgICAgICAgIEV2ZW50cy5PbkZvY3VzKGNvbnRyb2wsIGxpbmVJZHgsIHRydWUsIGZhbHNlKTtcclxuICAgICAgICAgICAgICBsZXQgcHJvZHVjZUNsaWNrOiBib29sZWFuID0gY29udHJvbC5pc0J1dHRvbigpIHx8IGNvbnRyb2wuaXNUZXh0Q29udHJvbCgpIHx8IGNvbnRyb2wuaXNMYWJlbCgpIHx8IGNvbnRyb2wuaXNUYWJDb250cm9sKCkgfHwgY29udHJvbC5pc1RhYmxlQ29udHJvbCgpIHx8IGNvbnRyb2wuaXNDb21ib0JveCgpIHx8IGNvbnRyb2wuaXNJbWFnZUNvbnRyb2woKTtcclxuICAgICAgICAgICAgICBFdmVudHMuT25Nb3VzZURvd24oY29udHJvbC5nZXRGb3JtKCksIGNvbnRyb2wsIG51bGwsIHRydWUsIGxpbmVJZHgsIGZhbHNlLCBwcm9kdWNlQ2xpY2spO1xyXG4gICAgICAgICAgICB9XHJcbiAgICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgICAgY2FzZSBcImZvY3VzXCI6XHJcblxyXG4gICAgICAgICAgICBpZiAodGhpcy5fYnV0dG9uSXNDbGlja2VkKVxyXG4gICAgICAgICAgICAgIHJldHVybjtcclxuXHJcbiAgICAgICAgICAgIC8vIC8vIElmIHdlIGF0dGVtcHQgdG8gcGFyayBvbiBjb250cm9sIHdoZXJlIHdlIGRvIG5vdCB3YW50IHRvIGFsbG93IHRvIHBhcmsgbGlrZSBpZiBjb250cm9sIGhhcyBleHByZXNzaW9uIG9yIHBhcmVudCB0YXNrIGZpZWxkLCBkbyBub3QgYWxsb3cgdG8gZm9jdXMgb24gaXQuXHJcbiAgICAgICAgICAgIC8vIC8vIFNldCBGb2N1cyBiYWNrIHRvIGxhc3RGb2N1c2VkQ29udHJvbC4gV2UgbmVlZCBfbGFzdEZhaWxlZEZvY3VzZWRDb250cm9sIGFzIHdlIGdldCBmb2N1cyBldmVudCBvbiBhIGNvbnRyb2wgdHdpY2UuIFNvIHRvIGF2b2lkIHRvIHNob3cgZXJyb3IgbXVsdGlwbGUgdGltZXMsXHJcbiAgICAgICAgICAgIC8vIC8vIG1haW50YWluIF9sYXN0RmFpbGVkRm9jdXNlZENvbnRyb2wuXHJcbiAgICAgICAgICAgIC8vIGlmIChjb250cm9sICE9PSB0aGlzLl9sYXN0RmFpbGVkRm9jdXNlZENvbnRyb2wgJiYgY29udHJvbC5IYXNFeHBPclBhcmVudFRhc2tGaWVsZCgpKSB7XHJcbiAgICAgICAgICAgIC8vICAgLy8gU2hvdyBlcnJvciBvbmx5IGZvciBjb250cm9scyBoYXZpbmcgRGF0YSBwcm9wZXJ0eSBhbmQgcGFya2FibGVcclxuICAgICAgICAgICAgLy8gICBpZihjb250cm9sLmlzTGlzdEJveCgpIHx8IGNvbnRyb2wuaXNDaGVja0JveCgpIHx8IGNvbnRyb2wuaXNDb21ib0JveCgpIHx8IGNvbnRyb2wuaXNUZXh0Q29udHJvbCgpIHx8IGNvbnRyb2wuaXNSYWRpbygpIHx8IGNvbnRyb2wuaXNUYWJDb250cm9sKCkpIHtcclxuICAgICAgICAgICAgLy8gICAgIENvbW1hbmRzLm1lc3NhZ2VCb3gobnVsbCwgXCJFcnJvclwiLCBcIkNvbnRyb2wgZWl0aGVyIGhhcyBleHByZXNzaW9uIG9yIGhhcyBwYXJlbnQgdGFzayB2YXJpYWJsZSBhdHRhY2hlZCB0byBpdC4gRmFpbGVkIHRvIHBhcmsgb24gY29udHJvbCB3aXRoIG5hbWU6IFwiICsgY29udHJvbC5OYW1lLCBTdHlsZXMuTVNHQk9YX0lDT05fRVJST1IgfCBTdHlsZXMuTVNHQk9YX0JVVFRPTl9PSyk7XHJcbiAgICAgICAgICAgIC8vICAgICAvLyBEZWZlY3QgMTUyNzI4OiBVc2UgbGFzdCBmb2N1c2VkIHRhc2sgaW5zdGVhZCBjb250cm9sIHRhc2sgYXMgY29udHJvbCBjYW4gYmUgZnJvbSBzdWJmb3JtLlxyXG4gICAgICAgICAgICAvLyAgICAgRm9jdXNNYW5hZ2VyLlNldEZvY3VzKExhc3RGb2N1c2VkTWFuYWdlci5JbnN0YW5jZS5nZXRMYXN0Rm9jdXNlZFRhc2soKSwgTGFzdEZvY3VzZWRNYW5hZ2VyLmdldExhc3RGb2N1c2VkQ29udHJvbCgpLCAwLFxyXG4gICAgICAgICAgICAvLyAgICAgICAgICAgICAgIExhc3RGb2N1c2VkTWFuYWdlci5nZXRMYXN0Rm9jdXNlZENvbnRyb2woKSA9PT0gbnVsbCA/IHRydWUgOiBmYWxzZSk7XHJcbiAgICAgICAgICAgIC8vICAgICB0aGlzLl9sYXN0RmFpbGVkRm9jdXNlZENvbnRyb2wgPSBjb250cm9sO1xyXG4gICAgICAgICAgICAvLyAgICAgcmV0dXJuO1xyXG4gICAgICAgICAgICAvLyAgIH1cclxuICAgICAgICAgICAgLy8gfVxyXG4gICAgICAgICAgICAvL1xyXG4gICAgICAgICAgICAvLyB0aGlzLl9sYXN0RmFpbGVkRm9jdXNlZENvbnRyb2wgPSBudWxsO1xyXG5cclxuICAgICAgICAgICAgRXZlbnRzLk9uRm9jdXMoY29udHJvbCwgbGluZUlkeCwgdHJ1ZSwgZmFsc2UpO1xyXG5cclxuICAgICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgICBjYXNlIFwiZGJsY2xpY2tcIjpcclxuICAgICAgICAgICAgRXZlbnRzLk9uRGJsQ2xpY2soY29udHJvbCwgbGluZUlkeCk7XHJcbiAgICAgICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgICAgIGNhc2UgXCJzZWxlY3Rpb25jaGFuZ2VkXCI6XHJcbiAgICAgICAgICAgIEV2ZW50cy5PblNlbGVjdGlvbihndWlFdmVudC5WYWx1ZSwgY29udHJvbCwgbGluZUlkeCwgdHJ1ZSk7XHJcbiAgICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgICAgY2FzZSBcImtleWRvd25cIjpcclxuICAgICAgICAgICAgRXZlbnRzLk9uS2V5RG93biggdGFzay5nZXRGb3JtKCksIGNvbnRyb2wsIGd1aUV2ZW50Lm1vZGlmaWVycywgZ3VpRXZlbnQua2V5Q29kZSwgMCwwLCAnJyxmYWxzZSwnJyx0cnVlICk7XHJcbiAgICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgICAgY2FzZSBcInNldGV4dGVybmFsdmFsdWVcIjpcclxuICAgICAgICAgICAgRXZlbnRzTWFuYWdlci5JbnN0YW5jZS5BZGRFeHRlcm5hbFZhbHVlRXZlbnQoKDxNZ0NvbnRyb2w+Y29udHJvbCksIGd1aUV2ZW50LmV4dGVybmFsVmFsdWUsIGd1aUV2ZW50LnBhcmFtKTtcclxuICAgICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgICBkZWZhdWx0OlxyXG4gICAgICAgICAgICAvLyBjb25zb2xlLmxvZyhcInVuaGFuZGxlZCBldmVudDogXFxcIlwiICsgZ3VpRXZlbnQuRXZlbnRUeXBlICsgXCJcXFwiXCIpO1xyXG4gICAgICAgICAgICBicmVhaztcclxuICAgICAgICB9XHJcbiAgICAgIH1cclxuICAgICAgZWxzZSB7XHJcbiAgICAgICAgc3dpdGNoIChndWlFdmVudC5FdmVudFR5cGUpIHtcclxuICAgICAgICAgIGNhc2UgXCJmb2N1c09uTm9uTWFnaWNDb250cm9sXCI6XHJcbiAgICAgICAgICAgIGNvbnRyb2wgPSBMYXN0Rm9jdXNlZE1hbmFnZXIuZ2V0TGFzdEZvY3VzZWRDb250cm9sKCk7XHJcbiAgICAgICAgICAgIGlmIChjb250cm9sICE9PSBudWxsKSB7XHJcbiAgICAgICAgICAgICAgRXZlbnRzLk9uRm9jdXNPbk5vbk1hZ2ljQ29udHJvbChjb250cm9sKTtcclxuICAgICAgICAgICAgfVxyXG4gICAgICAgICAgICBicmVhaztcclxuICAgICAgICAgIGNhc2UgXCJjbGlja1wiOlxyXG4gICAgICAgICAgICBpZihndWlFdmVudC5Db250cm9sTmFtZSA9PT0gXCJtYWdpY1Jvd1wiKSB7XHJcbiAgICAgICAgICAgICAgZHZSb3dJZHggPSBndWlFdmVudC5MaW5lIC0gKDxEYXRhVmlldz50YXNrLkRhdGFWaWV3KS5SZWNvcmRzQmVmb3JlQ3VycmVudFZpZXc7XHJcbiAgICAgICAgICAgICAgRXZlbnRzTWFuYWdlci5JbnN0YW5jZS5hZGRHdWlUcmlnZ2VyZWRFdmVudFdpdGhUYXNrQW5kQ29kZUFuZExpbmUodGFzaywgSW50ZXJuYWxJbnRlcmZhY2UuTUdfQUNUX1JFQ19QUkVGSVgsIGR2Um93SWR4KTtcclxuICAgICAgICAgICAgfVxyXG4gICAgICAgICAgICBicmVhaztcclxuICAgICAgICAgIGNhc2UgXCJuYXZpZ2F0ZUluVGFibGVcIjpcclxuICAgICAgICAgICAgaWYgKGd1aUV2ZW50LkNvbnRyb2xOYW1lID09PSBcIm1hZ2ljUm93XCIpXHJcbiAgICAgICAgICAgICAgdGhpcy5oYW5kbGVOYXZpZ2F0ZUluUm93RXZlbnQodGFzaywgZ3VpRXZlbnQuVmFsdWUpO1xyXG4gICAgICAgICAgICBicmVhaztcclxuICAgICAgICAgIGNhc2UgXCJjbG9zZVwiOlxyXG4gICAgICAgICAgICBpZiAodGFzay5Jc1JvdXRlKCkpXHJcbiAgICAgICAgICAgICAgRXZlbnRzTWFuYWdlci5JbnN0YW5jZS5BZGRSb3V0ZXJDbG9zZWRFdmVudCh0YXNrKTtcclxuICAgICAgICAgICAgZWxzZVxyXG4gICAgICAgICAgICAgIEV2ZW50cy5DbG9zZUZvcm1FdmVudCh0YXNrLmdldEZvcm0oKSk7XHJcbiAgICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgICAgY2FzZSBcImRpc3Bvc2VcIjpcclxuICAgICAgICAgICAgTUdEYXRhQ29sbGVjdGlvbi5JbnN0YW5jZS5kZWxldGVNR0RhdGFUcmVlKHRhc2suZ2V0TWdkSUQoKSk7XHJcbiAgICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgICAgY2FzZSBcInJlc2l6ZVwiOlxyXG4gICAgICAgICAgICBFdmVudHMuVGFibGVSZXNpemVFdmVudCh0YXNrLmdldEZvcm0oKS5nZXRUYWJsZUN0cmwoKSwgZ3VpRXZlbnQuUGFnZVNpemUpO1xyXG4gICAgICAgICAgLy9jYXNlICdicmVhaycgaXMgbWlzc2luZyBmb3IgJ3Jlc2l6ZScgd2l0aCBpbnRlbnRpb24gdG8gZXhlY3V0ZSBnZXRSb3dzIHdpdGggdGhlIGRlc2lyZWQgZGF0YXZpZXcgcm93IGluZGV4LlxyXG4gICAgICAgICAgY2FzZSBcImdldFJvd3NcIjpcclxuICAgICAgICAgICAgZHZSb3dJZHggPSBsaW5lSWR4IC0gKDxEYXRhVmlldz50YXNrLkRhdGFWaWV3KS5SZWNvcmRzQmVmb3JlQ3VycmVudFZpZXc7XHJcbiAgICAgICAgICAgIGxldCBsYXN0Rm9jdXNlZFZhbDogTGFzdEZvY3VzZWRWYWwgPSB7Z3VpTWdDb250cm9sOiB0YXNrLmdldEZvcm0oKS5nZXRUYWJsZUN0cmwoKSwgTGluZTogZHZSb3dJZHgsIFZhbDogXCJcIn07XHJcbiAgICAgICAgICAgIEV2ZW50cy5PbkdldFJvd3NEYXRhKHRhc2suZ2V0Rm9ybSgpLmdldFRhYmxlQ3RybCgpLCBkdlJvd0lkeCwgdHJ1ZSwgbGFzdEZvY3VzZWRWYWwpO1xyXG4gICAgICAgICAgICBicmVhaztcclxuICAgICAgICAgIGNhc2UgXCJSb3V0ZXJOYXZpZ2F0ZVwiOlxyXG4gICAgICAgICAgICBFdmVudHNNYW5hZ2VyLkluc3RhbmNlLkFkZFJvdXRlRXZlbnQodGFzaywgSW50ZXJuYWxJbnRlcmZhY2UuTUdfQUNUX1dFQkNMSUVOVF9ST1VURSwgZ3VpRXZlbnQuUm91dGVyUGF0aCwgZ3VpRXZlbnQuUm91dGVyT3V0bGV0TmFtZSxcclxuICAgICAgICAgICAgICBndWlFdmVudC5Sb3V0ZXJQYXJhbXMpO1xyXG4gICAgICAgICAgICBicmVhaztcclxuICAgICAgICAgIGNhc2UgXCJzZXRleHRlcm5hbHZhbHVlXCI6XHJcbiAgICAgICAgICAgIExvZ2dlci5JbnN0YW5jZS5Xcml0ZUVycm9yVG9Mb2coTlN0cmluZy5Gb3JtYXQoXCJVbmFibGUgdG8gZmluZCB7MH06IFRyeWluZyB0byBzZXQgdmFsdWUgZm9yIGEgbm9uLWV4aXN0aW5nIGNvbnRyb2wuXCIsIGd1aUV2ZW50LkNvbnRyb2xOYW1lKSk7XHJcbiAgICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgICAgZGVmYXVsdDpcclxuICAgICAgICAgICAgLy8gY29uc29sZS5sb2coIFwidW5oYW5kbGVkIGV2ZW50OiBcXFwiXCIgKyBndWlFdmVudC5FdmVudFR5cGUgKyBcIlxcXCJcIik7XHJcbiAgICAgICAgICAgIGJyZWFrO1xyXG4gICAgICAgIH1cclxuICAgICAgfVxyXG4gICAgfVxyXG4gICAgZWxzZSAvL3Rhc2sgPT0gbnVsbFxyXG4gICAgICBzd2l0Y2ggKGd1aUV2ZW50LkV2ZW50VHlwZSkge1xyXG4gICAgICAgIGNhc2UgXCJjb25maXJtYXRpb25DbG9zZWRcIjpcclxuICAgICAgICAgIHRoaXMuQ29uZmlybWF0aW9uQ2xvc2VkKGd1aUV2ZW50LmtleUNvZGUpO1xyXG4gICAgICAgICAgYnJlYWs7XHJcbiAgICAgIH1cclxuICB9XHJcblxyXG4gIHByaXZhdGUgaGFuZGxlTmF2aWdhdGVJblJvd0V2ZW50KHRhc2s6IFRhc2ssIGV2dDogc3RyaW5nKSB7XHJcbiAgICBsZXQgaW50ZXJuYWxFdnQgPSBJbnRlcm5hbEludGVyZmFjZS5NR19BQ1RfTk9ORTtcclxuICAgIGV2dCA9IGV2dC50b0xvd2VyQ2FzZSgpO1xyXG5cclxuICAgIHN3aXRjaCAoZXZ0KSB7XHJcbiAgICAgIGNhc2UgXCJiZWdpbi1wYWdlXCI6XHJcbiAgICAgICAgaW50ZXJuYWxFdnQgPSBJbnRlcm5hbEludGVyZmFjZS5NR19BQ1RfVEJMX0JFR1BBR0U7XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlIFwicHJldmlvdXMtcGFnZVwiOlxyXG4gICAgICAgIGludGVybmFsRXZ0ID0gSW50ZXJuYWxJbnRlcmZhY2UuTUdfQUNUX1RCTF9QUlZQQUdFO1xyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSBcImVuZC1wYWdlXCI6XHJcbiAgICAgICAgaW50ZXJuYWxFdnQgPSBJbnRlcm5hbEludGVyZmFjZS5NR19BQ1RfVEJMX0VORFBBR0U7XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlIFwibmV4dC1wYWdlXCI6XHJcbiAgICAgICAgaW50ZXJuYWxFdnQgPSBJbnRlcm5hbEludGVyZmFjZS5NR19BQ1RfVEJMX05YVFBBR0U7XHJcbiAgICAgICAgYnJlYWs7XHJcblxyXG4gICAgICBjYXNlIFwicHJldmlvdXMtbGluZVwiOlxyXG4gICAgICAgIGludGVybmFsRXZ0ID0gSW50ZXJuYWxJbnRlcmZhY2UuTUdfQUNUX1RCTF9QUlZMSU5FO1xyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSBcIm5leHQtbGluZVwiOlxyXG4gICAgICAgIGludGVybmFsRXZ0ID0gSW50ZXJuYWxJbnRlcmZhY2UuTUdfQUNUX1RCTF9OWFRMSU5FO1xyXG4gICAgICAgIGJyZWFrO1xyXG5cclxuICAgICAgY2FzZSBcImJlZ2luLXRhYmxlXCI6XHJcbiAgICAgICAgaW50ZXJuYWxFdnQgPSBJbnRlcm5hbEludGVyZmFjZS5NR19BQ1RfVEJMX0JFR1RCTDtcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGNhc2UgXCJlbmQtdGFibGVcIjpcclxuICAgICAgICBpbnRlcm5hbEV2dCA9IEludGVybmFsSW50ZXJmYWNlLk1HX0FDVF9UQkxfRU5EVEJMO1xyXG4gICAgICAgIGJyZWFrO1xyXG4gICAgfVxyXG5cclxuICAgIGlmIChpbnRlcm5hbEV2dCAhPSBJbnRlcm5hbEludGVyZmFjZS5NR19BQ1RfTk9ORSlcclxuICAgICAgRXZlbnRzTWFuYWdlci5JbnN0YW5jZS5hZGRHdWlUcmlnZ2VyZWRFdmVudFdpdGhUYXNrQW5kQ29kZSh0YXNrLCBpbnRlcm5hbEV2dCk7XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vIHJhaXNlIGNhbGxiYWNrIG9uIGNsb3NpbmcgY29uZmlybWF0aW9uXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICBDb25maXJtYXRpb25DbG9zZWQocmVzOiBudW1iZXIpOiB2b2lkXHJcbiAge1xyXG4gICAgQ29tbWFuZHMuUmVzb2x2ZVByb21pc2UocmVzKTtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gdGhlIHdvcmsgdGhyZWFkJ3MgZW50cnkgbWV0aG9kXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICBhc3luYyBXb3JrVGhyZWFkRXhlY3V0aW9uKCk6IFByb21pc2U8dm9pZD4ge1xyXG4gICAgaWYgKExvZ2dlci5JbnN0YW5jZS5TaG91bGRMb2coKSkge1xyXG4gICAgICBMb2dnZXIuSW5zdGFuY2UuV3JpdGVUb0xvZyhcIlwiLCB0cnVlKTtcclxuICAgICAgTG9nZ2VyLkluc3RhbmNlLldyaXRlVG9Mb2coXCJTdGFydGVkIG9uIFwiICsgRGF0ZVRpbWVVdGlscy5Ub1N0cmluZyhEYXRlVGltZS5Ob3csIFhNTENvbnN0YW50cy5FUlJPUl9MT0dfREFURV9GT1JNQVQsIExvZ2dlci5JbnN0YW5jZSksIHRydWUpO1xyXG4gICAgICB0aGlzLldyaXRlRXhlY3V0aW9uUHJvcGVydGllc1RvTG9nKCk7XHJcbiAgICB9XHJcblxyXG4gICAgdHJ5IHtcclxuICAgICAgaWYgKGF3YWl0IENvbW1hbmRzUHJvY2Vzc29yTWFuYWdlci5TdGFydFNlc3Npb24oKSkge1xyXG4gICAgICAgIGxldCBtZ0RhdGFUYWI6IE1HRGF0YUNvbGxlY3Rpb24gPSBNR0RhdGFDb2xsZWN0aW9uLkluc3RhbmNlO1xyXG4gICAgICAgIGxldCBzdGFydHVwTWdEYXRhOiBNR0RhdGEgPSBtZ0RhdGFUYWIuU3RhcnR1cE1nRGF0YTtcclxuICAgICAgICBtZ0RhdGFUYWIuY3Vyck1nZElEID0gc3RhcnR1cE1nRGF0YS5HZXRJZCgpO1xyXG5cclxuICAgICAgICBsZXQgbm9uSW50ZXJhY3RpdmVUYXNrOiBUYXNrID0gbnVsbDtcclxuICAgICAgICBpZiAoIXN0YXJ0dXBNZ0RhdGEuSXNBYm9ydGluZylcclxuICAgICAgICAgIG5vbkludGVyYWN0aXZlVGFzayA9IGF3YWl0IFRhc2suU3RhcnRQcm9ncmFtKGZhbHNlLCBmYWxzZSk7XHJcblxyXG4gICAgICAgIGlmIChub25JbnRlcmFjdGl2ZVRhc2sgPT09IG51bGwpXHJcbiAgICAgICAgICBhd2FpdCBFdmVudHNNYW5hZ2VyLkluc3RhbmNlLkV2ZW50c0xvb3Aoc3RhcnR1cE1nRGF0YSk7XHJcbiAgICAgICAgZWxzZVxyXG4gICAgICAgICAgYXdhaXQgRXZlbnRzTWFuYWdlci5JbnN0YW5jZS5Ob25JbnRlcmFjdGl2ZUV2ZW50c0xvb3Aoc3RhcnR1cE1nRGF0YSwgbm9uSW50ZXJhY3RpdmVUYXNrKTtcclxuXHJcbiAgICAgICAgLy8gc2V0IHRoZSBcImVuZCBvZiB3b3JrXCIgZmxhZyBvbmx5IGZvciB0aGUgbWFpbiB3aW5kb3dcclxuICAgICAgICBFdmVudHNNYW5hZ2VyLkluc3RhbmNlLnNldEVuZE9mV29yayh0cnVlKTtcclxuICAgICAgfVxyXG4gICAgfVxyXG4gICAgY2F0Y2ggKGV4KSB7XHJcbiAgICAgIGlmIChleCBpbnN0YW5jZW9mIEFwcGxpY2F0aW9uRXhjZXB0aW9uKSB7XHJcbiAgICAgICAgbGV0IGlzTm9SZXN1bHRFcnJvcjogYm9vbGVhbiA9IGZhbHNlO1xyXG4gICAgICAgIGlmIChleCBpbnN0YW5jZW9mIFNlcnZlckVycm9yICYmICg8U2VydmVyRXJyb3I+ZXgpLkdldENvZGUoKSA9PT0gU2VydmVyRXJyb3IuSU5GX05PX1JFU1VMVClcclxuICAgICAgICAgIGlzTm9SZXN1bHRFcnJvciA9IHRydWU7XHJcblxyXG4gICAgICAgIGlmICghaXNOb1Jlc3VsdEVycm9yKVxyXG4gICAgICAgICAgYXdhaXQgRXZlbnRzTWFuYWdlci5JbnN0YW5jZS5Qcm9jZXNzQWJvcnRpbmdFcnJvcihleCk7XHJcblxyXG4gICAgICAgIEV2ZW50c01hbmFnZXIuSW5zdGFuY2Uuc2V0RW5kT2ZXb3JrKHRydWUpO1xyXG4gICAgICB9XHJcbiAgICAgIGVsc2VcclxuICAgICAgICB0aHJvdyBleDtcclxuICAgIH1cclxuICAgIGZpbmFsbHkge1xyXG4gICAgICBhd2FpdCBFdmVudHNNYW5hZ2VyLkluc3RhbmNlLkNoZWNrQW5kU2hvd1NwaW5uZXIoZmFsc2UpO1xyXG5cclxuICAgICAgQ29tbWFuZHMuYWRkR2xvYmFsQ29tbWFuZChDb21tYW5kVHlwZS5FWElUX01BR0lDKTtcclxuICAgICAgQ29tbWFuZHMuYmVnaW5JbnZva2UoKTtcclxuICAgIH1cclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gV3JpdGUgRXhlY3V0aW9uIHByb3BlcnRpZXMgdG8gbG9nLlxyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgV3JpdGVFeGVjdXRpb25Qcm9wZXJ0aWVzVG9Mb2coKTogdm9pZCB7XHJcbiAgICBpZiAoTG9nZ2VyLkluc3RhbmNlLlNob3VsZExvZ1NlcnZlclJlbGF0ZWRNZXNzYWdlcygpKSB7XHJcbiAgICAgIExvZ2dlci5JbnN0YW5jZS5Xcml0ZVRvTG9nKFwiLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cIiwgdHJ1ZSk7XHJcbiAgICAgIC8vIFdyaXRlIGhlYWRpbmdcclxuICAgICAgTG9nZ2VyLkluc3RhbmNlLldyaXRlVG9Mb2coXCJFeGVjdXRpb24gUHJvcGVydGllczpcIiwgdHJ1ZSk7XHJcblxyXG4gICAgICBsZXQgcHJvcHNTdHJpbmcgPSBKU09OLnN0cmluZ2lmeShTZXJ2ZXJDb25maWcuSW5zdGFuY2UpO1xyXG4gICAgICBMb2dnZXIuSW5zdGFuY2UuV3JpdGVUb0xvZyhwcm9wc1N0cmluZywgdHJ1ZSk7XHJcblxyXG4gICAgICBMb2dnZXIuSW5zdGFuY2UuV3JpdGVUb0xvZyhcIi0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXCIsIHRydWUpO1xyXG4gICAgfVxyXG4gIH1cclxuXHJcblxyXG4gIGNvbnN0cnVjdG9yKCkge1xyXG4gICAgUnVudGltZUNvbnRleHRCYXNlLkluc3RhbmNlLkluaXQoUmVtb3RlQ29tbWFuZHNQcm9jZXNzb3IuUkNfTk9fQ09OVEVYVF9JRCk7XHJcbiAgICBMYXN0Rm9jdXNlZE1hbmFnZXIuSW5zdGFuY2UuSW5pdChNR0RhdGFDb2xsZWN0aW9uLkluc3RhbmNlKTtcclxuICAgIExhc3RGb2N1c2VkTWFuYWdlci5JbnN0YW5jZS5MYXN0QWN0aW9uVGltZSA9IE1pc2MuZ2V0U3lzdGVtTWlsbGlzZWNvbmRzKCk7XHJcblxyXG5cclxuICAgIFJlbW90ZUNvbW1hbmRzUHJvY2Vzc29yLlNob3VsZFNjcmFtYmxlQW5kVW5zY3JhbWJsZU1lc3NhZ2VzID0gdHJ1ZTtcclxuXHJcblxyXG4gICAgLy8gVE9ETzogSG93IHRvIGdldCBQcm9jZXNzLkdldEN1cnJlbnRQcm9jZXNzKCkuSWQ/XHJcbiAgICB0aGlzLl9nbG9iYWxVbmlxdWVTZXNzaW9uSWQgPSBVbmlxdWVJRFV0aWxzLkdldFVuaXF1ZU1hY2hpbmVJRCgpICsgXCJfXCI7IC8vICsgUHJvY2Vzcy5HZXRDdXJyZW50UHJvY2VzcygpLklkO1xyXG5cclxuICAgIHRoaXMuUmVnaXN0ZXJEZWxlZ2F0ZXMoKTtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT4gSW5pdGlhbGl6ZSB0aGUgZGF0YSBuZWVkZWQgYnkgdGhlIGd1aSBsaWJyYXJ5XHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICAvLy8gPHJldHVybnM+IENsaWVudE1hbmFnZXIgaW5zdGFuY2U8L3JldHVybnM+XHJcbiAgcHJpdmF0ZSBJbml0R3VpTWFuYWdlcigpOiB2b2lkIHtcclxuICAgIE1hbmFnZXIuRXZlbnRzTWFuYWdlciA9IEV2ZW50c01hbmFnZXIuSW5zdGFuY2U7XHJcbiAgICBNYW5hZ2VyLkRlZmF1bHRTZXJ2ZXJOYW1lID0gU2VydmVyQ29uZmlnLkluc3RhbmNlLmdldFNlcnZlcigpO1xyXG4gICAgTWFuYWdlci5EZWZhdWx0UHJvdG9jb2wgPSBTZXJ2ZXJDb25maWcuSW5zdGFuY2UuZ2V0UHJvdG9jb2woKTtcclxuICAgIEd1aUVudmlyb25tZW50LkVudmlyb25tZW50ID0gRW52aXJvbm1lbnQuSW5zdGFuY2U7XHJcbiAgICBHdWlEYXRhQ29sbGVjdGlvbi5NR0RhdGFUYWJsZSA9IE1HRGF0YUNvbGxlY3Rpb24uSW5zdGFuY2U7XHJcbiAgICBSdW50aW1lQ29udGV4dEJhc2Uubm9Db250ZXh0SWQgPSBSZW1vdGVDb21tYW5kc1Byb2Nlc3Nvci5SQ19OT19DT05URVhUX0lEO1xyXG4gIH1cclxuXHJcbiAgLy8vIEluaXRpYWxpemUgZ2xvYmFsIGRhdGEgaW4gb2JqZWN0cyB1c2VkIHRvIG92ZXJjb21lIHRoZSBjaXJjdWxhciByZWZlcmVuY2VzIGlzc3VlXHJcbiAgcHJpdmF0ZSBJbml0Q2lyY3VsYXJSZWZlcmVuY2VzQnlwYXNzZXMoKSB7XHJcbiAgICBUYXNrc1RhYmxlLmNyZWF0ZU5ld3Rhc2sgPSBUYXNrLkNyZWF0ZVRhc2s7XHJcbiAgICBPcGVuVVJMQ29tbWFuZC5jcmVhdGVOZXdDb21tYW5kc1RhYmxlID0gQ29tbWFuZHNUYWJsZS5jcmVhdGVDb21tYW5kc1RhYmxlO1xyXG5cclxuICAgIEFjY2Vzc0hlbHBlci5ldmVudHNNYW5hZ2VyID0gRXZlbnRzTWFuYWdlci5JbnN0YW5jZTtcclxuICAgIEFjY2Vzc0hlbHBlci5tZ0RhdGFUYWJsZSA9IE1HRGF0YUNvbGxlY3Rpb24uSW5zdGFuY2U7XHJcbiAgICBBY2Nlc3NIZWxwZXIuZW52UGFyYW1zVGFibGUgPSBFbnZQYXJhbXNUYWJsZS5JbnN0YW5jZTtcclxuICAgIEFjY2Vzc0hlbHBlci5nbG9iYWxQYXJhbXMgPSBHbG9iYWxQYXJhbXMuSW5zdGFuY2U7XHJcbiAgICBBY2Nlc3NIZWxwZXIubGFuZ3VhZ2VEYXRhID0gTGFuZ3VhZ2VEYXRhLkluc3RhbmNlO1xyXG4gICAgQWNjZXNzSGVscGVyLmVudmlyb25tZW50ID0gRW52aXJvbm1lbnQuSW5zdGFuY2U7XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+IEVudHJ5IHBvaW50IHRvIHRoZSBXZWIgQ2xpZW50IDwvc3VtbWFyeT5cclxuICAvLy8gPHBhcmFtIG5hbWU9XCJhcmdzXCI+YXJyYXkgb2YgY29tbWFuZCBsaW5lIGFyZ3VtZW50czwvcGFyYW0+XHJcbiAgc3RhdGljIE1haW4oaHR0cENsaWVudDogSHR0cENsaWVudCk6IHZvaWQge1xyXG4gICAgLy8gVE9ETyA6IENoZWNrIGhvdyB0byBsb2FkIGZyb20gU3R1ZGlvIGkuZS4gRjdcclxuICAgIC8vIGlmIChhcmdzLmxlbmd0aCAhPT0gMCAmJiBOU3RyaW5nLlN0YXJ0c1dpdGgoYXJnc1swXSwgXCIvc3Bhd25haGVhZFwiKSkge1xyXG4gICAgLy8gICBDbGllbnRNYW5hZ2VyLl9zdGFydGVkRnJvbVN0dWRpbyA9IHRydWU7XHJcbiAgICAvLyAgIFN0dWRpb0xhdW5jaGVyLkluaXQoYXJncyk7XHJcbiAgICAvLyB9XHJcbiAgICAvLyBlbHNlXHJcbiAgICBDbGllbnRNYW5hZ2VyLlN0YXJ0RXhlY3V0aW9uKGh0dHBDbGllbnQpO1xyXG5cclxuICAgIExvZ2dlci5JbnN0YW5jZS5GbHVzaCgpO1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PiBUZXJtaW5hdGUgdGhlIGNsaWVudCBhbmQgdGVybWluYXRlIHRoZSBjb250ZXh0IDwvc3VtbWFyeT5cclxuICBzdGF0aWMgYXN5bmMgU3RvcEV4ZWN1dGlvbigpOiBQcm9taXNlIDx2b2lkPiB7XHJcbiAgICB0cnkge1xyXG4gICAgICBIdHRwTWFuYWdlci5HZXRJbnN0YW5jZSgpLklzQWJvcnRpbmdNYWdpY0VuZ2luZSA9IHRydWU7XHJcbiAgICAgIE1HRGF0YUNvbGxlY3Rpb24uSW5zdGFuY2UuU3RhcnR1cE1nRGF0YS5nZXRGaXJzdFRhc2soKS5DbG9zaW5nRm9ybVVJID0gdHJ1ZTtcclxuXHJcbiAgICAgIGF3YWl0IE1HRGF0YUNvbGxlY3Rpb24uSW5zdGFuY2UuU3RhcnR1cE1nRGF0YS5nZXRGaXJzdFRhc2soKS5lbmRUYXNrKHRydWUsIGZhbHNlLFxyXG4gICAgICAgIGZhbHNlKTtcclxuICAgIH1cclxuICAgIGNhdGNoIChleDMpIHtcclxuICAgICAgaWYgKGV4MyBpbnN0YW5jZW9mIEFwcGxpY2F0aW9uRXhjZXB0aW9uKSB7XHJcbiAgICAgICAgTG9nZ2VyLkluc3RhbmNlLldyaXRlRXhjZXB0aW9uVG9Mb2coZXgzKTtcclxuICAgICAgfVxyXG4gICAgICBlbHNlXHJcbiAgICAgICAgdGhyb3cgZXgzO1xyXG4gICAgfVxyXG4gIH1cclxuXHJcbiAgc3RhdGljIGFzeW5jIFN0YXJ0RXhlY3V0aW9uKGh0dHBDbGllbnQ6SHR0cENsaWVudCk6IFByb21pc2U8dm9pZD4ge1xyXG4gICAgSHR0cE1hbmFnZXIuc2V0TmF0aXZlSHR0cENsaWVudChodHRwQ2xpZW50KTtcclxuICAgIGxldCBsb2FkZWQ6IGJvb2xlYW4gPSBhd2FpdCBDbGllbnRNYW5hZ2VyLkluc3RhbmNlLkxvYWRFeGVjdXRpb25Qcm9wcygpO1xyXG5cclxuICAgIGlmIChsb2FkZWQpIHtcclxuICAgICAgbmV3IEd1aUV2ZW50c1Byb2Nlc3NvcigpO1xyXG4gICAgICBDbGllbnRNYW5hZ2VyLkluc3RhbmNlLkluaXRHdWlNYW5hZ2VyKCk7XHJcbiAgICAgIENsaWVudE1hbmFnZXIuSW5zdGFuY2UuSW5pdENpcmN1bGFyUmVmZXJlbmNlc0J5cGFzc2VzKCk7XHJcbiAgICAgIGxldCB2ZXJzaW9uU3RyaW5nID0gQ3VycmVudENsaWVudFZlcnNpb24gPT09ICAnMC4wLjAuMCcgPyBcIkRldmVsb3BtZW50IG1vZGVcIiA6IFwiVmVyc2lvbiA6IFwiICsgQ3VycmVudENsaWVudFZlcnNpb24gO1xyXG5cclxuICAgICAgY29uc29sZS5sb2coXCJNYWdpYyBXZWIgQ2xpZW50LiBcIiArIHZlcnNpb25TdHJpbmcpO1xyXG5cclxuICAgICAgSHR0cE1hbmFnZXIuR2V0SW5zdGFuY2UoKS5IdHRwQ29tbXVuaWNhdGlvblRpbWVvdXRNUyA9IFNlcnZlckNvbmZpZy5JbnN0YW5jZS5HZXRGaXJzdEh0dHBSZXF1ZXN0VGltZW91dCgpO1xyXG5cclxuICAgICAgRXZlbnRzTWFuYWdlci5JbnN0YW5jZS5Jbml0KCk7XHJcblxyXG4gICAgICBMb2dnZXIuSW5zdGFuY2UuSW5pdGlhbGl6ZShSZW1vdGVDb21tYW5kc1Byb2Nlc3Nvci5HZXRJbnN0YW5jZSgpLnBhcnNlTG9nTGV2ZWwoU2VydmVyQ29uZmlnLkluc3RhbmNlLmdldEludGVybmFsTG9nTGV2ZWwoKSksZmFsc2UpO1xyXG5cclxuICAgICAgbGV0IG1nZDogTUdEYXRhID0gbmV3IE1HRGF0YShcclxuICAgICAgICBuZXcgQ29tbWFuZHNUYWJsZSgpLFxyXG4gICAgICAgIG5ldyBDb21tYW5kc1RhYmxlKCksXHJcbiAgICAgICAgMCwgbnVsbCwgZmFsc2UpO1xyXG4gICAgICBhd2FpdCBNR0RhdGFDb2xsZWN0aW9uLkluc3RhbmNlLmFkZE1HRGF0YShtZ2QsIDAsIHRydWUpO1xyXG5cclxuICAgICAgYXdhaXQgQ2xpZW50TWFuYWdlci5JbnN0YW5jZS5Xb3JrVGhyZWFkRXhlY3V0aW9uKCk7XHJcblxyXG4gICAgICBpZiAoTG9nZ2VyLkluc3RhbmNlLlNob3VsZExvZygpKSB7XHJcbiAgICAgICAgTG9nZ2VyLkluc3RhbmNlLldyaXRlVG9Mb2coT1NFbnZpcm9ubWVudC5Fb2xTZXEgKyBcIkVuZGVkIG9uIFwiICsgRGF0ZVRpbWVVdGlscy5Ub1N0cmluZyhEYXRlVGltZS5Ob3csIFhNTENvbnN0YW50cy5FUlJPUl9MT0dfREFURV9GT1JNQVQsIExvZ2dlci5JbnN0YW5jZSkgKyBPU0Vudmlyb25tZW50LkVvbFNlcSArIE9TRW52aXJvbm1lbnQuRW9sU2VxLCBmYWxzZSk7XHJcbiAgICAgIH1cclxuICAgIH1cclxuICAgIGVsc2VcclxuICAgIHtcclxuICAgICAgYXdhaXQgR1VJTWFuYWdlci5JbnN0YW5jZS5NZXNzYWdlQm94KFwiRXJyb3JcIiwgXCJGYWlsZWQgdG8gbG9hZCBzZXJ2ZXIgY29uZmlndXJhdGlvbiBmaWxlIChcIiArIHRoaXMuX2V4ZWN1dGlvblByb3BlcnRpZXNGaWxlTmFtZSArIFwiKVwiLCBTdHlsZXMuTVNHQk9YX0lDT05fRVJST1IgfCBTdHlsZXMuTVNHQk9YX0JVVFRPTl9PSyk7XHJcbiAgICB9XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+IExvYWRzIHRoZSBleGVjdXRpb24gcHJvcGVydGllcyBmaWxlIHRoYXQgY29udGFpbnMgaW5mb3JtYXRpb24gYWJvdXQgdGhlIHNlcnZlcixcclxuICAvLy8gYXBwbGljYXRpb24gYW5kIHByb2dyYW0gdG8gZXhlY3V0ZVxyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgYXN5bmMgTG9hZEV4ZWN1dGlvblByb3BzKCk6IFByb21pc2U8Ym9vbGVhbj4ge1xyXG4gICAgLy8gcmV0dXJuIHRydWUsIGlmIHN1Y2Nlc3NmdWxseSBsb2FkZWQuXHJcbiAgICBsZXQgcmVzcG9uc2U6IHN0cmluZyA9IFwiXCI7XHJcbiAgICBsZXQgc3VjY2VlZGVkOiBib29sZWFuID0gdHJ1ZTtcclxuXHJcbiAgICB0cnkge1xyXG4gICAgICAvLyBpc0Vycm9yIGNhbm5vdCBiZSB0cnVlIGhlcmUsIGJlY2F1c2UgaXQgaXMgcmV0dXJuZWQgYnkgcnVudGltZSBlbmdpbmUgYW5kIHRvIHJlYWQgZXhlY3V0aW9uIHByb3BlcnRpZXMsIHdlIGRvIG5vdCByZWFjaCB0aGUgZW5naW5lLlxyXG4gICAgICBsZXQgaXNFcnJvcjogUmVmUGFyYW08Ym9vbGVhbj4gPSBuZXcgUmVmUGFyYW0oZmFsc2UpO1xyXG4gICAgICByZXNwb25zZSA9IGF3YWl0IEh0dHBNYW5hZ2VyLkdldEluc3RhbmNlKCkuR2V0Q29udGVudChDbGllbnRNYW5hZ2VyLl9leGVjdXRpb25Qcm9wZXJ0aWVzRmlsZU5hbWUsIG51bGwsIGZhbHNlLCBpc0Vycm9yKTtcclxuICAgICAgU2VydmVyQ29uZmlnLkluc3RhbmNlLkluaXQoSlNPTi5wYXJzZShyZXNwb25zZSkpO1xyXG4gICAgfVxyXG4gICAgY2F0Y2ggKGV4KSB7XHJcbiAgICAgIHN1Y2NlZWRlZCA9IGZhbHNlO1xyXG4gICAgICAvLyBUT0RPOiBzaG91bGQgd2Ugd3JpdGUgdGhlIGVycm9yIG1lc3NhZ2UgdG8gbG9nP1xyXG4gICAgICAvLyBUT0RPOiBzaG91bGQgd2UgZGlzcGxheSBhbiBlcnJvciBtZXNzYWdlIG9uIHRoZSBicm93c2VyP1xyXG4gICAgfVxyXG5cclxuICAgIHJldHVybiBzdWNjZWVkZWQ7XHJcbiAgfVxyXG5cclxuXHJcblxyXG4gIC8vLyA8c3VtbWFyeT5nbG9iYWwgdW5pcXVlIElEIG9mIHRoZSBzZXNzaW9uID0gJ21hY2hpbmUgdW5pcXVlIElEJyArICdwcm9jZXNzIElEJzwvc3VtbWFyeT5cclxuICAvLy8gPHJldHVybnM+PC9yZXR1cm5zPlxyXG4gIEdldEdsb2JhbFVuaXF1ZVNlc3Npb25JRCgpOiBzdHJpbmcge1xyXG4gICAgcmV0dXJuIHRoaXMuX2dsb2JhbFVuaXF1ZVNlc3Npb25JZDtcclxuICB9XHJcblxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vICAgR2V0IGNvbnRyb2wncyBwaWN0dXJlXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICAvLy8gPHJldHVybnM+PC9yZXR1cm5zPlxyXG4gIHN0YXRpYyAgR2V0Q29udHJvbFBpY3R1cmVNYXNrKHRhc2tJZDogc3RyaW5nLCBjb250cm9sTmFtZTogc3RyaW5nKTogUElDXHJcbiAge1xyXG4gICAgbGV0IHBpYzogUElDID0gbnVsbDtcclxuICAgIGxldCB0YXNrOiBUYXNrID0gPFRhc2s+TUdEYXRhQ29sbGVjdGlvbi5JbnN0YW5jZS5HZXRUYXNrQnlJRCh0YXNrSWQpO1xyXG5cclxuICAgIGxldCBjb250cm9sOiBNZ0NvbnRyb2xCYXNlID0gbnVsbDtcclxuXHJcbiAgICBpZiAoY29udHJvbE5hbWUgIT09IG51bGwgJiYgdGFzayAhPSBudWxsKSB7XHJcbiAgICAgICAgY29udHJvbCA9IHRhc2suZ2V0Rm9ybSgpLkdldEN0cmwoY29udHJvbE5hbWUpO1xyXG4gICAgICAgIHBpYyA9IGNvbnRyb2wuZ2V0UElDKCk7XHJcbiAgICB9XHJcblxyXG4gICAgcmV0dXJuIHBpYztcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gICBHZXQgZm9ybWF0dGVkIHZhbHVlXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICAvLy8gPHJldHVybnM+PC9yZXR1cm5zPlxyXG4gIHN0YXRpYyAgR2V0Rm9ybWF0dGVkVmFsdWUodGFza0lkOiBzdHJpbmcsIGNvbnRyb2xOYW1lOiBzdHJpbmcsIHZhbHVlOiBhbnksIHJvd0lkPzogc3RyaW5nKTogc3RyaW5nXHJcbiAge1xyXG4gICAgbGV0IHBpYzogUElDID0gbnVsbDtcclxuICAgIGxldCB0YXNrOiBUYXNrID0gPFRhc2s+TUdEYXRhQ29sbGVjdGlvbi5JbnN0YW5jZS5HZXRUYXNrQnlJRCh0YXNrSWQpO1xyXG5cclxuICAgIGxldCBjb250cm9sOiBNZ0NvbnRyb2xCYXNlID0gbnVsbDtcclxuICAgIGxldCBuZXdWYWx1ZTogYW55ID0gdmFsdWU7XHJcblxyXG4gICAgaWYgKGNvbnRyb2xOYW1lICE9PSBudWxsICAmJiB0YXNrICE9PSBudWxsKVxyXG4gICAgICBjb250cm9sID0gdGFzay5nZXRGb3JtKCkuR2V0Q3RybChjb250cm9sTmFtZSk7XHJcbiAgICBpZiAoY29udHJvbCAhPSBudWxsICYmIGNvbnRyb2wuRGF0YVR5cGUgPT0gU3RvcmFnZUF0dHJpYnV0ZS5OVU1FUklDKSB7XHJcbiAgICAgIGlmIChjb250cm9sLmlzUmVwZWF0YWJsZSgpICYmICBjb250cm9sLmdldFByb3AoUHJvcEludGVyZmFjZS5QUk9QX1RZUEVfRk9STUFUKS5pc0V4cHJlc3Npb24oKSlcclxuICAgICAge1xyXG4gICAgICAgICAgbGV0IHBpY3ZhbCA9IGNvbnRyb2wuZ2V0UHJvcChQcm9wSW50ZXJmYWNlLlBST1BfVFlQRV9GT1JNQVQpLmdldFByZXZWYWx1ZSgrcm93SWQpO1xyXG4gICAgICAgICAgcGljID0gbmV3IFBJQyAocGljdmFsLCBTdG9yYWdlQXR0cmlidXRlLk5VTUVSSUMsICBjb250cm9sLmdldEZvcm0oKS5nZXRUYXNrKCkuZ2V0Q29tcElkeCgpKVxyXG4gICAgICB9XHJcbiAgICAgIGVsc2VcclxuICAgICAgICBwaWMgPSBjb250cm9sLmdldFBJQygpO1xyXG5cclxuICAgICAgbGV0IGRpc3BWYWx1ZSA9IFN0cmluZyh2YWx1ZSk7XHJcbiAgICAgIGxldCAgbmV3UGljID0gcGljLmdldFVuZm9ybWF0dGVkTnVtZXJpY1BpYygpO1xyXG5cclxuICAgICAgbGV0IG51bSA9IERpc3BsYXlDb252ZXJ0b3IuSW5zdGFuY2UudG9OdW0oZGlzcFZhbHVlLCBuZXdQaWMsIDApO1xyXG4gICAgICBuZXdWYWx1ZSA9IERpc3BsYXlDb252ZXJ0b3IuSW5zdGFuY2UuZnJvbU51bShudW0sIHBpYyk7XHJcbiAgICB9XHJcbiAgICByZXR1cm4gbmV3VmFsdWU7XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vICAgZ2V0IHJhbmdlZCB2YWx1ZSBmb3IgaW5wdXQgdmFsdWVcclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIHN0YXRpYyAgR2V0UmFuZ2VkVmFsdWUodGFza0lkOiBzdHJpbmcsIGNvbnRyb2xOYW1lOiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcpOiBzdHJpbmdcclxuICB7XHJcbiAgICBsZXQgdGFzazogVGFzayA9IDxUYXNrPk1HRGF0YUNvbGxlY3Rpb24uSW5zdGFuY2UuR2V0VGFza0J5SUQodGFza0lkKTtcclxuXHJcbiAgICBsZXQgY29udHJvbDogTWdDb250cm9sQmFzZSA9IG51bGw7XHJcblxyXG4gICAgaWYgKGNvbnRyb2xOYW1lICE9PSBudWxsKVxyXG4gICAge1xyXG4gICAgICBjb250cm9sID0gIHRhc2suZ2V0Rm9ybSgpLkdldEN0cmwoY29udHJvbE5hbWUpO1xyXG4gICAgICByZXR1cm4gY29udHJvbC5nZXRSYW5nZWRWYWx1ZSh2YWx1ZSk7XHJcbiAgICB9XHJcbiAgICByZXR1cm4gbnVsbDtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gICB2YWxpZGF0ZSB1c2VyIGlucHV0IHZhbHVlXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICBzdGF0aWMgIFZhbGlkYXRlQ29udHJvbFZhbHVlKHRhc2tJZDogc3RyaW5nLCBjb250cm9sTmFtZTogc3RyaW5nLCB2YWx1ZTogYW55KTogc3RyaW5nXHJcbiAge1xyXG4gICAgbGV0IHRhc2s6IFRhc2sgPSA8VGFzaz5NR0RhdGFDb2xsZWN0aW9uLkluc3RhbmNlLkdldFRhc2tCeUlEKHRhc2tJZCk7XHJcblxyXG4gICAgbGV0IGNvbnRyb2w6IE1nQ29udHJvbCA9IG51bGw7XHJcblxyXG4gICAgaWYgKGNvbnRyb2xOYW1lICE9PSBudWxsKVxyXG4gICAge1xyXG4gICAgICBjb250cm9sID0gIHRhc2suZ2V0Rm9ybSgpLkdldEN0cmwoY29udHJvbE5hbWUpIGFzIE1nQ29udHJvbDtcclxuICAgICAgcmV0dXJuIGNvbnRyb2wudmFsaWRhdGUodmFsdWUpO1xyXG4gICAgfVxyXG4gICAgcmV0dXJuIG51bGw7XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vICAgR2V0IGZsZCByYW5nZXNcclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIHN0YXRpYyBHZXRGbGRSYW5nZXModGFza0lkOiBzdHJpbmcsIGNvbnRyb2xOYW1lOiBzdHJpbmcpOiBzdHJpbmcge1xyXG4gICAgbGV0IHRhc2s6IFRhc2sgPSA8VGFzaz5NR0RhdGFDb2xsZWN0aW9uLkluc3RhbmNlLkdldFRhc2tCeUlEKHRhc2tJZCk7XHJcblxyXG4gICAgbGV0IGNvbnRyb2w6IE1nQ29udHJvbCA9IG51bGw7XHJcblxyXG4gICAgaWYgKGNvbnRyb2xOYW1lICE9PSBudWxsKVxyXG4gICAge1xyXG4gICAgICBjb250cm9sID0gIHRhc2suZ2V0Rm9ybSgpLkdldEN0cmwoY29udHJvbE5hbWUpIGFzIE1nQ29udHJvbDtcclxuICAgICAgcmV0dXJuIGNvbnRyb2wuZ2V0UmFuZ2VzKCk7XHJcbiAgICB9XHJcbiAgICByZXR1cm4gbnVsbDtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gICBSZXR1cm5zIHdoZXRoZXIgY2xpZW50L3NlcnZlciBjYWNoZSBleGlzdHNcclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIHN0YXRpYyBUcmFuc0NhY2hlRXhpc3RzKCk6IGJvb2xlYW4ge1xyXG4gICAgbGV0IGNsaWVudFRyYW5zQ2FjaGVFeGlzdHM6IGJvb2xlYW4gPSBBY2Nlc3NIZWxwZXIubWdEYXRhVGFibGUuZ2V0Q3Vyck1HRGF0YSgpLmNsaWVudFRyYW5zYWN0aW9uQ2FjaGVFeGlzdHMoKTtcclxuICAgIGxldCBzZXJ2ZXJUcmFuc2FjdGlvbkNhY2hlRXhpc3RzOiBib29sZWFuID0gQWNjZXNzSGVscGVyLm1nRGF0YVRhYmxlLmdldEN1cnJNR0RhdGEoKS5zZXJ2ZXJUcmFuc2FjdGlvbkNhY2hlRXhpc3RzKCk7XHJcblxyXG4gICAgcmV0dXJuIGNsaWVudFRyYW5zQ2FjaGVFeGlzdHMgfHwgc2VydmVyVHJhbnNhY3Rpb25DYWNoZUV4aXN0cztcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gU3Vic2NyaWJlIGZvciBldmVudHMgZGVmaW5lZCBhdCBIdHRwQ2xpZW50RXZlbnRzIGFuZFxyXG4gIC8vLyBhdHRhY2ggYXBwcm9wcmlhdGUgZXZlbnQgaGFuZGxlcnNcclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIC8vLyA8cmV0dXJucz48L3JldHVybnM+XHJcbiAgUmVnaXN0ZXJEZWxlZ2F0ZXMoKTogdm9pZCB7XHJcbiAgICBIdHRwQ2xpZW50RXZlbnRzLkdldEV4ZWN1dGlvblByb3BlcnR5X0V2ZW50ID0gU2VydmVyQ29uZmlnLkluc3RhbmNlLkdldEV4ZWN1dGlvblByb3BlcnR5LmJpbmQoU2VydmVyQ29uZmlnLkluc3RhbmNlKTtcclxuICAgIEh0dHBDbGllbnRFdmVudHMuR2V0UnVudGltZUN0eElEX0V2ZW50ID0gdGhpcy5HZXRSdW50aW1lQ3R4SUQuYmluZCh0aGlzKTtcclxuICAgIEh0dHBDbGllbnRFdmVudHMuR2V0R2xvYmFsVW5pcXVlU2Vzc2lvbklEX0V2ZW50ID0gdGhpcy5HZXRHbG9iYWxVbmlxdWVTZXNzaW9uSUQuYmluZCh0aGlzKTtcclxuICAgIEh0dHBDbGllbnRFdmVudHMuU2hvdWxkRGlzcGxheUdlbmVyaWNFcnJvcl9FdmVudCA9IFNlcnZlckNvbmZpZy5JbnN0YW5jZS5TaG91bGREaXNwbGF5R2VuZXJpY0Vycm9yLmJpbmQoU2VydmVyQ29uZmlnLkluc3RhbmNlKTtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gUmV0cmlldmVzIHJ1bnRpbWUgY29udGV4dCBJRFxyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgLy8vIDxyZXR1cm5zPjwvcmV0dXJucz5cclxuICBHZXRSdW50aW1lQ3R4SUQoKTogc3RyaW5nIHtcclxuICAgIHJldHVybiBSdW50aW1lQ29udGV4dEJhc2UuSW5zdGFuY2UuQ29udGV4dElEO1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PlxyXG4gIC8vLyBTZW5kIGEgcmVxdWVzdCB0byBzZXJ2ZXIgdG8gdGVybWluYXRlIHRoZSBjb250ZXh0IHVzaW5nIGZldGNoIEFQSSB3aXRoIGtlZXAtYWxpdmVcclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIHN0YXRpYyBUZXJtaW5hdGVDb250ZXh0VXNpbmdGZXRjaEFQSSgpOiB2b2lkIHtcclxuXHJcbiAgICAvLyBJZiBTZXNzaW9uIGlzIHJlLWluaXRpYWxpemluZyBtZWFucyB0aGUgY29udGV4dCBpcyBub3QgZm91bmQgYXQgc2VydmVyLlxyXG4gICAgLy8gdGhlbiB0aGVyZSBpcyBubyBwb2ludCBpbiB0ZXJtaW5hdGluZyBjb250ZXh0LlxyXG4gICAgaWYgKFJlbW90ZUNvbW1hbmRzUHJvY2Vzc29yLklzU2Vzc2lvblJlSW5pdGlhbGl6aW5nKVxyXG4gICAgICByZXR1cm47XHJcbiAgICBIdHRwTWFuYWdlci5HZXRJbnN0YW5jZSgpLklzQWJvcnRpbmdNYWdpY0VuZ2luZSA9IHRydWU7XHJcblxyXG4gICAgLy8gRHVlIHRvIHBheWxvYWQgbGltaXQgb2YgZmV0Y2ggQVBJLCBzZW5kIG9ubHkgdGhlIGNvbnRleHQgdGVybWluYXRpb24gY29tbWFuZCB0byBzZXJ2ZXJcclxuICAgIE1HRGF0YUNvbGxlY3Rpb24uSW5zdGFuY2UuZ2V0Q3Vyck1HRGF0YSgpLkNtZHNUb1NlcnZlci5DbGVhcigpO1xyXG5cclxuICAgIGxldCBjb250ZXh0VGVybWluYXRpb25FdmVudENvbW1hbmQ6IENvbnRleHRUZXJtaW5hdGlvbkV2ZW50Q29tbWFuZCA9IENvbW1hbmRGYWN0b3J5LlxyXG4gICAgQ3JlYXRlQ29udGV4dFRlcm1pbmF0aW9uRXZlbnRDb21tYW5kKE1HRGF0YUNvbGxlY3Rpb24uSW5zdGFuY2UuU3RhcnR1cE1nRGF0YS5nZXRGaXJzdFRhc2soKS5nZXRUYXNrVGFnKCkpO1xyXG5cclxuICAgIE1HRGF0YUNvbGxlY3Rpb24uSW5zdGFuY2UuZ2V0Q3Vyck1HRGF0YSgpLkNtZHNUb1NlcnZlci5BZGQoPElDbGllbnRDb21tYW5kPmNvbnRleHRUZXJtaW5hdGlvbkV2ZW50Q29tbWFuZCk7XHJcblxyXG4gICAgLy8gcmVzcG9uc2UgbmVlZCBub3QgdG8gYmUgaGFuZGxlZCBhcyB0aGUgYnJvd3NlciB1bmxvYWQgY2Fubm90IGJlIHN0b3BwZWQsIHNvIG5vIG5lZWQgdG8gd2FpdFxyXG4gICAgUmVtb3RlQ29tbWFuZHNQcm9jZXNzb3IuR2V0SW5zdGFuY2UoKS5FeGVjdXRlKENvbW1hbmRzUHJvY2Vzc29yQmFzZV9TZW5kaW5nSW5zdHJ1Y3Rpb24uT05MWV9DT01NQU5EUyk7XHJcbiAgfVxyXG59XHJcbiJdfQ==