import { IClientTargetedCommand } from './commands/IClientTargetedCommand';
import { XMLBasedCommandBuilder } from './commands/ServerToClient/XMLBasedCommandBuilder';
import { ResultCommand } from './commands/ServerToClient/ResultCommand';
import { AbortCommand } from './commands/ServerToClient/AbortCommand';
import { List } from '@magic-xpa/mscorelib';
import { VerifyCommand } from './commands/ServerToClient/VerifyCommand';
import { ConstInterface } from './ConstInterface';
import { ClientOriginatedCommandSerializer } from './commands/ClientToServer/ClientOriginatedCommandSerializer';
import { GlobalCommandsManager } from './GlobalCommandsManager';
import { AccessHelper } from './AccessHelper';
export class CommandsTable {
    constructor() {
        this._cmds = null;
        this._iterationIdx = 0;
        this._cmds = new List();
    }
    static createCommandsTable() {
        return new CommandsTable();
    }
    fillData(parser) {
        while (this.initInnerObjects(parser.getNextTag(), parser)) {
        }
    }
    initInnerObjects(foundTagName, parser) {
        if (foundTagName == null)
            return false;
        if (foundTagName === ConstInterface.MG_TAG_COMMAND) {
            let cmd = new XMLBasedCommandBuilder().fillData(parser);
            this._cmds.push(cmd);
        }
        else
            return false;
        return true;
    }
    async Execute(res) {
        let currFrame = GlobalCommandsManager.Instance.getCmdFrame();
        GlobalCommandsManager.Instance.add1toCmdFrame();
        let firstBlocking = -1;
        let i, j;
        let cmd;
        for (i = 0; i < this.getSize(); i = i + 1) {
            cmd = this.getCmd(i);
            if (cmd.IsBlocking && firstBlocking === -1)
                firstBlocking = i;
            else if (cmd instanceof ResultCommand && firstBlocking !== -1) {
                this._cmds.Insert(firstBlocking, cmd);
                this._cmds.RemoveAt(i + 1);
            }
        }
        for (i = 0; i < this.getSize(); i = i + 1) {
            cmd = this.getCmd(i);
            cmd.Frame = currFrame;
            if (cmd.IsBlocking) {
                for (j = i + 1; j < this.getSize(); j = j + 1)
                    GlobalCommandsManager.Instance.addUnframedCmd(this.getCmd(j));
                this._cmds.SetSize(i + 1);
                break;
            }
        }
        while (this.getSize() > 0) {
            const tmpCmd = this.extractCmd(currFrame);
            if (tmpCmd instanceof IClientTargetedCommand)
                cmd = tmpCmd;
            if (cmd != null) {
                if (cmd.ShouldExecute) {
                    await cmd.Execute(res);
                }
                else {
                    GlobalCommandsManager.Instance.addCommandsExecutedAfterTaskStarted(cmd);
                }
            }
            else
                break;
        }
    }
    extractCmd(frame) {
        let size = this.getSize();
        let cmd = null;
        let verifyCmd = null;
        let verifyIndex;
        let verifyTask = null;
        let abortTask = null;
        let startIterationIdx;
        let tid;
        if (size === 0)
            return null;
        if (this._iterationIdx >= size)
            this._iterationIdx = 0;
        startIterationIdx = this._iterationIdx;
        for (; this._iterationIdx < size; this._iterationIdx++) {
            cmd = ((this.getCmd(this._iterationIdx) instanceof AbortCommand) ? this.getCmd(this._iterationIdx) : null);
            if (cmd != null && !cmd.TaskTag.startsWith('-') && cmd.Frame === frame) {
                for (verifyIndex = 0; verifyIndex < this._iterationIdx; verifyIndex++) {
                    verifyCmd = ((this.getCmd(verifyIndex) instanceof VerifyCommand) ? this.getCmd(verifyIndex) : null);
                    if (verifyCmd != null && verifyCmd.Frame === frame) {
                        verifyTask = AccessHelper.mgDataTable.GetTaskByID(verifyCmd.TaskTag);
                        abortTask = AccessHelper.mgDataTable.GetTaskByID(cmd.TaskTag);
                        if (verifyTask != null && verifyTask.isDescendentOf(abortTask)) {
                            this._cmds.RemoveAt(verifyIndex);
                            this._iterationIdx--;
                            return verifyCmd;
                        }
                    }
                }
                this._cmds.RemoveAt(this._iterationIdx);
                size = this.getSize();
                for (startIterationIdx = 0; startIterationIdx < size; startIterationIdx++) {
                    tid = this.getCmd(startIterationIdx).TaskTag;
                    if (tid != null && tid.substr(1) === cmd.TaskTag) {
                        this._cmds.RemoveAt(startIterationIdx);
                        break;
                    }
                }
                return cmd;
            }
        }
        for (verifyIndex = 0; verifyIndex < size; verifyIndex++) {
            verifyCmd = ((this.getCmd(verifyIndex) instanceof VerifyCommand) ? this.getCmd(verifyIndex) : null);
            if (verifyCmd != null && verifyCmd.Frame === frame &&
                AccessHelper.mgDataTable.GetTaskByID(verifyCmd.TaskTag) != null) {
                this._cmds.RemoveAt(verifyIndex);
                return verifyCmd;
            }
        }
        for (this._iterationIdx = 0; this._iterationIdx < size; this._iterationIdx++) {
            cmd = this.getCmd(this._iterationIdx);
            if (cmd.Frame === frame) {
                this._cmds.RemoveAt(this._iterationIdx);
                return cmd;
            }
        }
        return null;
    }
    getCmd(idx) {
        return this._cmds.get_Item(idx);
    }
    buildXML(message) {
        for (let i = 0; i < this._cmds.length; i = i + 1) {
            let cmd = this._cmds.get_Item(i);
            message.Append(ClientOriginatedCommandSerializer.Serialize(cmd, AccessHelper.mgDataTable.getCurrMGData()));
        }
        this.clear();
    }
    Add(cmd) {
        this._cmds.push(cmd);
    }
    Clear() {
        this._cmds.Clear();
    }
    ExtractCommand(commandIndex) {
        let command = this.getCmd(commandIndex);
        this._cmds.RemoveAt(commandIndex);
        return command;
    }
    clear() {
        this._cmds.Clear();
    }
    getSize() {
        return this._cmds.length;
    }
}
//# sourceMappingURL=data:application/json;base64,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