import { CtrlButtonTypeGui, InternalInterface, Logger, MgControlType, Misc } from '@magic-xpa/utils';
import { Events, EventsProcessor, RuntimeContextBase } from '@magic-xpa/gui';
import { RunTimeEvent } from './event/RunTimeEvent';
import { RCTimer } from './tasks/RCTimer';
import { MGDataCollection } from './tasks/MGDataCollection';
import { EventsManager } from './event/EventsManager';
import { LanguageData } from './env/LanguageData';
import { EnvParamsTable } from './env/EnvVariablesTable';
import { LastFocusedManager } from './rt/LastFocusedManager';
import { ServerConfig } from './ServerConfig';
export class GuiEventsProcessor extends EventsProcessor {
    RegisterSubclassHandlers() {
        Events.FocusEvent = GuiEventsProcessor.processFocus;
        Events.FocusOnNonMagicControlEvent = GuiEventsProcessor.processFocusOnNonMagicControl;
        Events.MouseUpEvent = GuiEventsProcessor.processMouseUp;
        Events.SelectionEvent = GuiEventsProcessor.processSelection;
        Events.DisposeEvent = GuiEventsProcessor.processDispose;
        Events.TimerEvent = GuiEventsProcessor.ProcessTimer;
        Events.TableResizeEvent = GuiEventsProcessor.processTableResize;
        Events.GetRowsDataEvent = GuiEventsProcessor.processGetRowsData;
        Events.NonParkableLastParkedCtrlEvent = GuiEventsProcessor.OnNonParkableLastParkedCtrl;
        Events.ShowSessionStatisticsEvent = GuiEventsProcessor.ShowSessionStatistics;
        Events.WriteExceptionToLogEvent = Logger.Instance.WriteExceptionToLogWithMsg.bind(Logger.Instance);
        Events.WriteErrorToLogEvent = Logger.Instance.WriteErrorToLog.bind(Logger.Instance);
        Events.WriteWarningToLogEvent = Logger.Instance.WriteWarningToLogWithMsg.bind(Logger.Instance);
        Events.ShouldLogEvent = Logger.Instance.ShouldLog.bind(Logger.Instance);
        Events.WriteGuiToLogEvent = Logger.Instance.WriteGuiToLog.bind(Logger.Instance);
        Events.WriteDevToLogEvent = Logger.Instance.WriteDevToLog.bind(Logger.Instance);
        Events.TranslateEvent = LanguageData.Instance.translate.bind(LanguageData.Instance);
        Events.PeekEndOfWorkEvent = GuiEventsProcessor.peekEndOfWork;
        Events.TranslateLogicalNameEvent = EnvParamsTable.Instance.translate.bind(EnvParamsTable.Instance);
        Events.GetMainProgramEvent = GuiEventsProcessor.getMainProgram;
        Events.GetMessageStringEvent = LanguageData.Instance.getConstMessage.bind(LanguageData.Instance);
        Events.CtrlFocusEvent = GuiEventsProcessor.processCtrlFocus;
        Events.GetRuntimeContextEvent = GuiEventsProcessor.GetRuntimeContext;
        Events.SaveLastClickedCtrlEvent = GuiEventsProcessor.SaveLastClickedControlName;
        Events.SaveLastClickInfoEvent = GuiEventsProcessor.SaveLastClickInfo;
        Events.OnIsLogonRTLEvent = GuiEventsProcessor.IsLogonRTL;
        Events.ShouldAddEnterAsKeyEvent = GuiEventsProcessor.ShouldAddEnterAsKeyEvent;
    }
    constructor() {
        super();
    }
    static processCtrlFocus(iTask, mgControl) {
        LastFocusedManager.setLastFocusedControl(iTask, mgControl);
    }
    static processFocus(guiMgCtrl, line, isProduceClick, onMultiMark) {
        let mgControl = guiMgCtrl;
        EventsManager.Instance.addGuiTriggeredEventWithCtrlAndLineAndCodeAndIsProductClick(mgControl, InternalInterface.MG_ACT_CTRL_FOCUS, line, isProduceClick);
    }
    static processFocusOnNonMagicControl(guiMgCtrl) {
        let mgControl = guiMgCtrl;
        EventsManager.Instance.addGuiTriggeredEventWithCtrlAndCodeAndLine(mgControl, InternalInterface.MG_ACT_CTRL_FOCUS_ON_NON_MAGIC_CONTROL, mgControl.getDisplayLine(true));
    }
    static processMouseUp(guiMgCtrl, line) {
        let mgControl = guiMgCtrl;
        let rtEvt = new RunTimeEvent(mgControl, line, true);
        let produceClick = true;
        if (mgControl.Type === MgControlType.CTRL_TYPE_BUTTON && guiMgCtrl.ButtonStyle === CtrlButtonTypeGui.Hypertext)
            produceClick = false;
        rtEvt.setProduceClick(produceClick);
        rtEvt.setInternal(InternalInterface.MG_ACT_CTRL_MOUSEUP);
        EventsManager.Instance.addToTail(rtEvt);
    }
    static processDispose(guiMgForm) {
        let form = guiMgForm;
        let task = form.getTask();
        let mgd = task.getMGData();
        RCTimer.StopAll(mgd);
        if (mgd.IsAborting)
            MGDataCollection.Instance.deleteMGDataTree(mgd.GetId());
    }
    static ProcessTimer(mgTimer) {
        let mgd = mgTimer.GetMgdata();
        let task = mgd.getFirstTask();
        let seconds = Math.floor((mgTimer.TimerIntervalMiliSeconds) / 1000);
        let isIdle = mgTimer.IsIdleTimer;
        if (mgd.IsAborting)
            return;
        let rtEvt = new RunTimeEvent(task, true);
        rtEvt.setTimer(seconds, mgd.GetId(), isIdle);
        rtEvt.setMainPrgCreator(rtEvt.getTask());
        if (!isIdle)
            rtEvt.setCtrl(task.getLastParkedCtrl());
        rtEvt.setInternal(InternalInterface.MG_ACT_TIMER);
        EventsManager.Instance.addToTail(rtEvt);
    }
    static processTableResize(guiMgCtrl, newRowsInPage) {
        let mgControl = guiMgCtrl;
        if (mgControl != null) {
            let rtEvt = new RunTimeEvent(mgControl, newRowsInPage, true);
            rtEvt.setInternal(InternalInterface.MG_ACT_RESIZE);
            EventsManager.Instance.addToTail(rtEvt);
        }
    }
    static processGetRowsData(guiMgCtrl, desiredTopIndex, sendAll, lastFocusedVal) {
        let mgControl = guiMgCtrl;
        if (mgControl != null && mgControl.Type === MgControlType.CTRL_TYPE_TABLE) {
            let rtEvt = new RunTimeEvent(mgControl, desiredTopIndex, true);
            rtEvt.setInternal(InternalInterface.MG_ACT_ROW_DATA_CURR_PAGE);
            rtEvt.setSendAll(sendAll);
            rtEvt.LastFocusedVal = lastFocusedVal;
            EventsManager.Instance.addToTail(rtEvt);
        }
    }
    static OnNonParkableLastParkedCtrl(ctrl) {
    }
    static processSelection(val, guiMgCtrl, line, produceClick) {
        let mgControl = guiMgCtrl;
        if (mgControl.Type === MgControlType.CTRL_TYPE_BUTTON && mgControl.getForm().getTask().getLastParkedCtrl() !== mgControl)
            produceClick = true;
        if (guiMgCtrl.ConnectedControl) {
            let rtEvt = new RunTimeEvent(mgControl, +line, true);
            rtEvt.setInternal(InternalInterface.MG_ACT_ZOOM);
            rtEvt.setProduceClick(produceClick);
            EventsManager.Instance.addToTail(rtEvt);
        }
        else {
            let rtEvt = new RunTimeEvent(mgControl, +line, true);
            rtEvt.setInternal(InternalInterface.MG_ACT_SELECTION);
            rtEvt.setValue(val);
            rtEvt.setProduceClick(produceClick);
            EventsManager.Instance.addToTail(rtEvt);
        }
    }
    processKeyDown(guiMgForm, guiMgCtrl, modifier, keyCode, start, end, text, isActChar, suggestedValue, handled) {
        LastFocusedManager.Instance.LastActionTime = Misc.getSystemMilliseconds();
        return super.processKeyDown(guiMgForm, guiMgCtrl, modifier, keyCode, start, end, text, isActChar, suggestedValue, handled);
    }
    static peekEndOfWork() {
        return EventsManager.Instance.getEndOfWork();
    }
    static getMainProgram(contextID, ctlIdx) {
        return MGDataCollection.Instance.GetMainProgByCtlIdx(contextID, ctlIdx);
    }
    static GetRuntimeContext(contextID) {
        return RuntimeContextBase.Instance;
    }
    static SaveLastClickedControlName(ctrlName) {
        RuntimeContextBase.Instance.LastClickedCtrlName = ctrlName;
    }
    static SaveLastClickInfo(controlName, X, Y, offsetX, offsetY, LastClickCoordinatesAreInPixels) {
        RuntimeContextBase.Instance.SaveLastClickInfo(controlName, X, Y, offsetX, offsetY, LastClickCoordinatesAreInPixels);
    }
    static IsLogonRTL() {
        return ServerConfig.Instance.IsLogonRTL();
    }
    static ShowSessionStatistics() {
    }
    static ShouldAddEnterAsKeyEvent() {
        return false;
    }
}
//# sourceMappingURL=data:application/json;base64,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