import { Logger, XMLConstants, XmlParser } from "@magic-xpa/utils";
import { ApplicationException, Debug } from "@magic-xpa/mscorelib";
import { OpenURLCommand } from "./OpenURLCommand";
import { ClientTargetedCommandType } from "../../enums";
import { AbortCommand } from "./AbortCommand";
import { ClientRefreshCommand } from "./ClientRefreshCommand";
import { EnhancedVerifyCommand } from "./EnhancedVerifyCommand";
import { ResetRangeCommand } from "./ResetRangeCommand";
import { AddLocateCommand } from "./AddLocateCommand";
import { ResetLocateCommand } from "./ResetLocateCommand";
import { AddRangeCommand } from "./AddRangeCommand";
import { ResultCommand } from "./ResultCommand";
import { AddSortCommand } from "./AddSortCommand";
import { ResetSortCommand } from "./ResetSortCommand";
import { VerifyCommand } from "./VerifyCommand";
import { ConstInterface } from "../../ConstInterface";
export class XMLBasedCommandBuilder {
    constructor() {
        this.newTaskXML = null;
    }
    fillData(parser) {
        let endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_TERM, parser.getCurrIndex());
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            let tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(ConstInterface.MG_TAG_COMMAND) + ConstInterface.MG_TAG_COMMAND.length);
            let tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            let command = this.initElements(tokensVector);
            parser.setCurrIndex(endContext + XMLConstants.TAG_TERM.length);
            let openUrlCommand = command instanceof OpenURLCommand ? command : null;
            if (openUrlCommand != null) {
                this.getTaskXML(parser);
                openUrlCommand.NewTaskXML = this.newTaskXML;
            }
            return command;
        }
        Logger.Instance.WriteDevToLog("in Command.FillData() out of string bounds");
        return null;
    }
    initElements(tokensVector) {
        Debug.Assert(tokensVector.get_Item(0) === XMLConstants.MG_ATTR_TYPE, "The first attribute of a <command> element must be the command type.");
        let commandType = this.getType(tokensVector.get_Item(1).toLowerCase());
        let clientTargetedCommand = this.CreateCommand(commandType);
        for (let i = 2; i < tokensVector.length; i = i + 2) {
            let attribute = tokensVector.get_Item(i);
            let value = tokensVector.get_Item(i + 1);
            clientTargetedCommand.HandleAttribute(attribute, value);
        }
        return clientTargetedCommand;
    }
    CreateCommand(commandType) {
        switch (commandType) {
            case ClientTargetedCommandType.Abort:
                return new AbortCommand();
            case ClientTargetedCommandType.Verify:
                return new VerifyCommand();
            case ClientTargetedCommandType.EnhancedVerify:
                return new EnhancedVerifyCommand();
            case ClientTargetedCommandType.OpenURL:
                return new OpenURLCommand();
            case ClientTargetedCommandType.Result:
                return new ResultCommand();
            case ClientTargetedCommandType.AddRange:
                return new AddRangeCommand();
            case ClientTargetedCommandType.AddLocate:
                return new AddLocateCommand();
            case ClientTargetedCommandType.AddSort:
                return new AddSortCommand();
            case ClientTargetedCommandType.ResetRange:
                return new ResetRangeCommand();
            case ClientTargetedCommandType.ResetLocate:
                return new ResetLocateCommand();
            case ClientTargetedCommandType.ResetSort:
                return new ResetSortCommand();
            case ClientTargetedCommandType.ClientRefresh:
                return new ClientRefreshCommand();
            default:
                throw new ApplicationException("Unknown client targeted command type " + commandType);
        }
    }
    getTaskXML(parser) {
        if (parser.getNextTag() !== ConstInterface.MG_TAG_TASK_XML)
            throw new ApplicationException("Task's XML is missing");
        this.newTaskXML = parser.ReadContentOfCurrentElement();
    }
    getType(valueAttr) {
        let type;
        switch (valueAttr) {
            case ConstInterface.MG_ATTR_VAL_ABORT:
                type = ClientTargetedCommandType.Abort;
                break;
            case ConstInterface.MG_ATTR_VAL_VERIFY:
                type = ClientTargetedCommandType.Verify;
                break;
            case ConstInterface.MG_ATTR_VAL_ENHANCED_VERIFY:
                type = ClientTargetedCommandType.EnhancedVerify;
                break;
            case ConstInterface.MG_ATTR_VAL_RESULT:
                type = ClientTargetedCommandType.Result;
                break;
            case ConstInterface.MG_ATTR_VAL_OPENURL:
                type = ClientTargetedCommandType.OpenURL;
                break;
            case ConstInterface.MG_ATTR_VAL_ADD_RANGE:
                type = ClientTargetedCommandType.AddRange;
                break;
            case ConstInterface.MG_ATTR_VAL_ADD_LOCATE:
                type = ClientTargetedCommandType.AddLocate;
                break;
            case ConstInterface.MG_ATTR_VAL_ADD_SORT:
                type = ClientTargetedCommandType.AddSort;
                break;
            case ConstInterface.MG_ATTR_VAL_RESET_RANGE:
                type = ClientTargetedCommandType.ResetRange;
                break;
            case ConstInterface.MG_ATTR_VAL_RESET_LOCATE:
                type = ClientTargetedCommandType.ResetLocate;
                break;
            case ConstInterface.MG_ATTR_VAL_RESET_SORT:
                type = ClientTargetedCommandType.ResetSort;
                break;
            case ConstInterface.MG_ATTR_VAL_CLIENT_REFRESH:
                type = ClientTargetedCommandType.ClientRefresh;
                break;
            default:
                type = valueAttr[0];
                Logger.Instance.WriteExceptionToLogWithMsg("Command.getType there is no such SERVER to CLIENT command : " + valueAttr);
                break;
        }
        return type;
    }
}
//# sourceMappingURL=data:application/json;base64,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