import { ApplicationException, Debug, Int32, Int64, isNullOrUndefined, List, NNumber } from '@magic-xpa/mscorelib';
import { Constants, InternalInterface, Logger, Misc, StorageAttribute, XMLConstants, XmlParser } from '@magic-xpa/utils';
import { DataModificationTypes, DisplayConvertor, PIC, PropInterface } from '@magic-xpa/gui';
import { ConstInterface } from '../ConstInterface';
import { RecordsTable } from './RecordsTable';
import { Record } from './Record';
import { TaskBase } from '../tasks/TaskBase';
import { FieldsTableExt as FieldsTable } from './FieldsTable';
import { RecordOutOfDataViewException, RecordOutOfDataViewException_ExceptionType } from './RecordOutOfDataViewException';
import { CommandFactory } from '../commands/ClientToServer/CommandFactory';
import { XMLBasedDcValuesBuilder } from './XMLBasedDcValuesBuilder';
import { Field } from './Field';
import { EventSubType } from '../enums';
import { RemoteCommandsProcessor } from '../remote/RemoteCommandsProcessor';
import { FieldBase } from './FieldBase';
import { CommandsProcessorManager } from '../CommandsProcessorManager';
import { DataViewBase } from './DataViewBase';
import { AccessHelper, ClearEventsOnStopExecution } from '../AccessHelper';
import { MGDataCollection } from '../tasks/MGDataCollection';
export const SET_DISPLAYLINE_BY_DV = Int32.MinValue;
const COMPUTE_NEWREC_ON_CLIENT = 'C';
const UNKNOWN_RCMPS_NOT_INITED = 'M';
const UNKNOWN_RCMPS_FOUND = 'Y';
const INVOKED_FROM_OFFLINE_TASK = '-99999';
const CHUNK_CACHE_NEXT = 'N';
const CHUNK_CACHE_PREV = 'P';
const CHUNK_DV_BOTTOM = 'B';
const CHUNK_DV_TOP = 'T';
const COMPUTE_FLUSH_UPDATES = 'H';
const COMPUTE_NEWREC_ON_SERVER = 'S';
const END_DV_TAG = '</' + ConstInterface.MG_TAG_DATAVIEW + '>';
const RECOVERY_ACT_BEGIN_SCREEN = 'S';
const RECOVERY_ACT_BEGIN_TABLE = 'T';
const RECOVERY_ACT_CANCEL = 'C';
const RECOVERY_ACT_MOVE_DIRECTION_BEGIN = 'B';
const RECOVERY_ACT_NONE = 'N';
const TRANS_STAT_CLOSED = 'C';
const TRANS_STAT_OPENED = 'O';
export class DataView extends DataViewBase {
    set InsertAt(value) {
        this._insertAt = value;
    }
    get InsertAt() {
        return this._insertAt;
    }
    get CurrRec() {
        return this._currRec_DO_NOT_USE_DIRECTLY_USE_SETTER_GETTER;
    }
    set CurrRec(value) {
        if (this._currRec_DO_NOT_USE_DIRECTLY_USE_SETTER_GETTER !== null && value === null) {
            this._currRec_DO_NOT_USE_DIRECTLY_USE_SETTER_GETTER.resetDcValueId();
        }
        this._currRec_DO_NOT_USE_DIRECTLY_USE_SETTER_GETTER = value;
        if (this._currRec_DO_NOT_USE_DIRECTLY_USE_SETTER_GETTER !== null) {
            this._currRec_DO_NOT_USE_DIRECTLY_USE_SETTER_GETTER.SetDcValueId();
        }
    }
    get FlushUpdates() {
        return this._flushUpdates;
    }
    get HasMainTable() {
        return this._hasMainTable;
    }
    get CurrentRecId() {
        return this._currRecId;
    }
    get FirstRecord() {
        return this._recordsTab.getRecByIdx(0);
    }
    get LastRecord() {
        return this._recordsTab.getRecByIdx(this._recordsTab.getSize() - 1);
    }
    constructor(taskOrDataView) {
        super();
        this._cacheLruTimeStamp = 0;
        this._changed = false;
        this._chunkSize = 30;
        this._computeBy = '\0';
        this._currRecId = 0;
        this._currRecIdx = 0;
        this._dvPosValue = 0;
        this._firstDv = true;
        this._flushUpdates = false;
        this._hasMainTable = true;
        this._insertAt = ' ';
        this.InLocalDataviewCommand = false;
        this._lastCreatedRecId = 0;
        this._lastSessionCounter = Int64.MinValue;
        this._locateFirstRec = -1;
        this._modifiedRecordsTab = null;
        this._oldTopRecIdx = Int32.MinValue;
        this._original = null;
        this._pendingRecovery = RECOVERY_ACT_NONE;
        this._prevCurrRec = null;
        this._recordsTab = null;
        this._recovery = ConstInterface.RECOVERY_NONE;
        this._rmIdx = 0;
        this._rmSize = 0;
        this._skipParsing = false;
        this._topRecId = 0;
        this._topRecIdx = Int32.MinValue;
        this._topRecIdxModified = false;
        this._transCleared = false;
        this._unknownRcmp = UNKNOWN_RCMPS_NOT_INITED;
        this.IsOneWayKey = false;
        this._includesFirst = false;
        this._includesLast = false;
        this._serverTransCacheEmpty = true;
        this._currRec_DO_NOT_USE_DIRECTLY_USE_SETTER_GETTER = null;
        this.DBViewSize = 0;
        this.Dvcount = 0;
        this.TotalRecordsCount = 0;
        this.RecordsBeforeCurrentView = 0;
        if (taskOrDataView instanceof TaskBase) {
            this._task = taskOrDataView;
            this._fieldsTab = new FieldsTable();
            this._recordsTab = new RecordsTable(true);
            this._modifiedRecordsTab = new RecordsTable(false);
        }
        else {
            Object.assign(this, taskOrDataView);
        }
    }
    Init() {
        this.init();
    }
    IncludesFirst() {
        return this._includesFirst;
    }
    get ServerTransactionCacheEmpty() {
        return this._serverTransCacheEmpty;
    }
    IncludesLast() {
        return this._includesLast;
    }
    fillHeaderData(parser) {
        while (this.initHeaderInnerObjects(parser, parser.getNextTag())) {
        }
        this.Dvcount = this._task.ctl_itm_4_parent_vee(0, 1);
    }
    initHeaderInnerObjects(parser, foundTagName) {
        if (foundTagName === null)
            return false;
        if (foundTagName === XMLConstants.MG_TAG_DVHEADER) {
            Logger.Instance.WriteDevToLog('dvheader found');
            this.getHeaderAttributes(parser);
            this._fieldsTab.fillData(parser, this);
            this._fieldsTab.setRMPos(this._rmSize, this._rmIdx);
        }
        else if (foundTagName === ('/' + XMLConstants.MG_TAG_DVHEADER)) {
            parser.setCurrIndex2EndOfTag();
            return false;
        }
        else
            return false;
        return true;
    }
    getHeaderAttributes(parser) {
        let tokensVector;
        let tag;
        let attribute;
        let valueStr;
        let endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, parser.getCurrIndex());
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(XMLConstants.MG_TAG_DVHEADER) + XMLConstants.MG_TAG_DVHEADER.length);
            tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            for (let j = 0; j < tokensVector.length; j += 2) {
                attribute = (tokensVector.get_Item(j));
                valueStr = (tokensVector.get_Item(j + 1));
                switch (attribute) {
                    case ConstInterface.MG_ATTR_RMPOS: {
                        let i = valueStr.indexOf(',');
                        if (i > -1) {
                            this._rmSize = NNumber.Parse(valueStr.substr(0, i));
                            this._rmIdx = NNumber.Parse(valueStr.substr(i + 1));
                        }
                        break;
                    }
                    case ConstInterface.MG_ATTR_COMPUTE_BY:
                        this._computeBy = valueStr[0];
                        if (this._computeBy === COMPUTE_FLUSH_UPDATES) {
                            this._computeBy = COMPUTE_NEWREC_ON_SERVER;
                            this._flushUpdates = true;
                        }
                        else
                            this._flushUpdates = false;
                        break;
                    case ConstInterface.MG_ATTR_HAS_MAIN_TBL:
                        this._hasMainTable = XmlParser.getBoolean(valueStr);
                        break;
                    case ConstInterface.MG_ATTR_CHUNK_SIZE:
                        this._chunkSize = XmlParser.getInt(valueStr);
                        break;
                    case ConstInterface.MG_ATTR_CHUNK_SIZE_EXPRESSION:
                        break;
                    default:
                        Logger.Instance.WriteExceptionToLogWithMsg('Unknown attribute for <' + XMLConstants.MG_TAG_DVHEADER + '> tag: ' +
                            attribute);
                        break;
                }
            }
            parser.setCurrIndex(++endContext);
            return;
        }
        Logger.Instance.WriteExceptionToLogWithMsg('in DataView.getHeaderAttributes(): out of bounds');
    }
    async fillData(parser) {
        this._skipParsing = false;
        while (await this.initInnerObjects(parser, parser.getNextTag())) {
        }
        this.ResetFirstDv();
    }
    GetTransactionBegin() {
        let transBegin = '\0';
        let transBeginProp = this._task.getProp(PropInterface.PROP_TYPE_TRASACTION_BEGIN);
        if (transBeginProp !== null) {
            transBegin = (transBeginProp.GetComputedValue()).charAt(0);
        }
        return transBegin;
    }
    async initInnerObjects(parser, foundTagName) {
        let form = this._task.getForm();
        if (foundTagName == null)
            return false;
        if (foundTagName === (ConstInterface.MG_TAG_DATAVIEW)) {
            let invalidate = await this.getAttributes(parser);
            if (this._skipParsing) {
                let endContext = parser.getXMLdata().indexOf(END_DV_TAG, parser.getCurrIndex());
                parser.setCurrIndex(endContext + END_DV_TAG.length);
            }
            else {
                this.fillExecStack(parser);
                this.fillDataDc(parser);
                if (invalidate && form != null)
                    form.SetTableItemsCount(0, true);
                let taskRefreshType = this._recordsTab.fillData(this, this._insertAt, parser);
                if (form != null) {
                    if (form.Opened) {
                        if (this._insertAt === 'B' && (invalidate || this._recordsTab.InsertedRecordsCount > 0)) {
                            if (invalidate || !this._task.getForm().isTableWithPagination()) {
                                form.SetTableItemsCount(0, true);
                                form.setTableItemsCount(false);
                            }
                            else
                                form.InsertTableItems(0, this._recordsTab.InsertedRecordsCount);
                        }
                        else
                            form.setTableItemsCount(false);
                    }
                }
                if (this.CurrRec != null && this.CurrRec.InRecompute && taskRefreshType === Constants.TASK_REFRESH_NONE)
                    taskRefreshType = Constants.TASK_REFRESH_CURR_REC;
                if (invalidate) {
                    taskRefreshType = Constants.TASK_REFRESH_FORM;
                    if (this._task.IsSubForm)
                        this.getTask().DoSubformPrefixSuffix = true;
                }
                this.getTask().SetRefreshType(taskRefreshType);
                this.UpdateDataviewAfterInsert();
                let newRecId = this._recordsTab.getInitialCurrRecId();
                if (newRecId !== Int32.MinValue && (this._recovery === ConstInterface.RECOVERY_NONE || this._firstDv)) {
                    if (this._firstDv && this._recovery === ConstInterface.RECOVERY_NONE &&
                        (this._task.getLevel().charCodeAt(0) === 0 || this._task.getLevel() === Constants.TASK_LEVEL_TASK))
                        this._task.setLevel(Constants.TASK_LEVEL_TASK);
                    this._fieldsTab.invalidate(true, Field.LEAVE_FLAGS);
                    if (newRecId !== this._currRecId) {
                        await this.setCurrRec(newRecId, !this._firstDv);
                        this._original = this.CurrRec.replicate();
                        if (this._firstDv)
                            this.takeFldValsFromCurrRec();
                        if (this._task.getForm() != null)
                            (this._task.getForm()).updateDisplayLineByDV();
                    }
                    else
                        this.takeFldValsFromCurrRec();
                }
                else if (invalidate && this._recordsTab.getSize() > 0) {
                    if (form == null || form.isScreenMode()) {
                        try {
                            await this.setCurrRecByIdx(0, false, true, false, 0);
                        }
                        catch (exception) {
                            if (exception instanceof RecordOutOfDataViewException) {
                            }
                            else
                                throw exception;
                        }
                    }
                    else {
                        this._fieldsTab.invalidate(true, Field.LEAVE_FLAGS);
                        this.setTopRecIdx(0);
                        if (this._task.getForm() != null)
                            this._task.getForm().SetTableTopIndex();
                        try {
                            await this.setCurrRecByIdx(0, false, true, true, 0);
                            if (this._task.getForm() != null)
                                (this._task.getForm()).updateDisplayLineByDV();
                        }
                        catch (exception) {
                            if (exception instanceof RecordOutOfDataViewException) {
                            }
                            else
                                throw exception;
                        }
                        if (this._task.getLevel() === Constants.TASK_LEVEL_CONTROL && this._task.getLastParkedCtrl() != null)
                            this._task.getLastParkedCtrl().resetPrevVal();
                    }
                    this.saveOriginal();
                    this._fieldsTab.invalidate(true, FieldBase.CLEAR_FLAGS);
                }
                if (this._firstDv)
                    this.setPrevCurrRec();
                if (invalidate && this._task.IsSubForm)
                    this._task.getForm().getSubFormCtrl().RefreshOnVisible = false;
                if (this._recovery === ConstInterface.RECOVERY_NONE) {
                    if (this.getTask().getAfterRetry(ConstInterface.RECOVERY_RETRY) &&
                        !this.getTask().IsAfterRetryBeforeBuildXML)
                        this._task.setLevel(Constants.TASK_LEVEL_RECORD);
                }
                else {
                    let transBegin = this.GetTransactionBegin();
                    let stopExecution = true;
                    switch (this._recovery) {
                        case ConstInterface.RECOVERY_ROLLBACK:
                            if (transBegin === ConstInterface.TRANS_RECORD_PREFIX) {
                                if (this._task.getMode() === Constants.TASK_MODE_DELETE ||
                                    !this._task.IsInteractive)
                                    stopExecution = false;
                                else {
                                    this.getTask().setAfterRetry(ConstInterface.RECOVERY_ROLLBACK);
                                    this.CurrRec.restart(this.CurrRec.getMode());
                                    this.CurrRec.resetModified();
                                    this._original = this.CurrRec.replicate();
                                    this._pendingRecovery = RECOVERY_ACT_CANCEL;
                                }
                            }
                            else if (transBegin !== ConstInterface.TRANS_NONE) {
                                this._task.setLevel(Constants.TASK_LEVEL_TASK);
                                this._pendingRecovery = RECOVERY_ACT_BEGIN_TABLE;
                                this.getTask().setAfterRetry(ConstInterface.RECOVERY_ROLLBACK);
                            }
                            break;
                        case ConstInterface.RECOVERY_RETRY:
                            this.RecoveryRetry(transBegin);
                            break;
                    }
                    if (stopExecution) {
                        AccessHelper.eventsManager.setStopExecution(true);
                        this.getTask().TaskSuffixExecuted = false;
                    }
                }
            }
        }
        else if (foundTagName === '/' + ConstInterface.MG_TAG_DATAVIEW) {
            if (this._includesFirst && this._includesLast && this.isEmpty()) {
                this._currRecId = Int32.MinValue;
                this._currRecIdx = Int32.MinValue;
                this.CurrRec = null;
                this.setTopRecIdx(Int32.MinValue);
            }
            parser.setCurrIndex2EndOfTag();
            return false;
        }
        else
            return false;
        return true;
    }
    RecoveryRetry(transBegin) {
        if (transBegin === ConstInterface.TRANS_RECORD_PREFIX) {
            this._task.setLevel(Constants.TASK_LEVEL_RECORD);
            this.getTask().setTransactionFailed(true);
            this.getTask().setAfterRetry(ConstInterface.RECOVERY_RETRY);
            this.CurrRec.restart(this.CurrRec.getMode());
            if (!this._firstDv) {
                this.takeFldValsFromCurrRec();
                this._pendingRecovery = RECOVERY_ACT_MOVE_DIRECTION_BEGIN;
            }
        }
        else if (transBegin !== ConstInterface.TRANS_NONE) {
            this._task.setLevel(Constants.TASK_LEVEL_TASK);
            this._pendingRecovery = RECOVERY_ACT_BEGIN_SCREEN;
            this.getTask().setAfterRetry(ConstInterface.RECOVERY_RETRY);
            this.getTask().setTransactionFailed(true);
        }
    }
    UpdateDataviewAfterInsert() {
        if (this._insertAt === 'B') {
            let newRecs = this._recordsTab.InsertedRecordsCount;
            if (this._currRecIdx !== Int32.MinValue)
                this._currRecIdx += newRecs;
            if (this._topRecIdx !== Int32.MinValue)
                this._topRecIdx += newRecs;
            if (this._oldTopRecIdx !== Int32.MinValue)
                this._oldTopRecIdx += newRecs;
        }
    }
    async getAttributes(parser) {
        let tokensVector;
        let tag;
        let boolVal;
        let dataViewCommand = null;
        let isEmptyDataview = false;
        this._recovery = ConstInterface.RECOVERY_NONE;
        let endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, parser.getCurrIndex());
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(ConstInterface.MG_TAG_DATAVIEW) + ConstInterface.MG_TAG_DATAVIEW.length);
            tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            this._insertAt = ' ';
            let invalidate = this.peekInvalidate(tokensVector);
            this._locateFirstRec = 0;
            for (let j = 0; j < tokensVector.length; j += 2) {
                let attribute = (tokensVector.get_Item(j));
                let valueStr = (tokensVector.get_Item(j + 1));
                switch (attribute) {
                    case ConstInterface.MG_ATTR_INSERT_AT:
                        this._insertAt = valueStr[0];
                        break;
                    case ConstInterface.MG_ATTR_INCLUDE_FIRST:
                        boolVal = XmlParser.getBoolean(valueStr);
                        this.SetIncludesFirst(boolVal);
                        break;
                    case ConstInterface.MG_ATTR_INCLUDE_LAST:
                        boolVal = XmlParser.getBoolean(valueStr);
                        this.SetIncludesLast(boolVal);
                        break;
                    case ConstInterface.MG_ATTR_IS_ONEWAY_KEY:
                        boolVal = XmlParser.getBoolean(valueStr);
                        this.SetOneWayKey(boolVal);
                        break;
                    case ConstInterface.MG_ATTR_DBVIEWSIZE:
                        this.DBViewSize = XmlParser.getInt(valueStr);
                        break;
                    case ConstInterface.MG_ATTR_INVALIDATE:
                        {
                        }
                        break;
                    case ConstInterface.MG_ATTR_TOP_REC_ID:
                        this._topRecId = NNumber.Parse(valueStr);
                        break;
                    case ConstInterface.MG_ATTR_TASKMODE:
                        if (this._task.getOriginalTaskMode() === Constants.TASK_MODE_NONE || invalidate)
                            this._task.setOriginalTaskMode(valueStr);
                        this._task.setProp(PropInterface.PROP_TYPE_TASK_MODE, valueStr);
                        break;
                    case ConstInterface.MG_ATTR_EMPTY_DATAVIEW:
                        isEmptyDataview = true;
                        break;
                    case XMLConstants.MG_ATTR_TASKID:
                        continue;
                    case ConstInterface.MG_ATTR_LOW_ID:
                        this._lastCreatedRecId = NNumber.Parse(valueStr);
                        break;
                    case ConstInterface.MG_ATTR_TRANS_STAT:
                        let transStat = valueStr[0];
                        switch (transStat) {
                            case TRANS_STAT_OPENED:
                                dataViewCommand = CommandFactory.CreateSetTransactionStateDataviewCommand(this.getTask().getTaskTag(), true);
                                await this.getTask().DataviewManager.RemoteDataviewManager.Execute(dataViewCommand);
                                break;
                            case TRANS_STAT_CLOSED:
                                dataViewCommand = CommandFactory.CreateSetTransactionStateDataviewCommand(this.getTask().getTaskTag(), false);
                                await this.getTask().DataviewManager.RemoteDataviewManager.Execute(dataViewCommand);
                                this.getTask().setTransactionFailed(false);
                                break;
                            default:
                                break;
                        }
                        break;
                    case ConstInterface.MG_ATTR_RECOVERY:
                        if (this._task.IsTryingToStop)
                            this.getTask().setTryingToStop(false);
                        this.getTask().resetExecEndTask();
                        this._recovery = valueStr[0];
                        break;
                    case ConstInterface.MG_ATTR_DVPOS_VALUE:
                        this._dvPosValue = NNumber.Parse(valueStr);
                        break;
                    case ConstInterface.MG_ATTR_LOCATE_FIRST_ID:
                        this._locateFirstRec = NNumber.Parse(valueStr);
                        break;
                    case ConstInterface.MG_ATTR_OFFSET:
                        break;
                    case ConstInterface.MG_ATTR_USER_SORT:
                        let form = this._task.getForm();
                        form.clearTableColumnSortMark(XmlParser.getBoolean(valueStr));
                        break;
                    case ConstInterface.MG_ATTR_VAL_TOTAL_RECORDS_COUNT:
                        this.TotalRecordsCount = XmlParser.getInt(valueStr);
                        break;
                    case ConstInterface.MG_ATTR_VAL_RECORDS_BEFORE_CURRENT_VIEW:
                        this.RecordsBeforeCurrentView = XmlParser.getInt(valueStr);
                        break;
                    case ConstInterface.MG_ATTR_VAL_SERVER_TRANS_CACHE_EMPTY:
                        this._serverTransCacheEmpty = XmlParser.getBoolean(valueStr);
                        break;
                    default:
                        Logger.Instance.WriteExceptionToLogWithMsg('Unknown attribute for <' + ConstInterface.MG_TAG_DATAVIEW + '> tag: ' + attribute);
                        break;
                }
            }
            this.setEmptyDataview(isEmptyDataview);
            parser.setCurrIndex(++endContext);
            return invalidate;
        }
        Logger.Instance.WriteExceptionToLogWithMsg('in DataView.getAttributes(): out of bounds');
        return false;
    }
    SetOneWayKey(val) {
        this.IsOneWayKey = val;
    }
    SetIncludesLast(val) {
        this._includesLast = val;
    }
    SetIncludesFirst(val) {
        let prevIncludeFirst = this._includesFirst;
        this._includesFirst = val;
        if (this._includesFirst !== prevIncludeFirst) {
            let form = this._task.getForm();
            if (form !== null) {
                form.SetTableItemsCount(0, true);
            }
        }
    }
    fillDataDc(parser) {
        let dcvBuilder = new XMLBasedDcValuesBuilder();
        let foundTagName = parser.getNextTag();
        while (foundTagName !== null && foundTagName === ConstInterface.MG_TAG_DC_VALS) {
            let dcValsPosition = parser.getCurrIndex();
            dcvBuilder.SerializedDCVals = parser.ReadToEndOfCurrentElement();
            let dcv = dcvBuilder.Build();
            if (dcv !== null)
                this.AddDcValues(dcv);
            else
                Logger.Instance.WriteExceptionToLogWithMsg('Error while parsing DC values at position ' + dcValsPosition);
            foundTagName = parser.getNextTag();
        }
    }
    AddDcValues(dcv) {
        this._dcValsCollection.Add(dcv.getId().toString(), dcv);
    }
    fillExecStack(parser) {
        let foundTagName = parser.getNextTag();
        let execStackTagExists = false;
        while (foundTagName !== null && foundTagName === ConstInterface.MG_TAG_EXEC_STACK_ENTRY) {
            this.fillInnerExecStack(parser);
            foundTagName = parser.getNextTag();
            execStackTagExists = true;
        }
        if (execStackTagExists) {
            AccessHelper.eventsManager.reverseServerExecStack();
        }
    }
    fillInnerExecStack(parser) {
        let tokensVector;
        const endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_TERM, parser.getCurrIndex());
        let attribute;
        let taskId = '';
        let handlerId = '';
        let operIdx = 0;
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            let tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(ConstInterface.MG_TAG_EXEC_STACK_ENTRY) + ConstInterface.MG_TAG_EXEC_STACK_ENTRY.length);
            tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            for (let j = 0; j < tokensVector.length; j += 2) {
                attribute = (tokensVector.get_Item(j));
                let valueStr = (tokensVector.get_Item(j + 1));
                switch (attribute) {
                    case XMLConstants.MG_ATTR_TASKID:
                        taskId = valueStr;
                        break;
                    case ConstInterface.MG_ATTR_HANDLERID:
                        handlerId = valueStr;
                        break;
                    case ConstInterface.MG_ATTR_OPER_IDX:
                        operIdx = XmlParser.getInt(valueStr);
                        break;
                }
            }
            AccessHelper.eventsManager.pushServerExecStack(taskId, handlerId, operIdx);
            parser.setCurrIndex(endContext + XMLConstants.TAG_TERM.length);
        }
        else
            Logger.Instance.WriteExceptionToLogWithMsg('in DataView.fillInnerExecStack() out of bounds');
    }
    peekInvalidate(tokensVector) {
        let sessionCounter = CommandsProcessorManager.GetCommandsProcessor().GetSessionCounter();
        for (let j = 0; j < tokensVector.length; j += 2) {
            let attribute = tokensVector.get_Item(j);
            if (attribute === (ConstInterface.MG_ATTR_INVALIDATE)) {
                let valueStr = tokensVector.get_Item(j + 1);
                if (XmlParser.getBoolean(valueStr)) {
                    if (sessionCounter < this._lastSessionCounter)
                        this._skipParsing = true;
                    else {
                        this._lastSessionCounter = sessionCounter;
                        this.clear();
                        return true;
                    }
                }
                return false;
            }
        }
        return false;
    }
    getFieldByName(fldName) {
        if (this._fieldsTab !== null)
            return this._fieldsTab.getField(fldName);
        Logger.Instance.WriteExceptionToLogWithMsg('in DataView.getField(String): There is no fieldsTab object');
        return null;
    }
    async buildXML(message) {
        let dcv;
        let taskTag = this._task.getTaskTag();
        let brkLevelIndex = this._task.getBrkLevelIndex();
        let currRecDeleted = false;
        let serverParent;
        let contextTask = (this.getTask().GetContextTask());
        let invokingTaskTag = contextTask.IsOffline ? INVOKED_FROM_OFFLINE_TASK : contextTask.getTaskTag();
        message.Append('\n      <' + ConstInterface.MG_TAG_DATAVIEW + ' ' + XMLConstants.MG_ATTR_TASKID + '="' +
            taskTag + '"');
        message.Append(' ' + ConstInterface.MG_ATTR_TASKLEVEL + '="' + brkLevelIndex + '"');
        serverParent = '0';
        let triggeringTask = this.getTask().getTriggeringTask();
        if (triggeringTask != null) {
            if (triggeringTask.IsOffline) {
                let ctlIdx = triggeringTask.getCtlIdx();
                let mainPrg = AccessHelper.mgDataTable.GetMainProgByCtlIdx(-1, ctlIdx);
                serverParent = mainPrg.getTaskTag();
            }
            else
                serverParent = triggeringTask.getTaskTag();
        }
        if (this._currRecId > Int32.MinValue) {
            if (this.CurrRec != null)
                message.Append(' ' + ConstInterface.MG_ATTR_CURR_REC + '="' + this._currRecId + '"');
            message.Append(' ' + ConstInterface.MG_ATTR_TASKMODE + '="' + this._task.getMode() + '"');
            if (this._transCleared && !(MGDataCollection.Instance.StartupMgData.getFirstTask().InEndTask && MGDataCollection.Instance.StartupMgData.getFirstTask().ClosingFormUI)) {
                message.Append(' ' + ConstInterface.MG_ATTR_TRANS_CLEARED + '="1"');
                this._transCleared = false;
            }
            message.Append(' ' + ConstInterface.MG_ATTR_INVOKER_ID + '="' + invokingTaskTag + ',' + serverParent + '"');
            if (await this.getTask().isCached())
                message.Append(' ' + ConstInterface.MG_ATTR_DVPOS_VALUE + '="' + this._dvPosValue + '"');
            message.Append(' ' + ConstInterface.MG_ATTR_LOOP_COUNTER + '="' + this.getTask().getLoopCounter() + '"');
            if (this._task.IsSubForm) {
                let subformCtrl = this._task.getForm().getSubFormCtrl();
                if (!subformCtrl.isVisible() &&
                    !await subformCtrl.checkProp(PropInterface.PROP_TYPE_REFRESH_WHEN_HIDDEN, false))
                    message.Append(' ' + ConstInterface.MG_ATTR_SUBFORM_VISIBLE + '="0"');
            }
            if (!this._task.IsInteractive)
                message.Append(' ' + ConstInterface.MG_ATTR_TASK_COUNTER + '="' + this.getTask().getCounter() + '"');
            if (this._task != null && this._task.GetComputedProperty(PropInterface.PROP_TYPE_PRELOAD_VIEW).GetComputedValueBoolean()) {
                message.Append(' ' + ConstInterface.MG_ATTR_LAST_REC_ID + '="' +
                    (this._recordsTab.getRecByIdx(this.getSize() - 1).getId()) + '"');
                message.Append(' ' + ConstInterface.MG_ATTR_OLD_FIRST_REC_ID + '="' +
                    this._recordsTab.getRecByIdx(0).getId() + '"');
            }
            if (this._task.IsInteractive) {
                let form = this._task.getForm();
                if (form != null && this._recordsTab.getSize() > 1) {
                    message.Append(' ' + ConstInterface.MG_ATTR_CURR_REC_POS_IN_FORM + '="' +
                        form.getCurrRecPosInForm() + '"');
                }
            }
            message.Append(XMLConstants.TAG_CLOSE);
            if (this.getTask().ResetRange || this.getTask().UserRngs != null) {
                message.Append(XMLConstants.START_TAG + ConstInterface.USER_RANGES);
                if (this.getTask().ResetRange) {
                    message.Append(' ' + ConstInterface.CLEAR_RANGE + '="1"');
                    this.getTask().ResetRange = false;
                }
                if (this.getTask().UserRngs != null) {
                    this.getTask().buildXMLForRngs(message, this.getTask().UserRngs, false);
                    this.getTask().UserRngs.Clear();
                    this.getTask().UserRngs = null;
                }
                else
                    message.Append(XMLConstants.TAG_TERM);
            }
            if (this.getTask().ResetLocate || this.getTask().UserLocs != null) {
                message.Append(XMLConstants.START_TAG + ConstInterface.USER_LOCATES);
                if (this.getTask().ResetLocate) {
                    message.Append(' ' + ConstInterface.CLEAR_LOCATES + '="1"');
                    this.getTask().ResetLocate = false;
                }
                if (this.getTask().UserLocs != null) {
                    this.getTask().buildXMLForRngs(message, this.getTask().UserLocs, true);
                    this.getTask().UserLocs.Clear();
                    this.getTask().UserLocs = null;
                }
                else
                    message.Append(XMLConstants.TAG_TERM);
            }
            if (this.getTask().ResetSort || this.getTask().UserSorts != null) {
                message.Append(XMLConstants.START_TAG + ConstInterface.MG_TAG_SORTS);
                if (this.getTask().ResetSort) {
                    message.Append(' ' + ConstInterface.CLEAR_SORTS + '="1"');
                    this.getTask().ResetSort = false;
                }
                if (this.getTask().UserSorts != null)
                    this.getTask().buildXMLForSorts(message);
                else
                    message.Append(XMLConstants.TAG_TERM);
            }
            if (await this.getTask().isCached()) {
                let delList = this.getTask().getTaskCache().getDeletedListToXML();
                if (delList !== '') {
                    message.Append(XMLConstants.START_TAG + ConstInterface.MG_TAG_DEL_LIST);
                    message.Append(' ' + XMLConstants.MG_ATTR_ID + '="' + this._task.getTaskTag() + '"');
                    message.Append(' ' + ConstInterface.MG_ATTR_REMOVE + '="' + delList + '"');
                    message.Append(XMLConstants.TAG_TERM);
                    if (this.getTask().getTaskCache().isDeleted(this._dvPosValue))
                        this.setChanged(true);
                    this.getTask().getTaskCache().clearDeletedList();
                }
            }
            let dcValKeys = this._dcValsCollection.Keys;
            dcValKeys.forEach((dcValKey) => {
                dcv = this._dcValsCollection.get_Item(dcValKey);
                if (!dcv.HasReferences) {
                    message.Append(XMLConstants.START_TAG + ConstInterface.MG_TAG_DC_VALS);
                    message.Append(' ' + XMLConstants.MG_ATTR_ID + '="' + dcv.getId() + '"');
                    message.Append(' ' + ConstInterface.MG_ATTR_REMOVE + '="1"');
                    message.Append(XMLConstants.TAG_TERM);
                    this._dcValsCollection.Remove(dcValKey);
                }
            });
            if (this.CurrRec != null)
                currRecDeleted = (this.CurrRec.getMode() === DataModificationTypes.Delete
                    && this._modifiedRecordsTab.getRecord(this.CurrRec.getId()) != null && this.CurrRec.SendToServer);
            let skipCurrRec = false;
            if (this.CurrRec != null && this._modifiedRecordsTab.getRecord(this.CurrRec.getId()) != null)
                skipCurrRec = AccessHelper.eventsManager.isForceExitPreRecordUpdate(this.getTask());
            this._modifiedRecordsTab.buildXML(message, skipCurrRec, this.CurrRec.getId());
            if (this._original != null)
                this._original.clearMode();
            for (let i = this._modifiedRecordsTab.getSize() - 1; i >= 0; i--) {
                if (skipCurrRec && this._modifiedRecordsTab.getRecByIdx(i).getId() === this.CurrRec.getId())
                    continue;
                this._modifiedRecordsTab.getRecByIdx(i).clearFlagsHistory();
                this._modifiedRecordsTab.removeRecord(i);
            }
            if (this.CurrRec != null && !currRecDeleted) {
                let forceBuild = (this._task.isMainProg() && (RemoteCommandsProcessor.IsSessionReInitializing));
                this.CurrRec.buildXML(message, true, forceBuild);
            }
        }
        else {
            message.Append(' ' + ConstInterface.MG_ATTR_INVOKER_ID + '="' + invokingTaskTag + ',' + serverParent + '"' +
                XMLConstants.TAG_CLOSE);
        }
        message.Append('\n      </' + ConstInterface.MG_TAG_DATAVIEW + XMLConstants.TAG_CLOSE);
    }
    getCurrRec() {
        if (this.CurrRec === null && this._currRecId > Int32.MinValue) {
            Logger.Instance.WriteExceptionToLogWithMsg('DataView.getCurrRec(): record ' + this._currRecId + ' not found');
        }
        return this.CurrRec;
    }
    getPrevCurrRec() {
        return this._prevCurrRec;
    }
    getOriginalRec() {
        if (this._original === null || this._original.getId() !== this.CurrRec.getId())
            return this.CurrRec;
        return this._original;
    }
    async setCurrRec(id, compute) {
        let oldRecId = this._currRecId;
        if (id > Int32.MinValue) {
            let newIdx = this._recordsTab.getRecIdx(id);
            if (newIdx === -1)
                this._original = null;
            else {
                try {
                    let ignoreCurrRec = this._currRecId === Int32.MinValue;
                    await this.setCurrRecByIdx(newIdx, !ignoreCurrRec, ignoreCurrRec, compute, Int32.MinValue);
                    if (id !== oldRecId)
                        this.saveOriginal();
                    return true;
                }
                catch (exception) {
                    if (exception instanceof RecordOutOfDataViewException) {
                    }
                    else
                        throw exception;
                }
            }
        }
        else {
            this.CurrRec = null;
            this._original = null;
            this._currRecId = Int32.MinValue;
        }
        return false;
    }
    async setCurrRecByIdx(newIdx, doSuffix, ignoreCurrRec, compute, newDisplayLine) {
        let lastRecIdx = this._recordsTab.getSize() - 1;
        let oldFirstRec, nextRec;
        let recordSuff = false;
        if (newIdx === Constants.MG_DATAVIEW_FIRST_RECORD) {
            if (this.getTask().isTableWithAbsolutesScrollbar()) {
                this.RecordsBeforeCurrentView = 0;
            }
            if (this._includesFirst) {
                if (this._currRecIdx >= 0 || this._currRecIdx === Int32.MinValue) {
                    newIdx = 0;
                    recordSuff = true;
                }
                else
                    throw new RecordOutOfDataViewException(RecordOutOfDataViewException_ExceptionType.TOP);
            }
            else {
                await this.fetchChunkFromServer(CHUNK_DV_TOP, doSuffix);
                await this.setCurrRecByIdx(newIdx, false, ignoreCurrRec, compute, SET_DISPLAYLINE_BY_DV);
                return;
            }
        }
        else if (newIdx === Constants.MG_DATAVIEW_LAST_RECORD) {
            if (this._includesLast) {
                if (this._currRecIdx <= lastRecIdx) {
                    newIdx = lastRecIdx;
                    recordSuff = true;
                }
                else
                    throw new RecordOutOfDataViewException(RecordOutOfDataViewException_ExceptionType.BOTTOM);
            }
            else {
                if (this.getTask().isTableWithAbsolutesScrollbar())
                    this.RecordsBeforeCurrentView = this.TotalRecordsCount;
                await this.fetchChunkFromServer(CHUNK_DV_BOTTOM, doSuffix);
                if (!this._includesLast)
                    throw new RecordOutOfDataViewException(RecordOutOfDataViewException_ExceptionType.BOTTOM);
                let form = this._task.getForm();
                if (form != null) {
                    while (!this._includesFirst && form.getRowsInPage() >= this.getSize()) {
                        await this.fetchChunkFromServer(CHUNK_CACHE_PREV, false);
                    }
                }
                await this.setCurrRecByIdx(newIdx, false, ignoreCurrRec, compute, SET_DISPLAYLINE_BY_DV);
                return;
            }
        }
        else if (newIdx < 0) {
            if (this._includesFirst)
                throw new RecordOutOfDataViewException(RecordOutOfDataViewException_ExceptionType.TOP);
            oldFirstRec = this._recordsTab.getRecByIdx(0);
            if (this._recordsTab.getSize() > 0 || this.IncludesLast())
                await this.fetchChunkFromServer(CHUNK_CACHE_PREV, doSuffix);
            else if (this._recordsTab.getSize() === 0 && this.IncludesFirst())
                await this.fetchChunkFromServer(CHUNK_DV_TOP, doSuffix);
            else
                Logger.Instance.WriteExceptionToLogWithMsg('in DataView.fetchChunkFromServer() wrong option');
            let oldFirstNewIdx = oldFirstRec != null
                ? this._recordsTab.getRecIdx(oldFirstRec.getId())
                : this._recordsTab.getSize();
            if (oldFirstNewIdx !== 0)
                await this.setCurrRecByIdx(oldFirstNewIdx + newIdx, false, ignoreCurrRec, compute, SET_DISPLAYLINE_BY_DV);
            return;
        }
        else if (newIdx > lastRecIdx) {
            if (this._includesLast)
                throw new RecordOutOfDataViewException(RecordOutOfDataViewException_ExceptionType.BOTTOM);
            if (this.getForm().isRowCreated(newIdx))
                this.getForm().markRowNOTCreated(newIdx);
            await this.fetchChunkFromServer(CHUNK_CACHE_NEXT, doSuffix);
            await this.setCurrRecByIdx(newIdx, false, ignoreCurrRec, compute, SET_DISPLAYLINE_BY_DV);
            return;
        }
        let isCreateLineAboveAct = false;
        let lastEvent = isNullOrUndefined(AccessHelper.eventsManager.getLastRtEvent()) ? 0 :
            AccessHelper.eventsManager.getLastRtEvent().getInternalCode();
        if (newIdx >= 0 && !this.getTask().transactionFailed(ConstInterface.TRANS_RECORD_PREFIX) &&
            lastEvent === InternalInterface.MG_ACT_CRELINE_ABOVE) {
            if ((this.getTask().getInCreateLine()) && newIdx == this._currRecIdx) {
                let newRec = this._recordsTab.getRecByIdx(newIdx);
                if (newRec.getMode() == DataModificationTypes.Insert)
                    isCreateLineAboveAct = true;
            }
        }
        if (recordSuff || isCreateLineAboveAct || newIdx !== this._currRecIdx || ignoreCurrRec && compute) {
            if (!ignoreCurrRec && doSuffix && this.CurrRec != null) {
                let oldRecTabSize = this._recordsTab.getSize();
                nextRec = this._recordsTab.getRecByIdx(newIdx);
                if (nextRec.isNewRec()) {
                    if (this._task.getMode() !== Constants.TASK_MODE_CREATE &&
                        !await this._task.checkProp(PropInterface.PROP_TYPE_ALLOW_CREATE, true))
                        AccessHelper.eventsManager.setStopExecution(true);
                }
                if (!AccessHelper.eventsManager.GetStopExecutionFlag()) {
                    let orgTopRecIdx = 0;
                    if (this._topRecIdxModified) {
                        orgTopRecIdx = this.getTopRecIdx();
                        this.restoreTopRecIdx();
                    }
                    await AccessHelper.eventsManager.handleInternalEventWithTask(this._task, InternalInterface.MG_ACT_REC_SUFFIX);
                    if (this._topRecIdxModified)
                        this.setTopRecIdx(orgTopRecIdx);
                }
                if (AccessHelper.eventsManager.GetStopExecutionFlag())
                    throw new RecordOutOfDataViewException(RecordOutOfDataViewException_ExceptionType.REC_SUFFIX_FAILED);
                if (nextRec !== this.CurrRec && this.getTask().getAfterRetry())
                    this._task.setLevel(Constants.TASK_LEVEL_TASK);
                if (oldRecTabSize - 1 === this._recordsTab.getSize() && this._currRecIdx < newIdx)
                    newIdx--;
                if (newIdx === this._currRecIdx) {
                    if (this._task.getMode() === Constants.TASK_MODE_CREATE && !this.CurrRec.isNewRec() &&
                        !this._task.getForm().inRefreshDisplay())
                        this._task.setMode(Constants.TASK_MODE_MODIFY);
                    return;
                }
            }
            this.CurrRec = this._recordsTab.getRecByIdx(newIdx);
            if (this.CurrRec == null)
                throw new ApplicationException('in DataView.setCurrRecByIdx() current record not found!');
            this._currRecId = this.CurrRec.getId();
            this._currRecIdx = newIdx;
            if (this._task.getForm() != null) {
                if (newDisplayLine === SET_DISPLAYLINE_BY_DV) {
                    (this._task.getForm()).updateDisplayLineByDV();
                }
                else
                    this._task.getForm().DisplayLine = newDisplayLine;
            }
            if (this._task.getMode() === Constants.TASK_MODE_CREATE && !this.CurrRec.isNewRec() &&
                !this._task.getForm().inRefreshDisplay())
                this._task.setMode(Constants.TASK_MODE_MODIFY);
            if (compute)
                await this.currRecCompute(false);
        }
    }
    async currRecCompute(forceClientCompute) {
        this._fieldsTab.invalidate(false, true);
        await this.compute(false, forceClientCompute);
    }
    saveOriginal() {
        if (!this.CurrRec.Modified)
            this._original = this.CurrRec.replicate();
    }
    async fetchChunkFromServer(chunkId, doSuffix) {
        this._task.getForm().FormRefreshed = true;
        if (doSuffix && this.CurrRec != null) {
            await AccessHelper.eventsManager.handleInternalEventWithTask(this._task, InternalInterface.MG_ACT_REC_SUFFIX);
            if (AccessHelper.eventsManager.GetStopExecutionFlag() || this._task.isAborting())
                throw new RecordOutOfDataViewException(RecordOutOfDataViewException_ExceptionType.REC_SUFFIX_FAILED);
            this._task.getForm().setSuffixDone();
        }
        let eventCode;
        let clientRecId = this._currRecId;
        switch (chunkId) {
            case CHUNK_DV_TOP:
                eventCode = InternalInterface.MG_ACT_DATAVIEW_TOP;
                break;
            case CHUNK_DV_BOTTOM:
                eventCode = InternalInterface.MG_ACT_DATAVIEW_BOTTOM;
                break;
            case CHUNK_CACHE_PREV:
                clientRecId = this.FirstRecord == null ? 0 : this.FirstRecord.getId();
                eventCode = InternalInterface.MG_ACT_CACHE_PREV;
                break;
            case CHUNK_CACHE_NEXT:
                eventCode = InternalInterface.MG_ACT_CACHE_NEXT;
                clientRecId = this.LastRecord == null ? 0 : this.LastRecord.getId();
                break;
            default:
                Logger.Instance.WriteExceptionToLogWithMsg('in DataView.fetchChunkFromServer() unknown chunk id: ' + chunkId);
                return;
        }
        let cmd = CommandFactory.CreateEventCommand(this._task.getTaskTag(), eventCode);
        cmd.ClientRecId = clientRecId;
        await this.getTask().DataviewManager.Execute(cmd);
    }
    async FetchRecordsAheadFromServer() {
        if (!this._includesLast)
            await this.fetchChunkFromServer(CHUNK_CACHE_NEXT, false);
        else if (!this._includesFirst)
            await this.fetchChunkFromServer(CHUNK_CACHE_PREV, false);
    }
    recExists(recIdx) {
        let lastRecIdx = this._recordsTab.getSize() - 1;
        return recIdx >= 0 && recIdx <= lastRecIdx;
    }
    checkFirst(locateFirst) {
        return this._recordsTab.getRecord(locateFirst) !== null;
    }
    getCurrRecIdx() {
        if (this._currRecIdx === Int32.MinValue) {
            if (!this.setCurrRec(this._currRecId, true)) {
                Logger.Instance.WriteWarningToLogWithMsg('DataView.getCurrRecIdx(): _task ' +
                    (this.getTask()).PreviouslyActiveTaskId +
                    ', record ' + this._currRecId + ' not found');
            }
        }
        return this._currRecIdx;
    }
    getCurrDBViewRowIdx() {
        return this.CurrRec.getDBViewRowIdx();
    }
    setTopRecIdx(newTopRecIdx) {
        this._oldTopRecIdx = this.getTopRecIdx();
        this._topRecIdx = newTopRecIdx;
    }
    getTopRecIdx() {
        return this._topRecIdx;
    }
    setTopRecIdxModified(val) {
        this._topRecIdxModified = val;
    }
    getChunkSize() {
        return this._chunkSize;
    }
    restoreTopRecIdx() {
        this._topRecIdx = this._oldTopRecIdx;
    }
    clear() {
        this._recordsTab.removeAll();
        this._original = null;
        let mgForm = this._task.getForm();
        if (mgForm !== null) {
            this._task.getForm().SetTableItemsCount(0, true);
        }
        this._modifiedRecordsTab.removeAll();
        this.init();
    }
    init() {
        this._currRecId = Int32.MinValue;
        this._currRecIdx = Int32.MinValue;
        this.CurrRec = null;
        this._prevCurrRec = null;
        this._includesFirst = false;
        this._includesLast = false;
        this.IsOneWayKey = false;
        this.setTopRecIdx(Int32.MinValue);
        this._fieldsTab.invalidate(false, true);
    }
    async computeSubForms() {
        if (await this.getTask().prepareCache(true)) {
            if (!await this.getTask().testAndSet(true))
                await this.compute(true, false);
        }
        else
            await this.compute(true, false);
    }
    async compute(subForms, forceClientCompute) {
        let tmpRec = null;
        let computeByServer = !this.computeByClient();
        let orgComputeBy = this._computeBy;
        if (forceClientCompute)
            this._computeBy = COMPUTE_NEWREC_ON_CLIENT;
        try {
            this.CurrRec.setInCompute(true);
            if (!subForms) {
                if (this.CurrRec.isNewRec()) {
                    if (this._task.getMode() !== Constants.TASK_MODE_CREATE && !super.isEmptyDataview()) {
                        this._task.setMode(Constants.TASK_MODE_CREATE);
                    }
                }
                else if (this._task.getMode() === Constants.TASK_MODE_CREATE) {
                    this._task.setMode(Constants.TASK_MODE_MODIFY);
                }
            }
            let getRecordData = computeByServer;
            if (subForms || (getRecordData && !this.CurrRec.isComputed() && !forceClientCompute)) {
                let command = CommandFactory.CreateComputeEventCommand(this._task.getTaskTag(), subForms, this._currRecId);
                await this._task.DataviewManager.Execute(command);
            }
            else {
                if (getRecordData) {
                    tmpRec = new Record(this.CurrRec.getId(), this);
                    tmpRec.Initialize();
                    tmpRec.setInitialFldVals(true, true);
                }
                this._fieldsTab.setServerRcmp(false);
                let from = this._fieldsTab.getRMIdx();
                let size = this._fieldsTab.getRMSize();
                for (let i = from; i < from + size; i = i + 1) {
                    let fld = this._fieldsTab.getField(i);
                    if (getRecordData && this._fieldsTab.serverRcmpDone() && fld.IsVirtual && !fld.VirAsReal && !fld.hasInitExp()) {
                        let currRec2 = this.CurrRec;
                        this.CurrRec = tmpRec;
                        try {
                            fld.takeValFromRec();
                        }
                        finally {
                            this.CurrRec = currRec2;
                        }
                    }
                    if (!this._task.getForm().TransferringDataToGUI)
                        await fld.compute(false);
                }
            }
        }
        finally {
            this.CurrRec.setInCompute(false);
            this.CurrRec.setLateCompute(false);
            this._computeBy = orgComputeBy;
        }
    }
    async addRecord(doSuffix, ignoreCurrRec, isCrelineAbove) {
        let newCurrRecId = (--this._lastCreatedRecId);
        let mode = DataModificationTypes.Insert;
        return await this.addRecord_1(doSuffix, ignoreCurrRec, false, newCurrRecId, mode, isCrelineAbove);
    }
    async addRecord_1(doSuffix, ignoreCurrRec, isFirstRecord, newCurrRecId, mode, isCrelineAbove) {
        let rec = new Record(newCurrRecId, this);
        rec.Initialize();
        rec.setInitialFldVals(false, !isFirstRecord);
        let newCurrRecIdx = 0;
        if (isCrelineAbove) {
            newCurrRecIdx = (this._currRecIdx >= 0
                ? this._currRecIdx
                : 0);
        }
        else
            newCurrRecIdx = (this._currRecIdx != Int32.MinValue
                ? this._currRecIdx + 1
                : 0);
        this._recordsTab.insertRecord(rec, newCurrRecIdx);
        rec.setMode(mode);
        try {
            let compute = !isFirstRecord;
            await this.setCurrRecByIdx(newCurrRecIdx, doSuffix, ignoreCurrRec, compute, Int32.MinValue);
            if (isFirstRecord && this.getTask().getMode() !== Constants.TASK_MODE_CREATE)
                rec.setOldRec();
        }
        catch (execption) {
            if (execption instanceof RecordOutOfDataViewException) {
                if (AccessHelper.eventsManager.GetStopExecutionFlag())
                    newCurrRecIdx = -1;
                else
                    throw new ApplicationException('in DataView.addRecord() invalid exception');
            }
            else
                throw execption;
        }
        return newCurrRecIdx;
    }
    async removeCurrRec() {
        let mgForm = this._task.getForm();
        let onDelete;
        if (this.CurrRec.getMode() === DataModificationTypes.Insert &&
            !this.getTask().transactionFailed(ConstInterface.TRANS_RECORD_PREFIX)) {
            this._modifiedRecordsTab.removeRecord(this.CurrRec);
            onDelete = false;
        }
        else {
            this.CurrRec.clearMode();
            this.CurrRec.setMode(DataModificationTypes.Delete);
            onDelete = true;
        }
        this.CurrRec.removeRecFromDc();
        this._recordsTab.removeRecord(this._currRecIdx);
        this.setChanged(true);
        if (this.isEmpty() && !this._task.IsTryingToStop) {
            await this.MoveToEmptyDataviewIfAllowed();
        }
        let newDisplayLine = this.calcLineToMove(onDelete);
        try {
            if (mgForm.isScreenMode()) {
                await this.setCurrRecByIdx(newDisplayLine, false, false, true, SET_DISPLAYLINE_BY_DV);
            }
            else {
                await mgForm.setCurrRowByDisplayLine(newDisplayLine, false, true);
            }
        }
        catch (exception) {
            if (exception instanceof RecordOutOfDataViewException) {
                try {
                    if (!super.isEmptyDataview())
                        newDisplayLine = mgForm.getPrevLine(newDisplayLine);
                    if (mgForm.isScreenMode())
                        await this.setCurrRecByIdx(newDisplayLine, false, false, true, SET_DISPLAYLINE_BY_DV);
                    else
                        await mgForm.setCurrRowByDisplayLine(newDisplayLine, false, true);
                }
                catch (exception) {
                    if (exception instanceof RecordOutOfDataViewException) {
                        this.reset();
                    }
                    else
                        throw exception;
                }
            }
            else
                throw exception;
        }
    }
    RemoveRecord(record) {
        let recIdx = this._recordsTab.getRecIdx(record.getId());
        if (this._topRecIdx > recIdx)
            this._topRecIdx--;
        if (this._oldTopRecIdx > recIdx)
            this._oldTopRecIdx--;
        if (this._currRecIdx > recIdx)
            this._currRecIdx--;
        this._recordsTab.removeRecord(record);
    }
    async MoveToEmptyDataviewIfAllowed() {
        if (!super.isEmptyDataview() && this._task.getMode() !== Constants.TASK_MODE_CREATE) {
            if (await this._task.checkProp(PropInterface.PROP_TYPE_ALLOW_EMPTY_DATAVIEW, true))
                super.setEmptyDataview(true);
        }
        else {
            if (this._task.getMode() === Constants.TASK_MODE_CREATE && super.isEmptyDataview())
                super.setEmptyDataview(false);
        }
    }
    calcLineToMove(onDelete) {
        let newDisplayLine;
        if (onDelete)
            newDisplayLine = this._currRecIdx;
        else {
            if (this._currRecIdx > 0)
                newDisplayLine = this._currRecIdx - 1;
            else
                newDisplayLine = this._currRecIdx;
        }
        return newDisplayLine;
    }
    reset() {
        this._currRecId = Int32.MinValue;
        this._currRecIdx = Int32.MinValue;
        this.CurrRec = null;
        this.setTopRecIdx(Int32.MinValue);
        this.SetIncludesFirst(true);
        this.SetIncludesLast(true);
    }
    async cancelEdit(onlyReal, recomputeUnknowns) {
        let tempCurrRec = null;
        let recRemoved = false;
        let inEndTask = this._task.InEndTask;
        if (inEndTask)
            onlyReal = true;
        if (this.CurrRec.getMode() === DataModificationTypes.Insert && this.CurrRec.isNewRec()) {
            if ((this._includesFirst && this._includesLast && (this._recordsTab === null || this._recordsTab.getSize() === 1)) || !this.HasMainTable) {
                if (this._original !== null) {
                    this.CurrRec.setSameAs(this._original, onlyReal);
                }
            }
            else {
                await this.removeCurrRec();
                recRemoved = true;
                this._original = this.CurrRec.replicate();
            }
        }
        else {
            if (!this.CurrRec.Modified) {
                let currCtrl = this._task.getLastParkedCtrl();
                if (currCtrl !== null)
                    currCtrl.resetPrevVal();
            }
            if (recomputeUnknowns)
                tempCurrRec = this.CurrRec.replicate();
            if (this._original !== null)
                this.CurrRec.setSameAs(this._original, onlyReal);
            if ((this._original !== null && recomputeUnknowns) && !this.CurrRec.isSameRecData(tempCurrRec, false, false))
                await this.execUnknownRcmps(this._original);
            this.takeFldValsFromCurrRec();
            this._original = this.CurrRec.replicate();
        }
        return recRemoved;
    }
    isEmpty() {
        return this._includesFirst && this._includesLast && (this._recordsTab === null || this._recordsTab.getSize() === 0);
    }
    async addCurrToModified(setRecToOld) {
        if (isNullOrUndefined(setRecToOld)) {
            await this.addCurrToModified_0();
            return;
        }
        await this.addCurrToModified_1(setRecToOld);
    }
    async addCurrToModified_0() {
        await this.addCurrToModified(true);
    }
    async addCurrToModified_1(setRecToOld) {
        this._modifiedRecordsTab.addRecord(this.CurrRec);
        if (setRecToOld) {
            this.CurrRec.setOldRec();
        }
        let command = CommandFactory.CreateSetTransactionStateDataviewCommand(this.getTask().getTaskTag(), true);
        await this.getTask().DataviewManager.RemoteDataviewManager.Execute(command);
    }
    takeFldValsFromCurrRec() {
        this._fieldsTab.takeValsFromRec();
    }
    currEqualsPrev(fldList) {
        if (this.CurrRec !== null && this._prevCurrRec !== null) {
            if (this.CurrRec === this._prevCurrRec)
                return true;
            for (let i = 0; i < fldList.length; i = i + 1) {
                if (!this.CurrRec.fldValsEqual(this._prevCurrRec, fldList[i]))
                    return false;
            }
        }
        else if (this.CurrRec !== this._prevCurrRec)
            return false;
        return true;
    }
    getTask() {
        Debug.Assert(this._task instanceof TaskBase);
        return this._task;
    }
    getForm() {
        return this._task.getForm();
    }
    modifiedRecordsNumber() {
        return this._modifiedRecordsTab.getSize();
    }
    getSize() {
        return this._recordsTab.getSize();
    }
    computeByClient() {
        return this._computeBy === COMPUTE_NEWREC_ON_CLIENT;
    }
    recExistsById(id) {
        return this._recordsTab.getRecord(id) !== null;
    }
    ParametersExist(iPos, iLen) {
        if (arguments.length === 2)
            return this.ParametersExist_0(iPos, iLen);
        return this.ParametersExist_1();
    }
    ParametersExist_0(iPos, iLen) {
        for (let i = iPos; i < iPos + iLen; i = i + 1) {
            if (super.getField(i).isParam())
                return true;
        }
        return false;
    }
    ParametersExist_1() {
        return this.ParametersExist(this._rmIdx, this._rmSize);
    }
    recInModifiedTab(id) {
        let record = this._modifiedRecordsTab.getRecord(id);
        return record !== null;
    }
    setPrevCurrRec(rec) {
        if (arguments.length === 0) {
            this.setPrevCurrRec_0();
            return;
        }
        this.setPrevCurrRec_1(rec);
    }
    setPrevCurrRec_0() {
        this._prevCurrRec = this.CurrRec;
    }
    setPrevCurrRec_1(rec) {
        this._prevCurrRec = rec;
    }
    isPrevCurrRecNull() {
        return this._prevCurrRec === null;
    }
    async execUnknownRcmps(orgRec) {
        let i;
        let from = this._fieldsTab.getRMIdx();
        let size = this._fieldsTab.getRMSize();
        let rec = this.getCurrRec();
        if (this._unknownRcmp === UNKNOWN_RCMPS_NOT_INITED) {
            for (i = from; i < from + size && this._unknownRcmp === UNKNOWN_RCMPS_NOT_INITED; i++) {
                let field = this._fieldsTab.getField(i);
                if (field.IsVirtual && !field.hasInitExp() && field.isServerRcmp())
                    this._unknownRcmp = UNKNOWN_RCMPS_FOUND;
            }
        }
        if (this._unknownRcmp === UNKNOWN_RCMPS_FOUND && rec !== null) {
            for (i = from; i < from + size; i = i + 1) {
                let field = this._fieldsTab.getField(i);
                if (field.IsVirtual && !field.hasInitExp() && field.isServerRcmp() && !rec.fldValsEqual(orgRec, i)) {
                    let fieldValue = rec.GetFieldValue(i);
                    let isNullFld = rec.IsNull(i);
                    rec.setFieldValue(i, orgRec.GetFieldValue(i), false);
                    await field.setValueAndStartRecompute(fieldValue, isNullFld, false, false, false);
                }
            }
        }
    }
    setTransCleared() {
        this._transCleared = true;
    }
    async processRecovery() {
        if (!this._task.isAborting()) {
            let orgAction = this._pendingRecovery;
            let orgStopExecution = AccessHelper.eventsManager.GetStopExecutionFlag();
            this._pendingRecovery = RECOVERY_ACT_NONE;
            if (orgStopExecution)
                AccessHelper.eventsManager.setStopExecution(false);
            let temporaryResetInCtrlPrefix = this._task.InCtrlPrefix;
            if (temporaryResetInCtrlPrefix)
                this._task.InCtrlPrefix = false;
            switch (orgAction) {
                case RECOVERY_ACT_CANCEL:
                    (this._task.DataView.getCurrRec()).resetUpdated();
                    this.getTask().getTaskCache().clearCache();
                    if (this.getSize() === 1 && this._task.getMode() === Constants.TASK_MODE_CREATE) {
                        this.CurrRec.resetModified();
                        this.CurrRec.clearMode();
                        this.CurrRec.setMode(DataModificationTypes.Insert);
                        this.CurrRec.setNewRec();
                        this._original.setSameAs(this.CurrRec, false);
                    }
                    await AccessHelper.eventsManager.handleInternalEventWithTaskAndEventSubtype(this.getTask(), InternalInterface.MG_ACT_CANCEL, EventSubType.CancelWithNoRollback);
                    break;
                case RECOVERY_ACT_MOVE_DIRECTION_BEGIN:
                    let lastParkedControl = this._task.getLastParkedCtrl();
                    if (lastParkedControl !== null)
                        await lastParkedControl.invoke();
                    break;
                case RECOVERY_ACT_BEGIN_SCREEN:
                    this.getTask().setPreventRecordSuffix(true);
                    await AccessHelper.eventsManager.handleInternalEventWithTask(this._task, InternalInterface.MG_ACT_TBL_BEGPAGE);
                    this.getTask().setPreventRecordSuffix(false);
                    break;
                case RECOVERY_ACT_BEGIN_TABLE:
                    this._task.ActionManager.enable(InternalInterface.MG_ACT_TBL_BEGTBL, true);
                    await AccessHelper.eventsManager.handleInternalEventWithTask(this._task, InternalInterface.MG_ACT_TBL_BEGTBL);
                    this.getTask().getTaskCache().clearCache();
                    break;
                default:
                    AccessHelper.eventsManager.setStopExecution(orgStopExecution);
                    break;
            }
            if (temporaryResetInCtrlPrefix)
                this._task.InCtrlPrefix = true;
            if (orgAction !== RECOVERY_ACT_NONE)
                AccessHelper.eventsManager.setStopExecution(true, orgAction !== RECOVERY_ACT_BEGIN_TABLE ? ClearEventsOnStopExecution.ALL : ClearEventsOnStopExecution.SERVER);
        }
    }
    replicate() {
        let rep = new DataView(this);
        rep._recordsTab = this._recordsTab.replicate();
        rep._currRecId = rep._recordsTab.getRecByIdx(0).getId();
        rep._currRecIdx = 0;
        rep.CurrRec = rep._recordsTab.getRecByIdx(0);
        rep._prevCurrRec = rep._recordsTab.getRecByIdx(0);
        rep._original = rep._recordsTab.getRecByIdx(0);
        rep.setTopRecIdx(Int32.MinValue);
        return rep;
    }
    getDvPosValue() {
        return this._dvPosValue;
    }
    getFirstDv() {
        return this._firstDv;
    }
    ResetFirstDv() {
        this._firstDv = false;
    }
    getCacheLRU() {
        return this._cacheLruTimeStamp;
    }
    setCacheLRU() {
        this._cacheLruTimeStamp = Misc.getSystemMilliseconds();
    }
    async setSameAs(newDv) {
        this._rmSize = newDv._rmSize;
        this._rmIdx = newDv._rmIdx;
        this._computeBy = newDv._computeBy;
        this._flushUpdates = newDv._flushUpdates;
        this._currRecId = newDv._currRecId;
        this._currRecIdx = newDv._currRecIdx;
        this._topRecId = newDv._topRecId;
        this.CurrRec = newDv.CurrRec;
        this._original = newDv._original;
        this._prevCurrRec = null;
        this._fieldsTab = newDv._fieldsTab;
        this._recordsTab = newDv._recordsTab;
        this._modifiedRecordsTab = newDv._modifiedRecordsTab;
        this._includesFirst = newDv._includesFirst;
        this._includesLast = newDv._includesLast;
        this._insertAt = newDv._insertAt;
        this._lastCreatedRecId = newDv._lastCreatedRecId;
        this._task = newDv._task;
        this._dcValsCollection = newDv._dcValsCollection;
        this._recovery = newDv._recovery;
        this._pendingRecovery = newDv._pendingRecovery;
        this._hasMainTable = newDv._hasMainTable;
        this._chunkSize = newDv._chunkSize;
        this._firstDv = newDv._firstDv;
        this._lastSessionCounter = newDv._lastSessionCounter;
        this._skipParsing = newDv._skipParsing;
        this._unknownRcmp = newDv._unknownRcmp;
        this._transCleared = newDv._transCleared;
        this._dvPosValue = newDv._dvPosValue;
        this._cacheLruTimeStamp = newDv._cacheLruTimeStamp;
        this._changed = newDv._changed;
        this._locateFirstRec = newDv._locateFirstRec;
        super.setEmptyDataview(newDv.isEmptyDataview());
        this._task.setMode(newDv.taskModeFromCache);
        try {
            if (this.getTask().HasLoacte()) {
                await this.setCurrRec(this._locateFirstRec, true);
                this.setTopRecIdx(this._recordsTab.getRecIdx(this._locateFirstRec));
            }
            else {
                this.getTask().setPreventRecordSuffix(true);
                await this.setCurrRecByIdx(Int32.MinValue, true, false, true, Int32.MinValue);
                this.getTask().setPreventRecordSuffix(false);
                this.setTopRecIdx(0);
            }
            if (!this._task.getForm().isScreenMode()) {
                if (this._task.getForm().isTableWithPagination()) {
                    this.TotalRecordsCount = newDv.TotalRecordsCount;
                    this.RecordsBeforeCurrentView = newDv.RecordsBeforeCurrentView;
                }
                await this.setCurrRecByIdx(this._topRecIdx, true, false, true, Int32.MinValue);
            }
            if (this._task.getForm() !== null)
                this._task.getForm().updateDisplayLineByDV();
        }
        catch (ex) {
            if (ex instanceof RecordOutOfDataViewException) {
                Logger.Instance.WriteDevToLog('wrong top record idx in DataView.setSameAs' + ex.Message);
            }
            else
                throw ex;
        }
    }
    setChanged(val) {
        this._changed = val;
    }
    getChanged() {
        return this._changed;
    }
    getLocateFirstRec() {
        return this._locateFirstRec;
    }
    setLocateFirstRec(newFirst) {
        this._locateFirstRec = newFirst;
    }
    getRecByIdx(idx) {
        return this._recordsTab.getRecByIdx(idx);
    }
    getServerCurrRec() {
        return this._recordsTab.getServerCurrRec();
    }
    zeroServerCurrRec() {
        this._recordsTab.zeroServerCurrRec();
    }
    inRollback() {
        let inRollback = false;
        if (this._recovery === ConstInterface.RECOVERY_ROLLBACK && this.getTask().isTransactionOwner()) {
            inRollback = true;
        }
        return inRollback;
    }
    getRecIdx(id) {
        return this._recordsTab.getRecIdx(id);
    }
    backupCurrent() {
        return this.CurrRec.replicate();
    }
    restoreCurrent(recBackup) {
        Debug.Assert(recBackup !== null);
        this._currRecId = recBackup.getId();
        this._currRecIdx = this._recordsTab.getRecIdx(this._currRecId);
        this.CurrRec = this._recordsTab.getRecByIdx(this._currRecIdx);
        this.CurrRec.setSameAs(recBackup, false);
        this.takeFldValsFromCurrRec();
    }
    GetCurrentRecId() {
        return this._currRecId;
    }
    DataviewBoundriesAreChanged(orgIncludesFirst, orgIncludesLast) {
        return orgIncludesFirst !== this.IncludesFirst() || orgIncludesLast !== this.IncludesLast();
    }
    GetRouteParams() {
        let params = new List();
        for (let i = 0; i < this._fieldsTab.getSize(); i++) {
            let field = this._fieldsTab.getField(i);
            let val;
            if (field.IsExposedRouteParam && !field.isNull()) {
                switch (field.getType()) {
                    case StorageAttribute.ALPHA:
                    case StorageAttribute.UNICODE:
                        {
                            val = field.getDispValue();
                            if (field.isNull())
                                val = field.getNullDisplay();
                            val = val.trim();
                            params.push(val.length === 0 ? ' ' : val);
                        }
                        break;
                    case StorageAttribute.BOOLEAN:
                        {
                            if (field.isNull())
                                params.push(false);
                            else
                                params.push((field.getDispValue()).charAt(0) === '1');
                        }
                        break;
                    case StorageAttribute.NUMERIC: {
                        val = field.getDispValue();
                        if (field.isNull())
                            val = field.getNullDisplay();
                        let pic = new PIC(field.getPicture(), field.getType(), this.getTask().getCompIdx());
                        val = DisplayConvertor.Instance.mg2disp(val, '', pic, true, 0, false);
                        params.push(val);
                        break;
                    }
                }
            }
        }
        return (params.length > 0) ? params : null;
    }
}
//# sourceMappingURL=data:application/json;base64,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