import { ApplicationException, Debug, List, NNumber, NString, RefParam, isNullOrUndefined } from "@magic-xpa/mscorelib";
import { Constants, InternalInterface, Logger, MsgInterface, PICInterface, StorageAttribute, StrUtil, UtilStrByteMode, XMLConstants, XmlParser } from "@magic-xpa/utils";
import { BlobType, DisplayConvertor, ExpVal, FieldDef, NUM_TYPE, PIC, VectorType } from "@magic-xpa/gui";
import { YesNoExp } from "../exp/YesNoExp";
import { ExpressionEvaluator } from "../exp/ExpressionEvaluator";
import { Recompute, Recompute_RcmpBy } from "../rt/Recompute";
import { Boundary } from "../rt/Boundary";
import { ArgumentsList } from "../rt/ArgumentsList";
import { ConstInterface } from "../ConstInterface";
import { RunTimeEvent } from "../event/RunTimeEvent";
import { Record } from "./Record";
import { Environment } from "../env/Environment";
import { FieldBase } from "./FieldBase";
import { LastFocusedManager } from "../rt/LastFocusedManager";
import { LanguageData } from "../env/LanguageData";
import { AccessHelper } from "../AccessHelper";
export class Field extends FieldBase {
    constructor(dataview, id) {
        super(id);
        this._linkExp = null;
        this._recompute = null;
        this.CacheTableFldIdx = 0;
        this._causeTableInvalidation = false;
        this._form = null;
        this._hasChangeEvent = false;
        this._hasZoomHandler = false;
        this._inEvalProcess = false;
        this._initExp = null;
        this._invalidValue = true;
        this._isLinkFld = false;
        this._isVirtual = false;
        this._dataviewHeaderId = -1;
        this._linkCreate = false;
        this.Locate = null;
        this.Range = null;
        this._isParam = false;
        this._isExposedRouteParam = false;
        this._isExposedToClient = true;
        this._prevIsNull = false;
        this._prevValue = null;
        this._tableName = null;
        this._indexInTable = -1;
        this._virAsReal = false;
        this._val = null;
        this.modifiedAtLeastOnce = false;
        this._isNull = false;
        this._clearVectorType = true;
        this._vectorType = null;
        if (Field.NUM0 === null) {
            Field.NUM0 = new NUM_TYPE();
            Field.NUM1 = new NUM_TYPE();
            Field.NUM0.NUM_4_LONG(0);
            Field.NUM1.NUM_4_LONG(1);
        }
        this._dataview = dataview;
        this._linkExp = new YesNoExp(false);
        FieldDef._default_date = DisplayConvertor.Instance.disp2mg(PICInterface.DEFAULT_DATE, new PIC("6", StorageAttribute.NUMERIC, super.getTask().getCompIdx()), super.getTask().getCompIdx(), BlobType.CONTENT_TYPE_UNKNOWN);
    }
    get VirAsReal() {
        return this._virAsReal;
    }
    get IsVirtual() {
        return this._isVirtual;
    }
    get IsExposedRouteParam() {
        return this._isExposedRouteParam;
    }
    get IsExposedToClient() {
        return this._isExposedToClient;
    }
    get IndexInTable() {
        return this._indexInTable;
    }
    setAttribute(attribute, valueStr) {
        let isTagProcessed = super.setAttribute(attribute, valueStr);
        if (!isTagProcessed) {
            let data = StrUtil.tokenize(valueStr, ",");
            isTagProcessed = true;
            switch (attribute) {
                case ConstInterface.MG_ATTR_VIRTUAL:
                    this._isVirtual = (XmlParser.getInt(valueStr) !== 0);
                    break;
                case ConstInterface.MG_ATTR_PARAM:
                    this._isParam = (XmlParser.getInt(valueStr) !== 0);
                    if (!this._isVirtual && this._isParam)
                        throw new ApplicationException("in Field.initElements(): non virtual field is defined as a parameter");
                    break;
                case ConstInterface.MG_ATTR_EXPOSED_ROUTE_PARAMS:
                    this._isExposedRouteParam = (XmlParser.getInt(valueStr) !== 0);
                    if (!this._isParam && this._isExposedRouteParam)
                        throw new ApplicationException("in Field.initElements(): non parameter field is defined as a exposed route parameter");
                    break;
                case ConstInterface.MG_ATTR_EXPOSED_TO_CLIENT:
                    if (XmlParser.getInt(valueStr) == 0)
                        this._isExposedToClient = false;
                    break;
                case ConstInterface.MG_ATTR_VIR_AS_REAL:
                    this._virAsReal = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_LNK_CREATE:
                    this._linkCreate = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_LNKEXP:
                    this._linkExp.setVal((this.getTask()), valueStr);
                    break;
                case ConstInterface.MG_ATTR_TABLE_NAME:
                    this._tableName = XmlParser.unescape(valueStr);
                    break;
                case ConstInterface.MG_ATTR_INDEX_IN_TABLE:
                    this._indexInTable = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_INIT:
                    let expNum = XmlParser.getInt(valueStr);
                    this._initExp = this.getTask().getExpById(expNum);
                    break;
                case XMLConstants.MG_ATTR_NAME:
                    {
                    }
                    break;
                case ConstInterface.MG_ATTR_CHACHED_FLD_ID:
                    if (data[0] != "")
                        this.CacheTableFldIdx = NNumber.Parse(data[1]);
                    break;
                case ConstInterface.MG_ATTR_LOCATE:
                    this.Locate = new Boundary(this.getTask(), NNumber.Parse(data[1]), NNumber.Parse(data[0]), this.getType(), this.getSize(), this.CacheTableFldIdx);
                    break;
                case ConstInterface.MG_ATTR_RANGE:
                    this.Range = new Boundary(this.getTask(), NNumber.Parse(data[1]), NNumber.Parse(data[0]), this.getType(), this.getSize(), this.CacheTableFldIdx);
                    break;
                case ConstInterface.MG_ATTR_LINK:
                    this._dataviewHeaderId = NNumber.Parse(valueStr);
                    break;
                case ConstInterface.MG_ATTR_IS_LINK_FIELD:
                    this._isLinkFld = XmlParser.getBoolean(valueStr);
                    break;
                default:
                    isTagProcessed = false;
                    Logger.Instance.WriteExceptionToLogWithMsg(NString.Format("Unrecognized attribute: '{0}'", attribute));
                    break;
            }
        }
        return isTagProcessed;
    }
    setRecompute(recompRef) {
        this._recompute = recompRef;
    }
    RemoveSubformFromRecompute(subformTask) {
        if (this._recompute !== null) {
            this._recompute.RemoveSubform(subformTask);
        }
    }
    AddSubformRecompute(subformTask) {
        if (this._recompute === null) {
            this._recompute = new Recompute();
            this._recompute.Task = super.getTask();
            this._recompute.OwnerFld = this;
            this._recompute.RcmpMode = Recompute_RcmpBy.CLIENT;
        }
        this._recompute.AddSubform(subformTask);
    }
    SetControl(ctrl) {
        super.SetControl(ctrl);
        if (this._isVirtual && ctrl.IsRepeatable && this._initExp === null) {
            this._causeTableInvalidation = true;
        }
    }
    async compute(recompute) {
        let result = new RefParam(this.getValue(false));
        let isNullFld = new RefParam(this._isNull);
        let rec = this._dataview.getCurrRec();
        let task = super.getTask();
        let zeroReal = this._linkCreate && rec.InCompute && !recompute && task.getMode() === Constants.TASK_MODE_CREATE;
        if (this._form == null)
            this._form = task.getForm();
        if ((this._form != null && this._form.InRestore) || AccessHelper.eventsManager.GetStopExecutionFlag())
            zeroReal = false;
        let rtEvt = AccessHelper.eventsManager.getLastRtEvent();
        let computeOnCancelEvent = false;
        if (rtEvt != null)
            computeOnCancelEvent = rtEvt.getInternalCode() === InternalInterface.MG_ACT_CANCEL;
        if (!this._virAsReal && (this._form == null || !this._form.InRestore) &&
            ((this._isVirtual || (task.getMode() === Constants.TASK_MODE_CREATE && computeOnCancelEvent)) || recompute ||
                zeroReal || rec.isNewRec() && !rec.isComputed())) {
            if (!task.DataView.isEmptyDataview() || (task.DataView.isEmptyDataview() && !this.PartOfDataview)) {
                if ((this._isVirtual || task.getMode() === Constants.TASK_MODE_CREATE) || this._dataview.computeByClient() ||
                    !rec.InCompute || zeroReal || rec.lateCompute()) {
                    if (this._initExp != null) {
                        await this.EvaluateInitExpression(result, isNullFld);
                    }
                    else if (zeroReal)
                        result.value = this.getDefaultValue();
                }
            }
        }
        if (this._invalidValue) {
            this._prevValue = result.value;
            this._prevIsNull = isNullFld.value;
        }
        await this.setValue(result.value, isNullFld.value, recompute, false, recompute && !this._isVirtual, false, false);
    }
    async EvaluateInitExpression(result, isNullFld) {
        Debug.Assert(this._initExp !== null);
        try {
            this._inEvalProcess = true;
            result.value = await this._initExp.evaluateWithResultTypeAndLength(this._type, this._size);
        }
        finally {
            this._inEvalProcess = false;
        }
        isNullFld.value = (result.value === null);
        if (isNullFld.value && this.NullAllowed) {
            result.value = this.getValue(false);
        }
    }
    getValue(checkNullArithmetic) {
        if (super.getTask().getEvalOldValues())
            return this.getOriginalValue();
        if (this._invalidValue)
            this.takeValFromRec();
        if (this._isNull)
            this._val = this.getValueForNull(checkNullArithmetic);
        return this._val;
    }
    isChanged(newValue, isNull) {
        return this._prevValue !== null && (!(this._prevValue === newValue) || this._prevIsNull !== isNull);
    }
    takeValFromRec() {
        let rec = this._dataview.getCurrRec();
        if (rec !== null) {
            this._val = rec.GetFieldValue(this._id);
            this._isNull = rec.IsNull(this._id);
        }
        if (this._invalidValue) {
            if (rec !== null && this._causeTableInvalidation && this.isChanged(this._val, this._isNull) && this._form !== null) {
                rec.setCauseInvalidation(true);
            }
            this._prevValue = this._val;
            this._prevIsNull = this._isNull;
            this._invalidValue = false;
        }
        this._vectorType = null;
    }
    getOriginalValue() {
        let originalRec = this._dataview.getOriginalRec();
        if (originalRec.IsNull(this._id))
            return this.getMagicDefaultValue();
        return originalRec.GetFieldValue(this._id);
    }
    isOriginalValueNull() {
        let originalRec = this._dataview.getOriginalRec();
        return originalRec.IsNull(this._id);
    }
    getValueByRecIdx(idx) {
        let rec = this._dataview.getRecByIdx(idx);
        let fieldValue = rec.GetFieldValue(this._id);
        return rec.IsNull(this._id) ? this.getValueForNull(false) : fieldValue;
    }
    getModifiedAtLeastOnce() {
        return this.modifiedAtLeastOnce;
    }
    isNullByRecIdx(idx) {
        let rec = this._dataview.getRecByIdx(idx);
        return rec.IsNull(this._id);
    }
    isEqual(mgValue, isNullFld, type, recIdx) {
        let rec = (this._dataview.getRecByIdx(recIdx));
        let valsEqual = false;
        if (rec != null) {
            let fieldValue = rec.GetFieldValue(this._id);
            let fielsIsNull = rec.IsNull(this._id);
            valsEqual = ExpressionEvaluator.mgValsEqual(fieldValue, fielsIsNull, this._type, mgValue, isNullFld, type);
        }
        return valsEqual;
    }
    getNewRecValue(clobberedOnly) {
        if (this._isVirtual && this._initExp == null)
            return this._val;
        return !clobberedOnly ? this.getDefaultValue() : null;
    }
    async setValueAndStartRecompute(val, isNullFld, recompute, setRecordUpdated, isArgUpdate, enforceVariableChange) {
        if (isNullOrUndefined(enforceVariableChange))
            enforceVariableChange = false;
        await this.setValueAndStartRecompute_1(val, isNullFld, recompute, setRecordUpdated, isArgUpdate, false);
    }
    async setValueAndStartRecompute_1(val, isNullFld, recompute, setRecordUpdated, isArgUpdate, enforceVariableChange) {
        let task = this.getTask();
        if (task != null)
            task.VewFirst++;
        let valueWasUpdated = await this.setValue(val, isNullFld, recompute, setRecordUpdated, !this._isVirtual, isArgUpdate, enforceVariableChange);
        if (task !== null && task.VewFirst === 1 && this._form !== null && !this.IsEventHandlerField) {
            if (recompute && valueWasUpdated)
                await this._form.refreshOnExpressions();
        }
        if (task != null)
            task.VewFirst--;
    }
    async setValue(newVal, isNullFld, recompute, setRecordModified, setCrsrModified, isArgUpdate, enforceValiableChange) {
        let rec;
        let recFldVal;
        let valsEqual = false;
        let remainder;
        let zeroReal;
        let args = null;
        let mainCtrl = this.getCtrl();
        let pendingEvents;
        let evts;
        let forceUpdate = false;
        let checkForVariableChange = recompute || enforceValiableChange;
        let task = this.getTask();
        if (this._form === null)
            this._form = task.getForm();
        if (isNullFld && !this.NullAllowed && this._type !== StorageAttribute.BLOB_VECTOR) {
            newVal = this.getDefaultValue();
            isNullFld = false;
        }
        if (this._type === StorageAttribute.NUMERIC && newVal != null && !NUM_TYPE.numHexStrIsLong(newVal)) {
            let numVal = new NUM_TYPE(newVal);
            let pic = new PIC(this._picture, this._type, super.getTask().getCompIdx());
            let decs = pic.getDec();
            numVal.round(decs);
            newVal = numVal.toXMLrecord();
        }
        rec = this._dataview.getCurrRec();
        if (rec == null)
            throw new ApplicationException(LanguageData.Instance.getConstMessage(MsgInterface.RT_STR_NO_RECS_IN_RNG));
        if (!this.IsVirtual &&
            (task.getMode() === Constants.TASK_MODE_QUERY &&
                !Environment.Instance.allowUpdateInQueryMode(task.getCompIdx())
                || rec.InForceUpdate && rec.InCompute))
            return !valsEqual;
        if (checkForVariableChange && this.getHasChangeEvent()) {
            let argsList = new Array(2);
            argsList[0] = new ExpVal(StorageAttribute.NUMERIC, false, (mainCtrl !== null && mainCtrl.IsInteractiveUpdate
                ? Field.NUM0.toXMLrecord()
                : Field.NUM1.toXMLrecord()));
            argsList[1] = new ExpVal(this.getType(), this.isNull(), this.getValue(false));
            args = new ArgumentsList(argsList);
        }
        if (mainCtrl != null)
            mainCtrl.IsInteractiveUpdate = false;
        if (this._type !== StorageAttribute.UNICODE && UtilStrByteMode.isLocaleDefLangDBCS()) {
            if (newVal != null)
                remainder = this._size - UtilStrByteMode.lenB(newVal);
            else
                remainder = 0;
            if (newVal != null && this._type !== StorageAttribute.BLOB && this._type !== StorageAttribute.BLOB_VECTOR &&
                remainder < 0)
                this._val = UtilStrByteMode.leftB(newVal, this._size);
            else if (this._type === StorageAttribute.ALPHA && remainder > 0)
                this._val = newVal + NString.FromChars(this._spaces, 0, remainder);
            else if (this._type === StorageAttribute.BLOB)
                this._val = BlobType.copyBlob(this._val, newVal);
            else
                this._val = newVal;
        }
        else {
            if (newVal != null)
                remainder = this._size - newVal.length;
            else
                remainder = 0;
            if (newVal != null && this._type !== StorageAttribute.BLOB && this._type !== StorageAttribute.BLOB_VECTOR && remainder < 0)
                this._val = newVal.substr(0, this._size);
            else if ((this._type === StorageAttribute.ALPHA || this._type === StorageAttribute.UNICODE) && remainder > 0)
                this._val = newVal + NString.FromChars(this._spaces, 0, remainder);
            else if (this._type === StorageAttribute.BLOB) {
                if (this._val === null)
                    this._val = this.getMagicDefaultValue();
                this._val = BlobType.copyBlob(this._val, newVal);
            }
            else
                this._val = newVal;
        }
        {
            recFldVal = rec.GetFieldValue(this._id);
            let oldNullVal = this._isNull;
            if (!recompute && !isArgUpdate && this._initExp == null)
                oldNullVal = rec.IsNull(this._id);
            try {
                valsEqual = ExpressionEvaluator.mgValsEqual(this._val, isNullFld, this._type, recFldVal, oldNullVal, this._type);
            }
            catch (exception) {
                this._val = this.getMagicDefaultValue();
                valsEqual = false;
            }
        }
        if (this._clearVectorType)
            this._vectorType = null;
        if (this._invalidValue) {
            this._invalidValue = false;
            rec.clearFlag(this._id, Record.FLAG_UPDATED);
            rec.clearFlag(this._id, Record.FLAG_MODIFIED);
        }
        zeroReal = this._linkCreate && rec.InCompute && !recompute && (this._form == null || !this._form.InRestore) &&
            !AccessHelper.eventsManager.GetStopExecutionFlag();
        if ((this._isVirtual || zeroReal) && !rec.isNewRec() && !valsEqual)
            forceUpdate = true;
        if (checkForVariableChange || forceUpdate || rec.isNewRec()) {
            this.UpdateNull(isNullFld, rec);
            if (setCrsrModified)
                rec.setFlag(this._id, Record.FLAG_CRSR_MODIFIED);
            rec.setFieldValue(this._id, this._val, setRecordModified);
            evts = Field._variableChangeEvts;
            pendingEvents = evts.length;
            if (checkForVariableChange && this.getHasChangeEvent() && !evts.Contains(this) && !valsEqual) {
                let rtEvt = new RunTimeEvent(this);
                rtEvt.setInternal(InternalInterface.MG_ACT_VARIABLE);
                rtEvt.setArgList(args);
                evts.push(this);
                evts.push(rtEvt);
            }
            if (this._recompute != null && (recompute || forceUpdate) && !this._dataview.InLocalDataviewCommand) {
                try {
                    if (forceUpdate)
                        rec.InForceUpdate = true;
                    await this._recompute.execute(rec);
                }
                finally {
                    if (forceUpdate)
                        rec.InForceUpdate = false;
                }
            }
            if (this._causeTableInvalidation && this.isChanged(this._val, isNullFld))
                rec.setCauseInvalidation(true);
            if ((recompute || forceUpdate) && !this._dataview.InLocalDataviewCommand && mainCtrl != null) {
                if (mainCtrl.isChoiceControl())
                    await this.updateDisplay();
            }
            if (pendingEvents === 0 && checkForVariableChange && this.getHasChangeEvent()) {
                while (evts.length > 0) {
                    let evt = evts.get_Item(1);
                    evts.RemoveAt(0);
                    evts.RemoveAt(0);
                    if (LastFocusedManager.Instance.getLastFocusedTask() == null)
                        evt.setTask(this.getTask());
                    else {
                        let taskRef = LastFocusedManager.Instance.getLastFocusedTask();
                        if (taskRef.TaskSuffixExecuted) {
                            taskRef = (taskRef.PathParentTask != null) ? taskRef.PathParentTask : taskRef;
                        }
                        evt.setTask(taskRef);
                    }
                    evt.setCtrl(LastFocusedManager.getLastFocusedControl());
                    await AccessHelper.eventsManager.handleEvent(evt, false);
                }
            }
        }
        if (recompute && !valsEqual)
            this.setModified();
        return !valsEqual;
    }
    UpdateNull(isNullFld, rec) {
        if (isNullFld)
            rec.setFlag(this._id, Record.FLAG_NULL);
        else
            rec.clearFlag(this._id, Record.FLAG_NULL);
        this._isNull = isNullFld;
    }
    async setCellVecValue(idx, newVal, valIsNull) {
        let res = false;
        if (this._type === StorageAttribute.BLOB_VECTOR) {
            if (this._vectorType == null) {
                this._vectorType = this.isNull()
                    ? new VectorType(this._vecCellsType, this._vecCellsContentType, this.DefaultValue, this.isNullDefault(), this.NullAllowed, this._vecCellsSize)
                    : new VectorType(this._val);
            }
            if (this._vecCellsType === StorageAttribute.BLOB) {
                let tmpNewVal = this.getCellDefualtValue();
                newVal = BlobType.copyBlob(tmpNewVal, newVal);
            }
            res = this._vectorType.setVecCell(idx, newVal, valIsNull);
            if (res) {
                this._clearVectorType = false;
                await this.setValueAndStartRecompute(this._vectorType.toString(), false, true, false, false);
                this._clearVectorType = true;
            }
        }
        return res;
    }
    getVecCellValue(idx) {
        if (this._type === StorageAttribute.BLOB_VECTOR) {
            if (this._vectorType == null)
                this._vectorType = new VectorType(this._val);
            return this._vectorType.getVecCell(idx);
        }
        return null;
    }
    setRecomputed() {
    }
    invalidate(forceInvalidate, clearFlags) {
        let rec = this._dataview.getCurrRec();
        if (!this._isVirtual || this._virAsReal || this._initExp !== null || forceInvalidate) {
            this._invalidValue = true;
        }
        if (clearFlags && rec !== null) {
            rec.clearFlag(this._id, Record.FLAG_UPDATED);
            rec.clearFlag(this._id, Record.FLAG_MODIFIED);
        }
    }
    getTableName() {
        if (this._tableName == null)
            return "";
        return this._tableName;
    }
    getName() {
        if (this._isParam)
            return "Parameter." + this.getVarName();
        else if (this._isVirtual)
            return "Virtual." + this.getVarName();
        else
            return this.getTableName() + "." + this.getVarName();
    }
    isNull() {
        if (super.getTask().getEvalOldValues())
            return this.isOriginalValueNull();
        else if (this._invalidValue)
            this.takeValFromRec();
        return this._isNull;
    }
    PrevIsNull() {
        return this._prevIsNull;
    }
    isModified() {
        let currRec = this._dataview.getCurrRec();
        return currRec.isFldModified(this._id);
    }
    setModified() {
        let rec = (this._dataview.getCurrRec());
        rec.setFlag(this._id, Record.FLAG_MODIFIED);
        rec.setFlag(this._id, Record.FLAG_MODIFIED_ATLEAST_ONCE);
        if (this.IsVirtual)
            this.modifiedAtLeastOnce = true;
    }
    setUpdated() {
        let currRec = this._dataview.getCurrRec();
        currRec.setFlag(this._id, Record.FLAG_UPDATED);
        currRec.setUpdated();
    }
    isParam() {
        return this._isParam;
    }
    isInEvalProcess() {
        return this._inEvalProcess;
    }
    hasInitExp() {
        return this._initExp !== null;
    }
    isServerRcmp() {
        return this._recompute !== null && this._recompute.isServerRcmp();
    }
    getPicture() {
        return this._picture;
    }
    getDataviewHeaderId() {
        return this._dataviewHeaderId;
    }
    get IsLinkField() {
        return this._isLinkFld;
    }
    async getInitExpVal(res, isNull) {
        if (this._initExp != null)
            await this.EvaluateInitExpression(res, isNull);
        else
            res.value = this.getDefaultValue();
    }
    setHasChangeEvent() {
        this._hasChangeEvent = true;
    }
    getHasChangeEvent() {
        return this._hasChangeEvent;
    }
    setHasZoomHandler() {
        this._hasZoomHandler = true;
    }
    getHasZoomHandler() {
        return this._hasZoomHandler;
    }
    isCauseTableInvalidation() {
        return this._causeTableInvalidation;
    }
    causeTableInvalidation(causeTableInvalidation) {
        this._causeTableInvalidation = causeTableInvalidation;
    }
    getValueForNull(checkNullArithmetic) {
        let val;
        if (this.NullAllowed) {
            if (checkNullArithmetic && super.getTask().getNullArithmetic() === Constants.NULL_ARITH_NULLIFY) {
                val = null;
            }
            else {
                if (this._nullValue === null) {
                    this._nullValue = super.getMagicDefaultValue();
                }
                val = this._nullValue;
            }
        }
        else {
            val = super.getMagicDefaultValue();
        }
        return val;
    }
    getCtrl() {
        let ctrl;
        if (this._controls != null) {
            for (let i = 0; i < this._controls.length; i++) {
                ctrl = this._controls.get_Item(i);
                if (ctrl.getForm().getTask() === this.getTask())
                    return ctrl;
            }
        }
        return null;
    }
    async updateDisplay() {
        await super.updateDisplayWithValue(this.getDispValue(), this.isNull());
    }
    getDispValue() {
        if (this.isNull()) {
            if (this._nullDisplay != null)
                return this.getNullDisplay();
            else
                return super.getMagicDefaultValue();
        }
        else
            return this.getValue(false);
    }
    get ShouldCheckRangeInCompute() {
        return this.IsLinkField || (this.IsVirtual && this._dataview.HasMainTable);
    }
    IsForArgument(taskHasParameters) {
        if (taskHasParameters) {
            if (!this.isParam()) {
                return false;
            }
        }
        else {
            if (!this.IsVirtual) {
                return false;
            }
        }
        return true;
    }
}
Field.LEAVE_FLAGS = false;
Field.NUM0 = null;
Field.NUM1 = null;
Field._variableChangeEvts = new List();
//# sourceMappingURL=data:application/json;base64,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