import { DataModificationTypes, EMPTY_DCREF, ExpVal, ObjectReferenceBase, ObjectReferencesCollection, RecordUtils } from '@magic-xpa/gui';
import { ApplicationException, Int32, ISO_8859_1_Encoding, NNumber, NString, RefParam, StringBuilder, isNullOrUndefined } from '@magic-xpa/mscorelib';
import { Base64, Constants, Logger, StorageAttribute, StorageAttributeCheck, StrUtil, UtilStrByteMode, XMLConstants, XmlParser } from '@magic-xpa/utils';
import { ExpressionEvaluator, NullValueException } from '../exp/ExpressionEvaluator';
import { ConstInterface } from '../ConstInterface';
import { Environment } from '../env/Environment';
import { DataViewBase } from './DataViewBase';
import { RemoteCommandsProcessor } from "../remote/RemoteCommandsProcessor";
export class Record {
    get InCompute() {
        return this._inCompute;
    }
    get InRecompute() {
        return this._inRecompute;
    }
    constructor(dvOrTableCacheOrCIdOrRecord, dataview) {
        this._id = Int32.MinValue;
        this.dbViewRowIdx = 0;
        this._mode = DataModificationTypes.None;
        this._dcRefs = null;
        this._inCompute = false;
        this._inRecompute = false;
        this._newRec = false;
        this._dataview = null;
        this._sendToServer = true;
        if (dvOrTableCacheOrCIdOrRecord instanceof Record)
            Object.assign(this, dvOrTableCacheOrCIdOrRecord);
        else if (arguments.length === 1)
            this.constructor_0(dvOrTableCacheOrCIdOrRecord);
        else
            this.constructor_1(dvOrTableCacheOrCIdOrRecord, dataview);
    }
    constructor_0(dvOrTableCache) {
        if (dvOrTableCache instanceof DataViewBase)
            this._dataview = dvOrTableCache;
        else
            this._tableCache = dvOrTableCache;
        this._fieldsData = new Array(this.getFieldsTab().getSize());
        this._flags = new Uint8Array(this.getFieldsTab().getSize());
        this._flagsHistory = new Uint8Array(this.getFieldsTab().getSize());
        this._dcRefs = new ObjectReferencesCollection();
    }
    Initialize() {
        for (let i = 0; i < this._flags.length; i = i + 1) {
            let field = this.getFieldsTab().getField(i);
            this._flags[i] = ((field.isNullDefault() || field.getType() === StorageAttribute.BLOB_VECTOR) ? Record.FLAG_NULL : 0);
            if (this._dataview !== null && field.IsVirtual) {
                let record = this._dataview.getCurrRec();
                if (record != null && record.IsFldModifiedAtLeastOnce(i)) {
                    this.setFlag(i, Record.FLAG_MODIFIED_ATLEAST_ONCE);
                }
            }
            this._flagsHistory[i] = this._flags[i];
        }
        this.setNewRec();
    }
    constructor_1(cId, dataview) {
        this.constructor_0(dataview);
        this.setId(cId);
    }
    get Modified() {
        return this._modified;
    }
    get Updated() {
        return this._updated;
    }
    get SendToServer() {
        return this._sendToServer;
    }
    CompareTo(obj) {
        let res = 0;
        let compTo = obj;
        let key = this._tableCache.GetKeyById(this._tableCache.GetCurrSortKey());
        if (key != null) {
            for (let i = 0; i < key.Columns.length; i++) {
                let currFld = key.Columns.get_Item(i);
                let currFldId = currFld.getId();
                if (this.IsNull(currFldId) && compTo.IsNull(currFldId))
                    continue;
                else if (this.IsNull(currFldId) && !compTo.IsNull(currFldId)) {
                    res = 1;
                    break;
                }
                else if (!this.IsNull(currFldId) && compTo.IsNull(currFldId)) {
                    res = -1;
                    break;
                }
                else {
                    try {
                        let first = new ExpVal(currFld.getType(), false, this.GetFieldValue(currFldId));
                        let second = new ExpVal(currFld.getType(), false, compTo.GetFieldValue(currFldId));
                        res = ExpressionEvaluator.val_cmp_any(first, second);
                        if (res !== 0)
                            break;
                    }
                    catch (err) {
                        if (err instanceof NullValueException)
                            Logger.Instance.WriteExceptionToLogWithMsg(" in Record.CompareTo null value was reached");
                    }
                }
            }
            return res;
        }
        else {
            res = -1;
            return res;
        }
    }
    getMode() {
        return this._mode;
    }
    getId() {
        return this._id;
    }
    isNewRec() {
        return this._newRec;
    }
    getFieldDataXML(fldIdx, getOldVal) {
        let fldVal;
        if (!getOldVal)
            fldVal = this.GetFieldValue(fldIdx);
        else
            fldVal = (this.getFieldsTab().getField(fldIdx)).getOriginalValue();
        if (fldVal == null)
            throw new ApplicationException("in Record.buildFieldsData() null field value!\nField id: " + fldIdx);
        let tmpBuf = this.getXMLForValue(fldIdx, fldVal);
        return XmlParser.escape(tmpBuf.toString()).toString();
    }
    getFieldsTab() {
        if (this._dataview !== null)
            return this._dataview.GetFieldsTab();
        else
            return this._tableCache.FldsTab;
    }
    setInitialFldVals(clobberedOnly, isNewRec) {
        let i, size;
        let fld = null;
        size = this.getSizeFld(true);
        for (i = 0; i < size; i++) {
            if (this._fieldsData[i] === null || typeof this._fieldsData[i] === "undefined") {
                fld = this.getFieldsTab().getField(i);
                this._fieldsData[i] = fld.getNewRecValue(clobberedOnly);
                if (isNewRec && fld.IsVirtual && !fld.isNull())
                    this.clearFlag(fld.getId(), Record.FLAG_NULL);
                if (fld.IsVirtual && fld.getModifiedAtLeastOnce())
                    this.setFlag(fld.getId(), Record.FLAG_MODIFIED_ATLEAST_ONCE);
                if (this._fieldsData[i] === null || typeof this._fieldsData[i] === "undefined")
                    this._fieldsData[i] = fld.getDefaultValue();
            }
        }
    }
    setId(cId) {
        this._id = cId;
        this._hashKey = this._id;
    }
    setDBViewRowIdx(rowId) {
        this.dbViewRowIdx = rowId;
    }
    getDBViewRowIdx() {
        return this.dbViewRowIdx;
    }
    fillData(parser) {
        let isCurrRec = false;
        let endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_TERM, parser.getCurrIndex());
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            let tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(ConstInterface.MG_TAG_REC) + ConstInterface.MG_TAG_REC.length);
            let tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            isCurrRec = this.initElements(tokensVector);
            parser.setCurrIndex(endContext + XMLConstants.TAG_TERM.length);
        }
        else
            Logger.Instance.WriteExceptionToLogWithMsg("in Record.FillInnerData() out of bounds");
        return isCurrRec;
    }
    initElements(tokensVector) {
        let attribute, valueStr;
        let recFieldsData = null;
        let recFlags = null;
        let j = 0;
        let isCurrRec;
        isCurrRec = this.peekIsCurrRec(tokensVector);
        for (; j < tokensVector.length; j += 2) {
            attribute = (tokensVector.get_Item(j));
            valueStr = (tokensVector.get_Item(j + 1));
            switch (attribute) {
                case ConstInterface.MG_ATTR_MODE:
                    this._mode = valueStr[0];
                    break;
                case XMLConstants.MG_ATTR_ID:
                    this.setId(XmlParser.getInt(valueStr));
                    break;
                case XMLConstants.MG_ATTR_VB_VIEW_ROWIDX:
                    this.setDBViewRowIdx(XmlParser.getInt(valueStr));
                    break;
                case XMLConstants.MG_ATTR_VALUE:
                    if (this.ValueInBase64) {
                        recFieldsData = Base64.decode(valueStr);
                    }
                    else {
                        recFieldsData = XmlParser.unescape(valueStr);
                    }
                    break;
                case ConstInterface.MG_ATTR_CURR_REC:
                    {
                    }
                    break;
                case ConstInterface.MG_ATTR_ADD_AFTER:
                    this._addAfter = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_FLAGS:
                    recFlags = valueStr;
                    break;
                case ConstInterface.MG_ATTR_MODIFIED:
                    this._modified = true;
                    break;
                case ConstInterface.MG_ATTR_DC_REFS:
                    this.fillDCRef(valueStr);
                    break;
                case ConstInterface.MG_ATTR_DBPOS:
                    this._dbPosBase64Val = valueStr;
                    break;
                default:
                    Logger.Instance.WriteExceptionToLogWithMsg("in Record.initElements() unknown attribute: " + attribute);
                    break;
            }
        }
        if (this.ValueInBase64) {
            let fldValInBytes = ISO_8859_1_Encoding.ISO_8859_1.GetBytes(recFieldsData);
            this.fillFieldsData(fldValInBytes, recFlags);
        }
        else {
            this.fillFieldsData(recFieldsData, recFlags, isCurrRec);
        }
        return isCurrRec;
    }
    peekIsCurrRec(tokensVector) {
        let attribute;
        let valueStr;
        for (let j = 0; j < tokensVector.length; j += 2) {
            attribute = (tokensVector.get_Item(j));
            if (attribute === ConstInterface.MG_ATTR_CURR_REC) {
                valueStr = (tokensVector.get_Item(j + 1));
                return XmlParser.getBoolean(valueStr);
            }
        }
        return false;
    }
    fillFieldsData(fldValInBytesOrFldsVal, recFlags, isCurrRec) {
        if (isNullOrUndefined(isCurrRec))
            this.fillFieldsData_0();
        else
            this.fillFieldsData_1(fldValInBytesOrFldsVal, recFlags, isCurrRec);
    }
    fillFieldsData_0() {
    }
    fillFieldsData_1(fldsVal, recFlags, isCurrRec) {
        let val = null;
        let tmp = null;
        let parsedLen = new RefParam(0);
        let i, j = 0, from, size;
        let currType;
        let useHex;
        let fld = null;
        let valueNotPassed;
        from = this.getFromFldIdx(isCurrRec);
        size = this.getSizeFld(isCurrRec);
        for (i = from; j < size; i++, j++) {
            fld = this.getFieldsTab().getField(i);
            currType = fld.getType();
            useHex = (Environment.Instance.GetDebugLevel() > 1 ||
                currType === StorageAttribute.ALPHA ||
                currType === StorageAttribute.UNICODE ||
                StorageAttributeCheck.isTypeLogical(currType));
            if (recFlags[j] === '.')
                this._flags[i] = 0;
            else if (recFlags[j] === '/')
                this._flags[i] = 1;
            else {
                tmp = recFlags.substr(j * 2, 2);
                this._flags[i] = parseInt(tmp, 16);
            }
            valueNotPassed = Record.FLAG_VALUE_NOT_PASSED === (this._flags[i] & Record.FLAG_VALUE_NOT_PASSED);
            this._flags[i] = this._flags[i] & ~Record.FLAG_VALUE_NOT_PASSED;
            this._flagsHistory[i] = this._flags[i];
            if (Record.FLAG_UPDATED === (this._flags[i] & Record.FLAG_UPDATED))
                this._updated = true;
            if (valueNotPassed) {
                if (Record.FLAG_NULL === (this._flags[i] & Record.FLAG_NULL)) {
                    val = fld.getDefaultValue();
                }
                else {
                    val = this._dataview.getCurrRec().GetFieldValue(i);
                }
            }
            else {
                val = RecordUtils.deSerializeItemVal(fldsVal, currType, fld.getSize(), useHex, fld.getCellsType(), parsedLen);
                fldsVal = fldsVal.substr(parsedLen.value);
            }
            this._fieldsData[i] = val;
        }
        this.setInitialFldVals(false, false);
    }
    fillDCRef(valueStr) {
        if (!NString.IsNullOrEmpty(valueStr)) {
            let couples = StrUtil.tokenize(valueStr, "$");
            let size = couples.length;
            for (let i = 0; i < size; i = i + 1) {
                this._dcRefs.Add(DcValuesReference.Parse(couples[i], this._dataview));
            }
        }
    }
    buildFieldsData(message, isCurrRec, getOldVal) {
        let tmpStr = new StringBuilder();
        let from = this.getFromFldIdx(isCurrRec);
        let size = this.getSizeFld(isCurrRec);
        for (let i = from; i < from + size; i = i + 1) {
            let fld = this.getFieldsTab().getField(i);
            if (fld.IsExposedToClient === false)
                this.setShrinkFlag(i);
            if (Record.FLAG_VALUE_NOT_PASSED === (this._flags[i] & Record.FLAG_VALUE_NOT_PASSED))
                continue;
            tmpStr.Append(this.getFieldDataXML(i, getOldVal).toString());
        }
        if (tmpStr.Length === 0) {
            tmpStr.Append(" ");
        }
        message.Append(tmpStr.ToString());
    }
    getXMLForValue(fldIdx, fldVal) {
        let cellAttr = StorageAttribute.SKIP;
        let fldAttr = this.getFieldsTab().getType(fldIdx);
        if (fldAttr === StorageAttribute.BLOB_VECTOR)
            cellAttr = this.getFieldsTab().getField(fldIdx).getCellsType();
        let toBase64 = Environment.Instance.GetDebugLevel() <= 1;
        return Record.serializeItemVal(fldVal, fldAttr, cellAttr, toBase64);
    }
    getFromFldIdx(isCurrRec) {
        if (isCurrRec)
            return 0;
        else
            return this.getFieldsTab().getRMIdx();
    }
    getSizeFld(isCurrRec) {
        if (isCurrRec)
            return this.getFieldsTab().getSize();
        else
            return this.getFieldsTab().getRMSize();
    }
    buildXML(message, isCurrRec, forceBuild) {
        if (isNullOrUndefined(forceBuild))
            this.buildXML_0(message, isCurrRec);
        else
            this.buildXML_1(message, isCurrRec, forceBuild);
    }
    buildXML_0(message, isCurrRec) {
        this.buildXML_1(message, isCurrRec, false);
    }
    buildXML_1(message, isCurrRec, forceBuild) {
        let recFlags;
        let from, size, i;
        let aFlag;
        let hexFlag;
        if (!forceBuild)
            if (this._mode !== DataModificationTypes.Insert && this._mode !== DataModificationTypes.Update &&
                this._mode !== DataModificationTypes.Delete && !isCurrRec)
                return;
        message.Append("\n            <" + ConstInterface.MG_TAG_REC);
        if (this._id > Int32.MinValue)
            message.Append(" " + XMLConstants.MG_ATTR_ID + "=\"" + this._id + "\"");
        message.Append(" " + ConstInterface.MG_ATTR_MODE + "=\"" + this._mode + "\"");
        if (this._mode === DataModificationTypes.Insert && this._prev != null)
            message.Append(" " + ConstInterface.MG_ATTR_ADD_AFTER + "=\"" + this._prev.getId() + "\"");
        if (!forceBuild && !isCurrRec)
            this.clearMode();
        message.Append(" " + XMLConstants.MG_ATTR_VALUE + "=\"");
        if (!forceBuild && isCurrRec)
            this.setShrinkFlags();
        let getOldVal;
        if (this.Synced)
            getOldVal = false;
        else {
            let task = this._dataview.getTask();
            getOldVal = (!isCurrRec && (task.getLevel() === Constants.TASK_LEVEL_RECORD ||
                task.getLevel() === Constants.TASK_LEVEL_CONTROL) &&
                this._dataview.getCurrRec() != null && this._dataview.getCurrRec().getId() === this._id);
        }
        this.buildFieldsData(message, isCurrRec, getOldVal);
        message.Append("\"");
        recFlags = new StringBuilder();
        from = this.getFromFldIdx(isCurrRec);
        size = this.getSizeFld(isCurrRec);
        for (i = from; i < from + size; i++) {
            aFlag = this._flagsHistory[i];
            aFlag = aFlag & Record.FLAG_CRSR_MODIFIED;
            aFlag = aFlag | this._flags[i];
            hexFlag = NNumber.ToString(aFlag, "X2");
            recFlags.Append(hexFlag);
            if (isCurrRec)
                this._flags[i] = (this._flags[i] & ~Record.FLAG_VALUE_NOT_PASSED);
        }
        message.Append(" " + ConstInterface.MG_ATTR_FLAGS + "=\"" + recFlags.ToString() + "\"");
        if (isCurrRec) {
            message.Append(" " + ConstInterface.MG_ATTR_CURR_REC + "=\"1\"");
            if (this.Modified)
                message.Append(" " + ConstInterface.MG_ATTR_MODIFIED + "=\"1\"");
        }
        if (this._linksFldsPos != null)
            message.Append(" " + ConstInterface.MG_ATTR_DBPOS + "=\"" + XmlParser.escape(this._linksFldsPos) + "\" ");
        message.Append(XMLConstants.TAG_TERM);
    }
    setShrinkFlags() {
        let i;
        let size = this.getSizeFld(true);
        let serverCurrRec = this._dataview.getServerCurrRec();
        if (serverCurrRec == null)
            return;
        for (i = 0; i < size; i++) {
            if (this.fldValsEqual(serverCurrRec, i) || (Record.FLAG_NULL === (this._flags[i] & Record.FLAG_NULL))) {
                this._flags[i] = this._flags[i] | Record.FLAG_VALUE_NOT_PASSED;
            }
        }
    }
    setShrinkFlag(fldIdx) {
        this._flags[fldIdx] |= Record.FLAG_VALUE_NOT_PASSED;
    }
    SetFieldValue(idx, isNull, value) {
        let field = this.getFieldsTab().getField(idx);
        field.UpdateNull(isNull, this);
        value = this.CheckMgValue(value, isNull, field);
        this._fieldsData[idx] = value;
        field.takeValFromRec();
        this.clearFlag(field.getId(), Record.FLAG_INVALID);
        field.invalidate(true, false);
    }
    setFieldValue(fldIdx, mgVal, setRecordUpdated) {
        let fld = this.getFieldsTab().getField(fldIdx);
        if (fld.PrevIsNull() === fld.isNull() && mgVal === this.GetFieldValue(fldIdx))
            return;
        mgVal = this.CheckMgValue(mgVal, fld.isNull(), fld);
        if (fldIdx >= 0 && fldIdx < this.getFieldsTab().getSize()) {
            this._fieldsData[fldIdx] = mgVal;
            if (setRecordUpdated) {
                if (fld.PartOfDataview)
                    this._modified = true;
                if (RemoteCommandsProcessor.GetInstance().DelayCommandExecution) {
                    RemoteCommandsProcessor.GetInstance().DelayCommandExecution = false;
                    this._dataview.getTask().DelayCommandExecution = true;
                }
                this._dataview.setChanged(true);
                this.setMode(DataModificationTypes.Update);
            }
        }
        else
            Logger.Instance.WriteExceptionToLogWithMsg("in Record.setFieldValue() illegal field index: " + fldIdx);
    }
    CheckMgValue(mgVal, isNull, fld) {
        if (mgVal === null || isNull) {
            mgVal = fld.getNullValue();
            if (mgVal === null)
                mgVal = fld.getMagicDefaultValue();
        }
        let size = fld.getSize();
        if (UtilStrByteMode.isLocaleDefLangDBCS() && fld.getType() !== StorageAttribute.UNICODE) {
            if (mgVal !== null && UtilStrByteMode.lenB(mgVal) > size)
                mgVal = UtilStrByteMode.leftB(mgVal, size);
        }
        else {
            if (mgVal !== null && mgVal.length > size)
                mgVal = mgVal.substr(0, size);
        }
        return mgVal;
    }
    setModified() {
        this._modified = true;
    }
    resetModified() {
        this._modified = false;
    }
    setUpdated() {
        this._updated = true;
    }
    resetUpdated() {
        this._updated = false;
    }
    GetFieldValue(fldIdx) {
        let val = null;
        if (fldIdx >= 0 && fldIdx < this._fieldsData.length)
            val = this._fieldsData[fldIdx];
        return val.toString();
    }
    replicate() {
        let record = new Record(this);
        record._fieldsData = this._fieldsData.slice();
        record._flags = this._flags.subarray(0);
        record._flagsHistory = this._flagsHistory.subarray(0);
        record._dcRefs = this._dcRefs.Clone();
        return record;
    }
    setSameAs(rec, realOnly, rId) {
        if (arguments.length === 3)
            this.setSameAs_0(rec, realOnly, rId);
        else
            this.setSameAs_1(rec, realOnly);
    }
    setSameAs_0(rec, realOnly, rId) {
        this.setSameAs(rec, realOnly);
        this.setId(rId);
    }
    setSameAs_1(rec, realOnly) {
        this._id = rec._id;
        this._hashKey = rec._hashKey;
        if (!this._dataview.getTask().transactionFailed(ConstInterface.TRANS_RECORD_PREFIX))
            this.setMode(rec._mode);
        this._addAfter = rec._addAfter;
        if (realOnly) {
            for (let i = 0; i < this.getFieldsTab().getSize(); i = i + 1) {
                let afield = rec.getFieldsTab().getField(i);
                if (!afield.IsVirtual) {
                    this._fieldsData[i] = rec._fieldsData[i];
                    this._flags[i] = rec._flags[i];
                    this._flagsHistory[i] = rec._flagsHistory[i];
                }
            }
        }
        else {
            this._fieldsData = rec._fieldsData;
            this._flags = rec._flags;
            this._flagsHistory = rec._flagsHistory;
        }
        this._modified = rec._modified;
        this._dataview = rec._dataview;
        this._computed = rec._computed;
        this._updated = rec._updated;
        this.setDcRefs(rec._dcRefs);
        if (rec._next !== this) {
            this._next = rec._next;
        }
        if (rec._prev !== this) {
            this._prev = rec._prev;
        }
    }
    IsNull(fldIdx) {
        this.checkFlags(fldIdx);
        return ((this._flags[fldIdx] & Record.FLAG_NULL) === Record.FLAG_NULL);
    }
    isLinkInvalid(fldIdx) {
        this.checkFlags(fldIdx);
        return ((this._flags[fldIdx] & Record.FLAG_INVALID) === Record.FLAG_INVALID);
    }
    isFldModified(fldIdx) {
        this.checkFlags(fldIdx);
        return ((this._flags[fldIdx] & Record.FLAG_MODIFIED) === Record.FLAG_MODIFIED);
    }
    IsFldModifiedAtLeastOnce(fldIdx) {
        this.checkFlags(fldIdx);
        return ((this._flags[fldIdx] & Record.FLAG_MODIFIED_ATLEAST_ONCE) === Record.FLAG_MODIFIED_ATLEAST_ONCE);
    }
    isFldUpdated(fldIdx) {
        this.checkFlags(fldIdx);
        return ((this._flags[fldIdx] & Record.FLAG_UPDATED) === Record.FLAG_UPDATED);
    }
    setFlag(fldIdx, aFlag) {
        this.checkFlags(fldIdx);
        this._flags[fldIdx] = this._flags[fldIdx] | aFlag;
        if (aFlag === Record.FLAG_CRSR_MODIFIED)
            this._flagsHistory[fldIdx] = this._flagsHistory[fldIdx] | aFlag;
    }
    clearFlag(fldIdx, aFlags) {
        this.checkFlags(fldIdx);
        this._flags[fldIdx] = this._flags[fldIdx] & ~aFlags;
    }
    clearFlagsHistory() {
        for (let i = 0; i < this._flags.length; i = i + 1)
            this._flagsHistory[i] = 0;
    }
    clearFlags(aFlags) {
        for (let i = 0; i < this._flags.length; i = i + 1)
            this.clearFlag(i, aFlags);
    }
    clearFlagsForRealOnly(aFlags) {
        for (let i = 0; i < this._flags.length; i = i + 1) {
            let field = this.getFieldsTab().getField(i);
            if (!field.IsVirtual) {
                this.clearFlag(i, aFlags);
            }
        }
    }
    clearHistoryFlag(fldIdx) {
        if (this._flagsHistory !== null && fldIdx < this.getFieldsTab().getSize() && fldIdx >= 0) {
            this._flagsHistory[fldIdx] = 0;
        }
    }
    checkFlags(fldIdx) {
        if (this._flags === null || fldIdx >= this.getFieldsTab().getSize() || fldIdx < 0) {
            throw new ApplicationException("Cannot find flags");
        }
    }
    restart(oldMode) {
        let task = this._dataview.getTask();
        if (oldMode === DataModificationTypes.None) {
            let rec = this._dataview.getCurrRec();
            let isCurrRec = (rec != null && rec.getId() === this._id);
            if ((this._dataview.getTask()).getMode() === Constants.TASK_MODE_CREATE)
                if (isCurrRec && !task.getAfterRetry() && task.TryingToCommit && !this._dataview.inRollback())
                    oldMode = DataModificationTypes.Update;
                else
                    oldMode = DataModificationTypes.Insert;
            else
                oldMode = DataModificationTypes.Update;
        }
        this.setMode(oldMode);
        if (oldMode === DataModificationTypes.Insert)
            this.setNewRec();
    }
    setLateCompute(val) {
        this._lateCompute = val;
    }
    lateCompute() {
        return this._lateCompute;
    }
    setInDeleteProcess(val) {
        this._inDeleteProcess = val;
    }
    inDeleteProcess() {
        return this._inDeleteProcess;
    }
    removeRecFromDc() {
        if (this._dcRefs !== null) {
            this._dcRefs.Dispose();
        }
    }
    isSameRecData(rec, currRec, checkOnlyParetOfDataview) {
        let size = this.getSizeFld(currRec);
        let start = this.getFromFldIdx(currRec);
        let i;
        let field;
        if (this === rec)
            return true;
        try {
            if (rec.getSizeFld(currRec) !== size)
                return false;
            if (rec.getFromFldIdx(currRec) !== start)
                return false;
            for (i = 0; i < start + size; i++) {
                field = this.getFieldsTab().getField(i);
                if (checkOnlyParetOfDataview && !field.PartOfDataview)
                    continue;
                if (!this.fldValsEqual(rec, i))
                    return false;
            }
        }
        catch (err) {
            if (err instanceof ApplicationException)
                return false;
            throw err;
        }
        return true;
    }
    fldValsEqual(rec, fldIdx) {
        let type = this.getFieldsTab().getField(fldIdx).getType();
        return ExpressionEvaluator.mgValsEqual(this.GetFieldValue(fldIdx), this.IsNull(fldIdx), type, rec.GetFieldValue(fldIdx), rec.IsNull(fldIdx), type);
    }
    getHashKey() {
        return this._hashKey;
    }
    setNextRec(nextRec) {
        this._next = nextRec;
    }
    setPrevRec(prevRec) {
        this._prev = prevRec;
    }
    getPrevRec() {
        return this._prev;
    }
    getNextRec() {
        return this._next;
    }
    getDcRefs() {
        return this._dcRefs.Clone();
    }
    setDcRefs(newDcRefs) {
        this.removeRecFromDc();
        this._dcRefs = newDcRefs.Clone();
        this.SetDcValueId();
    }
    realModified() {
        let bRc = false;
        let tabSize = this.getFieldsTab().getSize();
        let j;
        for (j = 0; j < tabSize; j++) {
            if ((this._flags[j] & Record.FLAG_MODIFIED) === Record.FLAG_MODIFIED) {
                if (!(this.getFieldsTab().getField(j).IsVirtual)) {
                    bRc = true;
                    break;
                }
            }
        }
        return bRc;
    }
    ToString() {
        let str = new StringBuilder();
        this.buildXML(str, (this._dataview.getCurrRec()).getId() === this._id, true);
        return str.ToString();
    }
    setSendToServer(val) {
        this._sendToServer = val;
    }
    getRecSize() {
        let sum = 0;
        for (let i = 0; i < this.getFieldsTab().getSize(); i = i + 1) {
            let field = this.getFieldsTab().getField(i);
            sum = sum + field.getSize();
        }
        return sum;
    }
    copyCrsrModifiedFlags(rec) {
        let tabSize = this.getFieldsTab().getSize();
        let j;
        for (j = 0; j < tabSize; j++) {
            this._flags[j] = rec._flags[j];
            this._flags[j] = this._flags[j] & Record.FLAG_CRSR_MODIFIED;
        }
    }
    getDbPosVal() {
        return this._dbPosBase64Val;
    }
    buildLinksPosStr() {
        let tbl = this._dataview.getTask().getDataviewHeaders();
        if (tbl !== null) {
            this._linksFldsPos = tbl.buildDbPosString();
        }
    }
    setForceSaveOrg(val) {
        this._forceSaveOrg = val;
    }
    getForceSaveOrg() {
        return this._forceSaveOrg;
    }
    isCauseInvalidation() {
        return this._causeInvalidation;
    }
    setCauseInvalidation(causeInvalidation) {
        this._causeInvalidation = causeInvalidation;
    }
    isComputed() {
        return this._computed;
    }
    setComputed(val) {
        this._computed = val;
    }
    setInCompute(val) {
        if (!val)
            this.setComputed(true);
        this._inCompute = val;
    }
    setInRecompute(val) {
        this._inRecompute = val;
    }
    setMode(newMode) {
        switch (newMode) {
            case DataModificationTypes.None:
            case DataModificationTypes.Insert:
            case DataModificationTypes.Update:
            case DataModificationTypes.Delete:
                if (this._mode === DataModificationTypes.None ||
                    this._mode === DataModificationTypes.Update && newMode === DataModificationTypes.Delete)
                    this._mode = newMode;
                break;
            default:
                Logger.Instance.WriteExceptionToLogWithMsg("in Record.setMode(): illegal mode: " + newMode);
                break;
        }
    }
    clearMode() {
        this._mode = DataModificationTypes.None;
    }
    setOldRec() {
        this._newRec = false;
    }
    setNewRec() {
        this._newRec = true;
    }
    static serializeItemVal(itemVal, itemAttr, cellAttr, ToBase64) {
        return RecordUtils.serializeItemVal(itemVal, itemAttr, cellAttr, ToBase64);
    }
    SetDcValueId() {
        let form = this._dataview.getTask().getForm();
        let mgControl = null;
        if (form != null && this._dcRefs != null) {
            this._dcRefs.Refs.forEach((dcRef) => {
                mgControl = form.getCtrl(dcRef.ditIdx);
                if (mgControl != null)
                    mgControl.setDcValId(dcRef.DcValues.getId());
            });
        }
    }
    resetDcValueId() {
        let form = this._dataview.getTask().getForm();
        let mgControl = null;
        if (form != null && this._dcRefs != null) {
            this._dcRefs.Refs.forEach(function (dcRef) {
                mgControl = form.getCtrl(dcRef.ditIdx);
                if (mgControl != null)
                    mgControl.setDcValId(EMPTY_DCREF);
            });
        }
    }
    AddDcValuesReference(controlId, dcValuesId) {
        let dcValues = this._dataview.getDcValues(dcValuesId);
        let dcRef = new DcValuesReference(controlId, dcValues);
        if (this._dcRefs === null) {
            this._dcRefs = new ObjectReferencesCollection();
        }
        this._dcRefs.Add(dcRef);
    }
}
Record.FLAG_NULL = (0x01);
Record.FLAG_INVALID = (0x02);
Record.FLAG_MODIFIED = (0x04);
Record.FLAG_UPDATED = (0x08);
Record.FLAG_CRSR_MODIFIED = (0x10);
Record.FLAG_VALUE_NOT_PASSED = (0x20);
Record.FLAG_MODIFIED_ATLEAST_ONCE = (0x40);
Record.INCREASE = true;
Record.DECREASE = false;
export class DcValuesReference extends ObjectReferenceBase {
    get DcValues() {
        return this.Referent;
    }
    constructor(controlId, referencedDcValues) {
        super(referencedDcValues);
        this.ditIdx = 0;
        this.ditIdx = controlId;
    }
    Clone() {
        return new DcValuesReference(this.ditIdx, this.Referent);
    }
    static Parse(couple, dcValuesOwner) {
        let commaPos = couple.indexOf(",");
        let controlId = NNumber.Parse(couple.substr(0, commaPos));
        let dcId = NNumber.Parse(couple.substr(commaPos + 1));
        return new DcValuesReference(controlId, dcValuesOwner.getDcValues(dcId));
    }
}
//# sourceMappingURL=data:application/json;base64,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