import { Record } from "./Record";
import { DataModificationTypes } from "@magic-xpa/gui";
import { ApplicationException, Hashtable, Int32, List } from "@magic-xpa/mscorelib";
import { Logger } from "@magic-xpa/utils";
export class RecordsTable {
    constructor(withLinkedListOrRecordTable) {
        this._useLinkedList = false;
        this._hashTab = null;
        this._initialCurrRecId = Int32.MinValue;
        this._records = null;
        this._serverCurrRec = null;
        this.InsertedRecordsCount = 0;
        if (withLinkedListOrRecordTable.constructor === Boolean) {
            this._records = new List();
            this._hashTab = new Hashtable(100, 0.7);
            this._useLinkedList = withLinkedListOrRecordTable;
            this.InsertedRecordsCount = Int32.MinValue;
        }
        else
            Object.assign(this, withLinkedListOrRecordTable);
    }
    fillData(dataview, insertAt, parser) {
        let c = 'N';
        let record = null;
        let flag = false;
        this.InsertedRecordsCount = 0;
        this._initialCurrRecId = Int32.MinValue;
        let nextTag = parser.getNextTag();
        while (nextTag !== null && nextTag === "rec") {
            let record2 = new Record(dataview);
            record2.Initialize();
            let isCurrRec = record2.fillData(parser);
            if (isCurrRec && !dataview.HasMainTable) {
                let dcRefs2;
                let dcRefs = dcRefs2 = record2.getDcRefs();
                try {
                    for (let i = 0; i < this.getSize(); i = i + 1) {
                        this.getRecByIdx(i).setDcRefs(dcRefs);
                    }
                }
                finally {
                    if (dcRefs2 !== null) {
                        (dcRefs2).Dispose();
                    }
                }
            }
            if (isCurrRec) {
                if (dataview.getTask().getMode() !== 'C') {
                    record2.setOldRec();
                }
                this._initialCurrRecId = record2.getId();
                this._serverCurrRec = record2.replicate();
                if (c === 'N') {
                    c = 'C';
                }
                else {
                    c = 'F';
                }
            }
            else {
                record2.setOldRec();
                c = 'T';
            }
            let foundRec = this.getRecord(record2.getId());
            if (foundRec !== null) {
                let updated = foundRec.Updated;
                let modified = foundRec.Modified;
                let computed = foundRec.isComputed();
                let prevRec = foundRec.getPrevRec();
                let nextRec = foundRec.getNextRec();
                foundRec.setSameAs(record2, false);
                foundRec.setPrevRec(prevRec);
                foundRec.setNextRec(nextRec);
                foundRec.setComputed(computed);
                if (modified)
                    foundRec.setModified();
                if (updated)
                    foundRec.setUpdated();
            }
            else {
                this.InsertRecord(insertAt, record2);
                if (dataview.getTask().isTableWithAbsolutesScrollbar()) {
                    if (dataview.TotalRecordsCount === 0 && dataview.RecordsBeforeCurrentView == 0)
                        dataview.TotalRecordsCount = dataview.TotalRecordsCount + 1;
                    if (dataview.RecordsBeforeCurrentView > 0 && insertAt === 'B')
                        dataview.RecordsBeforeCurrentView = dataview.RecordsBeforeCurrentView - 1;
                }
                if (isCurrRec && dataview.getTask().getMode() === 'C')
                    record2.setMode(DataModificationTypes.Insert);
                record2.setComputed(true);
            }
            nextTag = parser.getNextTag();
            if (!isCurrRec)
                record = record2;
            flag = true;
        }
        let num = 0;
        while (!dataview.HasMainTable && record !== null && num < dataview.getChunkSize() - 1) {
            let record4 = new Record(dataview);
            record4.Initialize();
            record4.setSameAs(record, false, record.getId() + num + 1);
            record4.setOldRec();
            this.addRec(record4);
            num = num + 1;
        }
        if (this._initialCurrRecId === Int32.MinValue) {
            if (flag)
                this._serverCurrRec = null;
            else {
                if (dataview.getCurrRec() !== null)
                    this._serverCurrRec = (dataview.getCurrRec()).replicate();
                else
                    this._serverCurrRec = null;
            }
        }
        return c;
    }
    InsertRecord(insertAt, record) {
        if (insertAt === 'B') {
            let insertedRecordsCount = this.InsertedRecordsCount;
            this.InsertedRecordsCount = insertedRecordsCount + 1;
            this.insertRecord(record, 0);
        }
        else
            this.addRec(record);
    }
    buildXML(message, skipCurrRec, currRecId) {
        for (let i = 0; i < this._records.length; i = i + 1) {
            let record = this._records.get_Item(i);
            if (!record.SendToServer || (skipCurrRec && record.getId() === currRecId))
                continue;
            let prevRec;
            let record2 = prevRec = record.getPrevRec();
            let recIdx;
            while (prevRec !== null && typeof prevRec !== "undefined" &&
                (prevRec.getMode() === DataModificationTypes.Insert || !prevRec.SendToServer) &&
                record.getMode() === DataModificationTypes.Insert &&
                ((recIdx = this.getRecIdx(prevRec.getId())) > i || recIdx === RecordsTable.REC_NOT_FOUND || !prevRec.SendToServer)) {
                prevRec = prevRec.getPrevRec();
                record.setPrevRec(prevRec);
            }
            record.buildXML(message, false);
            if (prevRec !== record2) {
                record.setPrevRec(record2);
            }
        }
    }
    getRecord(id) {
        return this._hashTab.get_Item(id);
    }
    getRecIdx(id) {
        let result = RecordsTable.REC_NOT_FOUND;
        let record = this.getRecord(id);
        if (record !== null) {
            result = this._records.indexOf(record);
        }
        return result;
    }
    getRecByIdx(idx) {
        if (idx < 0 || idx >= this._records.length) {
            Logger.Instance.WriteDevToLog("in RecordsTable.getRecByIdx() index out of bounds: " + idx);
            return null;
        }
        return this._records.get_Item(idx);
    }
    removeAll() {
        this._records.Clear();
        this._hashTab.Clear();
    }
    getSize() {
        return this._records.length;
    }
    getInitialCurrRecId() {
        return this._initialCurrRecId;
    }
    addRecord(rec) {
        if (this._records.indexOf(rec) < 0) {
            this.addRec(rec);
        }
    }
    addRec(rec) {
        if (this._useLinkedList && this._records.length > 0) {
            let record = this._records.get_Item(this._records.length - 1);
            record.setNextRec(rec);
            rec.setPrevRec(record);
        }
        this._records.push(rec);
        this._hashTab.set_Item(rec.getHashKey(), rec);
    }
    insertRecord(rec, idx) {
        if (this._useLinkedList) {
            if (idx > 0) {
                let record = this._records.get_Item(idx - 1);
                rec.setPrevRec(record);
                record.setNextRec(rec);
            }
            if (idx !== this._records.length) {
                let record2 = this._records.get_Item(idx);
                rec.setNextRec(record2);
                record2.setPrevRec(rec);
            }
        }
        this._records.Insert(idx, rec);
        this._hashTab.set_Item(rec.getHashKey(), rec);
    }
    removeRecord(recIdxOrRec) {
        if (arguments.length === 1 && (recIdxOrRec === null || recIdxOrRec.constructor === Number)) {
            this.removeRecord_0(recIdxOrRec);
            return;
        }
        this.removeRecord_1(recIdxOrRec);
    }
    removeRecord_0(recIdx) {
        if (recIdx >= 0 && recIdx < this._records.length) {
            let record = this._records.get_Item(recIdx);
            let useLinkedList = this._useLinkedList;
            if (useLinkedList) {
                if (recIdx > 0) {
                    let record2 = this._records.get_Item(recIdx - 1);
                    record2.setNextRec(record.getNextRec());
                }
                if (recIdx + 1 < this._records.length) {
                    let record2 = this._records.get_Item(recIdx + 1);
                    record2.setPrevRec(record.getPrevRec());
                }
            }
            this._hashTab.Remove(record.getHashKey());
            this._records.RemoveAt(recIdx);
            return;
        }
        throw new ApplicationException("in RecordsTable.removeRecord(): invalid index: " + recIdx);
    }
    removeRecord_1(rec) {
        let num = this._records.indexOf(rec);
        if (num >= 0) {
            this.removeRecord(num);
        }
    }
    replicate() {
        let recordsTable = new RecordsTable(this);
        recordsTable._records = new List();
        recordsTable._hashTab = new Hashtable(100, 0.7);
        for (let i = 0; i < this._records.length; i = i + 1) {
            let record = this._records.get_Item(i).replicate();
            recordsTable._records.push(record);
            recordsTable._hashTab.set_Item(record.getHashKey(), record);
        }
        return recordsTable;
    }
    getServerCurrRec() {
        return this._serverCurrRec;
    }
    zeroServerCurrRec() {
        this._serverCurrRec = null;
    }
    GetSize() {
        return this.getSize();
    }
    GetRecByIdx(idx) {
        return this.getRecByIdx(idx);
    }
    RemoveAll() {
        this.removeAll();
    }
}
RecordsTable.REC_NOT_FOUND = -1;
//# sourceMappingURL=data:application/json;base64,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