import { DisplayConvertor } from '@magic-xpa/gui';
import { Debug, Encoding, Hashtable, NString } from '@magic-xpa/mscorelib';
import { Constants, Logger, OSEnvironment, UtilDateJpn, UtilStrByteMode, XMLConstants, XmlParser } from '@magic-xpa/utils';
import { CommandsProcessorManager } from '../CommandsProcessorManager';
import { ConstInterface } from '../ConstInterface';
import { ServerConfig } from '../ServerConfig';
import { AccessHelper } from '../AccessHelper';
export class Environment {
    constructor() {
        this._environments = new Hashtable();
        this._contextInactivityTimeout = 0;
        this._clientContextInactivityTimeout = 0;
        this._contextInactivityWarningTime = 0;
        this._toolitipTimeout = 0;
        this._contextUnloadTimeout = 0;
        this._canReplaceDecimalSeparator = false;
        this._codePage = 0;
        this._dateSeparator = '\0';
        this._decimalSeparator = '\0';
        this._thousandsSeparator = '\0';
        this._timeSeparator = '\0';
        this._guid = null;
        this._imeAutoOff = false;
        this._language = '\0';
        this._localAs400Set = false;
        this._localExtraGengo = null;
        this._localFlags = null;
        this._lowHigh = true;
        this._owner = null;
        this._debugMode = 0;
        this._significantNumSize = 0;
        this._specialAnsiExpression = false;
        this._specialImmediatehandlerEnd = false;
        this._specialShowStatusBarPanes = false;
        this._specialRouteToRootProgOnContextRecreation = false;
        this._specialCancelOnCreate = false;
        this._system = null;
        this._terminal = 0;
        this._forwardSlash = Constants.ForwardSlashWebUsage;
        this._dropUserFormats = null;
        this._webclientReinitSession = "N";
        this._specialValNewPolicy = false;
        this.IgnoreReplaceDecimalSeparator = false;
    }
    get Language() {
        return this._language;
    }
    get SpecialRestoreMaximizedForm() {
        return false;
    }
    set ForwardSlashUsage(value) {
        this._forwardSlash = value;
    }
    get ForwardSlashUsage() {
        return this._forwardSlash;
    }
    fillData(tokensVectorOrParser) {
        if (arguments[0] instanceof XmlParser) {
            this.fillData_0(tokensVectorOrParser);
            return;
        }
        this.fillData_1(tokensVectorOrParser);
    }
    fillData_0(parser) {
        let endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_TERM, parser.getCurrIndex());
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            let tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(ConstInterface.MG_TAG_ENV) + ConstInterface.MG_TAG_ENV.length);
            let tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            this.fillData(tokensVector);
            parser.setCurrIndex(endContext + XMLConstants.TAG_TERM.length);
        }
        else
            Logger.Instance.WriteExceptionToLogWithMsg("in Environment.FillData() out of string bounds");
    }
    fillData_1(tokensVector) {
        let attribute;
        let valueStr;
        let hashKey;
        let env;
        env = new EnvironmentDetails();
        for (let j = 0; j < tokensVector.length; j += 2) {
            attribute = (tokensVector.get_Item(j));
            valueStr = (tokensVector.get_Item(j + 1));
            valueStr = XmlParser.unescape(valueStr);
            switch (attribute) {
                case ConstInterface.MG_ATTR_THOUSANDS:
                    this._thousandsSeparator = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_DECIMAL_SEPARATOR:
                    this._decimalSeparator = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_WEBCLIENT_REINIT_SESSION:
                    this._webclientReinitSession = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_DATE:
                    this._dateSeparator = valueStr[0];
                    DisplayConvertor.Instance.setDateChar(this._dateSeparator.charCodeAt(0));
                    break;
                case ConstInterface.MG_ATTR_TIME:
                    this._timeSeparator = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_OWNER:
                    this._owner = valueStr;
                    break;
                case ConstInterface.MG_ATTR_SIGNIFICANT_NUM_SIZE:
                    this._significantNumSize = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_DEBUG_MODE:
                    this._debugMode = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_POINT_TRANSLATION:
                    this._canReplaceDecimalSeparator = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_SPECIAL_EXITCTRL:
                    break;
                case ConstInterface.MG_ATTR_LOWHIGH:
                    this._lowHigh = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_ACCESS_TEST:
                    break;
                case ConstInterface.MG_ATTR_SPECIAL_TEXT_SIZE_FACTORING:
                    break;
                case ConstInterface.MG_ATTR_SPECIAL_FLAT_EDIT_ON_CLASSIC_THEME:
                    break;
                case ConstInterface.MG_ATTR_ENCODING:
                    if (valueStr !== " ") {
                        this._codePage = XmlParser.getInt(valueStr);
                        UtilStrByteMode.Encoding = this.GetEncoding();
                    }
                    break;
                case ConstInterface.MG_ATTR_SYSTEM:
                    this._system = XmlParser.unescape(valueStr);
                    break;
                case ConstInterface.MG_ATTR_COMPONENT:
                    env.CompIdx = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_DATEMODE:
                    env.DateMode = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_CENTURY:
                    env.Century = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_IDLETIME:
                    env.IdleTime = XmlParser.getInt(valueStr);
                    if (env.IdleTime === 0)
                        env.IdleTime = 1;
                    break;
                case ConstInterface.MG_ATTR_UPD_IN_QUERY:
                    env.UpdateInQueryMode = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_CRE_IN_MODIFY:
                    env.CreateInModifyMode = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_DEFAULT_COLOR:
                    break;
                case ConstInterface.MG_ATTR_DEFAULT_FOCUS_COLOR:
                    break;
                case ConstInterface.MG_ATTR_CONTEXT_INACTIVITY_TIMEOUT:
                    this._contextInactivityTimeout = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_CLIENT_CONTEXT_INACTIVITY_TIMEOUT:
                    this._clientContextInactivityTimeout = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_CONTEXT_INACTIVITY_WARNING_TIME:
                    this._contextInactivityWarningTime = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_TOOLTIP_TIMEOUT:
                    this._toolitipTimeout = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_CONTEXT_UNLOAD_TIMEOUT:
                    this._contextUnloadTimeout = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_IME_AUTO_OFF:
                    this._imeAutoOff = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_LOCAL_AS400SET:
                    this._localAs400Set = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_LOCAL_EXTRA_GENGO:
                    this._localExtraGengo = valueStr;
                    UtilDateJpn.getInstance().addExtraGengo(this._localExtraGengo);
                    break;
                case ConstInterface.MG_ATTR_LOCAL_FLAGS:
                    this._localFlags = valueStr;
                    break;
                case ConstInterface.MG_ATTR_SPECIAL_IMMEDIATE_HANDLER_END:
                    this._specialImmediatehandlerEnd = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_SPEACIAL_ANSI_EXP:
                    this._specialAnsiExpression = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_SPECIAL_SHOW_STATUSBAR_PANES:
                    this._specialShowStatusBarPanes = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_SPECIAL_ROUTE_TO_ROOT_PROG_ON_CONTEXT_RECREATION:
                    this._specialRouteToRootProgOnContextRecreation = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_SPECIAL_SPECIAL_EDIT_LEFT_ALIGN:
                    break;
                case ConstInterface.MG_ATTR_SPEACIAL_SWF_CONTROL_NAME:
                    break;
                case ConstInterface.MG_ATTR_LANGUAGE:
                    this._language = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_USERID:
                    break;
                case ConstInterface.MG_TAG_USERNAME:
                    ServerConfig.Instance.setUsername(valueStr);
                    break;
                case ConstInterface.MG_ATTR_TERMINAL:
                    this._terminal = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_USERINFO:
                    break;
                case ConstInterface.MG_ATTR_GUID:
                    this._guid = valueStr;
                    break;
                case ConstInterface.MG_ATTR_CONTROLS_PERSISTENCY_PATH:
                    break;
                case ConstInterface.MG_ATTR_PROJDIR:
                    env.ProjDir = valueStr;
                    break;
                case ConstInterface.MG_ATTR_DROP_USERFORMATS:
                    this._dropUserFormats = valueStr;
                    break;
                case ConstInterface.MG_ATTR_SPECIAL_VAL_NEW_POLICY:
                    this._specialValNewPolicy = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_SPECIAL_CANCEL_ON_CREATE:
                    this._specialCancelOnCreate = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_SPECIAL_LEGACY_TRIM_RANGE:
                case ConstInterface.MG_ATTR_SPECIAL_LEGACY_EMPTY_RANGE:
                case ConstInterface.MG_ATTR_DEFAULT_DISABLED_COLOR:
                case ConstInterface.MG_ATTR_SPECIAL_ANSI_BLOB_CHECK_NULL:
                case ConstInterface.MG_ATTR_SPECIAL_TABLE_REDUCE_TITLE_HEIGHT:
                case ConstInterface.MG_ATTR_SPECIAL_DOTNET_ALLOW_ZERO_DATE:
                case ConstInterface.MG_ATTR_SPECIAL_REUSE_TABLE_EDITOR:
                    break;
                default:
                    Logger.Instance.WriteExceptionToLogWithMsg("in Environment.fillData(): unknown attribute: " + attribute);
                    break;
            }
        }
        hashKey = env.CompIdx;
        this._environments.set_Item(hashKey, env);
    }
    async fillFromUrl(tagName, parser) {
        let XMLdata = parser.getXMLdata();
        let endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_TERM, parser.getCurrIndex());
        if (endContext !== -1 && endContext < XMLdata.length) {
            let tagAndAttributes = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tagAndAttributes.indexOf(tagName) + tagName.length);
            let tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            Debug.Assert(tokensVector.get_Item(0) === XMLConstants.MG_ATTR_VALUE);
            let cachedFileUrl = (tokensVector.get_Item(1));
            if (cachedFileUrl.trim() === "")
                Logger.Instance.WriteErrorToLog(NString.Format("Empty cached file URL: '{0}'", tagAndAttributes.trim()));
            else {
                let Content = await CommandsProcessorManager.GetContent(cachedFileUrl, true);
                try {
                    switch (tagName) {
                        case ConstInterface.MG_TAG_KBDMAP_URL:
                            break;
                        case ConstInterface.MG_TAG_ENV_PARAM_URL:
                            let innerXmlParser = new XmlParser(Content);
                            while (AccessHelper.envParamsTable.mirrorFromXML(innerXmlParser.getNextTag(), innerXmlParser)) {
                            }
                            break;
                    }
                }
                catch (ex) {
                    switch (tagName) {
                        case ConstInterface.MG_TAG_KBDMAP_URL:
                            Logger.Instance.WriteExceptionToLogWithMsg(NString.Format("Keyboard Mapping: '{0}'{1}{2}", cachedFileUrl, OSEnvironment.EolSeq, ex.Message));
                            break;
                        case ConstInterface.MG_TAG_ENV_PARAM_URL:
                            Logger.Instance.WriteExceptionToLogWithMsg(NString.Format("Env Params Table: '{0}'{1}{2}", cachedFileUrl, OSEnvironment.EolSeq, ex.Message));
                    }
                }
            }
            endContext = XMLdata.indexOf(XMLConstants.TAG_OPEN, endContext);
            if (endContext !== -1)
                parser.setCurrIndex(endContext);
        }
    }
    getEnvDet(compIdx) {
        let environmentDetails = this._environments.get_Item(compIdx);
        if (environmentDetails === null)
            Logger.Instance.WriteExceptionToLogWithMsg("in Environment.getEnvDet() there is no env");
        return environmentDetails;
    }
    GetDateMode(compIdx) {
        return this.getEnvDet(compIdx).DateMode;
    }
    GetThousands() {
        return this._thousandsSeparator;
    }
    GetDecimal() {
        return this._decimalSeparator;
    }
    setDecimalSeparator(value) {
        this._decimalSeparator = value;
    }
    setThousandsSeparator(value) {
        this._thousandsSeparator = value;
    }
    setDateSeparator(value) {
        this._dateSeparator = value;
    }
    setTimeSeparator(value) {
        this._timeSeparator = value;
    }
    GetDate() {
        return this._dateSeparator;
    }
    getIdleTime(compIdx) {
        return this.getEnvDet(compIdx).IdleTime;
    }
    getOwner() {
        return this._owner;
    }
    GetTime() {
        return this._timeSeparator;
    }
    GetCentury(compIdx) {
        return this.getEnvDet(compIdx).Century;
    }
    allowUpdateInQueryMode(compIdx) {
        return this.getEnvDet(compIdx).allowUpdateInQueryMode();
    }
    allowCreateInModifyMode(compIdx) {
        return this.getEnvDet(compIdx).allowCreateInModifyMode();
    }
    GetSignificantNumSize() {
        return this._significantNumSize;
    }
    setSignificantNumSize(newSize) {
        this._significantNumSize = newSize;
    }
    getCodePage() {
        return this._codePage;
    }
    GetEncoding() {
        let encoding;
        try {
            encoding = Encoding.GetEncoding(this._codePage);
        }
        catch (exception) {
            encoding = null;
        }
        return encoding;
    }
    getSystem() {
        return this._system;
    }
    GetDebugLevel() {
        return this._debugMode;
    }
    CanReplaceDecimalSeparator() {
        return !this.IgnoreReplaceDecimalSeparator && this._canReplaceDecimalSeparator;
    }
    getClientContextInactivityTimeout() {
        return this._clientContextInactivityTimeout;
    }
    getContextInactivityTimeout() {
        return this._contextInactivityTimeout;
    }
    getContextInactivityWarningTime() {
        return this._contextInactivityWarningTime;
    }
    getContextUnloadTimeout() {
        return this._contextUnloadTimeout;
    }
    GetTooltipTimeout() {
        return this._toolitipTimeout;
    }
    getLowHigh() {
        return this._lowHigh;
    }
    GetImeAutoOff() {
        return this._imeAutoOff;
    }
    GetLocalAs400Set() {
        return this._localAs400Set;
    }
    getLocalExtraGengo() {
        return this._localExtraGengo;
    }
    GetLocalFlag(f) {
        return this._localFlags !== null && this._localFlags.indexOf(f) >= 0;
    }
    getSpecialImmediateHandlerEnd() {
        return this._specialImmediatehandlerEnd;
    }
    getSpecialAnsiExpression() {
        return this._specialAnsiExpression;
    }
    getSpecialShowStatusBarPanes() {
        return this._specialShowStatusBarPanes;
    }
    getTerminal() {
        return this._terminal;
    }
    getWebClientReinitSession() {
        return this._webclientReinitSession;
    }
    setWebClientReinitSession(session) {
        this._webclientReinitSession = session;
    }
    getSpecialRouteToRootProgOnContextRecreation() {
        return this._specialRouteToRootProgOnContextRecreation;
    }
    GetGUID() {
        return this._guid;
    }
    getProjDir(compIdx) {
        return this.getEnvDet(compIdx).ProjDir;
    }
    GetSpecialLogInternalExceptions() {
        return true;
    }
    setOwner(val) {
        this._owner = val;
    }
    setDateMode(compIdx, val) {
        this.getEnvDet(compIdx).DateMode = val;
    }
    setCentury(compIdx, val) {
        this.getEnvDet(compIdx).Century = val;
    }
    setIdleTime(compIdx, val) {
        this.getEnvDet(compIdx).IdleTime = val;
    }
    setAllowUpdateInQueryMode(compIdx, val) {
        this.getEnvDet(compIdx).UpdateInQueryMode = val;
    }
    setAllowCreateInModifyMode(compIdx, val) {
        this.getEnvDet(compIdx).CreateInModifyMode = val;
    }
    setContextInactivityTimeout(val) {
        this._contextInactivityTimeout = val;
    }
    setContextInactivityWarningTime(val) {
        this._contextInactivityWarningTime = val;
    }
    setContextUnloadTimeout(val) {
        this._contextUnloadTimeout = val;
    }
    setTerminal(val) {
        this._terminal = val;
    }
    GetDropUserFormats() {
        return this._dropUserFormats;
    }
    GetSpecialValNewPolicy() {
        return this._specialValNewPolicy;
    }
    GetSpecialCancelOnCreate() {
        return this._specialCancelOnCreate;
    }
}
Environment.Instance = new Environment();
export class EnvironmentDetails {
    set UpdateInQueryMode(value) {
        this._updateInQueryMode = value;
    }
    set CreateInModifyMode(value) {
        this._createInModifyMode = value;
    }
    constructor() {
        this._createInModifyMode = false;
        this._updateInQueryMode = false;
        this.CompIdx = 0;
        this.DateMode = '\0';
        this.Century = 0;
        this.IdleTime = 0;
        this.ProjDir = null;
        this.CompIdx = 0;
    }
    allowUpdateInQueryMode() {
        return this._updateInQueryMode;
    }
    allowCreateInModifyMode() {
        return this._createInModifyMode;
    }
}
//# sourceMappingURL=data:application/json;base64,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