import { Hashtable, NString, isUndefined } from "@magic-xpa/mscorelib";
import { JSON_Utils, Logger, Misc, MsgInterface, StrUtil, XMLConstants } from "@magic-xpa/utils";
import { ConstInterface } from "../ConstInterface";
import { CommandsProcessorManager } from "../CommandsProcessorManager";
const MLS_EOF_CHARS_TO_READ = 10;
const START_TAG = "<ErrorCodes>";
const END_TAG = "</ErrorCodes>";
export class LanguageData {
    constructor() {
        this._constMessages = null;
        this._constMessagesContent = null;
        this._constMessagesUrl = null;
        this._mlsContent = null;
        this._mlsFileUrl = null;
        this._mlsStrings = null;
    }
    async fillData(parser) {
        while (this.initInnerObjects(parser, parser.getNextTag())) {
        }
        await this.InitConstMessagesTable();
        await this.fillMlsMessagesTable();
    }
    initInnerObjects(parser, foundTagName) {
        if (foundTagName == null)
            return false;
        switch (foundTagName) {
            case ConstInterface.MG_TAG_LANGUAGE:
                parser.setCurrIndex(parser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, parser.getCurrIndex()) + 1);
                break;
            case ConstInterface.MG_TAG_CONSTMESSAGES: {
                let endUrlValue;
                parser.setCurrIndex(parser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, parser.getCurrIndex()) + 1);
                endUrlValue = parser.getXMLdata().indexOf(XMLConstants.TAG_OPEN, parser.getCurrIndex());
                this._constMessagesUrl = parser.getXMLsubstring(endUrlValue);
                parser.setCurrIndex(endUrlValue);
                break;
            }
            case ConstInterface.MG_TAG_CONSTMESSAGESCONTENT: {
                let endUrlValue;
                parser.setCurrIndex(parser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, parser.getCurrIndex()) + 1);
                endUrlValue = parser.getXMLdata().indexOf(ConstInterface.MG_TAG_CONSTMESSAGESCONTENT, parser.getCurrIndex()) - 2;
                this._constMessagesContent = parser.getXMLsubstring(endUrlValue);
                parser.setCurrIndex(endUrlValue);
                break;
            }
            case ConstInterface.MG_TAG_MLS_CONTENT: {
                let endUrlValue;
                parser.setCurrIndex(parser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, parser.getCurrIndex()) + 1);
                endUrlValue = parser.getXMLdata().indexOf(ConstInterface.MG_TAG_MLS_CONTENT, parser.getCurrIndex()) - 2;
                this._mlsContent = parser.getXMLsubstring(endUrlValue);
                parser.setCurrIndex(endUrlValue);
                if (!NString.IsNullOrEmpty(this._mlsContent)) {
                    if (this._mlsStrings != null) {
                        this._mlsStrings.Clear();
                        this._mlsStrings = null;
                    }
                }
                if (NString.IsNullOrEmpty(this._mlsContent) && this._mlsContent != null) {
                    this._mlsContent = null;
                    if (this._mlsStrings != null) {
                        this._mlsStrings.Clear();
                        this._mlsStrings = null;
                    }
                }
                break;
            }
            case ConstInterface.MG_TAG_MLS_FILE_URL: {
                let endUrlValue;
                parser.setCurrIndex(parser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, parser.getCurrIndex()) + 1);
                endUrlValue = parser.getXMLdata().indexOf(XMLConstants.TAG_OPEN, parser.getCurrIndex());
                this._mlsFileUrl = parser.getXMLsubstring(endUrlValue).trim();
                if (!NString.IsNullOrEmpty(this._mlsFileUrl)) {
                    if (this._mlsStrings != null) {
                        this._mlsStrings.Clear();
                        this._mlsStrings = null;
                    }
                }
                else if (NString.IsNullOrEmpty(this._mlsFileUrl) && this._mlsFileUrl != null) {
                    this._mlsFileUrl = null;
                    if (this._mlsStrings != null) {
                        this._mlsStrings.Clear();
                        this._mlsStrings = null;
                    }
                }
                parser.setCurrIndex(endUrlValue);
                break;
            }
            case ConstInterface.MG_TAG_CONSTMESSAGES_END:
            case ConstInterface.MG_TAG_MLS_FILE_URL_END:
            case ConstInterface.MG_TAG_CONSTMESSAGESCONTENT_END:
            case ConstInterface.MG_TAG_MLS_CONTENT_END:
                parser.setCurrIndex2EndOfTag();
                break;
            case ConstInterface.MG_TAG_LANGUAGE_END:
                parser.setCurrIndex2EndOfTag();
                return false;
            default:
                Logger.Instance.WriteExceptionToLogWithMsg("There is no such tag in LanguageData.initInnerObjects(): " + foundTagName);
                return false;
        }
        return true;
    }
    getConstMessage(msgId) {
        if (this._constMessages === null) {
            this._constMessages = new Hashtable();
            for (let i = 0; i <= MsgInterface.DefaultMessages.length - 1; i = i + 1) {
                this._constMessages.set_Item(MsgInterface.DefaultMessages[i].MsgID, MsgInterface.DefaultMessages[i].MsgString);
            }
        }
        return this._constMessages.get_Item(msgId);
    }
    async InitConstMessagesTable() {
        try {
            let msgsInCommentString = null;
            if (this._constMessagesUrl != null || this._constMessagesContent != null) {
                if (this._constMessagesUrl != null) {
                    try {
                        msgsInCommentString = await CommandsProcessorManager.GetContent(this._constMessagesUrl, true);
                    }
                    catch (err) {
                        Logger.Instance.WriteExceptionToLogWithMsg(NString.Format("Unknown message file: \"{0}\"", this._constMessagesUrl));
                    }
                }
                else {
                    msgsInCommentString = this._constMessagesContent;
                }
                if (msgsInCommentString != null) {
                    let startData = msgsInCommentString.indexOf(START_TAG);
                    let endData = msgsInCommentString.indexOf(END_TAG) + END_TAG.length;
                    let retrievedConstMessages = msgsInCommentString.substr(startData, (endData) - (startData));
                    this.fillConstMessagesTable(retrievedConstMessages);
                }
            }
            else {
                Logger.Instance.WriteDevToLog("'constMessagesUrl' and 'constMessagesContent' were not initialized.");
            }
        }
        catch (ex) {
            Logger.Instance.WriteExceptionToLog(ex);
        }
    }
    fillConstMessagesTable(constMessages) {
        try {
            if (constMessages !== null) {
                this._constMessages = new Hashtable();
                JSON_Utils.JSONFromXML(constMessages, this.FillConstMessagesFromJSON.bind(this));
            }
        }
        catch (ex) {
            Logger.Instance.WriteExceptionToLog(ex);
        }
    }
    FillConstMessagesFromJSON(error, result) {
        if (error != null) {
            throw error;
        }
        let errorMessages = result['ErrorCodes']['ErrorMessage'];
        for (let i = 0; i < errorMessages.length; i++) {
            let errorId = errorMessages[i]['$']['id'];
            let errorMessage = errorMessages[i]['_'];
            this._constMessages.set_Item(errorId, (isUndefined(errorMessage) ? "" : errorMessage.trim()));
        }
    }
    translate(fromString) {
        let toString = null;
        if (this._mlsStrings == null)
            return fromString;
        try {
            toString = this._mlsStrings.get_Item(fromString);
            if (toString == null)
                toString = fromString;
        }
        catch (e) {
            console.log(e);
        }
        return toString;
    }
    async fillMlsMessagesTable() {
        try {
            if (this._mlsContent != null) {
                if (this._mlsContent.length > 0) {
                    let linePairs = 0;
                    let srcLine = this._mlsContent + 3;
                    let transLine;
                    this._mlsStrings = new Hashtable();
                    let linesStr = this._mlsContent.substr(this._mlsContent.length - MLS_EOF_CHARS_TO_READ, 8);
                    linePairs = parseInt(linesStr.toString(), 16);
                    let tokens = StrUtil.tokenize(srcLine.toString(), "\n");
                    for (let pairNum = 0; pairNum < linePairs * 2; pairNum += 2) {
                        srcLine = tokens[pairNum].substr(0, tokens[pairNum].length - 1);
                        transLine = tokens[pairNum + 1].substr(0, tokens[pairNum + 1].length - 1);
                        if (this._mlsStrings.get_Item(srcLine.toString()) == null)
                            this._mlsStrings.set_Item(srcLine.toString(), transLine);
                    }
                }
            }
            else if (this._mlsFileUrl != null) {
                if (this._mlsFileUrl.startsWith("./"))
                    this._mlsFileUrl = NString.Replace(this._mlsFileUrl, './', './assets/cache/');
                let contentStr = await CommandsProcessorManager.GetContent(this._mlsFileUrl, true);
                let buffer = contentStr;
                if (buffer != null && buffer.length > 0) {
                    let contentLen = buffer.length;
                    this._mlsStrings = new Hashtable();
                    let linesStr = buffer.substr(contentLen - MLS_EOF_CHARS_TO_READ, 8);
                    let linePairs = parseInt(linesStr, 16);
                    let srcLine;
                    let transLine;
                    let tokens = StrUtil.tokenize(buffer.toString(), "\n");
                    for (let pairNum = 0; pairNum < linePairs * 2; pairNum += 2) {
                        srcLine = tokens[pairNum].substr(0, tokens[pairNum].length - 1);
                        transLine = tokens[pairNum + 1].substr(0, tokens[pairNum + 1].length - 1);
                        if (this._mlsStrings.get_Item(srcLine) == null)
                            this._mlsStrings.set_Item(srcLine, transLine);
                    }
                }
            }
        }
        catch (e) {
            Logger.Instance.WriteExceptionToLog(e);
            Misc.WriteStackTrace(e);
        }
    }
}
LanguageData.Instance = new LanguageData();
//# sourceMappingURL=data:application/json;base64,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