import { ApplicationException, Char, HashUtils, Int32, NNumber, NString, NumberStyles, RefParam, StringBuilder } from '@magic-xpa/mscorelib';
import { Base64, ForceExit, InternalInterface, Logger, Misc, StorageAttribute, StrUtil, XMLConstants, XmlParser } from '@magic-xpa/utils';
import { DisplayConvertor, KeyboardItem, Modifiers, RecordUtils, TaskDefinitionIdTableSaxHandler } from '@magic-xpa/gui';
import { ConstInterface } from '../ConstInterface';
import { LastFocusedManager } from '../rt/LastFocusedManager';
import { AccessHelper } from '../AccessHelper';
export class Event {
    constructor() {
        this.Exp = null;
        this._forceExit = ForceExit.None;
        this.kbdItm = null;
        this._paramAttrs = null;
        this._paramNulls = null;
        this._paramVals = null;
        this._parameters = 0;
        this._returnExp = 0;
        this._seconds = Int32.MinValue;
        this._timestamp = 0;
        this._type = Char.MinValue;
        this._userDefinedFuncName = null;
        this._userDefinedFuncNameHashCode = 0;
        this._userEvtDesc = '';
        this._userEvtIdx = Int32.MinValue;
        this.InternalEvent = 0;
        this.PublicName = null;
        this.UserEvt = null;
        this.UserEvtTaskTag = null;
        this.constructor_0();
    }
    constructor_0() {
        this.InternalEvent = Int32.MinValue;
        this.setTimestamp();
    }
    constructor_1(evt) {
        this._type = evt._type;
        this.kbdItm = evt.kbdItm;
        this.InternalEvent = evt.InternalEvent;
        this._seconds = evt._seconds;
        this.Exp = evt.Exp;
        this._forceExit = evt._forceExit;
        this.UserEvt = evt.UserEvt;
        this.UserEvtTaskTag = evt.UserEvtTaskTag;
        this._userEvtIdx = evt._userEvtIdx;
        this._timestamp = 0;
    }
    get ForceExit() {
        return this._forceExit;
    }
    fillData(xmlParser, taskRef) {
        let userValue;
        let endContext = xmlParser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, xmlParser.getCurrIndex());
        if (endContext !== -1 && endContext < xmlParser.getXMLdata().length) {
            let tag = xmlParser.getXMLsubstring(endContext);
            xmlParser.add2CurrIndex(tag.indexOf(ConstInterface.MG_TAG_EVENT) + ConstInterface.MG_TAG_EVENT.length);
            let tokensVector = XmlParser.getTokens(xmlParser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            let refUserValue = new RefParam(userValue);
            this.initElements(tokensVector, taskRef, refUserValue);
            userValue = refUserValue.value;
            xmlParser.setCurrIndex(endContext + XMLConstants.TAG_CLOSE.length);
            this.InitInnerObjects(xmlParser);
            if (userValue !== null)
                this.InitUserData(taskRef, userValue);
        }
        else
            Logger.Instance.WriteExceptionToLogWithMsg('in Event.FillData() out of string bounds');
    }
    InitInnerObjects(xmlParser) {
        let nextTag = xmlParser.getNextTag();
        switch (nextTag) {
            case XMLConstants.MG_TAG_TASKDEFINITIONID_ENTRY:
                let xmlBuffer = xmlParser.ReadToEndOfCurrentElement();
                this.InitTaskDefinitionId(xmlBuffer);
                let endContext = xmlParser.getXMLdata().indexOf(ConstInterface.MG_TAG_EVENT + XMLConstants.TAG_CLOSE, xmlParser.getCurrIndex());
                xmlParser.setCurrIndex(endContext + (ConstInterface.MG_TAG_EVENT + XMLConstants.TAG_CLOSE).length);
                break;
            default:
                break;
        }
    }
    initElements(tokensVector, taskRef, refUserValue) {
        let modifier = Modifiers.MODIFIER_NONE;
        let keyCode = -1;
        refUserValue.value = null;
        for (let j = 0; j < tokensVector.length; j += 2) {
            let attribute = (tokensVector.get_Item(j));
            let valueStr = (tokensVector.get_Item(j + 1));
            switch (attribute) {
                case ConstInterface.MG_ATTR_EVENTTYPE:
                    this._type = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_MODIFIER:
                    modifier = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_INTERNALEVENT:
                    this.InternalEvent = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_KEYCODE:
                    keyCode = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_SECONDS:
                    this._seconds = XmlParser.getInt(valueStr);
                    break;
                case XMLConstants.MG_ATTR_EXP:
                    this.Exp = taskRef.getExpById(XmlParser.getInt(valueStr));
                    break;
                case ConstInterface.MG_ATTR_FORCE_EXIT:
                    this._forceExit = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_DESC:
                    this._userEvtDesc = XmlParser.unescape(valueStr);
                    break;
                case ConstInterface.MG_ATTR_PUBLIC:
                    this.PublicName = XmlParser.unescape(valueStr);
                    break;
                case ConstInterface.MG_ATTR_PARAMS:
                    this._parameters = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_PAR_ATTRS:
                    this._paramAttrs = valueStr;
                    break;
                case XMLConstants.MG_ATTR_VALUE:
                    let paramStr = XmlParser.unescape(valueStr);
                    this._paramVals = new Array(this._parameters);
                    this.parseParamVal(paramStr);
                    break;
                case ConstInterface.MG_ATTR_NULLS:
                    this._paramNulls = valueStr;
                    break;
                case ConstInterface.MG_ATTR_USER:
                    refUserValue.value = valueStr;
                    break;
                case XMLConstants.MG_ATTR_NAME:
                    this.setUserDefinedFuncName(valueStr);
                    break;
                case ConstInterface.MG_TAG_USR_DEF_FUC_RET_EXP_ID:
                    this._returnExp = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_TAG_USR_DEF_FUC_RET_EXP_ATTR:
                    break;
                default:
                    Logger.Instance.WriteExceptionToLogWithMsg(NString.Format('There is no such tag in Event class. Insert case to Event.initElements for {0}', attribute));
                    break;
            }
        }
        if (this._type === ConstInterface.EVENT_TYPE_SYSTEM)
            this.kbdItm = new KeyboardItem(keyCode, modifier);
    }
    InitUserData(taskRef, userValue) {
        let comma = userValue.indexOf(',');
        if (comma > -1) {
            let ueTaskId = userValue.substr(0, comma);
            if (taskRef !== null)
                ueTaskId = taskRef.TaskService.GetEventTaskId(ueTaskId);
            let ueTask = AccessHelper.mgDataTable.GetTaskByID(ueTaskId);
            let ueIdx = XmlParser.getInt(userValue.substr(comma + 1));
            if (ueTask !== null) {
                this.UserEvt = ueTask.getUserEvent(ueIdx);
                if (this.UserEvt === null) {
                    throw new ApplicationException('in Event.fillData(): user event not found. task id = ' + ueTaskId +
                        ', user event index = ' + ueIdx);
                }
                this.UserEvtTaskTag = ueTaskId;
                this._userEvtIdx = ueIdx;
            }
            else {
                this.UserEvtTaskTag = ueTaskId;
                this._userEvtIdx = ueIdx;
                this.UserEvt = null;
            }
        }
    }
    InitTaskDefinitionId(xmlBuffer) {
        let handler = new TaskDefinitionIdTableSaxHandler(this.SetTaskDefinitionId);
        handler.parse(xmlBuffer);
    }
    SetTaskDefinitionId(taskDefinitionId) {
    }
    findUserEvent() {
        if (this._type === ConstInterface.EVENT_TYPE_USER && this.UserEvt === null) {
            let ueTask = AccessHelper.mgDataTable.GetTaskByID(this.UserEvtTaskTag);
            if (ueTask !== null)
                this.UserEvt = ueTask.getUserEvent(this._userEvtIdx);
            if (this.UserEvt === null)
                throw new ApplicationException(NString.Format('in Event.findUserEvent(): user event not found. task id = {0} doesn\'t exist', this.UserEvtTaskTag));
        }
    }
    equals(evt) {
        let bEqual = false;
        if (this === evt)
            bEqual = true;
        else if (this._type === evt._type) {
            switch (this._type) {
                case ConstInterface.EVENT_TYPE_SYSTEM:
                    bEqual = this.kbdItm.equals(evt.kbdItm);
                    break;
                case ConstInterface.EVENT_TYPE_INTERNAL:
                    bEqual = (this.InternalEvent === evt.InternalEvent);
                    break;
                case ConstInterface.EVENT_TYPE_TIMER:
                    bEqual = (this._seconds === evt._seconds);
                    break;
                case ConstInterface.EVENT_TYPE_EXPRESSION:
                case ConstInterface.EVENT_TYPE_PUBLIC:
                    bEqual = (this.Exp === evt.Exp);
                    break;
                case ConstInterface.EVENT_TYPE_USER:
                    bEqual = (this.UserEvt === evt.UserEvt);
                    break;
                case ConstInterface.EVENT_TYPE_USER_FUNC:
                    if (this._userDefinedFuncNameHashCode === evt.getUserDefinedFuncNameHashCode()) {
                        if (NString.Equals(evt.getUserDefinedFuncName(), this._userDefinedFuncName, true))
                            bEqual = true;
                    }
                    break;
                default:
                    bEqual = false;
                    break;
            }
        }
        if (this._type === ConstInterface.EVENT_TYPE_PUBLIC || evt._type === ConstInterface.EVENT_TYPE_PUBLIC) {
            let name1 = null;
            let name2 = null;
            if (this._type !== ConstInterface.EVENT_TYPE_USER && evt._type !== ConstInterface.EVENT_TYPE_USER)
                bEqual = false;
            else if (this._type === ConstInterface.EVENT_TYPE_PUBLIC) {
                name1 = this.PublicName;
                name2 = evt.getUserEvent().PublicName;
            }
            else {
                name1 = evt.PublicName;
                name2 = this.getUserEvent().PublicName;
            }
            if (name1 === null || name2 === null)
                bEqual = false;
            else
                bEqual = (name1 === name2);
        }
        else if (this._type === ConstInterface.EVENT_TYPE_INTERNAL && evt._type === ConstInterface.EVENT_TYPE_SYSTEM) {
            if (LastFocusedManager.Instance.getCurrTask() !== null && (AccessHelper.eventsManager.getMatchingAction(evt.getKbdItm())) === this.getInternalCode())
                bEqual = true;
        }
        else if (this._type === ConstInterface.EVENT_TYPE_USER && evt._type !== ConstInterface.EVENT_TYPE_USER) {
            this.findUserEvent();
            bEqual = this.UserEvt.equals(evt);
        }
        return bEqual;
    }
    async setPublicName() {
        if (this._type === ConstInterface.EVENT_TYPE_PUBLIC && this.Exp !== null) {
            this.PublicName = (await this.Exp.evaluateWithResLength(255)).mgVal;
            if (this.PublicName !== null)
                this.PublicName = StrUtil.rtrim(this.PublicName);
        }
    }
    setInternal(code) {
        this._type = ConstInterface.EVENT_TYPE_INTERNAL;
        this.InternalEvent = code;
        if (code === InternalInterface.MG_ACT_HIT || code === InternalInterface.MG_ACT_CTRL_HIT)
            LastFocusedManager.Instance.LastActionTime = Misc.getSystemMilliseconds();
    }
    setExpression(expRef) {
        this._type = ConstInterface.EVENT_TYPE_EXPRESSION;
        this.Exp = expRef;
    }
    getExpression() {
        return this.Exp;
    }
    getUsrEvntDesc() {
        return (this.UserEvt !== null) ? this.UserEvt._userEvtDesc : this._userEvtDesc;
    }
    setSystem(keyboardItem) {
        this._type = ConstInterface.EVENT_TYPE_SYSTEM;
        this.kbdItm = keyboardItem;
    }
    setKeyboardItem(keyboardItem) {
        this.kbdItm = keyboardItem;
    }
    getType() {
        return this._type;
    }
    setType(aType) {
        this._type = aType;
    }
    getInternalCode() {
        if (this._type === ConstInterface.EVENT_TYPE_INTERNAL)
            return this.InternalEvent;
        return Int32.MinValue;
    }
    getKbdItm() {
        if (this._type === ConstInterface.EVENT_TYPE_SYSTEM)
            return this.kbdItm;
        return null;
    }
    getKbdItmAlways() {
        return this.kbdItm;
    }
    async dataEventIsTrue() {
        if (this._type !== ConstInterface.EVENT_TYPE_EXPRESSION &&
            !(this._type === ConstInterface.EVENT_TYPE_USER && this.getUserEventType() === ConstInterface.EVENT_TYPE_EXPRESSION))
            return true;
        if (this._type !== ConstInterface.EVENT_TYPE_USER)
            return DisplayConvertor.toBoolean(await this.Exp.evaluateWithResultTypeAndLength(StorageAttribute.BOOLEAN, 0));
        return DisplayConvertor.toBoolean(await this.UserEvt.getExpression().evaluateWithResultTypeAndLength(StorageAttribute.BOOLEAN, 0));
    }
    getSeconds() {
        return this._seconds;
    }
    getBrkLevel(flowMonitor) {
        if (arguments.length === 0) {
            return this.getBrkLevel_0();
        }
        return this.getBrkLevel_1(flowMonitor);
    }
    getBrkLevel_0() {
        return this.getBrkLevel(false);
    }
    getBrkLevel_1(flowMonitor) {
        let level = '';
        switch (this._type) {
            case ConstInterface.EVENT_TYPE_INTERNAL:
                switch (this.InternalEvent) {
                    case InternalInterface.MG_ACT_TASK_PREFIX:
                        level = flowMonitor ? ConstInterface.BRK_LEVEL_TASK_PREFIX_FM : ConstInterface.BRK_LEVEL_TASK_PREFIX;
                        break;
                    case InternalInterface.MG_ACT_TASK_SUFFIX:
                        level = flowMonitor ? ConstInterface.BRK_LEVEL_TASK_SUFFIX_FM : ConstInterface.BRK_LEVEL_TASK_SUFFIX;
                        break;
                    case InternalInterface.MG_ACT_REC_PREFIX:
                        level = flowMonitor ? ConstInterface.BRK_LEVEL_REC_PREFIX_FM : ConstInterface.BRK_LEVEL_REC_PREFIX;
                        break;
                    case InternalInterface.MG_ACT_REC_SUFFIX:
                        level = flowMonitor ? ConstInterface.BRK_LEVEL_REC_SUFFIX_FM : ConstInterface.BRK_LEVEL_REC_SUFFIX;
                        break;
                    case InternalInterface.MG_ACT_CTRL_PREFIX:
                        level = flowMonitor ? ConstInterface.BRK_LEVEL_CTRL_PREFIX_FM : ConstInterface.BRK_LEVEL_CTRL_PREFIX + '_';
                        break;
                    case InternalInterface.MG_ACT_CTRL_SUFFIX:
                        level = flowMonitor ? ConstInterface.BRK_LEVEL_CTRL_SUFFIX_FM : ConstInterface.BRK_LEVEL_CTRL_SUFFIX + '_';
                        break;
                    case InternalInterface.MG_ACT_CTRL_VERIFICATION:
                        level = flowMonitor ? ConstInterface.BRK_LEVEL_CTRL_VERIFICATION_FM : ConstInterface.BRK_LEVEL_CTRL_VERIFICATION + '_';
                        break;
                    case InternalInterface.MG_ACT_VARIABLE:
                        level = flowMonitor ? ConstInterface.BRK_LEVEL_VARIABLE_FM : ConstInterface.BRK_LEVEL_VARIABLE + '_';
                        break;
                    default:
                        level = flowMonitor ? ConstInterface.BRK_LEVEL_HANDLER_INTERNAL_FM : ConstInterface.BRK_LEVEL_HANDLER_INTERNAL + '_';
                        break;
                }
                level += Event.getInternalEvtDescription(this.InternalEvent);
                break;
            case ConstInterface.EVENT_TYPE_SYSTEM:
                level = flowMonitor ? ConstInterface.BRK_LEVEL_HANDLER_SYSTEM_FM : ConstInterface.BRK_LEVEL_HANDLER_SYSTEM + '_';
                level += this.getKeyboardItemString();
                break;
            case ConstInterface.EVENT_TYPE_TIMER:
                level = flowMonitor ? ConstInterface.BRK_LEVEL_HANDLER_TIMER_FM + ' ' + this.seconds2String() : ConstInterface.BRK_LEVEL_HANDLER_TIMER + '_' + this.seconds2String();
                break;
            case ConstInterface.EVENT_TYPE_EXPRESSION:
                level = flowMonitor ? ConstInterface.BRK_LEVEL_HANDLER_EXPRESSION_FM : ConstInterface.BRK_LEVEL_HANDLER_EXPRESSION + '_' + this.Exp.getId();
                break;
            case ConstInterface.EVENT_TYPE_USER:
                level = (flowMonitor ? (ConstInterface.BRK_LEVEL_HANDLER_USER_FM + ': ') : (ConstInterface.BRK_LEVEL_HANDLER_USER + '_')) + (this.UserEvt != null ? this.UserEvt._userEvtDesc : this._userEvtDesc);
                break;
            case ConstInterface.EVENT_TYPE_USER_FUNC:
                level = (flowMonitor ? (ConstInterface.BRK_LEVEL_USER_FUNCTION_FM + ':') : (ConstInterface.BRK_LEVEL_USER_FUNCTION + '_')) + this._userDefinedFuncName;
                break;
        }
        return level;
    }
    static getInternalEvtDescription(internalEventCode) {
        let description;
        switch (internalEventCode) {
            case InternalInterface.MG_ACT_ABOUT:
                description = 'About Magic';
                break;
            case InternalInterface.MG_ACT_ACTION_LIST:
                description = 'Action List';
                break;
            case InternalInterface.MG_ACT_ADD_HYPERLINK:
                description = '';
                break;
            case InternalInterface.MG_ACT_ALIGN_LEFT:
                description = 'Left';
                break;
            case InternalInterface.MG_ACT_ALIGN_RIGHT:
                description = 'Right';
                break;
            case InternalInterface.MG_ACT_APPL_PROPERTIES:
                description = '';
                break;
            case InternalInterface.MG_ACT_APPLICATIONS:
                description = 'Applications';
                break;
            case InternalInterface.MG_ACT_APPLICATIONS_LIST:
                description = 'Applications List';
                break;
            case InternalInterface.MG_ACT_ATTACH_TO_TABLE:
                description = 'Attach to Table';
                break;
            case InternalInterface.MG_ACT_AUTHORIZE:
                description = 'Authorize';
                break;
            case InternalInterface.MG_ACT_EDT_BEGFLD:
                description = 'Begin Field';
                break;
            case InternalInterface.MG_ACT_EDT_BEGFORM:
                description = 'Begin Form';
                break;
            case InternalInterface.MG_ACT_EDT_BEGLINE:
                description = 'Begin Line';
                break;
            case InternalInterface.MG_ACT_EDT_BEGNXTLINE:
                description = 'Begin Next Line';
                break;
            case InternalInterface.MG_ACT_EDT_BEGPAGE:
                description = 'Begin Page';
                break;
            case InternalInterface.MG_ACT_TBL_BEGLINE:
                description = 'Begin Row';
                break;
            case InternalInterface.MG_ACT_TBL_BEGPAGE:
                description = 'Begin Screen';
                break;
            case InternalInterface.MG_ACT_TBL_BEGTBL:
                description = 'Begin table';
                break;
            case InternalInterface.MG_ACT_BOTTOM:
                description = 'Bottom';
                break;
            case InternalInterface.MG_ACT_BULLET:
                description = 'Bullet';
                break;
            case InternalInterface.MG_ACT_BUTTON:
                description = 'Button Press';
                break;
            case InternalInterface.MG_ACT_CANCEL:
                description = 'Cancel';
                break;
            case InternalInterface.MG_ACT_RT_QUIT:
                description = 'Quit';
                break;
            case InternalInterface.MG_ACT_CENTER:
                description = 'Center';
                break;
            case InternalInterface.MG_ACT_CHECK_SYNTAX:
                description = 'Check Syntax';
                break;
            case InternalInterface.MG_ACT_CHECK_TO_END:
                description = 'Check to End';
                break;
            case InternalInterface.MG_ACT_CLEAR_TEMPLATE:
                description = 'Clear Template';
                break;
            case InternalInterface.MG_ACT_CLEAR_VALUE:
                description = 'Clear Value';
                break;
            case InternalInterface.MG_ACT_CLOSE:
                description = 'Close';
                break;
            case InternalInterface.MG_ACT_ACT_CLOSE_APPL:
                description = 'Close Application';
                break;
            case InternalInterface.MG_ACT_COMMUNICATIONS:
                description = 'Communications';
                break;
            case InternalInterface.MG_ACT_COMPONENTS:
                description = 'Components';
                break;
            case InternalInterface.MG_ACT_CTRL_HIT:
                description = 'Control Hit';
                break;
            case InternalInterface.MG_ACT_CONTROL_NAME_LIST:
                description = 'Control Name List';
                break;
            case InternalInterface.MG_ACT_CLIP_COPY:
                description = 'Copy';
                break;
            case InternalInterface.MG_ACT_COPY_LAYOUT:
                description = 'Copy Layout';
                break;
            case InternalInterface.MG_ACT_COPY_SUBTREE:
                description = 'Copy Subtree';
                break;
            case InternalInterface.MG_ACT_CRELINE:
                description = 'Create Line';
                break;
            case InternalInterface.MG_ACT_CRELINE_ABOVE:
                description = 'Create Line Above';
                break;
            case InternalInterface.MG_ACT_CREATE_PARENT:
                description = 'Create Parent';
                break;
            case InternalInterface.MG_ACT_RTO_CREATE:
                description = 'Create Records';
                break;
            case InternalInterface.MG_ACT_CREATE_SUBTASK:
                description = 'Create Subtask';
                break;
            case InternalInterface.MG_ACT_CROSS_REFERENCE:
                description = 'Cross Reference';
                break;
            case InternalInterface.MG_ACT_CUT:
                description = 'Cut';
                break;
            case InternalInterface.MG_ACT_DB_TABLES:
                description = '';
                break;
            case InternalInterface.MG_ACT_DBMS:
                description = 'DBMS';
                break;
            case InternalInterface.MG_ACT_DDF_MAKE:
                description = 'DDF Make';
                break;
            case InternalInterface.MG_ACT_DATABASES:
                description = 'Databases';
                break;
            case InternalInterface.MG_ACT_DEFAULT_LAYOUT:
                description = 'Default Layout';
                break;
            case InternalInterface.MG_ACT_DEFINE_EXPRESSION:
                description = 'Define Expression';
                break;
            case InternalInterface.MG_ACT_EDT_DELCURCH:
                description = 'Current Char';
                break;
            case InternalInterface.MG_ACT_EDT_DELPRVCH:
                description = 'Previous Char';
                break;
            case InternalInterface.MG_ACT_DELETE_HYPERLINK:
                description = 'Delete Hyperlink';
                break;
            case InternalInterface.MG_ACT_DELLINE:
                description = 'Delete Line';
                break;
            case InternalInterface.MG_ACT_DELETE_SUBTREE:
                description = 'Delete Subtree';
                break;
            case InternalInterface.MG_ACT_RT_COPYFLD:
                description = 'Ditto';
                break;
            case InternalInterface.MG_ACT_DISPLAY_REFRESH:
                description = 'Display Refresh';
                break;
            case InternalInterface.MG_ACT_EDIT_MAIN_FORM:
                description = 'Edit Main Form';
                break;
            case InternalInterface.MG_ACT_EDT_ENDFLD:
                description = 'End Field';
                break;
            case InternalInterface.MG_ACT_EDT_ENDFORM:
                description = 'End Form';
                break;
            case InternalInterface.MG_ACT_EDT_ENDLINE:
                description = 'End Line';
                break;
            case InternalInterface.MG_ACT_EDT_ENDPAGE:
                description = 'End Page';
                break;
            case InternalInterface.MG_ACT_TBL_ENDLINE:
                description = 'End Row';
                break;
            case InternalInterface.MG_ACT_TBL_ENDPAGE:
                description = 'End Screen';
                break;
            case InternalInterface.MG_ACT_TBL_ENDTBL:
                description = 'End Table';
                break;
            case InternalInterface.MG_ACT_ENVIRONMENT:
                description = 'Environment';
                break;
            case InternalInterface.MG_ACT_EXECUTE_PROGRAM:
                description = 'Execute Program';
                break;
            case InternalInterface.MG_ACT_EXECUTE_REPORT:
                description = 'Execute Report';
                break;
            case InternalInterface.MG_ACT_EXIT:
                description = 'Exit';
                break;
            case InternalInterface.MG_ACT_EXIT_SYSTEM:
                description = 'Exit System';
                break;
            case InternalInterface.MG_ACT_EXPORT_IMPORT:
                description = 'Export/Import';
                break;
            case InternalInterface.MG_ACT_EXPRESSION_RULES:
                description = 'Expression rules';
                break;
            case InternalInterface.MG_ACT_EXTERNAL_EDITOR:
                description = 'External Editor';
                break;
            case InternalInterface.MG_ACT_EXT_EVENT:
                description = 'External Event';
                break;
            case InternalInterface.MG_ACT_FORMS:
                description = 'Forms';
                break;
            case InternalInterface.MG_ACT_FROM_VALUE:
                description = 'From Value';
                break;
            case InternalInterface.MG_ACT_FUNCTION_LIST:
                description = 'Function List';
                break;
            case InternalInterface.MG_ACT_GENERATE_FORM:
                description = 'Generate Form';
                break;
            case InternalInterface.MG_ACT_GENERATE_PROGRAM:
                description = 'Generate Program';
                break;
            case InternalInterface.MG_ACT_GO_TO_TOP:
                description = 'Go to Top';
                break;
            case InternalInterface.MG_ACT_H_CENTER_OF_FORM:
                description = '';
                break;
            case InternalInterface.MG_ACT_HTML_STYLES:
                description = 'HTML Styles';
                break;
            case InternalInterface.MG_ACT_HELP:
                description = 'Help';
                break;
            case InternalInterface.MG_ACT_HELP_SCREENS:
                description = 'Help Screens';
                break;
            case InternalInterface.MG_ACT_HORIZ_CENTER:
                description = 'Horiz. Center';
                break;
            case InternalInterface.MG_ACT_I_O_FILES:
                description = 'Tables';
                break;
            case InternalInterface.MG_ACT_INDENT:
                description = 'Indent';
                break;
            case InternalInterface.MG_ACT_INSERT_OBJECT:
                description = 'Insert Object';
                break;
            case InternalInterface.MG_ACT_JUMP_TO_ROW:
                description = 'Jump to Row';
                break;
            case InternalInterface.MG_ACT_KEYBOARD_LIST:
                description = 'Keyboard List';
                break;
            case InternalInterface.MG_ACT_KEYBOARD_MAPPING:
                description = 'Keyboard Mapping';
                break;
            case InternalInterface.MG_ACT_LANGUAGES:
                description = 'languages';
                break;
            case InternalInterface.MG_ACT_RTO_SEARCH:
                description = 'Locate Next';
                break;
            case InternalInterface.MG_ACT_RTO_LOCATE:
                description = 'Locate a Record';
                break;
            case InternalInterface.MG_ACT_LOGICALPNAMES:
                description = '';
                break;
            case InternalInterface.MG_ACT_LOGON:
                description = 'Logon';
                break;
            case InternalInterface.MG_ACT_USING_HELP:
                description = 'Help Topics';
                break;
            case InternalInterface.MG_ACT_EDT_MARKNXTCH:
                description = 'Mark Next Char';
                break;
            case InternalInterface.MG_ACT_MARK_NEXT_LINE:
                description = 'Mark Next Line MLE';
                break;
            case InternalInterface.MG_ACT_EDT_MARKPRVCH:
                description = 'Mark Previous Char';
                break;
            case InternalInterface.MG_ACT_EDT_MARKPRVLINE:
                description = 'Mark Previous Line MLE';
                break;
            case InternalInterface.MG_ACT_MARK_SUBTREE:
                description = '';
                break;
            case InternalInterface.MG_ACT_EDT_MARKTOBEG:
                description = 'Mark To Beginning';
                break;
            case InternalInterface.MG_ACT_EDT_MARKTOEND:
                description = 'Mark To End';
                break;
            case InternalInterface.MG_ACT_MAXIMUM_HEIGHT:
                description = 'Maximum Height';
                break;
            case InternalInterface.MG_ACT_MAXIMUM_WIDTH:
                description = 'Maximum Width';
                break;
            case InternalInterface.MG_ACT_MENU_BAR:
                description = 'Menu Bar';
                break;
            case InternalInterface.MG_ACT_MENUS:
                description = '';
                break;
            case InternalInterface.MG_ACT_MINIMUM_HEIGHT:
                description = 'Minimum Height';
                break;
            case InternalInterface.MG_ACT_MINIMUM_WIDTH:
                description = 'Minimum Width';
                break;
            case InternalInterface.MG_ACT_RTO_MODIFY:
                description = 'Modify Records';
                break;
            case InternalInterface.MG_ACT_MONITOR_DEBUGGER:
                description = 'Monitor/Debugger';
                break;
            case InternalInterface.MG_ACT_MOVE_ENTRY:
                description = 'Move Entry';
                break;
            case InternalInterface.MG_ACT_MOVE_SUBTREE:
                description = 'Move Subtree';
                break;
            case InternalInterface.MG_ACT_NULL_SETTINGS:
                description = 'NULL Settings';
                break;
            case InternalInterface.MG_ACT_EDT_NXTCHAR:
                description = 'Next Char';
                break;
            case InternalInterface.MG_ACT_EDT_NXTLINE:
                description = 'Next Line';
                break;
            case InternalInterface.MG_ACT_EDT_NXTPAGE:
                description = 'Next Page';
                break;
            case InternalInterface.MG_ACT_TBL_NXTLINE:
                description = 'Next Row';
                break;
            case InternalInterface.MG_ACT_TBL_NXTPAGE:
                description = 'Next Screen';
                break;
            case InternalInterface.MG_ACT_EDT_NXTTAB:
                description = 'Next Tab';
                break;
            case InternalInterface.MG_ACT_EDT_NXTWORD:
                description = 'Next Word';
                break;
            case InternalInterface.MG_ACT_NORMAL:
                description = 'Normal';
                break;
            case InternalInterface.MG_ACT_OK:
                description = 'OK';
                break;
            case InternalInterface.MG_ACT_OLE2:
                description = 'OLE2';
                break;
            case InternalInterface.MG_ACT_OPEN_APPLICATION:
                description = 'Open Application';
                break;
            case InternalInterface.MG_ACT_OVERWRITE_ENTRY:
                description = 'Overwrite Entry';
                break;
            case InternalInterface.MG_ACT_OVERWRITE_SUBTREE:
                description = 'Overwrite Subtree';
                break;
            case InternalInterface.MG_ACT_PPD:
                description = 'PPD';
                break;
            case InternalInterface.MG_ACT_PAGE_FOOTER:
                description = 'Page Footer';
                break;
            case InternalInterface.MG_ACT_PAGE_HEADER:
                description = 'Page Header';
                break;
            case InternalInterface.MG_ACT_CLIP_PASTE:
                description = 'Paste';
                break;
            case InternalInterface.MG_ACT_PASTE_LINK:
                description = 'Paste Link';
                break;
            case InternalInterface.MG_ACT_PASTE_SUBTREE:
                description = 'Paste Subtree';
                break;
            case InternalInterface.MG_ACT_EDT_PRVCHAR:
                description = 'Previous Char';
                break;
            case InternalInterface.MG_ACT_EDT_PRVLINE:
                description = 'Previous Line';
                break;
            case InternalInterface.MG_ACT_EDT_PRVPAGE:
                description = 'Previous Page';
                break;
            case InternalInterface.MG_ACT_TBL_PRVLINE:
                description = 'Previous Row';
                break;
            case InternalInterface.MG_ACT_TBL_PRVPAGE:
                description = 'Previous Screen';
                break;
            case InternalInterface.MG_ACT_EDT_PRVTAB:
                description = 'Previous Tab';
                break;
            case InternalInterface.MG_ACT_EDT_PRVWORD:
                description = 'Previous Word';
                break;
            case InternalInterface.MG_ACT_PRINT_ATTRIBUTES:
                description = 'Print Attributes';
                break;
            case InternalInterface.MG_ACT_PRINTER_SETUP:
                description = 'Printer Setup';
                break;
            case InternalInterface.MG_ACT_PRINTERS:
                description = 'Printers';
                break;
            case InternalInterface.MG_ACT_PROGRAMS:
                description = 'Programs';
                break;
            case InternalInterface.MG_ACT_PROPERTIES:
                description = 'Properties';
                break;
            case InternalInterface.MG_ACT_RTO_QUERY:
                description = 'Query Records';
                break;
            case InternalInterface.MG_ACT_RTO_RANGE:
                description = 'Range of Records';
                break;
            case InternalInterface.MG_ACT_RT_REFRESH_RECORD:
                description = 'Record Flush';
                break;
            case InternalInterface.MG_ACT_REDIRECT_FILES:
                description = 'Redirect Files';
                break;
            case InternalInterface.MG_ACT_REDRAW_LAYOUT:
                description = '';
                break;
            case InternalInterface.MG_ACT_REPEAT_ENTRY:
                description = 'Repeat Entry';
                break;
            case InternalInterface.MG_ACT_REPEAT_SUBTREE:
                description = 'Repeat Subtree';
                break;
            case InternalInterface.MG_ACT_REPORT_GENERATOR:
                description = '';
                break;
            case InternalInterface.MG_ACT_RESTORE_DEFAULTS:
                description = 'Restore Defaults';
                break;
            case InternalInterface.MG_ACT_RETURN_TO_TREE:
                description = '';
                break;
            case InternalInterface.MG_ACT_RIGHT:
                description = 'Right';
                break;
            case InternalInterface.MG_ACT_RIGHTS:
                description = '';
                break;
            case InternalInterface.MG_ACT_RIGHTS_LIST:
                description = 'Rights List';
                break;
            case InternalInterface.MG_ACT_SQL_ASSIST:
                description = 'SQL Assist';
                break;
            case InternalInterface.MG_ACT_SQL_COLUMNS:
                description = 'SQL Columns';
                break;
            case InternalInterface.MG_ACT_SQL_COMMAND:
                description = 'SQL Command';
                break;
            case InternalInterface.MG_ACT_SQL_FLIP_DESC:
                description = 'SQL Flip Desc.';
                break;
            case InternalInterface.MG_ACT_SQL_KEYWORDS:
                description = 'SQL Keywords';
                break;
            case InternalInterface.MG_ACT_SQL_OPERATORS:
                description = 'SQL Operators';
                break;
            case InternalInterface.MG_ACT_SQL_TABLES:
                description = 'SQL Tables';
                break;
            case InternalInterface.MG_ACT_SQL_WHERE_CLAUSE:
                description = '';
                break;
            case InternalInterface.MG_ACT_RT_REFRESH_SCREEN:
                description = 'Screen Refresh';
                break;
            case InternalInterface.MG_ACT_SECRET_NAMES:
                description = 'Secret Names';
                break;
            case InternalInterface.MG_ACT_SELECT:
                description = 'Select';
                break;
            case InternalInterface.MG_ACT_EDT_MARKALL:
                description = 'Select All';
                break;
            case InternalInterface.MG_ACT_SERVERS:
                description = '';
                break;
            case InternalInterface.MG_ACT_SERVICES:
                description = '';
                break;
            case InternalInterface.MG_ACT_RT_EDT_NULL:
                description = 'Set to NULL';
                break;
            case InternalInterface.MG_ACT_SHELL_TO_OS:
                description = 'Shell to OS';
                break;
            case InternalInterface.MG_ACT_SHOW_EXPRESSION:
                description = 'Show Expression';
                break;
            case InternalInterface.MG_ACT_SHOW_FULL_LAYOUT:
                description = '';
                break;
            case InternalInterface.MG_ACT_SHRINK_TREE:
                description = '';
                break;
            case InternalInterface.MG_ACT_SORT:
                description = 'Sort';
                break;
            case InternalInterface.MG_ACT_SORT_RECORDS:
                description = 'Sort Records';
                break;
            case InternalInterface.MG_ACT_TABLE_LOCATE:
                description = 'Table Locate';
                break;
            case InternalInterface.MG_ACT_TABLE_LOCATE_NEXT:
                description = 'Table Locate Next';
                break;
            case InternalInterface.MG_ACT_TABLES:
                description = '';
                break;
            case InternalInterface.MG_ACT_TASK_CONTROL:
                description = '';
                break;
            case InternalInterface.MG_ACT_TASK_EVENTS:
                description = '';
                break;
            case InternalInterface.MG_ACT_TO_VALUE:
                description = 'To Value';
                break;
            case InternalInterface.MG_ACT_TOOLKIT_RUNTIME:
                description = 'Toolkit/Runtime';
                break;
            case InternalInterface.MG_ACT_TOP:
                description = 'Top';
                break;
            case InternalInterface.MG_ACT_TYPES:
                description = '';
                break;
            case InternalInterface.MG_ACT_EDT_UNDO:
                description = 'Undo Editing';
                break;
            case InternalInterface.MG_ACT_UNINDENT:
                description = 'Unindent';
                break;
            case InternalInterface.MG_ACT_UPDATE_LINK:
                description = '';
                break;
            case InternalInterface.MG_ACT_USER_ACTION_1:
            case InternalInterface.MG_ACT_USER_ACTION_2:
            case InternalInterface.MG_ACT_USER_ACTION_3:
            case InternalInterface.MG_ACT_USER_ACTION_4:
            case InternalInterface.MG_ACT_USER_ACTION_5:
            case InternalInterface.MG_ACT_USER_ACTION_6:
            case InternalInterface.MG_ACT_USER_ACTION_7:
            case InternalInterface.MG_ACT_USER_ACTION_8:
            case InternalInterface.MG_ACT_USER_ACTION_9:
            case InternalInterface.MG_ACT_USER_ACTION_10:
            case InternalInterface.MG_ACT_USER_ACTION_11:
            case InternalInterface.MG_ACT_USER_ACTION_12:
            case InternalInterface.MG_ACT_USER_ACTION_13:
            case InternalInterface.MG_ACT_USER_ACTION_14:
            case InternalInterface.MG_ACT_USER_ACTION_15:
            case InternalInterface.MG_ACT_USER_ACTION_16:
            case InternalInterface.MG_ACT_USER_ACTION_17:
            case InternalInterface.MG_ACT_USER_ACTION_18:
            case InternalInterface.MG_ACT_USER_ACTION_19:
            case InternalInterface.MG_ACT_USER_ACTION_20:
                description = 'User Action ' + (internalEventCode - InternalInterface.MG_ACT_USER_ACTION_1 + 1);
                break;
            case InternalInterface.MG_ACT_USER_GROUPS:
                description = '';
                break;
            case InternalInterface.MG_ACT_USER_IDS:
                description = 'User IDs';
                break;
            case InternalInterface.MG_ACT_V_CENTER_OF_FORM:
                description = 'Center of Form';
                break;
            case InternalInterface.MG_ACT_VERTICAL_CENTER:
                description = 'Vertical Center';
                break;
            case InternalInterface.MG_ACT_VIEW_BY_KEY:
                description = 'View by Key';
                break;
            case InternalInterface.MG_ACT_RT_REFRESH_VIEW:
                description = 'View Refresh';
                break;
            case InternalInterface.MG_ACT_VIRTUAL_VARIABLES:
                description = 'Variables';
                break;
            case InternalInterface.MG_ACT_VISUAL_CONNECTION:
                description = 'Visual Connection';
                break;
            case InternalInterface.MG_ACT_HIT:
                description = 'Window Hit';
                break;
            case InternalInterface.MG_ACT_WINMOVE:
                description = 'Window Reposition';
                break;
            case InternalInterface.MG_ACT_WINSIZE:
                description = 'Window Resize';
                break;
            case InternalInterface.MG_ACT_ZOOM:
                description = 'Zoom';
                break;
            case InternalInterface.MG_ACT_LOCKING_DETAILS:
                description = 'Locking Details';
                break;
            case InternalInterface.MG_ACT_LOCKING_ABORT:
                description = 'Locking Abort';
                break;
            case InternalInterface.MG_ACT_LOCKING_RETRY:
                description = 'Locking Retry';
                break;
            case InternalInterface.MG_ACT_CHECK_OBJECT_LIST:
                description = 'Check Object List';
                break;
            case InternalInterface.MG_ACT_USERS_LIST:
                description = 'Users List';
                break;
            case InternalInterface.MG_ACT_MODELS:
                description = 'Models';
                break;
            case InternalInterface.MG_ACT_WEB_CLICK:
                description = 'Click';
                break;
            case InternalInterface.MG_ACT_WEB_ON_DBLICK:
                description = 'DblClick';
                break;
            case InternalInterface.MG_ACT_BROWSER_ESC:
                description = 'Esc';
                break;
            case InternalInterface.MG_ACT_ROLLBACK:
                description = 'Rollback';
                break;
            case InternalInterface.MG_ACT_EMPTY_DATAVIEW:
                description = 'Empty Dataview';
                break;
            case InternalInterface.MG_ACT_CTRL_MODIFY:
                description = 'Control Modify';
                break;
            case InternalInterface.MG_ACT_PRINT_DATA:
                description = 'Print Data';
                break;
            case InternalInterface.MG_ACT_POST_REFRESH_BY_PARENT:
                description = 'Post Refresh by Parent';
                break;
            case InternalInterface.MG_ACT_SWITCH_TO_OFFLINE:
                description = 'Switch To Offline';
                break;
            case InternalInterface.MG_ACT_UNAVAILABLE_SERVER:
                description = 'Unavailable Server';
                break;
            case InternalInterface.MG_ACT_CONTEXT_MENU:
                description = 'Context Menu';
                break;
            case InternalInterface.MG_ACT_ENTER_ROW_EDITING:
                description = 'Enter Row Editing';
                break;
            case InternalInterface.MG_ACT_NO_PROG_EXE_RIGHTS:
                description = 'No Program Execution Rights';
                break;
            case InternalInterface.MG_ACT_WEBCLIENT_ROUTE:
                description = 'Route';
                break;
            case InternalInterface.MG_ACT_COL_SORT:
                description = 'Column Sort';
                break;
            case InternalInterface.MG_ACT_ROW_DATA_CURR_PAGE:
                description = 'Get Rows';
                break;
            case InternalInterface.MG_ACT_SUBFORM_CLOSE:
                description = 'Subform Close';
                break;
            case InternalInterface.MG_ACT_SUBFORM_REFRESH:
                description = 'Subform Refresh';
                break;
            case InternalInterface.MG_ACT_SUBFORM_OPEN:
                description = 'Subform Open';
                break;
            case InternalInterface.MG_ACT_COMPUTE:
                description = 'Compute';
                break;
            case InternalInterface.MG_ACT_CACHE_NEXT:
                description = 'Get Next Data';
                break;
            case InternalInterface.MG_ACT_CACHE_PREV:
                description = 'Get Previous Data';
                break;
            case InternalInterface.MG_ACT_FETCH_DATA_CONTROL_VALUES:
                description = 'Fetch data control values';
                break;
            case InternalInterface.MG_ACT_INDEX_CHANGE:
                description = 'Index Change';
                break;
            case InternalInterface.MG_ACT_DUMP_ENVIRONMENT:
                description = 'Dump Environment';
                break;
            default:
                description = '';
                break;
        }
        return description;
    }
    getUserEventType() {
        if (this.UserEvt === null)
            return ConstInterface.EVENT_TYPE_NOTINITED;
        return this.UserEvt.getType();
    }
    getSecondsOfUserEvent() {
        return this.UserEvt.getSeconds();
    }
    getUserEvent() {
        return this.UserEvt;
    }
    getKeyboardItemString() {
        let kbi = this.getKbdItm();
        if (kbi === null)
            return '';
        return kbi.ToString();
    }
    seconds2String() {
        let time = new Array(3);
        let buffer = '';
        time[0] = Math.floor(this._seconds / 3600);
        time[1] = Math.floor((this._seconds - time[0] * 3600) / 60);
        time[2] = this._seconds % 60;
        for (let i = 0; i < time.length; i++) {
            if (time[i] === 0)
                buffer += '00';
            else if (time[i] < 10)
                buffer += ('0' + time[i]);
            else
                buffer += time[i];
            if (i < time.length - 1)
                buffer += ':';
        }
        return buffer;
    }
    parseParamVal(valueStr) {
        let startOfs = 0, endOfs = 0, nLen, j;
        let sLen, sValue = null;
        for (j = 0; j < this._parameters; j++) {
            switch (this._paramAttrs[j]) {
                case StorageAttribute.SKIP:
                    sValue = '';
                    break;
                case StorageAttribute.ALPHA:
                case StorageAttribute.BLOB_VECTOR:
                case StorageAttribute.BLOB:
                case StorageAttribute.MEMO:
                case StorageAttribute.UNICODE:
                    endOfs = startOfs + 4;
                    sLen = valueStr.substr(startOfs, (endOfs) - (startOfs));
                    nLen = NNumber.Parse(sLen, NumberStyles.HexNumber);
                    startOfs = endOfs;
                    if (AccessHelper.environment.GetDebugLevel() > 1 || this._paramAttrs[j] === StorageAttribute.ALPHA || this._paramAttrs[j] === StorageAttribute.UNICODE || this._paramAttrs[j] === StorageAttribute.MEMO) {
                        if ((nLen & 0x8000) > 0) {
                            nLen = (nLen & 0x7FFF);
                            nLen *= 2;
                            let res = new StringBuilder();
                            endOfs += RecordUtils.getSpannedField(valueStr, nLen, startOfs, this._paramAttrs[j], res, true);
                            valueStr = res.ToString();
                        }
                        else {
                            endOfs = startOfs + nLen;
                            sValue = valueStr.substr(startOfs, (endOfs) - (startOfs));
                        }
                    }
                    else {
                        if ((nLen & 0x8000) > 0) {
                            nLen = (nLen & 0x7FFF);
                            nLen = Math.floor((nLen + 2) / 3) * 4;
                            let res = new StringBuilder();
                            endOfs += RecordUtils.getSpannedField(valueStr, nLen, startOfs, this._paramAttrs[j], res, false);
                            sValue = res.ToString();
                        }
                        else {
                            endOfs = startOfs + Math.floor((nLen + 2) / 3) * 4;
                            sValue = Base64.decode(valueStr.substr(startOfs, (endOfs) - (startOfs)));
                        }
                    }
                    break;
                case StorageAttribute.NUMERIC:
                case StorageAttribute.DATE:
                case StorageAttribute.TIME:
                    if (AccessHelper.environment.GetDebugLevel() > 1) {
                        endOfs = startOfs + AccessHelper.environment.GetSignificantNumSize() * 2;
                        sValue = valueStr.substr(startOfs, (endOfs) - (startOfs));
                    }
                    else {
                        endOfs = startOfs + (Math.floor((AccessHelper.environment.GetSignificantNumSize() + 2) / 3) * 4);
                        sValue = Base64.decodeToHex(valueStr.substr(startOfs, (endOfs) - (startOfs)));
                    }
                    break;
                case StorageAttribute.BOOLEAN:
                    endOfs = startOfs + 1;
                    sValue = valueStr.substr(startOfs, (endOfs) - (startOfs));
                    break;
            }
            this._paramVals[j] = sValue;
            startOfs = endOfs;
        }
        return endOfs;
    }
    getParamNum() {
        return this._parameters;
    }
    isControlHeader() {
        return this.getType() == ConstInterface.EVENT_TYPE_INTERNAL &&
            (this.getInternalCode() == InternalInterface.MG_ACT_CTRL_PREFIX ||
                this.getInternalCode() == InternalInterface.MG_ACT_CTRL_SUFFIX ||
                this.getInternalCode() == InternalInterface.MG_ACT_CTRL_VERIFICATION);
    }
    isVariableHeader() {
        return this.getType() == ConstInterface.EVENT_TYPE_INTERNAL &&
            this.getInternalCode() == InternalInterface.MG_ACT_VARIABLE;
    }
    AppendDescription(buffer) {
        switch (this._type) {
            case ConstInterface.EVENT_TYPE_INTERNAL:
                buffer.Append(Event.getInternalEvtDescription(this.InternalEvent));
                break;
            case ConstInterface.EVENT_TYPE_SYSTEM:
                buffer.Append(this.getKeyboardItemString());
                break;
            case ConstInterface.EVENT_TYPE_USER:
                buffer.Append(this.getUsrEvntDesc());
                break;
            case ConstInterface.EVENT_TYPE_USER_FUNC:
                buffer.Append(this._userDefinedFuncName);
                break;
            case ConstInterface.EVENT_TYPE_TIMER:
                buffer.Append(this.seconds2String());
                break;
            case ConstInterface.EVENT_TYPE_EXPRESSION:
                buffer.Append('ID ' + this.Exp.getId().toString());
                break;
            default:
                buffer.Append('unknown event type ');
                break;
        }
    }
    isNewInternalEvent() {
        return this.InternalEvent >= 1001 && this.InternalEvent <= 1008;
    }
    getNewInternalEventDescription() {
        let eventDesc = NString.Empty;
        switch (this.InternalEvent) {
            case InternalInterface.MG_ACT_TASK_PREFIX:
                eventDesc = ConstInterface.BRK_LEVEL_TASK_PREFIX_FM;
                break;
            case InternalInterface.MG_ACT_TASK_SUFFIX:
                eventDesc = ConstInterface.BRK_LEVEL_TASK_SUFFIX_FM;
                break;
            case InternalInterface.MG_ACT_REC_PREFIX:
                eventDesc = ConstInterface.BRK_LEVEL_REC_PREFIX_FM;
                break;
            case InternalInterface.MG_ACT_REC_SUFFIX:
                eventDesc = ConstInterface.BRK_LEVEL_REC_SUFFIX_FM;
                break;
            case InternalInterface.MG_ACT_CTRL_PREFIX:
                eventDesc = ConstInterface.BRK_LEVEL_CTRL_PREFIX_FM;
                break;
            case InternalInterface.MG_ACT_CTRL_SUFFIX:
                eventDesc = ConstInterface.BRK_LEVEL_CTRL_SUFFIX_FM;
                break;
            case InternalInterface.MG_ACT_CTRL_VERIFICATION:
                eventDesc = ConstInterface.BRK_LEVEL_CTRL_VERIFICATION_FM;
                break;
            case InternalInterface.MG_ACT_VARIABLE:
                eventDesc = ConstInterface.BRK_LEVEL_VARIABLE_FM;
                break;
        }
        return eventDesc;
    }
    getEventType() {
        return this.isNewInternalEvent()
            ? NString.TrimEnd(this.getNewInternalEventDescription())
            : this.getEventTypeString();
    }
    getEventTypeString() {
        let eventType;
        switch (this.getType()) {
            case ConstInterface.EVENT_TYPE_INTERNAL:
                eventType = 'Internal Event:';
                break;
            case ConstInterface.EVENT_TYPE_SYSTEM:
                eventType = 'System Event:';
                break;
            case ConstInterface.EVENT_TYPE_EXPRESSION:
                eventType = 'Expression Event:';
                break;
            case ConstInterface.EVENT_TYPE_TIMER:
                eventType = 'Timer Event:';
                break;
            case ConstInterface.EVENT_TYPE_USER:
                eventType = 'User Event:';
                break;
            default:
                eventType = '';
                break;
        }
        return eventType;
    }
    setUserDefinedFuncName(name) {
        this._userDefinedFuncName = name;
        let functionNameUpper = this._userDefinedFuncName.toUpperCase();
        this._userDefinedFuncNameHashCode = HashUtils.GetHashCode(functionNameUpper);
    }
    getUserDefinedFuncName() {
        return this._userDefinedFuncName;
    }
    getUserDefinedFuncNameHashCode() {
        return this._userDefinedFuncNameHashCode;
    }
    getUserDefinedFuncRetExp() {
        return this._returnExp;
    }
    setTimestamp() {
        this._timestamp = Event._lastTimestamp++;
    }
    getTimestamp() {
        return this._timestamp;
    }
    toString() {
        return NString.Format('{{Event: {0}}}', this.getBrkLevel());
    }
}
Event._lastTimestamp = 0;
//# sourceMappingURL=data:application/json;base64,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