import { InternalInterface, Logger } from "@magic-xpa/utils";
import { ConstInterface } from "../ConstInterface";
import { AccessHelper } from "../AccessHelper";
import { isNullOrUndefined } from "@magic-xpa/mscorelib";
import { FlowMonitorQueue } from "../util/FlowMonitorQueue";
export class EventHandlerPosition {
    constructor() {
        this._handlerIdx = 0;
        this._handlersTab = null;
        this._orgPrevTask = null;
        this._orgTask = null;
        this._phase = 0;
        this._prevTask = null;
        this._rtEvt = null;
        this._task = null;
    }
    init(rtEvent) {
        this._rtEvt = rtEvent;
        this._task = this._rtEvt.getTask();
        if (this._task.isMainProg()) {
            this._prevTask = this._rtEvt.getMainPrgCreator();
            if (this._prevTask !== null && this._prevTask.isMainProg())
                this._prevTask = null;
        }
        if (rtEvent.getType() === ConstInterface.EVENT_TYPE_USER_FUNC)
            this._phase = EventHandlerPosition.PHASE_CONTROL_NON_SPECIFIC;
        else {
            if (rtEvent.getType() === ConstInterface.EVENT_TYPE_USER_FUNC)
                this._phase = EventHandlerPosition.PHASE_CONTROL_NON_SPECIFIC;
            else
                this._phase = EventHandlerPosition.PHASE_CONTROL_SPECIFIC;
        }
        this._orgTask = this._task;
        this._orgPrevTask = this._prevTask;
        this._handlersTab = this._task.getHandlersTab();
        if (this._handlersTab === null) {
            this.goUpTaskChain();
        }
        this._handlerIdx = -1;
    }
    getNext(evt) {
        if (!isNullOrUndefined(evt))
            return this.getNext_0(evt);
        else
            return this.getNext_1();
    }
    getNext_0(evt) {
        this._rtEvt = evt;
        return this.getNext();
    }
    getNext_1() {
        if (this._handlersTab === null)
            return null;
        if (this._rtEvt.getType() === ConstInterface.EVENT_TYPE_INTERNAL && this._rtEvt.getInternalCode() !== InternalInterface.MG_ACT_VARIABLE) {
            switch (this._rtEvt.getInternalCode()) {
                case InternalInterface.MG_ACT_TASK_PREFIX:
                case InternalInterface.MG_ACT_TASK_SUFFIX:
                case InternalInterface.MG_ACT_REC_PREFIX:
                case InternalInterface.MG_ACT_REC_SUFFIX:
                case InternalInterface.MG_ACT_CTRL_PREFIX:
                case InternalInterface.MG_ACT_CTRL_SUFFIX: {
                    if (this._rtEvt.getInternalCode() == InternalInterface.MG_ACT_REC_PREFIX) {
                        FlowMonitorQueue.Instance.addDataViewFlow(this._task);
                    }
                    if (this._handlerIdx === -1) {
                        for (this._handlerIdx = this._handlersTab.getSize() - 1; this._handlerIdx >= 0; this._handlerIdx--) {
                            let handler = this._handlersTab.getHandler(this._handlerIdx);
                            if ((handler.isNonSpecificHandlerOf(this._rtEvt)) || (handler.isSpecificHandlerOf(this._rtEvt))) {
                                return handler;
                            }
                        }
                    }
                    return null;
                }
                default:
                    break;
            }
        }
        while (this.setNextHandlerIdx()) {
            let handler = this._handlersTab.getHandler(this._handlerIdx);
            switch (this._phase) {
                case EventHandlerPosition.PHASE_CONTROL_SPECIFIC:
                    if (handler.isSpecificHandlerOf(this._rtEvt))
                        return handler;
                    continue;
                case EventHandlerPosition.PHASE_CONTROL_NON_SPECIFIC:
                    if (handler.isNonSpecificHandlerOf(this._rtEvt))
                        return handler;
                    continue;
                case EventHandlerPosition.PHASE_GLOBAL_SPECIFIC:
                    if (handler.isGlobalSpecificHandlerOf(this._rtEvt))
                        return handler;
                    continue;
                case EventHandlerPosition.PHASE_GLOBAL:
                    if (handler.isGlobalHandlerOf(this._rtEvt))
                        return handler;
                    continue;
                default:
                    Logger.Instance.WriteExceptionToLogWithMsg("in EventHandlerPosition.getNext() invalid phase: " + this._phase);
                    break;
            }
        }
        return null;
    }
    setNextHandlerIdx() {
        if (this._handlerIdx < 0)
            this._handlerIdx = this._handlersTab.getSize() - 1;
        else
            this._handlerIdx = this._handlerIdx - 1;
        if (this._handlerIdx < 0 || this._task.isAborting()) {
            if (this.goUpTaskChain())
                return this.setNextHandlerIdx();
            else
                return false;
        }
        return true;
    }
    goUpTaskChain() {
        let mGData = this._task.getMGData();
        let ctlIdx = this._task.getCtlIdx();
        switch (this._phase) {
            case EventHandlerPosition.PHASE_CONTROL_SPECIFIC:
            case EventHandlerPosition.PHASE_CONTROL_NON_SPECIFIC: {
                if (!this._task.isMainProg()) {
                    this.getParentOrCompMainPrg();
                    break;
                }
                else {
                    if ((this._rtEvt.getType() === ConstInterface.EVENT_TYPE_PUBLIC ||
                        this._rtEvt.getType() === ConstInterface.EVENT_TYPE_INTERNAL ||
                        this._rtEvt.getType() === ConstInterface.EVENT_TYPE_SYSTEM ||
                        this._rtEvt.getType() === ConstInterface.EVENT_TYPE_USER) && ctlIdx > 0) {
                        if (this._prevTask === null) {
                            this._task = mGData.getNextMainProg(ctlIdx);
                            if (this._task === null && ctlIdx > 0)
                                this._task = AccessHelper.mgDataTable.GetMainProgByCtlIdx(-1, AccessHelper.eventsManager.getCompMainPrgTab().getCtlIdx(0));
                        }
                        else {
                            this._task = this._prevTask.PathParentTask;
                            this._prevTask = null;
                        }
                        this._rtEvt.setMainPrgCreator(null);
                        break;
                    }
                    if (this._phase === EventHandlerPosition.PHASE_CONTROL_SPECIFIC) {
                        this._phase = EventHandlerPosition.PHASE_GLOBAL_SPECIFIC;
                    }
                    else {
                        this._phase = EventHandlerPosition.PHASE_GLOBAL;
                    }
                    this._task = AccessHelper.mgDataTable.GetMainProgByCtlIdx(-1, AccessHelper.eventsManager.getCompMainPrgTab().getCtlIdx(0));
                    this._rtEvt.setMainPrgCreator(this._task);
                    if (this._task === null)
                        return false;
                    break;
                }
            }
            case EventHandlerPosition.PHASE_GLOBAL_SPECIFIC:
            case EventHandlerPosition.PHASE_GLOBAL: {
                this._task = mGData.getNextMainProg(ctlIdx);
                if (this._task === null) {
                    if (this._phase === EventHandlerPosition.PHASE_GLOBAL)
                        return false;
                    this._phase = EventHandlerPosition.PHASE_CONTROL_NON_SPECIFIC;
                    this._task = this._orgTask;
                    this._prevTask = this._orgPrevTask;
                }
                break;
            }
            default:
                Logger.Instance.WriteExceptionToLogWithMsg("in EventHandlerPosition.goUpTaskChain() invalid phase: " + this._phase);
                break;
        }
        if (this._task === null)
            return false;
        this._handlersTab = this._task.getHandlersTab();
        if (this._handlersTab === null)
            return this.goUpTaskChain();
        this._handlerIdx = -1;
        return true;
    }
    getParentOrCompMainPrg() {
        let ctlIdx = this._task.getCtlIdx();
        this._prevTask = this._task;
        let pathParentTask = this._task.PathParentTask;
        if (pathParentTask === null) {
            this._task = null;
            return;
        }
        if (ctlIdx !== pathParentTask.getCtlIdx()) {
            this._rtEvt.setMainPrgCreator(this._task);
            this._task = AccessHelper.mgDataTable.GetMainProgByCtlIdx(-1, ctlIdx);
        }
        else {
            this._rtEvt.setMainPrgCreator(null);
            this._task = pathParentTask;
        }
    }
}
EventHandlerPosition.PHASE_CONTROL_SPECIFIC = 1;
EventHandlerPosition.PHASE_CONTROL_NON_SPECIFIC = 2;
EventHandlerPosition.PHASE_GLOBAL = 3;
EventHandlerPosition.PHASE_GLOBAL_SPECIFIC = 4;
//# sourceMappingURL=data:application/json;base64,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