import { Constants, ForceExit, InternalInterface, Logger, Logger_LogLevels, MgControlType, Misc, MsgInterface, Priority, RaiseAt, StorageAttribute, StrUtil, SubformType } from '@magic-xpa/utils';
import { ApplicationException, Int32, List, NString, RefParam, Stack, StringBuilder, Thread, isNullOrUndefined } from '@magic-xpa/mscorelib';
import { Commands, CommandType, DataModificationTypes, ExpVal, FocusManager, GuiConstants, GuiEnvironment, KeyboardItem, Manager, Modifiers, PropInterface, RuntimeContextBase, Styles, TaskDefinitionId } from '@magic-xpa/gui';
import { EventSubType } from '../enums';
import { RollbackEventCommand_RollbackType } from '../commands/ClientToServer/RollbackEventCommand';
import { MgControl } from '../gui/MgControl';
import { MgPriorityBlockingQueue } from '../util/MgPriorityBlockingQueue';
import { RunTimeEvent } from './RunTimeEvent';
import { FlowMonitorQueue } from '../util/FlowMonitorQueue';
import { GUIManager } from '../GUIManager';
import { Task } from '../tasks/Task';
import { Task_SubformExecModeEnum } from '../tasks/TaskBase';
import { ArgumentsList } from '../rt/ArgumentsList';
import { MGData } from '../tasks/MGData';
import { MGDataCollection } from '../tasks/MGDataCollection';
import { RecordOutOfDataViewException } from '../data/RecordOutOfDataViewException';
import { CommandsProcessorBase_SendingInstruction } from '../CommandsProcessorBase';
import { CommandFactory } from '../commands/ClientToServer/CommandFactory';
import { CommandsProcessorManager } from '../CommandsProcessorManager';
import { ClientRefreshCommand } from '../commands/ServerToClient/ClientRefreshCommand';
import { RemoteCommandsProcessor } from '../remote/RemoteCommandsProcessor';
import { ExecutionStack } from '../rt/ExecutionStack';
import { EventHandlerPosition } from './EventHandlerPosition';
import { ConstInterface } from '../ConstInterface';
import { FlowMonitorInterface } from '../FlowMonitorInterface';
import { DataViewCommandType } from '../commands/ClientToServer/DataviewCommand';
import { ServerError } from '../remote/ServerError';
import { Environment } from '../env/Environment';
import { LastFocusedManager } from '../rt/LastFocusedManager';
import { EventsAllowedType } from './IEventsManager';
import { LanguageData } from '../env/LanguageData';
import { Subject } from 'rxjs';
import { ClientManager } from '../ClientManager';
import { ClearEventsOnStopExecution } from '../AccessHelper';
var EventScope;
(function (EventScope) {
    EventScope["NONE"] = " ";
    EventScope["TRANS"] = "T";
})(EventScope || (EventScope = {}));
export class EventsManager {
    constructor() {
        this._eventsQueue = null;
        this._execStack = null;
        this._rtEvents = null;
        this._serverExecStack = null;
        this._allowEvents = EventsAllowedType.ALL;
        this._compMainPrgTab = null;
        this._currCtrl = null;
        this._currField = null;
        this._currKbdItem = null;
        this._endOfWork = false;
        this._eventScope = EventScope.NONE;
        this._forceExit = ForceExit.None;
        this._forceExitTask = null;
        this._ignoreUnknownAbort = false;
        this._initialized = false;
        this._isHandlingForceExit = false;
        this._isNonReversibleExit = false;
        this._isInRecordFlushEvent = false;
        this.lastSavedRouteEvent = null;
        this._stopExecution = false;
        this._processingTopMostEndTask = false;
        this._stopExecutionCtrl = null;
        this.guiPendingEvent = null;
        this.KBI_DOWN = null;
        this.KBI_UP = null;
        this.confirmationDialogMsg = 'The server context serving this application is about to time out. Do you want to continue your work?';
        this.confirmationDialogTitle = 'Session is about to be terminated!';
        this._lastFailedFocusedControl = null;
        this.isSpinnerShown = false;
        this.isInViewRefresh = false;
        this.SpinnerStopped = new Subject();
        this._rtEvents = new Stack();
        this._eventsQueue = new MgPriorityBlockingQueue();
        this._execStack = new Stack();
        this._execStack.push(new ExecutionStack());
        this._serverExecStack = new Stack();
        this._serverExecStack.push(null);
        this.KBI_DOWN = new KeyboardItem(GuiConstants.KEY_DOWN, Modifiers.MODIFIER_NONE);
        this.KBI_UP = new KeyboardItem(GuiConstants.KEY_UP, Modifiers.MODIFIER_NONE);
    }
    getPendingGuiEvent() {
        return this.guiPendingEvent;
    }
    setPendingGuiEvent(guiEvent) {
        this.guiPendingEvent = guiEvent;
    }
    addToTail(rtEvt) {
        if (!rtEvt.isIdleTimer()) {
            if (EventsManager.clientInactivityTimerON) {
                EventsManager.clientContextInactivityTimerAborted = true;
            }
        }
        else {
            if (this.getLastSavedRouteEvent() != null) {
                this.addToTail(this.getLastSavedRouteEvent());
                this.setLastSavedRouteEvent(null);
            }
        }
        this._eventsQueue.put(rtEvt);
    }
    AddKeyboardEvent(mgForm, mgControl, modifier, keyCode, start, end, text, isActChar, suggestedValue, code) {
        let rtEvt = (mgControl !== null) ? new RunTimeEvent(mgControl, true, false) : new RunTimeEvent(mgForm.getTask(), true);
        rtEvt.setInternal(code);
        let newKbItem = new KeyboardItem(keyCode, modifier);
        if (isActChar && (mgControl !== null && (mgControl.isTextControl())))
            newKbItem.setAction(InternalInterface.MG_ACT_CHAR);
        rtEvt.setKeyboardItem(newKbItem);
        if (code === InternalInterface.MG_ACT_CTRL_KEYDOWN) {
            rtEvt.setEditParms(start, end, text);
        }
        else
            rtEvt.setValue(suggestedValue);
        this.addToTail(rtEvt);
        return;
    }
    async checkParkability() {
        let current = LastFocusedManager.getLastFocusedControl();
        if (current !== null && (current.getForm().getTask()).getLevel() === Constants.TASK_LEVEL_CONTROL && !await current.IsParkable())
            await this.HandleNonParkableControls(current.getForm().getTask());
    }
    async HandleParkabilityOnRowChange(itask) {
        await this.HandleNonParkableControls(itask);
    }
    async HandleNonParkableControls(itask) {
        let task = itask;
        if (task.getMode() === Constants.TASK_MODE_CREATE && task.IsInteractive) {
            await task.enableModes();
            let oldMode = task.getOriginalTaskMode();
            await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
            if (!this.GetStopExecutionFlag()) {
                if (oldMode !== Constants.TASK_MODE_CREATE) {
                    task.setMode(oldMode);
                    await task.DataView.currRecCompute(true);
                    await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
                    await task.getForm().RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
                    task.setLastParkedCtrl(null);
                    await FocusManager.SetFocus(task, null, -1, true);
                    return;
                }
            }
        }
        if (task.getLevel() === Constants.TASK_LEVEL_CONTROL)
            await this.handleInternalEventWithMgControlBase(task.getLastParkedCtrl(), InternalInterface.MG_ACT_CTRL_SUFFIX);
        task.setLastParkedCtrl(null);
        MgControl.ReturnToCtrl = null;
        await FocusManager.SetFocus(task, null, -1, true);
    }
    async handleEvent(rtEvt, returnedFromServer) {
        let handler = null;
        let pos = new EventHandlerPosition();
        let task = rtEvt.getTask();
        let ctrl = task.getLastParkedCtrl();
        let forceExitDone = false;
        let endTaskError = false;
        let oldRtEvt = null;
        let oldForceExit = this.getForceExit();
        let oldForceExitTask = this.getForceExitTask();
        let oldIsHandlingForceExit = this._isHandlingForceExit;
        let restoreIsForceExit = false;
        let restoreCurrCtrl;
        let restoreCurrField = null;
        if ((rtEvt.getType() === ConstInterface.EVENT_TYPE_INTERNAL)) {
            if (rtEvt.InternalEvent < InternalInterface.MG_ACT_TOT_CNT && (!task.ActionManager.isEnabled(rtEvt.InternalEvent) || (!rtEvt.isImmediate() && this.NoEventsAllowed())))
                return;
        }
        else if (!rtEvt.isImmediate() && this.NoEventsAllowed())
            return;
        restoreCurrCtrl = this._currCtrl;
        this._currCtrl = ctrl;
        let propagate = rtEvt.getType() !== ConstInterface.EVENT_TYPE_INTERNAL || rtEvt.getInternalCode() < 1000;
        try {
            this.pushRtEvent(rtEvt);
            await this.CheckAndShowSpinner(true);
            let bRcBefore = await this.commonHandlerBefore(rtEvt);
            restoreCurrField = this._currField;
            this._currField = task.getCurrField();
            if (bRcBefore) {
                pos.init(rtEvt);
                handler = pos.getNext();
                if (handler !== null && handler.getEvent() !== null && handler.getEvent().UserEvt !== null)
                    this.setForceExit(handler.getEvent().UserEvt.ForceExit, task);
                while (handler !== null) {
                    let handlerContextTask = null;
                    let rtEvtChanged;
                    try {
                        let enabledCndCheckedAndTrue = false;
                        handlerContextTask = handler.getTask().GetContextTask();
                        if (rtEvt.getType() === ConstInterface.EVENT_TYPE_INTERNAL && rtEvt.getInternalCode() >= 1000 && rtEvt.getInternalCode() !== InternalInterface.MG_ACT_VARIABLE)
                            handler.getTask().SetContextTask(handler.getTask());
                        else
                            handler.getTask().SetContextTask(rtEvt.getTask());
                        if (handler.getEvent().getType() === ConstInterface.EVENT_TYPE_USER && await handler.isEnabled()) {
                            enabledCndCheckedAndTrue = true;
                            this._isHandlingForceExit = false;
                            restoreIsForceExit = true;
                            if (await this.handleForceExitBefore(handler.getEvent(), rtEvt)) {
                                forceExitDone = true;
                            }
                        }
                        if (rtEvt !== handler.getEvent()) {
                            oldRtEvt = rtEvt;
                            rtEvt = new RunTimeEvent(handler.getEvent(), rtEvt);
                            rtEvtChanged = true;
                            this.popRtEvent();
                            this.pushRtEvent(rtEvt);
                        }
                        else
                            rtEvtChanged = false;
                        if (rtEvt.getType() === ConstInterface.EVENT_TYPE_INTERNAL && rtEvt.getInternalCode() === InternalInterface.MG_ACT_REC_SUFFIX)
                            rtEvt.getTask().setInRecordSuffix(true);
                        this.createEventArguments(rtEvt);
                        let handlerRetVals = await handler.execute(rtEvt, returnedFromServer, enabledCndCheckedAndTrue);
                        if (rtEvt.getType() === ConstInterface.EVENT_TYPE_INTERNAL && rtEvt.getInternalCode() === InternalInterface.MG_ACT_REC_SUFFIX)
                            rtEvt.getTask().setInRecordSuffix(false);
                        propagate = handlerRetVals.Propagate;
                        if (!handlerRetVals.Enabled && rtEvtChanged) {
                            rtEvt = oldRtEvt;
                            oldRtEvt = null;
                            rtEvtChanged = false;
                            this.popRtEvent();
                            this.pushRtEvent(rtEvt);
                        }
                        if (rtEvt.getInternalCode() !== InternalInterface.MG_ACT_TASK_PREFIX && task.isStarted() && task.getExecEndTask()) {
                            endTaskError = await task.endTask(true, false, false);
                            if (endTaskError)
                                break;
                        }
                        if (this.GetStopExecutionFlag()) {
                            if (await this.handleStopExecution(rtEvt))
                                break;
                        }
                    }
                    finally {
                        handler.getTask().SetContextTask(handlerContextTask);
                    }
                    if (!propagate)
                        break;
                    if (rtEvtChanged)
                        handler = pos.getNext(rtEvt);
                    else
                        handler = pos.getNext();
                    if (handler !== null)
                        FlowMonitorQueue.Instance.addTaskEvent(rtEvt.getBrkLevel(true), FlowMonitorInterface.FLWMTR_PROPAGATE, task.GetTaskDetails());
                }
                if (propagate && !endTaskError)
                    await this.commonHandler(rtEvt);
            }
            if (!endTaskError) {
                await this.commonHandlerAfter(rtEvt, bRcBefore);
            }
            if (forceExitDone && this._isHandlingForceExit) {
                this._isHandlingForceExit = false;
                await this.handleForceExitAfter(rtEvt);
            }
            this.setForceExit(oldForceExit, oldForceExitTask);
            if (restoreIsForceExit)
                this._isHandlingForceExit = oldIsHandlingForceExit;
        }
        finally {
            this._currCtrl = restoreCurrCtrl;
            this._currField = restoreCurrField;
            this.popRtEvent();
        }
        GUIManager.Instance.execGuiCommandQueue();
    }
    async simulateSelection(ctrl, val, line, produceClick) {
        let mgControl = ctrl;
        let currTask = mgControl.getForm().getTask();
        let cancelWasRaised = currTask.cancelWasRaised();
        if (mgControl.Type === MgControlType.CTRL_TYPE_BUTTON) {
            let aRtEvt = new RunTimeEvent(mgControl);
            let raiseAt = (await (mgControl.getProp(PropInterface.PROP_TYPE_RAISE_AT)).getValueInt());
            if (aRtEvt.getType() === ConstInterface.EVENT_TYPE_INTERNAL && (aRtEvt.InternalEvent === InternalInterface.MG_ACT_CANCEL || aRtEvt.InternalEvent === InternalInterface.MG_ACT_RT_QUIT)) {
                if (raiseAt === RaiseAt.TaskInFocus)
                    currTask = LastFocusedManager.Instance.getLastFocusedTask();
                cancelWasRaised = currTask.cancelWasRaised();
                currTask.setCancelWasRaised(true);
            }
            if (raiseAt === RaiseAt.TaskInFocus)
                mgControl.setRtEvtTask(LastFocusedManager.Instance.getLastFocusedTask());
        }
        await this.handleFocus(mgControl, line, produceClick);
        currTask.setCancelWasRaised(cancelWasRaised);
        if (this._stopExecution)
            return;
        let controlIsParkable = await mgControl.IsParkable();
        if (mgControl.Type !== MgControlType.CTRL_TYPE_BUTTON || (mgControl.getField() === null || controlIsParkable)) {
            if (mgControl.Type === MgControlType.CTRL_TYPE_BUTTON && !controlIsParkable) {
                let subformTask = mgControl.getForm().getTask();
                if (subformTask.IsSubForm) {
                    currTask = LastFocusedManager.Instance.getLastFocusedTask();
                    if (subformTask !== currTask && subformTask.pathContains(currTask)) {
                        if (subformTask.getLevel() === Constants.TASK_LEVEL_TASK) {
                            await this.handleInternalEventWithTask(subformTask, InternalInterface.MG_ACT_REC_PREFIX);
                        }
                        await this.HandleNonParkableControls(subformTask);
                    }
                }
            }
            let rtEvt = new RunTimeEvent(mgControl);
            RuntimeContextBase.Instance.CurrentClickedCtrl = mgControl;
            rtEvt.setInternal(InternalInterface.MG_ACT_CTRL_HIT);
            await this.handleEvent(rtEvt, false);
            if (this._stopExecution)
                return;
        }
        if (mgControl.Type !== MgControlType.CTRL_TYPE_BUTTON)
            await this.handleSelection(mgControl, line, val);
        if (this._stopExecution)
            return;
        await this.handleMouseUp(mgControl, line, produceClick);
    }
    async handleInternalEventWithTask(itask, eventCode) {
        await this.handleInternalEventWithTaskAndSubformRefresh(itask, eventCode, false);
    }
    async handleInternalEventWithTaskAndSubformRefresh(itask, eventCode, subformRefresh) {
        let task = itask;
        if (eventCode === InternalInterface.MG_ACT_REC_PREFIX && task.IsSubForm && task.SubformExecMode !== Task_SubformExecModeEnum.FIRST_TIME) {
            task.SubformExecMode = (subformRefresh ? Task_SubformExecModeEnum.REFRESH : Task_SubformExecModeEnum.SET_FOCUS);
        }
        if (subformRefresh)
            await this.handleInternalEventWithTask(itask, InternalInterface.MG_ACT_POST_REFRESH_BY_PARENT);
        if (task.DataView.isEmptyDataview()) {
            if (eventCode === InternalInterface.MG_ACT_REC_PREFIX || eventCode === InternalInterface.MG_ACT_REC_SUFFIX) {
                if (eventCode === InternalInterface.MG_ACT_REC_PREFIX)
                    await task.emptyDataviewOpen(subformRefresh);
                else
                    task.emptyDataviewClose();
                if (task.getForm() !== null) {
                    if (eventCode === InternalInterface.MG_ACT_REC_PREFIX)
                        await task.getForm().RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
                    task.getForm().SelectRow();
                }
            }
            else
                await this.handleInternalEventWithTaskAndEventSubtype(task, eventCode, EventSubType.Normal);
        }
        else
            await this.handleInternalEventWithTaskAndEventSubtype(task, eventCode, EventSubType.Normal);
    }
    async handleInternalEventWithMgControlBase(ctrl, eventCode) {
        await this.handleInternalEventWithMgControl(ctrl, eventCode, EventSubType.Normal);
    }
    GetStopExecutionFlag() {
        return this._stopExecution;
    }
    getStopExecutionCtrl() {
        return this._stopExecutionCtrl;
    }
    Init() {
        if (this._initialized) {
            this._isHandlingForceExit = false;
            this._forceExit = ForceExit.None;
            this._forceExitTask = null;
            this._currField = null;
            this.setEndOfWork(true);
        }
        this._initialized = true;
    }
    addInternalEventWithItaskAndCode(itask, code) {
        this.addGuiTriggeredEventWithTaskAndCodeAndOnMultiMark(itask, code);
    }
    addInternalEventWithCtrlAndCode(ctrl, code) {
        let rtEvt = new RunTimeEvent(ctrl);
        rtEvt.setInternal(code);
        this.addToTail(rtEvt);
    }
    addInternalEventWithCtrlAndDisplayLineAndCode(ctrl, DisplayLine, code) {
        let rtEvt = new RunTimeEvent(ctrl, DisplayLine, false);
        rtEvt.setInternal(code);
        this.addToTail(rtEvt);
    }
    addInternalEventWithCtrlAndCodeAndPriority(ctrl, code, priority) {
        let rtEvt = new RunTimeEvent(ctrl);
        rtEvt.setInternal(code);
        rtEvt.setPriority(priority);
        this.addToTail(rtEvt);
    }
    addGuiTriggeredEventWithTaskAndCodeAndOnMultiMark(itask, code) {
        let task = itask;
        let rtEvt = new RunTimeEvent(task);
        rtEvt.setInternal(code);
        this.addToTail(rtEvt);
    }
    addGuiTriggeredEventWithCtrlAndCodeAndLineAndModifier(ctrl, code, line) {
        let rtEvt = new RunTimeEvent(ctrl, line, false);
        rtEvt.setInternal(code);
        this.addToTail(rtEvt);
    }
    addGuiTriggeredEventWithTaskAndCodeAndLine(task, code, line) {
        let tsk = null;
        if (task !== null)
            tsk = task;
        let rtEvt = new RunTimeEvent(tsk);
        rtEvt.setInternal(code);
        rtEvt.DisplayLine = line;
        this.addToTail(rtEvt);
    }
    AddColumnClickEvent(columnCtrl, direction, columnHeader) {
        let rtEvt = new RunTimeEvent(columnCtrl, direction, 0);
        rtEvt.setInternal(InternalInterface.MG_ACT_COL_CLICK);
        let argsList = new Array();
        argsList[0] = new ExpVal(StorageAttribute.UNICODE, false, columnHeader);
        let args = new ArgumentsList(argsList);
        rtEvt.setArgList(args);
        this.addToTail(rtEvt);
    }
    AddRouteEvent(task, code, routerpath, routerOutletName, routeParams) {
        let rtEvt = new RunTimeEvent(task, true);
        rtEvt.setInternal(code);
        let argsList = new Array(2);
        argsList[0] = new ExpVal(StorageAttribute.ALPHA, false, routerpath);
        if (routerOutletName === null)
            routerOutletName = '';
        argsList[1] = new ExpVal(StorageAttribute.ALPHA, false, routerOutletName);
        let args = new ArgumentsList(argsList);
        rtEvt.setArgList(args);
        rtEvt.setRouteParamList(routeParams);
        this.addToTail(rtEvt);
    }
    AddRouterClosedEvent(task) {
        let subFormControl = task.getForm().getSubFormCtrl();
        let rtEvt = new RunTimeEvent(subFormControl.getForm().getTask(), true);
        rtEvt.setInternal(InternalInterface.MG_ACT_SUBFORM_CLOSE);
        let argsList = new Array(1);
        argsList[0] = new ExpVal(StorageAttribute.ALPHA, false, subFormControl.getName());
        let args = new ArgumentsList(argsList);
        rtEvt.setArgList(args);
        EventsManager.Instance.addToTail(rtEvt);
    }
    AddLastSavedRouteEvent(srcTask, args) {
        let rtEvt = new RunTimeEvent(LastFocusedManager.Instance.getLastFocusedTask(), true);
        rtEvt.setInternal(InternalInterface.MG_ACT_WEBCLIENT_ROUTE);
        if (srcTask != null) {
            rtEvt.setTask(srcTask);
            if (args != null) {
                rtEvt.setArgList(args);
            }
            this.setLastSavedRouteEvent(rtEvt);
        }
    }
    AddExternalValueEvent(ctrl, value, refreshDisplay) {
        let rtEvt = new RunTimeEvent(ctrl, 0, true);
        rtEvt.setInternal(InternalInterface.MG_ACT_SET_EXTERNAL_VALUE);
        rtEvt.setValue(value);
        let argsList = new Array(1);
        argsList[0] = new ExpVal(StorageAttribute.BOOLEAN, false, refreshDisplay ? '1' : '0');
        let args = new ArgumentsList(argsList);
        rtEvt.setArgList(args);
        this.addToTail(rtEvt);
    }
    addGuiTriggeredEventWithCtrlAndCodeAndLine(ctrl, code, line) {
        let rtEvt = new RunTimeEvent(ctrl, line, true);
        rtEvt.setInternal(code);
        this.addToTail(rtEvt);
    }
    addGuiTriggeredEventWithCtrlAndLineAndCodeAndIsProductClick(ctrl, code, line, isProduceClick) {
        let rtEvt = new RunTimeEvent(ctrl, line, true);
        rtEvt.setInternal(code);
        rtEvt.setProduceClick(isProduceClick);
        this.addToTail(rtEvt);
    }
    addGuiTriggeredEventWithCtrlAndCodeAndList(ctrl, code, list) {
        let rtEvt = new RunTimeEvent(ctrl, list, true);
        rtEvt.setInternal(code);
        this.addToTail(rtEvt);
    }
    addGuiTriggeredEventWithTaskAndCode(task, code) {
        let rtEvt = new RunTimeEvent(task, true);
        rtEvt.setInternal(code);
        this.addToTail(rtEvt);
    }
    addGuiTriggeredEventTaskAndCodeAndRaisedBy(task, code) {
        this.addGuiTriggeredEventWithTaskAndCode(task, code);
    }
    getEvent() {
        return this._eventsQueue.poll();
    }
    getForceExit() {
        return this._forceExit;
    }
    setForceExit(forceExit, task) {
        this._forceExit = forceExit;
        this.setForceExitTask(task);
    }
    getForceExitTask() {
        return this._forceExitTask;
    }
    setForceExitTask(task) {
        this._forceExitTask = task;
    }
    isForceExitPreRecordUpdate(task) {
        return this._isHandlingForceExit && this.getForceExit() === ForceExit.PreRecordUpdate && task === this.getForceExitTask();
    }
    async ExecuteClientContextInactivityTimer() {
        let startTime = Misc.getSystemMilliseconds();
        let clientCtxInactivityTimeout = Environment.Instance.getClientContextInactivityTimeout() * 100;
        EventsManager.clientInactivityTimerON = true;
        while (EventsManager.clientInactivityTimerON) {
            await Thread.Sleep(1000);
            var timeLapsed = Misc.getSystemMilliseconds() - startTime;
            if (EventsManager.clientContextInactivityTimerAborted || timeLapsed >= clientCtxInactivityTimeout) {
                EventsManager.clientInactivityTimerON = false;
            }
        }
    }
    async CheckAndShowSpinner(show) {
        let sendCommand = false;
        if (this.getLastRtEvent() !== null && this.getLastRtEvent().getPriority() === Priority.LOWEST)
            show = false;
        if (show != this.isSpinnerShown) {
            if (show) {
                if (Misc.getSystemMilliseconds() - this.GetEventTime() > 1000) {
                    sendCommand = true;
                    EventsManager.clientBusyTimeStart = Misc.getSystemMilliseconds();
                }
            }
            else
                sendCommand = true;
            if (sendCommand) {
                Commands.addShowSpinner(show);
                Commands.beginInvoke();
                await Thread.Sleep(50);
                this.isSpinnerShown = show;
                if (!this.isSpinnerShown) {
                    let totalClientBusyTime = Misc.getSystemMilliseconds() - EventsManager.clientBusyTimeStart;
                    this.SpinnerStopped.next(totalClientBusyTime);
                    EventsManager.clientBusyTimeStart = 0;
                }
            }
        }
    }
    async EventsLoop(mgData) {
        let savedAllowEvents = this.getAllowEvents();
        this.setAllowEvents(EventsAllowedType.ALL);
        this.pushNewExecStacks();
        while (!mgData.IsAborting) {
            await this.CheckAndShowSpinner(false);
            await this._eventsQueue.waitForElement();
            await this.handleEvents(mgData, 0);
            if (!mgData.IsAborting)
                await this.checkParkability();
        }
        this.popNewExecStacks();
        this.setAllowEvents(savedAllowEvents);
    }
    async NonInteractiveEventsLoop(mgData, taskBase) {
        let task = taskBase;
        let savedAllowEvents = this.getAllowEvents();
        this.setNonInteractiveAllowEvents(await task.isAllowEvents());
        this.pushNewExecStacks();
        while (!mgData.IsAborting) {
            if (task.getMode() === Constants.TASK_MODE_DELETE)
                await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_CYCLE_NEXT_DELETE_REC);
            else
                await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_CYCLE_NEXT_REC);
            await Thread.Sleep(10);
            if (!mgData.IsAborting && await task.isAllowEvents())
                await this.handleEvents(mgData, 1);
        }
        this.popNewExecStacks();
        this.setAllowEvents(savedAllowEvents);
    }
    async handleEvents(baseMgDataOrHandlersTab, eventsToHandleCntOrRtEvt) {
        if ((baseMgDataOrHandlersTab === null || baseMgDataOrHandlersTab instanceof MGData) && (eventsToHandleCntOrRtEvt === null || eventsToHandleCntOrRtEvt.constructor === Number)) {
            await this.handleEvents_0(baseMgDataOrHandlersTab, eventsToHandleCntOrRtEvt);
            return;
        }
        await this.handleEvents_1(baseMgDataOrHandlersTab, eventsToHandleCntOrRtEvt);
    }
    async handleEvents_0(baseMgData, eventsToHandleCnt) {
        let task = null;
        let delta = 0;
        let eventsPolledCnt = 0;
        this.setStopExecution(false);
        while (!this._eventsQueue.isEmpty() && !this._stopExecution && (eventsToHandleCnt === 0 || eventsPolledCnt < eventsToHandleCnt)) {
            let rtEvt = this.getEvent();
            if (rtEvt !== null && rtEvt.isGuiTriggeredEvent()) {
                delta = Misc.getSystemMilliseconds();
                this._currKbdItem = null;
                let blockedByModalWindow = rtEvt.isBlockedByModalWindow(baseMgData);
                if (blockedByModalWindow && rtEvt.getInternalCode() !== InternalInterface.MG_ACT_TIMER) {
                    continue;
                }
                if (!rtEvt.isIdleTimer() && !blockedByModalWindow && rtEvt.getInternalCode() !== InternalInterface.MG_ACT_ENABLE_EVENTS && rtEvt.getInternalCode() !== InternalInterface.MG_ACT_DISABLE_EVENTS && rtEvt.getTask() !== null && !rtEvt.getTask().getMGData().IsAborting) {
                    MGDataCollection.Instance.currMgdID = rtEvt.getTask().getMGData().GetId();
                }
            }
            let currMgd = MGDataCollection.Instance.getCurrMGData();
            if (currMgd !== null) {
                while (currMgd !== null && currMgd.IsAborting) {
                    currMgd = currMgd.getParentMGdata();
                    if (currMgd !== null)
                        MGDataCollection.Instance.currMgdID = MGDataCollection.Instance.getMgDataIdx(currMgd);
                }
            }
            else {
                task = rtEvt.getTask();
                while (currMgd === null && task !== null && task.getParent() !== null) {
                    task = task.getParent();
                    currMgd = task.getMGData();
                    if (currMgd.IsAborting)
                        currMgd = null;
                    if (currMgd !== null)
                        MGDataCollection.Instance.currMgdID = MGDataCollection.Instance.getMgDataIdx(currMgd);
                }
            }
            if (currMgd === null)
                break;
            if (rtEvt.getTask() === null) {
                task = LastFocusedManager.Instance.getLastFocusedTask();
                while (task !== null && (task.isAborting() || task.InEndTask))
                    task = task.getParent();
                if (task !== null) {
                    rtEvt.setCtrl(task.getLastParkedCtrl());
                    rtEvt.setTask(task);
                }
            }
            else
                task = rtEvt.getTask();
            if (task !== null) {
                await this.handleEvent(rtEvt, false);
            }
            GUIManager.Instance.execGuiCommandQueue();
            this._currCtrl = null;
            if (task !== null)
                await Task.CommandsProcessor.SendMonitorOnly();
            else {
                await Task.CommandsProcessor.SendMonitorOnly();
            }
            if (rtEvt !== null && rtEvt.isGuiTriggeredEvent()) {
                if (delta > 0) {
                    delta = Misc.getSystemMilliseconds() - delta;
                }
            }
            if (eventsToHandleCnt > 0 && rtEvt !== null && rtEvt.getType() === ConstInterface.EVENT_TYPE_INTERNAL && rtEvt.InternalEvent < InternalInterface.MG_ACT_TOT_CNT)
                eventsPolledCnt++;
        }
    }
    async handleStopExecution(rtEvt) {
        let stop = false;
        if (!this.GetStopExecutionFlag())
            return stop;
        let task = rtEvt.getTask();
        let currCtrl = rtEvt.Control;
        let lastParkedCtrl = LastFocusedManager.getLastFocusedControl();
        let endTaskDueToVerifyError = false;
        if (rtEvt.getType() === ConstInterface.EVENT_TYPE_INTERNAL) {
            switch (rtEvt.getInternalCode()) {
                case InternalInterface.MG_ACT_TASK_PREFIX:
                    endTaskDueToVerifyError = true;
                    break;
                case InternalInterface.MG_ACT_REC_PREFIX:
                    endTaskDueToVerifyError = true;
                    break;
                case InternalInterface.MG_ACT_REC_SUFFIX:
                    this.setStopExecutionCtrl(null);
                    this.setStopExecutionCtrl(task.getLastParkedCtrl());
                    break;
                case InternalInterface.MG_ACT_CTRL_PREFIX:
                    this.setStopExecution(false);
                    task.InCtrlPrefix = false;
                    if (lastParkedCtrl === null) {
                        task.setCurrField(null);
                        this.setStopExecutionCtrl(null);
                        this.setStopExecutionCtrl(currCtrl);
                        await this.HandleNonParkableControls(task);
                    }
                    else {
                        task.setCurrField(lastParkedCtrl.getField());
                        await FocusManager.SetFocus(lastParkedCtrl, -1);
                    }
                    this.setStopExecution(true);
                    break;
                case InternalInterface.MG_ACT_CTRL_SUFFIX:
                    currCtrl.setInControlSuffix(false);
                    break;
            }
            if (endTaskDueToVerifyError) {
                await task.endTask(true, false, true);
                stop = true;
            }
        }
        return stop;
    }
    async handleForceExitBefore(evt, rtEvt) {
        let forceExit = evt.UserEvt.ForceExit;
        let isNewRec = false;
        let oldIdx = 0;
        if (forceExit === ForceExit.None || rtEvt.isImmediate())
            return false;
        this._isHandlingForceExit = true;
        let task = rtEvt.getTask();
        let oldRec = task.DataView.getCurrRec();
        if (oldRec !== null && oldRec.getMode() === DataModificationTypes.Insert && oldRec.isNewRec()) {
            isNewRec = true;
            oldIdx = oldRec.getId();
        }
        if (forceExit !== ForceExit.Editing && task.getLevel() === Constants.TASK_LEVEL_CONTROL) {
            let ctrl = task.getLastParkedCtrl();
            if (ctrl !== null)
                await this.handleInternalEventWithMgControlBase(ctrl, InternalInterface.MG_ACT_CTRL_SUFFIX);
        }
        if ((forceExit === ForceExit.PreRecordUpdate || forceExit === ForceExit.PostRecordUpdate) && task.getLevel() === Constants.TASK_LEVEL_RECORD) {
            await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
            if (task.getMode() === Constants.TASK_MODE_CREATE && !this._stopExecution)
                task.setMode(Constants.TASK_MODE_MODIFY);
            let currRec = task.DataView.getCurrRec();
            if (currRec !== null && this._stopExecution && isNewRec && oldIdx === currRec.getId() && task.getMode() === Constants.TASK_MODE_CREATE) {
                currRec.clearMode();
                currRec.restart(DataModificationTypes.Insert);
            }
        }
        if (forceExit === ForceExit.PostRecordUpdate && !this._stopExecution) {
            if (task.getLevel() === Constants.TASK_LEVEL_TASK)
                await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
        }
        if (forceExit === ForceExit.Editing) {
            let evtCtrl = rtEvt.Control;
            if (evtCtrl !== null && evtCtrl.getField() !== null) {
                let ctrlVal = Manager.GetCtrlVal(evtCtrl);
                await evtCtrl.validateAndSetValue(ctrlVal, true);
            }
        }
        return true;
    }
    async handleForceExitAfter(rtEvt) {
        let forceExit = rtEvt.UserEvt.ForceExit;
        let task = rtEvt.getTask();
        if (forceExit === ForceExit.PreRecordUpdate) {
            await this.commitRecord(task, true);
            if (!task.transactionFailed(ConstInterface.TRANS_RECORD_PREFIX) && !task.isAborting()) {
                if (await task.evalEndCond(ConstInterface.END_COND_EVAL_AFTER))
                    await task.endTask(true, false, false);
            }
        }
        if (forceExit !== ForceExit.PostRecordUpdate) {
            task = rtEvt.getTask();
            if (task.getLevel() === Constants.TASK_LEVEL_TASK)
                await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
        }
        let ctrl = LastFocusedManager.getLastFocusedControl();
        if (ctrl !== null && forceExit !== ForceExit.Editing) {
            await this.handleInternalEventWithMgControlBase(ctrl, InternalInterface.MG_ACT_CTRL_PREFIX);
        }
        if (forceExit === ForceExit.Editing) {
            let evtCtrl = rtEvt.Control;
            if (evtCtrl !== null && evtCtrl.getField() !== null)
                await evtCtrl.getField().updateDisplay();
        }
    }
    async handleEvents_1(handlersTab, rtEvt) {
        try {
            this.pushRtEvent(rtEvt);
            for (let i = 0; i < handlersTab.getSize(); i++) {
                let handler = handlersTab.getHandler(i);
                if (rtEvt.getType() === ConstInterface.EVENT_TYPE_EXPRESSION || handler.isSpecificHandlerOf(rtEvt) || handler.isNonSpecificHandlerOf(rtEvt) || handler.isGlobalHandlerOf(rtEvt)) {
                    await handler.execute(rtEvt, false, false);
                }
            }
        }
        finally {
            this.popRtEvent();
        }
    }
    async handleKeyDown(task, ctrl, evt) {
        let kbdItem = evt.getKbdItmAlways();
        let keyCode = kbdItem.getKeyCode();
        this._currKbdItem = kbdItem;
        if (evt.IgnoreSpecifiedControl) {
            ctrl = this.getCurrCtrl();
            task = (((ctrl.getForm().getTask() instanceof Task) ? ctrl.getForm().getTask() : null));
            if (ctrl.Type !== MgControlType.CTRL_TYPE_TEXT)
                return;
        }
        Logger.Instance.WriteDevToLog('Start handling KEYDOWN event. Key code: ' + keyCode);
        try {
            if (this._currKbdItem.equals(this.KBI_DOWN) || this._currKbdItem.equals(this.KBI_UP)) {
                if (ctrl !== null && (ctrl.Type === MgControlType.CTRL_TYPE_LIST || ctrl.Type === MgControlType.CTRL_TYPE_RADIO))
                    return;
            }
            let rtEvt = new RunTimeEvent(task, ctrl);
            rtEvt.setSystem(this._currKbdItem);
            rtEvt.setEditParms(evt.getStartSelection(), evt.getEndSelection(), evt.getValue());
            try {
                await this.handleEvent(rtEvt, false);
            }
            finally {
                if (this._stopExecution) {
                    let returnToCtrl = MgControl.ReturnToCtrl;
                    if (returnToCtrl !== null && MgControl.validReturnToCtrl())
                        returnToCtrl.getForm().getTask().setLastParkedCtrl(returnToCtrl);
                }
            }
            let mgd = MGDataCollection.Instance.getCurrMGData();
            if (mgd !== null && !mgd.IsAborting)
                await this.handleExpressionHandlers();
            Logger.Instance.WriteDevToLog('End handling KEYDOWN event. Key code: ' + keyCode);
            if (ctrl !== null && ctrl.IsRepeatable && ctrl === MgControl.ReturnToCtrl) {
                await ctrl.getForm().bringRecordToPage();
            }
        }
        finally {
        }
    }
    async handleSelection(ctrl, line, val) {
        if (ctrl.Type === MgControlType.CTRL_TYPE_TAB || ctrl.Type === MgControlType.CTRL_TYPE_CHECKBOX || ctrl.Type === MgControlType.CTRL_TYPE_RADIO || ctrl.Type === MgControlType.CTRL_TYPE_COMBO || ctrl.Type === MgControlType.CTRL_TYPE_LIST) {
            let checkFocus = (ctrl.Type !== MgControlType.CTRL_TYPE_TAB && ctrl.Type !== MgControlType.CTRL_TYPE_CHECKBOX);
            let allowSelection = (checkFocus ? (ctrl.getForm().getTask().getLastParkedCtrl() === ctrl) : true);
            if (!allowSelection || !ctrl.isModifiable() || !await ctrl.IsParkable())
                await ctrl.restoreOldValue();
            else {
                if (ctrl.isChoiceControl())
                    Commands.setGetSuggestedValueOfChoiceControlOnTagData(ctrl, ctrl.IsRepeatable ? line : 0, true);
                if (ctrl.isDifferentValue(val, ctrl.IsNull, true)) {
                    ctrl.ModifiedByUser = true;
                    await this.handleCtrlModify(ctrl, InternalInterface.MG_ACT_SELECTION);
                }
                if (ctrl.isChoiceControl())
                    Commands.setGetSuggestedValueOfChoiceControlOnTagData(ctrl, ctrl.IsRepeatable ? line : 0, false);
                if (!ctrl.IsMultipleSelectionListBox())
                    await ctrl.validateAndSetValue(val, true);
            }
        }
    }
    async handleMouseUp(ctrl, line, produceClick) {
        if (ctrl === null)
            return;
        if (ctrl.Type === MgControlType.CTRL_TYPE_RADIO)
            RuntimeContextBase.Instance.CurrentClickedRadio = ctrl;
        try {
            if (ctrl.Type !== MgControlType.CTRL_TYPE_TABLE) {
                if (!await ctrl.checkPropWithLine(PropInterface.PROP_TYPE_ENABLED, true, line)) {
                    return;
                }
                let canGotoCtrl = this.canGoToControl(ctrl, true);
                let StopFocus = ctrl.isFocusedStopExecution();
                let returnToCtrl = MgControl.ReturnToCtrl;
                if ((!canGotoCtrl || ((this._stopExecution || StopFocus) && returnToCtrl !== ctrl)) && returnToCtrl !== null) {
                    MGDataCollection.Instance.currMgdID = returnToCtrl.getForm().getTask().getMgdID();
                    if (returnToCtrl.IsRepeatable)
                        await returnToCtrl.getForm().bringRecordToPage();
                    await FocusManager.SetFocus(returnToCtrl, -1);
                    returnToCtrl.getForm().getTask().setLastParkedCtrl(returnToCtrl);
                }
                if (this._stopExecution || StopFocus)
                    return;
            }
            if (produceClick && (ctrl.Type === MgControlType.CTRL_TYPE_BUTTON || ctrl.Type === MgControlType.CTRL_TYPE_CHECKBOX || ctrl.Type === MgControlType.CTRL_TYPE_LIST)) {
                let rtEvt = new RunTimeEvent(ctrl);
                rtEvt.setInternal(InternalInterface.MG_ACT_WEB_CLICK);
                this.addToTail(rtEvt);
            }
        }
        finally {
            RuntimeContextBase.Instance.CurrentClickedRadio = null;
        }
    }
    checkAndSaveRouteEventIfCtxRemovedFromSrvr(aRtEvt) {
        if (aRtEvt.getType() == ConstInterface.EVENT_TYPE_INTERNAL && aRtEvt.getInternalCode() ==
            InternalInterface.MG_ACT_WEBCLIENT_ROUTE && RuntimeContextBase.Instance.RemovedContextFromServer) {
            this.setLastSavedRouteEvent(aRtEvt);
        }
    }
    async handleFocus(ctrl, line, onClick) {
        try {
            if (ctrl.isStatic())
                return;
            if (ctrl !== this._lastFailedFocusedControl && ctrl.HasExpOrParentTaskField()) {
                if (ctrl.isListBox() || ctrl.isCheckBox() || ctrl.isComboBox() || ctrl.isTextControl() || ctrl.isRadio() || ctrl.isTabControl()) {
                    Logger.Instance.WriteErrorToLog(NString.Format('Control either has expression or has parent task variable attached to it. Failed to park on control with name: {0}', ctrl.Name));
                    FocusManager.SetFocus(LastFocusedManager.Instance.getLastFocusedTask(), LastFocusedManager.getLastFocusedControl(), 0, LastFocusedManager.getLastFocusedControl() === null ? true : false);
                    this._lastFailedFocusedControl = ctrl;
                    return;
                }
            }
            this._lastFailedFocusedControl = null;
            if (onClick)
                RuntimeContextBase.Instance.CurrentClickedCtrl = ctrl;
            let prevCtrl = LastFocusedManager.getLastFocusedControl();
            if (ctrl === prevCtrl && !ctrl.isRepeatable()) {
                await ctrl.getForm().bringRecordToPage();
                return;
            }
            let prevTask = LastFocusedManager.Instance.getLastFocusedTask();
            if (ctrl === null || ctrl.Type === MgControlType.CTRL_TYPE_TABLE) {
                if (prevCtrl === null)
                    await this.HandleNonParkableControls(ctrl.getForm().getTask());
                return;
            }
            let currTask;
            if ((ctrl.Type === MgControlType.CTRL_TYPE_BUTTON) && ((await (ctrl.getProp(PropInterface.PROP_TYPE_RAISE_AT)).getValueInt()) === RaiseAt.TaskInFocus))
                currTask = prevTask;
            else
                currTask = ctrl.getForm().getTask();
            let canGotoCtrl = this.canGoToControl(ctrl, onClick);
            if (canGotoCtrl) {
                if (this.guiPendingEvent != null)
                    return;
                let handled = await this.HandleSubformModality(ctrl, prevTask, currTask);
                if (handled)
                    return;
                let eventLine = (ctrl.isRepeatable() || ctrl.isRadio()) ? line : Int32.MinValue;
                let rtEvt = new RunTimeEvent(ctrl, eventLine);
                rtEvt.setInternal(InternalInterface.MG_ACT_CTRL_PREFIX);
                await this.handleEvent(rtEvt, false);
            }
            else if (prevCtrl !== null) {
                if (!prevCtrl.isModifiable()) {
                    if (prevCtrl.isChoiceControl())
                        await prevCtrl.restoreOldValue();
                }
                else if (!currTask.cancelWasRaised()) {
                    let newVal = Manager.GetCtrlVal(prevCtrl);
                    if (!await prevCtrl.validateAndSetValue(newVal, true))
                        this.setStopExecution(true);
                }
                MgControl.ReturnToCtrl = prevCtrl;
            }
            else
                await this.HandleNonParkableControls(ctrl.getForm().getTask());
            if (!this._stopExecution && MgControl.ReturnToCtrl === ctrl)
                LastFocusedManager.setLastFocusedControl(ctrl.getForm().getTask(), ctrl);
            return;
        }
        finally {
            ctrl.setFocusedStopExecution(this._stopExecution);
            if (this._stopExecution) {
                {
                    let returnToCtrl = MgControl.ReturnToCtrl;
                    if (returnToCtrl !== null) {
                        MGDataCollection.Instance.currMgdID = returnToCtrl.getForm().getTask().getMgdID();
                        await FocusManager.SetFocus(returnToCtrl, -1);
                        returnToCtrl.getForm().getTask().setLastParkedCtrl(returnToCtrl);
                    }
                }
            }
        }
    }
    async HandleSubformModality(ctrl, prevTask, currTask) {
        let handledSubformModality = false;
        let length = prevTask.getMGData().getTasksCount();
        let topTask = prevTask.getMGData().getTask(length - 1);
        if (currTask == prevTask || prevTask != topTask) {
            if (topTask != null && prevTask != topTask && topTask.IsSubForm) {
                let subformCtrl = topTask.getForm().getSubFormCtrl();
                if (subformCtrl.IsModal())
                    prevTask = topTask;
            }
        }
        if (prevTask != ctrl.getForm().getTask() && prevTask.getIsDestinationCall()) {
            if (!currTask.isDescendentOf(prevTask)) {
                let subformCtrl = prevTask.getForm().getSubFormCtrl();
                if (!subformCtrl.IsModal() && !currTask.getIsDestinationCall())
                    subformCtrl = prevTask.GetSubformModalParentTask();
                if (subformCtrl != null && (ctrl != subformCtrl) && subformCtrl.IsModal()) {
                    await this.performSubformClose(subformCtrl);
                    if (!this._stopExecution)
                        this.guiPendingEvent = this.getLastRtEvent();
                    handledSubformModality = true;
                }
            }
        }
        return handledSubformModality;
    }
    async handleFocusOnNonMagicControl(ctrl) {
        try {
            await EventsManager.Instance.handleInternalEventWithMgControlBase(ctrl, InternalInterface.MG_ACT_CTRL_SUFFIX);
        }
        finally {
            ctrl.setFocusedStopExecution(this._stopExecution);
            if (this._stopExecution) {
                {
                    let returnToCtrl = MgControl.ReturnToCtrl;
                    if (returnToCtrl !== null) {
                        MGDataCollection.Instance.currMgdID = returnToCtrl.getForm().getTask().getMgdID();
                        await FocusManager.SetFocus(returnToCtrl, -1);
                        returnToCtrl.getForm().getTask().setLastParkedCtrl(returnToCtrl);
                    }
                }
            }
        }
    }
    canGoToControl(ctrl, onClick) {
        if (onClick && (ctrl.Type === MgControlType.CTRL_TYPE_SUBFORM || ctrl.Type === MgControlType.CTRL_TYPE_BROWSER))
            return false;
        let canGotoCtrl = true;
        let currTask = ctrl.getForm().getTask();
        let lastTask = LastFocusedManager.Instance.getLastFocusedTask();
        if (!currTask.IsInteractive || (lastTask != null && lastTask.getLastParkedCtrl() != null &&
            lastTask.cancelWasRaised()))
            return false;
        if (ctrl.Type === MgControlType.CTRL_TYPE_IMAGE && !ctrl.IsRepeatable)
            canGotoCtrl = false;
        return canGotoCtrl;
    }
    async handleTimer(evt) {
        evt.setType(ConstInterface.EVENT_TYPE_TIMER);
        if (evt.isIdleTimer()) {
            await this.handleExpressionHandlers();
            await this.handleContextInactivityAlert(evt);
        }
        else {
            let mgd = MGDataCollection.Instance.getMGData(evt.getMgdID());
            await this.handleEvents(mgd.getTimerHandlers(), evt);
        }
    }
    async handleContextInactivityAlert(evt) {
        let timeNow = Misc.getSystemMilliseconds();
        let mgd = evt.getTask().getMGData();
        let lastRequestTime = RemoteCommandsProcessor.GetInstance().GetLastRequestTime;
        let contextInactivityTimeout = Environment.Instance.getContextInactivityTimeout() / 10;
        let contextInactivityWarningTime = Environment.Instance.getContextInactivityWarningTime() / 10;
        let idleTime = (timeNow - lastRequestTime) / 1000;
        if (contextInactivityTimeout == 0 || contextInactivityWarningTime == 0)
            return;
        if (EventsManager.userGrantedForContextActivity && idleTime >= (contextInactivityTimeout - contextInactivityWarningTime)) {
            let userConfirmation = await GUIManager.Instance.MessageBox(this.confirmationDialogTitle, this.confirmationDialogMsg, Styles.MSGBOX_BUTTON_YES_NO);
            if (userConfirmation == Styles.MSGBOX_RESULT_YES) {
                let cmd = CommandFactory.CreateContextTimeoutResetEventCommand();
                mgd.CmdsToServer.Add(cmd);
                await RemoteCommandsProcessor.GetInstance().Execute(CommandsProcessorBase_SendingInstruction.ONLY_COMMANDS);
            }
            else {
                EventsManager.userGrantedForContextActivity = false;
            }
        }
    }
    async handleExpressionHandlers() {
        let rtEvt = new RunTimeEvent(LastFocusedManager.Instance.getLastFocusedTask(), LastFocusedManager.getLastFocusedControl());
        rtEvt.setExpression(null);
        rtEvt.setMainPrgCreator(rtEvt.getTask());
        for (let i = 0; i < MGDataCollection.Instance.getSize(); i++) {
            let mgd = MGDataCollection.Instance.getMGData(i);
            if (mgd !== null && !mgd.IsAborting)
                await this.handleEvents(mgd.getExpHandlers(), rtEvt);
        }
    }
    handleResize(ctrl, rowsInPage) {
        ctrl.getForm().setRowsInPage(rowsInPage);
    }
    async handleRowDataCurPage(table, desiredTopIndex, sendAll, lastFocusedVal) {
        let form = table.getForm();
        await form.setRowData(desiredTopIndex, sendAll);
    }
    async handleTransferDataToGui(ctrl) {
        let form = ctrl.getForm();
        await form.transferDataToGui();
    }
    async fetchRecordsAheadFromServer(ctrl) {
        let form = ctrl.getForm();
        await form.FetchRecordsAheadFromServer();
    }
    handleEnableEvents(evt, enable) {
        evt.getTask().ActionManager.enableList(evt.getActEnableList(), enable, true);
    }
    async handleCtrlPrefix(evt) {
        let task = evt.getTask();
        let currClickedCtrl = RuntimeContextBase.Instance.CurrentClickedCtrl;
        let form = task.getForm();
        let displayLine = evt.getDisplayLine();
        let ctrl = evt.Control;
        if (form.isLineMode()) {
            if (displayLine === Int32.MinValue || ctrl.isRadio())
                displayLine = form.DisplayLine;
        }
        else
            displayLine = Int32.MinValue;
        let lastParkedCtrl = LastFocusedManager.getLastFocusedControl();
        if (lastParkedCtrl !== null && lastParkedCtrl.getForm().getTask().getMGData().GetId() === ctrl.getForm().getTask().getMGData().GetId() && lastParkedCtrl.getForm().getTask().getLevel() === Constants.TASK_LEVEL_CONTROL) {
            if ((ctrl !== lastParkedCtrl || (ctrl.isRepeatable() && displayLine !== ctrl.getDisplayLine(true))) && !lastParkedCtrl.isInControlSuffix()) {
                await this.handleInternalEventWithMgControlBase(lastParkedCtrl, InternalInterface.MG_ACT_CTRL_SUFFIX);
                if (this._stopExecution) {
                    if (ctrl.Type === MgControlType.CTRL_TYPE_LIST || ctrl.Type === MgControlType.CTRL_TYPE_RADIO || ctrl.Type === MgControlType.CTRL_TYPE_CHECKBOX || ctrl.isTabControl()) {
                        ctrl.resetPrevVal();
                        await ctrl.RefreshDisplay();
                    }
                    return false;
                }
            }
            else
                return false;
        }
        if (task.InCtrlPrefix)
            return false;
        task.InCtrlPrefix = true;
        let prevCtrl = lastParkedCtrl;
        if (currClickedCtrl !== null && currClickedCtrl.getForm().getTask().getMgdID() === ctrl.getForm().getTask().getMgdID() && ctrl.onDiffForm(currClickedCtrl))
            prevCtrl = currClickedCtrl;
        if (prevCtrl !== null && ctrl.onDiffForm(prevCtrl)) {
            let prevTask = prevCtrl.getForm().getTask();
            await prevTask.execSubformRecSuffix(ctrl.getForm().getTask());
            if (this._stopExecution) {
                if (ctrl.Type === MgControlType.CTRL_TYPE_LIST || ctrl.Type === MgControlType.CTRL_TYPE_RADIO || ctrl.Type === MgControlType.CTRL_TYPE_CHECKBOX || ctrl.isTabControl()) {
                    ctrl.resetPrevVal();
                    await ctrl.RefreshDisplay();
                }
                task.InCtrlPrefix = false;
                return false;
            }
            if (ctrl.getForm().getTask().isAborting()) {
                task.InCtrlPrefix = false;
                return false;
            }
            await EventsManager.doSyncForSubformParent(task);
        }
        if (task.DataView.isEmptyDataview())
            await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
        else {
            task.SubformExecMode = Task_SubformExecModeEnum.SET_FOCUS;
            let rtEvt = new RunTimeEvent(ctrl, displayLine);
            rtEvt.setInternal(InternalInterface.MG_ACT_REC_PREFIX);
            await this.handleEvent(rtEvt, false);
        }
        if (this.GetStopExecutionFlag()) {
            task.InCtrlPrefix = false;
            return false;
        }
        if (!await ctrl.IsParkable() && ctrl !== lastParkedCtrl) {
            task.InCtrlPrefix = false;
            if (lastParkedCtrl === null || !await lastParkedCtrl.invoke())
                await this.HandleNonParkableControls(ctrl.getForm().getTask());
            return false;
        }
        Logger.Instance.WriteSupportToLog(NString.Format('{0} Entering control: control {1}, row {2}', task.GetTaskDetails(), ctrl.Name, displayLine < 0 ? 0 : displayLine), true);
        task.setLevel(Constants.TASK_LEVEL_CONTROL);
        ctrl.KeyStrokeOn = false;
        ctrl.ModifiedByUser = false;
        task.setCurrField(ctrl.getField());
        if (ctrl.IsRepeatable)
            await form.bringRecordToPage();
        if (lastParkedCtrl !== ctrl) {
            if ((lastParkedCtrl === null && ctrl.getForm().isSubForm()) || (lastParkedCtrl !== null && (ctrl.getForm().isSubForm() || lastParkedCtrl.getForm().isSubForm()))) {
                let lastParkedTask = null;
                if (lastParkedCtrl !== null)
                    lastParkedTask = lastParkedCtrl.getForm().getTask();
                if (lastParkedTask !== task) {
                    let firstTime = true;
                    let isSonSubform;
                    let parentTask;
                    if (lastParkedTask !== null) {
                        isSonSubform = lastParkedTask.IsSubForm;
                        for (parentTask = lastParkedTask; parentTask !== null && isSonSubform; parentTask = parentTask.getParent()) {
                            if (!firstTime) {
                                isSonSubform = parentTask.IsSubForm;
                                parentTask.ActionManager.enable(InternalInterface.MG_ACT_ZOOM, false);
                                parentTask.ActionManager.enable(InternalInterface.MG_ACT_DELLINE, false);
                                parentTask.enableCreateActs(false);
                            }
                            else
                                firstTime = false;
                        }
                    }
                    isSonSubform = task.IsSubForm;
                    for (parentTask = task.getParent(); (parentTask !== null) && isSonSubform; parentTask = parentTask.getParent()) {
                        isSonSubform = parentTask.IsSubForm;
                        if (parentTask.getEnableZoomHandler())
                            parentTask.ActionManager.enable(InternalInterface.MG_ACT_ZOOM, true);
                        await parentTask.setCreateDeleteActsEnableState();
                    }
                    await task.enableModes();
                }
            }
        }
        if (task.getEnableZoomHandler() || ctrl.useZoomHandler())
            task.ActionManager.enable(InternalInterface.MG_ACT_ZOOM, true);
        if (ctrl.isTextControl() || ctrl.isChoiceControl())
            ctrl.SetKeyboardLanguage();
        if (ctrl.Type === MgControlType.CTRL_TYPE_BUTTON)
            task.ActionManager.enable(InternalInterface.MG_ACT_BUTTON, true);
        await task.setCreateDeleteActsEnableState();
        let field = ctrl.getField();
        if (field !== null && field.NullAllowed && ctrl.isModifiable() && ctrl.Type !== MgControlType.CTRL_TYPE_BUTTON)
            task.ActionManager.enable(InternalInterface.MG_ACT_RT_EDT_NULL, true);
        return true;
    }
    async commonHandlerBefore(evt) {
        let evtSave = evt;
        let task = evt.getTask();
        let rec = null;
        let prevRec = null;
        let val;
        let oldDisplayLine = Int32.MinValue;
        if (this._stopExecution)
            return false;
        if (task.isAborting())
            return false;
        if (evt.getType() === ConstInterface.EVENT_TYPE_SYSTEM) {
            let actId = this.getMatchingAction(evt.getKbdItm());
            if (actId !== 0 && evt.Control !== null) {
                evt = new RunTimeEvent(evt, evt);
                evt.setInternal(actId);
                evt.setArgList(null);
                let mgControl = evt.Control;
                evtSave.setCtrl(mgControl);
                if (mgControl === null)
                    return false;
            }
        }
        if (evt.getType() === ConstInterface.EVENT_TYPE_INTERNAL) {
            let form = task.getForm();
            let displayLine = evt.getDisplayLine();
            let dv = task.DataView;
            rec = dv.getCurrRec();
            let ctrl = evt.Control;
            let intEvtCode = evt.getInternalCode();
            let lastFocusedCtrl = LastFocusedManager.getLastFocusedControl();
            switch (intEvtCode) {
                case InternalInterface.MG_ACT_ZOOM:
                    let boolval = await this.selectProg(ctrl);
                    if (boolval)
                        return false;
                case InternalInterface.MG_ACT_WEB_ON_DBLICK:
                    break;
                case InternalInterface.MG_ACT_CTRL_FOCUS:
                    await this.handleFocus(ctrl, displayLine, evt.isProduceClick());
                    return false;
                case InternalInterface.MG_ACT_CTRL_FOCUS_ON_NON_MAGIC_CONTROL:
                    await this.handleFocusOnNonMagicControl(ctrl);
                    return false;
                case InternalInterface.MG_ACT_CTRL_KEYDOWN:
                    await this.handleKeyDown(task, ctrl, evt);
                    return false;
                case InternalInterface.MG_ACT_CTRL_MOUSEUP:
                    await this.handleMouseUp(ctrl, displayLine, evt.isProduceClick());
                    return false;
                case InternalInterface.MG_ACT_SELECTION:
                    await this.onActSelection(ctrl, form, evt);
                    return false;
                case InternalInterface.MG_ACT_TIMER:
                    await this.handleTimer(evt);
                    return false;
                case InternalInterface.MG_ACT_RESIZE:
                    this.handleResize(ctrl, displayLine);
                    return false;
                case InternalInterface.MG_ACT_ROW_DATA_CURR_PAGE:
                    await this.handleRowDataCurPage(ctrl, displayLine, evt.isSendAll(), evt.LastFocusedVal);
                    return false;
                case InternalInterface.MG_ACT_DV_TO_GUI:
                    await this.handleTransferDataToGui(ctrl);
                    return false;
                case InternalInterface.MG_ACT_FETCH_RECORDS_AHEAD_FROM_SERVER:
                    await this.fetchRecordsAheadFromServer(ctrl);
                    break;
                case InternalInterface.MG_ACT_ENABLE_EVENTS:
                    this.handleEnableEvents(evt, true);
                    return false;
                case InternalInterface.MG_ACT_DISABLE_EVENTS:
                    this.handleEnableEvents(evt, false);
                    return false;
                case InternalInterface.MG_ACT_TASK_PREFIX:
                    task.setLevel(Constants.TASK_LEVEL_TASK);
                    break;
                case InternalInterface.MG_ACT_TASK_SUFFIX:
                    task.setLevel(Constants.TASK_LEVEL_TASK);
                    break;
                case InternalInterface.MG_ACT_REC_PREFIX:
                    if (task.IsSubForm && task.getParent().InCommonHandlerBeforeRP)
                        return false;
                    task.setAfterRetry(ConstInterface.RECOVERY_NONE);
                    task.DataSynced = false;
                    if (!task.isStarted())
                        return false;
                    if (form.getRowsInPage() > Int32.MinValue && displayLine > Int32.MinValue) {
                        while (!dv.recExists(displayLine) && displayLine > 0) {
                            displayLine--;
                        }
                        if (displayLine >= 0) {
                            prevRec = dv.getCurrRec();
                            let oldTaskMode = task.getMode();
                            let oldRecId = prevRec.getId();
                            try {
                                oldDisplayLine = form.DisplayLine;
                                form.PrevDisplayLine = oldDisplayLine;
                                await form.setCurrRowByDisplayLine(displayLine, true, false);
                                if (prevRec != null && !dv.recExistsById(prevRec.getId())) {
                                    await form.RefreshDisplay(Constants.TASK_REFRESH_TABLE);
                                    await form.RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
                                }
                                rec = dv.getCurrRec();
                            }
                            catch (e) {
                                if (e instanceof RecordOutOfDataViewException) {
                                    if (this._stopExecution) {
                                        if (oldTaskMode === Constants.TASK_MODE_CREATE && oldDisplayLine > 0 &&
                                            !dv.recExistsById(oldRecId)) {
                                            let prevLine = form.getPrevLine(oldDisplayLine);
                                            await form.restoreOldDisplayLine(prevLine);
                                        }
                                        else
                                            await form.restoreOldDisplayLine(oldDisplayLine);
                                        await form.RefreshDisplay(Constants.TASK_REFRESH_FORM);
                                        this.setStopExecution(false);
                                        lastFocusedCtrl = task.getLastParkedCtrl();
                                        task.getLastParkedCtrl();
                                        if (lastFocusedCtrl != null)
                                            await lastFocusedCtrl.invoke();
                                        this.setStopExecution(true);
                                        return false;
                                    }
                                    else
                                        Logger.Instance.WriteDevToLog(e.StackTrace);
                                }
                                else
                                    throw e;
                            }
                        }
                        else
                            throw new ApplicationException('in EventsManager.commonHandlerBefore(): invalid line number: ' +
                                displayLine);
                    }
                    if (rec == null)
                        throw new ApplicationException('in EventsManager.commonHandlerBefore(): no current record available !');
                    if (!task.IsInteractive && task.getLevel() === Constants.TASK_LEVEL_TASK)
                        task.increaseCounter();
                    if (await task.evalEndCond(ConstInterface.END_COND_EVAL_BEFORE)) {
                        await task.endTask(true, false, false);
                        return false;
                    }
                    if (form.HasTable())
                        await form.bringRecordToPage();
                    if (task.getLevel() !== Constants.TASK_LEVEL_TASK)
                        return false;
                    await form.RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
                    Logger.Instance.WriteDevToLog('RECORD PREFIX: ' + task.queryTaskPath());
                    (dv.getCurrRec()).setForceSaveOrg(false);
                    (dv.getCurrRec()).Synced = false;
                    dv.saveOriginal();
                    if (task.IsSubForm && task.PerformParentRecordPrefix) {
                        let parentTask = task.getParent();
                        while (parentTask.IsSubForm) {
                            if (parentTask.getLevel() === Constants.TASK_LEVEL_TASK) {
                                await this.handleInternalEventWithTask(parentTask, InternalInterface.MG_ACT_REC_PREFIX);
                                parentTask = parentTask.getParent();
                                if (this.GetStopExecutionFlag())
                                    break;
                            }
                            else
                                break;
                        }
                    }
                    task.InCommonHandlerBeforeRP = true;
                    if (await task.CheckRefreshSubTasks())
                        await dv.computeSubForms();
                    task.InCommonHandlerBeforeRP = false;
                    task.setLevel(Constants.TASK_LEVEL_RECORD);
                    await task.enableRecordActions();
                    break;
                case InternalInterface.MG_ACT_REC_SUFFIX:
                    let forceSuffix = false;
                    let isSelectionTable;
                    task.ConfirmUpdateNo = false;
                    if (rec == null)
                        return false;
                    forceSuffix = await task.checkProp(PropInterface.PROP_TYPE_FORCE_SUFFIX, false);
                    if ((rec.Modified || forceSuffix) && rec.getMode() === DataModificationTypes.Delete &&
                        task.getMode() === Constants.TASK_MODE_DELETE)
                        return true;
                    if (task.getLevel() === Constants.TASK_LEVEL_TASK)
                        return false;
                    if (task.getLevel() === Constants.TASK_LEVEL_CONTROL) {
                        task.setInRecordSuffix(true);
                        await this.handleInternalEventWithMgControlBase(task.getLastParkedCtrl(), InternalInterface.MG_ACT_CTRL_SUFFIX);
                        task.setInRecordSuffix(false);
                        if (this._stopExecution)
                            return false;
                    }
                    await task.handleEventOnSlaveTasks(InternalInterface.MG_ACT_REC_SUFFIX);
                    if (this._stopExecution)
                        return false;
                    Logger.Instance.WriteDevToLog('RECORD SUFFIX: ' + task.queryTaskPath());
                    task.setLevel(Constants.TASK_LEVEL_TASK);
                    isSelectionTable = await task.checkProp(PropInterface.PROP_TYPE_SELECTION, false);
                    if (task.getMode() === Constants.TASK_MODE_QUERY && !forceSuffix && !rec.Modified &&
                        !isSelectionTable)
                        return false;
                    if (!rec.Modified && !forceSuffix && (!isSelectionTable || !task.InSelect)) {
                        if (rec.getMode() === DataModificationTypes.Delete) {
                            return await this.executeControlVerification(task);
                        }
                        return false;
                    }
                    if (task.getMode() !== Constants.TASK_MODE_DELETE && rec.Modified) {
                        if (!await this.updateConfirmed(task))
                            return false;
                    }
                    let stopExecution = !await this.executeControlVerification(task);
                    if (stopExecution)
                        return false;
                    break;
                case InternalInterface.MG_ACT_CTRL_PREFIX:
                    if (await this.handleCtrlPrefix(evt))
                        return true;
                    else {
                        evt.Control.InControl = false;
                        return false;
                    }
                case InternalInterface.MG_ACT_CANCEL:
                    Manager.CleanMessagePane(task);
                    break;
                case InternalInterface.MG_ACT_CTRL_SUFFIX:
                    if (ctrl == null || this.getStopExecutionCtrl() != null)
                        return false;
                    if (task.getLevel() !== Constants.TASK_LEVEL_CONTROL)
                        return false;
                    if (!ctrl.isSubform()) {
                        val = Manager.GetCtrlVal(ctrl);
                        task.CurrentEditingControl = null;
                        if (!task.cancelWasRaised() &&
                            (ctrl.Type !== MgControlType.CTRL_TYPE_BROWSER && (!isNullOrUndefined(ctrl._field) && (ctrl._field.getType() === StorageAttribute.DATE || ctrl._field.getType() == StorageAttribute.NUMERIC) || val !== null) && typeof val != 'undefined' &&
                                !await ctrl.validateAndSetValue(val, true))) {
                            this.setStopExecution(true);
                            return false;
                        }
                    }
                    if (!this._stopExecution)
                        Manager.CleanMessagePane(task);
                    ctrl.InControl = false;
                    ctrl.ClipBoardDataExists = false;
                    ctrl.ValidateControl = true;
                    ctrl.setInControlSuffix(true);
                    break;
            }
        }
        return true;
    }
    async selectProg(ctrl) {
        let cmd;
        let value;
        let mgVal;
        let progExecuted = false;
        let task = ctrl.getForm().getTask();
        if (ctrl.HasSelectProgram()) {
            if (ctrl.IsParkable()) {
                value = Manager.GetCtrlVal(ctrl);
                if (!ctrl.validateAndSetValue(value, false))
                    this.setStopExecution(true);
                else {
                    let encodedVal = null;
                    mgVal = await ctrl.getMgValue(value);
                    var rec = task.DataView.getCurrRec();
                    encodedVal = rec.getXMLForValue(ctrl.getField().getId(), mgVal);
                    cmd = CommandFactory.CreateSelectProgramCommand(task.getTaskTag(), null, Int32.MinValue, ctrl.getDitIdx(), encodedVal);
                    task.getMGData().CmdsToServer.Add(cmd);
                    let commandsProcessorServer = CommandsProcessorManager.GetCommandsProcessor();
                    await commandsProcessorServer.Execute(CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS);
                    progExecuted = true;
                }
            }
        }
        return progExecuted;
    }
    async executeControlVerification(task) {
        let table = task.getHandlersTab();
        for (let i = 0; i < table.getSize(); i++) {
            let handler = table.getHandler(i);
            let evt = handler.getEvent();
            if (evt.getInternalCode() === InternalInterface.MG_ACT_CTRL_VERIFICATION) {
                let control = ((task.getForm() !== null) ? task.getForm().getCtrlByCtrlName(handler.getCtrlName()) : null);
                if (!isNullOrUndefined(control)) {
                    let runTimeEvent = new RunTimeEvent(control);
                    runTimeEvent.setInternal(InternalInterface.MG_ACT_CTRL_VERIFICATION);
                    await this.handleEvent(runTimeEvent, false);
                    if (this._stopExecution) {
                        LastFocusedManager.setLastFocusedControl(task, control);
                        this.setStopExecutionCtrl(null);
                        this.setStopExecutionCtrl(control);
                        return false;
                    }
                }
            }
        }
        return true;
    }
    async onActSelection(ctrl, form, evt) {
        let displayLine = evt.getDisplayLine();
        if (displayLine === Int32.MinValue)
            displayLine = form.DisplayLine;
        let simulate = ctrl.Type === MgControlType.CTRL_TYPE_CHECKBOX || ctrl.Type === MgControlType.CTRL_TYPE_LIST || ctrl.Type === MgControlType.CTRL_TYPE_RADIO || ctrl.Type === MgControlType.CTRL_TYPE_BUTTON || (ctrl.Type === MgControlType.CTRL_TYPE_TAB && !evt.isProduceClick());
        if (simulate)
            await this.simulateSelection(ctrl, evt.getValue(), displayLine, evt.isProduceClick());
        else
            await this.handleSelection(ctrl, displayLine, evt.getValue());
    }
    async onSubformClose(task, evt) {
        if ((evt.getArgList() != null) && (evt.getArgList().getSize() === 1)) {
            let subformCtrl = null;
            let closingSubformName = await evt.getArgList().getArgValue(0, StorageAttribute.ALPHA, 0);
            if (closingSubformName == null)
                return;
            for (let parentTask = task; parentTask.getForm() != null && parentTask.getMgdID() === task.getMgdID(); parentTask = parentTask.getParent()) {
                subformCtrl = parentTask.getForm().getCtrlByName(closingSubformName.trim(), MgControlType.CTRL_TYPE_SUBFORM);
                if (subformCtrl != null) {
                    let subformTask = subformCtrl.getSubformTask();
                    if (subformTask != null && !subformTask.IsRoute()) {
                        {
                            await this.performSubformClose(subformCtrl);
                        }
                        break;
                    }
                }
            }
        }
    }
    async performSubformClose(subformCtrl) {
        let subformTask = subformCtrl.getSubformTask();
        let parentTask = subformCtrl.getForm().getTask();
        let mgd = subformTask.getMGData();
        this.pushNewExecStacks();
        await subformTask.endTask(true, false, false, true);
        this.popNewExecStacks();
        if (!this._stopExecution) {
            parentTask.TaskService.RemoveRecomputes(parentTask);
            await subformTask.stop();
            subformCtrl.resetSubformTask();
            await CommandsProcessorManager.GetCommandsProcessor().Execute(CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS);
            Commands.addClosesubform(CommandType.CLOSE_SUB_FORM, subformCtrl, subformTask.getTaskTag(), subformCtrl.getName(), subformTask.getForm().buildFormName());
            if (subformCtrl.IsModal())
                mgd.abort();
        }
    }
    createEventArguments(evt) {
        if (evt.getArgList() === null) {
            if (evt.getType() === ConstInterface.EVENT_TYPE_INTERNAL) {
                switch (evt.getInternalCode()) {
                    default:
                        break;
                }
            }
        }
    }
    async DoTaskLevelRecordSuffix(task, refDoSuffix) {
        let shouldReturn = false;
        if (task.getLevel() === Constants.TASK_LEVEL_RECORD) {
            refDoSuffix.value = false;
            await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
            if (this.GetStopExecutionFlag())
                shouldReturn = true;
        }
        return shouldReturn;
    }
    async DoTaskLevelRecordPrefix(task) {
        if (!this.GetStopExecutionFlag()) {
            await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
        }
    }
    async commonHandler(evt) {
        let task = evt.getTask();
        let mgData = task.getMGData();
        let form = task.getForm();
        let ctrl = evt.Control;
        let lastParkedCtrl;
        let doSuffix = true;
        let oldMode;
        let server = Task.CommandsProcessor;
        if (this._stopExecution)
            return;
        if (evt.getType() === ConstInterface.EVENT_TYPE_USER)
            evt = new RunTimeEvent(evt.getUserEvent(), evt);
        try {
            this.pushRtEvent(evt);
            if (evt.getType() === ConstInterface.EVENT_TYPE_SYSTEM) {
                let actId = this.getMatchingAction(evt.getKbdItm());
                if (actId !== 0) {
                    evt = new RunTimeEvent(evt, evt);
                    evt.setInternal(actId);
                    evt.setArgList(null);
                }
            }
            let cmd;
            if (evt.getType() === ConstInterface.EVENT_TYPE_INTERNAL && !evt.isImmediate() && this.ActAllowed(evt.getInternalCode()) && !this._processingTopMostEndTask) {
                let cmdsToServer = mgData.CmdsToServer;
                let dv = task.DataView;
                let rec = dv.getCurrRec();
                let intEvtCode = evt.getInternalCode();
                switch (intEvtCode) {
                    case InternalInterface.MG_ACT_SUBFORM_CLOSE:
                        await this.onSubformClose(task, evt);
                        break;
                    case InternalInterface.MG_ACT_BUTTON:
                    case InternalInterface.MG_ACT_CTRL_HIT:
                        if (ctrl.Type === MgControlType.CTRL_TYPE_BUTTON &&
                            (intEvtCode === InternalInterface.MG_ACT_BUTTON || !evt.isGuiTriggeredEvent() || ctrl.isParkable(true))) {
                            let aRtEvt = new RunTimeEvent(ctrl);
                            if (ctrl.getRtEvtTask() != null)
                                aRtEvt.setTask(ctrl.getRtEvtTask());
                            if (ConstInterface.EVENT_TYPE_NONE !== aRtEvt.getType()) {
                                if (ctrl.useZoomHandler()) {
                                    let taskToEnable = aRtEvt.getTask();
                                    if (taskToEnable != null)
                                        taskToEnable.ActionManager.enable(InternalInterface.MG_ACT_ZOOM, true);
                                }
                                aRtEvt.setArgList(ctrl.ArgList);
                                await aRtEvt.setPublicName();
                                this.addToTail(aRtEvt);
                                this.checkAndSaveRouteEventIfCtxRemovedFromSrvr(aRtEvt);
                            }
                        }
                        else if (ctrl.Type === MgControlType.CTRL_TYPE_TAB) {
                            this.addInternalEventWithCtrlAndCode(ctrl, InternalInterface.MG_ACT_ZOOM);
                        }
                        else if (ctrl.Type === MgControlType.CTRL_TYPE_RADIO) {
                            ctrl.setControlToFocus();
                        }
                        break;
                    case InternalInterface.MG_ACT_TOGGLE_INSERT:
                        RuntimeContextBase.Instance.ToggleInsertMode();
                        break;
                    case InternalInterface.MG_ACT_CLOSE:
                        let rtEvt = new RunTimeEvent(evt, evt);
                        rtEvt.setInternal(InternalInterface.MG_ACT_EXIT);
                        this.addToTail(rtEvt);
                        break;
                    case InternalInterface.MG_ACT_EXIT:
                        let taskToEnd = (task.isMainProg() ? MGDataCollection.Instance.StartupMgData.getFirstTask() : task);
                        await taskToEnd.endTask(evt.reversibleExit(), false, evt.ExitDueToError());
                        break;
                    case InternalInterface.MG_ACT_EXIT_SYSTEM:
                        await MGDataCollection.Instance.StartupMgData.getFirstTask().endTask(evt.reversibleExit(), false, false);
                        break;
                    case InternalInterface.MG_ACT_TBL_NXTLINE:
                        if (form != null)
                            await form.moveInView(Constants.MOVE_UNIT_ROW, Constants.MOVE_DIRECTION_NEXT);
                        break;
                    case InternalInterface.MG_ACT_TBL_PRVLINE:
                        if (form != null)
                            await form.moveInView(Constants.MOVE_UNIT_ROW, Constants.MOVE_DIRECTION_PREV);
                        break;
                    case InternalInterface.MG_ACT_TBL_NXTPAGE:
                        if (form != null)
                            await form.moveInView(Constants.MOVE_UNIT_PAGE, Constants.MOVE_DIRECTION_NEXT);
                        break;
                    case InternalInterface.MG_ACT_TBL_PRVPAGE:
                        if (form != null)
                            await form.moveInView(Constants.MOVE_UNIT_PAGE, Constants.MOVE_DIRECTION_PREV);
                        break;
                    case InternalInterface.MG_ACT_TBL_BEGPAGE:
                        if (form != null)
                            await form.moveInView(Constants.MOVE_UNIT_PAGE, Constants.MOVE_DIRECTION_BEGIN);
                        break;
                    case InternalInterface.MG_ACT_TBL_ENDPAGE:
                        if (form != null)
                            await form.moveInView(Constants.MOVE_UNIT_PAGE, Constants.MOVE_DIRECTION_END);
                        break;
                    case InternalInterface.MG_ACT_TBL_BEGTBL:
                        lastParkedCtrl = task.getLastParkedCtrl();
                        if (lastParkedCtrl == null || !lastParkedCtrl.isListBox()) {
                            if (form != null)
                                await form.moveInView(Constants.MOVE_UNIT_TABLE, Constants.MOVE_DIRECTION_BEGIN);
                        }
                        break;
                    case InternalInterface.MG_ACT_TBL_ENDTBL:
                        lastParkedCtrl = task.getLastParkedCtrl();
                        if (lastParkedCtrl === null || !lastParkedCtrl.isListBox()) {
                            if (form != null)
                                await form.moveInView(Constants.MOVE_UNIT_TABLE, Constants.MOVE_DIRECTION_END);
                        }
                        break;
                    case InternalInterface.MG_ACT_WEB_ON_DBLICK:
                        {
                            let ctrlType = ctrl.Type;
                            let putAction = InternalInterface.MG_ACT_NONE;
                            if (ctrlType !== MgControlType.CTRL_TYPE_TABLE && ctrlType !== MgControlType.CTRL_TYPE_LABEL &&
                                ctrlType !== MgControlType.CTRL_TYPE_IMAGE) {
                                if (ctrl.getForm().getTask().ActionManager.isEnabled(InternalInterface.MG_ACT_SELECT))
                                    putAction = InternalInterface.MG_ACT_SELECT;
                                else {
                                    if (ctrl.getForm().getTask().ActionManager.isEnabled(InternalInterface.MG_ACT_ZOOM))
                                        putAction = InternalInterface.MG_ACT_ZOOM;
                                }
                                if (putAction !== InternalInterface.MG_ACT_NONE)
                                    this.addInternalEventWithCtrlAndCode(ctrl, putAction);
                            }
                        }
                        break;
                    case InternalInterface.MG_ACT_DELLINE:
                        if (await this.HandleActionDelline(task, dv, rec, false))
                            return;
                        break;
                    case InternalInterface.MG_ACT_CRELINE:
                    case InternalInterface.MG_ACT_CRELINE_ABOVE:
                        if (task.DataView.isEmptyDataview()) {
                            await this.gotoCreateMode(task);
                            break;
                        }
                        if (task.getInCreateLine())
                            break;
                        task.setInCreateLine(true);
                        try {
                            if (task.getLevel() === Constants.TASK_LEVEL_CONTROL) {
                                await this.handleInternalEventWithMgControlBase(task.getLastParkedCtrl(), InternalInterface.MG_ACT_CTRL_SUFFIX);
                                if (this.GetStopExecutionFlag())
                                    return;
                            }
                            if (task.getMode() === Constants.TASK_MODE_CREATE && rec != null && !rec.Modified) {
                                if (await task.checkProp(PropInterface.PROP_TYPE_FORCE_SUFFIX, false)) {
                                    doSuffix = false;
                                    await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
                                    if (this.GetStopExecutionFlag() || task.isAborting())
                                        return;
                                }
                                else {
                                    task.setLevel(Constants.TASK_LEVEL_TASK);
                                    await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
                                    if (!this.GetStopExecutionFlag() && form != null) {
                                        let lastParkedControl = task.getLastParkedCtrl();
                                        if (lastParkedControl !== null)
                                            await lastParkedControl.invoke();
                                    }
                                    return;
                                }
                            }
                            let refDoSuffix = new RefParam(doSuffix);
                            if (await this.DoTaskLevelRecordSuffix(task, refDoSuffix))
                                return;
                            doSuffix = refDoSuffix.value;
                            if (form != null) {
                                if ((task.getMode() !== Constants.TASK_MODE_CREATE) || !task.ConfirmUpdateNo)
                                    await form.addRec(doSuffix, intEvtCode == InternalInterface.MG_ACT_CRELINE_ABOVE ? true : false);
                            }
                            await this.DoTaskLevelRecordPrefix(task);
                            await this.HandleActEnterRowEditing(task);
                        }
                        finally {
                            task.setInCreateLine(false);
                        }
                        break;
                    case InternalInterface.MG_ACT_CANCEL:
                        await this.HandleActCancel(evt, task, form, ctrl, rec, true);
                        break;
                    case InternalInterface.MG_ACT_ENTER_ROW_EDITING:
                        await this.HandleActEnterRowEditing(task);
                        break;
                    case InternalInterface.MG_ACT_RT_QUIT:
                        await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_RT_QUIT);
                        break;
                    case InternalInterface.MG_ACT_RTO_CREATE:
                        if (task.getMode() === Constants.TASK_MODE_CREATE ||
                            await (task.getProp(PropInterface.PROP_TYPE_ALLOW_OPTION)).getValueBoolean()) {
                            await this.gotoCreateMode(task);
                        }
                        break;
                    case InternalInterface.MG_ACT_RTO_QUERY:
                        if (task.getMode() !== Constants.TASK_MODE_QUERY &&
                            (await (task.getProp(PropInterface.PROP_TYPE_ALLOW_OPTION)).getValueBoolean()) === false)
                            break;
                        await task.enableModes();
                        oldMode = task.getOriginalTaskMode();
                        await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
                        if (!this.GetStopExecutionFlag()) {
                            task.setMode(Constants.TASK_MODE_QUERY);
                            if (oldMode !== Constants.TASK_MODE_CREATE) {
                                await dv.currRecCompute(true);
                                await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
                                if (ctrl != null)
                                    await ctrl.invoke();
                                await form.RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
                            }
                            else {
                                task.setPreventRecordSuffix(true);
                                task.setPreventControlChange(true);
                                await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_RT_REFRESH_VIEW);
                                if (ctrl != null)
                                    await ctrl.invoke();
                                task.setPreventRecordSuffix(false);
                                task.setPreventControlChange(false);
                                if (!dv.isEmpty() && (dv.getCurrRec()).getMode() !== DataModificationTypes.Insert)
                                    task.setMode(Constants.TASK_MODE_QUERY);
                                else
                                    task.setMode(Constants.TASK_MODE_CREATE);
                                await form.RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
                            }
                            task.setOriginalTaskMode(Constants.TASK_MODE_QUERY);
                            await task.setCreateDeleteActsEnableState();
                        }
                        break;
                    case InternalInterface.MG_ACT_RTO_MODIFY:
                        if (task.getMode() !== Constants.TASK_MODE_MODIFY &&
                            (await (task.getProp(PropInterface.PROP_TYPE_ALLOW_OPTION)).getValueBoolean()) === false)
                            break;
                        await task.enableModes();
                        oldMode = task.getOriginalTaskMode();
                        await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
                        if (!this.GetStopExecutionFlag()) {
                            task.setMode(Constants.TASK_MODE_MODIFY);
                            if (oldMode !== Constants.TASK_MODE_CREATE) {
                                await dv.currRecCompute(true);
                                await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
                                if (ctrl != null)
                                    await ctrl.invoke();
                            }
                            else {
                                task.setPreventRecordSuffix(true);
                                task.setPreventControlChange(true);
                                await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_RT_REFRESH_VIEW);
                                if (ctrl != null)
                                    await ctrl.invoke();
                                task.setPreventRecordSuffix(false);
                                task.setPreventControlChange(false);
                                if (!dv.isEmpty() && (dv.getCurrRec()).getMode() !== DataModificationTypes.Insert)
                                    task.setMode(Constants.TASK_MODE_MODIFY);
                                else
                                    task.setMode(Constants.TASK_MODE_CREATE);
                                await form.RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
                            }
                            task.setOriginalTaskMode(Constants.TASK_MODE_MODIFY);
                            await task.setCreateDeleteActsEnableState();
                        }
                        break;
                    case InternalInterface.MG_ACT_SELECT:
                        try {
                            task.InSelect = true;
                            await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
                        }
                        finally {
                            task.InSelect = false;
                        }
                        if (!this._stopExecution) {
                            await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_EXIT);
                        }
                        break;
                    case InternalInterface.MG_ACT_SUBFORM_REFRESH:
                        if ((evt.getArgList() != null) && (evt.getArgList().getSize() === 1)) {
                            let subformCtrl = null;
                            let RefreshedSubformName = await evt.getArgList().getArgValue(0, StorageAttribute.ALPHA, 0);
                            if (RefreshedSubformName == null)
                                break;
                            for (let parentTask = task; parentTask.getForm() != null && parentTask.getMgdID() === task.getMgdID(); parentTask = parentTask.getParent()) {
                                subformCtrl = parentTask.getForm().getCtrlByName(RefreshedSubformName.trim(), MgControlType.CTRL_TYPE_SUBFORM);
                                if (subformCtrl != null) {
                                    let subformTask = subformCtrl.getSubformTask();
                                    if (subformTask == null) {
                                        if (await (subformCtrl.getProp(PropInterface.PROP_TYPE_SUBFORM_TYPE)).getValueInt() !== SubformType.None)
                                            await this.OpenSubform(parentTask, subformCtrl);
                                    }
                                    else
                                        await parentTask.SubformRefresh(subformTask, true);
                                    break;
                                }
                            }
                        }
                        else if ((evt.getArgList() != null) && (evt.getArgList().getSize() === 2)) {
                            let subformTask = task, parentTask;
                            let taskTag = await evt.getArgList().getArgValue(1, StorageAttribute.NUMERIC, 0);
                            while (subformTask.IsSubForm) {
                                parentTask = subformTask.getParent();
                                if (subformTask.getTaskTag() === taskTag) {
                                    await parentTask.SubformRefresh(subformTask, true);
                                    break;
                                }
                                subformTask = parentTask;
                            }
                        }
                        break;
                    case InternalInterface.MG_ACT_SUBFORM_OPEN:
                        await this.OpenSubform(task, evt.Control);
                        break;
                    case InternalInterface.MG_ACT_RTO_SEARCH:
                        await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
                        if (!this.GetStopExecutionFlag()) {
                            cmd = CommandFactory.CreateEventCommand(task.getTaskTag(), intEvtCode);
                            task.getTaskCache().clearCache();
                            await task.DataviewManager.Execute(cmd);
                            if (!form.isScreenMode()) {
                                dv.setTopRecIdx(0);
                                if (form.HasTable()) {
                                    await form.restoreOldDisplayLine(form.recordIdx2DisplayLine(dv.getCurrRecIdx()));
                                }
                            }
                            await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
                            if (ctrl != null)
                                await ctrl.invoke();
                        }
                        break;
                    case InternalInterface.MG_ACT_RT_REFRESH_SCREEN:
                    case InternalInterface.MG_ACT_RT_REFRESH_VIEW:
                    case InternalInterface.MG_ACT_RT_REFRESH_RECORD:
                        await this.handleRefreshEvents(evt);
                        break;
                    case InternalInterface.MG_ACT_ACT_CLOSE_APPL:
                        cmd =
                            CommandFactory.CreateEventCommand(MGDataCollection.Instance.StartupMgData.getFirstTask().getTaskTag(), InternalInterface.MG_ACT_BROWSER_ESC);
                        cmdsToServer.Add(cmd);
                        await server.Execute(CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS);
                        break;
                    case InternalInterface.MG_ACT_CONTEXT_REMOVE:
                        let currRecord = LastFocusedManager.Instance.getLastFocusedTask().DataView.getCurrRec();
                        if (ClientManager.TransCacheExists() || (currRecord != null && currRecord.Modified)) {
                            console.log('Pending transaction/record is modified, discarded \'Context Remove\' event.');
                        }
                        else {
                            if (Environment.Instance.getClientContextInactivityTimeout() > 0) {
                                await this.ExecuteClientContextInactivityTimer();
                            }
                            if (!EventsManager.clientContextInactivityTimerAborted) {
                                let contextTerminationEventCommand = CommandFactory.CreateContextTerminationEventCommand(MGDataCollection.Instance.StartupMgData.getFirstTask().getTaskTag());
                                MGDataCollection.Instance.getCurrMGData().CmdsToServer.Add(contextTerminationEventCommand);
                                RemoteCommandsProcessor.GetInstance().Execute(CommandsProcessorBase_SendingInstruction.ONLY_COMMANDS);
                                console.log('Removed context from server');
                                RuntimeContextBase.Instance.RemovingContextFromServer = true;
                            }
                            else {
                                console.log('Discarded \'Context Remove\' event as user event occurred.');
                                EventsManager.clientContextInactivityTimerAborted = false;
                                break;
                            }
                        }
                        break;
                    case InternalInterface.MG_ACT_SERVER_TERMINATION:
                    case InternalInterface.MG_ACT_DUMP_ENVIRONMENT:
                        cmd = CommandFactory.CreateEventCommand(task.getTaskTag(), intEvtCode);
                        cmdsToServer.Add(cmd);
                        await CommandsProcessorManager.GetCommandsProcessor().Execute(CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS);
                        break;
                    case InternalInterface.MG_ACT_RT_EDT_NULL:
                        if (this._currField != null && this._currField.NullAllowed) {
                            await this._currField.setValueAndStartRecompute(this._currField.getMagicDefaultValue(), true, true, true, false);
                            await this._currField.updateDisplay();
                        }
                        break;
                    case InternalInterface.MG_ACT_ROLLBACK:
                        await EventsManager.HandleRollbackAction(task, cmdsToServer, RollbackEventCommand_RollbackType.ROLLBACK);
                        break;
                    case InternalInterface.MG_ACT_OK:
                    case InternalInterface.MG_ACT_INDEX_CHANGE:
                    case InternalInterface.MG_ACT_COL_SORT:
                        if ((intEvtCode === InternalInterface.MG_ACT_INDEX_CHANGE && task.DataviewManager.HasRemoteData) ||
                            (intEvtCode === InternalInterface.MG_ACT_COL_SORT && ctrl.getField() != null)) {
                            let lastFocusedTask = LastFocusedManager.Instance.getLastFocusedTask();
                            let focusedCtrl = LastFocusedManager.getLastFocusedControl(LastFocusedManager.LastFocusMgdID);
                            if (focusedCtrl != null)
                                await this.handleInternalEventWithMgControlBase(focusedCtrl, InternalInterface.MG_ACT_CTRL_SUFFIX);
                            if (!this.GetStopExecutionFlag())
                                await task.ExecuteNestedRS(lastFocusedTask);
                            if (!this.GetStopExecutionFlag() && task.getLevel() !== Constants.TASK_LEVEL_TASK)
                                await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
                            if (!this.GetStopExecutionFlag()) {
                                task.getTaskCache().clearCache();
                                if (intEvtCode === InternalInterface.MG_ACT_INDEX_CHANGE) {
                                    let taskForm = task.getForm();
                                    if (taskForm.HasTable())
                                        form.clearTableColumnSortMark(true);
                                    cmd = await CommandFactory.CreateIndexChangeCommand(task.getTaskTag(), task.DataView.GetCurrentRecId(), evt.getArgList());
                                }
                                else {
                                    cmd = CommandFactory.CreateColumnSortCommand(task.getTaskTag(), evt.Direction, ctrl.getDitIdx(), ctrl.getField().getId(), task.DataView.GetCurrentRecId());
                                }
                                await task.DataviewManager.Execute(cmd);
                                dv.setTopRecIdx(0);
                                await form.restoreOldDisplayLine(0);
                                if (!(task === LastFocusedManager.Instance.getLastFocusedTask() || !task.pathContains(LastFocusedManager.Instance.getLastFocusedTask()))) {
                                    await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
                                    if (!this.GetStopExecutionFlag() && !task.getPreventControlChange())
                                        await lastFocusedTask.ExecuteNestedRS(task);
                                    if (!this.GetStopExecutionFlag() && focusedCtrl != null)
                                        await focusedCtrl.invoke();
                                }
                                else
                                    await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
                            }
                        }
                        break;
                    case InternalInterface.MG_ACT_COL_CLICK:
                        if (!this.GetStopExecutionFlag() && evt.isGuiTriggeredEvent() && ctrl.isColumnSortable()) {
                            let columnChildControl = ctrl.getColumnChildControl();
                            if (columnChildControl != null) {
                                let aRtEvt = new RunTimeEvent(columnChildControl, evt.Direction, 0);
                                aRtEvt.setInternal(InternalInterface.MG_ACT_COL_SORT);
                                this.addToTail(aRtEvt);
                            }
                        }
                        break;
                }
            }
        }
        finally {
            this.popRtEvent();
        }
    }
    async HandleActEnterRowEditing(task) {
        if (task.getMode() === Constants.TASK_MODE_QUERY)
            task.WriteToMessagePane(`Cannot enter Row Editing when in Query Mode.`);
        else {
            let myform = task.getForm();
            await myform.startRowEditing(myform.DisplayLine);
        }
    }
    async HandleStopRowEditing(task) {
        let myform = task.getForm();
        await myform.stopRowEditing(myform.DisplayLine);
    }
    async OpenSubform(task, subformControl) {
        task.TaskService.RemoveRecomputes(task);
        let cmd = CommandFactory.CreateSubformOpenCommand(task.getTaskTag(), subformControl.getDitIdx());
        task.getMGData().CmdsToServer.Add(cmd);
        await Task.CommandsProcessor.Execute(CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS);
    }
    static async HandleRollbackAction(task, cmdsToServer, rollbackType) {
        let commandsProcessor = Task.CommandsProcessor;
        ClientRefreshCommand.ClientIsInRollbackCommand = true;
        task = task.TaskService.GetOwnerTransactionTask(task);
        let cmd = CommandFactory.CreateRollbackEventCommand(task.getTaskTag(), rollbackType);
        cmdsToServer.Add(cmd);
        await commandsProcessor.Execute(CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS);
        ClientRefreshCommand.ClientIsInRollbackCommand = false;
    }
    async HandleActCancel(evt, task, form, ctrl, rec, isMG_ACT_CANCEL) {
        if (task.InHandleActCancel)
            return;
        task.InHandleActCancel = true;
        let oldMode = task.getMode();
        let saveDisplayLine = task.getForm().DisplayLine;
        if (oldMode !== Constants.TASK_MODE_QUERY && !rec.Updated) {
            if (isMG_ACT_CANCEL) {
                if (rec.Modified || (task.getLastParkedCtrl() !== null && task.getLastParkedCtrl().ModifiedByUser)) {
                    if (await task.checkProp(PropInterface.PROP_TYPE_CONFIRM_CANCEL, false)) {
                        let confirmed = await GUIManager.Instance.confirm_bool(MsgInterface.CONFIRM_STR_CANCEL);
                        if (!confirmed) {
                            task.InHandleActCancel = false;
                            return;
                        }
                    }
                }
            }
            let oldRecMode = rec.getMode();
            if (GuiEnvironment.Environment.GetSpecialCancelOnCreate() && task.getMode() == Constants.TASK_MODE_CREATE
                && task.getOriginalTaskMode() == Constants.TASK_MODE_CREATE) {
                let enable = await task.checkProp(PropInterface.PROP_TYPE_ALLOW_MODIFY, true);
                if (enable) {
                    let allowEmptyDataView = await task.checkProp(PropInterface.PROP_TYPE_ALLOW_EMPTY_DATAVIEW, true);
                    if (task.DataView.getSize() == 1 && allowEmptyDataView) {
                        task.setMode('M');
                        task.DataView.setEmptyDataview(true);
                    }
                }
            }
            if (isMG_ACT_CANCEL) {
                if (task.isTransactionOnLevel(ConstInterface.TRANS_RECORD_PREFIX) && evt.RollbackInCancel()) {
                    await EventsManager.HandleRollbackAction(task, task.getMGData().CmdsToServer, RollbackEventCommand_RollbackType.CANCEL);
                }
            }
            await EventsManager.HandleCancelEdit(evt, task, form, rec, oldRecMode);
        }
        else {
            task.setLevel(Constants.TASK_LEVEL_TASK);
            let lastFocusedControl = LastFocusedManager.getLastFocusedControl();
            if (lastFocusedControl !== null) {
                lastFocusedControl.getForm().getTask().setLevel(Constants.TASK_LEVEL_TASK);
            }
            if (!evt.isQuit()) {
                await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
                await this.HandleParkabilityOnRowChange(task);
            }
        }
        if (evt.isQuit())
            await this.handleInternalEventWithMgControlBase(ctrl, InternalInterface.MG_ACT_EXIT);
        await form.stopRowEditing(saveDisplayLine);
        task.InHandleActCancel = false;
    }
    static async HandleCancelEdit(evt, task, form, rec, oldRecMode) {
        if (rec.Synced && oldRecMode === DataModificationTypes.Insert)
            rec.setMode(oldRecMode);
        if (form !== null) {
            if (task.getLastParkedCtrl() !== null)
                task.getLastParkedCtrl().resetPrevVal();
            await form.cancelEdit(true, evt.isQuit());
        }
    }
    async HandleActionDelline(task, dv, rec, nonInteractiveDelete) {
        let lastParkedCtrl;
        let myform = task.getForm();
        let saveIsRowEditing = myform.getIsRowEditingFromGui(myform.DisplayLine);
        let oldRecMode;
        if (await task.checkProp(PropInterface.PROP_TYPE_ALLOW_DELETE, true)) {
            lastParkedCtrl = task.getLastParkedCtrl();
            if (lastParkedCtrl !== null)
                await this.handleInternalEventWithMgControlBase(lastParkedCtrl, InternalInterface.MG_ACT_CTRL_SUFFIX);
            if (!this.GetStopExecutionFlag()) {
                oldRecMode = rec.getMode();
                rec.clearMode();
                rec.setMode(DataModificationTypes.Delete);
                if (oldRecMode === DataModificationTypes.Insert)
                    rec.setSendToServer(false);
                if (rec.Modified || await task.checkProp(PropInterface.PROP_TYPE_FORCE_SUFFIX, false)) {
                    task.setMode(Constants.TASK_MODE_MODIFY);
                    await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
                    if (this._stopExecution) {
                        if (!rec.SendToServer) {
                            await dv.addCurrToModified();
                            rec.setSendToServer(true);
                        }
                        rec.clearMode();
                        rec.setMode(oldRecMode);
                        this.setStopExecution(false);
                        await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
                        this.setStopExecution(true);
                        return true;
                    }
                }
                rec.setInDeleteProcess(true);
                task.setMode(Constants.TASK_MODE_DELETE);
                let skipConfirmDialog = nonInteractiveDelete;
                if (!skipConfirmDialog) {
                    if (task.mustConfirmInDeleteMode() || await task.checkProp(PropInterface.PROP_TYPE_CONFIRM_UPDATE, false)) {
                        let confirmed = await GUIManager.Instance.confirm_bool(MsgInterface.CONFIRM_STR_DELETE);
                        if (!confirmed) {
                            rec.clearMode();
                            rec.setMode(oldRecMode);
                            task.setMode(Constants.TASK_MODE_MODIFY);
                            if (lastParkedCtrl !== null)
                                await this.handleInternalEventWithMgControlBase(lastParkedCtrl, InternalInterface.MG_ACT_CTRL_PREFIX);
                            if (saveIsRowEditing) {
                                let myform = task.getForm();
                                await myform.startRowEditing(myform.DisplayLine);
                            }
                            return true;
                        }
                    }
                }
                await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
                rec.setInDeleteProcess(false);
                if (this._stopExecution) {
                    if (!rec.SendToServer) {
                        await dv.addCurrToModified();
                        rec.setSendToServer(true);
                    }
                    rec.clearMode();
                    rec.setMode(oldRecMode);
                    this.setStopExecution(false);
                    task.setMode(Constants.TASK_MODE_MODIFY);
                    await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
                    this.setStopExecution(true);
                    return true;
                }
                if (oldRecMode === DataModificationTypes.Insert) {
                    rec.clearMode();
                    rec.setMode(oldRecMode);
                }
                let newCurrRec = dv.getCurrRec();
                try {
                    if (rec !== newCurrRec)
                        await dv.setCurrRec(rec.getId(), false);
                    if (!task.isAborting())
                        await this.delCurrRec(task, false);
                }
                finally {
                    if (rec !== newCurrRec) {
                        await dv.setCurrRec(newCurrRec.getId(), false);
                        if (dv.getSize() === 1)
                            task.setMode(Constants.TASK_MODE_CREATE);
                    }
                    if (nonInteractiveDelete)
                        task.setMode(Constants.TASK_MODE_DELETE);
                }
            }
        }
        return false;
    }
    async handleRefreshEvents(evt) {
        let task = evt.getTask();
        let form = task.getForm();
        let dv = task.DataView;
        let intEvtCode = evt.getInternalCode();
        let cmd = null;
        if (intEvtCode === InternalInterface.MG_ACT_RT_REFRESH_RECORD)
            this._isInRecordFlushEvent = true;
        if (intEvtCode === InternalInterface.MG_ACT_RT_REFRESH_VIEW && task.getMode() === Constants.TASK_MODE_CREATE && !await task.checkProp(PropInterface.PROP_TYPE_ALLOW_MODIFY, true)) {
            await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_EXIT);
        }
        else {
            await task.ExecuteNestedRS(LastFocusedManager.Instance.getLastFocusedTask());
            if (!this.GetStopExecutionFlag()) {
                await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
            }
            if (!this.GetStopExecutionFlag() && !task.InEndTask && !task.isAborting()) {
                if (intEvtCode !== InternalInterface.MG_ACT_RT_REFRESH_SCREEN) {
                    let currentRow = 0;
                    if (!evt.getRefreshType()) {
                        if (evt.RtViewRefreshUseCurrentRow() && task.IsInteractive) {
                            currentRow = form.getCurrRecPosInForm();
                        }
                        cmd = CommandFactory.CreateInternalRefreshCommand(task.getTaskTag(), intEvtCode, dv.CurrentRecId, currentRow);
                    }
                    else {
                        if (task.IsInteractive)
                            currentRow = form.getCurrRecPosInForm();
                        let refetchRecord = true;
                        if (intEvtCode === InternalInterface.MG_ACT_RT_REFRESH_RECORD && evt.getArgList() !== null && evt.getArgList().getSize() !== 0) {
                            let refetchRecStr;
                            refetchRecStr = await evt.getArgList().getArgValue(0, StorageAttribute.BOOLEAN, 0);
                            refetchRecord = (isNullOrUndefined(refetchRecStr) || refetchRecStr === '1');
                        }
                        if (refetchRecord)
                            cmd = await CommandFactory.CreateRealRefreshCommand(task.getTaskTag(), intEvtCode, currentRow, evt.getArgList(), dv.CurrentRecId);
                    }
                }
                else {
                    let id;
                    if (task.getForm() === null)
                        id = 0;
                    else if (task.getForm().isScreenMode())
                        id = (task.DataView.getCurrRec()).getId();
                    else
                        id = task.DataView.getRecByIdx(task.DataView.getTopRecIdx()).getId();
                    cmd = CommandFactory.CreateScreenRefreshCommand(task.getTaskTag(), id, dv.CurrentRecId);
                }
                if (intEvtCode !== InternalInterface.MG_ACT_RT_REFRESH_SCREEN)
                    task.getTaskCache().clearCache();
                if (cmd !== null)
                    await task.DataviewManager.Execute(cmd);
                this._isInRecordFlushEvent = false;
                if (!this.GetStopExecutionFlag() && !task.InEndTask && !task.isAborting()) {
                    if (intEvtCode === InternalInterface.MG_ACT_RT_REFRESH_VIEW && !form.isScreenMode()) {
                        dv.setTopRecIdx(0);
                        if (form.HasTable()) {
                            form.clearTableColumnSortMark(true);
                            await form.restoreOldDisplayLine(form.recordIdx2DisplayLine(dv.getCurrRecIdx()));
                        }
                    }
                    if (intEvtCode === InternalInterface.MG_ACT_RT_REFRESH_RECORD && task.getMode() !== Constants.TASK_MODE_QUERY && !task.getAfterRetry()) {
                        if (task.getMode() === Constants.TASK_MODE_CREATE) {
                            task.setMode(Constants.TASK_MODE_MODIFY);
                            (task.DataView.getCurrRec()).setOldRec();
                        }
                        await dv.currRecCompute(true);
                    }
                    else if (intEvtCode === InternalInterface.MG_ACT_RT_REFRESH_SCREEN && task.getMode() !== Constants.TASK_MODE_QUERY && !task.getAfterRetry()) {
                        await dv.currRecCompute(true);
                    }
                    this.setIsInViewRefresh(true);
                    await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
                    if (!this.GetStopExecutionFlag() && !task.getPreventControlChange()) {
                        if (!(task === LastFocusedManager.Instance.getLastFocusedTask() || !task.pathContains(LastFocusedManager.Instance.getLastFocusedTask())))
                            await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
                    }
                }
            }
        }
    }
    async gotoCreateMode(task) {
        let cmd;
        await task.enableModes();
        await task.ExecuteNestedRS(LastFocusedManager.Instance.getLastFocusedTask());
        if (!this.GetStopExecutionFlag())
            await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
        if (!this.GetStopExecutionFlag()) {
            cmd = CommandFactory.CreateEventCommand(task.getTaskTag(), InternalInterface.MG_ACT_RTO_CREATE);
            await task.DataviewManager.Execute(cmd);
            if (task.getLevel() === Constants.TASK_LEVEL_TASK)
                await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
            await this.HandleNonParkableControls(task);
            await task.setCreateDeleteActsEnableState();
            task.setOriginalTaskMode(Constants.TASK_MODE_CREATE);
            await this.HandleActEnterRowEditing(task);
        }
    }
    async commonHandlerAfter(evt, rcBefore) {
        let task = evt.getTask();
        let dv;
        let rec = null;
        let ctrl;
        let recModified = false;
        let recModeIsDelete = false;
        let forceSuffix;
        if (this._stopExecution && evt.getInternalCode() !== InternalInterface.MG_ACT_REC_SUFFIX && evt.getInternalCode() !== InternalInterface.MG_ACT_CTRL_SUFFIX)
            return;
        if (evt.getType() === ConstInterface.EVENT_TYPE_INTERNAL) {
            dv = task.DataView;
            rec = dv.getCurrRec();
            ctrl = evt.Control;
            switch (evt.getInternalCode()) {
                case InternalInterface.MG_ACT_REC_PREFIX:
                    if (rcBefore) {
                        if (await task.CheckRefreshSubTasks())
                            await task.doSubformRecPrefixSuffix();
                        await task.handleEventOnSlaveTasks(InternalInterface.MG_ACT_REC_PREFIX);
                        await task.enableModes();
                    }
                    break;
                case InternalInterface.MG_ACT_REC_SUFFIX:
                    task.isFirstRecordCycle(false);
                    if (task.getPreventRecordSuffix()) {
                        task.setPreventRecordSuffix(false);
                        return;
                    }
                    task.DoSubformPrefixSuffix = false;
                    if (this._stopExecution) {
                        if (rec.getMode() === DataModificationTypes.Delete)
                            rec.clearMode();
                        if (task.getLevel() !== Constants.TASK_LEVEL_CONTROL) {
                            let last = this.getStopExecutionCtrl();
                            if (last === null)
                                last = task.getLastParkedCtrl();
                            this.setStopExecution(false);
                            this.setStopExecutionCtrl(null);
                            if (last != null) {
                                task.setLevel(Constants.TASK_LEVEL_RECORD);
                                let inCtrlPrefix = task.InCtrlPrefix;
                                task.InCtrlPrefix = false;
                                await this.handleInternalEventWithMgControlBase(last, InternalInterface.MG_ACT_CTRL_PREFIX);
                                task.InCtrlPrefix = inCtrlPrefix;
                            }
                            this.setStopExecution(true);
                        }
                        return;
                    }
                    else {
                        await this.HandleStopRowEditing(task);
                    }
                    if (rec != null) {
                        recModified = rec.Modified;
                        if (!this._stopExecution && rec.isCauseInvalidation()) {
                            task.getForm().invalidateTable();
                            rec.setCauseInvalidation(false);
                        }
                        recModeIsDelete = (rec.getMode() === DataModificationTypes.Delete);
                        forceSuffix = await task.checkProp(PropInterface.PROP_TYPE_FORCE_SUFFIX, false);
                        if (recModeIsDelete && task.getMode() === Constants.TASK_MODE_DELETE ||
                            !recModeIsDelete && !rec.inDeleteProcess() && (recModified || forceSuffix)) {
                            rec.setMode(DataModificationTypes.Update);
                            await dv.addCurrToModified();
                            dv.setChanged(true);
                            if (recModified && task.getMode() === Constants.TASK_MODE_QUERY) {
                                if (rec.realModified() &&
                                    !Environment.Instance.allowUpdateInQueryMode(task.getCompIdx())) {
                                    await dv.cancelEdit(EventsManager.REAL_ONLY, false);
                                    task.WriteToMessagePanebyMsgId(MsgInterface.RT_STR_UPDATE_IN_QUERY);
                                    await task.getForm().RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
                                }
                            }
                        }
                        if (!this.isForceExitPreRecordUpdate(task) &&
                            (!task.isAborting() || MGDataCollection.Instance.GetTaskByID(task.getTaskTag()) != null)) {
                            if ((!recModeIsDelete || task.getMode() === Constants.TASK_MODE_DELETE))
                                await this.commitRecord(task, evt.reversibleExit());
                        }
                        if (!task.transactionFailed(ConstInterface.TRANS_RECORD_PREFIX) && !task.isAborting()) {
                            if ((rec.getMode() === DataModificationTypes.Delete &&
                                task.getMode() === Constants.TASK_MODE_DELETE) ||
                                rec.getMode() !== DataModificationTypes.Delete)
                                if (await task.evalEndCond(ConstInterface.END_COND_EVAL_AFTER))
                                    await task.endTask(true, false, false);
                        }
                    }
                    break;
                case InternalInterface.MG_ACT_CTRL_PREFIX:
                    if (rcBefore) {
                        if (await ctrl.IsParkable() || ctrl === LastFocusedManager.getLastFocusedControl()) {
                            ctrl.InControl = true;
                            if (ctrl.ShouldRefreshOnControlEnter())
                                await ctrl.getField().updateDisplay();
                            task.setLastParkedCtrl(ctrl);
                            RuntimeContextBase.Instance.CurrentClickedCtrl = null;
                            await ctrl.SetFocus(ctrl, ctrl.getDisplayLine(false), true);
                            this.setStopExecutionCtrl(null);
                            MgControl.ReturnToCtrl = ctrl;
                            task.InCtrlPrefix = false;
                        }
                        else {
                            task.InCtrlPrefix = false;
                            await this.HandleNonParkableControls(ctrl.getForm().getTask());
                        }
                    }
                    break;
                case InternalInterface.MG_ACT_CTRL_SUFFIX:
                    if (!this.GetStopExecutionFlag()) {
                        if (ctrl.isChoiceControl()) {
                            ctrl.SetKeyboardLanguage();
                        }
                        if (ctrl.Type === MgControlType.CTRL_TYPE_BUTTON)
                            task.ActionManager.enable(InternalInterface.MG_ACT_BUTTON, false);
                        if (task.getEnableZoomHandler() || ctrl.useZoomHandler())
                            task.ActionManager.enable(InternalInterface.MG_ACT_ZOOM, false);
                        task.ActionManager.enable(InternalInterface.MG_ACT_RT_EDT_NULL, false);
                    }
                    if (rcBefore) {
                        if (this.GetStopExecutionFlag()) {
                            ctrl.InControl = true;
                            return;
                        }
                        let fld = ctrl.getField();
                        if (fld != null && (ctrl.Type === MgControlType.CTRL_TYPE_TEXT))
                            await fld.updateDisplay();
                        task.setLevel(Constants.TASK_LEVEL_RECORD);
                        ctrl.setInControlSuffix(false);
                        ctrl.IsInteractiveUpdate = false;
                        if (ctrl.PropertyExists(PropInterface.PROP_TYPE_CUSTOM_VALIDATORS))
                            await ctrl.getProp(PropInterface.PROP_TYPE_CUSTOM_VALIDATORS).RefreshDisplay(true);
                        ctrl.ValidateControl = false;
                        LastFocusedManager.setLastFocusedControl(task, null);
                        if (await task.evalEndCond(ConstInterface.END_COND_EVAL_IMMIDIATE))
                            await task.endTask(true, false, false);
                    }
                    break;
                case InternalInterface.MG_ACT_TASK_PREFIX:
                    task.setLevel(Constants.TASK_LEVEL_TASK);
                    break;
                case InternalInterface.MG_ACT_TASK_SUFFIX:
                    task.setLevel(Constants.TASK_LEVEL_NONE);
                    break;
                case InternalInterface.MG_ACT_CYCLE_NEXT_REC:
                    let form = task.getForm();
                    if (!task.isMainProg() && form != null) {
                        await form.moveInView(form.isLineMode()
                            ? Constants.MOVE_UNIT_ROW
                            : Constants.MOVE_UNIT_PAGE, Constants.MOVE_DIRECTION_NEXT);
                        if (task.getExecEndTask())
                            await task.endTask(true, false, false);
                    }
                    break;
                case InternalInterface.MG_ACT_CYCLE_NEXT_DELETE_REC:
                    if (task.DataView.isEmptyDataview())
                        (task).setExecEndTask();
                    else
                        await this.HandleActionDelline(task, dv, rec, true);
                    if (task.getExecEndTask())
                        await task.endTask(true, false, false);
                    break;
                case InternalInterface.MG_ACT_SET_EXTERNAL_VALUE:
                    let refreshDisplay = await evt.getArgList().getArgValue(1, StorageAttribute.BOOLEAN, 0) === '1' ? true : false;
                    await ctrl.validateAndSetValueIfAllowed(evt.getValue(), refreshDisplay);
                    break;
            }
        }
    }
    async handleNonReversibleEvent(task, eventCode) {
        let rtEvt = new RunTimeEvent(task);
        let exitCommand = (eventCode === InternalInterface.MG_ACT_EXIT || eventCode === InternalInterface.MG_ACT_CLOSE);
        rtEvt.setInternal(eventCode);
        if (exitCommand)
            this._isNonReversibleExit = true;
        try {
            rtEvt.setNonReversibleExit();
            rtEvt.setCtrl(task.getLastParkedCtrl());
            rtEvt.setArgList(null);
            await this.handleEvent(rtEvt, false);
        }
        finally {
            if (exitCommand)
                this._isNonReversibleExit = false;
        }
    }
    async handleInternalEventWithTaskAndEventSubtype(task, eventCode, eventSubType) {
        let lastParkedCtrl = task.getLastParkedCtrl();
        let rtEvt;
        if (eventCode < InternalInterface.MG_ACT_TOT_CNT && task != null && !(eventSubType === EventSubType.CancelIsQuit) && !task.ActionManager.isEnabled(eventCode))
            return;
        if (lastParkedCtrl !== null)
            await this.handleInternalEventWithMgControl(lastParkedCtrl, eventCode, eventSubType);
        else {
            if (eventCode === InternalInterface.MG_ACT_RT_QUIT)
                await this.handleInternalEventWithTaskAndEventSubtype(task, InternalInterface.MG_ACT_CANCEL, EventSubType.CancelIsQuit);
            else {
                rtEvt = new RunTimeEvent(task);
                rtEvt.setInternal(eventCode);
                rtEvt.SetEventSubType(eventSubType);
                if (InternalInterface.BuiltinEvent(eventCode))
                    await this.handleEvent(rtEvt, false);
                else
                    await this.commonHandler(rtEvt);
            }
        }
    }
    async handleInternalEventWithMgControl(ctrl, eventCode, eventSubType) {
        let rtEvt;
        if (eventCode === InternalInterface.MG_ACT_RT_QUIT)
            await this.handleInternalEventWithMgControl(ctrl, InternalInterface.MG_ACT_CANCEL, EventSubType.CancelIsQuit);
        else {
            rtEvt = new RunTimeEvent(ctrl);
            rtEvt.setInternal(eventCode);
            rtEvt.SetEventSubType(eventSubType);
            if (eventCode === InternalInterface.MG_ACT_RT_REFRESH_VIEW)
                rtEvt.setIsRealRefresh(false);
            if (InternalInterface.BuiltinEvent(eventCode))
                await this.handleEvent(rtEvt, false);
            else
                await this.commonHandler(rtEvt);
        }
    }
    getCurrField() {
        return this._currField;
    }
    getCurrCtrl() {
        return this._currCtrl;
    }
    getCurrTask() {
        if (this._currField == null)
            return null;
        return this._currField.getTask();
    }
    setStopExecution(stop, clearEventsOnStopExecution) {
        if (arguments.length === 1)
            this.setStopExecution_0(stop);
        else
            this.setStopExecution_1(stop, clearEventsOnStopExecution);
    }
    setStopExecution_0(stop) {
        this.setStopExecution(stop, ClearEventsOnStopExecution.ALL);
    }
    setStopExecution_1(stop, clearEventsOnStopExecution) {
        if (stop)
            if (clearEventsOnStopExecution == ClearEventsOnStopExecution.ALL)
                this._eventsQueue.clear();
            else if (clearEventsOnStopExecution == ClearEventsOnStopExecution.SERVER) {
                let tmpVec = new List();
                let rtEvt;
                let i;
                while (!this._eventsQueue.isEmpty()) {
                    rtEvt = this._eventsQueue.poll();
                    if (rtEvt.isFromServer()) {
                        rtEvt.resetFromServer();
                        tmpVec.push(rtEvt);
                    }
                }
                for (i = 0; i < tmpVec.length; i++)
                    this._eventsQueue.put(tmpVec.get_Item(i));
            }
        this._stopExecution = stop;
    }
    setStopExecutionCtrl(ctrl) {
        if (this._stopExecutionCtrl === null || ctrl === null)
            this._stopExecutionCtrl = ctrl;
    }
    async delCurrRec(task, inForceDel) {
        let dv = task.DataView;
        task.setMode(Constants.TASK_MODE_MODIFY);
        await task.getForm().delCurrRec();
        if (!task.DataView.isEmptyDataview()) {
            if (dv.getCurrRecIdx() >= 0) {
                if ((dv.getCurrRec()).getMode() !== DataModificationTypes.Delete && !inForceDel) {
                    let ctrl = task.getLastParkedCtrl();
                    await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
                    if (!this.GetStopExecutionFlag())
                        if (ctrl === null)
                            await this.HandleNonParkableControls(task);
                        else
                            await this.handleInternalEventWithMgControlBase(ctrl, InternalInterface.MG_ACT_CTRL_PREFIX);
                }
            }
            else if (!task.getInCreateLine())
                await this.handleNonReversibleEvent(task, InternalInterface.MG_ACT_EXIT);
        }
        else {
            await this.HandleNonParkableControls(task);
            await this.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_PREFIX);
        }
    }
    async updateConfirmed(task) {
        let confirm;
        let answer;
        let ans;
        if (task == null)
            return false;
        confirm = await task.checkProp(PropInterface.PROP_TYPE_CONFIRM_UPDATE, false);
        if (!this._isInRecordFlushEvent && confirm && !task.DataSynced) {
            ans = await GUIManager.Instance.confirm(MsgInterface.CRF_STR_CONF_UPD, Styles.MSGBOX_ICON_QUESTION | Styles.MSGBOX_BUTTON_YES_NO_CANCEL);
            if (ans === Styles.MSGBOX_RESULT_YES)
                answer = 'Y';
            else if (ans === Styles.MSGBOX_RESULT_NO)
                answer = 'N';
            else
                answer = 'C';
        }
        else
            answer = 'Y';
        switch (answer) {
            case 'Y':
                break;
            case 'N':
                await task.getForm().cancelEdit(false, false);
                await task.getForm().RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
                task.ConfirmUpdateNo = true;
                return false;
            case 'C':
                this.setStopExecution(true);
                this.setStopExecutionCtrl(LastFocusedManager.getLastFocusedControl());
                await FocusManager.SetFocus(LastFocusedManager.getLastFocusedControl(), -1);
                return false;
            default:
                Logger.Instance.WriteExceptionToLogWithMsg('in ClientManager.Instance.updateConfirmed() illegal confirmation code: ' +
                    confirm);
                break;
        }
        return true;
    }
    setCompMainPrgTab(compMainPrgTable) {
        this._compMainPrgTab = compMainPrgTable;
    }
    setEndOfWork(endOfWork_) {
        this._endOfWork = endOfWork_;
    }
    getEndOfWork() {
        return this._endOfWork;
    }
    getCompMainPrgTab() {
        return this._compMainPrgTab;
    }
    inNonReversibleExit() {
        return this._isNonReversibleExit;
    }
    pushRtEvent(rtEvt) {
        this._rtEvents.push(rtEvt);
    }
    popRtEvent() {
        if (this._rtEvents.count() === 0)
            return null;
        return this._rtEvents.pop();
    }
    storeLastSavedRouteEventOnLocalStorage() {
        let dataStorage = window.localStorage;
        if (this.getLastSavedRouteEvent() != null) {
            let rtEvt = this.getLastSavedRouteEvent();
            dataStorage.setItem(ConstInterface.LAST_ROUTE_EVENT, '1');
            dataStorage.setItem(ConstInterface.LAST_ROUTE_EVENT_SRC_TSK, JSON.stringify(rtEvt.getTask().TaskDefinitionId));
            if (rtEvt.getArgList().getSize() > 0) {
                let str = new StringBuilder();
                rtEvt.getArgList().buildXML(str);
                dataStorage.setItem(ConstInterface.LAST_ROUTE_EVENT_ARG_LIST, str.toString());
            }
        }
    }
    restoreLastSavedRouteEventFromLocalStorage() {
        let dataStorage = window.localStorage;
        let lastRouteEvent = dataStorage.getItem(ConstInterface.LAST_ROUTE_EVENT);
        if (lastRouteEvent === '1') {
            let srcTaskDefinitionStr = dataStorage.getItem(ConstInterface.LAST_ROUTE_EVENT_SRC_TSK);
            let routeArgListStr = dataStorage.getItem(ConstInterface.LAST_ROUTE_EVENT_ARG_LIST);
            let taskDefinitionId = new TaskDefinitionId();
            taskDefinitionId.fromJSON(srcTaskDefinitionStr);
            let srcTask = MGDataCollection.Instance.getCurrMGData().getTaskByTaskDefinitionID(taskDefinitionId);
            let args = null;
            if (routeArgListStr != null) {
                args = new ArgumentsList();
                args.fillList(routeArgListStr, srcTask);
            }
            EventsManager.Instance.AddLastSavedRouteEvent(srcTask, args);
        }
    }
    getLastSavedRouteEvent() {
        return this.lastSavedRouteEvent;
    }
    setLastSavedRouteEvent(rtEvt) {
        if (rtEvt == null)
            this.lastSavedRouteEvent = null;
        else
            this.lastSavedRouteEvent = rtEvt.replicate();
    }
    getLastRtEvent() {
        if (this._rtEvents.count() === 0)
            return null;
        return this._rtEvents.peek();
    }
    setIgnoreUnknownAbort(val) {
        this._ignoreUnknownAbort = val;
    }
    ignoreUnknownAbort() {
        return this._ignoreUnknownAbort;
    }
    async failTaskOverlay(taskId, mgd) {
        let cmd;
        if (taskId !== null) {
            this.setIgnoreUnknownAbort(true);
            try {
                cmd = CommandFactory.CreateNonReversibleExitCommand(taskId, false);
                mgd.CmdsToServer.Add(cmd);
                await RemoteCommandsProcessor.GetInstance().Execute(CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS);
            }
            finally {
                this.setIgnoreUnknownAbort(false);
            }
        }
    }
    isEventScopeTrans() {
        return this._eventScope === EventScope.TRANS;
    }
    setEventScopeTrans() {
        this._eventScope = EventScope.TRANS;
    }
    clearEventScope() {
        this._eventScope = EventScope.NONE;
    }
    GetEventTime() {
        return this._eventsQueue.GetTime();
    }
    getMatchingAction(kbItm) {
        let act = 0;
        if (kbItm == null)
            return act;
        if (act === 0) {
            if (kbItm.getAction() !== 0)
                act = kbItm.getAction();
            else if (kbItm.getKeyCode() === 45 && kbItm.getModifier() === Modifiers.MODIFIER_NONE && kbItm.getStates() === 0)
                act = InternalInterface.MG_ACT_TOGGLE_INSERT;
        }
        return (act);
    }
    getExecStack() {
        return this._execStack.peek();
    }
    pushExecStack(taskId, handlerId, operIdx) {
        this._execStack.peek().push(MGDataCollection.Instance.getTaskIdById(taskId), handlerId, operIdx);
    }
    popExecStack() {
        return this._execStack.peek().pop();
    }
    pushNewExecStacks() {
        this._serverExecStack.push(null);
        this._execStack.push(new ExecutionStack());
    }
    popNewExecStacks() {
        this._serverExecStack.pop();
        this._execStack.pop();
    }
    pushServerExecStack(taskId, handlerId, operIdx) {
        if (this._serverExecStack.peek() === null) {
            this._serverExecStack.pop();
            this._serverExecStack.push(new ExecutionStack());
        }
        this._serverExecStack.peek().push(taskId, handlerId, operIdx);
    }
    clearServerExecStack() {
        if (this._serverExecStack.peek() !== null) {
            this._serverExecStack.peek().clear();
        }
    }
    reverseServerExecStack() {
        if (this._serverExecStack.peek() !== null)
            this._serverExecStack.peek().reverse();
    }
    getNextOperIdx(oper, clearWhenFound) {
        let tmpServerExecStackEntry = null;
        let nextOperIdx = -1;
        if (this._serverExecStack.count() !== 0 && this._serverExecStack.peek() !== null && !this._serverExecStack.peek().empty()) {
            if (this._serverExecStack.peek().size() > this._execStack.peek().size()) {
                let tmpServerExecStack = new ExecutionStack(this._serverExecStack.peek());
                while (tmpServerExecStack.size() > this._execStack.peek().size()) {
                    tmpServerExecStackEntry = tmpServerExecStack.pop();
                }
                if (this._execStack.peek().Equals(tmpServerExecStack) && tmpServerExecStackEntry.TaskId === oper.getTaskTag() && tmpServerExecStackEntry.HandlerId === oper.getHandlerId() && tmpServerExecStackEntry.OperIdx >= oper.getServerId()) {
                    nextOperIdx = tmpServerExecStackEntry.OperIdx;
                    if (clearWhenFound && this._serverExecStack.peek().size() === this._execStack.peek().size() + 1) {
                        this._serverExecStack.peek().clear();
                    }
                }
                else
                    nextOperIdx = EventsManager.MAX_OPER;
            }
            else {
                nextOperIdx = EventsManager.MAX_OPER;
            }
        }
        return nextOperIdx;
    }
    setProcessingTopMostEndTask(inProcessingTopMostEndTask) {
        this._processingTopMostEndTask = inProcessingTopMostEndTask;
    }
    getProcessingTopMostEndTask() {
        return this._processingTopMostEndTask;
    }
    setAllowEvents(AllowEvents) {
        this._allowEvents = AllowEvents;
    }
    setNonInteractiveAllowEvents(AllowEvents) {
        if (AllowEvents)
            this._allowEvents = EventsAllowedType.NON_INTERACTIVE;
        else
            this._allowEvents = EventsAllowedType.NONE;
    }
    getAllowEvents() {
        return this._allowEvents;
    }
    NoEventsAllowed() {
        return this._allowEvents === EventsAllowedType.NONE;
    }
    async handleCtrlModify(ctrl, act) {
        if (ctrl == null || !ctrl.isModifiable())
            return;
        switch (act) {
            case InternalInterface.MG_ACT_RT_EDT_NULL:
            case InternalInterface.MG_ACT_CTRL_MODIFY:
            case InternalInterface.MG_ACT_SELECTION:
                break;
            default:
                return;
        }
        let rtEvt;
        rtEvt = new RunTimeEvent(ctrl);
        rtEvt.setInternal(InternalInterface.MG_ACT_CTRL_MODIFY);
        await this.handleEvent(rtEvt, false);
    }
    ActAllowed(act) {
        let allowed = true;
        switch (this.getAllowEvents()) {
            case EventsAllowedType.ALL:
                allowed = true;
                break;
            case EventsAllowedType.NONE:
                allowed = (act === InternalInterface.MG_ACT_CRELINE);
                break;
            case EventsAllowedType.NON_INTERACTIVE:
                switch (act) {
                    case InternalInterface.MG_ACT_EXIT_SYSTEM:
                    case InternalInterface.MG_ACT_EXIT:
                    case InternalInterface.MG_ACT_CLOSE:
                    case InternalInterface.MG_ACT_CRELINE:
                    case InternalInterface.MG_ACT_CRELINE_ABOVE:
                    case InternalInterface.MG_ACT_ROLLBACK:
                        allowed = true;
                        break;
                    default:
                        allowed = false;
                        break;
                }
                break;
        }
        return allowed;
    }
    async commitRecord(task, isReversibleExit) {
        let dv = task.DataView;
        let rec = dv.getCurrRec();
        let forceSuffix = await task.checkProp(PropInterface.PROP_TYPE_FORCE_SUFFIX, false);
        let isSelectionTable = await task.checkProp(PropInterface.PROP_TYPE_SELECTION, false);
        let isNewRec = rec.isNewRec();
        let savePrevCurrRec = true;
        let recModified = rec.Modified;
        if (!dv.recInModifiedTab(rec.getId()))
            rec.clearFlagsHistory();
        await task.TaskTransactionManager.checkAndCommit(isReversibleExit, ConstInterface.TRANS_RECORD_PREFIX, task.ConfirmUpdateNo);
        if (task.transactionFailed(ConstInterface.TRANS_RECORD_PREFIX)) {
            try {
                task.DataSynced = true;
                this.setStopExecution(false);
                await task.handleEventOnSlaveTasks(InternalInterface.MG_ACT_REC_PREFIX);
            }
            finally {
                this.setStopExecution(true);
            }
        }
        else if (task.isAborting())
            this.setStopExecution(true);
        else {
            if (recModified)
                rec.setLateCompute(true);
            if (!this._stopExecution)
                task.setLevel(Constants.TASK_LEVEL_TASK);
            rec.resetUpdated();
            rec.resetModified();
            if (!recModified && !forceSuffix && (!isSelectionTable || !task.InSelect)) {
                if (isNewRec) {
                    dv.setPrevCurrRec();
                    savePrevCurrRec = false;
                    rec.setNewRec();
                    await task.getForm().cancelEdit(false, false);
                    await this.HandleStopRowEditing(task);
                }
                else if (!rec.getForceSaveOrg())
                    await dv.cancelEdit(EventsManager.REAL_ONLY, true);
                else
                    dv.saveOriginal();
            }
            else
                dv.saveOriginal();
        }
        if (savePrevCurrRec)
            dv.setPrevCurrRec();
    }
    incLocateTimerCB(taskObj) {
        let task = taskObj;
        let rtEvt = new RunTimeEvent(task, true);
        rtEvt.setInternal(InternalInterface.MG_ACT_INCREMENTAL_LOCATE);
        rtEvt.setMainPrgCreator(task);
        rtEvt.setCtrl(task.getLastParkedCtrl());
        this.addToTail(rtEvt);
    }
    static async doSyncForSubformParent(task) {
        let parentTask = task.getParent();
        let parentDv = parentTask.DataView;
        let parentRec = parentDv.getCurrRec();
        if (task.IsSubForm && parentRec != null && parentRec.Modified && !parentDv.isEmptyDataview()) {
            parentRec.setMode(DataModificationTypes.Update);
            await parentDv.addCurrToModified(false);
            ((parentDv.getCurrRec())).Synced = true;
            parentDv.setChanged(true);
            let dataViewExecuteLocalUpdatesCommand = CommandFactory.CreateDataViewCommand(parentTask.getTaskTag(), DataViewCommandType.ExecuteLocalUpdates);
            await parentTask.DataviewManager.Execute(dataViewExecuteLocalUpdatesCommand);
        }
    }
    async ProcessAbortingError(ex) {
        let eMsg;
        let errForConsole;
        let errForMsgBox;
        if (ex instanceof ServerError)
            eMsg = ex.GetMessage();
        else
            eMsg = ((ex.Message !== null) ? ex.Message : ex.GetType());
        errForConsole = StrUtil.getConsoleErorString(eMsg);
        errForMsgBox = StrUtil.getMsgBoxErorString(eMsg);
        if (ex.InnerException !== null)
            Logger.Instance.WriteExceptionToLog(ex.InnerException, errForConsole);
        else {
            if (!(ex instanceof ServerError) || Logger.Instance.LogLevel !== Logger_LogLevels.Basic) {
                Logger.Instance.WriteExceptionToLogWithMsg(errForConsole);
            }
        }
        let sendUnloadToServer = RuntimeContextBase.Instance.ContextID !== RuntimeContextBase.noContextId;
        MGDataCollection.Instance.currMgdID = 0;
        let messageString = LanguageData.Instance.getConstMessage(MsgInterface.BRKTAB_STR_ERROR);
        let style = 16;
        if (ex.Message.startsWith('<HTML')) {
            this.processHTMLContent(ex.Message);
        }
        else {
            if (this.ShowErrorInMessageBox(ex)) {
                await GUIManager.Instance.MessageBox(messageString, errForMsgBox, style);
            }
            if (ex instanceof ServerError) {
                if (ex.GetCode() == ServerError.ERR_WEBCLIENT_PROGRAM_RELOADED ||
                    ex.GetCode() == ServerError.ERR_EXECUTED_PROGRAM_CHANGED) {
                    Commands.addGlobalCommand(CommandType.RELOAD_PAGE);
                    Commands.beginInvoke();
                }
            }
        }
        if (ex instanceof ServerError) {
            if (ex.GetCode() === ServerError.ERR_CTX_NOT_FOUND ||
                ex.GetCode() === ServerError.ERR_WEBCLIENT_PROGRAM_RELOADED ||
                ex.GetCode() === ServerError.ERR_EXECUTED_PROGRAM_CHANGED) {
                sendUnloadToServer = false;
            }
        }
        if (sendUnloadToServer) {
            try {
                let cmd = CommandFactory.CreateUnloadCommand();
                MGDataCollection.Instance.getMGData(0).CmdsToServer.Add(cmd);
                await CommandsProcessorManager.GetCommandsProcessor().Execute(CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS);
            }
            catch (ex3) {
                if (ex3 instanceof ApplicationException) {
                    Logger.Instance.WriteExceptionToLog(ex3);
                }
                else
                    throw ex3;
            }
        }
        GUIManager.Instance.abort();
    }
    processHTMLContent(htmlContent) {
        Commands.addShowHTMLError(CommandType.SHOW_HTML_ERROR, htmlContent);
        Commands.beginInvoke();
    }
    ShowErrorInMessageBox(ex) {
        if (!(ex instanceof ServerError))
            return true;
        if ((ex.GetCode() == ServerError.ERR_EXECUTED_PROGRAM_CHANGED ||
            ex.GetCode() == ServerError.ERR_WEBCLIENT_PROGRAM_RELOADED) &&
            RuntimeContextBase.Instance.ContextID === RuntimeContextBase.noContextId)
            return false;
        return true;
    }
    getIsInViewRefresh() {
        return this.isInViewRefresh;
    }
    setIsInViewRefresh(inViewRefresh) {
        this.isInViewRefresh = inViewRefresh;
    }
}
EventsManager.REAL_ONLY = true;
EventsManager.MAX_OPER = 9999;
EventsManager.userGrantedForContextActivity = true;
EventsManager.clientInactivityTimerON = false;
EventsManager.clientContextInactivityTimerAborted = false;
EventsManager.Instance = new EventsManager();
EventsManager.clientBusyTimeStart = 0;
//# sourceMappingURL=data:application/json;base64,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