import { NString, NNumber, NumberStyles } from "@magic-xpa/mscorelib";
import { StorageAttribute, Logger, Misc, Base64, XmlParser, XMLConstants } from "@magic-xpa/utils";
import { ExpVal, VectorType, RecordUtils, NUM_TYPE } from "@magic-xpa/gui";
import { ExpressionEvaluator } from "./ExpressionEvaluator";
import { CommandFactory } from "../commands/ClientToServer/CommandFactory";
import { ConstInterface } from "../ConstInterface";
import { Environment } from "../env/Environment";
import { AccessHelper } from "../AccessHelper";
export class Expression {
    constructor() {
        this._cmdToServer = null;
        this._cmdsToServer = null;
        this._computeBy = '\0';
        this._expBytes = null;
        this._id = -1;
        this._prevResType = StorageAttribute.NONE;
        this._resultValue = null;
        this._task = null;
        this._type = StorageAttribute.NONE;
        this._clientStorageAttribute = StorageAttribute.NONE;
    }
    GetStorageAttribute() {
        return this._clientStorageAttribute;
    }
    async evaluateWithResultTypeAndLength(resType, length) {
        let retVal;
        let retExpressionStorage = StorageAttribute.NONE;
        if (this.computedByClient()) {
            let expVal = await ExpressionEvaluator.eval(this._expBytes, resType, this._task);
            if (expVal.IsNull)
                retVal = null;
            else if (resType === StorageAttribute.BLOB_VECTOR && expVal.Attr === StorageAttribute.BLOB) {
                if (VectorType.validateBlobContents(expVal.StrVal))
                    retVal = expVal.ToMgVal();
                else
                    retVal = null;
            }
            else if (expVal.Attr === StorageAttribute.BLOB_VECTOR && resType !== StorageAttribute.BLOB_VECTOR && resType !== StorageAttribute.BLOB)
                retVal = null;
            else
                retVal = expVal.ToMgVal();
            if (!expVal.IsNull)
                retExpressionStorage = expVal.Attr;
        }
        else {
            let rtEvt = AccessHelper.eventsManager.getLastRtEvent();
            let mprgCreator = null;
            if (rtEvt !== null)
                mprgCreator = rtEvt.getMainPrgCreator();
            if (resType !== this._prevResType)
                this._cmdToServer = CommandFactory.CreateEvaluateCommand(this._task.getTaskTag(), resType, this._id, length, mprgCreator);
            await AccessHelper.mgDataTable.execRequestWithSubformRecordCycle(this._cmdsToServer, this._cmdToServer, this);
            if (resType !== StorageAttribute.BLOB && resType !== StorageAttribute.BLOB_VECTOR)
                retVal = this._resultValue;
            else if (this._resultValue !== null && this._resultValue === " ")
                retVal = "";
            else
                retVal = RecordUtils.byteStreamToString(this._resultValue);
        }
        this._prevResType = resType;
        if (resType === StorageAttribute.NONE)
            this._clientStorageAttribute = retExpressionStorage;
        return retVal;
    }
    async DiscardCndRangeResult() {
        return await ExpressionEvaluator.DiscardCndRangeResult(this._expBytes, this._task);
    }
    async evaluateWithResType(resType) {
        let isNull = false;
        let expVal;
        if (this.computedByClient())
            expVal = await ExpressionEvaluator.eval(this._expBytes, resType, this._task);
        else {
            let rtEvt = AccessHelper.eventsManager.getLastRtEvent();
            let mprgCreator = null;
            if (rtEvt !== null)
                mprgCreator = rtEvt.getMainPrgCreator();
            if (resType !== this._prevResType)
                this._cmdToServer = CommandFactory.CreateEvaluateCommand(this._task.getTaskTag(), resType, this._id, 0, mprgCreator);
            await AccessHelper.mgDataTable.execRequestWithSubformRecordCycle(this._cmdsToServer, this._cmdToServer, this);
            let retVal;
            if (resType !== StorageAttribute.BLOB && resType !== StorageAttribute.BLOB_VECTOR)
                retVal = this._resultValue;
            else if (this._resultValue !== null && this._resultValue === " ")
                retVal = "";
            else
                retVal = RecordUtils.byteStreamToString(this._resultValue);
            if (retVal === null)
                isNull = true;
            if (resType === StorageAttribute.NONE)
                resType = this._type;
            expVal = new ExpVal(resType, isNull, retVal);
        }
        this._prevResType = resType;
        return expVal;
    }
    async evaluateWithResLength(length) {
        let resType = StorageAttribute.NONE;
        let val;
        if (this.computedByClient()) {
            let expVal = await ExpressionEvaluator.eval(this._expBytes, resType, this._task);
            if (expVal.IsNull)
                val = null;
            else
                val = expVal.ToMgVal();
            resType = expVal.Attr;
        }
        else {
            val = await this.evaluateWithResultTypeAndLength(resType, length);
            resType = this._type;
        }
        return new Expression_ReturnValue(val, resType);
    }
    fillData(taskRef, parser) {
        let tokensVector;
        let endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_TERM, parser.getCurrIndex());
        if (this._task == null)
            this._task = taskRef;
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            let tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(ConstInterface.MG_TAG_EXP) + ConstInterface.MG_TAG_EXP.length);
            tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            this.initElements(tokensVector);
            parser.setCurrIndex(endContext + XMLConstants.TAG_TERM.length);
        }
        else
            Logger.Instance.WriteExceptionToLogWithMsg("in Command.FillData() out of string bounds");
    }
    initElements(tokensVector) {
        let expStr;
        let attribute, valueStr;
        for (let j = 0; j < tokensVector.length; j += 2) {
            attribute = (tokensVector.get_Item(j));
            valueStr = (tokensVector.get_Item(j + 1));
            switch (attribute) {
                case XMLConstants.MG_ATTR_VALUE:
                    if (Environment.Instance.GetDebugLevel() > 1) {
                        expStr = valueStr;
                        this.buildByteArray(expStr);
                    }
                    else
                        this._expBytes = Misc.ToSByteArray(Base64.decodeToByte(valueStr));
                    break;
                case XMLConstants.MG_ATTR_ID:
                    this._id = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_COMPUTE_BY:
                    this._computeBy = valueStr[0];
                    break;
                default:
                    Logger.Instance.WriteExceptionToLogWithMsg("There is no such tag in Expression.initElements class. Insert case to Expression.initElements for " + attribute);
                    break;
            }
        }
        if (!this.computedByClient())
            this._cmdsToServer = this._task.getMGData().CmdsToServer;
    }
    buildByteArray(expStr) {
        if (expStr === null || expStr.length === 0 || expStr.length % 2 > 0) {
            Logger.Instance.WriteExceptionToLogWithMsg("in Expression.buildByteArray() expStr cannot be changed " + expStr);
            return;
        }
        this._expBytes = new Int8Array(expStr.length / 2);
        for (let i = 0; i < expStr.length; i = i + 2) {
            let twoHexDigits = expStr.substr(i, 2);
            this._expBytes[i / 2] = NUM_TYPE.toSByte(NNumber.Parse(twoHexDigits, NumberStyles.HexNumber));
        }
    }
    getId() {
        return this._id;
    }
    computedByClient() {
        return this._computeBy !== 'S';
    }
    computedByClientOnly() {
        return this._computeBy === 'C';
    }
    computedByServerOnly() {
        return this._computeBy === 'S';
    }
    getTask() {
        return this._task;
    }
    SetResultValue(result, type_) {
        this._resultValue = result;
        this._type = type_;
    }
    toString() {
        return NString.Format("{{Expression #{0} of {1}}}", this._id, this._task);
    }
}
export class Expression_ReturnValue {
    constructor(mgVal_, type_) {
        this.mgVal = null;
        this.type = StorageAttribute.NONE;
        this.mgVal = mgVal_;
        this.type = type_;
    }
}
//# sourceMappingURL=data:application/json;base64,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