import { BlobType, Commands, DataModificationTypes, DisplayConvertor, ExpressionInterface, ExpVal, GuiExpressionEvaluator, NUM_TYPE, PIC, PropInterface, VectorType } from '@magic-xpa/gui';
import { Constants, InternalInterface, Logger, Misc, MsgInterface, PICInterface, StorageAttribute, StorageAttributeCheck, StrUtil, UtilDateJpn, UtilStrByteMode } from '@magic-xpa/utils';
import { ApplicationException, Array_Enumerator, Debug, Encoding, Exception, List, NChar, NNumber, NString, RefParam, Stack, StringBuilder, isNullOrUndefined, Thread } from '@magic-xpa/mscorelib';
import { ExpressionLocalJpn } from './ExpressionLocalJpn';
import { ExpressionDict } from './ExpressionDict';
import { RunTimeEvent } from '../event/RunTimeEvent';
import { EventHandlerPosition } from '../event/EventHandlerPosition';
import { ArgumentsList } from '../rt/ArgumentsList';
import { CommandFactory } from '../commands/ClientToServer/CommandFactory';
import { Task_SubformExecModeEnum } from '../tasks/TaskBase';
import { GlobalParams } from './GlobalParamsTable';
import { UserDetails } from '../security/UserDetails';
import { Operation } from '../rt/Operation';
import { ConstInterface } from '../ConstInterface';
import { LanguageData } from '../env/LanguageData';
import { EnvParamsTable } from '../env/EnvVariablesTable';
import { Environment } from '../env/Environment';
import { LastFocusedManager } from '../rt/LastFocusedManager';
import { ServerConfig } from '../ServerConfig';
import { AccessHelper } from '../AccessHelper';
import { CookieService } from '../util/cookie.service';
export class ExpressionEvaluator extends GuiExpressionEvaluator {
    constructor() {
        super();
        this._charsToTrim = [' ', '\0'];
        this._expressionLocalJpn = null;
        this._expressionLocalJpn = (UtilStrByteMode.isLocaleDefLangJPN() ? new ExpressionLocalJpn(this) : null);
    }
    getExpBasicValue(expStrTracker, opCode) {
        let Val = new ExpVal();
        let len;
        let parent, vee;
        let field;
        let fldVal;
        switch (opCode) {
            case ExpressionInterface.EXP_OP_A:
            case ExpressionInterface.EXP_OP_H:
                Val.Attr = StorageAttribute.UNICODE;
                len = expStrTracker.get4ByteNumber();
                Val.StrVal = expStrTracker.getString(len, true, true);
                len = expStrTracker.get4ByteNumber();
                expStrTracker.getString(len, true, false);
                break;
            case ExpressionInterface.EXP_OP_EXT_A:
                Val.Attr = StorageAttribute.ALPHA;
                len = expStrTracker.get4ByteNumber();
                expStrTracker.getString(len, true, true);
                len = expStrTracker.get4ByteNumber();
                Val.StrVal = expStrTracker.getString(len, true, false);
                break;
            case ExpressionInterface.EXP_OP_N:
            case ExpressionInterface.EXP_OP_T:
            case ExpressionInterface.EXP_OP_D:
            case ExpressionInterface.EXP_OP_E:
                if (opCode === ExpressionInterface.EXP_OP_D)
                    Val.Attr = StorageAttribute.DATE;
                else if (opCode === ExpressionInterface.EXP_OP_T)
                    Val.Attr = StorageAttribute.TIME;
                else
                    Val.Attr = StorageAttribute.NUMERIC;
                len = expStrTracker.get2ByteNumber();
                Val.MgNumVal = expStrTracker.getMagicNumber(len, true);
                break;
            case ExpressionInterface.EXP_OP_L:
                Val.Attr = StorageAttribute.BOOLEAN;
                Val.BoolVal = (expStrTracker.get2ByteNumber() === 1);
                break;
            case ExpressionInterface.EXP_OP_V:
                parent = expStrTracker.getVarIdx();
                vee = expStrTracker.get4ByteNumber();
                field = this.ExpTask.getFieldDef(parent, vee - 1);
                Val.Attr = field.getType();
                let nullAllowed = (field.getType() === StorageAttribute.BLOB_VECTOR)
                    ? true
                    : field.NullAllowed;
                Val.IsNull = field.isNull() && nullAllowed;
                Val.OriginalNull = Val.IsNull;
                fldVal = field.getValue(true);
                if (fldVal != null && Val.IsNull &&
                    field.getTask().getNullArithmetic() === Constants.NULL_ARITH_USE_DEF)
                    Val.IsNull = false;
                switch (Val.Attr) {
                    case StorageAttribute.ALPHA:
                    case StorageAttribute.UNICODE:
                        Val.StrVal = fldVal;
                        break;
                    case StorageAttribute.BLOB_VECTOR:
                        Val.VectorField = field;
                    case StorageAttribute.BLOB:
                        Val.StrVal = fldVal;
                        if (Val.StrVal == null) {
                            Val.IsNull = true;
                            Val.IncludeBlobPrefix = false;
                        }
                        else
                            Val.IncludeBlobPrefix = true;
                        break;
                    case StorageAttribute.NUMERIC:
                    case StorageAttribute.DATE:
                    case StorageAttribute.TIME:
                        Val.MgNumVal = fldVal != null ? new NUM_TYPE(fldVal) : null;
                        break;
                    case StorageAttribute.BOOLEAN:
                        if (fldVal != null)
                            Val.BoolVal = DisplayConvertor.toBoolean(fldVal);
                        break;
                }
                break;
            case ExpressionInterface.EXP_OP_FORM:
                parent = expStrTracker.getVarIdx();
                let formDisplayIndexInTask = expStrTracker.get4ByteNumber();
                formDisplayIndexInTask = this.ExpTask.GetRealMainDisplayIndexOnDepth(formDisplayIndexInTask);
                this.ConstructMagicNum(Val, formDisplayIndexInTask, StorageAttribute.NUMERIC);
                break;
            case ExpressionInterface.EXP_OP_VAR:
                parent = expStrTracker.getVarIdx();
                vee = expStrTracker.get4ByteNumber();
                let itm = this.ExpTask.ctl_itm_4_parent_vee(parent, vee);
                this.ConstructMagicNum(Val, itm, StorageAttribute.NUMERIC);
                break;
            case ExpressionInterface.EXP_OP_RIGHT_LITERAL:
                len = expStrTracker.get2ByteNumber();
                Val.MgNumVal = expStrTracker.getMagicNumber(len, true);
                len = expStrTracker.get2ByteNumber();
                Val.Attr = StorageAttribute.NUMERIC;
                expStrTracker.getString(len, true, false);
                break;
        }
        return Val;
    }
    static isBasicItem(opCode) {
        return (opCode <= ExpressionInterface.EXP_OP_L ||
            opCode === ExpressionInterface.EXP_OP_EXT_A ||
            opCode === ExpressionInterface.EXP_OP_RIGHT_LITERAL ||
            opCode === ExpressionInterface.EXP_OP_E ||
            opCode === ExpressionInterface.EXP_OP_FORM);
    }
    static isVarArgList(expDesc) {
        if (expDesc.ArgCount_ < 0 || expDesc.ArgAttr_.length > expDesc.ArgCount_)
            return true;
        return false;
    }
    async execOperation(opCode, expStrTracker, valStack, addedOpers, expectedType) {
        let val1;
        let val2;
        let val3;
        let val4;
        let val5;
        let val6;
        let val7;
        let val8;
        let dynOper = new DynamicOperation();
        let whole;
        let dec;
        let resVal = new ExpVal();
        let Exp_params;
        let addResult = true;
        let nArgs;
        let specialAnsiExpression = Environment.Instance.getSpecialAnsiExpression();
        let ofs, len, LenMax, j = 0;
        let pic;
        let expDesc = ExpressionDict.expDesc[opCode];
        if (ExpressionEvaluator.isVarArgList(expDesc)) {
            nArgs = expStrTracker.get1ByteNumber();
            for (j = 0; j < nArgs; j++) {
                try {
                    await this.execExpression(expStrTracker, valStack, StorageAttribute.NONE);
                }
                catch (exception) {
                    if (exception instanceof Exception) {
                        break;
                    }
                    else
                        throw exception;
                }
            }
            if (ExpressionEvaluator.isVarArgList(expDesc) && j === nArgs) {
                let temp_object = nArgs;
                valStack.push(temp_object);
            }
            else
                Logger.Instance.WriteExceptionToLogWithMsg('ExpressionEvaluator.execOperation() there is problem with arguments of ' +
                    opCode + '(see ExpressionDict for name)');
        }
        switch (opCode) {
            case ExpressionInterface.EXP_OP_ADD:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.add(val1.MgNumVal, val2.MgNumVal);
                resVal.Attr = StorageAttribute.NUMERIC;
                resVal.IsNull = (resVal.MgNumVal == null);
                break;
            case ExpressionInterface.EXP_OP_SUB:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.sub(val1.MgNumVal, val2.MgNumVal);
                resVal.Attr = StorageAttribute.NUMERIC;
                resVal.IsNull = (resVal.MgNumVal == null);
                break;
            case ExpressionInterface.EXP_OP_MUL:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.mul(val1.MgNumVal, val2.MgNumVal);
                resVal.Attr = StorageAttribute.NUMERIC;
                resVal.IsNull = (resVal.MgNumVal == null);
                break;
            case ExpressionInterface.EXP_OP_DIV:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.div(val1.MgNumVal, val2.MgNumVal);
                resVal.Attr = StorageAttribute.NUMERIC;
                resVal.IsNull = (resVal.MgNumVal == null);
                break;
            case ExpressionInterface.EXP_OP_MOD:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.mod(val1.MgNumVal, val2.MgNumVal);
                resVal.Attr = StorageAttribute.NUMERIC;
                resVal.IsNull = (resVal.MgNumVal == null);
                break;
            case ExpressionInterface.EXP_OP_NEG:
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.NUMERIC;
                if (val1.MgNumVal == null) {
                    super.SetNULL(resVal, StorageAttribute.NUMERIC);
                    break;
                }
                val1.MgNumVal.num_neg();
                resVal.MgNumVal = new NUM_TYPE(val1.MgNumVal);
                break;
            case ExpressionInterface.EXP_OP_FIX:
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.NUMERIC;
                if (val1.MgNumVal == null || val2.MgNumVal == null || val3.MgNumVal == null) {
                    super.SetNULL(resVal, StorageAttribute.NUMERIC);
                    break;
                }
                resVal.MgNumVal = new NUM_TYPE(val1.MgNumVal);
                whole = val2.MgNumVal.NUM_2_LONG();
                resVal.MgNumVal.num_fix(whole);
                dec = val3.MgNumVal.NUM_2_LONG();
                resVal.MgNumVal.num_trunc(dec);
                break;
            case ExpressionInterface.EXP_OP_ROUND:
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.NUMERIC;
                if (val1.MgNumVal == null || val2.MgNumVal == null || val3.MgNumVal == null) {
                    super.SetNULL(resVal, StorageAttribute.NUMERIC);
                    break;
                }
                resVal.MgNumVal = new NUM_TYPE(val1.MgNumVal);
                whole = val2.MgNumVal.NUM_2_LONG();
                resVal.MgNumVal.num_fix(whole);
                dec = val3.MgNumVal.NUM_2_LONG();
                resVal.MgNumVal.round(dec);
                break;
            case ExpressionInterface.EXP_OP_EQ:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.BOOLEAN;
                try {
                    resVal.BoolVal = (ExpressionEvaluator.val_cmp_any(val1, val2) === 0);
                }
                catch (nullValueException) {
                    if (nullValueException instanceof NullValueException) {
                        resVal.BoolVal = false;
                    }
                    else
                        throw nullValueException;
                }
                expStrTracker.resetNullResult();
                break;
            case ExpressionInterface.EXP_OP_NE:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.BOOLEAN;
                try {
                    resVal.BoolVal = (ExpressionEvaluator.val_cmp_any(val1, val2) !== 0);
                }
                catch (nullValueException) {
                    if (nullValueException instanceof NullValueException) {
                        super.SetNULL(resVal, StorageAttribute.BOOLEAN);
                    }
                    else
                        throw nullValueException;
                }
                break;
            case ExpressionInterface.EXP_OP_LE:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.BOOLEAN;
                try {
                    resVal.BoolVal = (ExpressionEvaluator.val_cmp_any(val1, val2) <= 0);
                }
                catch (nullValueException) {
                    if (nullValueException instanceof NullValueException) {
                        super.SetNULL(resVal, StorageAttribute.BOOLEAN);
                    }
                    else
                        throw nullValueException;
                }
                break;
            case ExpressionInterface.EXP_OP_LT:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.BOOLEAN;
                try {
                    resVal.BoolVal = (ExpressionEvaluator.val_cmp_any(val1, val2) < 0);
                }
                catch (nullValueException) {
                    if (nullValueException instanceof NullValueException) {
                        super.SetNULL(resVal, StorageAttribute.BOOLEAN);
                    }
                    else
                        throw nullValueException;
                }
                break;
            case ExpressionInterface.EXP_OP_GE:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.BOOLEAN;
                try {
                    resVal.BoolVal = (ExpressionEvaluator.val_cmp_any(val1, val2) >= 0);
                }
                catch (nullValueException) {
                    if (nullValueException instanceof NullValueException) {
                        super.SetNULL(resVal, StorageAttribute.BOOLEAN);
                    }
                    else
                        throw nullValueException;
                }
                break;
            case ExpressionInterface.EXP_OP_GT:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.BOOLEAN;
                try {
                    resVal.BoolVal = (ExpressionEvaluator.val_cmp_any(val1, val2) > 0);
                }
                catch (nullValueException) {
                    if (nullValueException instanceof NullValueException) {
                        super.SetNULL(resVal, StorageAttribute.BOOLEAN);
                    }
                    else
                        throw nullValueException;
                }
                break;
            case ExpressionInterface.EXP_OP_NOT:
                resVal = valStack.pop();
                resVal.BoolVal = !resVal.BoolVal;
                break;
            case ExpressionInterface.EXP_OP_OR:
                resVal = valStack.pop();
                dynOper = new DynamicOperation();
                dynOper.argCount_ = 1;
                if (resVal.BoolVal) {
                    dynOper.opCode_ = ExpressionInterface.EXP_OP_IGNORE;
                    dynOper.argCount_ = 1;
                    addedOpers.push(dynOper);
                }
                else {
                    dynOper.opCode_ = ExpressionInterface.EXP_OP_EVALX;
                    dynOper.argCount_ = 0;
                    addedOpers.push(dynOper);
                    addResult = false;
                }
                break;
            case ExpressionInterface.EXP_OP_AND:
                resVal = valStack.pop();
                dynOper = new DynamicOperation();
                if (!resVal.BoolVal) {
                    dynOper.opCode_ = ExpressionInterface.EXP_OP_IGNORE;
                    dynOper.argCount_ = 1;
                    addedOpers.push(dynOper);
                }
                else {
                    dynOper.opCode_ = ExpressionInterface.EXP_OP_EVALX;
                    dynOper.argCount_ = 0;
                    addedOpers.push(dynOper);
                    addResult = false;
                }
                break;
            case ExpressionInterface.EXP_OP_IF:
                val1 = valStack.pop();
                if (val1.BoolVal) {
                    dynOper = new DynamicOperation();
                    dynOper.opCode_ = ExpressionInterface.EXP_OP_EVALX;
                    dynOper.argCount_ = 0;
                    addedOpers.push(dynOper);
                    dynOper = new DynamicOperation();
                    {
                        dynOper.opCode_ = ExpressionInterface.EXP_OP_IGNORE;
                        dynOper.argCount_ = 1;
                    }
                    addedOpers.push(dynOper);
                }
                else {
                    dynOper = new DynamicOperation();
                    {
                        dynOper.opCode_ = ExpressionInterface.EXP_OP_IGNORE;
                        dynOper.argCount_ = 1;
                    }
                    addedOpers.push(dynOper);
                    dynOper = new DynamicOperation();
                    {
                        dynOper.opCode_ = ExpressionInterface.EXP_OP_EVALX;
                        dynOper.argCount_ = 0;
                    }
                    addedOpers.push(dynOper);
                }
                addResult = false;
                expStrTracker.resetNullResult();
                break;
            case ExpressionInterface.EXP_OP_LEN:
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.NUMERIC;
                if (val1.StrVal == null) {
                    super.SetNULL(resVal, StorageAttribute.NUMERIC);
                    break;
                }
                resVal.MgNumVal = new NUM_TYPE();
                if (specialAnsiExpression || val1.Attr !== StorageAttribute.UNICODE)
                    resVal.MgNumVal.NUM_4_LONG(UtilStrByteMode.lenB(val1.StrVal));
                else
                    resVal.MgNumVal.NUM_4_LONG(val1.StrVal.length);
                break;
            case ExpressionInterface.EXP_OP_CON:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.UNICODE;
                if (val1.Attr === StorageAttribute.UNICODE && val1.StrVal == null ||
                    val2.Attr === StorageAttribute.UNICODE && val2.StrVal == null)
                    super.SetNULL(resVal, StorageAttribute.UNICODE);
                else
                    resVal.StrVal = (!NString.IsNullOrEmpty(val1.StrVal) ? val1.StrVal : '') + (!NString.IsNullOrEmpty(val2.StrVal) ? val2.StrVal : '');
                break;
            case ExpressionInterface.EXP_OP_MID:
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = val1.Attr;
                if (val2.MgNumVal == null || val3.MgNumVal == null || val1.StrVal == null) {
                    super.SetNULL(resVal, resVal.Attr);
                    break;
                }
                ofs = val2.MgNumVal.NUM_2_LONG();
                ofs = (ofs > 1
                    ? ofs - 1
                    : 0);
                len = val3.MgNumVal.NUM_2_LONG();
                if (specialAnsiExpression || val1.Attr !== StorageAttribute.UNICODE) {
                    resVal.Attr = StorageAttribute.ALPHA;
                    resVal.StrVal = UtilStrByteMode.midB(val1.StrVal, ofs, len);
                }
                else {
                    LenMax = val1.StrVal.length - ofs;
                    if (LenMax < len)
                        len = LenMax;
                    if (len < 0)
                        len = 0;
                    resVal.StrVal = val1.StrVal.substr(ofs, len);
                }
                if (resVal.StrVal == null)
                    resVal.StrVal = '';
                break;
            case ExpressionInterface.EXP_OP_LEFT:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = val1.Attr;
                if (val2.MgNumVal == null || val1.StrVal == null) {
                    super.SetNULL(resVal, val1.Attr);
                    break;
                }
                len = val2.MgNumVal.NUM_2_LONG();
                if (specialAnsiExpression || val1.Attr !== StorageAttribute.UNICODE) {
                    resVal.Attr = StorageAttribute.ALPHA;
                    resVal.StrVal = UtilStrByteMode.leftB(val1.StrVal, len);
                    if (resVal.StrVal == null)
                        resVal.StrVal = '';
                }
                else {
                    if (len > val1.StrVal.length)
                        len = val1.StrVal.length;
                    if (len < 0)
                        len = 0;
                    resVal.StrVal = val1.StrVal.substr(0, len);
                }
                break;
            case ExpressionInterface.EXP_OP_RIGHT:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = val1.Attr;
                if (val2.MgNumVal == null || val1.StrVal == null) {
                    super.SetNULL(resVal, val1.Attr);
                    break;
                }
                len = val2.MgNumVal.NUM_2_LONG();
                if (specialAnsiExpression || val1.Attr !== StorageAttribute.UNICODE) {
                    resVal.Attr = StorageAttribute.ALPHA;
                    resVal.StrVal = UtilStrByteMode.rightB(val1.StrVal, len);
                    if (resVal.StrVal == null)
                        resVal.StrVal = '';
                }
                else {
                    if (len > val1.StrVal.length)
                        len = val1.StrVal.length;
                    if (len < 0)
                        len = 0;
                    ofs = val1.StrVal.length - len;
                    resVal.StrVal = val1.StrVal.substr(ofs);
                }
                break;
            case ExpressionInterface.EXP_OP_FILL:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.UNICODE;
                if (val2.MgNumVal == null || val1.StrVal == null) {
                    super.SetNULL(resVal, StorageAttribute.UNICODE);
                    break;
                }
                len = val1.StrVal.length;
                j = val2.MgNumVal.NUM_2_LONG();
                if (j < 0)
                    j = 0;
                LenMax = len * j;
                if (LenMax > 0x7FFF)
                    LenMax = Math.floor(0x7FFF / len) * len;
                if (LenMax > 0) {
                    if (len <= 0)
                        resVal.StrVal = '';
                    else if (LenMax === 1)
                        resVal.StrVal = val1.StrVal;
                    else {
                        let tmpBuffer = new StringBuilder(LenMax);
                        for (; LenMax > 0; LenMax -= len)
                            tmpBuffer.Append(val1.StrVal);
                        resVal.StrVal = tmpBuffer.ToString();
                    }
                }
                if (resVal.StrVal == null)
                    resVal.StrVal = '';
                break;
            case ExpressionInterface.EXP_OP_INSTR:
                val2 = valStack.pop();
                val1 = valStack.pop();
                ofs = 0;
                resVal.Attr = StorageAttribute.NUMERIC;
                if (val1.StrVal == null || val2.StrVal == null) {
                    super.SetNULL(resVal, StorageAttribute.NUMERIC);
                    break;
                }
                resVal.MgNumVal = new NUM_TYPE();
                if (specialAnsiExpression ||
                    !(val1.Attr === StorageAttribute.UNICODE ||
                        val2.Attr === StorageAttribute.UNICODE)) {
                    ofs = UtilStrByteMode.instrB(val1.StrVal, val2.StrVal);
                }
                else {
                    if (val2.StrVal.length === 0) {
                        resVal.MgNumVal.NUM_4_LONG(ofs);
                        break;
                    }
                    ofs = val1.StrVal.indexOf(val2.StrVal);
                    if (ofs < 0)
                        ofs = 0;
                    else
                        ofs++;
                }
                resVal.MgNumVal.NUM_4_LONG(ofs);
                break;
            case ExpressionInterface.EXP_OP_TRIM:
            case ExpressionInterface.EXP_OP_LTRIM:
            case ExpressionInterface.EXP_OP_RTRIM:
                val1 = valStack.pop();
                resVal.Attr = val1.Attr === StorageAttribute.ALPHA
                    ? StorageAttribute.ALPHA
                    : StorageAttribute.UNICODE;
                if (val1.StrVal == null) {
                    super.SetNULL(resVal, resVal.Attr);
                    break;
                }
                switch (opCode) {
                    case ExpressionInterface.EXP_OP_TRIM:
                        val1.StrVal = ExpressionEvaluator.trimStr(val1.StrVal, 'B');
                        break;
                    case ExpressionInterface.EXP_OP_LTRIM:
                        val1.StrVal = ExpressionEvaluator.trimStr(val1.StrVal, 'L');
                        break;
                    case ExpressionInterface.EXP_OP_RTRIM:
                        val1.StrVal = ExpressionEvaluator.trimStr(val1.StrVal, 'R');
                        break;
                }
                resVal.StrVal = val1.StrVal;
                break;
            case ExpressionInterface.EXP_OP_STR:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.UNICODE;
                if (val2.StrVal == null || val1.MgNumVal == null) {
                    super.SetNULL(resVal, StorageAttribute.UNICODE);
                    break;
                }
                pic = new PIC(ExpressionEvaluator.set_a_pic(val2.StrVal), StorageAttribute.NUMERIC, this.ExpTask.getCompIdx());
                resVal.StrVal = val1.MgNumVal.to_a(pic);
                break;
            case ExpressionInterface.EXP_OP_VAL:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.NUMERIC;
                if (val2.StrVal == null || val1.StrVal == null) {
                    super.SetNULL(resVal, StorageAttribute.NUMERIC);
                    break;
                }
                pic = new PIC(ExpressionEvaluator.set_a_pic(val2.StrVal), StorageAttribute.NUMERIC, this.ExpTask.getCompIdx());
                resVal.MgNumVal = new NUM_TYPE(val1.StrVal, pic, this.ExpTask.getCompIdx());
                break;
            case ExpressionInterface.EXP_OP_M:
                len = expStrTracker.get2ByteNumber();
                let codes = expStrTracker.getString(len, true, false);
                this.eval_op_m(resVal, codes);
                break;
            case ExpressionInterface.EXP_OP_K:
                resVal.Attr = StorageAttribute.NUMERIC;
                len = expStrTracker.get2ByteNumber();
                resVal.MgNumVal = expStrTracker.getMagicNumber(len, true);
                break;
            case ExpressionInterface.EXP_OP_F:
            case ExpressionInterface.EXP_OP_P:
                resVal.Attr = StorageAttribute.NUMERIC;
                len = expStrTracker.get2ByteNumber();
                resVal.MgNumVal = expStrTracker.getMagicNumber(len, true);
                len = expStrTracker.get2ByteNumber();
                expStrTracker.getMagicNumber(len, true);
                break;
            case ExpressionInterface.EXP_OP_STAT:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_stat(resVal, val1, val2);
                break;
            case ExpressionInterface.EXP_OP_SUBFORM_EXEC_MODE:
                val1 = valStack.pop();
                this.eval_op_subformExecMode(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_IS_LOGGED_IN:
                this.eval_op_IsLoggedIn(resVal);
                break;
            case ExpressionInterface.EXP_OP_SYS:
                this.eval_op_appname(resVal);
                break;
            case ExpressionInterface.EXP_OP_PROG:
                this.eval_op_prog(resVal);
                break;
            case ExpressionInterface.EXP_OP_LEVEL:
                val1 = valStack.pop();
                this.eval_op_level(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_COUNTER:
                val1 = valStack.pop();
                this.eval_op_counter(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_EMPTY_DATA_VIEW:
                val1 = valStack.pop();
                this.eval_op_emptyDataview(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_MAINLEVEL:
                val1 = valStack.pop();
                this.eval_op_mainlevel(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_MAINDISPLAY:
                val1 = valStack.pop();
                this.eval_op_maindisplay(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_ISFIRSTRECORDCYCLE:
                val1 = valStack.pop();
                this.eval_op_IsFirstRecordCycle(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_DATE:
            case ExpressionInterface.EXP_OP_UTCDATE:
                resVal.MgNumVal = new NUM_TYPE();
                resVal.MgNumVal.NUM_4_LONG(DisplayConvertor.Instance.date_magic(opCode === ExpressionInterface.EXP_OP_UTCDATE));
                resVal.Attr = StorageAttribute.DATE;
                break;
            case ExpressionInterface.EXP_OP_ADDDT:
                val8 = valStack.pop();
                val7 = valStack.pop();
                val6 = valStack.pop();
                val5 = valStack.pop();
                val4 = valStack.pop();
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                await this.eval_op_addDateTime(resVal, val1, val2, val3, val4, val5, val6, val7, val8);
                break;
            case ExpressionInterface.EXP_OP_DIFDT:
                val6 = valStack.pop();
                val5 = valStack.pop();
                val4 = valStack.pop();
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                await this.eval_op_difdt(resVal, val1, val2, val3, val4, val5, val6);
                break;
            case ExpressionInterface.EXP_OP_VARPREV:
                val1 = valStack.pop();
                this.exp_get_var(resVal, val1, true);
                break;
            case ExpressionInterface.EXP_OP_VARCURR:
                val1 = valStack.pop();
                this.exp_get_var(resVal, val1, false);
                break;
            case ExpressionInterface.EXP_OP_VARMOD:
                val1 = valStack.pop();
                this.eval_op_varmod(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_VARINP:
                val1 = valStack.pop();
                this.eval_op_varinp(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_VARNAME:
                val1 = valStack.pop();
                this.eval_op_varname(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_VARDISPLAYNAME:
                val1 = valStack.pop();
                this.eval_op_VarDisplayName(resVal, val1);
                expStrTracker.resetNullResult();
                break;
            case ExpressionInterface.EXP_OP_CONTROL_ITEMS_REFRESH:
                val2 = valStack.pop();
                val1 = valStack.pop();
                await this.eval_op_controlItemRefresh(val1, val2, resVal);
                break;
            case ExpressionInterface.EXP_OP_VARCONTROLID:
                val1 = valStack.pop();
                this.eval_op_VarControlID(resVal, val1);
                expStrTracker.resetNullResult();
                break;
            case ExpressionInterface.EXP_OP_CONTROLITEMSLIST:
                val1 = valStack.pop();
                await this.eval_op_ControlItemsList(resVal, val1);
                expStrTracker.resetNullResult();
                break;
            case ExpressionInterface.EXP_OP_CONTROLDISPLAYLIST:
                val1 = valStack.pop();
                await this.eval_op_ControlDisplayList(resVal, val1);
                expStrTracker.resetNullResult();
                break;
            case ExpressionInterface.EXP_OP_VIEWMOD:
                val1 = valStack.pop();
                this.eval_op_viewmod(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_TIME:
            case ExpressionInterface.EXP_OP_UTCTIME:
                resVal.MgNumVal = new NUM_TYPE();
                resVal.MgNumVal.NUM_4_LONG(DisplayConvertor.Instance.time_magic(opCode === ExpressionInterface.EXP_OP_UTCTIME));
                resVal.Attr = StorageAttribute.TIME;
                break;
            case ExpressionInterface.EXP_OP_MTIME:
            case ExpressionInterface.EXP_OP_UTCMTIME:
                resVal.MgNumVal = new NUM_TYPE();
                resVal.MgNumVal.NUM_4_LONG(DisplayConvertor.Instance.mtime_magic(opCode === ExpressionInterface.EXP_OP_UTCMTIME));
                resVal.Attr = StorageAttribute.TIME;
                break;
            case ExpressionInterface.EXP_OP_PWR:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.MgNumVal = new NUM_TYPE();
                resVal.MgNumVal = NUM_TYPE.eval_op_pwr(val1.MgNumVal, val2.MgNumVal);
                resVal.IsNull = (resVal.MgNumVal == null);
                resVal.Attr = StorageAttribute.NUMERIC;
                break;
            case ExpressionInterface.EXP_OP_LOG:
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.eval_op_log(val1.MgNumVal);
                resVal.IsNull = (resVal.MgNumVal == null);
                resVal.Attr = StorageAttribute.NUMERIC;
                break;
            case ExpressionInterface.EXP_OP_EXP:
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.eval_op_exp(val1.MgNumVal);
                resVal.IsNull = (resVal.MgNumVal == null);
                resVal.Attr = StorageAttribute.NUMERIC;
                break;
            case ExpressionInterface.EXP_OP_ABS:
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.eval_op_abs(val1.MgNumVal);
                resVal.IsNull = (resVal.MgNumVal == null);
                resVal.Attr = StorageAttribute.NUMERIC;
                break;
            case ExpressionInterface.EXP_OP_SIN:
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.eval_op_sin(val1.MgNumVal);
                resVal.IsNull = (resVal.MgNumVal == null);
                resVal.Attr = StorageAttribute.NUMERIC;
                break;
            case ExpressionInterface.EXP_OP_COS:
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.eval_op_cos(val1.MgNumVal);
                resVal.IsNull = (resVal.MgNumVal == null);
                resVal.Attr = StorageAttribute.NUMERIC;
                break;
            case ExpressionInterface.EXP_OP_TAN:
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.eval_op_tan(val1.MgNumVal);
                resVal.IsNull = (resVal.MgNumVal == null);
                resVal.Attr = StorageAttribute.NUMERIC;
                break;
            case ExpressionInterface.EXP_OP_ASIN:
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.eval_op_asin(val1.MgNumVal);
                resVal.IsNull = (resVal.MgNumVal == null);
                resVal.Attr = StorageAttribute.NUMERIC;
                break;
            case ExpressionInterface.EXP_OP_ACOS:
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.eval_op_acos(val1.MgNumVal);
                resVal.IsNull = (resVal.MgNumVal == null);
                resVal.Attr = StorageAttribute.NUMERIC;
                break;
            case ExpressionInterface.EXP_OP_ATAN:
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.eval_op_atan(val1.MgNumVal);
                resVal.IsNull = (resVal.MgNumVal == null);
                resVal.Attr = StorageAttribute.NUMERIC;
                break;
            case ExpressionInterface.EXP_OP_RAND:
                val1 = valStack.pop();
                resVal.MgNumVal = NUM_TYPE.eval_op_rand(val1.MgNumVal);
                resVal.IsNull = (resVal.MgNumVal == null);
                resVal.Attr = StorageAttribute.NUMERIC;
                break;
            case ExpressionInterface.EXP_OP_MIN:
            case ExpressionInterface.EXP_OP_MAX:
                nArgs = valStack.pop();
                this.val_cpy(valStack.pop(), resVal);
                try {
                    for (j = 1; j < nArgs; j++) {
                        val1 = valStack.pop();
                        if (opCode === ExpressionInterface.EXP_OP_MIN) {
                            if (ExpressionEvaluator.val_cmp_any(val1, resVal) < 0)
                                this.val_cpy(val1, resVal);
                        }
                        else {
                            if (ExpressionEvaluator.val_cmp_any(val1, resVal) > 0)
                                this.val_cpy(val1, resVal);
                        }
                    }
                }
                catch (oneOfValuesIsNull) {
                    if (oneOfValuesIsNull instanceof NullValueException) {
                        for (; valStack.count() > 0 && j < nArgs; j++)
                            valStack.pop();
                        resVal.IsNull = true;
                        resVal.Attr = oneOfValuesIsNull.getAttr();
                    }
                    else
                        throw oneOfValuesIsNull;
                }
                break;
            case ExpressionInterface.EXP_OP_RANGE:
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_range(val1, val2, val3, resVal);
                break;
            case ExpressionInterface.EXP_OP_REP:
                val4 = valStack.pop();
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                if (specialAnsiExpression ||
                    !(val1.Attr === StorageAttribute.UNICODE ||
                        val2.Attr === StorageAttribute.UNICODE)) {
                    resVal.Attr = StorageAttribute.ALPHA;
                    resVal.StrVal = UtilStrByteMode.repB(val1.StrVal, val2.StrVal, val3.MgNumVal.NUM_2_LONG(), val4.MgNumVal.NUM_2_LONG());
                }
                else {
                    this.eval_op_rep_1(resVal, val1, val2, val3, val4);
                }
                break;
            case ExpressionInterface.EXP_OP_INS:
                val4 = valStack.pop();
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                if (specialAnsiExpression ||
                    !(val1.Attr === StorageAttribute.UNICODE ||
                        val2.Attr === StorageAttribute.UNICODE)) {
                    resVal.Attr = StorageAttribute.ALPHA;
                    resVal.StrVal = UtilStrByteMode.insB(val1.StrVal, val2.StrVal, val3.MgNumVal.NUM_2_LONG(), val4.MgNumVal.NUM_2_LONG());
                }
                else {
                    this.eval_op_ins(resVal, val1, val2, val3, val4);
                }
                break;
            case ExpressionInterface.EXP_OP_DEL:
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                if (specialAnsiExpression || val1.Attr !== StorageAttribute.UNICODE) {
                    resVal.Attr = StorageAttribute.ALPHA;
                    resVal.StrVal = UtilStrByteMode.delB(val1.StrVal, val2.MgNumVal.NUM_2_LONG(), val3.MgNumVal.NUM_2_LONG());
                }
                else {
                    this.eval_op_del(resVal, val1, val2, val3);
                }
                break;
            case ExpressionInterface.EXP_OP_FLIP:
                val1 = valStack.pop();
                this.eval_op_flip(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_UPPER:
                val1 = valStack.pop();
                this.eval_op_upper(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_LOWER:
                val1 = valStack.pop();
                this.eval_op_lower(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_CRC:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_crc(resVal, val1, val2);
                break;
            case ExpressionInterface.EXP_OP_CHKDGT:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_chkdgt(resVal, val1, val2);
                break;
            case ExpressionInterface.EXP_OP_SOUNDX:
                val1 = valStack.pop();
                this.eval_op_soundx(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_HSTR:
                val1 = valStack.pop();
                this.eval_op_hstr(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_HVAL:
                val1 = valStack.pop();
                this.eval_op_hval(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_CHR:
                val1 = valStack.pop();
                this.eval_op_chr(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_ASC:
                val1 = valStack.pop();
                this.eval_op_asc(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_MSTR:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_mstr(resVal, val1, val2);
                break;
            case ExpressionInterface.EXP_OP_MVAL:
                val1 = valStack.pop();
                this.eval_op_mval(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_DSTR:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_dstr(resVal, val1, val2, DisplayConvertor.Instance);
                break;
            case ExpressionInterface.EXP_OP_DVAL:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_dval(resVal, val1, val2, DisplayConvertor.Instance);
                break;
            case ExpressionInterface.EXP_OP_TSTR:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_tstr(resVal, val1, val2, DisplayConvertor.Instance, false);
                break;
            case ExpressionInterface.EXP_OP_MTSTR:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_tstr(resVal, val1, val2, DisplayConvertor.Instance, true);
                break;
            case ExpressionInterface.EXP_OP_TVAL:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_tval(resVal, val1, val2, DisplayConvertor.Instance, false);
                break;
            case ExpressionInterface.EXP_OP_MTVAL:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_tval(resVal, val1, val2, DisplayConvertor.Instance, true);
                break;
            case ExpressionInterface.EXP_OP_DAY:
                val1 = valStack.pop();
                this.eval_op_day(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_MONTH:
                val1 = valStack.pop();
                this.eval_op_month(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_YEAR:
                val1 = valStack.pop();
                this.eval_op_year(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_DOW:
                val1 = valStack.pop();
                this.eval_op_dow(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_CDOW:
                val1 = valStack.pop();
                this.eval_op_cdow(resVal, val1.MgNumVal, DisplayConvertor.Instance);
                break;
            case ExpressionInterface.EXP_OP_CMONTH:
                val1 = valStack.pop();
                this.eval_op_cmonth(resVal, val1.MgNumVal, DisplayConvertor.Instance);
                break;
            case ExpressionInterface.EXP_OP_NDOW:
                val1 = valStack.pop();
                this.eval_op_ndow(resVal, val1, DisplayConvertor.Instance);
                break;
            case ExpressionInterface.EXP_OP_NMONTH:
                val1 = valStack.pop();
                this.eval_op_nmonth(resVal, val1, DisplayConvertor.Instance);
                break;
            case ExpressionInterface.EXP_OP_SECOND:
                val1 = valStack.pop();
                this.eval_op_second(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_MINUTE:
                val1 = valStack.pop();
                this.eval_op_minute(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_HOUR:
                val1 = valStack.pop();
                this.eval_op_hour(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_DELAY:
                val1 = valStack.pop();
                this.eval_op_delay();
                resVal.Attr = StorageAttribute.BOOLEAN;
                resVal.BoolVal = true;
                break;
            case ExpressionInterface.EXP_OP_IDLE:
                this.eval_op_idle(resVal);
                break;
            case ExpressionInterface.EXP_OP_ADDDATE:
                val4 = valStack.pop();
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_adddate(resVal, val1, val2, val3, val4);
                break;
            case ExpressionInterface.EXP_OP_ADDTIME:
                val4 = valStack.pop();
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_addtime(resVal, val1, val2, val3, val4);
                break;
            case ExpressionInterface.EXP_OP_OWNER:
                resVal.Attr = StorageAttribute.ALPHA;
                resVal.StrVal = Environment.Instance.getOwner();
                break;
            case ExpressionInterface.EXP_OP_VARATTR:
                val1 = valStack.pop();
                this.eval_op_varattr(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_VARPIC:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_varpic(resVal, val1, val2);
                break;
            case ExpressionInterface.EXP_OP_DBROUND:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_dbround(resVal, val1, val2);
                break;
            case ExpressionInterface.EXP_OP_NULL:
            case ExpressionInterface.EXP_OP_NULL_A:
            case ExpressionInterface.EXP_OP_NULL_N:
            case ExpressionInterface.EXP_OP_NULL_B:
            case ExpressionInterface.EXP_OP_NULL_D:
            case ExpressionInterface.EXP_OP_NULL_T:
            case ExpressionInterface.EXP_OP_NULL_U:
            case ExpressionInterface.EXP_OP_NULL_O:
                this.exp_null_val_get(expectedType, opCode, resVal);
                break;
            case ExpressionInterface.EXP_OP_ISNULL:
                val1 = valStack.pop();
                resVal.BoolVal = val1.IsNull;
                if (!val1.IsNull && val1.OriginalNull)
                    resVal.BoolVal = val1.OriginalNull;
                resVal.Attr = StorageAttribute.BOOLEAN;
                if (expStrTracker.isNull()) {
                    let myArray = new List();
                    let prevNull = false;
                    let i;
                    while (!(valStack.count() === 0) && !prevNull) {
                        myArray.push(valStack.pop());
                        if (myArray.get_Item(myArray.length - 1).IsNull)
                            prevNull = true;
                    }
                    for (i = myArray.length - 1; i >= 0; i--)
                        valStack.push(myArray.get_Item(i));
                    if (!prevNull)
                        expStrTracker.resetNullResult();
                }
                break;
            case ExpressionInterface.EXP_OP_BOM:
                val1 = valStack.pop();
                this.eval_op_bom(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_BOY:
                val1 = valStack.pop();
                this.eval_op_boy(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_EOM:
                val1 = valStack.pop();
                this.eval_op_eom(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_EOY:
                val1 = valStack.pop();
                this.eval_op_eoy(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_VARSET:
                val2 = valStack.pop();
                val1 = valStack.pop();
                await this.eval_op_varset(resVal, val2, val1);
                break;
            case ExpressionInterface.EXP_OP_CTRL_NAME:
                this.eval_op_ctrl_name(resVal);
                break;
            case ExpressionInterface.EXP_OP_TDEPTH:
                let currTsk = this.ExpTask.GetContextTask();
                len = (currTsk.getTaskDepth(false)) - 1;
                this.ConstructMagicNum(resVal, len, StorageAttribute.NUMERIC);
                break;
            case ExpressionInterface.EXP_OP_ISDEFAULT:
                val1 = valStack.pop();
                this.exp_is_default(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_STRTOKEN:
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_strtok(resVal, val1, val2, val3);
                break;
            case ExpressionInterface.EXP_OP_STRTOK_CNT:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_strTokenCnt(val1, val2, resVal);
                break;
            case ExpressionInterface.EXP_OP_STRTOKEN_IDX:
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_strTokenIdx(val1, val2, val3, resVal);
                break;
            case ExpressionInterface.EXP_OP_BLOBSIZE:
                val1 = valStack.pop();
                this.eval_op_blobsize(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_WEB_REFERENCE:
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.ALPHA;
                resVal.StrVal = '%' + val1.StrVal + '%';
                break;
            case ExpressionInterface.EXP_OP_CASE:
                nArgs = valStack.pop();
                Exp_params = new Array(nArgs);
                for (j = 0; j < nArgs; j++)
                    Exp_params[nArgs - 1 - j] = valStack.pop();
                this.val_cpy(Exp_params[0], resVal);
                for (j = 1; j < nArgs; j += 2) {
                    val1 = Exp_params[j];
                    let valueMatched;
                    try {
                        valueMatched = (ExpressionEvaluator.val_cmp_any(val1, resVal) === 0);
                    }
                    catch (nullValueException) {
                        if (nullValueException instanceof NullValueException) {
                            valueMatched = false;
                        }
                        else
                            throw nullValueException;
                    }
                    if (valueMatched) {
                        this.val_cpy(Exp_params[j + 1], resVal);
                        break;
                    }
                    if (j === (nArgs - 3)) {
                        this.val_cpy(Exp_params[j + 2], resVal);
                        break;
                    }
                }
                expStrTracker.resetNullResult();
                break;
            case ExpressionInterface.EXP_OP_THIS:
                this.eval_op_this(resVal);
                break;
            case ExpressionInterface.EXP_OP_LIKE:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_like(val1, val2, resVal);
                break;
            case ExpressionInterface.EXP_OP_REPSTR:
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_repstr(val1, val2, val3, resVal);
                break;
            case ExpressionInterface.EXP_OP_VARCURRN:
                val1 = valStack.pop();
                this.exp_get_var(val1, resVal);
                break;
            case ExpressionInterface.EXP_OP_VARINDEX:
                val1 = valStack.pop();
                this.exp_get_indx(val1, resVal);
                break;
            case ExpressionInterface.EXP_OP_JCDOW:
                if (this._expressionLocalJpn == null)
                    super.SetNULL(resVal, StorageAttribute.ALPHA);
                else {
                    val1 = valStack.pop();
                    this._expressionLocalJpn.eval_op_jcdow(resVal, val1.MgNumVal, DisplayConvertor.Instance);
                }
                break;
            case ExpressionInterface.EXP_OP_JMONTH:
                if (this._expressionLocalJpn == null)
                    super.SetNULL(resVal, StorageAttribute.ALPHA);
                else {
                    val1 = valStack.pop();
                    this._expressionLocalJpn.eval_op_jmonth(resVal, val1);
                }
                break;
            case ExpressionInterface.EXP_OP_JNDOW:
                if (this._expressionLocalJpn == null)
                    super.SetNULL(resVal, StorageAttribute.ALPHA);
                else {
                    val1 = valStack.pop();
                    this._expressionLocalJpn.eval_op_jndow(resVal, val1, DisplayConvertor.Instance);
                }
                break;
            case ExpressionInterface.EXP_OP_JYEAR:
                if (this._expressionLocalJpn == null)
                    super.SetNULL(resVal, StorageAttribute.ALPHA);
                else {
                    val1 = valStack.pop();
                    this._expressionLocalJpn.eval_op_jyear(resVal, val1);
                }
                break;
            case ExpressionInterface.EXP_OP_JGENGO:
                if (this._expressionLocalJpn == null)
                    super.SetNULL(resVal, StorageAttribute.ALPHA);
                else {
                    val2 = valStack.pop();
                    val1 = valStack.pop();
                    this._expressionLocalJpn.eval_op_jgengo(resVal, val1.MgNumVal, val2.MgNumVal, DisplayConvertor.Instance);
                }
                break;
            case ExpressionInterface.EXP_OP_HAN:
                val1 = valStack.pop();
                resVal.Attr = val1.Attr;
                if (this._expressionLocalJpn == null)
                    resVal.StrVal = val1.StrVal;
                else if (val1.StrVal == null)
                    super.SetNULL(resVal, val1.Attr);
                else {
                    resVal.StrVal = this._expressionLocalJpn.eval_op_han(val1.StrVal);
                    if (resVal.StrVal == null)
                        resVal.StrVal = '';
                }
                break;
            case ExpressionInterface.EXP_OP_ZEN:
                val1 = valStack.pop();
                resVal.Attr = val1.Attr;
                if (this._expressionLocalJpn == null)
                    resVal.StrVal = val1.StrVal;
                else if (val1.StrVal == null)
                    super.SetNULL(resVal, val1.Attr);
                else {
                    resVal.StrVal = this._expressionLocalJpn.eval_op_zens(val1.StrVal, 0);
                    if (resVal.StrVal == null)
                        resVal.StrVal = '';
                }
                break;
            case ExpressionInterface.EXP_OP_ZENS:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = val1.Attr;
                if (this._expressionLocalJpn == null)
                    resVal.StrVal = val1.StrVal;
                else if (val2.MgNumVal == null || val1.StrVal == null)
                    super.SetNULL(resVal, val1.Attr);
                else {
                    resVal.StrVal = this._expressionLocalJpn.eval_op_zens(val1.StrVal, val2.MgNumVal.NUM_2_LONG());
                    if (resVal.StrVal == null)
                        resVal.StrVal = '';
                }
                break;
            case ExpressionInterface.EXP_OP_ZIMEREAD:
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.ALPHA;
                if (this._expressionLocalJpn == null)
                    resVal.StrVal = val1.StrVal;
                else {
                    resVal.StrVal = this._expressionLocalJpn.eval_op_zimeread();
                    if (resVal.StrVal == null)
                        resVal.StrVal = '';
                }
                break;
            case ExpressionInterface.EXP_OP_ZKANA:
                val2 = valStack.pop();
                val1 = valStack.pop();
                resVal.Attr = val1.Attr;
                if (this._expressionLocalJpn == null)
                    resVal.StrVal = val1.StrVal;
                else if (val2.MgNumVal == null || val1.StrVal == null)
                    super.SetNULL(resVal, val1.Attr);
                else {
                    resVal.StrVal = this._expressionLocalJpn.eval_op_zkana(val1.StrVal, val2.MgNumVal.NUM_2_LONG());
                    if (resVal.StrVal == null)
                        resVal.StrVal = '';
                }
                break;
            case ExpressionInterface.EXP_OP_GOTO_CTRL:
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_gotoCtrl(val1, val2, val3, resVal);
                break;
            case ExpressionInterface.EXP_OP_TRANSLATE:
                val1 = valStack.pop();
                this.eval_op_translate(val1, resVal);
                break;
            case ExpressionInterface.EXP_OP_ASTR:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_astr(val1, val2, resVal);
                break;
            case ExpressionInterface.EXP_OP_LOOPCOUNTER:
                this.ConstructMagicNum(resVal, this.ExpTask.getLoopCounter(), StorageAttribute.NUMERIC);
                break;
            case ExpressionInterface.EXP_OP_VECCELLATTR:
                val1 = valStack.pop();
                ExpressionEvaluator.eval_op_vecCellAttr(val1, resVal);
                break;
            case ExpressionInterface.EXP_OP_VECGET:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_vecGet(val1, val2, resVal);
                break;
            case ExpressionInterface.EXP_OP_VECSET:
                val3 = valStack.pop();
                val2 = valStack.pop();
                val1 = valStack.pop();
                await this.eval_op_vecSet(val1, val2, val3, resVal);
                resVal.IsNull = false;
                if (expStrTracker.isNull())
                    expStrTracker.resetNullResult();
                break;
            case ExpressionInterface.EXP_OP_VECSIZE:
                val1 = valStack.pop();
                this.eval_op_vecSize(val1, resVal);
                expStrTracker.resetNullResult();
                break;
            case ExpressionInterface.EXP_OP_IN:
                nArgs = valStack.pop();
                resVal.Attr = StorageAttribute.BOOLEAN;
                resVal.BoolVal = false;
                Exp_params = new Array(nArgs);
                for (j = 0; j < nArgs; j++)
                    Exp_params[nArgs - 1 - j] = valStack.pop();
                try {
                    for (j = 1; j < nArgs; j++) {
                        if (ExpressionEvaluator.val_cmp_any(Exp_params[0], Exp_params[j]) === 0) {
                            resVal.BoolVal = true;
                            break;
                        }
                    }
                }
                catch (nullValueException) {
                    if (nullValueException instanceof NullValueException) {
                        super.SetNULL(resVal, StorageAttribute.BOOLEAN);
                    }
                    else
                        throw nullValueException;
                }
                break;
            case ExpressionInterface.EXP_OP_ISCOMPONENT:
                this.eval_op_iscomponent(resVal);
                break;
            case ExpressionInterface.EXP_OP_USER_DEFINED_FUNC:
                nArgs = valStack.pop();
                if (nArgs > 0) {
                    nArgs--;
                    Exp_params = new Array(nArgs);
                    for (j = 0; j < nArgs; j++)
                        Exp_params[nArgs - 1 - j] = valStack.pop();
                    let funcName = valStack.pop().StrVal;
                    await this.eval_op_ExecUserDefinedFunc(funcName, Exp_params, resVal, expectedType);
                    expStrTracker.resetNullResult();
                }
                break;
            case ExpressionInterface.EXP_OP_UNICODEASC:
                val1 = valStack.pop();
                resVal.Attr = StorageAttribute.NUMERIC;
                resVal.IsNull = false;
                if (!val1.IsNull && val1.StrVal.length > 0) {
                    resVal.MgNumVal = new NUM_TYPE();
                    resVal.MgNumVal.NUM_4_LONG(val1.StrVal[0].charCodeAt(0));
                }
                else
                    resVal.IsNull = true;
                break;
            case ExpressionInterface.EXP_OP_PUBLICNAME:
                val1 = valStack.pop();
                this.eval_op_publicName(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_TASKID:
                val1 = valStack.pop();
                this.eval_op_taskId(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_DBVIEWSIZE:
                val1 = valStack.pop();
                this.eval_op_dbviewsize(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_DBVIEWROWIDX:
                val1 = valStack.pop();
                this.eval_op_dbviewrowidx(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_PROJECTDIR:
                this.eval_op_projectdir(resVal);
                break;
            case ExpressionInterface.EXP_OP_MLS_TRANS:
                val1 = valStack.pop();
                this.eval_op_MlsTrans(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_STR_BUILD:
                nArgs = valStack.pop();
                this.eval_op_StrBuild(valStack, resVal, nArgs);
                break;
            case ExpressionInterface.EXP_OP_STATUSBARSETTEXT:
                val1 = valStack.pop();
                this.eval_op_statusbar_set_text(resVal, val1);
                expStrTracker.resetNullResult();
                break;
            case ExpressionInterface.EXP_OP_SET_TITLE:
                val1 = valStack.pop();
                this.eval_op_set_title(resVal, val1);
                expStrTracker.resetNullResult();
                break;
            case ExpressionInterface.EXP_OP_IS_ROW_EDITING:
                val1 = valStack.pop();
                this.eval_op_is_row_editing(resVal, val1);
                expStrTracker.resetNullResult();
                break;
            case ExpressionInterface.EXP_OP_GETPARAM:
                val1 = valStack.pop();
                this.eval_op_getParam(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_SETPARAM:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_setParam(resVal, val1, val2);
                break;
            case ExpressionInterface.EXP_OP_INIPUT:
                val2 = valStack.pop();
                val1 = valStack.pop();
                await this.eval_op_iniput(resVal, val1, val2);
                break;
            case ExpressionInterface.EXP_OP_INIGET:
                val1 = valStack.pop();
                this.eval_op_iniget(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_INIGETLN:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_inigetln(resVal, val1, val2);
                break;
            case ExpressionInterface.EXP_OP_EXPCALC:
                val1 = valStack.pop();
                await this.eval_op_expcalc(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_TASKTYPE:
                val1 = valStack.pop();
                this.eval_op_taskType(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_TSK_INSTANCE:
                val1 = valStack.pop();
                this.eval_op_tsk_instance(resVal, val1);
                break;
            case ExpressionInterface.EXP_OP_TERM:
                this.eval_op_terminal(resVal);
                break;
            case ExpressionInterface.EXP_OP_CLIENTSESSION_SET:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_ClientSessionSet(val1, val2, resVal);
                break;
            case ExpressionInterface.EXP_OP_CND_RANGE:
                val2 = valStack.pop();
                val1 = valStack.pop();
                this.eval_op_CndRange(resVal, val1, val2);
                break;
            case ExpressionInterface.EXP_OP_CONTROL_SELECT_PROGRAM:
                val1 = valStack.pop();
                await this.eval_op_control_select_program(val1, resVal);
                break;
            case ExpressionInterface.EXP_OP_CALLJS:
                nArgs = valStack.pop();
                if (nArgs > 0) {
                    nArgs--;
                    Exp_params = new Array(nArgs);
                    for (j = 0; j < nArgs; j++)
                        Exp_params[nArgs - 1 - j] = valStack.pop();
                    await this.eval_op_CallJS(this.ExpTask, valStack.pop().StrVal, Exp_params, resVal);
                }
                break;
            case ExpressionInterface.EXP_OP_SET_COOKIE:
                nArgs = valStack.pop();
                if (nArgs > 0) {
                    nArgs--;
                    Exp_params = new Array(nArgs);
                    for (j = 0; j < nArgs; j++)
                        Exp_params[nArgs - 1 - j] = valStack.pop();
                    this.eval_op_set_cookie(valStack.pop().StrVal, Exp_params, resVal);
                }
                break;
            case ExpressionInterface.EXP_OP_GET_COOKIE:
                resVal.Attr = StorageAttribute.ALPHA;
                resVal.IsNull = false;
                this.eval_op_get_cookie(valStack.pop().StrVal, resVal);
                break;
            case ExpressionInterface.EXP_OP_DELETE_COOKIE:
                resVal.Attr = StorageAttribute.BOOLEAN;
                resVal.IsNull = false;
                this.eval_op_delete_cookie(valStack.pop().StrVal, resVal);
                break;
            case ExpressionInterface.EXP_OP_ROUTEGET:
                this.eval_op_route_get(resVal);
                break;
            default:
                return;
        }
        if (addResult) {
            this.ConvertExpVal(resVal, expectedType);
            valStack.push(resVal);
            if (resVal.IsNull)
                expStrTracker.setNullResult();
        }
    }
    eval_op_ClientSessionSet(val1, val2, resVal) {
        resVal.BoolVal = false;
        resVal.Attr = StorageAttribute.BOOLEAN;
        if (!val1.isEmptyString()) {
            switch (val1.StrVal) {
                case ConstInterface.ENABLE_COMMUNICATION_DIALOGS:
                    if (val2.Attr === StorageAttribute.BOOLEAN) {
                        resVal.BoolVal = true;
                    }
                    else
                        Logger.Instance.WriteExceptionToLogWithMsg('Invalid attribute for ' + ConstInterface.ENABLE_COMMUNICATION_DIALOGS + ' key in ClientSessionSet()');
                    break;
                default:
                    Logger.Instance.WriteExceptionToLogWithMsg('Invalid Key in ClientSessionSet()');
                    break;
            }
        }
    }
    async execExpression(expStrTracker, valStack, expectedType) {
        let expVal;
        let i;
        let addedOpers = new List();
        let dynOper;
        let opCode = expStrTracker.getOpCode();
        if (ExpressionEvaluator.isBasicItem(opCode)) {
            expVal = this.getExpBasicValue(expStrTracker, opCode);
            this.ConvertExpVal(expVal, expectedType);
            valStack.push(expVal);
            if (expVal.IsNull)
                expStrTracker.setNullResult();
            return;
        }
        let expDesc = ExpressionDict.expDesc[opCode];
        let nullArgs = false;
        if (expDesc.ArgEvalCount_ > 0) {
            for (i = 0; i < expDesc.ArgEvalCount_; i++) {
                expStrTracker.resetNullResult();
                await this.execExpression(expStrTracker, valStack, expDesc.ArgAttr_[i]);
                if (expStrTracker.isNull())
                    nullArgs = true;
            }
        }
        if (nullArgs)
            expStrTracker.setNullResult();
        await this.execOperation(opCode, expStrTracker, valStack, addedOpers, expectedType);
        let nDynOpers = addedOpers.length;
        if (nDynOpers > 0) {
            for (i = 0; i < nDynOpers; i++) {
                dynOper = addedOpers.get_Item(0);
                addedOpers.RemoveAt(0);
                switch (dynOper.opCode_) {
                    case ExpressionInterface.EXP_OP_IGNORE:
                        let j;
                        for (j = 0; j < dynOper.argCount_; j++)
                            expStrTracker.skipOperator();
                        break;
                    case ExpressionInterface.EXP_OP_EVALX:
                        await this.execExpression(expStrTracker, valStack, expectedType);
                        break;
                }
            }
        }
    }
    async DiscardCndRangeExpression(expStrTracker, valStack) {
        let expVal;
        let opCode = expStrTracker.getOpCode();
        if (opCode !== ExpressionInterface.EXP_OP_CND_RANGE) {
            return false;
        }
        let expDesc = ExpressionDict.expDesc[opCode];
        await this.execExpression(expStrTracker, valStack, expDesc.ArgAttr_[0]);
        expVal = valStack.pop();
        return expVal.Attr === StorageAttribute.BOOLEAN && !expVal.BoolVal;
    }
    static val_cmp_any(val1, val2) {
        let retval = 0;
        let attr1 = val1.Attr;
        let attr2 = val2.Attr;
        let expVal = new ExpressionEvaluator();
        if (val1.IsNull && val2.IsNull)
            return 0;
        if (val1.IsNull || val2.IsNull)
            throw new NullValueException(attr1);
        if (StorageAttributeCheck.isTypeBlob(attr1)) {
            if (val1.IncludeBlobPrefix && BlobType.getContentType(val1.StrVal) === BlobType.CONTENT_TYPE_BINARY) {
                val1.StrVal = BlobType.removeBlobPrefix(val1.StrVal);
                val1.Attr = StorageAttribute.ALPHA;
                val1.IncludeBlobPrefix = false;
            }
            else
                expVal.ConvertExpVal(val1, StorageAttribute.UNICODE);
        }
        if (StorageAttributeCheck.isTypeBlob(attr2)) {
            if (val2.IncludeBlobPrefix && BlobType.getContentType(val2.StrVal) === BlobType.CONTENT_TYPE_BINARY) {
                val2.StrVal = BlobType.removeBlobPrefix(val2.StrVal);
                val2.Attr = StorageAttribute.ALPHA;
                val2.IncludeBlobPrefix = false;
            }
            else
                expVal.ConvertExpVal(val2, StorageAttribute.UNICODE);
        }
        attr1 = val1.Attr;
        attr2 = val2.Attr;
        if (attr1 !== attr2) {
            if ((StorageAttributeCheck.isTypeNumeric(attr1) && StorageAttributeCheck.isTypeNumeric(attr2)) ||
                (StorageAttributeCheck.IsTypeAlphaOrUnicode(attr1) && StorageAttributeCheck.IsTypeAlphaOrUnicode(attr2))) {
            }
            else
                return 1;
        }
        switch (attr1) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.BLOB:
            case StorageAttribute.BLOB_VECTOR:
            case StorageAttribute.UNICODE:
                if (val1.StrVal === null && val2.StrVal === null)
                    return 0;
                if (val1.StrVal === null || val2.StrVal === null)
                    throw new NullValueException(attr1);
                let str1 = StrUtil.rtrim(val1.StrVal);
                let str2 = StrUtil.rtrim(val2.StrVal);
                if (Environment.Instance.getSpecialAnsiExpression() ||
                    (UtilStrByteMode.isLocaleDefLangDBCS() &&
                        attr1 === StorageAttribute.ALPHA && attr2 === StorageAttribute.ALPHA))
                    retval = UtilStrByteMode.strcmp(str1, str2);
                else
                    retval = NString.CompareOrdinal(str1, str2);
                break;
            case StorageAttribute.NUMERIC:
            case StorageAttribute.DATE:
            case StorageAttribute.TIME:
                if (val1.MgNumVal === null && val2.MgNumVal === null)
                    return 0;
                if (val1.MgNumVal === null || val2.MgNumVal === null)
                    throw new NullValueException(attr1);
                retval = NUM_TYPE.num_cmp(val1.MgNumVal, val2.MgNumVal);
                break;
            case StorageAttribute.BOOLEAN:
                retval = (val1.BoolVal ? 1 : 0) - (val2.BoolVal ? 1 : 0);
                break;
        }
        return (retval);
    }
    static async eval(exp, expectedType, task) {
        let res = null;
        let valStack = new Stack();
        if (exp !== null && exp.length > 0) {
            let me = new ExpressionEvaluator();
            let expStrTracker = new ExpStrTracker(exp, task.getNullArithmetic() === Constants.NULL_ARITH_NULLIFY);
            me.ExpTask = task;
            await me.execExpression(expStrTracker, valStack, expectedType);
            res = valStack.pop();
            if (expStrTracker.isNull())
                res.IsNull = true;
        }
        return res;
    }
    static async DiscardCndRangeResult(exp, task) {
        let valStack = new Stack();
        if (exp !== null && exp.length > 0) {
            let me = new ExpressionEvaluator();
            let expStrTracker = new ExpStrTracker(exp, task.getNullArithmetic() === Constants.NULL_ARITH_NULLIFY);
            me.ExpTask = task;
            return await me.DiscardCndRangeExpression(expStrTracker, valStack);
        }
        return false;
    }
    static set_a_pic(val) {
        let len = Math.min(val.length, 99);
        return StrUtil.ZstringMake(val, len);
    }
    eval_op_range(val1, val2, val3, resVal) {
        resVal.BoolVal = false;
        resVal.Attr = StorageAttribute.BOOLEAN;
        try {
            if (ExpressionEvaluator.val_cmp_any(val1, val2) >= 0) {
                val2 = val3;
                if (ExpressionEvaluator.val_cmp_any(val1, val2) <= 0)
                    resVal.BoolVal = true;
            }
        }
        catch (ex) {
            if (ex instanceof NullValueException) {
                super.SetNULL(resVal, ex.getAttr());
            }
            else
                throw ex;
        }
    }
    eval_op_rep_1(resVal, val1, val2, val3, val4) {
        if (val1.StrVal === null || val2.StrVal === null || val3.MgNumVal === null || val4.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.UNICODE);
            return;
        }
        ExpressionEvaluator.val_s_cpy(val1, resVal);
        let i = val3.MgNumVal.NUM_2_LONG();
        if (i < 1)
            i = 1;
        let j = val4.MgNumVal.NUM_2_LONG();
        if (j > val2.StrVal.length)
            j = val2.StrVal.length;
        if (i + j - 1 > resVal.StrVal.length)
            j = resVal.StrVal.length - i + 1;
        if ((j <= 0))
            return;
        let tmp_s = StrUtil.memcpy('', 0, resVal.StrVal, i + j - 1, resVal.StrVal.length - i - j + 1);
        resVal.StrVal = StrUtil.memcpy(resVal.StrVal, i - 1, val2.StrVal, 0, j);
        resVal.StrVal = StrUtil.memcpy(resVal.StrVal, i - 1 + j, tmp_s, 0, resVal.StrVal.length - i - j + 1);
        if (j - val2.StrVal.length > 0)
            resVal.StrVal = StrUtil.memset(resVal.StrVal, i + val2.StrVal.length - 1, ' ', j - val2.StrVal.length);
    }
    static val_s_cpy(val, resVal) {
        resVal.Attr = val.Attr;
        resVal.StrVal = val.StrVal;
        resVal.IsNull = val.IsNull;
        resVal.IncludeBlobPrefix = val.IncludeBlobPrefix;
    }
    val_cpy(val, resVal) {
        switch (val.Attr) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.UNICODE:
            case StorageAttribute.BLOB:
            case StorageAttribute.BLOB_VECTOR:
                ExpressionEvaluator.val_s_cpy(val, resVal);
                break;
            case StorageAttribute.NUMERIC:
            case StorageAttribute.DATE:
            case StorageAttribute.TIME:
                if (!val.IsNull)
                    resVal.MgNumVal = new NUM_TYPE(val.MgNumVal);
                break;
            case StorageAttribute.BOOLEAN:
                resVal.BoolVal = val.BoolVal;
                break;
            default:
                Logger.Instance.WriteExceptionToLogWithMsg('Expression Evaluator.val_cpy no such type of attribute : ' + val.Attr);
                break;
        }
        resVal.Attr = val.Attr;
        resVal.IsNull = val.IsNull;
    }
    eval_op_ins(resVal, val1, val2, val3, val4) {
        resVal.Attr = StorageAttribute.UNICODE;
        if (val1.StrVal === null || val2.StrVal === null || val3.MgNumVal === null || val4.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.UNICODE);
            return;
        }
        let i = val1.StrVal.length;
        let ins_ofs = val3.MgNumVal.NUM_2_LONG() - 1;
        ins_ofs = Math.max(ins_ofs, 0);
        ins_ofs = Math.min(ins_ofs, i);
        let ins_len = val4.MgNumVal.NUM_2_LONG();
        ins_len = Math.max(ins_len, 0);
        ins_len = Math.min(ins_len, val2.StrVal.length);
        resVal.StrVal = StrUtil.memcpy('', 0, val1.StrVal, 0, ins_ofs);
        resVal.StrVal = StrUtil.memcpy(resVal.StrVal, ins_ofs, val2.StrVal, 0, ins_len);
        resVal.StrVal = StrUtil.memcpy(resVal.StrVal, ins_ofs + ins_len, val1.StrVal, ins_ofs, i - ins_ofs);
    }
    eval_op_del(resVal, val1, val2, val3) {
        if (val1.StrVal === null || val2.MgNumVal === null || val3.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.UNICODE);
            return;
        }
        ExpressionEvaluator.val_s_cpy(val1, resVal);
        let i = val2.MgNumVal.NUM_2_LONG() - 1;
        if (i < 0) {
            i = 0;
        }
        if (i > resVal.StrVal.length)
            i = resVal.StrVal.length;
        let j = val3.MgNumVal.NUM_2_LONG();
        if (i + j > resVal.StrVal.length)
            j = resVal.StrVal.length - i;
        if (j <= 0)
            return;
        resVal.StrVal = StrUtil.memcpy(resVal.StrVal, i, resVal.StrVal, i + j, resVal.StrVal.length - i);
    }
    eval_op_flip(resVal, val1) {
        resVal.Attr = StorageAttribute.UNICODE;
        if (val1.StrVal === null) {
            super.SetNULL(resVal, StorageAttribute.UNICODE);
            return;
        }
        let rev_str = new StringBuilder(val1.StrVal);
        resVal.StrVal = StrUtil.ReverseString(rev_str).ToString();
    }
    eval_op_upper(resVal, val1) {
        resVal.Attr = StorageAttribute.UNICODE;
        if (val1.StrVal === null) {
            super.SetNULL(resVal, StorageAttribute.UNICODE);
            return;
        }
        resVal.StrVal = val1.StrVal.toUpperCase();
    }
    eval_op_lower(resVal, val1) {
        resVal.Attr = StorageAttribute.UNICODE;
        if (val1.StrVal == null) {
            super.SetNULL(resVal, StorageAttribute.UNICODE);
            return;
        }
        resVal.StrVal = val1.StrVal.toLowerCase();
    }
    eval_op_crc(resVal, val1, val2) {
        resVal.Attr = StorageAttribute.ALPHA;
        if (val1.StrVal == null || val2.MgNumVal == null) {
            super.SetNULL(resVal, StorageAttribute.ALPHA);
            return;
        }
        let mode = val2.MgNumVal.NUM_2_LONG();
        let res = 0;
        switch (mode) {
            case 0:
                res = ExpressionEvaluator.eval_crc_16(val1.StrVal);
                break;
        }
        let left = String.fromCharCode(res % 256);
        let right = String.fromCharCode(res / 256);
        resVal.StrVal = '' + left + right;
    }
    static eval_crc_16(buf) {
        let crc_16_table = [
            40961, 61441, 55297, 52225, 50689, 49921, 49537, 49345
        ];
        let buffer_idx = 0;
        let len = buf.length;
        let crc = 0;
        for (; len > 0; len--) {
            let bt = buf.charCodeAt(buffer_idx++);
            bt = (bt ^ ExpressionEvaluator.LO_CHAR(crc));
            crc = ExpressionEvaluator.LO_CHAR(ExpressionEvaluator.MK_SHRT(0, ExpressionEvaluator.HI_CHAR(crc)));
            let mask;
            let tbl_idx;
            for (tbl_idx = 0, mask = ExpressionEvaluator.LO_CHAR(0x80); tbl_idx < 8; tbl_idx++, mask = (ExpressionEvaluator.LO_CHAR(mask) >> 1)) {
                if (ExpressionEvaluator.LO_CHAR((bt & ExpressionEvaluator.LO_CHAR(mask))) !== 0)
                    crc = (crc ^ crc_16_table[tbl_idx]);
            }
        }
        return (crc);
    }
    static LO_CHAR(n) {
        return (n & 0xff);
    }
    static HI_CHAR(n) {
        return ((n & 0xff00) >> 8);
    }
    static MK_SHRT(c1, c2) {
        return ((c1 << 8) | c2);
    }
    static MK_LONG(c1, c2, c3, c4) {
        return ((c1 << 24) | (c2 << 16) | (c3 << 8) | c4);
    }
    eval_op_chkdgt(resVal, val1, val2) {
        let weight_vals = [
            String.fromCharCode(1), String.fromCharCode(2), String.fromCharCode(5), String.fromCharCode(3), String.fromCharCode(6), String.fromCharCode(4), String.fromCharCode(8),
            String.fromCharCode(7), String.fromCharCode(10), String.fromCharCode(9)
        ];
        let pos;
        let mul;
        let c_str;
        let digits;
        if (val1.StrVal === null || val2.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.NUMERIC);
            return;
        }
        let mode = val2.MgNumVal.NUM_2_LONG();
        let res = 0;
        switch (mode) {
            case 0:
                mul = 2;
                for (pos = val1.StrVal.length; pos >= 1; pos--) {
                    c_str = val1.StrVal.substr(pos - 1);
                    digits = this.a_2_long(c_str, 1) * mul;
                    res += digits + (digits > 9 ? 1 : 0);
                    mul = 3 - mul;
                }
                res %= 10;
                if (res !== 0)
                    res = 10 - res;
                break;
            case 1:
                for (pos = val1.StrVal.length; pos >= 1; pos--) {
                    mul = weight_vals[(val1.StrVal.length - pos) % 10].charCodeAt(0);
                    let c_char = val1.StrVal[pos - 1].toUpperCase();
                    c_str = val1.StrVal.substr(pos - 1);
                    if (UtilStrByteMode.isDigit(c_char))
                        digits = this.a_2_long(c_str, 1) * mul;
                    else if (NChar.IsUpper(c_str[0]))
                        digits = (c_str[0].charCodeAt(0) - 'A'.charCodeAt(0) + 1) * mul;
                    else
                        digits = 0;
                    res += digits;
                }
                res %= 11;
                if (res !== 0)
                    res = 11 - res;
                break;
        }
        super.ConstructMagicNum(resVal, res, StorageAttribute.NUMERIC);
    }
    a_2_long(str, len) {
        let n = 0;
        for (let pos = 0; pos < len; pos = pos + 1) {
            if (UtilStrByteMode.isDigit(str.charAt(pos))) {
                n = n * 10;
                n += (str.charCodeAt(pos) - '0'.charCodeAt(0));
            }
        }
        return n;
    }
    eval_op_soundx(resVal, val1) {
        let soundx_vals = [
            '0', '1', '2', '3', '0', '1', '2', '0', '0', '2', '2', '4', '5', '5', '0', '1', '2',
            '6',
            '2', '3', '0', '1', '0', '2', '0', '2'
        ];
        if (val1.StrVal === null) {
            super.SetNULL(resVal, StorageAttribute.ALPHA);
            return;
        }
        resVal.Attr = StorageAttribute.ALPHA;
        resVal.StrVal = '0000';
        let lastc = ' ';
        let outpos = 0;
        for (let inpos = 0; inpos < val1.StrVal.length; inpos++) {
            let inc = val1.StrVal.charAt(inpos).toUpperCase();
            let outc = soundx_vals[(inc.charCodeAt(0) - 'A'.charCodeAt(0))];
            if (inpos === 0)
                resVal.StrVal = ExpressionEvaluator.setAt(resVal.StrVal, outpos++, inc);
            else if (outc > '0' && outc !== lastc) {
                resVal.StrVal = ExpressionEvaluator.setAt(resVal.StrVal, outpos++, outc);
                if (outpos > 3)
                    break;
            }
            lastc = outc;
        }
    }
    static setAt(str, pos, ch) {
        let buffer = new StringBuilder(str);
        if (pos < str.length) {
            buffer.set_Item(pos, ch);
        }
        else {
            buffer.Append(ch);
        }
        return buffer.ToString();
    }
    eval_op_hstr(resVal, val1) {
        let num16 = new NUM_TYPE();
        let newnum;
        let digit;
        let outstr = new Array(30);
        let tmpOutStr = new StringBuilder(outstr.length);
        let negative = false;
        resVal.Attr = StorageAttribute.ALPHA;
        if (val1.MgNumVal == null) {
            super.SetNULL(resVal, StorageAttribute.ALPHA);
            return;
        }
        num16.num_4_a_std('16');
        let orgnum = new NUM_TYPE(val1.MgNumVal);
        if (orgnum.num_is_neg()) {
            negative = true;
            orgnum.num_abs();
        }
        let digits = 0;
        while (true) {
            newnum = NUM_TYPE.mod(orgnum, num16);
            orgnum = NUM_TYPE.div(orgnum, num16);
            orgnum.num_trunc(0);
            digit = newnum.NUM_2_LONG();
            digits++;
            ExpressionEvaluator.int_2_hex(outstr, outstr.length - digits, 1, digit, 0);
            if (orgnum.num_is_zero())
                break;
        }
        if (negative) {
            digits++;
            outstr[outstr.length - digits] = '-';
        }
        for (digit = outstr.length - digits; digit < outstr.length; digit++)
            tmpOutStr.Append(outstr[digit]);
        resVal.StrVal = tmpOutStr.ToString();
    }
    static int_2_hex(str, strPos, len, n, lead) {
        let pos = len;
        do {
            let digit = n % 16;
            if (digit < 10)
                str[--pos + strPos] = String.fromCharCode('0'.charCodeAt(0) + digit);
            else
                str[--pos + strPos] = String.fromCharCode('A'.charCodeAt(0) + digit - 10);
            n = Math.floor(n / 16);
        } while (pos > 0 && n !== 0);
        ExpressionEvaluator.lib_a_fill(str, len, pos + strPos, lead);
        return;
    }
    static lib_a_fill(str, len, pos, lead) {
        if (lead === 0) {
            len -= pos;
            if (len > 0 && pos > 0) {
                StrUtil.memcpy(str, 0, str, pos, len);
                StrUtil.memset(str, len, ' ', pos);
            }
        }
        else {
            if (pos > 0)
                StrUtil.memset(str, 0, String.fromCharCode(lead), pos);
        }
        return;
    }
    eval_op_hval(resVal, val1) {
        let num16 = new NUM_TYPE();
        let num = new NUM_TYPE();
        let digits;
        let state = 0;
        let negative = false;
        if (val1.StrVal === null) {
            super.SetNULL(resVal, StorageAttribute.NUMERIC);
            return;
        }
        num16.num_4_a_std('16');
        resVal.MgNumVal = new NUM_TYPE();
        resVal.MgNumVal.NUM_ZERO();
        for (digits = 0; digits < val1.StrVal.length; digits++) {
            let digitc = val1.StrVal[digits];
            if (digitc === '-' && state === 0)
                negative = true;
            let digit = this.hex_2_long(val1.StrVal, digits, 1);
            if (digit === 0 && digitc !== '0')
                continue;
            state = 1;
            resVal.MgNumVal = NUM_TYPE.mul(resVal.MgNumVal, num16);
            num.NUM_4_LONG(digit);
            resVal.MgNumVal = NUM_TYPE.add(resVal.MgNumVal, num);
        }
        if (negative)
            resVal.MgNumVal.num_neg();
        resVal.Attr = StorageAttribute.NUMERIC;
    }
    hex_2_long(str, strCount, len) {
        let pos;
        let n = 0;
        for (pos = strCount; pos < strCount + len; pos++) {
            let digit = str.charAt(pos).toUpperCase();
            if (UtilStrByteMode.isDigit(digit)) {
                n *= 16;
                n += digit.charCodeAt(0) - '0'.charCodeAt(0);
            }
            else if (digit >= 'A' && digit <= 'F') {
                n *= 16;
                n += digit.charCodeAt(0) - 'A'.charCodeAt(0) + 10;
            }
        }
        return n;
    }
    eval_op_chr(resVal, val1) {
        resVal.Attr = StorageAttribute.ALPHA;
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.ALPHA);
            return;
        }
        resVal.StrVal = String.fromCharCode(val1.MgNumVal.NUM_2_LONG())[0];
    }
    eval_op_asc(resVal, val1) {
        if (val1.StrVal === null) {
            super.SetNULL(resVal, StorageAttribute.NUMERIC);
            return;
        }
        let c = (val1.StrVal.length > 0) ? (Encoding.ASCII.GetBytes(val1.StrVal))[0] : 0;
        this.ConstructMagicNum(resVal, c, StorageAttribute.NUMERIC);
    }
    eval_op_mstr(resVal, val1, val2) {
        resVal.Attr = StorageAttribute.ALPHA;
        if (val1.MgNumVal == null && val2.MgNumVal == null) {
            super.SetNULL(resVal, StorageAttribute.ALPHA);
            return;
        }
        let num = new NUM_TYPE(val1.MgNumVal);
        if (num.NUM_IS_LONG())
            num.num_4_std_long();
        let len = val2.MgNumVal.NUM_2_LONG();
        len = Math.max(len, 2);
        let tmpArray = new Int8Array(len);
        for (let i = 0; i < len; i++)
            tmpArray[i] = num.Data[i];
        let tmpBytes = Misc.ToByteArray(tmpArray);
        resVal.StrVal = Environment.Instance.GetEncoding().GetString(tmpBytes, 0, tmpBytes.length);
    }
    eval_op_mval(resVal, val1) {
        resVal.Attr = StorageAttribute.NUMERIC;
        if (val1.StrVal == null) {
            super.SetNULL(resVal, StorageAttribute.NUMERIC);
            return;
        }
        resVal.MgNumVal = new NUM_TYPE();
        resVal.MgNumVal.NUM_SET_ZERO();
        let len = Math.min(val1.StrVal.length, NUM_TYPE.NUM_SIZE);
        resVal.MgNumVal = new NUM_TYPE(Misc.ToSByteArray(Environment.Instance.GetEncoding().GetBytes(val1.StrVal)), 0, len);
    }
    eval_op_dstr(resVal, val1, val2, displayConvertor) {
        resVal.Attr = StorageAttribute.UNICODE;
        if (val1.MgNumVal == null || val2.StrVal == null) {
            super.SetNULL(resVal, StorageAttribute.UNICODE);
            return;
        }
        resVal.StrVal = displayConvertor.to_a(resVal.StrVal, 100, val1.MgNumVal.NUM_2_ULONG(), val2.StrVal, this.ExpTask.getCompIdx());
    }
    eval_op_tstr(resVal, val1, val2, displayConvertor, milliSeconds) {
        resVal.Attr = StorageAttribute.ALPHA;
        if (val1.MgNumVal == null || val2.StrVal == null) {
            super.SetNULL(resVal, StorageAttribute.ALPHA);
            return;
        }
        resVal.StrVal = displayConvertor.time_2_a(resVal.StrVal, 100, val1.MgNumVal.NUM_2_ULONG(), val2.StrVal, this.ExpTask.getCompIdx(), milliSeconds);
    }
    eval_op_dval(resVal, val1, val2, displayConvertor) {
        if (val1.StrVal === null || val2.StrVal === null) {
            super.SetNULL(resVal, StorageAttribute.DATE);
            return;
        }
        if (val1.Attr === StorageAttribute.UNICODE)
            if (val1.StrVal.length + 2 <= NUM_TYPE.NUM_SIZE) {
            }
            else {
                let tmp = new Array(val1.StrVal.length + 2);
                for (let _ai = 0; _ai < tmp.length; ++_ai)
                    tmp[_ai] = '\0';
                StrUtil.memcpy(tmp, 0, NString.ToCharArray(val1.StrVal), 0, val1.StrVal.length);
            }
        let l = displayConvertor.a_2_date(val1.StrVal, val2.StrVal, this.ExpTask.getCompIdx());
        if (l >= 1000000000)
            l = 0;
        super.ConstructMagicNum(resVal, l, StorageAttribute.DATE);
    }
    eval_op_tval(resVal, val1, val2, displayConvertor, milliSeconds) {
        if (val1.StrVal === null || val2.StrVal === null) {
            super.SetNULL(resVal, StorageAttribute.TIME);
            return;
        }
        let pic = new PIC(val2.StrVal, StorageAttribute.TIME, this.ExpTask.getCompIdx());
        let l = displayConvertor.a_2_time(val1.StrVal, pic, milliSeconds);
        if (l >= 1000000000)
            l = 0;
        super.ConstructMagicNum(resVal, l, StorageAttribute.TIME);
    }
    eval_op_day(resVal, val1) {
        this.eval_op_date_brk(resVal, val1.MgNumVal, 0);
    }
    eval_op_month(resVal, val1) {
        this.eval_op_date_brk(resVal, val1.MgNumVal, 1);
    }
    eval_op_year(resVal, val1) {
        this.eval_op_date_brk(resVal, val1.MgNumVal, 2);
    }
    eval_op_dow(resVal, val1) {
        this.eval_op_date_brk(resVal, val1.MgNumVal, 3);
    }
    eval_op_second(resVal, val1) {
        this.eval_op_time_brk(resVal, val1.MgNumVal, 2);
    }
    eval_op_minute(resVal, val1) {
        this.eval_op_time_brk(resVal, val1.MgNumVal, 1);
    }
    eval_op_hour(resVal, val1) {
        this.eval_op_time_brk(resVal, val1.MgNumVal, 0);
    }
    eval_op_date_brk(resVal, val1, typ) {
        if (val1 === null) {
            super.SetNULL(resVal, StorageAttribute.NUMERIC);
            return;
        }
        let d = val1.NUM_2_LONG();
        let breakParams = DisplayConvertor.Instance.getNewDateBreakParams();
        DisplayConvertor.Instance.date_break_datemode(breakParams, d, false, this.ExpTask.getCompIdx());
        let year = breakParams.year;
        let month = breakParams.month;
        let day = breakParams.day;
        let doy = breakParams.doy;
        let dow = breakParams.dow;
        switch (typ) {
            case 0:
                d = day;
                break;
            case 1:
                d = month;
                break;
            case 2:
                d = year;
                break;
            case 3:
                d = dow;
                break;
            case 4:
                d = UtilDateJpn.getInstance().date_jpn_year_ofs(year, doy);
                break;
            default:
                d = 0;
                break;
        }
        super.ConstructMagicNum(resVal, d, StorageAttribute.NUMERIC);
    }
    eval_op_time_brk(resVal, val1, typ) {
        if (val1 === null) {
            super.SetNULL(resVal, StorageAttribute.NUMERIC);
            return;
        }
        let d = val1.NUM_2_ULONG();
        let breakParams = DisplayConvertor.Instance.getNewTimeBreakParams();
        DisplayConvertor.time_break(breakParams, d);
        let hour = breakParams.hour;
        let minute = breakParams.minute;
        let second = breakParams.second;
        switch (typ) {
            case 0:
                d = hour;
                break;
            case 1:
                d = minute;
                break;
            case 2:
                d = second;
                break;
            default:
                d = 0;
                break;
        }
        super.ConstructMagicNum(resVal, d, StorageAttribute.NUMERIC);
    }
    async eval_op_addDateTime(resVal, dateVal, timeVal, yearsVal, monthsVal, daysVal, hoursVal, minutesVal, secondsVal) {
        let tmpVal = new ExpVal();
        if (dateVal.MgNumVal === null || timeVal.MgNumVal === null) {
            resVal.Attr = StorageAttribute.BOOLEAN;
            resVal.BoolVal = false;
            return;
        }
        let fldTime = this.GetFieldOfContextTask(timeVal.MgNumVal.NUM_2_LONG());
        super.SetVal(tmpVal, fldTime.getType(), fldTime.getValue(true), null);
        this.eval_op_addtime(resVal, tmpVal, hoursVal, minutesVal, secondsVal);
        let time = resVal.MgNumVal.NUM_2_LONG();
        let date = Math.floor(time / (60 * 60 * 24));
        time = time % (60 * 60 * 24);
        if (time < 0) {
            date--;
            time = (60 * 60 * 24) - (-time);
        }
        let fldDate = this.GetFieldOfContextTask(dateVal.MgNumVal.NUM_2_LONG());
        super.SetVal(tmpVal, fldDate.getType(), fldDate.getValue(true), null);
        this.eval_op_adddate(resVal, tmpVal, yearsVal, monthsVal, daysVal);
        date += resVal.MgNumVal.NUM_2_LONG();
        tmpVal.MgNumVal.NUM_4_LONG(time);
        await fldTime.setValueAndStartRecompute(tmpVal.ToMgVal(), false, true, false, false);
        await fldTime.updateDisplay();
        tmpVal.MgNumVal.NUM_4_LONG(date);
        await fldDate.setValueAndStartRecompute(tmpVal.ToMgVal(), false, true, false, false);
        await fldDate.updateDisplay();
        resVal.Attr = StorageAttribute.BOOLEAN;
        resVal.BoolVal = true;
    }
    async eval_op_difdt(resVal, date1val, time1val, date2val, time2val, difDateVal, difTimeVal) {
        let tmpVal = new ExpVal();
        tmpVal.MgNumVal = new NUM_TYPE();
        tmpVal.Attr = StorageAttribute.NUMERIC;
        if (difDateVal.MgNumVal === null || difTimeVal.MgNumVal === null) {
            resVal.Attr = StorageAttribute.BOOLEAN;
            resVal.BoolVal = false;
            return;
        }
        let difDate = date1val.MgNumVal.NUM_2_LONG() - date2val.MgNumVal.NUM_2_LONG();
        let difTime = time1val.MgNumVal.NUM_2_LONG() - time2val.MgNumVal.NUM_2_LONG();
        if ((difTime < 0) && (difTime > -86400) && (difDate > 0)) {
            difDate--;
            difTime = 86400 - (-difTime);
        }
        let fldDate = this.GetFieldOfContextTask(difDateVal.MgNumVal.NUM_2_LONG());
        let fldTime = this.GetFieldOfContextTask(difTimeVal.MgNumVal.NUM_2_LONG());
        tmpVal.MgNumVal.NUM_4_LONG(difDate);
        await fldDate.setValueAndStartRecompute(tmpVal.ToMgVal(), false, true, false, false);
        await fldDate.updateDisplay();
        tmpVal.MgNumVal.NUM_4_LONG(difTime);
        await fldTime.setValueAndStartRecompute(tmpVal.ToMgVal(), false, true, false, false);
        await fldTime.updateDisplay();
        resVal.Attr = StorageAttribute.BOOLEAN;
        resVal.BoolVal = true;
    }
    eval_op_ndow(resVal, val1, displayConvertor) {
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.NUMERIC);
            return;
        }
        val1.MgNumVal = this.mul_add(val1.MgNumVal, 0, 6);
        this.eval_op_cdow(resVal, val1.MgNumVal, displayConvertor);
    }
    eval_op_nmonth(resVal, val1, displayConvertor) {
        if (val1.MgNumVal == null) {
            super.SetNULL(resVal, StorageAttribute.NUMERIC);
            return;
        }
        val1.MgNumVal = this.mul_add(val1.MgNumVal, 31, -30);
        this.eval_op_cmonth(resVal, val1.MgNumVal, displayConvertor);
    }
    mul_add(num, mul, add) {
        let tmp = new NUM_TYPE();
        if (num === null)
            return null;
        if (mul !== 0) {
            tmp.NUM_4_LONG(mul);
            num = NUM_TYPE.mul(num, tmp);
        }
        if (add !== 0) {
            tmp.NUM_4_LONG(add);
            num = NUM_TYPE.add(num, tmp);
        }
        return num;
    }
    eval_op_cdow(resVal, val1, displayConvertor) {
        this.eval_op_date_str(resVal, val1, 'WWWWWWWWWWT', displayConvertor);
    }
    eval_op_cmonth(resVal, val1, displayConvertor) {
        this.eval_op_date_str(resVal, val1, 'MMMMMMMMMMT', displayConvertor);
    }
    eval_op_date_str(resVal, val1, format, displayConvertor) {
        resVal.Attr = StorageAttribute.ALPHA;
        if (val1 === null) {
            super.SetNULL(resVal, StorageAttribute.ALPHA);
            return;
        }
        let dow = val1.NUM_2_ULONG();
        resVal.StrVal = displayConvertor.to_a(resVal.StrVal, 10, dow, format, this.ExpTask.getCompIdx());
    }
    eval_op_delay() {
    }
    eval_op_idle(resVal) {
        let n = 0;
        let idleTime = Environment.Instance.getIdleTime(this.ExpTask.getCompIdx());
        if (idleTime > 0) {
            let CurrTimeMilli = Misc.getSystemMilliseconds();
            n = Math.floor(Math.floor((CurrTimeMilli - LastFocusedManager.Instance.LastActionTime) / 1000) / idleTime);
        }
        resVal.MgNumVal = new NUM_TYPE();
        resVal.Attr = StorageAttribute.NUMERIC;
        resVal.MgNumVal.NUM_4_LONG(n * idleTime * 10);
    }
    eval_op_adddate(resVal, val1, val2, val3, val4) {
        let tries;
        if (val1.MgNumVal === null || val2.MgNumVal === null || val3.MgNumVal === null || val4.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.DATE);
            return;
        }
        let date = val1.MgNumVal.NUM_2_LONG();
        let breakParams = DisplayConvertor.Instance.getNewDateBreakParams();
        DisplayConvertor.Instance.date_break_datemode(breakParams, date, false, this.ExpTask.getCompIdx());
        let year = breakParams.year;
        let month = breakParams.month;
        let day = breakParams.day;
        if (Environment.Instance.GetDateMode(this.ExpTask.getCompIdx()) === 'B')
            year = Math.max(year - PICInterface.DATE_BUDDHIST_GAP, 0);
        year += val2.MgNumVal.NUM_2_LONG();
        month += val3.MgNumVal.NUM_2_LONG();
        let day1 = val4.MgNumVal.NUM_2_LONG();
        let add_day = (day === 0 && year !== 0 && month !== 0 && day1 !== 0) ? 1 : 0;
        let month1 = month + year * 12;
        year = Math.floor((month1 - 1) / 12);
        month = (month1 - 1) % 12 + 1;
        for (tries = 0; tries < 4; tries++) {
            date = DisplayConvertor.Instance.date_4_calender(year, month, day + add_day, 0, false);
            if (date < 1000000000)
                break;
            day--;
        }
        if (date === 1000000000)
            date = 0;
        else {
            date += day1 - add_day;
            date = Math.max(date, 0);
        }
        super.ConstructMagicNum(resVal, date, StorageAttribute.DATE);
    }
    eval_op_addtime(resVal, val1, val2, val3, val4) {
        if (val1.MgNumVal === null || val2.MgNumVal === null || val3.MgNumVal === null || val4.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.TIME);
            return;
        }
        let time = val1.MgNumVal.NUM_2_ULONG() + val2.MgNumVal.NUM_2_LONG() * 3600 + val3.MgNumVal.NUM_2_LONG() * 60 + val4.MgNumVal.NUM_2_LONG();
        super.ConstructMagicNum(resVal, time, StorageAttribute.TIME);
    }
    eval_op_bom(resVal, val1) {
        let breakParams;
        if (val1.MgNumVal == null) {
            super.SetNULL(resVal, StorageAttribute.DATE);
            return;
        }
        let date = val1.MgNumVal.NUM_2_ULONG();
        if (date !== 0) {
            breakParams = DisplayConvertor.Instance.getNewDateBreakParams();
            DisplayConvertor.Instance.date_break_datemode(breakParams, date, false, this.ExpTask.getCompIdx());
            let day = breakParams.day;
            date -= (day - 1);
        }
        super.ConstructMagicNum(resVal, date, StorageAttribute.DATE);
    }
    eval_op_boy(resVal, val1) {
        let breakParams;
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.DATE);
            return;
        }
        let date = val1.MgNumVal.NUM_2_ULONG();
        if (date !== 0) {
            breakParams = DisplayConvertor.Instance.getNewDateBreakParams();
            DisplayConvertor.Instance.date_break_datemode(breakParams, date, false, this.ExpTask.getCompIdx());
            let year = breakParams.year;
            let day = 1;
            let month = 1;
            if (Environment.Instance.GetDateMode(this.ExpTask.getCompIdx()) === 'B')
                year = Math.max(year - PICInterface.DATE_BUDDHIST_GAP, 0);
            date = DisplayConvertor.Instance.date_4_calender(year, month, day, 0, false);
        }
        super.ConstructMagicNum(resVal, date, StorageAttribute.DATE);
    }
    eval_op_eom(resVal, val1) {
        let breakParams;
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.DATE);
            return;
        }
        let date = val1.MgNumVal.NUM_2_ULONG();
        if (date !== 0) {
            breakParams = DisplayConvertor.Instance.getNewDateBreakParams();
            DisplayConvertor.Instance.date_break_datemode(breakParams, date, false, this.ExpTask.getCompIdx());
            let year = breakParams.year;
            let month = breakParams.month;
            let day = 31;
            if (Environment.Instance.GetDateMode(this.ExpTask.getCompIdx()) === 'B')
                year = Math.max(year - PICInterface.DATE_BUDDHIST_GAP, 0);
            let tries;
            for (tries = 0; tries < 4; tries++) {
                date = DisplayConvertor.Instance.date_4_calender(year, month, day, 0, false);
                if (date < 1000000000)
                    break;
                day--;
            }
        }
        super.ConstructMagicNum(resVal, date, StorageAttribute.DATE);
    }
    eval_op_eoy(resVal, val1) {
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.DATE);
            return;
        }
        let date = val1.MgNumVal.NUM_2_ULONG();
        if (date !== 0) {
            let breakParams = DisplayConvertor.Instance.getNewDateBreakParams();
            DisplayConvertor.Instance.date_break_datemode(breakParams, date, false, this.ExpTask.getCompIdx());
            let year = breakParams.year;
            let month = 12;
            let day = 31;
            if (Environment.Instance.GetDateMode(this.ExpTask.getCompIdx()) === 'B')
                year = Math.max(year - PICInterface.DATE_BUDDHIST_GAP, 0);
            date = DisplayConvertor.Instance.date_4_calender(year, month, day, 0, false);
        }
        super.ConstructMagicNum(resVal, date, StorageAttribute.DATE);
    }
    eval_op_strtok(resVal, val1, val2, val3) {
        let tmp_s;
        let tmp_str;
        let idx;
        let delim;
        let ret_str = '';
        if (val1.StrVal === null || val2.MgNumVal === null || val3.StrVal === null) {
            super.SetNULL(resVal, StorageAttribute.UNICODE);
        }
        if (!NString.IsNullOrEmpty(val1.StrVal) && val3.StrVal.length > 0) {
            tmp_s = new StringBuilder(val1.StrVal.length + 2);
            idx = val2.MgNumVal.NUM_2_LONG();
            if (idx > 0) {
                delim = NString.TrimEnd(val3.StrVal, this._charsToTrim);
                tmp_s.Append(val1.StrVal);
                if (delim.length === 0) {
                    if (idx === 1)
                        ret_str = tmp_s.ToString();
                }
                else {
                    tmp_str = tmp_s.ToString();
                    let i;
                    for (i = 0; i < idx; i++) {
                        ret_str = StrUtil.strstr(tmp_str, delim);
                        if (ret_str === null) {
                            if (i === idx - 1)
                                ret_str = tmp_str;
                            break;
                        }
                        ret_str = tmp_str.substr(0, tmp_str.length - ret_str.length);
                        tmp_str = tmp_str.substr(ret_str.length + delim.length);
                    }
                }
            }
        }
        if (ret_str !== null) {
            resVal.Attr = StorageAttribute.UNICODE;
            resVal.StrVal = ret_str;
        }
        else {
            resVal.Attr = StorageAttribute.UNICODE;
            resVal.StrVal = '';
        }
        idx = val2.MgNumVal.NUM_2_LONG();
        if (!NString.IsNullOrEmpty(val1.StrVal) && val3.StrVal.length === 0 && idx === 1) {
            resVal.Attr = StorageAttribute.UNICODE;
            resVal.StrVal = val1.StrVal;
        }
    }
    eval_op_dbround(resVal, val1, val2) {
        if (val1.MgNumVal === null || val2.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.NUMERIC);
            return;
        }
        resVal.MgNumVal = new NUM_TYPE(val1.MgNumVal);
        let whole = val2.MgNumVal.NUM_2_LONG();
        if (whole < 0)
            resVal.MgNumVal.dbRound(-whole);
        else
            resVal.MgNumVal.round(whole);
        resVal.Attr = StorageAttribute.NUMERIC;
    }
    eval_op_varpic(resVal, val1, val2) {
        let fld;
        let ctrl;
        if (val2.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.UNICODE);
            return;
        }
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.UNICODE);
            return;
        }
        let mode = val2.MgNumVal.NUM_2_LONG();
        let itm = val1.MgNumVal.NUM_2_LONG();
        resVal.Attr = StorageAttribute.UNICODE;
        try {
            fld = this.GetFieldOfContextTask(itm);
        }
        catch (ex) {
            if (ex instanceof Exception) {
                Logger.Instance.WriteExceptionToLog(ex);
                fld = null;
            }
            else
                throw ex;
        }
        if (fld === null) {
            Logger.Instance.WriteExceptionToLogWithMsg('ExpressionEvaluator.eval_op_varpic there is no control number ' + itm);
            resVal.StrVal = '';
            return;
        }
        if (mode !== 0) {
            ctrl = fld.getCtrl();
            if (ctrl !== null) {
                resVal.StrVal = ctrl.getPIC().getFormat();
                return;
            }
        }
        if (fld.getType() !== StorageAttribute.BLOB && fld.getType() !== StorageAttribute.BLOB_VECTOR) {
            resVal.StrVal = fld.getPicture();
            return;
        }
        resVal.StrVal = '';
        return;
    }
    eval_op_varattr(resVal, val1) {
        let fld;
        resVal.Attr = StorageAttribute.ALPHA;
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.ALPHA);
            return;
        }
        let itm = val1.MgNumVal.NUM_2_LONG();
        try {
            fld = this.GetFieldOfContextTask(itm);
        }
        catch (ex) {
            if (ex instanceof Exception) {
                Logger.Instance.WriteExceptionToLog(ex);
                fld = null;
            }
            else
                throw ex;
        }
        if (fld === null) {
            Logger.Instance.WriteExceptionToLogWithMsg('ExpressionEvaluator.eval_op_varattr there is no control number ' + itm);
            resVal.StrVal = '';
            return;
        }
        let attr = fld.getType();
        resVal.StrVal = '' + ExpressionEvaluator.GetAttributeChar(attr);
    }
    exp_null_val_get(exp_attr, opcode, null_parm) {
        let ptr = '';
        let num_val = 0;
        null_parm.IsNull = true;
        if (opcode === ExpressionInterface.EXP_OP_NULL) {
            null_parm.Attr = exp_attr;
            if (exp_attr === StorageAttribute.ALPHA)
                ptr = '';
            switch (exp_attr) {
                case StorageAttribute.ALPHA:
                case StorageAttribute.BLOB:
                case StorageAttribute.BLOB_VECTOR:
                case StorageAttribute.MEMO:
                    opcode = ExpressionInterface.EXP_OP_NULL_A;
                    break;
                case StorageAttribute.NUMERIC:
                    opcode = ExpressionInterface.EXP_OP_NULL_N;
                    break;
                case StorageAttribute.BOOLEAN:
                    opcode = ExpressionInterface.EXP_OP_NULL_B;
                    break;
                case StorageAttribute.DATE:
                    opcode = ExpressionInterface.EXP_OP_NULL_D;
                    break;
                case StorageAttribute.TIME:
                    opcode = ExpressionInterface.EXP_OP_NULL_T;
                    break;
                case StorageAttribute.UNICODE:
                    opcode = ExpressionInterface.EXP_OP_NULL_U;
                    break;
            }
        }
        switch (opcode) {
            case ExpressionInterface.EXP_OP_NULL:
            case ExpressionInterface.EXP_OP_NULL_A:
                if (exp_attr !== StorageAttribute.BLOB &&
                    exp_attr !== StorageAttribute.BLOB_VECTOR)
                    null_parm.Attr = StorageAttribute.ALPHA;
                ptr = '';
                break;
            case ExpressionInterface.EXP_OP_NULL_N:
                null_parm.Attr = StorageAttribute.NUMERIC;
                break;
            case ExpressionInterface.EXP_OP_NULL_B:
                null_parm.Attr = StorageAttribute.BOOLEAN;
                num_val = 0;
                break;
            case ExpressionInterface.EXP_OP_NULL_D:
                null_parm.Attr = StorageAttribute.DATE;
                num_val = NNumber.Parse(PICInterface.DEFAULT_DATE);
                break;
            case ExpressionInterface.EXP_OP_NULL_T:
                null_parm.Attr = StorageAttribute.TIME;
                num_val = NNumber.Parse(PICInterface.DEFAULT_TIME);
                break;
            case ExpressionInterface.EXP_OP_NULL_U:
                if (exp_attr !== StorageAttribute.BLOB &&
                    exp_attr !== StorageAttribute.BLOB_VECTOR)
                    null_parm.Attr = StorageAttribute.UNICODE;
                ptr = '';
                break;
            case ExpressionInterface.EXP_OP_NULL_O:
                null_parm.Attr = StorageAttribute.BLOB;
                break;
        }
        switch (null_parm.Attr) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.BLOB:
            case StorageAttribute.BLOB_VECTOR:
                null_parm.StrVal = ptr;
                break;
            case StorageAttribute.NUMERIC:
            case StorageAttribute.DATE:
            case StorageAttribute.TIME:
                if (opcode === ExpressionInterface.EXP_OP_NULL_N) {
                    let num_value = new Int8Array(NUM_TYPE.NUM_SIZE);
                    for (let i = 0; i < num_value.length; i++)
                        num_value[i] = 0;
                    null_parm.MgNumVal = new NUM_TYPE(num_value);
                }
                else
                    this.ConstructMagicNum(null_parm, num_val, null_parm.Attr);
                break;
            case StorageAttribute.UNICODE:
                null_parm.StrVal = ptr;
                break;
            case StorageAttribute.BOOLEAN:
                null_parm.BoolVal = false;
                break;
        }
    }
    exp_get_var(resValOrVal1, val1OrResVal, is_previous) {
        if (!isNullOrUndefined(is_previous))
            this.exp_get_var_0(resValOrVal1, val1OrResVal, is_previous);
        else
            this.exp_get_var_1(resValOrVal1, val1OrResVal);
    }
    exp_get_var_0(resVal, val1, is_previous) {
        let fld = null;
        if (val1.MgNumVal !== null) {
            let itm = val1.MgNumVal.NUM_2_LONG();
            fld = this.GetFieldOfContextTask(itm);
        }
        if (fld === null) {
            super.SetNULL(resVal, StorageAttribute.NONE);
            return;
        }
        if (fld.IsExposedToClient === false) {
            let sb = new StringBuilder();
            sb.Append(NString.Format('Error: Variable \'{0}\' is not exposed to client.', fld.getVarName()));
            alert(sb.ToString());
            resVal.IsNull = true;
            return;
        }
        if (is_previous)
            fld.getTask().setEvalOldValues(true);
        super.SetVal(resVal, fld.getType(), fld.getValue(true), null);
        if (is_previous) {
            if (fld.isOriginalValueNull())
                super.SetNULL(resVal, StorageAttribute.NONE);
        }
        else
            resVal.IsNull = fld.isNull();
        if (is_previous)
            fld.getTask().setEvalOldValues(false);
    }
    eval_op_varmod(resVal, val1) {
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.BOOLEAN);
            return;
        }
        let itm = val1.MgNumVal.NUM_2_LONG();
        let fld = this.GetFieldOfContextTask(itm);
        resVal.Attr = StorageAttribute.BOOLEAN;
        resVal.BoolVal = false;
        if (fld !== null) {
            let idx = fld.getId();
            let dv = fld.getTask().DataView;
            let rec = dv.getCurrRec();
            resVal.BoolVal = !rec.fldValsEqual(dv.getOriginalRec(), idx);
        }
    }
    eval_op_varinp(resVal, val1) {
        let i = 0;
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.NUMERIC);
            return;
        }
        let expkern_parent = val1.MgNumVal.NUM_2_LONG();
        if ((expkern_parent >= 0 && expkern_parent < (this.ExpTask.getTaskDepth(false))) || expkern_parent === ExpressionEvaluator.TRIGGER_TASK) {
            let tsk = super.GetContextTask(expkern_parent);
            if (tsk !== null)
                i = tsk.ctl_itm_4_parent_vee(0, tsk.getCurrFieldIdx() + 1);
        }
        super.ConstructMagicNum(resVal, i, StorageAttribute.NUMERIC);
    }
    eval_op_varname(resVal, val1) {
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.ALPHA);
            return;
        }
        let itm = val1.MgNumVal.NUM_2_LONG();
        let fld = this.GetFieldOfContextTask(itm);
        let buffer = (fld !== null) ? fld.getName() : '';
        resVal.StrVal = buffer;
        resVal.Attr = StorageAttribute.ALPHA;
    }
    eval_op_VarDisplayName(resVal, val1) {
        resVal.Attr = StorageAttribute.UNICODE;
        resVal.StrVal = '';
        if (val1.MgNumVal !== null) {
            let itm = val1.MgNumVal.NUM_2_LONG();
            let fld = this.GetFieldOfContextTask(itm);
            if (fld !== null)
                resVal.StrVal = fld.VarDisplayName;
        }
    }
    async eval_op_controlItemRefresh(val1, val2, resVal) {
        let success = false;
        let parent = val2.MgNumVal.NUM_2_LONG();
        resVal.Attr = StorageAttribute.BOOLEAN;
        if ((parent >= 0 && parent < (this.ExpTask.getTaskDepth(false))) || parent === ExpressionEvaluator.TRIGGER_TASK) {
            let tsk = super.GetContextTask(parent);
            if (tsk != null && tsk.getForm() != null) {
                let control = tsk.getForm().GetCtrl(val1.StrVal);
                if (control != null && control.isChoiceControl() && control.isDataCtrl()) {
                    let command = CommandFactory.CreateControlItemsRefreshCommand(tsk.getTaskTag(), control);
                    await tsk.DataviewManager.CurrentDataviewManager.Execute(command);
                    success = true;
                }
            }
        }
        resVal.BoolVal = success;
    }
    eval_op_VarControlID(resVal, val1) {
        let ret = 0;
        if (val1.MgNumVal !== null) {
            let itm = val1.MgNumVal.NUM_2_LONG();
            if (itm === ExpressionEvaluator.TRIGGER_TASK)
                itm = (this.ExpTask.GetContextTask().ctl_itm_4_parent_vee(0, this.ExpTask.GetContextTask().getCurrFieldIdx() + 1));
            ret = this.ExpTask.GetContextTask().GetControlIDFromVarItem(itm - 1);
        }
        resVal.Attr = StorageAttribute.NUMERIC;
        resVal.MgNumVal = new NUM_TYPE();
        resVal.MgNumVal.NUM_4_LONG(ret);
    }
    async eval_op_ControlItemsList(resVal, val1) {
        resVal.Attr = StorageAttribute.UNICODE;
        resVal.StrVal = '';
        if (val1.MgNumVal !== null) {
            let controlID = val1.MgNumVal.NUM_2_LONG();
            let parent = 0;
            let refParent = new RefParam(parent);
            let mgControl = (this.ExpTask.GetContextTask().GetControlFromControlID(controlID - 1, refParent));
            parent = refParent.value;
            if (mgControl !== null && mgControl.isChoiceControl())
                resVal.StrVal = await mgControl.getForm().GetChoiceControlItemList(mgControl);
        }
    }
    async eval_op_ControlDisplayList(resVal, val1) {
        resVal.Attr = StorageAttribute.UNICODE;
        resVal.StrVal = '';
        if (val1.MgNumVal !== null) {
            let controlID = val1.MgNumVal.NUM_2_LONG();
            let parent = 0;
            let refParent = new RefParam(parent);
            let mgControl = this.ExpTask.GetContextTask().GetControlFromControlID(controlID - 1, refParent);
            parent = refParent.value;
            if (mgControl !== null && mgControl.isChoiceControl())
                resVal.StrVal = await mgControl.getForm().GetChoiceControlDisplayList(mgControl);
        }
    }
    eval_op_viewmod(resVal, val1) {
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.BOOLEAN);
            return;
        }
        let parent = val1.MgNumVal.NUM_2_LONG();
        if ((parent >= 0 && parent < (this.ExpTask.getTaskDepth(false))) || parent === ExpressionEvaluator.TRIGGER_TASK) {
            let tsk = super.GetContextTask(parent);
            if (!tsk.isMainProg()) {
                resVal.BoolVal = (tsk.DataView.getCurrRec()).Modified;
                if (!resVal.BoolVal) {
                    let currRec = tsk.DataView.getCurrRec();
                    let originalRec = tsk.DataView.getOriginalRec();
                    resVal.BoolVal = !currRec.isSameRecData(originalRec, true, true);
                }
            }
            else {
                resVal.BoolVal = false;
            }
        }
        else
            resVal.BoolVal = false;
        resVal.Attr = StorageAttribute.BOOLEAN;
    }
    eval_op_level(resVal, val1) {
        let outstr = '';
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.ALPHA);
            return;
        }
        let parent = val1.MgNumVal.NUM_2_LONG();
        if ((parent >= 0 && parent < (this.ExpTask.getTaskDepth(false))) || parent === ExpressionEvaluator.TRIGGER_TASK) {
            let tsk = super.GetContextTask(parent);
            outstr = tsk.getBrkLevel();
            if (parent !== ExpressionEvaluator.TRIGGER_TASK && tsk !== this.ExpTask) {
                if (outstr.toUpperCase() === ConstInterface.BRK_LEVEL_REC_MAIN.toUpperCase()) {
                    if (tsk.isMainProg()) {
                        outstr = ConstInterface.BRK_LEVEL_MAIN_PROG;
                    }
                    else {
                        let tskTree = new Array((this.ExpTask.getTaskDepth(false)));
                        for (let _ai = 0; _ai < tskTree.length; ++_ai)
                            tskTree[_ai] = null;
                        this.ExpTask.pathToRoot(tskTree, false);
                        if (parent > 0 && tskTree[parent - 1].IsSubForm)
                            outstr = ConstInterface.BRK_LEVEL_SUBFORM;
                    }
                }
            }
        }
        resVal.StrVal = outstr;
        resVal.Attr = StorageAttribute.ALPHA;
    }
    eval_op_counter(resVal, val1) {
        let num = 0;
        let parent = val1.MgNumVal.NUM_2_LONG();
        if ((parent >= 0 && parent < this.ExpTask.getTaskDepth(false)) || parent === ExpressionEvaluator.TRIGGER_TASK) {
            let task = super.GetContextTask(parent);
            num = ((task === null) ? 0 : task.getCounter());
        }
        resVal.Attr = StorageAttribute.NUMERIC;
        resVal.MgNumVal = new NUM_TYPE();
        resVal.MgNumVal.NUM_4_LONG(num);
    }
    eval_op_emptyDataview(resVal, val1) {
        let ret = false;
        let parent = val1.MgNumVal.NUM_2_LONG();
        if ((parent >= 0 && parent < this.ExpTask.getTaskDepth(false)) || parent === ExpressionEvaluator.TRIGGER_TASK) {
            let tsk = super.GetContextTask(parent);
            if (tsk !== null && tsk.DataView.isEmptyDataview())
                ret = true;
        }
        resVal.Attr = StorageAttribute.BOOLEAN;
        resVal.BoolVal = ret;
    }
    eval_op_mainlevel(resVal, val1) {
        let outstr = '';
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.ALPHA);
            return;
        }
        let parent = val1.MgNumVal.NUM_2_LONG();
        if ((parent >= 0 && parent < (this.ExpTask.getTaskDepth(false))) || parent === ExpressionEvaluator.TRIGGER_TASK) {
            let tsk = super.GetContextTask(parent);
            outstr = tsk.getMainLevel();
        }
        resVal.StrVal = outstr;
        resVal.Attr = StorageAttribute.ALPHA;
    }
    eval_op_maindisplay(resVal, val1) {
        let mainDspIdx = 0;
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.NUMERIC);
            return;
        }
        let parent = val1.MgNumVal.NUM_2_LONG();
        if ((parent >= 0 && parent < (this.ExpTask.getTaskDepth(false))) || parent === ExpressionEvaluator.TRIGGER_TASK) {
            let tsk = super.GetContextTask(parent);
            mainDspIdx = (tsk.getProp(PropInterface.PROP_TYPE_MAIN_DISPLAY)).GetComputedValueInteger();
        }
        super.ConstructMagicNum(resVal, mainDspIdx, StorageAttribute.NUMERIC);
    }
    eval_op_IsFirstRecordCycle(resVal, val1) {
        let isFirstRecCycle = false;
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.ALPHA);
            return;
        }
        let parent = val1.MgNumVal.NUM_2_LONG();
        if ((parent >= 0 && parent < (this.ExpTask.getTaskDepth(false))) || parent === ExpressionEvaluator.TRIGGER_TASK) {
            let tsk = super.GetContextTask(parent);
            isFirstRecCycle = tsk.isFirstRecordCycle();
        }
        resVal.BoolVal = isFirstRecCycle;
        resVal.Attr = StorageAttribute.BOOLEAN;
    }
    exp_is_default(resVal, val1) {
        resVal.Attr = StorageAttribute.BOOLEAN;
        resVal.BoolVal = false;
        let itm = val1.MgNumVal.NUM_2_LONG();
        let fld = (this.ExpTask.ctl_itm_2_parent_vee(itm - 1));
        if (fld === null)
            return;
        if (fld.isNull() && (fld.isNullDefault() || fld.getType() === StorageAttribute.BLOB_VECTOR))
            resVal.BoolVal = true;
        let val = fld.getValue(false);
        let defVal = fld.getDefaultValue();
        let type = fld.getType();
        resVal.BoolVal = ExpressionEvaluator.mgValsEqual(val, fld.isNull(), type, defVal, fld.isNullDefault(), type);
    }
    eval_op_IsLoggedIn(resVal) {
        resVal.Attr = StorageAttribute.BOOLEAN;
        resVal.BoolVal = UserDetails.Instance.getIsLoggedIn();
    }
    eval_op_appname(resVal) {
        resVal.Attr = StorageAttribute.ALPHA;
        resVal.StrVal = ServerConfig.Instance.getAppName();
    }
    eval_op_prog(resVal) {
        resVal.Attr = StorageAttribute.ALPHA;
        resVal.StrVal = ((this.ExpTask.queryTaskPath())).ToString();
    }
    eval_op_this(resVal) {
        let triggerTask = this.ExpTask;
        let Result = 0;
        if (triggerTask !== null) {
            Result = ExpressionEvaluator.TRIGGER_TASK;
        }
        super.ConstructMagicNum(resVal, Result, StorageAttribute.NUMERIC);
    }
    eval_op_stat(resVal, Parent, Modes) {
        let Ret = false;
        let iParent = Parent.MgNumVal.NUM_2_LONG();
        if ((iParent >= 0 && iParent < (this.ExpTask.getTaskDepth(false))) || iParent === ExpressionEvaluator.TRIGGER_TASK) {
            let Tsk = super.GetContextTask(iParent);
            if (Tsk !== null) {
                let tskMode = (Tsk.getMode()).toUpperCase();
                for (let i = 0; i < Modes.StrVal.length; i = i + 1) {
                    let mode = Modes.StrVal.charAt(i).toUpperCase();
                    let code = this.cst_code_trans_buf('I', 'MCDEPLRKSONB', mode, MsgInterface.EXPTAB_TSK_MODE_RT);
                    if (code === '\0') {
                        code = mode.toUpperCase();
                        if (code === 'Q')
                            code = 'E';
                    }
                    if (code === tskMode) {
                        Ret = true;
                        break;
                    }
                }
            }
        }
        resVal.Attr = StorageAttribute.BOOLEAN;
        resVal.BoolVal = Ret;
    }
    eval_op_subformExecMode(resVal, generation) {
        let subformExecMode = Task_SubformExecModeEnum.NO_SUBFORM;
        let iParent = generation.MgNumVal.NUM_2_LONG();
        if ((iParent >= 0 && iParent < (this.ExpTask.getTaskDepth(false))) || iParent === ExpressionEvaluator.TRIGGER_TASK) {
            let task = super.GetContextTask(iParent);
            if (task !== null) {
                if (task.IsSubForm)
                    subformExecMode = task.SubformExecMode;
            }
        }
        super.ConstructMagicNum(resVal, subformExecMode, StorageAttribute.NUMERIC);
    }
    async eval_op_varset(resVal, val, num) {
        resVal.BoolVal = true;
        resVal.Attr = StorageAttribute.BOOLEAN;
        if (num.MgNumVal === null) {
            return;
        }
        let itm = num.MgNumVal.NUM_2_LONG();
        let fld = this.GetFieldOfContextTask(itm);
        if (fld === null) {
            resVal.BoolVal = false;
            return;
        }
        if (StorageAttributeCheck.StorageFldAlphaUnicodeOrBlob(fld.getType(), val.Attr))
            this.ConvertExpVal(val, fld.getType());
        let bufptr;
        if (StorageAttributeCheck.isTheSameType(fld.getType(), val.Attr)) {
            switch (fld.getType()) {
                case StorageAttribute.ALPHA:
                case StorageAttribute.UNICODE:
                    bufptr = val.StrVal;
                    break;
                case StorageAttribute.NUMERIC:
                case StorageAttribute.DATE:
                case StorageAttribute.TIME:
                    bufptr = val.MgNumVal.toXMLrecord();
                    break;
                case StorageAttribute.BOOLEAN:
                    bufptr = val.BoolVal ? '1' : '0';
                    break;
                case StorageAttribute.BLOB:
                    bufptr = val.ToMgVal();
                    break;
                case StorageAttribute.BLOB_VECTOR:
                    bufptr = val.ToMgVal();
                    if (!val.IsNull) {
                        if (val.Attr === StorageAttribute.BLOB)
                            if (!VectorType.validateBlobContents(val.StrVal))
                                bufptr = null;
                        if (bufptr != null)
                            bufptr = Operation.operUpdateVectors(fld, bufptr);
                        if (bufptr != null)
                            break;
                    }
                default:
                    super.SetNULL(resVal, StorageAttribute.BOOLEAN);
                    return;
            }
        }
        else {
            bufptr = fld.getDefaultValue();
        }
        if (val.IsNull) {
            bufptr = fld.getNullValue();
            if (bufptr === null)
                fld.getMagicDefaultValue();
        }
        let setRecordUpdated = fld.getTask() !== this.ExpTask || fld.getTask().getBrkLevel() === ConstInterface.BRK_LEVEL_REC_SUFFIX;
        await fld.setValueAndStartRecompute(bufptr, val.IsNull, true, setRecordUpdated, false);
        await fld.updateDisplay();
    }
    eval_op_like(source, maskOrg, resVal) {
        let i;
        let j;
        let Source = source.StrVal;
        let MaskOrg = maskOrg.StrVal;
        let asteriskCnt = 0;
        let Same = true;
        let esc_ch;
        if (Source === null || MaskOrg === null) {
            super.SetNULL(resVal, StorageAttribute.BOOLEAN);
            return;
        }
        let SourceLen = Source.length;
        let MaskLen = MaskOrg.length;
        let Mask = new Array(MaskLen);
        let buffer = new StringBuilder(MaskLen);
        for (i = 0, j = 0, esc_ch = false; i < MaskLen; i++) {
            let currChr = MaskOrg[i];
            switch (currChr) {
                case '\\':
                    if (esc_ch)
                        Mask[j++] = currChr;
                    esc_ch = !esc_ch;
                    break;
                case '*':
                    if (esc_ch)
                        Mask[j++] = currChr;
                    else {
                        Mask[j++] = ExpressionEvaluator.ASTERISK_CHR;
                        asteriskCnt++;
                    }
                    esc_ch = false;
                    break;
                case '?':
                    Mask[j++] = esc_ch ? currChr : ExpressionEvaluator.QUESTION_CHR;
                    esc_ch = false;
                    break;
                default:
                    Mask[j++] = currChr;
                    esc_ch = false;
                    break;
            }
        }
        MaskLen = j;
        MaskOrg = ExpressionEvaluator.arrToStr(Mask, 0, Mask.length);
        let ast_last_ptr = MaskOrg.lastIndexOf(ExpressionEvaluator.ASTERISK_CHR);
        if (ast_last_ptr === -1)
            Same = ExpressionEvaluator.op_like_cmp(Source, MaskOrg);
        else {
            for (i = 0; Mask[i] !== ExpressionEvaluator.ASTERISK_CHR && Same; i++, MaskLen--, SourceLen--) {
                if (SourceLen === 0)
                    Same = false;
                else
                    Same = (Mask[i] === ExpressionEvaluator.QUESTION_CHR ? true : Mask[i] === Source[i]);
            }
            Source = Source.substr(i);
            Mask = ExpressionEvaluator.cutArray(Mask, i);
            while (Same && asteriskCnt !== 1) {
                let ast_ptr;
                let tmp_len;
                for (ast_ptr = 1, tmp_len = 0; ast_ptr + i !== ast_last_ptr; ast_ptr++, tmp_len++)
                    if (Mask[ast_ptr] === ExpressionEvaluator.ASTERISK_CHR)
                        break;
                asteriskCnt--;
                SourceLen = Source.length;
                if (tmp_len !== 0) {
                    if (SourceLen > 0) {
                        buffer.Remove(0, buffer.Length);
                        buffer.Append(Source);
                        Same = ExpressionEvaluator.op_like_map(buffer, ExpressionEvaluator.arrToStr(Mask, 1, tmp_len + 1), false);
                        Source = buffer.ToString();
                    }
                    else
                        Same = false;
                }
                i += ast_ptr;
                Mask = ExpressionEvaluator.cutArray(Mask, ast_ptr);
            }
            if (Mask[0] === ExpressionEvaluator.ASTERISK_CHR)
                Mask = ExpressionEvaluator.cutArray(Mask, 1);
            if (Same && (Mask.length > 0)) {
                buffer.Remove(0, buffer.Length);
                buffer.Append(Source);
                Same = ExpressionEvaluator.op_like_map(buffer, ExpressionEvaluator.arrToStr(Mask), true);
            }
        }
        resVal.Attr = StorageAttribute.BOOLEAN;
        resVal.BoolVal = Same;
    }
    static cutArray(source, from) {
        let length = source.length - from;
        let buffer = new Array(length);
        for (let curr = 0; curr < length; curr++)
            buffer[curr] = source[from + curr];
        return buffer;
    }
    eval_op_repstr(source, orgSubstr, newSubstr, resVal) {
        if (source.StrVal === null || orgSubstr.StrVal === null || newSubstr.StrVal === null || !StorageAttributeCheck.IsTypeAlphaOrUnicode(source.Attr) || !StorageAttributeCheck.IsTypeAlphaOrUnicode(orgSubstr.Attr) || !StorageAttributeCheck.IsTypeAlphaOrUnicode(newSubstr.Attr)) {
            super.SetNULL(resVal, StorageAttribute.UNICODE);
            return;
        }
        resVal.StrVal = NString.Replace(source.StrVal, orgSubstr.StrVal, newSubstr.StrVal);
        resVal.Attr = StorageAttribute.UNICODE;
    }
    static arrToStr(arr, from, to) {
        if (arguments.length === 1)
            return ExpressionEvaluator.arrToStr_0(arr);
        else
            return ExpressionEvaluator.arrToStr_1(arr, from, to);
    }
    static arrToStr_0(arr) {
        return ExpressionEvaluator.arrToStr(arr, 0, arr.length);
    }
    static arrToStr_1(arr, from, to) {
        let buffer = new StringBuilder(to - from);
        for (; from < to; from++)
            buffer.Append(arr[from]);
        return buffer.ToString();
    }
    static op_like_cmp(Source, MaskOrg) {
        let Same = true;
        let Mask = MaskOrg;
        let SourceLen = Source.length;
        let MaskLen = MaskOrg.length;
        if (SourceLen < MaskLen)
            Same = false;
        else if (SourceLen > MaskLen)
            Mask = MaskOrg;
        while (Mask.length < SourceLen)
            Mask += ' ';
        for (let i = 0; i < SourceLen && Same; i++)
            Same = (Mask.charAt(i) === ExpressionEvaluator.QUESTION_CHR || Mask.charAt(i) === Source.charAt(i));
        return Same;
    }
    static op_like_map(source, mask, end) {
        let same = false;
        let ptr = source.ToString();
        let i = 0;
        if (!end && source.Length < mask.length)
            return false;
        let j = 0;
        for (j = 0; j < source.Length && !same; j++) {
            same = true;
            if (end) {
                same = ExpressionEvaluator.op_like_cmp(ptr.substr(j), mask);
            }
            else {
                for (i = 0; i < mask.length && same; i++) {
                    if (j + i === source.Length) {
                        return false;
                    }
                    same = (mask.charAt(i) === ExpressionEvaluator.QUESTION_CHR || mask.charAt(i) === ptr.charAt(j + i));
                }
            }
        }
        if (same) {
            source.Remove(0, source.Length);
            source.Append(ptr.substr(j + i - 1));
        }
        return same;
    }
    exp_get_var_1(val1, resVal) {
        let fldName = val1.StrVal;
        let fld = this.ExpTask.getFieldByName(fldName);
        if (fld === null) {
            super.SetNULL(resVal, StorageAttribute.NONE);
            return;
        }
        if (fld.IsExposedToClient === false) {
            let sb = new StringBuilder();
            sb.Append(NString.Format('Error: Variable \'{0}\' is not exposed to client.', fld.getVarName()));
            alert(sb.ToString());
            resVal.IsNull = true;
            return;
        }
        let fldValue = fld.getValue(true);
        resVal.IsNull = fld.isNull();
        super.SetVal(resVal, fld.getType(), fldValue, null);
    }
    exp_get_indx(val1, resVal) {
        let index = this.ExpTask.GetContextTask().getIndexOfFieldByName(val1.StrVal);
        super.ConstructMagicNum(resVal, index, StorageAttribute.NUMERIC);
    }
    getField(itm) {
        let fld = (itm !== ExpressionEvaluator.TRIGGER_TASK) ? (this.ExpTask.ctl_itm_2_parent_vee(itm - 1)) : AccessHelper.eventsManager.getCurrField();
        return fld;
    }
    GetFieldOfContextTask(itm) {
        return (itm !== ExpressionEvaluator.TRIGGER_TASK) ? (this.ExpTask.GetContextTask().ctl_itm_2_parent_vee(itm - 1)) : AccessHelper.eventsManager.getCurrField();
    }
    static trimStr(s, type) {
        let l = 0;
        if (NString.IsNullOrEmpty(s))
            return s;
        let r = s.length - 1;
        if (type !== 'R') {
            while (l < s.length && s.charAt(l) === ' ')
                l++;
        }
        if (type !== 'L') {
            while (r >= l && s.charAt(r) === ' ')
                r--;
        }
        r++;
        return r > l ? s.substr(l, (r) - (l)) : '';
    }
    static mgValsEqual(aVal, aIsNull, aDataType, bVal, bIsNull, bDataType) {
        let a = null;
        let b = null;
        let result = false;
        if (aIsNull === bIsNull && (aIsNull || (aVal !== null && bVal !== null))) {
            try {
                a = new ExpVal(aDataType, aIsNull, aVal);
                b = new ExpVal(bDataType, bIsNull, bVal);
                result = (ExpressionEvaluator.val_cmp_any(a, b) === 0);
            }
            catch (nullValueException) {
                if (nullValueException instanceof NullValueException) {
                    if (a.IsNull && b.IsNull)
                        result = true;
                }
                else
                    throw nullValueException;
            }
        }
        return result;
    }
    eval_op_translate(str, resVal) {
        resVal.Attr = StorageAttribute.ALPHA;
        resVal.StrVal = '';
        let name = str.StrVal;
        if (NString.IsNullOrEmpty(name))
            super.SetNULL(resVal, StorageAttribute.ALPHA);
        else
            resVal.StrVal = EnvParamsTable.Instance.translate(name);
    }
    eval_op_astr(source, format, resVal) {
        resVal.Attr = StorageAttribute.UNICODE;
        if (source.StrVal === null || format.StrVal === null) {
            super.SetNULL(resVal, StorageAttribute.UNICODE);
            return;
        }
        if (format.StrVal.length > 0 && source.StrVal.length > 0) {
            let pic = new PIC(ExpressionEvaluator.set_a_pic(format.StrVal), StorageAttribute.UNICODE, this.ExpTask.getCompIdx());
            resVal.StrVal = DisplayConvertor.Instance.mg2disp(source.ToMgVal(), null, pic, false, false, this.ExpTask.getCompIdx(), true, false);
        }
    }
    static eval_op_vecCellAttr(vec, res) {
        res.Attr = StorageAttribute.ALPHA;
        res.IsNull = false;
        let attr = StorageAttribute.NONE;
        if (vec.IsNull && vec.VectorField !== null) {
            attr = vec.VectorField.getCellsType();
        }
        else if (ExpressionEvaluator.IsValidVector(vec)) {
            attr = VectorType.getCellsAttr(vec.StrVal);
        }
        res.StrVal = '' + ExpressionEvaluator.GetAttributeChar(attr);
    }
    eval_op_vecGet(vec, cell, res) {
        if (cell.MgNumVal === null || !ExpressionEvaluator.IsValidVector(vec) || cell.MgNumVal.NUM_2_LONG() <= 0) {
            res.IsNull = true;
        }
        else {
            let cellAttr;
            let cellVal;
            if (vec.VectorField !== null) {
                cellAttr = vec.VectorField.getCellsType();
                cellVal = vec.VectorField.getVecCellValue(cell.MgNumVal.NUM_2_LONG());
            }
            else {
                let vector = new VectorType(vec.StrVal);
                cellVal = vector.getVecCell(cell.MgNumVal.NUM_2_LONG());
                cellAttr = vector.getCellsAttr();
            }
            if (cellVal == null)
                res.IsNull = true;
            else {
                switch (cellAttr) {
                    case StorageAttribute.ALPHA:
                    case StorageAttribute.MEMO:
                        res.Attr = StorageAttribute.ALPHA;
                        res.StrVal = cellVal;
                        break;
                    case StorageAttribute.UNICODE:
                        res.Attr = cellAttr;
                        res.StrVal = cellVal;
                        break;
                    case StorageAttribute.BLOB:
                        res.Attr = cellAttr;
                        res.StrVal = cellVal;
                        res.IncludeBlobPrefix = true;
                        break;
                    case StorageAttribute.BLOB_VECTOR:
                        res.Attr = cellAttr;
                        res.StrVal = cellVal;
                        res.IncludeBlobPrefix = true;
                        break;
                    case StorageAttribute.NUMERIC:
                        res.Attr = cellAttr;
                        res.MgNumVal = new NUM_TYPE(cellVal);
                        break;
                    case StorageAttribute.DATE:
                        res.Attr = cellAttr;
                        res.MgNumVal = new NUM_TYPE(cellVal);
                        break;
                    case StorageAttribute.TIME:
                        res.Attr = cellAttr;
                        res.MgNumVal = new NUM_TYPE(cellVal);
                        break;
                    case StorageAttribute.BOOLEAN:
                        res.Attr = cellAttr;
                        res.BoolVal = DisplayConvertor.toBoolean(cellVal);
                        break;
                    default:
                        throw new ApplicationException('in ExpressionEvaluator.eval_op_vecGet unknowen storage type: ' +
                            cellAttr);
                }
            }
        }
    }
    eval_op_vecSize(vec, res) {
        res.Attr = StorageAttribute.NUMERIC;
        res.IsNull = false;
        res.MgNumVal = new NUM_TYPE();
        res.MgNumVal.NUM_4_LONG(-1);
        if (ExpressionEvaluator.IsValidVector(vec))
            res.MgNumVal.NUM_4_LONG(new VectorType(vec.StrVal).getVecSize());
    }
    async eval_op_vecSet(vec, cell, newVal, res) {
        res.Attr = StorageAttribute.BOOLEAN;
        res.BoolVal = false;
        if (vec.MgNumVal !== null && cell.MgNumVal !== null) {
            let vecField;
            try {
                vecField = this.getField(vec.MgNumVal.NUM_2_LONG());
            }
            catch (ex) {
                if (ex instanceof Exception) {
                    Logger.Instance.WriteExceptionToLog(ex);
                    vecField = null;
                }
                else
                    throw ex;
            }
            if ((StorageAttributeCheck.IsTypeAlphaOrUnicode(vecField.getCellsType()) && StorageAttributeCheck.IsTypeAlphaOrUnicode(newVal.Attr)) || (vecField !== null && (vecField.IsVirtual || vecField.getTask().getMode() === Constants.TASK_MODE_CREATE || vecField.DbModifiable) && vecField.getType() === StorageAttribute.BLOB_VECTOR && (StorageAttributeCheck.isTheSameType(vecField.getCellsType(), newVal.Attr) || newVal.IsNull || (StorageAttributeCheck.IsTypeAlphaOrUnicode(newVal.Attr) && vecField.getCellsType() === StorageAttribute.BLOB)))) {
                if (StorageAttributeCheck.IsTypeAlphaOrUnicode(newVal.Attr) && vecField.getCellsType() === StorageAttribute.BLOB) {
                    this.ConvertExpVal(newVal, StorageAttribute.BLOB);
                }
                res.BoolVal = await vecField.setCellVecValue(cell.MgNumVal.NUM_2_LONG(), newVal.ToMgVal(), newVal.IsNull);
            }
        }
    }
    eval_op_strTokenCnt(sourceString, delimiter, resVal) {
        resVal.Attr = StorageAttribute.NUMERIC;
        resVal.MgNumVal = new NUM_TYPE();
        let res = 0;
        if (!sourceString.IsNull && sourceString.StrVal !== null && NString.TrimEnd(sourceString.StrVal, this._charsToTrim).length !== 0) {
            if (delimiter.IsNull || NString.TrimEnd(delimiter.StrVal, this._charsToTrim).length === 0)
                res = 1;
            else {
                let tokensSize = this.strTokenCount(sourceString.StrVal, delimiter.StrVal);
                if (tokensSize === 0)
                    res = 1;
                else
                    res = tokensSize;
            }
        }
        resVal.MgNumVal.NUM_4_LONG(res);
    }
    eval_op_strTokenIdx(sourceString, token, delimiter, resVal) {
        resVal.Attr = StorageAttribute.NUMERIC;
        resVal.MgNumVal = new NUM_TYPE();
        let res = 0;
        if (!sourceString.IsNull && sourceString.StrVal !== null && !token.IsNull && token.StrVal !== null && NString.TrimEnd(sourceString.StrVal, this._charsToTrim).length !== 0 && token.StrVal.trim().length > 0) {
            if (!delimiter.IsNull && NString.TrimEnd(delimiter.StrVal, this._charsToTrim).length !== 0)
                res = this.strTokenIndex(sourceString.StrVal, delimiter.StrVal, token.StrVal);
            else if (sourceString.StrVal === token.StrVal)
                res = 1;
            else
                res = 0;
        }
        resVal.MgNumVal.NUM_4_LONG(res);
    }
    strTokenIndex(source, delimiter, token) {
        let trimDelim = NString.TrimEnd(delimiter, this._charsToTrim);
        let trimSource = trimDelim + NString.TrimEnd(source, this._charsToTrim) + trimDelim;
        let trimToken = (token !== null) ? NString.TrimEnd(token, this._charsToTrim) : null;
        if (trimToken === null)
            return 0;
        trimToken = trimDelim + trimToken + trimDelim;
        let tokenOffset = trimSource.indexOf(trimToken);
        if (tokenOffset === -1)
            return 0;
        else {
            let tokenIndex = this.strTokenCount(trimSource.substr(0, tokenOffset + trimDelim.length), trimDelim);
            tokenIndex--;
            return tokenIndex;
        }
    }
    strTokenCount(source, delimiter) {
        let counter = 1;
        let trimDelim = NString.TrimEnd(delimiter, this._charsToTrim);
        let trimSource = NString.TrimEnd(source, this._charsToTrim);
        let delimLength = trimDelim.length;
        let data = trimSource;
        let fromOffset = 0;
        let delimiterOffset = 0;
        if (source === null || source.length === 0)
            return 0;
        delimiterOffset = data.indexOf(trimDelim, fromOffset);
        while (delimiterOffset >= 0) {
            counter++;
            fromOffset = delimiterOffset + delimLength;
            delimiterOffset = data.indexOf(trimDelim, fromOffset);
        }
        return counter;
    }
    eval_op_blobsize(resVal, blobVal) {
        let size = 0;
        switch (blobVal.Attr) {
            case StorageAttribute.BLOB_VECTOR:
                size = VectorType.getVecSize(blobVal.StrVal);
                break;
            case StorageAttribute.BLOB:
                size = BlobType.getBlobSize(blobVal.StrVal);
                break;
            default:
                break;
        }
        resVal.Attr = StorageAttribute.NUMERIC;
        resVal.MgNumVal = new NUM_TYPE();
        resVal.MgNumVal.NUM_4_LONG(size);
    }
    eval_op_iscomponent(resVal) {
        let currTsk = AccessHelper.eventsManager.getCurrTask() || this.ExpTask;
        resVal.Attr = StorageAttribute.BOOLEAN;
        resVal.BoolVal = (currTsk.getCompIdx() !== 0);
    }
    async eval_op_ExecUserDefinedFunc(funcName, Exp_params, resVal, expectedType) {
        let rtEvt = new RunTimeEvent(this.ExpTask);
        rtEvt.setType(ConstInterface.EVENT_TYPE_USER_FUNC);
        rtEvt.setUserDefinedFuncName(funcName);
        let evtHanPos = new EventHandlerPosition();
        evtHanPos.init(rtEvt);
        let handler = evtHanPos.getNext();
        let argsMatch = handler !== null && handler.argsMatch(Exp_params);
        let val = null;
        let valIniated = false;
        if (argsMatch) {
            let handlerContextTask = handler.getTask().GetContextTask();
            if (handler.getTask() === handler.getTask().GetContextTask())
                handler.getTask().SetContextTask(rtEvt.getTask());
            let argList = new ArgumentsList(Exp_params);
            rtEvt.setArgList(argList);
            AccessHelper.eventsManager.pushNewExecStacks();
            await handler.execute(rtEvt, false, false);
            AccessHelper.eventsManager.popNewExecStacks();
            let exp = handler.getTask().getExpById(handler.getEvent().getUserDefinedFuncRetExp());
            if (exp !== null) {
                val = await exp.evaluateWithResType(expectedType);
                valIniated = true;
            }
            handler.getTask().SetContextTask(handlerContextTask);
        }
        if (!valIniated) {
            if (expectedType === StorageAttribute.NONE) {
                expectedType = StorageAttribute.ALPHA;
            }
            val = new ExpVal(expectedType, true, null);
        }
        resVal.Copy(val);
    }
    eval_op_publicName(resVal, Parent) {
        let publicName = 'NULL';
        let iParent = Parent.MgNumVal.NUM_2_LONG();
        if ((iParent >= 0 && iParent < (this.ExpTask.getTaskDepth(false))) || iParent === ExpressionEvaluator.TRIGGER_TASK) {
            let task = super.GetContextTask(iParent);
            if (task !== null && task.isProgram() && !task.isMainProg())
                publicName = task.getPublicName();
        }
        resVal.Attr = StorageAttribute.ALPHA;
        resVal.StrVal = publicName;
    }
    eval_op_taskId(resVal, Parent) {
        let taskId = 'NULL';
        let iParent = Parent.MgNumVal.NUM_2_LONG();
        if ((iParent >= 0 && iParent < (this.ExpTask.getTaskDepth(false))) || iParent === ExpressionEvaluator.TRIGGER_TASK) {
            let task = super.GetContextTask(iParent);
            if (task !== null)
                taskId = task.getExternalTaskId();
        }
        resVal.Attr = StorageAttribute.ALPHA;
        resVal.StrVal = taskId;
    }
    eval_op_dbviewsize(resVal, Parent) {
        let size = 0;
        let iParent = Parent.MgNumVal.NUM_2_LONG();
        if ((iParent >= 0 && iParent < this.ExpTask.getTaskDepth(false)) || iParent === ExpressionEvaluator.TRIGGER_TASK) {
            let task = super.GetContextTask(iParent);
            if (task !== null && task.DataView.HasMainTable) {
                if (task.GetComputedProperty(PropInterface.PROP_TYPE_PRELOAD_VIEW).GetComputedValueBoolean()) {
                    size = task.DataView.DBViewSize;
                }
                else if (task.isTableWithAbsolutesScrollbar())
                    size = task.DataView.TotalRecordsCount;
            }
        }
        super.ConstructMagicNum(resVal, size, StorageAttribute.NUMERIC);
    }
    eval_op_dbviewrowidx(resVal, Parent) {
        let idx = 0;
        let iParent = Parent.MgNumVal.NUM_2_LONG();
        if ((iParent >= 0 && iParent < this.ExpTask.getTaskDepth(false)) || iParent === ExpressionEvaluator.TRIGGER_TASK) {
            let task = super.GetContextTask(iParent);
            if (task !== null && task.GetComputedProperty(PropInterface.PROP_TYPE_PRELOAD_VIEW).GetComputedValueBoolean()) {
                if ((task.DataView.getCurrRec()).getMode() !== DataModificationTypes.Insert) {
                    idx = task.DataviewManager.CurrentDataviewManager.GetDbViewRowIdx();
                }
            }
        }
        super.ConstructMagicNum(resVal, idx, StorageAttribute.NUMERIC);
    }
    eval_op_MlsTrans(resVal, fromString) {
        resVal.Attr = StorageAttribute.UNICODE;
        resVal.StrVal = '';
        resVal.StrVal = LanguageData.Instance.translate(NString.TrimEnd(fromString.StrVal, this._charsToTrim));
    }
    eval_op_StrBuild(valStack, resVal, nArgs) {
        let Exp_parms = new Array(nArgs);
        for (let i = 0; i < nArgs; i++)
            Exp_parms[nArgs - 1 - i] = valStack.pop();
        this.val_cpy(Exp_parms[0], resVal);
        let resultStr = new StringBuilder(resVal.StrVal);
        for (let i = 1; i < nArgs; i++) {
            let toReplace = '@' + i.toString().trim() + '@';
            let indexFrom = 0;
            while (indexFrom !== -1) {
                let nextIndex = resultStr.ToString().indexOf(toReplace, indexFrom);
                if (nextIndex === -1)
                    break;
                let precededBySlash = false;
                let shashIndex = resultStr.ToString().indexOf('\\' + toReplace, indexFrom);
                if (shashIndex !== -1)
                    precededBySlash = true;
                if ((precededBySlash && nextIndex !== shashIndex + 1) || !precededBySlash) {
                    resultStr.Replace(resultStr.ToString(nextIndex, nextIndex + toReplace.length - nextIndex), Exp_parms[i].StrVal.trim(), nextIndex, nextIndex + toReplace.length - nextIndex);
                    indexFrom = nextIndex + Exp_parms[i].StrVal.trim().length;
                }
                else
                    indexFrom = nextIndex + 1;
            }
        }
        resultStr.Replace('\\@', '@');
        resVal.StrVal = resultStr.ToString();
    }
    eval_op_tsk_instance(resVal, Parent) {
        let tag = 0;
        let iParent = Parent.MgNumVal.NUM_2_LONG();
        if ((iParent >= 0 && iParent < (this.ExpTask.getTaskDepth(false))) || iParent === ExpressionEvaluator.TRIGGER_TASK) {
            let task = super.GetContextTask(iParent);
            if (task !== null) {
                tag = NNumber.Parse(task.getTaskTag());
            }
        }
        resVal.Attr = StorageAttribute.NUMERIC;
        resVal.MgNumVal = new NUM_TYPE();
        resVal.MgNumVal.NUM_4_LONG(tag);
    }
    eval_op_getParam(resVal, name) {
        Debug.Assert(!name.IsNull && name.StrVal !== null);
        let expVal = GlobalParams.Instance.get(name.StrVal);
        if (expVal !== null)
            resVal.Copy(expVal);
        else
            resVal.Nullify();
    }
    eval_op_setParam(resVal, name, value) {
        Debug.Assert(!name.IsNull && name.StrVal !== null);
        resVal.Attr = StorageAttribute.BOOLEAN;
        let globalParams = GlobalParams.Instance;
        globalParams.set(name.StrVal, value);
        resVal.BoolVal = true;
    }
    async eval_op_iniput(resVal, value, updateIni) {
        resVal.Attr = StorageAttribute.BOOLEAN;
        resVal.BoolVal = await EnvParamsTable.Instance.set(value.StrVal, updateIni.BoolVal);
    }
    eval_op_iniget(resVal, nameVal) {
        resVal.StrVal = EnvParamsTable.Instance.get(nameVal.StrVal);
        resVal.Attr = StorageAttribute.ALPHA;
    }
    eval_op_inigetln(resVal, sectionVal, numberVal) {
        resVal.StrVal = EnvParamsTable.Instance.getln(sectionVal.StrVal, numberVal.MgNumVal.NUM_2_LONG());
        resVal.Attr = StorageAttribute.ALPHA;
    }
    eval_op_m(resVal, codes) {
        resVal.Attr = StorageAttribute.ALPHA;
        resVal.StrVal = '';
        for (let i = 0; i < codes.length; i++) {
            let mode = this.cst_code_trans_buf('O', 'MCDELRKSON ', codes.charAt(i), MsgInterface.EXPTAB_TSK_MODE_RT);
            resVal.StrVal = resVal.StrVal + mode;
        }
    }
    cst_code_trans_buf(opr, intStr, code, strId) {
        let i;
        let constStr = LanguageData.Instance.getConstMessage(strId);
        let tokens = new Array_Enumerator(StrUtil.tokenize(constStr, ','));
        let token;
        let resVal = '\0';
        for (i = 0; i < intStr.length && tokens.MoveNext(); i++) {
            token = tokens.Current;
            let ofs = token.indexOf('&');
            ofs++;
            let currCode = token.charAt(ofs).toUpperCase();
            if (opr === 'I') {
                if (code === currCode) {
                    resVal = intStr.charAt(i);
                    break;
                }
            }
            else {
                if (code === intStr.charAt(i)) {
                    resVal = currCode;
                    break;
                }
            }
        }
        return resVal;
    }
    async eval_op_expcalc(resVal, expVal) {
        resVal.IsNull = true;
        if (ExpressionEvaluator._recursiveExpCalcCount < 50) {
            ExpressionEvaluator._recursiveExpCalcCount++;
            let expVal2 = await this.ExpTask.getExpById(expVal.MgNumVal.NUM_2_LONG()).evaluateWithResType(StorageAttribute.NONE);
            ExpressionEvaluator._recursiveExpCalcCount--;
            if (expVal2 !== null)
                resVal.Copy(expVal2);
        }
    }
    eval_op_CndRange(resVal, val1, val2) {
        if (val1.BoolVal)
            resVal.Copy(val2);
        else
            resVal.IsNull = true;
    }
    eval_op_taskType(resVal, Parent) {
        let iParent = Parent.MgNumVal.NUM_2_LONG();
        if ((iParent >= 0 && iParent < (this.ExpTask.getTaskDepth(false))) || iParent === ExpressionEvaluator.TRIGGER_TASK) {
            let task = super.GetContextTask(iParent);
            if (task !== null) {
                resVal.StrVal = (task.isMainProg() ? 'MW' : 'W');
            }
            else
                resVal.StrVal = ' ';
        }
        resVal.Attr = StorageAttribute.ALPHA;
    }
    eval_op_terminal(resVal) {
        let terminal = Environment.Instance.getTerminal();
        super.ConstructMagicNum(resVal, terminal, StorageAttribute.NUMERIC);
    }
    eval_op_projectdir(resVal) {
        resVal.StrVal = Environment.Instance.getProjDir(this.ExpTask.getCompIdx());
        resVal.Attr = StorageAttribute.ALPHA;
    }
    static IsValidVector(vec) {
        return vec !== null && vec.Attr === StorageAttribute.BLOB_VECTOR && VectorType.validateBlobContents(vec.StrVal);
    }
    static GetAttributeChar(storageAttr) {
        let attr = storageAttr;
        switch (storageAttr) {
            case StorageAttribute.BLOB:
                attr = 'B';
                break;
            case StorageAttribute.BOOLEAN:
                attr = 'L';
                break;
            default:
                break;
        }
        return attr;
    }
    HandleControlGoto(ctrlTask, ctrl, rowNo) {
        let task = ctrlTask;
        if (ctrl !== null) {
            let dv = task.DataView;
            let tCtrl = ctrl;
            let wantedLine = task.getForm().DisplayLine;
            if (ctrl.IsRepeatable && rowNo > 0) {
                task.getForm().getTopIndexFromGUI();
                let topRecIdx = dv.getTopRecIdx();
                wantedLine = topRecIdx + rowNo - 1;
                if (wantedLine - topRecIdx - 1 >= task.getForm().getRowsInPage())
                    return false;
                if (!task.getForm().IsValidRow(wantedLine))
                    return false;
            }
            if (tCtrl.isVisible()) {
                if (tCtrl.isSubform()) {
                    return false;
                }
                let frm = task.getForm();
                let isRowEditing = frm.getIsRowEditingFromGui(wantedLine);
                if (tCtrl.isRepeatable() && !isRowEditing) {
                    AccessHelper.eventsManager.addGuiTriggeredEventWithTaskAndCodeAndLine(task, InternalInterface.MG_ACT_REC_PREFIX, wantedLine);
                    console.log('Unable to park on control: ' + tCtrl.Name + ' as row is not in editing mode');
                    return true;
                }
                else {
                    let rtEvt = new RunTimeEvent(tCtrl, wantedLine);
                    rtEvt.setInternal(InternalInterface.MG_ACT_CTRL_FOCUS);
                    AccessHelper.eventsManager.addToTail(rtEvt);
                    return true;
                }
            }
        }
        return false;
    }
    eval_op_set_title(resVal, title) {
        resVal.Attr = StorageAttribute.BOOLEAN;
        resVal.BoolVal = true;
        if (!(title.Attr !== StorageAttribute.ALPHA && title.Attr !== StorageAttribute.UNICODE)) {
            if (!title.IsNull) {
                let text = title.StrVal;
                let task = this.ExpTask.GetContextTask();
                if (task.getTopMostForm() !== null) {
                    task.getTopMostForm().setTitle(text);
                }
            }
        }
    }
    eval_op_is_row_editing(resVal, val1) {
        let isRowEditing = false;
        if (val1.MgNumVal === null) {
            super.SetNULL(resVal, StorageAttribute.ALPHA);
            return;
        }
        let parent = val1.MgNumVal.NUM_2_LONG();
        if ((parent >= 0 && parent < (this.ExpTask.getTaskDepth(false))) || parent === ExpressionEvaluator.TRIGGER_TASK) {
            let tsk = super.GetContextTask(parent);
            let myform = tsk.getForm();
            isRowEditing = myform.getIsRowEditingFromGui(myform.DisplayLine);
        }
        resVal.BoolVal = isRowEditing;
        resVal.Attr = StorageAttribute.BOOLEAN;
    }
    async eval_op_control_select_program(expVal, resVal) {
        let controlID = expVal.MgNumVal.NUM_2_LONG();
        let parent = 0;
        let refParent = new RefParam(parent);
        let mgControl = this.ExpTask.GetContextTask().GetControlFromControlID(controlID - 1, refParent);
        parent = refParent.value;
        resVal.Attr = StorageAttribute.NUMERIC;
        resVal.MgNumVal = new NUM_TYPE();
        if (mgControl !== null && mgControl.HasSelectProgram()) {
            let selectProgProp = mgControl.getProp(PropInterface.PROP_TYPE_SELECT_PROGRAM);
            let realIndex = NNumber.Parse(await selectProgProp.getValue());
            let programIndex = 0;
            if (realIndex > 0) {
                if (parent > 0)
                    programIndex = realIndex + parent / 100;
                else
                    programIndex = realIndex;
            }
            resVal.MgNumVal = NUM_TYPE.from_double(programIndex);
        }
    }
    async eval_op_CallJS(task, methodName, expVal, resVal) {
        resVal.Attr = StorageAttribute.ALPHA;
        resVal.StrVal = '';
        if (task.isMainProg()) {
            console.error("CallJS cannot be used in Main Program");
            return;
        }
        let form = task.getForm();
        let args = this.params2arguments(expVal, 0, expVal.length);
        if (!isNullOrUndefined(args)) {
            if (task.isFirstRecordCycle()) {
                Commands.invoke();
                await Thread.Sleep(10);
            }
            resVal.StrVal = Commands.addCallJS(form, form.UniqueName, methodName, args);
            resVal.Attr = StorageAttribute.ALPHA;
        }
    }
    eval_op_set_cookie(cookieName, expVal, resVal) {
        resVal.Attr = StorageAttribute.BOOLEAN;
        resVal.StrVal = '';
        let args = this.params2arguments(expVal, 0, expVal.length);
        var expiresDate, path, domain, secure, sameSite;
        expiresDate = args[1];
        let expiresTime = args[2];
        let nativeDt;
        if (expiresDate) {
            let breakParams = DisplayConvertor.Instance.getNewDateBreakParams();
            DisplayConvertor.Instance.date_break_datemode(breakParams, expiresDate, false, this.ExpTask.getCompIdx());
            if (breakParams.year == 0 && breakParams.month == 0 && breakParams.day == 0)
                return null;
            nativeDt = new Date(breakParams.year, breakParams.month - 1, breakParams.day);
            if (expiresTime) {
                let timeBreakParams = DisplayConvertor.Instance.getNewTimeBreakParams();
                DisplayConvertor.time_break(timeBreakParams, expiresTime);
                nativeDt.setHours(timeBreakParams.hour);
                nativeDt.setMinutes(timeBreakParams.minute);
                nativeDt.setSeconds(timeBreakParams.second);
            }
        }
        path = args[3];
        domain = args[4];
        resVal.BoolVal = CookieService.setCookie(cookieName, (args[0] != null ? args[0].toString() : ""), nativeDt, path, domain, secure, sameSite);
    }
    eval_op_get_cookie(cookieName, resVal) {
        resVal.StrVal = CookieService.getCookie(cookieName);
    }
    eval_op_delete_cookie(cookieName, resVal) {
        resVal.Attr = StorageAttribute.BOOLEAN;
        resVal.BoolVal = CookieService.deleteCookie(cookieName);
        ;
    }
    eval_op_route_get(resVal) {
        resVal.Attr = StorageAttribute.UNICODE;
        resVal.IsNull = false;
        resVal.StrVal = Commands.getLastRoute();
    }
    params2arguments(Exp_params, startAt, count) {
        let NULL_STR = '';
        let attr = new Array();
        for (let i = startAt; i < startAt + count; i++) {
            if (isNullOrUndefined(Exp_params[i])) {
                attr.push(NULL_STR);
            }
            else {
                switch (Exp_params[i].Attr) {
                    case StorageAttribute.ALPHA:
                    case StorageAttribute.UNICODE:
                        if (Exp_params[i].StrVal != null && Exp_params[i].StrVal.length > 0)
                            attr.push(Exp_params[i].StrVal);
                        else
                            attr.push(NULL_STR);
                        break;
                    case StorageAttribute.BOOLEAN:
                        attr.push(Exp_params[i].BoolVal ? true : false);
                        break;
                    case StorageAttribute.DATE:
                    case StorageAttribute.TIME:
                        attr.push(Exp_params[i].MgNumVal.NUM_2_LONG());
                        break;
                    case StorageAttribute.NUMERIC:
                        attr.push(Exp_params[i].MgNumVal.to_double());
                        break;
                    case StorageAttribute.BLOB:
                        let blob = Exp_params[i].StrVal;
                        if (!isNullOrUndefined(blob)) {
                            {
                                let contentType = BlobType.getContentType(blob);
                                if (contentType !== BlobType.CONTENT_TYPE_BINARY) {
                                    attr.push(BlobType.getString(blob));
                                }
                                else {
                                    Logger.Instance.WriteErrorToLog('Argument ' + (i + 1) + ' of type Binary Blob is not supported in CallJS');
                                    return null;
                                }
                            }
                        }
                        break;
                    case StorageAttribute.NONE:
                    default:
                        Logger.Instance.WriteErrorToLog(' Argument ' + (i + 1) + ' is not supported in CallJS');
                        return null;
                }
            }
        }
        return attr;
    }
}
ExpressionEvaluator.ASTERISK_CHR = String.fromCharCode(1);
ExpressionEvaluator.QUESTION_CHR = String.fromCharCode(2);
ExpressionEvaluator.PARENT_LEN = 2;
ExpressionEvaluator.SHORT_OBJECT_LEN = 2;
ExpressionEvaluator.LONG_OBJECT_LEN = 4;
ExpressionEvaluator._recursiveExpCalcCount = 0;
ExpressionEvaluator.ErrorToBeWrittenInServerLog = null;
export class NullValueException extends Exception {
    constructor(attr) {
        super();
        this._attr = StorageAttribute.NONE;
        this._attr = attr;
        this.name = 'NullValueException';
    }
    getAttr() {
        return this._attr;
    }
}
export class ExpStrTracker {
    constructor(expBytes, nullArithmetic) {
        this._expBytes = null;
        this._lowHigh = true;
        this._nullArithmetic = false;
        this._isNull = false;
        this._posIdx = 0;
        this._expBytes = new Int8Array(expBytes.length);
        for (let _ai = 0; _ai < this._expBytes.length; ++_ai)
            this._expBytes[_ai] = 0;
        for (let i = 0; i < expBytes.length; i++)
            this._expBytes[i] = expBytes[i];
        this._nullArithmetic = nullArithmetic;
        this._lowHigh = Environment.Instance.getLowHigh();
    }
    setNullResult() {
        if (this._nullArithmetic)
            this._isNull = true;
    }
    resetNullResult() {
        if (this._nullArithmetic) {
            this._isNull = false;
        }
    }
    isNull() {
        return this._isNull;
    }
    get1ByteNumber() {
        let num = (this._expBytes[this._posIdx] >= 0) ? this._expBytes[this._posIdx] : (256 + this._expBytes[this._posIdx]);
        this._posIdx += 1;
        return num;
    }
    get2ByteNumber() {
        let c1 = (this._expBytes[this._posIdx] >= 0) ? this._expBytes[this._posIdx] : (256 + this._expBytes[this._posIdx]);
        this._posIdx += 1;
        let c2 = (this._expBytes[this._posIdx] >= 0) ? this._expBytes[this._posIdx] : (256 + this._expBytes[this._posIdx]);
        this._posIdx += 1;
        let num = (this._lowHigh ? ExpressionEvaluator.MK_SHRT(c2, c1) : ExpressionEvaluator.MK_SHRT(c1, c2));
        return num;
    }
    get4ByteNumber() {
        let c4 = (this._expBytes[this._posIdx] >= 0) ? this._expBytes[this._posIdx] : (256 + this._expBytes[this._posIdx]);
        this._posIdx += 1;
        let c3 = (this._expBytes[this._posIdx] >= 0) ? this._expBytes[this._posIdx] : (256 + this._expBytes[this._posIdx]);
        this._posIdx += 1;
        let c2 = (this._expBytes[this._posIdx] >= 0) ? this._expBytes[this._posIdx] : (256 + this._expBytes[this._posIdx]);
        this._posIdx += 1;
        let c1 = (this._expBytes[this._posIdx] >= 0) ? this._expBytes[this._posIdx] : (256 + this._expBytes[this._posIdx]);
        this._posIdx += 1;
        let num = this._lowHigh ? ExpressionEvaluator.MK_LONG(c1, c2, c3, c4) : ExpressionEvaluator.MK_LONG(c4, c3, c2, c1);
        return num;
    }
    getString(len, updateIdx, isUnicode) {
        let str = '';
        let bytes;
        if (isUnicode === false) {
            bytes = len;
            let tmpChar = new Array(this._expBytes.length);
            for (let i = 0; i < this._expBytes.length; i = i + 1)
                tmpChar[i] = String.fromCharCode(this._expBytes[i]);
            str = NString.FromChars(tmpChar, this._posIdx, len);
        }
        else {
            bytes = len * 2;
            let tmp = new Uint8Array(bytes);
            let increment = (!this._lowHigh) ? ConstInterface.BYTES_IN_CHAR : 1;
            for (let i = this._posIdx; i < this._posIdx + bytes; i = i + increment) {
                if (!this._lowHigh) {
                    tmp[i - this._posIdx] = this._expBytes[i + 1];
                    tmp[i + 1 - this._posIdx] = this._expBytes[i];
                }
                else
                    tmp[i - this._posIdx] = this._expBytes[i];
            }
            try {
                str = Encoding.Unicode.GetString(tmp, 0, tmp.length);
            }
            catch (ex) {
                if (ex instanceof Exception) {
                    Logger.Instance.WriteExceptionToLog(ex);
                }
                else
                    throw ex;
            }
        }
        if (updateIdx) {
            this._posIdx += bytes;
        }
        return str;
    }
    skipOpFunctionPtr() {
        this._posIdx += ExpressionInterface.EXP_OPER_FUNC_PTR_LEN;
    }
    getOpCode() {
        let tmp = [
            this._lowHigh ? this._expBytes[this._posIdx] : this._expBytes[this._posIdx + 1], this._lowHigh ? this._expBytes[this._posIdx + 1] : this._expBytes[this._posIdx]
        ];
        let num = (tmp[1] >= 0) ? tmp[1] : (256 + tmp[1]);
        num <<= 8;
        num = (num | ((tmp[0] >= 0) ? tmp[0] : (256 + tmp[0])));
        this._posIdx += ExpressionInterface.EXP_OPER_LEN;
        this.skipOpFunctionPtr();
        return num;
    }
    getVarIdx() {
        let flip = 0;
        if (this._lowHigh)
            flip = 1 - flip;
        let num = (this._expBytes[this._posIdx + flip] >= 0) ? this._expBytes[this._posIdx + flip] : (256 + this._expBytes[this._posIdx + flip]);
        num <<= 8;
        flip = 1 - flip;
        num |= ((this._expBytes[this._posIdx + flip] >= 0) ? this._expBytes[this._posIdx + flip] : (256 + this._expBytes[this._posIdx + flip]));
        this._posIdx += ExpressionInterface.EXP_OPER_LEN;
        return num;
    }
    getMagicNumber(len, updateIdx) {
        let mgNum = new NUM_TYPE(this._expBytes, this._posIdx, len);
        if (updateIdx)
            this._posIdx += len;
        return mgNum;
    }
    skipOperator() {
        let argsRemain = 1;
        let expDesc;
        let len;
        while (argsRemain > 0) {
            argsRemain--;
            let opCode = this.getOpCode();
            switch (opCode) {
                case ExpressionInterface.EXP_OP_A:
                case ExpressionInterface.EXP_OP_H:
                    len = this.get4ByteNumber();
                    this._posIdx += (len * ConstInterface.BYTES_IN_CHAR);
                    len = this.get4ByteNumber();
                    this._posIdx += len;
                    break;
                case ExpressionInterface.EXP_OP_EXT_A:
                    len = this.get2ByteNumber();
                    this._posIdx += len;
                    break;
                case ExpressionInterface.EXP_OP_N:
                case ExpressionInterface.EXP_OP_T:
                case ExpressionInterface.EXP_OP_D:
                case ExpressionInterface.EXP_OP_M:
                case ExpressionInterface.EXP_OP_K:
                case ExpressionInterface.EXP_OP_E:
                    len = this.get2ByteNumber();
                    this._posIdx += len;
                    break;
                case ExpressionInterface.EXP_OP_F:
                case ExpressionInterface.EXP_OP_P:
                    len = this.get2ByteNumber();
                    this._posIdx += len;
                    len = this.get2ByteNumber();
                    this._posIdx += len;
                    break;
                case ExpressionInterface.EXP_OP_L:
                    this._posIdx += 2;
                    break;
                case ExpressionInterface.EXP_OP_V:
                    this._posIdx += (ExpressionEvaluator.PARENT_LEN + ExpressionEvaluator.LONG_OBJECT_LEN);
                    break;
                case ExpressionInterface.EXP_OP_FORM:
                    this._posIdx += (ExpressionEvaluator.PARENT_LEN + ExpressionEvaluator.LONG_OBJECT_LEN);
                    break;
                case ExpressionInterface.EXP_OP_VAR:
                    this._posIdx += (ExpressionEvaluator.PARENT_LEN + ExpressionEvaluator.LONG_OBJECT_LEN);
                    break;
                case ExpressionInterface.EXP_OP_RIGHT_LITERAL:
                    len = this.get2ByteNumber();
                    this._posIdx += len;
                    len = this.get2ByteNumber();
                    this._posIdx += len;
                    break;
                default:
                    expDesc = ExpressionDict.expDesc[opCode];
                    if (expDesc.ArgCount_ < 0)
                        argsRemain += this.get1ByteNumber();
                    else
                        argsRemain += expDesc.ArgCount_;
                    break;
            }
        }
    }
}
class DynamicOperation {
    constructor() {
        this.argCount_ = 0;
        this.opCode_ = ExpressionInterface.EXP_OP_NONE;
    }
}
//# sourceMappingURL=data:application/json;base64,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