import { Commands, CommandType, GuiDataCollection, HtmlProperties, MgControlBase, PropInterface } from '@magic-xpa/gui';
import { InternalInterface, Logger, MgControlType, ScrollBarThumbType, StorageAttribute, SubformType, TableBehaviour } from '@magic-xpa/utils';
import { Debug, NString } from '@magic-xpa/mscorelib';
import { ArgumentsList } from '../rt/ArgumentsList';
import { TaskBase } from '../tasks/TaskBase';
import { ConstInterface } from '../ConstInterface';
import { AccessHelper } from '../AccessHelper';
export class MgControl extends MgControlBase {
    initialize() {
        this._focusedStopExecution = false;
        this._inControlSuffix = false;
        this._rtEvtTask = null;
        this._subformTaskId = null;
        this._subformTask = null;
        this.IsInteractiveUpdate = false;
        this.refreshOnString = null;
        this.SubformLoaded = false;
        this.HasZoomHandler = false;
        this.ArgList = null;
    }
    constructor(type, taskOrParentMgForm, parentControlOrParentControlIdx) {
        if (arguments.length === 0)
            super();
        else if (arguments.length === 3 && (type === null || type.constructor === Number) && (taskOrParentMgForm === null || taskOrParentMgForm instanceof TaskBase) && (parentControlOrParentControlIdx === null || parentControlOrParentControlIdx.constructor === Number))
            super(type, taskOrParentMgForm.getForm(), parentControlOrParentControlIdx);
        else
            super(type, taskOrParentMgForm, parentControlOrParentControlIdx);
        this.initialize();
    }
    GetVarIndex() {
        return super.getForm().getTask().DataView.Dvcount + this.veeIndx;
    }
    async fillData(mgForm, ditIdx, parser) {
        super.fillData(mgForm, ditIdx, parser);
        this.ArgList = await this.GetArgumentList();
    }
    SetAttribute(attribute, valueStr) {
        let isTagProcessed = super.SetAttribute(attribute, valueStr);
        if (!isTagProcessed) {
            switch (attribute) {
                case ConstInterface.MG_ATTR_SUBFORM_TASK:
                    if (valueStr.trim() != "")
                        this._subformTaskId = valueStr;
                    break;
                case ConstInterface.MG_ATTR_REFRESHON:
                    this.refreshOnString = valueStr.trim();
                    break;
                default:
                    isTagProcessed = false;
                    Logger.Instance.WriteExceptionToLogWithMsg(NString.Format("Unrecognized attribute: '{0}'", attribute));
                    break;
            }
        }
        return isTagProcessed;
    }
    async GetArgumentList() {
        let argList = null;
        if (this._propTab === null)
            return argList;
        let ArgListProp = this._propTab.getPropById(PropInterface.PROP_TYPE_PARAMETERS);
        if (ArgListProp === null)
            return argList;
        let parms = await ArgListProp.getValue();
        if (parms !== null) {
            argList = new ArgumentsList();
            argList.fillList(parms, super.getForm().getTask());
            if (this.refreshOnString !== null)
                argList.RefreshOnString = this.refreshOnString;
        }
        return argList;
    }
    isDescendent(parentForm) {
        let isDescendent = false;
        let checkCurrentForm = super.getForm();
        Debug.Assert(parentForm !== null);
        while (!isDescendent && checkCurrentForm !== null) {
            if (parentForm === checkCurrentForm)
                isDescendent = true;
            else {
                let currTask = checkCurrentForm.getTask();
                if (currTask.getParent() !== null)
                    checkCurrentForm = currTask.getParent().getForm();
                else
                    break;
            }
        }
        return isDescendent;
    }
    async refreshOnExpression() {
        let i;
        let size = (this._propTab === null ? 0 : this._propTab.getSize());
        let prop;
        let refresh = false;
        for (i = 0; i < size && !refresh; i++) {
            prop = this._propTab.getProp(i);
            switch (prop.getID()) {
                case PropInterface.PROP_TYPE_DATA:
                    if (!prop.isExpression() && this._field !== null && this._field.getTask() !== this.getForm().getTask()) {
                        refresh = true;
                        break;
                    }
                case PropInterface.PROP_TYPE_FORMAT:
                case PropInterface.PROP_TYPE_VISIBLE:
                case PropInterface.PROP_TYPE_ENABLED:
                case PropInterface.PROP_TYPE_RANGE:
                case PropInterface.PROP_TYPE_TRANSLATOR:
                    if (prop.isExpression())
                        refresh = true;
                    break;
                default:
                    break;
            }
            if (refresh)
                await this.RefreshDisplay();
        }
    }
    async invoke() {
        let invoked = false;
        if (await this.isParkable(true)) {
            await AccessHelper.eventsManager.handleInternalEventWithMgControlBase(this, InternalInterface.MG_ACT_CTRL_PREFIX);
            invoked = true;
        }
        return invoked;
    }
    async isParkable(checkEnabledAndVisible) {
        let result = !super.isImageControl();
        if (result) {
            await super.checkProp(PropInterface.PROP_TYPE_ENABLED, true);
            await super.checkProp(PropInterface.PROP_TYPE_VISIBLE, true);
            result = await super.isParkable(checkEnabledAndVisible);
        }
        return result;
    }
    getSubformTaskId() {
        return this._subformTaskId;
    }
    initSubformTask() {
        if (super.isSubform())
            this._subformTask = GuiDataCollection.MGDataTable.GetTaskByID(this._subformTaskId);
    }
    getParentTable() {
        return this._parentTable;
    }
    async restoreOldValue() {
        if (this._field !== null)
            await this._field.updateDisplay();
        else {
            super.resetPrevVal();
            await super.SetAndRefreshDisplayValue(await this.getMgValue(this.Value), this.IsNull);
        }
    }
    setRtEvtTask(curTask) {
        this._rtEvtTask = curTask;
    }
    getRtEvtTask() {
        return this._rtEvtTask;
    }
    setSubformTaskId(taskId) {
        this._subformTaskId = taskId;
    }
    isFocusedStopExecution() {
        return this._focusedStopExecution;
    }
    setFocusedStopExecution(focusedStopExecution) {
        this._focusedStopExecution = focusedStopExecution;
    }
    onTheSameSubFormControl(ctrl) {
        let retValue = false;
        if (ctrl !== null) {
            if (super.getForm().isSubForm() && ctrl.getForm().isSubForm()) {
                if (super.getForm().getTask() === ctrl.getForm().getTask())
                    retValue = true;
            }
        }
        return retValue;
    }
    onDiffForm(ctrl) {
        let isSubform = super.getForm().isSubForm() || ctrl.getForm().isSubForm();
        let isOnDiffForm = (isSubform) && !this.onTheSameSubFormControl(ctrl);
        return isOnDiffForm;
    }
    isInControlSuffix() {
        return this._inControlSuffix;
    }
    setInControlSuffix(inControlSuffix) {
        this._inControlSuffix = inControlSuffix;
    }
    HasSelectProgram() {
        let HasSelectProgram = false;
        let selectProgProp = super.GetComputedProperty(PropInterface.PROP_TYPE_SELECT_PROGRAM);
        if (selectProgProp !== null && selectProgProp.GetComputedValue() !== null)
            HasSelectProgram = true;
        return HasSelectProgram;
    }
    useZoomHandler() {
        return this.HasZoomHandler || (super.getField() !== null && (super.getField().getHasZoomHandler() || (this.HasSelectProgram())));
    }
    async updatePropertyLogicNesting(propId, commandType, val, updateThis) {
        if (val && super.haveToCheckParentValue())
            val = super.isParentPropValue(propId);
        if (val && super.isSubform() && commandType === CommandType.PROP_SET_VISIBLE && this.ShouldSetSubformInvisible())
            val = false;
        if (updateThis) {
            Commands.addOperationWithLine(CommandType.SET_PROPERTY, this, super.getDisplayLine(false), commandType === CommandType.PROP_SET_VISIBLE ? HtmlProperties.Visible : HtmlProperties.Enabled, val);
        }
        await super.updateChildrenPropValue(propId, commandType, val);
    }
    ShouldSetSubformInvisible() {
        let setInvisible = false;
        let subformType = super.GetComputedProperty(PropInterface.PROP_TYPE_SUBFORM_TYPE).GetComputedValueInteger();
        if (subformType === SubformType.None && this.getSubformTaskId() === null)
            setInvisible = true;
        return setInvisible;
    }
    async updateSubformChildrenPropValue(propId, commandType, val) {
        Debug.Assert(super.isSubform());
        if (!super.getForm().getTask().AfterFirstRecordPrefix)
            return;
        let subformType = super.GetComputedProperty(PropInterface.PROP_TYPE_SUBFORM_TYPE).GetComputedValueInteger();
        if (this._subformTaskId !== null || subformType === SubformType.None) {
            if (this._subformTask !== null && this._subformTask.isStarted()) {
                this.SubformLoaded = true;
                if (commandType === CommandType.PROP_SET_VISIBLE && val && this.RefreshOnVisible) {
                    this.RefreshOnVisible = false;
                    await super.getForm().getTask().SubformRefresh(this._subformTask, true);
                }
                let subformForm = this._subformTask.getForm();
                for (let i = 0; i < subformForm.getCtrlCount(); i++) {
                    let child = subformForm.getCtrl(i);
                    if (!child.isColumnControl())
                        continue;
                    let childValue = await child.checkProp(propId, true) && val;
                    if (childValue)
                        childValue = child.isParentPropValue(propId);
                    await child.updateChildrenPropValue(propId, commandType, childValue);
                }
            }
        }
        else if ((commandType === CommandType.PROP_SET_VISIBLE && val) && !this.SubformLoaded) {
            this.SubformLoaded = true;
            await AccessHelper.eventsManager.handleInternalEventWithMgControlBase(this, InternalInterface.MG_ACT_SUBFORM_OPEN);
        }
    }
    Init() {
        this.initSubformTask();
    }
    GetTableBehaviour() {
        return TableBehaviour.UnlimitedItems;
    }
    async validateAndSetValue(NewValue, updateCtrl) {
        super.UpdateModifiedByUser(NewValue);
        if (this.ModifiedByUser) {
            let ctrlCurrValue = this.Value;
            let vd;
            if (super.getPIC() === null)
                return true;
            vd = super.buildPicture(ctrlCurrValue, NewValue);
            vd.evaluate();
            if (vd.ValidationFailed) {
                this.ModifiedByUser = false;
                return false;
            }
            this.IsInteractiveUpdate = true;
            super.getForm().getTask().setDataSynced(false);
            if (updateCtrl) {
                if (this._field !== null && this._field.getType() === StorageAttribute.NUMERIC && typeof NewValue === 'string' && NewValue.trim().length > 0)
                    vd.setValue(NewValue);
                await this.setValue(vd);
            }
        }
        return true;
    }
    async setValue(vd) {
        let isNull = vd.getIsNull();
        let newValue = vd.getDispValue();
        let mgVal;
        if (!isNull || !this._field.hasNullDisplayValue())
            mgVal = await super.getMgValue(newValue);
        else
            mgVal = ((this._field.getType() === StorageAttribute.BLOB_VECTOR) ? this._field.getCellDefualtValue() : this._field.getDefaultValue());
        if (mgVal !== null) {
            if (this._field !== null && (!(newValue === this.Value) || this.KeyStrokeOn || this.IsNull !== isNull)) {
                await this._field.setValueAndStartRecompute(mgVal, isNull, true, true, false);
                if (super.isCheckBox()) {
                    this.forceRefresh = true;
                    mgVal = this._field.getValue(false);
                }
            }
            if ((this._field !== null && isNull) && this._field.hasNullDisplayValue())
                mgVal = newValue;
            await super.SetAndRefreshDisplayValue(mgVal, isNull);
            this.forceRefresh = false;
            this.KeyStrokeOn = false;
        }
    }
    getSubformTask() {
        return this._subformTask;
    }
    resetSubformTask() {
        this._subformTask = null;
        this._subformTaskId = "0";
    }
    GetSubformMgForm() {
        let form = null;
        if (this._subformTask !== null)
            form = this._subformTask.getForm();
        return form;
    }
    IsTableWithAbsoluteScrollbar() {
        let tableWithAbsoluteScrollbar = false;
        Debug.Assert(this.Type === MgControlType.CTRL_TYPE_TABLE, "In MgControl.IsTableWithAbsoluteScrollbar(): Not a table control.");
        if (this._propTab.propExists(PropInterface.PROP_TYPE_SCROLL_BAR_THUMB))
            tableWithAbsoluteScrollbar = (this._propTab.getPropById(PropInterface.PROP_TYPE_SCROLL_BAR_THUMB).GetComputedValueInteger() === ScrollBarThumbType.Absolute);
        return tableWithAbsoluteScrollbar;
    }
    static validReturnToCtrl() {
        if (MgControl.ReturnToCtrl !== null && MgControl.ReturnToCtrl.getForm().getTask().isAborting())
            return false;
        return true;
    }
    maySetValue() {
        switch (this.Type) {
            case MgControlType.CTRL_TYPE_CHECKBOX:
            case MgControlType.CTRL_TYPE_TEXT:
            case MgControlType.CTRL_TYPE_COMBO:
            case MgControlType.CTRL_TYPE_LIST:
            case MgControlType.CTRL_TYPE_RADIO:
            case MgControlType.CTRL_TYPE_TAB:
                break;
            default:
                return false;
        }
        if (this._field === null || this._valExpId > 0)
            return false;
        return this.isModifiable();
    }
    async validateAndSetValueIfAllowed(NewValue, refreshDisplay) {
        if (!this.maySetValue()) {
            Logger.Instance.WriteErrorToLog(NString.Format("Unable to change value of control {0}: The control is read-only.", this.Name));
            return false;
        }
        let retVal = await this.validateAndSetValue(NewValue, true);
        if (!retVal)
            Logger.Instance.WriteErrorToLog(NString.Format("Value validation failed for {0}.", this.Name));
        if (refreshDisplay) {
            Commands.addNoParameters(CommandType.REFRESH_TASK, this.getForm());
            Commands.invoke();
        }
        return retVal;
    }
}
MgControl.ReturnToCtrl = null;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiTWdDb250cm9sLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vcHJvamVjdHMvZW5naW5lL3NyYy9ndWkvTWdDb250cm9sLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLE9BQU8sRUFDTCxRQUFRLEVBQ1IsV0FBVyxFQUNYLGlCQUFpQixFQUNqQixjQUFjLEVBQ2QsYUFBYSxFQUdiLGFBQWEsRUFHZCxNQUFNLGdCQUFnQixDQUFDO0FBQ3hCLE9BQU8sRUFDTCxpQkFBaUIsRUFDakIsTUFBTSxFQUNOLGFBQWEsRUFDYixrQkFBa0IsRUFDbEIsZ0JBQWdCLEVBQ2hCLFdBQVcsRUFDWCxjQUFjLEVBRWYsTUFBTSxrQkFBa0IsQ0FBQztBQUMxQixPQUFPLEVBQUMsS0FBSyxFQUFFLE9BQU8sRUFBQyxNQUFNLHNCQUFzQixDQUFDO0FBQ3BELE9BQU8sRUFBQyxhQUFhLEVBQUMsTUFBTSxxQkFBcUIsQ0FBQztBQUNsRCxPQUFPLEVBQUMsUUFBUSxFQUFDLE1BQU0sbUJBQW1CLENBQUM7QUFJM0MsT0FBTyxFQUFDLGNBQWMsRUFBQyxNQUFNLG1CQUFtQixDQUFDO0FBQ2pELE9BQU8sRUFBQyxZQUFZLEVBQUMsTUFBTSxpQkFBaUIsQ0FBQztBQVE3QyxNQUFNLE9BQU8sU0FBVSxTQUFRLGFBQWE7SUFnQmxDLFVBQVU7UUFDaEIsSUFBSSxDQUFDLHFCQUFxQixHQUFHLEtBQUssQ0FBQztRQUNuQyxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsS0FBSyxDQUFDO1FBQzlCLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDO1FBQ3ZCLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDO1FBQzNCLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxtQkFBbUIsR0FBRyxLQUFLLENBQUM7UUFDakMsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUM7UUFDNUIsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUM7UUFDM0IsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUM7UUFDNUIsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLENBQUM7SUFDdEIsQ0FBQztJQVNELFlBQVksSUFBb0IsRUFBRSxrQkFBd0IsRUFBRSwrQkFBd0M7UUFDbEcsSUFBSSxTQUFTLENBQUMsTUFBTSxLQUFLLENBQUM7WUFDeEIsS0FBSyxFQUFFLENBQUM7YUFDTCxJQUFJLFNBQVMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLElBQUksSUFBSSxJQUFJLENBQUMsV0FBVyxLQUFLLE1BQU0sQ0FBQyxJQUFJLENBQUMsa0JBQWtCLEtBQUssSUFBSSxJQUFJLGtCQUFrQixZQUFZLFFBQVEsQ0FBQyxJQUFJLENBQUMsK0JBQStCLEtBQUssSUFBSSxJQUFJLCtCQUErQixDQUFDLFdBQVcsS0FBSyxNQUFNLENBQUM7WUFDbFEsS0FBSyxDQUFDLElBQUksRUFBRSxrQkFBa0IsQ0FBQyxPQUFPLEVBQUUsRUFBRSwrQkFBK0IsQ0FBQyxDQUFDOztZQUUzRSxLQUFLLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFLCtCQUErQixDQUFDLENBQUM7UUFFbkUsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO0lBQ3BCLENBQUM7SUFFRCxXQUFXO1FBQ1QsT0FBa0IsS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDLE9BQU8sRUFBRSxDQUFDLFFBQVMsQ0FBQyxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQztJQUMvRSxDQUFDO0lBT0QsS0FBSyxDQUFDLFFBQVEsQ0FBQyxNQUFrQixFQUFFLE1BQWMsRUFBRSxNQUFpQjtRQUNsRSxLQUFLLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDdkMsSUFBSSxDQUFDLE9BQU8sR0FBRyxNQUFNLElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQztJQUM5QyxDQUFDO0lBS0QsWUFBWSxDQUFDLFNBQWlCLEVBQUUsUUFBZ0I7UUFDOUMsSUFBSSxjQUFjLEdBQVksS0FBSyxDQUFDLFlBQVksQ0FBQyxTQUFTLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDdEUsSUFBSSxDQUFDLGNBQWMsRUFBRTtZQUNuQixRQUFRLFNBQVMsRUFBRTtnQkFDakIsS0FBSyxjQUFjLENBQUMsb0JBQW9CO29CQUN0QyxJQUFJLFFBQVEsQ0FBQyxJQUFJLEVBQUUsSUFBSSxFQUFFO3dCQUN2QixJQUFJLENBQUMsY0FBYyxHQUFHLFFBQVEsQ0FBQztvQkFDakMsTUFBTTtnQkFDUixLQUFLLGNBQWMsQ0FBQyxpQkFBaUI7b0JBQ25DLElBQUksQ0FBQyxlQUFlLEdBQUcsUUFBUSxDQUFDLElBQUksRUFBRSxDQUFDO29CQUN2QyxNQUFNO2dCQUNSO29CQUNFLGNBQWMsR0FBRyxLQUFLLENBQUM7b0JBQ3ZCLE1BQU0sQ0FBQyxRQUFRLENBQUMsMEJBQTBCLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQywrQkFBK0IsRUFBRSxTQUFTLENBQUMsQ0FBQyxDQUFDO29CQUN2RyxNQUFNO2FBQ1Q7U0FDRjtRQUNELE9BQU8sY0FBYyxDQUFDO0lBQ3hCLENBQUM7SUFLTyxLQUFLLENBQUMsZUFBZTtRQUMzQixJQUFJLE9BQU8sR0FBa0IsSUFBSSxDQUFDO1FBQ2xDLElBQUksSUFBSSxDQUFDLFFBQVEsS0FBSyxJQUFJO1lBQ3hCLE9BQU8sT0FBTyxDQUFDO1FBQ2pCLElBQUksV0FBVyxHQUFhLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQzFGLElBQUksV0FBVyxLQUFLLElBQUk7WUFDdEIsT0FBTyxPQUFPLENBQUM7UUFFakIsSUFBSSxLQUFLLEdBQVcsTUFBTSxXQUFXLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDakQsSUFBSSxLQUFLLEtBQUssSUFBSSxFQUFFO1lBQ2xCLE9BQU8sR0FBRyxJQUFJLGFBQWEsRUFBRSxDQUFDO1lBQzlCLE9BQU8sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFZLEtBQUssQ0FBQyxPQUFPLEVBQUUsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1lBQzdELElBQUksSUFBSSxDQUFDLGVBQWUsS0FBSyxJQUFJO2dCQUMvQixPQUFPLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUM7U0FDbEQ7UUFDRCxPQUFPLE9BQU8sQ0FBQztJQUNqQixDQUFDO0lBT0QsWUFBWSxDQUFDLFVBQWtCO1FBQzdCLElBQUksWUFBWSxHQUFZLEtBQUssQ0FBQztRQUNsQyxJQUFJLGdCQUFnQixHQUFtQixLQUFLLENBQUMsT0FBTyxFQUFFLENBQUM7UUFFdkQsS0FBSyxDQUFDLE1BQU0sQ0FBQyxVQUFVLEtBQUssSUFBSSxDQUFDLENBQUM7UUFFbEMsT0FBTyxDQUFDLFlBQVksSUFBSSxnQkFBZ0IsS0FBSyxJQUFJLEVBQUU7WUFDakQsSUFBSSxVQUFVLEtBQUssZ0JBQWdCO2dCQUNqQyxZQUFZLEdBQUcsSUFBSSxDQUFDO2lCQUVqQjtnQkFDSCxJQUFJLFFBQVEsR0FBdUIsZ0JBQWdCLENBQUMsT0FBTyxFQUFFLENBQUM7Z0JBQzlELElBQUksUUFBUSxDQUFDLFNBQVMsRUFBRSxLQUFLLElBQUk7b0JBQy9CLGdCQUFnQixHQUFXLFFBQVEsQ0FBQyxTQUFTLEVBQUUsQ0FBQyxPQUFPLEVBQUUsQ0FBQzs7b0JBRTFELE1BQU07YUFDVDtTQUNGO1FBQ0QsT0FBTyxZQUFZLENBQUM7SUFDdEIsQ0FBQztJQUtNLEtBQUssQ0FBQyxtQkFBbUI7UUFDOUIsSUFBSSxDQUFTLENBQUM7UUFDZCxJQUFJLElBQUksR0FBVyxDQUFDLElBQUksQ0FBQyxRQUFRLEtBQUssSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztRQUMxRSxJQUFJLElBQWMsQ0FBQztRQUNuQixJQUFJLE9BQU8sR0FBWSxLQUFLLENBQUM7UUFFN0IsS0FBSyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxJQUFJLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQyxFQUFFLEVBQUU7WUFDckMsSUFBSSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBRWhDLFFBQVEsSUFBSSxDQUFDLEtBQUssRUFBRSxFQUFFO2dCQUNwQixLQUFLLGFBQWEsQ0FBQyxjQUFjO29CQUMvQixJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksRUFBRSxJQUFJLElBQUksQ0FBQyxNQUFNLEtBQUssSUFBSSxJQUFJLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxFQUFFLEtBQUssSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDLE9BQU8sRUFBRSxFQUFFO3dCQUN0RyxPQUFPLEdBQUcsSUFBSSxDQUFDO3dCQUNmLE1BQU07cUJBQ1A7Z0JBRUgsS0FBSyxhQUFhLENBQUMsZ0JBQWdCLENBQUM7Z0JBQ3BDLEtBQUssYUFBYSxDQUFDLGlCQUFpQixDQUFDO2dCQUNyQyxLQUFLLGFBQWEsQ0FBQyxpQkFBaUIsQ0FBQztnQkFDckMsS0FBSyxhQUFhLENBQUMsZUFBZSxDQUFDO2dCQUNuQyxLQUFLLGFBQWEsQ0FBQyxvQkFBb0I7b0JBQ3JDLElBQUksSUFBSSxDQUFDLFlBQVksRUFBRTt3QkFDckIsT0FBTyxHQUFHLElBQUksQ0FBQztvQkFDakIsTUFBTTtnQkFDUjtvQkFDRSxNQUFNO2FBQ1Q7WUFDRCxJQUFJLE9BQU87Z0JBQ1QsTUFBTSxJQUFJLENBQUMsY0FBYyxFQUFFLENBQUM7U0FDL0I7SUFDSCxDQUFDO0lBRUQsS0FBSyxDQUFDLE1BQU07UUFDVixJQUFJLE9BQU8sR0FBWSxLQUFLLENBQUM7UUFHN0IsSUFBSSxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDL0IsTUFBTSxZQUFZLENBQUMsYUFBYSxDQUFDLG9DQUFvQyxDQUFDLElBQUksRUFBRSxpQkFBaUIsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1lBQ2xILE9BQU8sR0FBRyxJQUFJLENBQUM7U0FDaEI7UUFDRCxPQUFPLE9BQU8sQ0FBQztJQUNqQixDQUFDO0lBRUQsS0FBSyxDQUFDLFVBQVUsQ0FBQyxzQkFBK0I7UUFDOUMsSUFBSSxNQUFNLEdBQVksQ0FBQyxLQUFLLENBQUMsY0FBYyxFQUFFLENBQUM7UUFDOUMsSUFBSSxNQUFNLEVBQUU7WUFDVixNQUFNLEtBQUssQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLGlCQUFpQixFQUFFLElBQUksQ0FBQyxDQUFDO1lBRTdELE1BQU0sS0FBSyxDQUFDLFNBQVMsQ0FBQyxhQUFhLENBQUMsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFDN0QsTUFBTSxHQUFHLE1BQU0sS0FBSyxDQUFDLFVBQVUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1NBQ3pEO1FBRUQsT0FBTyxNQUFNLENBQUM7SUFDaEIsQ0FBQztJQUtELGdCQUFnQjtRQUNkLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQztJQUM3QixDQUFDO0lBS0QsZUFBZTtRQUNiLElBQUksS0FBSyxDQUFDLFNBQVMsRUFBRTtZQUNuQixJQUFJLENBQUMsWUFBWSxHQUFhLGlCQUFpQixDQUFDLFdBQVcsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ2pHLENBQUM7SUFLRCxjQUFjO1FBQ1osT0FBa0IsSUFBSSxDQUFDLFlBQVksQ0FBQztJQUN0QyxDQUFDO0lBS0QsS0FBSyxDQUFDLGVBQWU7UUFDbkIsSUFBSSxJQUFJLENBQUMsTUFBTSxLQUFLLElBQUk7WUFDdEIsTUFBYyxJQUFJLENBQUMsTUFBTyxDQUFDLGFBQWEsRUFBRSxDQUFDO2FBQ3hDO1lBQ0gsS0FBSyxDQUFDLFlBQVksRUFBRSxDQUFDO1lBQ3JCLE1BQU0sS0FBSyxDQUFDLHlCQUF5QixDQUFDLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ3ZGO0lBQ0gsQ0FBQztJQUtELFlBQVksQ0FBQyxPQUFpQjtRQUM1QixJQUFJLENBQUMsVUFBVSxHQUFHLE9BQU8sQ0FBQztJQUM1QixDQUFDO0lBS0QsWUFBWTtRQUNWLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQztJQUN6QixDQUFDO0lBS0QsZ0JBQWdCLENBQUMsTUFBYztRQUM3QixJQUFJLENBQUMsY0FBYyxHQUFHLE1BQU0sQ0FBQztJQUMvQixDQUFDO0lBTUQsc0JBQXNCO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLHFCQUFxQixDQUFDO0lBQ3BDLENBQUM7SUFNRCx1QkFBdUIsQ0FBQyxvQkFBNkI7UUFDbkQsSUFBSSxDQUFDLHFCQUFxQixHQUFHLG9CQUFvQixDQUFDO0lBQ3BELENBQUM7SUFLRCx1QkFBdUIsQ0FBQyxJQUFlO1FBQ3JDLElBQUksUUFBUSxHQUFZLEtBQUssQ0FBQztRQUM5QixJQUFJLElBQUksS0FBSyxJQUFJLEVBQUU7WUFDakIsSUFBSSxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUMsU0FBUyxFQUFFLElBQUksSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDLFNBQVMsRUFBRSxFQUFFO2dCQUM3RCxJQUFJLEtBQUssQ0FBQyxPQUFPLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUMsT0FBTyxFQUFFO29CQUN4RCxRQUFRLEdBQUcsSUFBSSxDQUFDO2FBQ25CO1NBQ0Y7UUFDRCxPQUFPLFFBQVEsQ0FBQztJQUNsQixDQUFDO0lBT0QsVUFBVSxDQUFDLElBQWU7UUFDeEIsSUFBSSxTQUFTLEdBQVksS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDLFNBQVMsRUFBRSxJQUFJLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQyxTQUFTLEVBQUUsQ0FBQztRQUNuRixJQUFJLFlBQVksR0FBWSxDQUFDLFNBQVMsQ0FBRSxJQUFJLENBQUMsSUFBSSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxDQUFDO1FBRWhGLE9BQU8sWUFBWSxDQUFDO0lBQ3RCLENBQUM7SUFNRCxpQkFBaUI7UUFDZixPQUFPLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQztJQUMvQixDQUFDO0lBTUQsa0JBQWtCLENBQUMsZUFBd0I7UUFDekMsSUFBSSxDQUFDLGdCQUFnQixHQUFHLGVBQWUsQ0FBQztJQUMxQyxDQUFDO0lBS0QsZ0JBQWdCO1FBQ2QsSUFBSSxnQkFBZ0IsR0FBWSxLQUFLLENBQUM7UUFDdEMsSUFBSSxjQUFjLEdBQWEsS0FBSyxDQUFDLG1CQUFtQixDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2pHLElBQUksY0FBYyxLQUFLLElBQUksSUFBSSxjQUFjLENBQUMsZ0JBQWdCLEVBQUUsS0FBSyxJQUFJO1lBQ3ZFLGdCQUFnQixHQUFHLElBQUksQ0FBQztRQUUxQixPQUFPLGdCQUFnQixDQUFDO0lBQzFCLENBQUM7SUFNRCxjQUFjO1FBQ1osT0FBTyxJQUFJLENBQUMsY0FBYyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxLQUFLLElBQUksSUFBSSxDQUFTLEtBQUssQ0FBQyxRQUFRLEVBQUcsQ0FBQyxpQkFBaUIsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDNUksQ0FBQztJQVNELEtBQUssQ0FBQywwQkFBMEIsQ0FBQyxNQUFjLEVBQUUsV0FBd0IsRUFBRSxHQUFZLEVBQUUsVUFBbUI7UUFDMUcsSUFBSSxHQUFHLElBQUksS0FBSyxDQUFDLHNCQUFzQixFQUFFO1lBQ3ZDLEdBQUcsR0FBRyxLQUFLLENBQUMsaUJBQWlCLENBQUMsTUFBTSxDQUFDLENBQUM7UUFHeEMsSUFBSSxHQUFHLElBQUksS0FBSyxDQUFDLFNBQVMsRUFBRSxJQUFJLFdBQVcsS0FBSyxXQUFXLENBQUMsZ0JBQWdCLElBQUksSUFBSSxDQUFDLHlCQUF5QixFQUFFO1lBQzlHLEdBQUcsR0FBRyxLQUFLLENBQUM7UUFFZCxJQUFJLFVBQVUsRUFBRTtZQUNkLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxXQUFXLENBQUMsWUFBWSxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxFQUMzRCxXQUFXLEtBQUssV0FBVyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsT0FBTyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1NBRXBJO1FBQ0QsTUFBTSxLQUFLLENBQUMsdUJBQXVCLENBQUMsTUFBTSxFQUFFLFdBQVcsRUFBRSxHQUFHLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBTU8seUJBQXlCO1FBQy9CLElBQUksWUFBWSxHQUFZLEtBQUssQ0FBQztRQUNsQyxJQUFJLFdBQVcsR0FBNkIsS0FBSyxDQUFDLG1CQUFtQixDQUFDLGFBQWEsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDLHVCQUF1QixFQUFFLENBQUM7UUFDdEksSUFBSSxXQUFXLEtBQUssV0FBVyxDQUFDLElBQUksSUFBSSxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsS0FBSyxJQUFJO1lBQ3RFLFlBQVksR0FBRyxJQUFJLENBQUM7UUFDdEIsT0FBTyxZQUFZLENBQUM7SUFDdEIsQ0FBQztJQVFELEtBQUssQ0FBQyw4QkFBOEIsQ0FBQyxNQUFjLEVBQUUsV0FBd0IsRUFBRSxHQUFZO1FBQ3pGLEtBQUssQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUM7UUFFaEMsSUFBSSxDQUFZLEtBQUssQ0FBQyxPQUFPLEVBQUUsQ0FBQyxPQUFPLEVBQUcsQ0FBQyxzQkFBc0I7WUFDL0QsT0FBTztRQUVULElBQUksV0FBVyxHQUE2QixLQUFLLENBQUMsbUJBQW1CLENBQUMsYUFBYSxDQUFDLHNCQUFzQixDQUFDLENBQUMsdUJBQXVCLEVBQUUsQ0FBQztRQUN0SSxJQUFJLElBQUksQ0FBQyxjQUFjLEtBQUssSUFBSSxJQUFJLFdBQVcsS0FBSyxXQUFXLENBQUMsSUFBSSxFQUFFO1lBQ3BFLElBQUksSUFBSSxDQUFDLFlBQVksS0FBSyxJQUFJLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxTQUFTLEVBQUUsRUFBRTtnQkFDL0QsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLENBQUM7Z0JBQzFCLElBQUksV0FBVyxLQUFLLFdBQVcsQ0FBQyxnQkFBZ0IsSUFBSSxHQUFHLElBQUksSUFBSSxDQUFDLGdCQUFnQixFQUFFO29CQUNoRixJQUFJLENBQUMsZ0JBQWdCLEdBQUcsS0FBSyxDQUFDO29CQUM5QixNQUFpQixLQUFLLENBQUMsT0FBTyxFQUFFLENBQUMsT0FBTyxFQUFHLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxZQUFZLEVBQUUsSUFBSSxDQUFDLENBQUM7aUJBQ3JGO2dCQUVELElBQUksV0FBVyxHQUFlLElBQUksQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFLENBQUM7Z0JBQzFELEtBQUssSUFBSSxDQUFDLEdBQVcsQ0FBQyxFQUFFLENBQUMsR0FBRyxXQUFXLENBQUMsWUFBWSxFQUFFLEVBQUUsQ0FBQyxFQUFFLEVBQUU7b0JBQzNELElBQUksS0FBSyxHQUFrQixXQUFXLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO29CQUdsRCxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsRUFBRTt3QkFDMUIsU0FBUztvQkFDWCxJQUFJLFVBQVUsR0FBWSxNQUFNLEtBQUssQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQztvQkFDckUsSUFBSSxVQUFVO3dCQUNaLFVBQVUsR0FBRyxLQUFLLENBQUMsaUJBQWlCLENBQUMsTUFBTSxDQUFDLENBQUM7b0JBSS9DLE1BQU0sS0FBSyxDQUFDLHVCQUF1QixDQUFDLE1BQU0sRUFBRSxXQUFXLEVBQUUsVUFBVSxDQUFDLENBQUM7aUJBQ3RFO2FBQ0Y7U0FDRjthQUNJLElBQUksQ0FBQyxXQUFXLEtBQUssV0FBVyxDQUFDLGdCQUFnQixJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsRUFBRTtZQUNyRixJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksQ0FBQztZQUMxQixNQUFNLFlBQVksQ0FBQyxhQUFhLENBQUMsb0NBQW9DLENBQUMsSUFBSSxFQUFFLGlCQUFpQixDQUFDLG1CQUFtQixDQUFDLENBQUM7U0FDcEg7SUFDSCxDQUFDO0lBS0QsSUFBSTtRQUNGLElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQztJQUN6QixDQUFDO0lBTUQsaUJBQWlCO1FBQ2YsT0FBTyxjQUFjLENBQUMsY0FBYyxDQUFDO0lBQ3ZDLENBQUM7SUFRRCxLQUFLLENBQUMsbUJBQW1CLENBQUMsUUFBYSxFQUFFLFVBQW1CO1FBQzFELEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUlyQyxJQUFJLElBQUksQ0FBQyxjQUFjLEVBQUU7WUFDdkIsSUFBSSxhQUFhLEdBQVEsSUFBSSxDQUFDLEtBQUssQ0FBQztZQUNwQyxJQUFJLEVBQXFCLENBQUM7WUFFMUIsSUFBSSxLQUFLLENBQUMsTUFBTSxFQUFFLEtBQUssSUFBSTtnQkFDekIsT0FBTyxJQUFJLENBQUM7WUFDZCxFQUFFLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxhQUFhLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFFakQsRUFBRSxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBR2QsSUFBSSxFQUFFLENBQUMsZ0JBQWdCLEVBQUU7Z0JBQ3ZCLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDO2dCQUM1QixPQUFPLEtBQUssQ0FBQzthQUNkO1lBRUQsSUFBSSxDQUFDLG1CQUFtQixHQUFHLElBQUksQ0FBQztZQUVoQyxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUMsT0FBTyxFQUFFLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBRy9DLElBQUksVUFBVSxFQUFFO2dCQU1kLElBQUksSUFBSSxDQUFDLE1BQU0sS0FBSyxJQUFJLElBQUksSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLEVBQUUsS0FBSyxnQkFBZ0IsQ0FBQyxPQUFPLElBQUksT0FBTyxRQUFRLEtBQUssUUFBUSxJQUFJLFFBQVEsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxNQUFNLEdBQUcsQ0FBQztvQkFDMUksRUFBRSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQztnQkFFeEIsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDO2FBQ3pCO1NBQ0Y7UUFDRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFPTyxLQUFLLENBQUMsUUFBUSxDQUFDLEVBQXFCO1FBQzFDLElBQUksTUFBTSxHQUFZLEVBQUUsQ0FBQyxTQUFTLEVBQUUsQ0FBQztRQUNyQyxJQUFJLFFBQVEsR0FBUSxFQUFFLENBQUMsWUFBWSxFQUFFLENBQUM7UUFDdEMsSUFBSSxLQUFhLENBQUM7UUFFbEIsSUFBSSxDQUFDLE1BQU0sSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsbUJBQW1CLEVBQUU7WUFDL0MsS0FBSyxHQUFHLE1BQU0sS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQzs7WUFHekMsS0FBSyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sRUFBRSxLQUFLLGdCQUFnQixDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLG1CQUFtQixFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQztRQUN6SSxJQUFJLEtBQUssS0FBSyxJQUFJLEVBQUU7WUFDbEIsSUFBSSxJQUFJLENBQUMsTUFBTSxLQUFLLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxRQUFRLEtBQUssSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLElBQUksQ0FBQyxXQUFXLElBQUksSUFBSSxDQUFDLE1BQU0sS0FBSyxNQUFNLENBQUMsRUFBRTtnQkFDdEcsTUFBYyxJQUFJLENBQUMsTUFBTyxDQUFDLHlCQUF5QixDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztnQkFPdkYsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFLEVBQUU7b0JBQ3RCLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDO29CQUN6QixLQUFLLEdBQVcsSUFBSSxDQUFDLE1BQU8sQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7aUJBQzlDO2FBQ0Y7WUFHRCxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sS0FBSyxJQUFJLElBQUksTUFBTSxDQUFDLElBQUksSUFBSSxDQUFDLE1BQU0sQ0FBQyxtQkFBbUIsRUFBRTtnQkFDdkUsS0FBSyxHQUFHLFFBQVEsQ0FBQztZQUVuQixNQUFNLEtBQUssQ0FBQyx5QkFBeUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7WUFDckQsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUM7WUFFMUIsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUM7U0FDMUI7SUFDSCxDQUFDO0lBS0QsY0FBYztRQUNaLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQztJQUMzQixDQUFDO0lBS0QsZ0JBQWdCO1FBQ2QsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUM7UUFDekIsSUFBSSxDQUFDLGNBQWMsR0FBRyxHQUFHLENBQUM7SUFDNUIsQ0FBQztJQUlELGdCQUFnQjtRQUNkLElBQUksSUFBSSxHQUFlLElBQUksQ0FBQztRQUM1QixJQUFJLElBQUksQ0FBQyxZQUFZLEtBQUssSUFBSTtZQUM1QixJQUFJLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFLRCw0QkFBNEI7UUFDMUIsSUFBSSwwQkFBMEIsR0FBWSxLQUFLLENBQUM7UUFDaEQsS0FBSyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLGFBQWEsQ0FBQyxlQUFlLEVBQUUsbUVBQW1FLENBQUMsQ0FBQztRQUMvSCxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQywwQkFBMEIsQ0FBQztZQUNwRSwwQkFBMEIsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDLHVCQUF1QixFQUFFLEtBQWEsa0JBQWtCLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFdkssT0FBTywwQkFBMEIsQ0FBQztJQUNwQyxDQUFDO0lBSUQsTUFBTSxDQUFDLGlCQUFpQjtRQUN0QixJQUFJLFNBQVMsQ0FBQyxZQUFZLEtBQUssSUFBSSxJQUFJLFNBQVMsQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFLENBQUMsT0FBTyxFQUFFLENBQUMsVUFBVSxFQUFFO1lBQzVGLE9BQU8sS0FBSyxDQUFDO1FBRWYsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBR0QsV0FBVztRQUNULFFBQVEsSUFBSSxDQUFDLElBQUksRUFBRTtZQUNqQixLQUFLLGFBQWEsQ0FBQyxrQkFBa0IsQ0FBQztZQUN0QyxLQUFLLGFBQWEsQ0FBQyxjQUFjLENBQUM7WUFDbEMsS0FBSyxhQUFhLENBQUMsZUFBZSxDQUFDO1lBQ25DLEtBQUssYUFBYSxDQUFDLGNBQWMsQ0FBQztZQUNsQyxLQUFLLGFBQWEsQ0FBQyxlQUFlLENBQUM7WUFDbkMsS0FBSyxhQUFhLENBQUMsYUFBYTtnQkFDOUIsTUFBTTtZQUVSO2dCQUNFLE9BQU8sS0FBSyxDQUFDO1NBQ2hCO1FBR0QsSUFBSSxJQUFJLENBQUMsTUFBTSxLQUFLLElBQUksSUFBSSxJQUFJLENBQUMsU0FBUyxHQUFHLENBQUM7WUFDNUMsT0FBTyxLQUFLLENBQUM7UUFFZixPQUFPLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztJQUM3QixDQUFDO0lBUUQsS0FBSyxDQUFDLDRCQUE0QixDQUFDLFFBQWEsRUFBRSxjQUF1QjtRQUN2RSxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxFQUFFO1lBQ3ZCLE1BQU0sQ0FBQyxRQUFRLENBQUMsZUFBZSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsa0VBQWtFLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDL0gsT0FBTyxLQUFLLENBQUM7U0FDZDtRQUVELElBQUksTUFBTSxHQUFHLE1BQU0sSUFBSSxDQUFDLG1CQUFtQixDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUM1RCxJQUFJLENBQUMsTUFBTTtZQUNULE1BQU0sQ0FBQyxRQUFRLENBQUMsZUFBZSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsa0NBQWtDLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7UUFFakcsSUFBSSxjQUFjLEVBQUU7WUFDbEIsUUFBUSxDQUFDLGVBQWUsQ0FBQyxXQUFXLENBQUMsWUFBWSxFQUFFLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1lBQ25FLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztTQUNuQjtRQUdELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUM7O0FBcGtCTSxzQkFBWSxHQUFjLElBQUksQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7XHJcbiAgQ29tbWFuZHMsXHJcbiAgQ29tbWFuZFR5cGUsXHJcbiAgR3VpRGF0YUNvbGxlY3Rpb24sXHJcbiAgSHRtbFByb3BlcnRpZXMsXHJcbiAgTWdDb250cm9sQmFzZSxcclxuICBNZ0Zvcm1CYXNlLFxyXG4gIFByb3BlcnR5LFxyXG4gIFByb3BJbnRlcmZhY2UsXHJcbiAgUHJvcFBhcmVudEludGVyZmFjZSxcclxuICBWYWxpZGF0aW9uRGV0YWlsc1xyXG59IGZyb20gJ0BtYWdpYy14cGEvZ3VpJztcclxuaW1wb3J0IHtcclxuICBJbnRlcm5hbEludGVyZmFjZSxcclxuICBMb2dnZXIsXHJcbiAgTWdDb250cm9sVHlwZSxcclxuICBTY3JvbGxCYXJUaHVtYlR5cGUsXHJcbiAgU3RvcmFnZUF0dHJpYnV0ZSxcclxuICBTdWJmb3JtVHlwZSxcclxuICBUYWJsZUJlaGF2aW91cixcclxuICBYbWxQYXJzZXJcclxufSBmcm9tICdAbWFnaWMteHBhL3V0aWxzJztcclxuaW1wb3J0IHtEZWJ1ZywgTlN0cmluZ30gZnJvbSAnQG1hZ2ljLXhwYS9tc2NvcmVsaWInO1xyXG5pbXBvcnQge0FyZ3VtZW50c0xpc3R9IGZyb20gJy4uL3J0L0FyZ3VtZW50c0xpc3QnO1xyXG5pbXBvcnQge1Rhc2tCYXNlfSBmcm9tICcuLi90YXNrcy9UYXNrQmFzZSc7XHJcbmltcG9ydCB7TWdGb3JtfSBmcm9tICcuL01nRm9ybSc7XHJcbmltcG9ydCB7RmllbGR9IGZyb20gJy4uL2RhdGEvRmllbGQnO1xyXG5pbXBvcnQge0RhdGFWaWV3fSBmcm9tICcuLi9kYXRhL0RhdGFWaWV3JztcclxuaW1wb3J0IHtDb25zdEludGVyZmFjZX0gZnJvbSAnLi4vQ29uc3RJbnRlcmZhY2UnO1xyXG5pbXBvcnQge0FjY2Vzc0hlbHBlcn0gZnJvbSAnLi4vQWNjZXNzSGVscGVyJztcclxuXHJcbi8vLyA8c3VtbWFyeT5cclxuLy8vICAgZGF0YSBmb3IgPGNvbnRyb2w+IC4uLjwvY29udHJvbD4gdGFnXHJcbi8vLyA8L3N1bW1hcnk+XHJcbi8vLyA8YXV0aG9yPiAgZWh1ZG1cclxuLy8vXHJcbi8vLyA8L2F1dGhvcj5cclxuZXhwb3J0IGNsYXNzIE1nQ29udHJvbCBleHRlbmRzIE1nQ29udHJvbEJhc2UgaW1wbGVtZW50cyBQcm9wUGFyZW50SW50ZXJmYWNlIHtcclxuICBwcml2YXRlIF9mb2N1c2VkU3RvcEV4ZWN1dGlvbjogYm9vbGVhbjsgLy8gdXNlZCB0byByZW1lbWJlciBzdG9wRXhlY3V0aW9uIGZsYWcgaW4gdGhlIGVuZCBvZlxyXG4gIHByaXZhdGUgX2luQ29udHJvbFN1ZmZpeDogYm9vbGVhbjsgLy8gdHJ1ZSwgaXMgY29udHJvbCBpcyBpbiBjb250cm9sIHN1ZmZpeFxyXG4gIHByaXZhdGUgX3J0RXZ0VGFzazogVGFza0Jhc2U7XHJcblxyXG4gIHByaXZhdGUgX3N1YmZvcm1UYXNrSWQ6IHN0cmluZzsgLy8gdGhlIHRhc2sgaWQgb2YgdGhlIHN1YiBmb3JtIHRhc2sgKGlmIHRoaXNcclxuICBwcml2YXRlIF9zdWJmb3JtVGFzazogVGFza0Jhc2U7IC8vIHJlZmVyZW5jZSB0byB0YXNrIG9mIHRoZSBzdWIgZm9ybSB0YXNrXHJcblxyXG4gIElzSW50ZXJhY3RpdmVVcGRhdGU6IGJvb2xlYW47IC8vIHRydWUgaWYgdGhpcyBjb250cm9sIHdhcyBtb2RpZmllZCBieSB1c2VyXHJcbiAgcHJpdmF0ZSByZWZyZXNoT25TdHJpbmc6IHN0cmluZzsgLy8gT25seSBmb3Igb2ZmbGluZS4gR2V0IGl0IGZyb20gWE1MIGFuZCBwdXQgaXQgb24gdGhlIHN1YmZvcm0gdGFzayB3aGVuIHRoZSBzdWJmb3JtIHRhc2sgaXMgb3BlbmVkXHJcbiAgU3ViZm9ybUxvYWRlZDogYm9vbGVhbjsgLy8gdHJ1ZSwgaWYgdGhlIHN1YmZvcm0gdGFzayB3YXMgbG9hZGVkIGFscmVhZHlcclxuICBIYXNab29tSGFuZGxlcjogYm9vbGVhbjsgLy8gaWYgVFJVRSwgdGhlcmUgaXMgYSB6b29tIGhhbmRsZXIgb24gdGhpcyBjdHJsXHJcbiAgQXJnTGlzdDogQXJndW1lbnRzTGlzdDsgLy8gYXJndW1lbnQgbGlzdCBidWlsdCBmcm9tIHRoZSBwcm9wZXJ0eVxyXG5cclxuICBzdGF0aWMgUmV0dXJuVG9DdHJsOiBNZ0NvbnRyb2wgPSBudWxsO1xyXG5cclxuICBwcml2YXRlIGluaXRpYWxpemUoKSB7XHJcbiAgICB0aGlzLl9mb2N1c2VkU3RvcEV4ZWN1dGlvbiA9IGZhbHNlO1xyXG4gICAgdGhpcy5faW5Db250cm9sU3VmZml4ID0gZmFsc2U7XHJcbiAgICB0aGlzLl9ydEV2dFRhc2sgPSBudWxsO1xyXG4gICAgdGhpcy5fc3ViZm9ybVRhc2tJZCA9IG51bGw7XHJcbiAgICB0aGlzLl9zdWJmb3JtVGFzayA9IG51bGw7XHJcbiAgICB0aGlzLklzSW50ZXJhY3RpdmVVcGRhdGUgPSBmYWxzZTtcclxuICAgIHRoaXMucmVmcmVzaE9uU3RyaW5nID0gbnVsbDtcclxuICAgIHRoaXMuU3ViZm9ybUxvYWRlZCA9IGZhbHNlO1xyXG4gICAgdGhpcy5IYXNab29tSGFuZGxlciA9IGZhbHNlO1xyXG4gICAgdGhpcy5BcmdMaXN0ID0gbnVsbDtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gICBDVE9SXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICBjb25zdHJ1Y3RvcigpO1xyXG4gIGNvbnN0cnVjdG9yKHR5cGU6IE1nQ29udHJvbFR5cGUpO1xyXG4gIGNvbnN0cnVjdG9yKHR5cGU6IE1nQ29udHJvbFR5cGUsIHRhc2s6IFRhc2tCYXNlLCBwYXJlbnRDb250cm9sOiBudW1iZXIpO1xyXG4gIGNvbnN0cnVjdG9yKHR5cGU6IE1nQ29udHJvbFR5cGUsIHBhcmVudE1nRm9ybTogTWdGb3JtQmFzZSwgcGFyZW50Q29udHJvbElkeDogbnVtYmVyKTtcclxuICBjb25zdHJ1Y3Rvcih0eXBlPzogTWdDb250cm9sVHlwZSwgdGFza09yUGFyZW50TWdGb3JtPzogYW55LCBwYXJlbnRDb250cm9sT3JQYXJlbnRDb250cm9sSWR4PzogbnVtYmVyKSB7XHJcbiAgICBpZiAoYXJndW1lbnRzLmxlbmd0aCA9PT0gMClcclxuICAgICAgc3VwZXIoKTtcclxuICAgIGVsc2UgaWYgKGFyZ3VtZW50cy5sZW5ndGggPT09IDMgJiYgKHR5cGUgPT09IG51bGwgfHwgdHlwZS5jb25zdHJ1Y3RvciA9PT0gTnVtYmVyKSAmJiAodGFza09yUGFyZW50TWdGb3JtID09PSBudWxsIHx8IHRhc2tPclBhcmVudE1nRm9ybSBpbnN0YW5jZW9mIFRhc2tCYXNlKSAmJiAocGFyZW50Q29udHJvbE9yUGFyZW50Q29udHJvbElkeCA9PT0gbnVsbCB8fCBwYXJlbnRDb250cm9sT3JQYXJlbnRDb250cm9sSWR4LmNvbnN0cnVjdG9yID09PSBOdW1iZXIpKVxyXG4gICAgICBzdXBlcih0eXBlLCB0YXNrT3JQYXJlbnRNZ0Zvcm0uZ2V0Rm9ybSgpLCBwYXJlbnRDb250cm9sT3JQYXJlbnRDb250cm9sSWR4KTtcclxuICAgIGVsc2VcclxuICAgICAgc3VwZXIodHlwZSwgdGFza09yUGFyZW50TWdGb3JtLCBwYXJlbnRDb250cm9sT3JQYXJlbnRDb250cm9sSWR4KTtcclxuXHJcbiAgICB0aGlzLmluaXRpYWxpemUoKTtcclxuICB9XHJcblxyXG4gIEdldFZhckluZGV4KCk6IG51bWJlciB7XHJcbiAgICByZXR1cm4gKDxEYXRhVmlldz5zdXBlci5nZXRGb3JtKCkuZ2V0VGFzaygpLkRhdGFWaWV3KS5EdmNvdW50ICsgdGhpcy52ZWVJbmR4O1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PlxyXG4gIC8vLyBTZXQgYXJndW1lbnRzIGZyb20gdGhlIHByb3BlcnR5XHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICAvLy8gPHBhcmFtIG5hbWU9XCJtZ0Zvcm1cIj48L3BhcmFtPlxyXG4gIC8vLyA8cGFyYW0gbmFtZT1cImRpdElkeFwiPjwvcGFyYW0+XHJcbiAgYXN5bmMgZmlsbERhdGEobWdGb3JtOiBNZ0Zvcm1CYXNlLCBkaXRJZHg6IG51bWJlciwgcGFyc2VyOiBYbWxQYXJzZXIpOiBQcm9taXNlPHZvaWQ+IHtcclxuICAgIHN1cGVyLmZpbGxEYXRhKG1nRm9ybSwgZGl0SWR4LCBwYXJzZXIpO1xyXG4gICAgdGhpcy5BcmdMaXN0ID0gYXdhaXQgdGhpcy5HZXRBcmd1bWVudExpc3QoKTtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gICBzZXQgdGhlIGNvbnRyb2wgYXR0cmlidXRlc1xyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgU2V0QXR0cmlidXRlKGF0dHJpYnV0ZTogc3RyaW5nLCB2YWx1ZVN0cjogc3RyaW5nKTogYm9vbGVhbiB7XHJcbiAgICBsZXQgaXNUYWdQcm9jZXNzZWQ6IGJvb2xlYW4gPSBzdXBlci5TZXRBdHRyaWJ1dGUoYXR0cmlidXRlLCB2YWx1ZVN0cik7XHJcbiAgICBpZiAoIWlzVGFnUHJvY2Vzc2VkKSB7XHJcbiAgICAgIHN3aXRjaCAoYXR0cmlidXRlKSB7XHJcbiAgICAgICAgY2FzZSBDb25zdEludGVyZmFjZS5NR19BVFRSX1NVQkZPUk1fVEFTSzpcclxuICAgICAgICAgIGlmICh2YWx1ZVN0ci50cmltKCkgIT0gXCJcIilcclxuICAgICAgICAgICAgdGhpcy5fc3ViZm9ybVRhc2tJZCA9IHZhbHVlU3RyO1xyXG4gICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgY2FzZSBDb25zdEludGVyZmFjZS5NR19BVFRSX1JFRlJFU0hPTjpcclxuICAgICAgICAgIHRoaXMucmVmcmVzaE9uU3RyaW5nID0gdmFsdWVTdHIudHJpbSgpO1xyXG4gICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgZGVmYXVsdDpcclxuICAgICAgICAgIGlzVGFnUHJvY2Vzc2VkID0gZmFsc2U7XHJcbiAgICAgICAgICBMb2dnZXIuSW5zdGFuY2UuV3JpdGVFeGNlcHRpb25Ub0xvZ1dpdGhNc2coTlN0cmluZy5Gb3JtYXQoXCJVbnJlY29nbml6ZWQgYXR0cmlidXRlOiAnezB9J1wiLCBhdHRyaWJ1dGUpKTtcclxuICAgICAgICAgIGJyZWFrO1xyXG4gICAgICB9XHJcbiAgICB9XHJcbiAgICByZXR1cm4gaXNUYWdQcm9jZXNzZWQ7XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vIGJ1aWxkIEFyZ3VtZW50IGxpc3QgZnJvbSB0aGUgcHJvcGVydHlcclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIHByaXZhdGUgYXN5bmMgR2V0QXJndW1lbnRMaXN0KCk6IFByb21pc2U8QXJndW1lbnRzTGlzdD4ge1xyXG4gICAgbGV0IGFyZ0xpc3Q6IEFyZ3VtZW50c0xpc3QgPSBudWxsO1xyXG4gICAgaWYgKHRoaXMuX3Byb3BUYWIgPT09IG51bGwpXHJcbiAgICAgIHJldHVybiBhcmdMaXN0O1xyXG4gICAgbGV0IEFyZ0xpc3RQcm9wOiBQcm9wZXJ0eSA9IHRoaXMuX3Byb3BUYWIuZ2V0UHJvcEJ5SWQoUHJvcEludGVyZmFjZS5QUk9QX1RZUEVfUEFSQU1FVEVSUyk7XHJcbiAgICBpZiAoQXJnTGlzdFByb3AgPT09IG51bGwpXHJcbiAgICAgIHJldHVybiBhcmdMaXN0O1xyXG5cclxuICAgIGxldCBwYXJtczogc3RyaW5nID0gYXdhaXQgQXJnTGlzdFByb3AuZ2V0VmFsdWUoKTtcclxuICAgIGlmIChwYXJtcyAhPT0gbnVsbCkge1xyXG4gICAgICBhcmdMaXN0ID0gbmV3IEFyZ3VtZW50c0xpc3QoKTtcclxuICAgICAgYXJnTGlzdC5maWxsTGlzdChwYXJtcywgPFRhc2tCYXNlPnN1cGVyLmdldEZvcm0oKS5nZXRUYXNrKCkpO1xyXG4gICAgICBpZiAodGhpcy5yZWZyZXNoT25TdHJpbmcgIT09IG51bGwpXHJcbiAgICAgICAgYXJnTGlzdC5SZWZyZXNoT25TdHJpbmcgPSB0aGlzLnJlZnJlc2hPblN0cmluZztcclxuICAgIH1cclxuICAgIHJldHVybiBhcmdMaXN0O1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PlxyXG4gIC8vLyAgIHJldHVybiB0cnVlIGlmIHRoaXMgY29udHJvbCBpcyBkZXNjZW5kYW50IG9mIHRoZSBwYXJlbnRGb3JtXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICAvLy8gPHBhcmFtIG5hbWUgPSBcInBhcmVudEZvcm1cIj48L3BhcmFtPlxyXG4gIC8vLyA8cmV0dXJucz48L3JldHVybnM+XHJcbiAgaXNEZXNjZW5kZW50KHBhcmVudEZvcm06IE1nRm9ybSk6IGJvb2xlYW4ge1xyXG4gICAgbGV0IGlzRGVzY2VuZGVudDogYm9vbGVhbiA9IGZhbHNlO1xyXG4gICAgbGV0IGNoZWNrQ3VycmVudEZvcm06IE1nRm9ybSA9IDxNZ0Zvcm0+c3VwZXIuZ2V0Rm9ybSgpO1xyXG5cclxuICAgIERlYnVnLkFzc2VydChwYXJlbnRGb3JtICE9PSBudWxsKTtcclxuXHJcbiAgICB3aGlsZSAoIWlzRGVzY2VuZGVudCAmJiBjaGVja0N1cnJlbnRGb3JtICE9PSBudWxsKSB7XHJcbiAgICAgIGlmIChwYXJlbnRGb3JtID09PSBjaGVja0N1cnJlbnRGb3JtKVxyXG4gICAgICAgIGlzRGVzY2VuZGVudCA9IHRydWU7XHJcblxyXG4gICAgICBlbHNlIHtcclxuICAgICAgICBsZXQgY3VyclRhc2s6IFRhc2tCYXNlID0gPFRhc2tCYXNlPmNoZWNrQ3VycmVudEZvcm0uZ2V0VGFzaygpO1xyXG4gICAgICAgIGlmIChjdXJyVGFzay5nZXRQYXJlbnQoKSAhPT0gbnVsbClcclxuICAgICAgICAgIGNoZWNrQ3VycmVudEZvcm0gPSA8TWdGb3JtPmN1cnJUYXNrLmdldFBhcmVudCgpLmdldEZvcm0oKTtcclxuICAgICAgICBlbHNlXHJcbiAgICAgICAgICBicmVhaztcclxuICAgICAgfVxyXG4gICAgfVxyXG4gICAgcmV0dXJuIGlzRGVzY2VuZGVudDtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gICBSZWZyZXNoIGNvbnRyb2wgZGlzcGxheSBpZiBpdCBoYXMgYSBwcm9wZXJ0eSB3aGljaCBpcyBhbiBleHByZXNzaW9uXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICBwdWJsaWMgYXN5bmMgcmVmcmVzaE9uRXhwcmVzc2lvbigpOiBQcm9taXNlPHZvaWQ+IHtcclxuICAgIGxldCBpOiBudW1iZXI7XHJcbiAgICBsZXQgc2l6ZTogbnVtYmVyID0gKHRoaXMuX3Byb3BUYWIgPT09IG51bGwgPyAwIDogdGhpcy5fcHJvcFRhYi5nZXRTaXplKCkpO1xyXG4gICAgbGV0IHByb3A6IFByb3BlcnR5O1xyXG4gICAgbGV0IHJlZnJlc2g6IGJvb2xlYW4gPSBmYWxzZTtcclxuXHJcbiAgICBmb3IgKGkgPSAwOyBpIDwgc2l6ZSAmJiAhcmVmcmVzaDsgaSsrKSB7XHJcbiAgICAgIHByb3AgPSB0aGlzLl9wcm9wVGFiLmdldFByb3AoaSk7XHJcblxyXG4gICAgICBzd2l0Y2ggKHByb3AuZ2V0SUQoKSkge1xyXG4gICAgICAgIGNhc2UgUHJvcEludGVyZmFjZS5QUk9QX1RZUEVfREFUQTpcclxuICAgICAgICAgIGlmICghcHJvcC5pc0V4cHJlc3Npb24oKSAmJiB0aGlzLl9maWVsZCAhPT0gbnVsbCAmJiB0aGlzLl9maWVsZC5nZXRUYXNrKCkgIT09IHRoaXMuZ2V0Rm9ybSgpLmdldFRhc2soKSkge1xyXG4gICAgICAgICAgICByZWZyZXNoID0gdHJ1ZTtcclxuICAgICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgICB9XHJcbiAgICAgICAgLyogIGZhbGxzIHRocm91Z2ggKi9cclxuICAgICAgICBjYXNlIFByb3BJbnRlcmZhY2UuUFJPUF9UWVBFX0ZPUk1BVDpcclxuICAgICAgICBjYXNlIFByb3BJbnRlcmZhY2UuUFJPUF9UWVBFX1ZJU0lCTEU6XHJcbiAgICAgICAgY2FzZSBQcm9wSW50ZXJmYWNlLlBST1BfVFlQRV9FTkFCTEVEOlxyXG4gICAgICAgIGNhc2UgUHJvcEludGVyZmFjZS5QUk9QX1RZUEVfUkFOR0U6XHJcbiAgICAgICAgY2FzZSBQcm9wSW50ZXJmYWNlLlBST1BfVFlQRV9UUkFOU0xBVE9SOlxyXG4gICAgICAgICAgaWYgKHByb3AuaXNFeHByZXNzaW9uKCkpXHJcbiAgICAgICAgICAgIHJlZnJlc2ggPSB0cnVlO1xyXG4gICAgICAgICAgYnJlYWs7XHJcbiAgICAgICAgZGVmYXVsdDpcclxuICAgICAgICAgIGJyZWFrO1xyXG4gICAgICB9XHJcbiAgICAgIGlmIChyZWZyZXNoKVxyXG4gICAgICAgIGF3YWl0IHRoaXMuUmVmcmVzaERpc3BsYXkoKTtcclxuICAgIH1cclxuICB9XHJcblxyXG4gIGFzeW5jIGludm9rZSgpOiBQcm9taXNlPGJvb2xlYW4+IHtcclxuICAgIGxldCBpbnZva2VkOiBib29sZWFuID0gZmFsc2U7XHJcblxyXG4gICAgLy8gaWYgdGhlIGNvbnRyb2wgaXMgbm90IHZpc2libGUsIG5vdCBlbmFibGVkIG9yIG5vdCBwYXJrYWJsZSBza2lwIHRoaXMgY29udHJvbFxyXG4gICAgaWYgKGF3YWl0IHRoaXMuaXNQYXJrYWJsZSh0cnVlKSkge1xyXG4gICAgICBhd2FpdCBBY2Nlc3NIZWxwZXIuZXZlbnRzTWFuYWdlci5oYW5kbGVJbnRlcm5hbEV2ZW50V2l0aE1nQ29udHJvbEJhc2UodGhpcywgSW50ZXJuYWxJbnRlcmZhY2UuTUdfQUNUX0NUUkxfUFJFRklYKTtcclxuICAgICAgaW52b2tlZCA9IHRydWU7XHJcbiAgICB9XHJcbiAgICByZXR1cm4gaW52b2tlZDtcclxuICB9XHJcblxyXG4gIGFzeW5jIGlzUGFya2FibGUoY2hlY2tFbmFibGVkQW5kVmlzaWJsZTogYm9vbGVhbik6IFByb21pc2U8Ym9vbGVhbj4ge1xyXG4gICAgbGV0IHJlc3VsdDogYm9vbGVhbiA9ICFzdXBlci5pc0ltYWdlQ29udHJvbCgpO1xyXG4gICAgaWYgKHJlc3VsdCkge1xyXG4gICAgICBhd2FpdCBzdXBlci5jaGVja1Byb3AoUHJvcEludGVyZmFjZS5QUk9QX1RZUEVfRU5BQkxFRCwgdHJ1ZSk7XHJcblxyXG4gICAgICBhd2FpdCBzdXBlci5jaGVja1Byb3AoUHJvcEludGVyZmFjZS5QUk9QX1RZUEVfVklTSUJMRSwgdHJ1ZSk7XHJcbiAgICAgIHJlc3VsdCA9IGF3YWl0IHN1cGVyLmlzUGFya2FibGUoY2hlY2tFbmFibGVkQW5kVmlzaWJsZSk7XHJcbiAgICB9XHJcblxyXG4gICAgcmV0dXJuIHJlc3VsdDtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gICByZXR1cm5zIHRoZSBzdWJmb3JtIHRhc2sgaWQgb2YgdGhlIGNvbnRyb2xcclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIGdldFN1YmZvcm1UYXNrSWQoKTogc3RyaW5nIHtcclxuICAgIHJldHVybiB0aGlzLl9zdWJmb3JtVGFza0lkO1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PlxyXG4gIC8vLyAgIGZpbmQgdGhlIHN1YmZvcm0gdGFzayBhY2NvcmRpbmcgdG8gaXRzIGlkIGFuZCBzYXZlIGl0XHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICBpbml0U3ViZm9ybVRhc2soKTogdm9pZCB7XHJcbiAgICBpZiAoc3VwZXIuaXNTdWJmb3JtKCkpXHJcbiAgICAgIHRoaXMuX3N1YmZvcm1UYXNrID0gPFRhc2tCYXNlPkd1aURhdGFDb2xsZWN0aW9uLk1HRGF0YVRhYmxlLkdldFRhc2tCeUlEKHRoaXMuX3N1YmZvcm1UYXNrSWQpO1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PlxyXG4gIC8vLyAgIGdldCB0aGUgY29udHJvbCdzIHBhcmVudFRhYmxlXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICBnZXRQYXJlbnRUYWJsZSgpOiBNZ0NvbnRyb2wge1xyXG4gICAgcmV0dXJuIDxNZ0NvbnRyb2w+dGhpcy5fcGFyZW50VGFibGU7XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vICAgcmVzdG9yZSBhbmQgZGlzcGxheSB0aGUgb2xkIHZhbHVlIG9mIHRoZSBjb250cm9sIHVzZWQgd2hlbiB0aGUgY29udHJvbCBpcyBub3QgbW9kaWZpYWJsZVxyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgYXN5bmMgcmVzdG9yZU9sZFZhbHVlKCk6IFByb21pc2U8dm9pZD4ge1xyXG4gICAgaWYgKHRoaXMuX2ZpZWxkICE9PSBudWxsKVxyXG4gICAgICBhd2FpdCAoPEZpZWxkPnRoaXMuX2ZpZWxkKS51cGRhdGVEaXNwbGF5KCk7XHJcbiAgICBlbHNlIHtcclxuICAgICAgc3VwZXIucmVzZXRQcmV2VmFsKCk7XHJcbiAgICAgIGF3YWl0IHN1cGVyLlNldEFuZFJlZnJlc2hEaXNwbGF5VmFsdWUoYXdhaXQgdGhpcy5nZXRNZ1ZhbHVlKHRoaXMuVmFsdWUpLCB0aGlzLklzTnVsbCk7XHJcbiAgICB9XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vICAgc2V0cyB0aGUgZXZlbnQgdGFza1xyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgc2V0UnRFdnRUYXNrKGN1clRhc2s6IFRhc2tCYXNlKTogdm9pZCB7XHJcbiAgICB0aGlzLl9ydEV2dFRhc2sgPSBjdXJUYXNrO1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PlxyXG4gIC8vLyAgIHJldHVybnMgdGhlIGV2ZW50IHRhc2tcclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIGdldFJ0RXZ0VGFzaygpOiBUYXNrQmFzZSB7XHJcbiAgICByZXR1cm4gdGhpcy5fcnRFdnRUYXNrO1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PlxyXG4gIC8vLyAgIHNldHMgdGhlIHN1YmZvcm0gdGFzayBpZCBvZiB0aGUgY29udHJvbFxyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgc2V0U3ViZm9ybVRhc2tJZCh0YXNrSWQ6IHN0cmluZyk6IHZvaWQge1xyXG4gICAgdGhpcy5fc3ViZm9ybVRhc2tJZCA9IHRhc2tJZDtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gICByZXR1cm5zIHN0b3BFeGVjdXRpb24gZmxhZyBpbiB0aGUgZW5kIG9mIGhhbmRsZSBmb2N1cyBvbiB0aGUgY29udHJvbFxyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgLy8vIDxyZXR1cm5zPjwvcmV0dXJucz5cclxuICBpc0ZvY3VzZWRTdG9wRXhlY3V0aW9uKCk6IGJvb2xlYW4ge1xyXG4gICAgcmV0dXJuIHRoaXMuX2ZvY3VzZWRTdG9wRXhlY3V0aW9uO1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PlxyXG4gIC8vLyAgIHJlbWVtYmVyIHN0b3BFeGVjdXRpb24gZmxhZyBpbiB0aGUgZW5kIG9mIGhhbmRsZSBmb2N1cyBvbiB0aGUgY29udHJvbFxyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgLy8vIDxwYXJhbSBuYW1lID0gXCJmb2N1c2VkU3RvcEV4ZWN1dGlvblwiPjwvcGFyYW0+XHJcbiAgc2V0Rm9jdXNlZFN0b3BFeGVjdXRpb24oZm9jdXNlZFN0b3BFeGVjdXRpb246IGJvb2xlYW4pOiB2b2lkIHtcclxuICAgIHRoaXMuX2ZvY3VzZWRTdG9wRXhlY3V0aW9uID0gZm9jdXNlZFN0b3BFeGVjdXRpb247XHJcbiAgfVxyXG5cclxuICAvLy8gPHBhcmFtIG5hbWUgPSBcImN0cmwtXCI+VGhlIGN0cmwgdG8gY2hlY2suIGNhbiBiZSBmcmFtZSBmb3JtIGFuZCBhbHNvIHN1YmZvcm0gY29udHJvbFxyXG4gIC8vLyA8L3BhcmFtPlxyXG4gIC8vLyA8cmV0dXJucz48L3JldHVybnM+XHJcbiAgb25UaGVTYW1lU3ViRm9ybUNvbnRyb2woY3RybDogTWdDb250cm9sKTogYm9vbGVhbiB7XHJcbiAgICBsZXQgcmV0VmFsdWU6IGJvb2xlYW4gPSBmYWxzZTtcclxuICAgIGlmIChjdHJsICE9PSBudWxsKSB7XHJcbiAgICAgIGlmIChzdXBlci5nZXRGb3JtKCkuaXNTdWJGb3JtKCkgJiYgY3RybC5nZXRGb3JtKCkuaXNTdWJGb3JtKCkpIHtcclxuICAgICAgICBpZiAoc3VwZXIuZ2V0Rm9ybSgpLmdldFRhc2soKSA9PT0gY3RybC5nZXRGb3JtKCkuZ2V0VGFzaygpKVxyXG4gICAgICAgICAgcmV0VmFsdWUgPSB0cnVlO1xyXG4gICAgICB9XHJcbiAgICB9XHJcbiAgICByZXR1cm4gcmV0VmFsdWU7XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vICAgdHJ1ZSBpZiB0aGUgY3RybHMgYXJlIG9uIGRpZmZlcmVudCBmb3Jtc1xyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgLy8vIDxwYXJhbSBuYW1lID0gXCJjdHJsXCI+PC9wYXJhbT5cclxuICAvLy8gPHJldHVybnM+PC9yZXR1cm5zPlxyXG4gIG9uRGlmZkZvcm0oY3RybDogTWdDb250cm9sKTogYm9vbGVhbiB7XHJcbiAgICBsZXQgaXNTdWJmb3JtOiBib29sZWFuID0gc3VwZXIuZ2V0Rm9ybSgpLmlzU3ViRm9ybSgpIHx8IGN0cmwuZ2V0Rm9ybSgpLmlzU3ViRm9ybSgpO1xyXG4gICAgbGV0IGlzT25EaWZmRm9ybTogYm9vbGVhbiA9IChpc1N1YmZvcm0gKSAmJiAhdGhpcy5vblRoZVNhbWVTdWJGb3JtQ29udHJvbChjdHJsKTtcclxuXHJcbiAgICByZXR1cm4gaXNPbkRpZmZGb3JtO1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PlxyXG4gIC8vLyAgIHRydWUgaWYgY29udHJvbCBpcyBpbiBjb250cm9sIHN1ZmZpeFxyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgLy8vIDxyZXR1cm5zPjwvcmV0dXJucz5cclxuICBpc0luQ29udHJvbFN1ZmZpeCgpOiBib29sZWFuIHtcclxuICAgIHJldHVybiB0aGlzLl9pbkNvbnRyb2xTdWZmaXg7XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vICAgc2V0IHRydWUgaWYgY29udHJvbCBpcyBpbiBjb250cm9sIHN1ZmZpeFxyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgLy8vIDxwYXJhbSBuYW1lID0gXCJpbkNvbnRyb2xTdWZmaXhcIj48L3BhcmFtPlxyXG4gIHNldEluQ29udHJvbFN1ZmZpeChpbkNvbnRyb2xTdWZmaXg6IGJvb2xlYW4pOiB2b2lkIHtcclxuICAgIHRoaXMuX2luQ29udHJvbFN1ZmZpeCA9IGluQ29udHJvbFN1ZmZpeDtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gICByZXR1cm4gdHJ1ZSBpZiB0aGUgY29udHJvbCBoYXMgc2VsZWN0IHByb2dyYW1cclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIEhhc1NlbGVjdFByb2dyYW0oKTogYm9vbGVhbiB7XHJcbiAgICBsZXQgSGFzU2VsZWN0UHJvZ3JhbTogYm9vbGVhbiA9IGZhbHNlO1xyXG4gICAgbGV0IHNlbGVjdFByb2dQcm9wOiBQcm9wZXJ0eSA9IHN1cGVyLkdldENvbXB1dGVkUHJvcGVydHkoUHJvcEludGVyZmFjZS5QUk9QX1RZUEVfU0VMRUNUX1BST0dSQU0pO1xyXG4gICAgaWYgKHNlbGVjdFByb2dQcm9wICE9PSBudWxsICYmIHNlbGVjdFByb2dQcm9wLkdldENvbXB1dGVkVmFsdWUoKSAhPT0gbnVsbClcclxuICAgICAgSGFzU2VsZWN0UHJvZ3JhbSA9IHRydWU7XHJcblxyXG4gICAgcmV0dXJuIEhhc1NlbGVjdFByb2dyYW07XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vICAgcmV0dXJuIHRydWUgaWYgdGhlIGNvbnRyb2wgdXNlIHRoZSBoYW5kbGVyIFpPT00gKGZvciBzZWxlY3QgcHJvZ3JhbSBvciBab29tIGhhbmRsZXIpXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICAvLy8gPHJldHVybnM+PC9yZXR1cm5zPlxyXG4gIHVzZVpvb21IYW5kbGVyKCk6IGJvb2xlYW4ge1xyXG4gICAgcmV0dXJuIHRoaXMuSGFzWm9vbUhhbmRsZXIgfHwgKHN1cGVyLmdldEZpZWxkKCkgIT09IG51bGwgJiYgKCg8RmllbGQ+c3VwZXIuZ2V0RmllbGQoKSkuZ2V0SGFzWm9vbUhhbmRsZXIoKSB8fCAodGhpcy5IYXNTZWxlY3RQcm9ncmFtKCkpKSk7XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vICAgVXBkYXRlIGJvb2xlYW4gcHJvcGVydHkoZXg6dmlzaWJsZVxcZW5hYmxlKSBvZiBjb250cm9sIGFuZCBpdCdzIGNoaWxkcmVuXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICAvLy8gPHBhcmFtIG5hbWUgPSBcInByb3BJZFwiPnByb3BlcnR5IGlkIDwvcGFyYW0+XHJcbiAgLy8vIDxwYXJhbSBuYW1lID0gXCJjb21tYW5kVHlwZVwiPmNvbW1hbmQgdHlwZSB0aGF0IGJlbG9uZyB0byB0aGUgcHJvcCBpZDwvcGFyYW0+XHJcbiAgLy8vIDxwYXJhbSBuYW1lID0gXCJ2YWxcIj48L3BhcmFtPlxyXG4gIC8vLyA8cGFyYW0gbmFtZSA9IFwidXBkYXRlVGhpc1wiPjwvcGFyYW0+XHJcbiAgYXN5bmMgdXBkYXRlUHJvcGVydHlMb2dpY05lc3RpbmcocHJvcElkOiBudW1iZXIsIGNvbW1hbmRUeXBlOiBDb21tYW5kVHlwZSwgdmFsOiBib29sZWFuLCB1cGRhdGVUaGlzOiBib29sZWFuKTogUHJvbWlzZTx2b2lkPiB7XHJcbiAgICBpZiAodmFsICYmIHN1cGVyLmhhdmVUb0NoZWNrUGFyZW50VmFsdWUoKSlcclxuICAgICAgdmFsID0gc3VwZXIuaXNQYXJlbnRQcm9wVmFsdWUocHJvcElkKTtcclxuXHJcbiAgICAvLyBpZiB0aGUgc3ViZm9ybSBtdXN0IGJlIGludmlzaWJsZSAoQ1RSTF9TVUJGT1JNX05PTkUpIHdlIGNoYW5nZSBpdHMgdmlzaWJpbGl0eSB0byBmYWxzZVxyXG4gICAgaWYgKHZhbCAmJiBzdXBlci5pc1N1YmZvcm0oKSAmJiBjb21tYW5kVHlwZSA9PT0gQ29tbWFuZFR5cGUuUFJPUF9TRVRfVklTSUJMRSAmJiB0aGlzLlNob3VsZFNldFN1YmZvcm1JbnZpc2libGUoKSlcclxuICAgICAgdmFsID0gZmFsc2U7XHJcblxyXG4gICAgaWYgKHVwZGF0ZVRoaXMpIHtcclxuICAgICAgQ29tbWFuZHMuYWRkT3BlcmF0aW9uV2l0aExpbmUoQ29tbWFuZFR5cGUuU0VUX1BST1BFUlRZLCB0aGlzLCBzdXBlci5nZXREaXNwbGF5TGluZShmYWxzZSksXHJcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIGNvbW1hbmRUeXBlID09PSBDb21tYW5kVHlwZS5QUk9QX1NFVF9WSVNJQkxFID8gSHRtbFByb3BlcnRpZXMuVmlzaWJsZSA6IEh0bWxQcm9wZXJ0aWVzLkVuYWJsZWQsIHZhbCk7XHJcblxyXG4gICAgfVxyXG4gICAgYXdhaXQgc3VwZXIudXBkYXRlQ2hpbGRyZW5Qcm9wVmFsdWUocHJvcElkLCBjb21tYW5kVHlwZSwgdmFsKTtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gSWYgdGhlIHN1YmZvcm0gbXVzdCBiZSBpbnZpc2libGUgKGZvciBleGFtcGxlIENUUkxfU1VCRk9STV9OT05FKSByZXR1cm5zIHRydWUuXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICAvLy8gPHJldHVybnM+PC9yZXR1cm5zPlxyXG4gIHByaXZhdGUgU2hvdWxkU2V0U3ViZm9ybUludmlzaWJsZSgpOiBib29sZWFuIHtcclxuICAgIGxldCBzZXRJbnZpc2libGU6IGJvb2xlYW4gPSBmYWxzZTtcclxuICAgIGxldCBzdWJmb3JtVHlwZTogU3ViZm9ybVR5cGUgPSA8U3ViZm9ybVR5cGU+c3VwZXIuR2V0Q29tcHV0ZWRQcm9wZXJ0eShQcm9wSW50ZXJmYWNlLlBST1BfVFlQRV9TVUJGT1JNX1RZUEUpLkdldENvbXB1dGVkVmFsdWVJbnRlZ2VyKCk7XHJcbiAgICBpZiAoc3ViZm9ybVR5cGUgPT09IFN1YmZvcm1UeXBlLk5vbmUgJiYgdGhpcy5nZXRTdWJmb3JtVGFza0lkKCkgPT09IG51bGwpXHJcbiAgICAgIHNldEludmlzaWJsZSA9IHRydWU7XHJcbiAgICByZXR1cm4gc2V0SW52aXNpYmxlO1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PlxyXG4gIC8vLyAgIFVwZGF0ZXMgdmlzaWJpbGl0eSBvciBlbmFibGUgcHJvcGVydHkgb2Ygc3ViZm9ybSBjb250cm9sc1xyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgLy8vIDxwYXJhbSBuYW1lID0gXCJwcm9wSWRcIj5JZCBvZiB2aXNpYmlsaXR5IG9yIGVuYWJsZSBwcm9wZXJ0eTwvcGFyYW0+XHJcbiAgLy8vIDxwYXJhbSBuYW1lID0gXCJjb21tYW5kVHlwZVwiPnR5cGUgb2YgdGhlIEdVSSBjb21tYW5kPC9wYXJhbT5cclxuICAvLy8gPHBhcmFtIG5hbWUgPSBcInZhbFwiPnZhbHVlIGZvciB0aGUgcHJvcGVydHk8L3BhcmFtPlxyXG4gIGFzeW5jIHVwZGF0ZVN1YmZvcm1DaGlsZHJlblByb3BWYWx1ZShwcm9wSWQ6IG51bWJlciwgY29tbWFuZFR5cGU6IENvbW1hbmRUeXBlLCB2YWw6IGJvb2xlYW4pOiBQcm9taXNlPHZvaWQ+IHtcclxuICAgIERlYnVnLkFzc2VydChzdXBlci5pc1N1YmZvcm0oKSk7XHJcblxyXG4gICAgaWYgKCEoPFRhc2tCYXNlPnN1cGVyLmdldEZvcm0oKS5nZXRUYXNrKCkpLkFmdGVyRmlyc3RSZWNvcmRQcmVmaXgpXHJcbiAgICAgIHJldHVybjtcclxuXHJcbiAgICBsZXQgc3ViZm9ybVR5cGU6IFN1YmZvcm1UeXBlID0gPFN1YmZvcm1UeXBlPnN1cGVyLkdldENvbXB1dGVkUHJvcGVydHkoUHJvcEludGVyZmFjZS5QUk9QX1RZUEVfU1VCRk9STV9UWVBFKS5HZXRDb21wdXRlZFZhbHVlSW50ZWdlcigpO1xyXG4gICAgaWYgKHRoaXMuX3N1YmZvcm1UYXNrSWQgIT09IG51bGwgfHwgc3ViZm9ybVR5cGUgPT09IFN1YmZvcm1UeXBlLk5vbmUpIHtcclxuICAgICAgaWYgKHRoaXMuX3N1YmZvcm1UYXNrICE9PSBudWxsICYmIHRoaXMuX3N1YmZvcm1UYXNrLmlzU3RhcnRlZCgpKSB7XHJcbiAgICAgICAgdGhpcy5TdWJmb3JtTG9hZGVkID0gdHJ1ZTtcclxuICAgICAgICBpZiAoY29tbWFuZFR5cGUgPT09IENvbW1hbmRUeXBlLlBST1BfU0VUX1ZJU0lCTEUgJiYgdmFsICYmIHRoaXMuUmVmcmVzaE9uVmlzaWJsZSkge1xyXG4gICAgICAgICAgdGhpcy5SZWZyZXNoT25WaXNpYmxlID0gZmFsc2U7XHJcbiAgICAgICAgICBhd2FpdCAoPFRhc2tCYXNlPnN1cGVyLmdldEZvcm0oKS5nZXRUYXNrKCkpLlN1YmZvcm1SZWZyZXNoKHRoaXMuX3N1YmZvcm1UYXNrLCB0cnVlKTtcclxuICAgICAgICB9XHJcblxyXG4gICAgICAgIGxldCBzdWJmb3JtRm9ybTogTWdGb3JtQmFzZSA9IHRoaXMuX3N1YmZvcm1UYXNrLmdldEZvcm0oKTtcclxuICAgICAgICBmb3IgKGxldCBpOiBudW1iZXIgPSAwOyBpIDwgc3ViZm9ybUZvcm0uZ2V0Q3RybENvdW50KCk7IGkrKykge1xyXG4gICAgICAgICAgbGV0IGNoaWxkOiBNZ0NvbnRyb2xCYXNlID0gc3ViZm9ybUZvcm0uZ2V0Q3RybChpKTtcclxuICAgICAgICAgIC8vIFFDUiAjOTgxNTU1LCBXZSBzaG91bGQgbm90IGFmZmVjdCB2aXNpYmlsaXR5IG9mIHRoZSBUYWJsZUNvbHVtbnMgc2luY2UgaXQgaXMgbWFuYWdlZCBieSB0aGUgdGFibGUuU2V0dGluZyB2aXNpYmlsaXR5XHJcbiAgICAgICAgICAvLyBvZiBhbGwgY29sdW1ucyB0byBmYWxzZSBwcmV2ZW50cyB0YWJsZSBwbGFjZW1lbnRcclxuICAgICAgICAgIGlmICghY2hpbGQuaXNDb2x1bW5Db250cm9sKCkpXHJcbiAgICAgICAgICAgIGNvbnRpbnVlO1xyXG4gICAgICAgICAgbGV0IGNoaWxkVmFsdWU6IGJvb2xlYW4gPSBhd2FpdCBjaGlsZC5jaGVja1Byb3AocHJvcElkLCB0cnVlKSAmJiB2YWw7XHJcbiAgICAgICAgICBpZiAoY2hpbGRWYWx1ZSlcclxuICAgICAgICAgICAgY2hpbGRWYWx1ZSA9IGNoaWxkLmlzUGFyZW50UHJvcFZhbHVlKHByb3BJZCk7XHJcblxyXG4gICAgICAgICAgLy8gaWYgKHRoaXMuX3N1YmZvcm1UYXNrLmlzU3RhcnRlZCgpKVxyXG4gICAgICAgICAgLy8gICBDb21tYW5kcy5hZGRBc3luYyhjb21tYW5kVHlwZSwgY2hpbGQsIGNoaWxkLmdldERpc3BsYXlMaW5lKGZhbHNlKSwgY2hpbGRWYWx1ZSk7XHJcbiAgICAgICAgICBhd2FpdCBjaGlsZC51cGRhdGVDaGlsZHJlblByb3BWYWx1ZShwcm9wSWQsIGNvbW1hbmRUeXBlLCBjaGlsZFZhbHVlKTtcclxuICAgICAgICB9XHJcbiAgICAgIH1cclxuICAgIH1cclxuICAgIGVsc2UgaWYgKChjb21tYW5kVHlwZSA9PT0gQ29tbWFuZFR5cGUuUFJPUF9TRVRfVklTSUJMRSAmJiB2YWwpICYmICF0aGlzLlN1YmZvcm1Mb2FkZWQpIHtcclxuICAgICAgdGhpcy5TdWJmb3JtTG9hZGVkID0gdHJ1ZTtcclxuICAgICAgYXdhaXQgQWNjZXNzSGVscGVyLmV2ZW50c01hbmFnZXIuaGFuZGxlSW50ZXJuYWxFdmVudFdpdGhNZ0NvbnRyb2xCYXNlKHRoaXMsIEludGVybmFsSW50ZXJmYWNlLk1HX0FDVF9TVUJGT1JNX09QRU4pO1xyXG4gICAgfVxyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PlxyXG4gIC8vLyBTZXQgc3ViZm9ybSB0YXNrIGlkIGZvciBhIHN1YmZvcm0gY29udHJvbFxyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgSW5pdCgpOiB2b2lkIHtcclxuICAgIHRoaXMuaW5pdFN1YmZvcm1UYXNrKCk7XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vIGdldCB0aGUgYmVoYXZpb3VyIGZvciB0aGUgdGFibGUgY29udHJvbFxyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgLy8vIDxyZXR1cm5zPjwvcmV0dXJucz5cclxuICBHZXRUYWJsZUJlaGF2aW91cigpOiBUYWJsZUJlaGF2aW91ciB7XHJcbiAgICByZXR1cm4gVGFibGVCZWhhdmlvdXIuVW5saW1pdGVkSXRlbXM7XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vICAgdmFsaWRhdGUgdGhlIHZhbHVlIG9mIHRoZSBjb250cm9sIGFuZCBzZXQgaXQncyB2YWx1ZVxyXG4gIC8vLyA8L3N1bW1hcnk+XHJcbiAgLy8vIDxwYXJhbSBuYW1lID0gXCJOZXdWYWx1ZVwiPnRoZSB2YWx1ZSBvZiB0aGUgY29udHJvbDwvcGFyYW0+XHJcbiAgLy8vIDxwYXJhbSBuYW1lID0gXCJ1cGRhdGVDdHJsXCI+aWYgdHJ1ZSB0aGVuIHRoZSBjb250cm9sIHdpbGwgYmUgdXBkYXRlZCB1cG9uIHN1Y2Nlc3NmdWwgY2hlY2s8L3BhcmFtPlxyXG4gIC8vLyA8cmV0dXJucz5yZXR1cm5zIGlmIHZhbGlkYXRpb24gaXMgc3VjY2Vzc2Z1bDwvcmV0dXJucz5cclxuICBhc3luYyB2YWxpZGF0ZUFuZFNldFZhbHVlKE5ld1ZhbHVlOiBhbnksIHVwZGF0ZUN0cmw6IGJvb2xlYW4pOiBQcm9taXNlPGJvb2xlYW4+IHtcclxuICAgIHN1cGVyLlVwZGF0ZU1vZGlmaWVkQnlVc2VyKE5ld1ZhbHVlKTtcclxuXHJcbiAgICAvLyBpZiB0aGUgdmFsdWUgb2YgdGhlIGNvbnRyb2wgaXMgbnVsbCBvciB3YXMgY2hhbmdlZFxyXG4gICAgLy8gY2hlY2sgdmFsaWRhdGlvbiBhbmQgdXBkYXRlIHRoZSBjb250cm9sIHZhbHVlXHJcbiAgICBpZiAodGhpcy5Nb2RpZmllZEJ5VXNlcikge1xyXG4gICAgICBsZXQgY3RybEN1cnJWYWx1ZTogYW55ID0gdGhpcy5WYWx1ZTtcclxuICAgICAgbGV0IHZkOiBWYWxpZGF0aW9uRGV0YWlscztcclxuXHJcbiAgICAgIGlmIChzdXBlci5nZXRQSUMoKSA9PT0gbnVsbClcclxuICAgICAgICByZXR1cm4gdHJ1ZTtcclxuICAgICAgdmQgPSBzdXBlci5idWlsZFBpY3R1cmUoY3RybEN1cnJWYWx1ZSwgTmV3VmFsdWUpO1xyXG5cclxuICAgICAgdmQuZXZhbHVhdGUoKTtcclxuXHJcbiAgICAgIC8vIHRoZSB2YWxpZGF0aW9uIG9mIHRoZSBmaWVsZHMgdmFsdWUgd2FzIHdyb25nXHJcbiAgICAgIGlmICh2ZC5WYWxpZGF0aW9uRmFpbGVkKSB7XHJcbiAgICAgICAgdGhpcy5Nb2RpZmllZEJ5VXNlciA9IGZhbHNlO1xyXG4gICAgICAgIHJldHVybiBmYWxzZTtcclxuICAgICAgfVxyXG5cclxuICAgICAgdGhpcy5Jc0ludGVyYWN0aXZlVXBkYXRlID0gdHJ1ZTtcclxuXHJcbiAgICAgIHN1cGVyLmdldEZvcm0oKS5nZXRUYXNrKCkuc2V0RGF0YVN5bmNlZChmYWxzZSk7XHJcblxyXG4gICAgICAvLyBzZXQgdGhlIHZhbHVlIChoYW5kbGUgcmVjb21wdXRlLCBldGMuKVxyXG4gICAgICBpZiAodXBkYXRlQ3RybCkge1xyXG4gICAgICAgIC8vIFFDUiM0NDEwMjA6QWZ0ZXIgdmFsaWRhdGluZyBkaXNwbGF5IHZhbHVlLCB0YWtlIHRoZSBzYW1lIGN0cmwncyB2YWx1ZSBmb3IgbWcgdmFsdWUgY29udmVyc2lvbi5cclxuICAgICAgICAvLyBVc2luZyBkaXNwbGF5IHZhbHVlIGdlbmVyYXRlZCBmcm9tIE1nVmFsdWUsIG1pZ2h0IGhhdmUgcHJvYmxlbSB0byBjb252ZXJ0IGJhY2sgdG8gTWdWYWwgdXNpbmcgcGljdHVyZSBkdWUgdG8gYWxpZ25tZW50IChjaGFuZ2VkIGFsaWdubWVudCApLlxyXG4gICAgICAgIC8vIFRoaXMgcHJvYmxlbSBvY2N1cnMgZm9yIG51bWVyaWMuXHJcbiAgICAgICAgLy8gVGhlIGZpeCBzaG91bGQgYmUgYXBwbGljYWJsZSBmb3IgbnVtZXJpYyB0eXBlIG9ubHkuIEJlY2F1c2UgZGF0ZSBhbmQgdGltZSBmaWVsZCB3aXRoIGNvbXBsZXggbWFza2luZyBoYXZlIHByb2JsZW0gb24gdXNpbmcgY3RybCdzIHZhbHVlLlxyXG4gICAgICAgIC8vIFJlZmVyIGRhdGUgcWNyczogIzI5MTY2NiwgIzMwNzM1MywgIzQzNDI0MSBhbmQgZm9yIHRpbWUsIGNoZWNrIGRlZmF1bHQgdmFsdWUgd2hlbiBmb3JtYXQgaXMgJ0hILU1NLVNTIFBNWkAnXHJcbiAgICAgICAgaWYgKHRoaXMuX2ZpZWxkICE9PSBudWxsICYmIHRoaXMuX2ZpZWxkLmdldFR5cGUoKSA9PT0gU3RvcmFnZUF0dHJpYnV0ZS5OVU1FUklDICYmIHR5cGVvZiBOZXdWYWx1ZSA9PT0gJ3N0cmluZycgJiYgTmV3VmFsdWUudHJpbSgpLmxlbmd0aCA+IDApXHJcbiAgICAgICAgICB2ZC5zZXRWYWx1ZShOZXdWYWx1ZSk7XHJcblxyXG4gICAgICAgIGF3YWl0IHRoaXMuc2V0VmFsdWUodmQpO1xyXG4gICAgICB9XHJcbiAgICB9XHJcbiAgICByZXR1cm4gdHJ1ZTtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gICBzZXQgdGhlIHZhbHVlIHJlY2VpdmVkIGZyb20gdGhlIHVzZXIgaW5wdXQgYWZ0ZXIgdmFsaWRhdGlvbiBhbmQgdXBkYXRlIHRoZSBmaWVsZCBjb25uZWN0ZWQgdG8gdGhpc1xyXG4gIC8vLyAgIGNvbnRyb2wgKGlmIGV4aXN0cylcclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIC8vLyA8cGFyYW0gbmFtZSA9IFwidmRcIj50aGUgdmFsaWRhdGlvbiBkZXRhaWxzIG9iamVjdDwvcGFyYW0+XHJcbiAgcHJpdmF0ZSBhc3luYyBzZXRWYWx1ZSh2ZDogVmFsaWRhdGlvbkRldGFpbHMpOiBQcm9taXNlPHZvaWQ+IHtcclxuICAgIGxldCBpc051bGw6IGJvb2xlYW4gPSB2ZC5nZXRJc051bGwoKTtcclxuICAgIGxldCBuZXdWYWx1ZTogYW55ID0gdmQuZ2V0RGlzcFZhbHVlKCk7XHJcbiAgICBsZXQgbWdWYWw6IHN0cmluZztcclxuXHJcbiAgICBpZiAoIWlzTnVsbCB8fCAhdGhpcy5fZmllbGQuaGFzTnVsbERpc3BsYXlWYWx1ZSgpKVxyXG4gICAgICBtZ1ZhbCA9IGF3YWl0IHN1cGVyLmdldE1nVmFsdWUobmV3VmFsdWUpO1xyXG4gICAgLy8gaXQncyBpbm5lciB2YWx1ZSAtPiB0cmFuc2xhdGUgaXQgdG8gdGhlIGRpc3BsYXlcclxuICAgIGVsc2VcclxuICAgICAgbWdWYWwgPSAoKHRoaXMuX2ZpZWxkLmdldFR5cGUoKSA9PT0gU3RvcmFnZUF0dHJpYnV0ZS5CTE9CX1ZFQ1RPUikgPyB0aGlzLl9maWVsZC5nZXRDZWxsRGVmdWFsdFZhbHVlKCkgOiB0aGlzLl9maWVsZC5nZXREZWZhdWx0VmFsdWUoKSk7XHJcbiAgICBpZiAobWdWYWwgIT09IG51bGwpIHtcclxuICAgICAgaWYgKHRoaXMuX2ZpZWxkICE9PSBudWxsICYmICghKG5ld1ZhbHVlID09PSB0aGlzLlZhbHVlKSB8fCB0aGlzLktleVN0cm9rZU9uIHx8IHRoaXMuSXNOdWxsICE9PSBpc051bGwpKSB7XHJcbiAgICAgICAgYXdhaXQgKDxGaWVsZD50aGlzLl9maWVsZCkuc2V0VmFsdWVBbmRTdGFydFJlY29tcHV0ZShtZ1ZhbCwgaXNOdWxsLCB0cnVlLCB0cnVlLCBmYWxzZSk7XHJcblxyXG4gICAgICAgIC8vIGRlZmVjdCAxMjc3MzQgOiB2YWx1ZSBvZiB0aGUgZmllbGQgbWlnaHQgYmUgY2hhbmdlZCBhbHJlYWR5IGluc2lkZSBhIGNvbnRyb2wgY2hhbmdlIGhhbmRsZXJcclxuICAgICAgICAvLyB0byBhdm9pZCB1cGRhdGluZyBvbGQgdmFsdWUgYW5kIG92ZXJydW5uaW5nIHRoZSB2YWx1ZSBmcjBtIHRoZSBjb250cm9sIGNoYW5nZVxyXG4gICAgICAgIC8vIGdldCB0aGUgdmFsdWUgZnJvbSB0aGUgZmllbGQuIChuZWVkIHRvIGJlIGRvbmUgZm9yIGNvbWJvIGFuZCBsaXN0IGJveCwgYnV0IGl0J3MgdG9vIHJpc2t5XHJcbiAgICAgICAgLy8gYW5kIHVubGlrZWx5IHRvIGhhcHBlbiwgc28gdGhlIGZpeCB3aWxsIGJlIGZvciBjaGVjayBib3ggb25seSlcclxuXHJcbiAgICAgICAgaWYgKHN1cGVyLmlzQ2hlY2tCb3goKSkge1xyXG4gICAgICAgICAgdGhpcy5mb3JjZVJlZnJlc2ggPSB0cnVlO1xyXG4gICAgICAgICAgbWdWYWwgPSAoPEZpZWxkPnRoaXMuX2ZpZWxkKS5nZXRWYWx1ZShmYWxzZSk7XHJcbiAgICAgICAgfVxyXG4gICAgICB9XHJcblxyXG4gICAgICAvLyBpZiBmaWVsZCB2YWx1ZSBpcyBudWxsICYgaGFzIG51bGwgZGlzcGxheSB2YWx1ZSwgdGFrZSBldmFsdWF0ZWQgbnVsbCBkaXNwbGF5IHZhbHVlLlxyXG4gICAgICBpZiAoKHRoaXMuX2ZpZWxkICE9PSBudWxsICYmIGlzTnVsbCkgJiYgdGhpcy5fZmllbGQuaGFzTnVsbERpc3BsYXlWYWx1ZSgpKVxyXG4gICAgICAgIG1nVmFsID0gbmV3VmFsdWU7XHJcblxyXG4gICAgICBhd2FpdCBzdXBlci5TZXRBbmRSZWZyZXNoRGlzcGxheVZhbHVlKG1nVmFsLCBpc051bGwpO1xyXG4gICAgICB0aGlzLmZvcmNlUmVmcmVzaCA9IGZhbHNlO1xyXG4gICAgICAvLyBRQ1IgIzQzODc1My4gUHJldmVudCBSUCZSUyBjeWNsZSwgd2hlbiBpdCdzIG5vdCBuZWVkZWQuXHJcbiAgICAgIHRoaXMuS2V5U3Ryb2tlT24gPSBmYWxzZTtcclxuICAgIH1cclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gICByZXR1cm5zIHRoZSBzdWJmb3JtIHRhc2sgb2YgdGhpcyBzdWJmb3JtIGNvbnRyb2xcclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIGdldFN1YmZvcm1UYXNrKCk6IFRhc2tCYXNlIHtcclxuICAgIHJldHVybiB0aGlzLl9zdWJmb3JtVGFzaztcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gICByZXR1cm5zIHRoZSBzdWJmb3JtIHRhc2sgb2YgdGhpcyBzdWJmb3JtIGNvbnRyb2xcclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIHJlc2V0U3ViZm9ybVRhc2soKTogdm9pZCB7XHJcbiAgICB0aGlzLl9zdWJmb3JtVGFzayA9IG51bGw7XHJcbiAgICB0aGlzLl9zdWJmb3JtVGFza0lkID0gXCIwXCI7XHJcbiAgfVxyXG4gIC8vLyA8c3VtbWFyeT5cclxuICAvLy8gcmV0dXJucyBNZ0Zvcm0gb2YgdGhlIHN1YmZvcm0gY29udHJvbCB0YXNrXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICBHZXRTdWJmb3JtTWdGb3JtKCk6IE1nRm9ybUJhc2Uge1xyXG4gICAgbGV0IGZvcm06IE1nRm9ybUJhc2UgPSBudWxsO1xyXG4gICAgaWYgKHRoaXMuX3N1YmZvcm1UYXNrICE9PSBudWxsKVxyXG4gICAgICBmb3JtID0gdGhpcy5fc3ViZm9ybVRhc2suZ2V0Rm9ybSgpO1xyXG4gICAgcmV0dXJuIGZvcm07XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vIENoZWNrIHdoZXRoZXIgdGFibGUgYmVpbmcgdXNlZCB3aXRoIGFic29sdXRlIHNjcm9sbGJhciB0aHVtYi5cclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIElzVGFibGVXaXRoQWJzb2x1dGVTY3JvbGxiYXIoKTogYm9vbGVhbiB7XHJcbiAgICBsZXQgdGFibGVXaXRoQWJzb2x1dGVTY3JvbGxiYXI6IGJvb2xlYW4gPSBmYWxzZTtcclxuICAgIERlYnVnLkFzc2VydCh0aGlzLlR5cGUgPT09IE1nQ29udHJvbFR5cGUuQ1RSTF9UWVBFX1RBQkxFLCBcIkluIE1nQ29udHJvbC5Jc1RhYmxlV2l0aEFic29sdXRlU2Nyb2xsYmFyKCk6IE5vdCBhIHRhYmxlIGNvbnRyb2wuXCIpO1xyXG4gICAgaWYgKHRoaXMuX3Byb3BUYWIucHJvcEV4aXN0cyhQcm9wSW50ZXJmYWNlLlBST1BfVFlQRV9TQ1JPTExfQkFSX1RIVU1CKSlcclxuICAgICAgdGFibGVXaXRoQWJzb2x1dGVTY3JvbGxiYXIgPSAodGhpcy5fcHJvcFRhYi5nZXRQcm9wQnlJZChQcm9wSW50ZXJmYWNlLlBST1BfVFlQRV9TQ1JPTExfQkFSX1RIVU1CKS5HZXRDb21wdXRlZFZhbHVlSW50ZWdlcigpID09PSA8bnVtYmVyPlNjcm9sbEJhclRodW1iVHlwZS5BYnNvbHV0ZSk7XHJcblxyXG4gICAgcmV0dXJuIHRhYmxlV2l0aEFic29sdXRlU2Nyb2xsYmFyO1xyXG4gIH1cclxuXHJcblxyXG4gIC8vLyA8c3VtbWFyeT4gY2hlY2sgaWYgdGhlIHJldHVybi10by1jb250cm9sIHBvaW50cyB0byBhbiBleGlzdGluZyB0YXNrLjwvc3VtbWFyeT5cclxuICBzdGF0aWMgdmFsaWRSZXR1cm5Ub0N0cmwoKTogYm9vbGVhbiB7XHJcbiAgICBpZiAoTWdDb250cm9sLlJldHVyblRvQ3RybCAhPT0gbnVsbCAmJiBNZ0NvbnRyb2wuUmV0dXJuVG9DdHJsLmdldEZvcm0oKS5nZXRUYXNrKCkuaXNBYm9ydGluZygpKVxyXG4gICAgICByZXR1cm4gZmFsc2U7XHJcblxyXG4gICAgcmV0dXJuIHRydWU7XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+IGNoZWNrIGlmIG9uZSBtYXkgZGlyZWN0bHkgKG5vdCB2aWEgdGhlIGd1aSkgc2V0IGEgdmFsdWUgb24gdGhpcyBjb250cm9sLjwvc3VtbWFyeT5cclxuICBtYXlTZXRWYWx1ZSgpOiBib29sZWFuIHtcclxuICAgIHN3aXRjaCAodGhpcy5UeXBlKSB7XHJcbiAgICAgIGNhc2UgTWdDb250cm9sVHlwZS5DVFJMX1RZUEVfQ0hFQ0tCT1g6XHJcbiAgICAgIGNhc2UgTWdDb250cm9sVHlwZS5DVFJMX1RZUEVfVEVYVDpcclxuICAgICAgY2FzZSBNZ0NvbnRyb2xUeXBlLkNUUkxfVFlQRV9DT01CTzpcclxuICAgICAgY2FzZSBNZ0NvbnRyb2xUeXBlLkNUUkxfVFlQRV9MSVNUOlxyXG4gICAgICBjYXNlIE1nQ29udHJvbFR5cGUuQ1RSTF9UWVBFX1JBRElPOlxyXG4gICAgICBjYXNlIE1nQ29udHJvbFR5cGUuQ1RSTF9UWVBFX1RBQjpcclxuICAgICAgICBicmVhaztcclxuXHJcbiAgICAgIGRlZmF1bHQ6XHJcbiAgICAgICAgcmV0dXJuIGZhbHNlO1xyXG4gICAgfVxyXG5cclxuICAgIC8vIHByZXZlbnQgdmFsdWUgc2V0dGluZyBpZiB0aGUgY29udHJvbCBkb2Vzbid0IGhhdmUgYSBmaWVsZCwgb3IgaWYgaXQgZ2V0cyBpdHMgdmFsdWUgZnJvbSBhbiBleHByZXNzaW9uXHJcbiAgICBpZiAodGhpcy5fZmllbGQgPT09IG51bGwgfHwgdGhpcy5fdmFsRXhwSWQgPiAwKVxyXG4gICAgICByZXR1cm4gZmFsc2U7XHJcblxyXG4gICAgcmV0dXJuIHRoaXMuaXNNb2RpZmlhYmxlKCk7XHJcbiAgfVxyXG5cclxuICAvLy8gPHN1bW1hcnk+XHJcbiAgLy8vICAgQ2hlY2tzIGlmIGEgdmFsdWUgbWF5IGJlIHNldCwgYW5kIHRyaWVzIHRvIHNldCB0aGUgdmFsdWUgb2YgdGhlIGNvbnRyb2xcclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIC8vLyA8cGFyYW0gbmFtZSA9IFwiTmV3VmFsdWVcIj50aGUgdmFsdWUgb2YgdGhlIGNvbnRyb2w8L3BhcmFtPlxyXG4gIC8vLyA8cGFyYW0gbmFtZSA9IFwidXBkYXRlQ3RybFwiPmlmIHRydWUgdGhlbiB0aGUgY29udHJvbCB3aWxsIGJlIHVwZGF0ZWQgdXBvbiBzdWNjZXNzZnVsIGNoZWNrPC9wYXJhbT5cclxuICAvLy8gPHJldHVybnM+cmV0dXJucyBpZiB2YWx1ZSBzZXR0aW5nIHdhcyBzdWNjZXNzZnVsIDwvcmV0dXJucz5cclxuICBhc3luYyB2YWxpZGF0ZUFuZFNldFZhbHVlSWZBbGxvd2VkKE5ld1ZhbHVlOiBhbnksIHJlZnJlc2hEaXNwbGF5OiBib29sZWFuKTogUHJvbWlzZTxib29sZWFuPiB7XHJcbiAgICBpZiAoIXRoaXMubWF5U2V0VmFsdWUoKSkge1xyXG4gICAgICBMb2dnZXIuSW5zdGFuY2UuV3JpdGVFcnJvclRvTG9nKE5TdHJpbmcuRm9ybWF0KFwiVW5hYmxlIHRvIGNoYW5nZSB2YWx1ZSBvZiBjb250cm9sIHswfTogVGhlIGNvbnRyb2wgaXMgcmVhZC1vbmx5LlwiLCB0aGlzLk5hbWUpKTtcclxuICAgICAgcmV0dXJuIGZhbHNlO1xyXG4gICAgfVxyXG5cclxuICAgIGxldCByZXRWYWwgPSBhd2FpdCB0aGlzLnZhbGlkYXRlQW5kU2V0VmFsdWUoTmV3VmFsdWUsIHRydWUpO1xyXG4gICAgaWYgKCFyZXRWYWwpXHJcbiAgICAgIExvZ2dlci5JbnN0YW5jZS5Xcml0ZUVycm9yVG9Mb2coTlN0cmluZy5Gb3JtYXQoXCJWYWx1ZSB2YWxpZGF0aW9uIGZhaWxlZCBmb3IgezB9LlwiLCB0aGlzLk5hbWUpKTtcclxuXHJcbiAgICBpZiAocmVmcmVzaERpc3BsYXkpIHtcclxuICAgICAgQ29tbWFuZHMuYWRkTm9QYXJhbWV0ZXJzKENvbW1hbmRUeXBlLlJFRlJFU0hfVEFTSywgdGhpcy5nZXRGb3JtKCkpO1xyXG4gICAgICBDb21tYW5kcy5pbnZva2UoKTtcclxuICAgIH1cclxuXHJcbiAgICAvL3JlZnJlc2ggdGFza1xyXG4gICAgcmV0dXJuIHJldFZhbDtcclxuICB9XHJcbn1cclxuIl19