import { Commands, CommandType, DataModificationTypes, GuiEnvironment, Manager, MgFormBase, PropInterface } from '@magic-xpa/gui';
import { Constants, InternalInterface, Logger, MgControlType, Misc, Priority, XMLConstants } from '@magic-xpa/utils';
import { ApplicationException, Debug, Exception, Int32, NString, RefParam, StringBuilder } from '@magic-xpa/mscorelib';
import { MgControl } from './MgControl';
import { RecordOutOfDataViewException } from '../data/RecordOutOfDataViewException';
import { SET_DISPLAYLINE_BY_DV } from '../data/DataView';
import { TaskBase } from '../tasks/TaskBase';
import { ServerError } from '../remote/ServerError';
import { ConstInterface } from '../ConstInterface';
import { LastFocusedManager } from '../rt/LastFocusedManager';
import { CreatedFormVector } from './CreatedFormVector';
import { AccessHelper } from '../AccessHelper';
export class MgForm extends MgFormBase {
    constructor() {
        super();
        this.IsMovingInView = false;
        this.PrevDisplayLine = 0;
        this.RefreshAllRows = false;
        this.IgnoreFirstRecordCycle = false;
        this.MovedToFirstControl = false;
        this._suffixDone = false;
        this.hiddenControlsIsnsList = null;
        this._transferringDataToGUI = false;
        this.isFetchingRecordsAhead = false;
    }
    static get CreatedForms() {
        return MgForm._createdForms;
    }
    get InRestore() {
        return this._inRestore;
    }
    get TransferringDataToGUI() {
        return this._transferringDataToGUI;
    }
    get shouldFetchRecordsAheadFromServer() {
        if (this.isTableWithPagination()) {
            let prop = this._tableMgControl.GetComputedProperty(PropInterface.PROP_TYPE_SERVER_READ_AHEAD);
            return (prop != null ? prop.GetComputedValueBoolean() : false);
        }
        else
            return false;
    }
    async FetchRecordsAheadFromServer() {
        this.isFetchingRecordsAhead = true;
        await this.GetDataview().FetchRecordsAheadFromServer();
        this.isFetchingRecordsAhead = false;
    }
    GetDataview() {
        let dv = null;
        if (this._task != null)
            dv = this._task.DataView;
        return (dv);
    }
    async moveInView(unit, direction) {
        let oldRecId = Int32.MinValue;
        let currRec = this.GetDataview().getCurrRec();
        let oldTaskMode = ' ';
        let returnToVisibleLine = false;
        let recordOutOfView = false;
        let visibleLine = 0;
        try {
            this.IsMovingInView = true;
            let oldDisplayLine = this.DisplayLine;
            if (unit === Constants.MOVE_UNIT_TABLE) {
                try {
                    if (direction === Constants.MOVE_DIRECTION_BEGIN) {
                        await this.GetDataview().setCurrRecByIdx(Constants.MG_DATAVIEW_FIRST_RECORD, true, false, true, SET_DISPLAYLINE_BY_DV);
                        if (this.isLineMode()) {
                            this.GetDataview().setTopRecIdx(0);
                            await this.setCurrRowByDisplayLine(0, false, false);
                        }
                    }
                    else {
                        await this.GetDataview().setCurrRecByIdx(Constants.MG_DATAVIEW_LAST_RECORD, true, false, true, SET_DISPLAYLINE_BY_DV);
                        if (this.isLineMode()) {
                            this.GetDataview().setTopRecIdx(this.GetDataview().getCurrRecIdx() - this._rowsInPage + 1);
                            if (this.GetDataview().getTopRecIdx() >= 0)
                                await this.setCurrRowByDisplayLine(this.GetDataview().getCurrRecIdx(), false, false);
                            else {
                                this.GetDataview().setTopRecIdx(0);
                                await this.setCurrRowByDisplayLine(this.GetDataview().getCurrRecIdx(), false, false);
                            }
                        }
                    }
                    this.updateDisplayLineByDV();
                    await this.RefreshDisplay(Constants.TASK_REFRESH_FORM);
                }
                catch (ex) {
                    if (ex instanceof RecordOutOfDataViewException)
                        Logger.Instance.WriteDevToLog(ex.Message);
                }
            }
            else {
                this.getTopIndexFromGUI();
                let size;
                if (unit === Constants.MOVE_UNIT_PAGE && this.isLineMode()) {
                    switch (direction) {
                        case Constants.MOVE_DIRECTION_BEGIN:
                            size = this.GetDataview().getCurrRecIdx() - this.GetDataview().getTopRecIdx();
                            break;
                        case Constants.MOVE_DIRECTION_END:
                            let last = this.GetDataview().getTopRecIdx() + this._rowsInPage - 1;
                            last = Math.min(last, this.GetDataview().getSize() - 1);
                            size = last - this.GetDataview().getCurrRecIdx();
                            break;
                        default:
                            returnToVisibleLine = true;
                            size = this._rowsInPage;
                            break;
                    }
                }
                else
                    size = 1;
                size = (direction === Constants.MOVE_DIRECTION_PREV ||
                    direction === Constants.MOVE_DIRECTION_BEGIN
                    ? -size
                    : size);
                if (this.isLineMode()) {
                    visibleLine = this.getVisibleLine();
                    if (visibleLine < 0) {
                        visibleLine = 0;
                        recordOutOfView = true;
                    }
                    if (visibleLine > this._rowsInPage + 1) {
                        visibleLine = this._rowsInPage;
                        recordOutOfView = true;
                    }
                    if (unit === Constants.MOVE_UNIT_PAGE &&
                        (direction === Constants.MOVE_DIRECTION_NEXT ||
                            direction === Constants.MOVE_DIRECTION_PREV) ||
                        unit === Constants.MOVE_UNIT_ROW &&
                            (direction === Constants.MOVE_DIRECTION_NEXT && visibleLine === this._rowsInPage - 1 ||
                                direction === Constants.MOVE_DIRECTION_PREV && visibleLine === 0)) {
                        if (direction === Constants.MOVE_DIRECTION_PREV &&
                            (unit === Constants.MOVE_UNIT_ROW || unit === Constants.MOVE_UNIT_PAGE) &&
                            visibleLine === 0 && this.GetDataview().getCurrRecIdx() === 0 && this.GetDataview().IncludesFirst())
                            return;
                        if (direction === Constants.MOVE_DIRECTION_NEXT && unit === Constants.MOVE_UNIT_PAGE &&
                            visibleLine === this.getLastValidRow() && this.GetDataview().getCurrRecIdx() === this.GetDataview().getSize() - 1 && this.GetDataview().IncludesLast())
                            return;
                        oldRecId = (this.GetDataview().getCurrRec()).getId();
                        this.GetDataview().setTopRecIdx(this.GetDataview().getTopRecIdx() + size);
                        this.GetDataview().setTopRecIdxModified(true);
                        try {
                            this._suffixDone = false;
                            await this.setCurrRowByDisplayLine(this.GetDataview().getCurrRecIdx() + size, true, false);
                            this.GetDataview().setTopRecIdxModified(false);
                            await this.RefreshDisplay(Constants.TASK_REFRESH_FORM);
                        }
                        catch (ex) {
                            if (ex instanceof RecordOutOfDataViewException) {
                                this.GetDataview().setTopRecIdxModified(false);
                                if (AccessHelper.eventsManager.GetStopExecutionFlag()) {
                                    this.GetDataview().restoreTopRecIdx();
                                    await this.restoreOldDisplayLine(oldDisplayLine);
                                    return;
                                }
                                if (this.GetDataview().getTopRecIdx() < 0) {
                                    this.GetDataview().setTopRecIdx(0);
                                    try {
                                        await this.setCurrRowByDisplayLine(0, true, false);
                                    }
                                    catch (exception) {
                                        if (exception instanceof Exception) {
                                            if (AccessHelper.eventsManager.GetStopExecutionFlag()) {
                                                this.GetDataview().restoreTopRecIdx();
                                                await this.restoreOldDisplayLine(oldDisplayLine);
                                                return;
                                            }
                                        }
                                    }
                                    await this.RefreshDisplay(Constants.TASK_REFRESH_FORM);
                                }
                                else if (unit !== Constants.MOVE_UNIT_ROW && this.GetDataview().recExists(this.GetDataview().getTopRecIdx())) {
                                    let newRecId = (this.GetDataview().getCurrRec()).getId();
                                    if (newRecId === oldRecId && !this._suffixDone) {
                                        await AccessHelper.eventsManager.handleInternalEventWithTask(this._task, InternalInterface.MG_ACT_REC_SUFFIX);
                                        if (AccessHelper.eventsManager.GetStopExecutionFlag()) {
                                            this.GetDataview().restoreTopRecIdx();
                                            await this.restoreOldDisplayLine(oldDisplayLine);
                                            return;
                                        }
                                        try {
                                            await this.RefreshDisplay(Constants.TASK_REFRESH_FORM);
                                            await this.setCurrRowByDisplayLine(this.GetDataview().getTopRecIdx() + this.getLastValidRow(), false, true);
                                            visibleLine = this.getLastValidRow();
                                        }
                                        catch (Exception) {
                                        }
                                    }
                                }
                                else {
                                    this.GetDataview().restoreTopRecIdx();
                                    await this.restoreOldDisplayLine(oldDisplayLine);
                                    if (unit === Constants.MOVE_UNIT_ROW && direction === Constants.MOVE_DIRECTION_NEXT) {
                                        if (!this._task.IsInteractive && this._task.getMode() !== Constants.TASK_MODE_CREATE)
                                            this._task.setExecEndTask();
                                        else
                                            await AccessHelper.eventsManager.handleInternalEventWithTask(this._task, InternalInterface.MG_ACT_CRELINE);
                                    }
                                    return;
                                }
                                Logger.Instance.WriteDevToLog(ex.Message);
                            }
                        }
                    }
                    else {
                        try {
                            oldTaskMode = this._task.getMode();
                            oldRecId = (this.GetDataview().getCurrRec()).getId();
                            await this.setCurrRowByDisplayLine(oldDisplayLine + size, true, false);
                            if (!this.GetDataview().recExistsById(oldRecId)) {
                                if (size > 0 || oldTaskMode !== Constants.TASK_MODE_CREATE)
                                    if (size !== -1 || this.GetDataview().recExists(oldDisplayLine))
                                        await this.setCurrRowByDisplayLine(oldDisplayLine, false, false);
                                    else
                                        Logger.Instance.WriteDevToLog(NString.Format("skipped setcurrRow for row {0}", oldDisplayLine));
                                await this.RefreshDisplay(Constants.TASK_REFRESH_FORM);
                            }
                            else if (AccessHelper.eventsManager.GetStopExecutionFlag()) {
                                await this.RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
                            }
                        }
                        catch (ex) {
                            if (ex instanceof RecordOutOfDataViewException) {
                                if (this.GetDataview().recExistsById(oldRecId)) {
                                    await this.restoreOldDisplayLine(oldDisplayLine);
                                    if (!currRec.Synced && oldTaskMode === Constants.TASK_MODE_CREATE) {
                                        this._task.setMode(Constants.TASK_MODE_CREATE);
                                        currRec.clearMode();
                                        currRec.setMode(DataModificationTypes.Insert);
                                        currRec.setNewRec();
                                    }
                                    await this.RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
                                }
                                else {
                                    if (oldTaskMode === Constants.TASK_MODE_CREATE && oldDisplayLine > 0)
                                        await this.restoreOldDisplayLine(oldDisplayLine - 1);
                                    await this.RefreshDisplay(Constants.TASK_REFRESH_FORM);
                                }
                                if (size <= 0 || AccessHelper.eventsManager.GetStopExecutionFlag())
                                    Logger.Instance.WriteDevToLog(ex.Message);
                                else {
                                    if (!this._task.IsInteractive && this._task.getMode() !== Constants.TASK_MODE_CREATE) {
                                        this._task.setExecEndTask();
                                        return;
                                    }
                                    else {
                                        if (this._task.ActionManager.isEnabled(InternalInterface.MG_ACT_CRELINE))
                                            await AccessHelper.eventsManager.handleInternalEventWithTask(this._task, InternalInterface.MG_ACT_CRELINE);
                                        else {
                                            let doRecordSuffix = true;
                                            let refdoRecordSuffix = new RefParam(doRecordSuffix);
                                            if (!await AccessHelper.eventsManager.DoTaskLevelRecordSuffix(this._task, refdoRecordSuffix))
                                                await AccessHelper.eventsManager.DoTaskLevelRecordPrefix(this._task);
                                            doRecordSuffix = refdoRecordSuffix.value;
                                        }
                                    }
                                }
                                return;
                            }
                        }
                    }
                }
                else {
                    try {
                        await this.GetDataview().setCurrRecByIdx(this.GetDataview().getCurrRecIdx() + size, true, false, true, SET_DISPLAYLINE_BY_DV);
                        await this.RefreshDisplay(Constants.TASK_REFRESH_FORM);
                    }
                    catch (ex) {
                        if (ex instanceof RecordOutOfDataViewException) {
                            if (size > 0) {
                                if (!this._task.IsInteractive && this._task.getMode() !== Constants.TASK_MODE_CREATE)
                                    this._task.setExecEndTask();
                                else
                                    await AccessHelper.eventsManager.handleInternalEventWithTask(this._task, InternalInterface.MG_ACT_CRELINE);
                            }
                            else
                                Logger.Instance.WriteDevToLog(ex.Message);
                            return;
                        }
                        else if (ex instanceof ServerError) {
                            await this._task.stop();
                            this._task.abort();
                            await AccessHelper.eventsManager.ProcessAbortingError(ex);
                        }
                        else if (ex instanceof Exception) {
                            Logger.Instance.WriteDevToLog(ex.Message);
                            return;
                        }
                    }
                }
            }
            if (returnToVisibleLine) {
                if (recordOutOfView) {
                    if (visibleLine === 0)
                        this.GetDataview().setTopRecIdx(this.GetDataview().getCurrRecIdx());
                    else
                        this.GetDataview().setTopRecIdx(this.GetDataview().getCurrRecIdx() - this._rowsInPage);
                    this.SetTableTopIndex();
                }
                else {
                    this.SetTableTopIndex();
                    await this.setCurrRowByDisplayLine(this.GetDataview().getTopRecIdx() + visibleLine, false, true);
                }
            }
            else
                this.SetTableTopIndex();
            await AccessHelper.eventsManager.handleInternalEventWithTask(this._task, InternalInterface.MG_ACT_REC_PREFIX);
        }
        catch (e) {
            if (e instanceof RecordOutOfDataViewException) {
                Logger.Instance.WriteDevToLog(e.Message);
            }
        }
        finally {
            this.SelectRow();
            this.IsMovingInView = false;
        }
    }
    async addRec(doSuffix, isCrelineAbove) {
        let newLine = 1;
        let currLine = -1;
        if (!this._task.DataView.isEmptyDataview())
            currLine = this.getVisibleLine();
        if (this.isTableWithAbsoluteScrollbar()) {
            this.GetDataview().TotalRecordsCount += 1;
            this.RefreshAllRows = true;
        }
        let newRecIdx = await this.GetDataview().addRecord(doSuffix, false, isCrelineAbove);
        if (newRecIdx > -1) {
            if (this.isLineMode()) {
                let newCurrRow;
                if (this.GetDataview().getSize() === 0 || this.GetDataview().getTopRecIdx() === Int32.MinValue) {
                    this.GetDataview().setTopRecIdx(0);
                    newCurrRow = 0;
                }
                else if (currLine === this._rowsInPage - 1) {
                    this.GetDataview().setTopRecIdx(this.GetDataview().getTopRecIdx() + 1);
                    newCurrRow = this.getVisibleLine();
                }
                else {
                    if (isCrelineAbove)
                        newCurrRow = currLine > 0 ? currLine : 0;
                    else
                        newCurrRow = currLine + 1;
                }
                newLine = this.GetDataview().getTopRecIdx() + newCurrRow;
                this.removeRecordsAfterIdx(this.GetDataview().getTopRecIdx() + newCurrRow);
                try {
                    if (!this._task.DataView.isEmptyDataview())
                        this._task.setMode(Constants.TASK_MODE_CREATE);
                    await this.setCurrRowByDisplayLine(newLine, false, false);
                }
                catch (ex) {
                    if (ex instanceof RecordOutOfDataViewException) {
                        Logger.Instance.WriteDevToLog(NString.Format("in Form.addRec() {0}", ex.Message));
                    }
                }
                await this.RefreshDisplay(Constants.TASK_REFRESH_TABLE);
            }
        }
        else if (this.isTableWithAbsoluteScrollbar()) {
            this.GetDataview().TotalRecordsCount -= 1;
            this.RefreshAllRows = false;
        }
    }
    async delCurrRec() {
        let topTableRow = super.isLineMode() && this.GetDataview().getTopRecIdx() === this.GetDataview().getCurrRecIdx();
        await this.GetDataview().removeCurrRec();
        if (this.GetDataview().isEmpty() && !this._task.IsTryingToStop) {
            if (this._task.DataView.isEmptyDataview()) {
                await this.addRec(false, false);
            }
            else {
                if (await this._task.checkProp(PropInterface.PROP_TYPE_ALLOW_CREATE, true)) {
                    if (this._task.getMode() !== Constants.TASK_MODE_CREATE)
                        this._task.setMode(Constants.TASK_MODE_CREATE);
                    this._task.enableCreateActs(true);
                    await AccessHelper.eventsManager.handleInternalEventWithTask(this._task, InternalInterface.MG_ACT_CRELINE);
                }
                else
                    await AccessHelper.eventsManager.handleInternalEventWithTask(this._task, InternalInterface.MG_ACT_EXIT);
            }
        }
        if (!this.GetDataview().isEmpty()) {
            if (topTableRow)
                this.GetDataview().setTopRecIdx(this.GetDataview().getCurrRecIdx());
            if (super.isLineMode()) {
                if (super.HasTable()) {
                    if (this.isTableWithAbsoluteScrollbar()) {
                        this.GetDataview().TotalRecordsCount--;
                        this.RefreshAllRows = true;
                    }
                    this.removeRecordsAfterIdx(this.GetDataview().getCurrRecIdx());
                    await this.RefreshDisplay(Constants.TASK_REFRESH_FORM);
                }
                else
                    await this.RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
            }
            else {
                try {
                    await this.GetDataview().setCurrRecByIdx(this.GetDataview().getCurrRecIdx(), true, true, true, SET_DISPLAYLINE_BY_DV);
                }
                catch (ex) {
                    if (ex instanceof RecordOutOfDataViewException) {
                    }
                    else
                        throw ex;
                }
                await this.RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
            }
        }
    }
    async cancelEdit(isActCancel, isQuitEvent) {
        let mode = this._task.getMode();
        let newMode;
        let row = 0;
        let execRecPrefix = false;
        if (this.GetDataview().getSize() === 1 && mode === Constants.TASK_MODE_CREATE)
            newMode = Constants.TASK_MODE_CREATE;
        else
            newMode = Constants.TASK_MODE_MODIFY;
        let recRemoved = await this.GetDataview().cancelEdit(false, false);
        this.GetDataview().setPrevCurrRec(null);
        if ((this.GetDataview().isEmpty() || ((isActCancel && recRemoved) && (!GuiEnvironment.Environment.GetSpecialCancelOnCreate()) && this._task.getOriginalTaskMode() === Constants.TASK_MODE_CREATE)) && !this._task.IsTryingToStop) {
            let orgCancelWasRaised = this._task.cancelWasRaised();
            this._task.setCancelWasRaised(isActCancel);
            await AccessHelper.eventsManager.handleInternalEventWithTask(this._task, InternalInterface.MG_ACT_CRELINE);
            this._task.setCancelWasRaised(orgCancelWasRaised);
        }
        else if (isActCancel)
            execRecPrefix = true;
        if (!this.GetDataview().isEmpty()) {
            if (super.isLineMode()) {
                try {
                    if (super.HasTable()) {
                        row = this.getVisibleLine();
                        if (row < 0 || row >= this._rowsInPage) {
                            let topRecIdx = this.GetDataview().getTopRecIdx();
                            let currRecIdx = this.GetDataview().getCurrRecIdx();
                            if (topRecIdx > currRecIdx)
                                this.GetDataview().setTopRecIdx(currRecIdx);
                            row = 0;
                        }
                        await this.setCurrRowByDisplayLine(this.GetDataview().getTopRecIdx() + row, false, true);
                    }
                }
                catch (ex) {
                    if (ex instanceof RecordOutOfDataViewException) {
                        if (row !== 0)
                            Logger.Instance.WriteDevToLog(NString.Format("in Form.cancelEdit() {0}", ex.Message));
                    }
                    else
                        throw ex;
                }
                if (super.HasTable()) {
                    if (this.isTableWithAbsoluteScrollbar() && recRemoved) {
                        this.GetDataview().TotalRecordsCount--;
                        this.RefreshAllRows = true;
                    }
                    this.removeRecordsAfterIdx(this.GetDataview().getCurrRecIdx());
                }
            }
            else {
                try {
                    await this.GetDataview().setCurrRecByIdx(this.GetDataview().getCurrRecIdx(), false, true, true, SET_DISPLAYLINE_BY_DV);
                }
                catch (ex) {
                    if (ex instanceof RecordOutOfDataViewException) {
                        Logger.Instance.WriteDevToLog("in Form.cancelEdit() error in Screen mode for Current Record");
                    }
                    else
                        throw ex;
                }
            }
            this._task.setMode(newMode);
            if (!this._task.InEndTask)
                await this.RefreshDisplay(Constants.TASK_REFRESH_FORM);
            if (!this._task.InEndTask)
                await this.RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
        }
        if (execRecPrefix) {
            this._task.setLevel(Constants.TASK_LEVEL_TASK);
            let lastParkedCtrl = LastFocusedManager.getLastFocusedControl();
            if (lastParkedCtrl !== null) {
                if (lastParkedCtrl.getForm().getTask().pathContains(this._task))
                    for (let parent = lastParkedCtrl.getForm().getTask(); parent !== null && parent !== this._task; parent = parent.getParent()) {
                        parent.setLevel(Constants.TASK_LEVEL_TASK);
                        (parent.DataView.getCurrRec()).resetModified();
                    }
                else
                    lastParkedCtrl.getForm().getTask().setLevel(Constants.TASK_LEVEL_TASK);
            }
            if (!isQuitEvent) {
                await AccessHelper.eventsManager.handleInternalEventWithTask(this._task, InternalInterface.MG_ACT_REC_PREFIX);
                await AccessHelper.eventsManager.HandleParkabilityOnRowChange(this._task);
            }
        }
    }
    async refreshOnExpressions() {
        for (let i = 0; i < this.CtrlTab.getSize(); i = i + 1) {
            let mgControl = ((this.CtrlTab.getCtrl(i) instanceof MgControl) ? this.CtrlTab.getCtrl(i) : null);
            if (mgControl != null)
                await mgControl.refreshOnExpression();
            else
                Debug.Assert(false);
        }
        await super.refreshPropsOnExpression();
    }
    removeRefsToCtrls() {
        let lastFocusedControl = LastFocusedManager.getLastFocusedControl();
        if (lastFocusedControl !== null && lastFocusedControl.getForm() === this)
            LastFocusedManager.setLastFocusedControl(this._task, null);
        super.removeRefsToCtrls();
    }
    getCtrlByCtrlName(ctrlName) {
        return this.CtrlTab.getCtrl(ctrlName);
    }
    getSubFormCtrlByName(ctrlName) {
        let guiParentTask;
        let destSubForm = null;
        for (guiParentTask = this._task; destSubForm === null && guiParentTask !== null && guiParentTask.getForm() !== null; guiParentTask = guiParentTask.getParent()) {
            destSubForm = guiParentTask.getForm().getCtrlByName(ctrlName, MgControlType.CTRL_TYPE_SUBFORM);
            if (destSubForm == null) {
                if (!guiParentTask.IsSubForm || NString.Equals(guiParentTask.getForm().getSubFormCtrl().Name, ctrlName, true))
                    break;
            }
        }
        return destSubForm;
    }
    setSuffixDone() {
        this._suffixDone = true;
    }
    getDestinationRow() {
        return this._destTblRow;
    }
    setSubFormCtrl(subFormCtrl) {
        Debug.Assert(this._subFormCtrl === null && subFormCtrl !== null);
        this._subFormCtrl = subFormCtrl;
    }
    getSubFormCtrlForTask(taskId) {
        let curr = null;
        let parentCtrlTab;
        let guiParentForm = null;
        let destSubForm = null;
        for (guiParentForm = this; guiParentForm != null; guiParentForm = guiParentForm.ParentForm) {
            parentCtrlTab = guiParentForm.CtrlTab;
            for (let i = 0; i < parentCtrlTab.getSize(); i++) {
                curr = ((parentCtrlTab.getCtrl(i) instanceof MgControl) ? parentCtrlTab.getCtrl(i) : null);
                if (curr != null && curr.isSubform() && (curr.getSubformTaskId() != null))
                    if (curr.getSubformTaskId() === taskId) {
                        destSubForm = curr;
                        break;
                    }
            }
            if (destSubForm != null)
                break;
        }
        return destSubForm;
    }
    removeRecordsAfterIdx(idx) {
        let removeAll = false;
        if (this.isTableWithPagination())
            this.setTableItemsCount(false);
        else {
            if (idx <= this._rowsInPage + 1 && this.GetDataview().getSize() > this._rowsInPage)
                idx = 0;
            if (idx === 0)
                removeAll = true;
            super.SetTableItemsCount(idx, removeAll);
            this.setTableItemsCount(false);
        }
    }
    isRowValidated(idx) {
        if (this.Rows.length <= idx || idx < 0)
            return false;
        let row = this.Rows.get_Item(idx);
        if (row == null)
            return false;
        return row.Created && row.Validated;
    }
    async transferDataToGui() {
        let saveRowIdx = this.GetDataview().getCurrRecIdx();
        let maxTime = Misc.getSystemMilliseconds() + MgForm.TIME_LIMIT;
        let updated = false;
        let currentEditingControl = super.getTask().CurrentEditingControl;
        try {
            let bkpRecord = this.GetDataview().backupCurrent();
            while (Misc.getSystemMilliseconds() < maxTime && this.RefreshRepeatableAllowed) {
                if (this._lastRowSent + 1 >= this.GetDataview().getSize()) {
                    this.RefreshUI();
                    break;
                }
                else if (this.isTableWithPagination() && this._lastRowSent + 1 >= this.getMaxRowsInPaginatedTable()) {
                    this.RefreshUI();
                    break;
                }
                this._lastRowSent++;
                this._transferringDataToGUI = true;
                this.AllowedSubformRecompute = false;
                if (!this.isRowValidated(this._lastRowSent)) {
                    super.checkAndCreateRow(this._lastRowSent);
                    if (this._lastRowSent !== saveRowIdx) {
                        super.getTask().CurrentEditingControl = null;
                        await this.setCurrRowByDisplayLine(this._lastRowSent, false, true);
                        await super.refreshControls(true);
                        updated = true;
                    }
                }
                this.AllowedSubformRecompute = true;
                this._transferringDataToGUI = false;
            }
            if (updated) {
                await this.restoreBackup(saveRowIdx, bkpRecord);
                await super.refreshControls(true);
                super.getTask().CurrentEditingControl = currentEditingControl;
            }
        }
        catch (ex) {
            if (ex instanceof RecordOutOfDataViewException) {
                Logger.Instance.WriteDevToLog(ex.Message);
            }
            else
                throw ex;
        }
        this._transferingData = false;
        this.checkAndCreateRowsEvent();
    }
    checkAndCreateRowsEvent() {
        if (this._tableMgControl !== null && this.Opened && this.RefreshRepeatableAllowed) {
            let size = this.GetDataview().getSize();
            if (this._lastRowSent >= this.Rows.length) {
                this._lastRowSent = this.Rows.length - 1;
            }
            let transferDataToGui = this._lastRowSent < (size - 1);
            if (this.isTableWithPagination() && (this._lastRowSent + 1) >= this.getMaxRowsInPaginatedTable())
                transferDataToGui = false;
            if (transferDataToGui && !this._transferingData) {
                AccessHelper.eventsManager.addInternalEventWithCtrlAndCodeAndPriority(this._tableMgControl, InternalInterface.MG_ACT_DV_TO_GUI, Priority.LOWEST);
                this._transferingData = true;
            }
            else if (this.shouldFetchRecordsAheadFromServer && !transferDataToGui && (!this.GetDataview().IncludesLast() || !this.GetDataview().IncludesFirst())) {
                AccessHelper.eventsManager.addInternalEventWithCtrlAndCodeAndPriority(this._tableMgControl, InternalInterface.MG_ACT_FETCH_RECORDS_AHEAD_FROM_SERVER, Priority.LOWEST);
            }
        }
    }
    async setRowData(desiredTopIndex, sendAll) {
        let saveRowIdx = this.GetDataview().getCurrRecIdx();
        let diff = 0;
        let getPrevChunk = false;
        let getNextChunk = false;
        let updated = false;
        if (desiredTopIndex !== Int32.MinValue) {
            if (desiredTopIndex < 0) {
                getPrevChunk = true;
                this.GetDataview().setTopRecIdx(0);
            }
            else if (desiredTopIndex + this._rowsInPage + 1 >= this.GetDataview().getSize())
                getNextChunk = true;
        }
        else
            desiredTopIndex = this.GetDataview().getTopRecIdx();
        let bkpRecord = this.GetDataview().backupCurrent();
        this._transferringDataToGUI = true;
        this.AllowedSubformRecompute = false;
        let diffRef = new RefParam(diff);
        updated = await this.refreshRows(desiredTopIndex, sendAll, saveRowIdx, diffRef);
        diff = diffRef.value;
        if ((getPrevChunk && diff !== 0) || getNextChunk) {
            this.GetDataview().setTopRecIdx(desiredTopIndex + diff);
            this.SetTableTopIndex();
            if (this._topIndexUpdated) {
                this._topIndexUpdated = false;
            }
        }
        this.AllowedSubformRecompute = true;
        this._transferringDataToGUI = false;
        if (updated) {
            await this.restoreBackup(saveRowIdx + diff, bkpRecord);
            await super.refreshControls(true);
        }
        super.SelectRow();
        if (getPrevChunk) {
            let lastFocusedControl = LastFocusedManager.getLastFocusedControl();
            if (lastFocusedControl !== null && lastFocusedControl.getParentTable() === this._tableMgControl) {
            }
        }
    }
    async refreshRows(desiredTopIndex, sendAll, saveRowIdx, diffRef) {
        let index = 0;
        let updated = false;
        diffRef.value = 0;
        let orgIncludesFirst = this.GetDataview().IncludesFirst();
        let orgIncludesLast = this.GetDataview().IncludesLast();
        for (let i = 0; i < this._rowsInPage; i++) {
            let idx = desiredTopIndex + i;
            let prevTopIdx = this.GetDataview().getTopRecIdx();
            try {
                index = idx + diffRef.value;
                if (!this.isRowValidated(idx) || sendAll || this.RefreshAllRows) {
                    if (index >= 0)
                        super.checkAndCreateRow(index);
                    if (saveRowIdx + diffRef.value !== index) {
                        await this.setCurrRowByDisplayLine(index, false, true);
                        await super.refreshControls(true);
                    }
                    updated = true;
                }
            }
            catch (ex) {
                if (ex instanceof RecordOutOfDataViewException) {
                    if (super.isRowCreated(index))
                        super.markRowNOTCreated(index);
                    break;
                }
                else
                    throw ex;
            }
            finally {
                diffRef.value = diffRef.value + (this.GetDataview().getTopRecIdx() - prevTopIdx);
                prevTopIdx = this.GetDataview().getTopRecIdx();
                if (!updated && this.GetDataview().DataviewBoundriesAreChanged(orgIncludesFirst, orgIncludesLast))
                    updated = true;
            }
        }
        this.RefreshUI();
        this.RefreshAllRows = false;
        return updated;
    }
    getVisibleLine() {
        let currRecIdx = 0;
        if (this.GetDataview().getCurrRecIdx() !== Int32.MinValue)
            currRecIdx = this.GetDataview().getCurrRecIdx();
        return currRecIdx - Math.max(this.GetDataview().getTopRecIdx(), 0);
    }
    async bringRecordToPage() {
        if (this._tableMgControl !== null && !this._task.DataView.isEmptyDataview()) {
            let topIndex = Commands.getTopIndex(this._tableMgControl);
            let currIdx = this.GetDataview().getCurrRecIdx();
            let newTopIndex = topIndex;
            if (!this.isTableWithPagination()) {
                if (topIndex > currIdx) {
                    this._topIndexUpdated = true;
                    newTopIndex = currIdx;
                }
                else if (topIndex + this._rowsInPage - 1 < currIdx) {
                    this._topIndexUpdated = true;
                    if (this._rowsInPage > 0)
                        newTopIndex = currIdx - this._rowsInPage + 1;
                    else
                        newTopIndex = currIdx;
                }
            }
            else {
                let currGuiIdx = this.getGuiRowidx(currIdx);
                let newTopGuiIndex = (currGuiIdx) - ((currGuiIdx) % this._rowsInPage);
                newTopIndex = this.getDvRowIdx(newTopGuiIndex);
                if (newTopIndex !== this._prevGuiTopIndex || this.IsMovingInView)
                    this._topIndexUpdated = true;
            }
            if (this.GetDataview().getTopRecIdx() !== newTopIndex || this._topIndexUpdated) {
                this.GetDataview().setTopRecIdx(newTopIndex);
                this.SetTableTopIndex();
                await this.setRowData(Int32.MinValue, false);
                this._topIndexUpdated = false;
            }
        }
    }
    invalidateTable() {
        for (let i = 0; i < this.Rows.length; i++) {
            let row = this.Rows.get_Item(i);
            if (row !== null)
                row.Validated = false;
        }
        this._lastRowSent = -1;
        this.checkAndCreateRowsEvent();
    }
    getPrevLine(line) {
        return line - 1;
    }
    clearTableColumnSortMark(clearSortMark) {
        if (clearSortMark) {
        }
    }
    async stopRowEditing(idx) {
        if (this.getIsRowEditingFromGui(idx)) {
            Commands.addValueWithLine(CommandType.SET_NOT_IS_ROW_EDITING, this, this.isLineMode() ? idx : 0, this.isLineMode());
            await this.RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
            Commands.addNoParameters(CommandType.REFRESH_TASK, this.getForm());
        }
    }
    async startRowEditing(idx) {
        if (!this.getIsRowEditingFromGui(idx)) {
            Commands.addValueWithLine(CommandType.SET_IS_ROW_EDITING, this, this.isLineMode() ? idx : 0, this.isLineMode());
            await this.RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
            Commands.addNoParameters(CommandType.REFRESH_TASK, this.getForm());
        }
    }
    getIsRowEditingFromGui(displayIndex) {
        let isRowEditing = Commands.getIsRowEditing(this, this.isLineMode() ? displayIndex : 0, this.isLineMode());
        return isRowEditing;
    }
    async RefreshDisplay(refreshType) {
        let task = this._task;
        if (this.isFetchingRecordsAhead)
            return false;
        if (task.isAborting())
            return false;
        let currRec = this.GetDataview().getCurrRec();
        if (currRec !== null && currRec.InCompute && !currRec.isNewRec())
            return false;
        if (this._inRefreshDisp)
            return false;
        this._inRefreshDisp = true;
        try {
            if (refreshType === Constants.TASK_REFRESH_FORM || refreshType === Constants.TASK_REFRESH_TABLE) {
                if (super.HasTable() && this.Opened)
                    await this.refreshTable();
            }
            if (refreshType === Constants.TASK_REFRESH_FORM || refreshType === Constants.TASK_REFRESH_CURR_REC) {
                if (refreshType === Constants.TASK_REFRESH_CURR_REC && super.hasTable()) {
                    this.SetTableTopIndex();
                    super.SelectRow();
                    await this.fetchPrevChunkForCurrentPage();
                }
                await super.refreshProps();
                await super.refreshControls(false);
            }
        }
        catch (ex) {
            if (ex instanceof Exception) {
                Logger.Instance.WriteExceptionToLogWithMsg(NString.Format("{0} : {1} in task {2}", ex.GetType(), ex.Message, super.getTask().PreviouslyActiveTaskId));
            }
            else
                throw ex;
        }
        finally {
            this._inRefreshDisp = false;
        }
        if (refreshType === Constants.TASK_REFRESH_CURR_REC && (!super.isLineMode() || this._tableRefreshed)) {
            this.FormRefreshed = true;
            if (this._task.isFirstRecordCycle() || AccessHelper.eventsManager.getIsInViewRefresh()) {
                Commands.addNoParameters(CommandType.SET_WC_IDLE, this);
                AccessHelper.eventsManager.setIsInViewRefresh(false);
            }
        }
        return true;
    }
    RefreshUI() {
        if (!this.getTask().isMainProg()) {
            Commands.addNoParameters(CommandType.REFRESH_TASK, this.getForm());
        }
    }
    async refreshTable() {
        if (this.isFetchingRecordsAhead)
            return;
        Logger.Instance.WriteGuiToLog("Start form.refreshTable()");
        this.updateDisplayLineByDV();
        let oldCurrRow = this.DisplayLine;
        let currRecIdx = this.GetDataview().getCurrRecIdx();
        if (!this._tableRefreshed && currRecIdx >= 0) {
            oldCurrRow = this.GetDataview().getCurrRecIdx();
            if (oldCurrRow + 1 > this._rowsInPage)
                this.GetDataview().setTopRecIdx(oldCurrRow - Math.floor(this._rowsInPage / 2) + 1);
            else
                this.GetDataview().setTopRecIdx(0);
        }
        else {
            if (oldCurrRow === Int32.MinValue)
                oldCurrRow = 0;
            if (this.GetDataview().getTopRecIdx() === Int32.MinValue)
                this.GetDataview().setTopRecIdx(currRecIdx);
        }
        if (this.GetDataview().getTopRecIdx() < 0)
            return;
        oldCurrRow = oldCurrRow - this.GetDataview().getTopRecIdx();
        let bkpRecord = this.GetDataview().backupCurrent();
        this._tableRefreshed = true;
        let oriTransferDataToGui = this._transferringDataToGUI;
        if (!oriTransferDataToGui)
            this._transferringDataToGUI = true;
        this.AllowedSubformRecompute = false;
        for (let i = 0; i < this._rowsInPage; i++) {
            let topIndex = this.GetDataview().getTopRecIdx();
            if (topIndex < 0) {
                this.GetDataview().setTopRecIdx(0);
                topIndex = 0;
            }
            let idx = topIndex + i;
            try {
                super.checkAndCreateRow(idx);
                if (i !== oldCurrRow) {
                    await this.setCurrRowByDisplayLine(idx, false, true);
                    await super.refreshControls(true);
                }
            }
            catch (ex) {
                if (ex.name == "RecordOutOfDataViewException") {
                    if (super.isRowCreated(idx))
                        super.markRowNOTCreated(idx);
                    break;
                }
                else
                    throw ex;
            }
        }
        this.AllowedSubformRecompute = true;
        if (!oriTransferDataToGui)
            this._transferringDataToGUI = false;
        await this.restoreBackup(oldCurrRow + this.GetDataview().getTopRecIdx(), bkpRecord);
        await super.refreshControls(true);
        this.RefreshUI();
        Logger.Instance.WriteGuiToLog("End form.refreshTable()");
    }
    updateDisplayLineByDV() {
        this.DisplayLine = this.recordIdx2DisplayLine(this.GetDataview().getCurrRecIdx());
    }
    async setCurrRowByDisplayLine(displayLine, doSuffix, ignoreCurrRec) {
        let recIdx = this.displayLine2RecordIdx(displayLine);
        await this.setCurrRowByRecIdx(recIdx, doSuffix, ignoreCurrRec, displayLine);
        if (super.HasTable())
            this.updateDisplayLineByDV();
        else
            this.DisplayLine = displayLine;
    }
    async setCurrRowByRecIdx(rowNum, doSuffix, ignoreCurrRec, displayLine) {
        if (this.GetDataview() === null || (!super.hasTable() && rowNum !== 0))
            throw new ApplicationException("in Form.setCurrRow() no table ");
        try {
            this._destTblRow = rowNum;
            await this.GetDataview().setCurrRecByIdx(rowNum, doSuffix, ignoreCurrRec, true, displayLine);
        }
        finally {
            this._destTblRow = rowNum - this.GetDataview().getTopRecIdx();
        }
    }
    getLastValidRow() {
        return Math.min(this._rowsInPage - 1, this.GetDataview().getSize() - 1 - this.GetDataview().getTopRecIdx());
    }
    recordIdx2DisplayLine(recIdx) {
        return recIdx;
    }
    displayLine2RecordIdx(displayLine) {
        return displayLine;
    }
    async restoreOldDisplayLine(displayLine) {
        try {
            this._inRestore = true;
            await this.setCurrRowByDisplayLine(displayLine, false, true);
        }
        catch (ex) {
            if (ex instanceof RecordOutOfDataViewException) {
            }
            else
                throw ex;
        }
        finally {
            this._inRestore = false;
        }
    }
    getTopIndexFromGUI() {
        let topDisplayLine = super.getTopIndexFromGUI();
        if (super.hasTable())
            this.GetDataview().setTopRecIdx(this.displayLine2RecordIdx(topDisplayLine));
        return topDisplayLine;
    }
    getCurrRecPosInForm() {
        let topDisplayLine = super.getTopIndexFromGUI();
        return this.DisplayLine - topDisplayLine + 1;
    }
    async restoreBackup(oldDisplayLine, bkpRecord) {
        if (bkpRecord.getId() >= 0 && this.GetDataview().getRecIdx(bkpRecord.getId()) >= 0 && oldDisplayLine >= 0 && oldDisplayLine < this.GetDataview().getSize() && this.GetDataview().getCurrRecIdx() >= 0 && bkpRecord.getMode() === DataModificationTypes.None) {
            this.GetDataview().restoreCurrent(bkpRecord);
            this.updateDisplayLineByDV();
        }
        else
            await this.restoreOldDisplayLine(oldDisplayLine);
    }
    ConstructMgControl(type, taskOrParentMgForm, parentControlOrParentControlIdx) {
        if (arguments.length === 0) {
            return this.ConstructMgControl_0();
        }
        if (arguments.length === 3 && (type === null || type.constructor === Number) && (taskOrParentMgForm === null || taskOrParentMgForm instanceof TaskBase) && (parentControlOrParentControlIdx === null || parentControlOrParentControlIdx.constructor === Number)) {
            return this.ConstructMgControl_1(type, taskOrParentMgForm, parentControlOrParentControlIdx);
        }
        return this.ConstructMgControl_2(type, taskOrParentMgForm, parentControlOrParentControlIdx);
    }
    ConstructMgControl_0() {
        return new MgControl();
    }
    ConstructMgControl_1(type, task, parentControl) {
        return new MgControl(type, task, parentControl);
    }
    ConstructMgControl_2(type, parentMgForm, parentControlIdx) {
        return new MgControl(type, parentMgForm, parentControlIdx);
    }
    alreadyMovedToFirstControl() {
        return this._task.getMGData().AlreadyMovedToFirstControl();
    }
    async firstTableRefresh() {
        if (this._tableMgControl !== null) {
            super.firstTableRefresh();
            this.setTableItemsCount(false);
            await this.refreshTable();
            super.SelectRow();
        }
    }
    InsertTableItems(idx, count) {
        if (this._tableMgControl != null) {
            this.Rows.InsertRange(new Array(count));
            this.InsertTableChildrenArrays(idx, count);
            this._lastRowSent = -1;
        }
        this.setTableItemsCount(false);
    }
    setTableItemsCount(removeAll) {
        let tableItemsCount;
        if (this.isTableWithAbsoluteScrollbar()) {
            if (this._task.getOriginalTaskMode() === Constants.TASK_MODE_CREATE
                && this._task.getMode() == Constants.TASK_MODE_CREATE) {
                tableItemsCount = this.GetDataview().getSize();
                if (this.GetDataview().RecordsBeforeCurrentView > 0) {
                    this.GetDataview().RecordsBeforeCurrentView = 0;
                    this.SetRecordsBeforeCurrentView(0);
                    this.RefreshUI();
                }
            }
            else {
                tableItemsCount = (this.GetDataview().TotalRecordsCount > 0) ? this.GetDataview().TotalRecordsCount : 1;
                this.SetRecordsBeforeCurrentView(this.GetDataview().RecordsBeforeCurrentView);
            }
        }
        else {
            tableItemsCount = this.GetDataview().getSize();
        }
        super.SetTableItemsCount(this.GetDataview().getSize(), tableItemsCount, removeAll);
    }
    InitTableControl(dvSize, tableItemsCount, removeAll) {
        if (arguments.length === 0) {
            this.InitTableControl_00();
            return;
        }
        this.InitTableControl_01(dvSize, tableItemsCount, removeAll);
    }
    InitTableControl_00() {
        let dvSize = 1;
        let tableItemsCount = 1;
        if (this.isTableWithAbsoluteScrollbar() && this.GetDataview().TotalRecordsCount > 0) {
            let recordsBeforeCurrentView = this.GetDataview().RecordsBeforeCurrentView;
            tableItemsCount = this.GetDataview().TotalRecordsCount;
            Commands.addWithNumber(CommandType.SET_RECORDS_BEFORE_CURRENT_VIEW, this._tableMgControl, recordsBeforeCurrentView);
        }
        this.InitTableControl(dvSize, tableItemsCount, false);
        this.SetTableTopIndex();
    }
    InitTableControl_01(dvSize, tableItemsCount, removeAll) {
        if (this._tableMgControl !== null) {
            if (removeAll) {
                Commands.addBoolWithLine(CommandType.SET_TABLE_INCLUDES_FIRST, this._tableMgControl, 0, true);
                Commands.addBoolWithLine(CommandType.SET_TABLE_INCLUDES_LAST, this._tableMgControl, 0, true);
            }
            else {
                Commands.addBoolWithLine(CommandType.SET_TABLE_INCLUDES_FIRST, this._tableMgControl, 0, this.GetDataview().IncludesFirst() || this.GetDataview().IsOneWayKey);
                Commands.addBoolWithLine(CommandType.SET_TABLE_INCLUDES_LAST, this._tableMgControl, 0, this.GetDataview().IncludesLast());
                Commands.addBoolWithLine(CommandType.SET_EMPTY_DATAVIEW, this._tableMgControl, 0, this.GetDataview().isEmptyDataview());
            }
            super.InitTableControl(dvSize, tableItemsCount, removeAll);
            this.checkAndCreateRowsEvent();
        }
    }
    SetTableTopIndex() {
        if (this._tableMgControl !== null) {
            let index = this.GetDataview().getTopRecIdx();
            if (index === Int32.MinValue)
                index = 0;
            if (!this.isTableWithPagination()) {
                if (this.GetDataview().IncludesLast()) {
                    if (index + this._rowsInPage > this.GetDataview().getSize()) {
                        if (this.GetDataview().IncludesFirst() || this.GetDataview().getSize() > this._rowsInPage + 1) {
                            index = Math.max(this.GetDataview().getSize() - this._rowsInPage, 0);
                            this.GetDataview().setTopRecIdx(index);
                            this._topIndexUpdated = true;
                        }
                    }
                }
            }
            if (index !== this._prevGuiTopIndex || this.IsMovingInView) {
                Commands.addWithNumber(CommandType.SET_TABLE_TOP_INDEX, this._tableMgControl, index);
                this._prevGuiTopIndex = this.getGuiRowidx(index);
            }
        }
    }
    createForm() {
        super.createForm();
        MgForm.CreatedForms.add(this);
    }
    UpdateStatusBar() {
    }
    initInnerObjects(foundTagName) {
        if (foundTagName === null)
            return false;
        let parser = Manager.GetCurrentRuntimeContext().Parser;
        if (foundTagName === XMLConstants.MG_TAG_RECOMPUTE) {
            Logger.Instance.WriteDevToLog("goes to recompute form");
            this._task.RecomputeFillData(parser);
            return true;
        }
        return super.initInnerObjects(foundTagName);
    }
    toString() {
        return "{" + this.constructor.name + ": task=" + this._task + ", Id=" + this._userStateId + "}";
    }
    GetHiddenControlListXML() {
        let xml = new StringBuilder();
        if (this.hiddenControlsIsnsList !== null && this.hiddenControlsIsnsList.length > 0) {
            xml.Append("<" + ConstInterface.MG_TAG_HIDDEN_CONTOLS + " " + ConstInterface.MG_ATTR_ISNS + "=\"");
            this.hiddenControlsIsnsList.forEach((change) => {
                xml.Append(change + ",");
            });
            xml.Remove(xml.Length - 1, 1);
            xml.Append("\"/>");
            this.hiddenControlsIsnsList = null;
        }
        return xml.ToString();
    }
    ShouldBehaveAsModal() {
        let shouldBeModal = false;
        if (!this._task.IsInteractive && !this._task.isMainProg())
            shouldBeModal = TaskBase.ShouldNonInteractiveBeModal();
        else {
            let parentTask = this._task.ParentTask;
            if (parentTask !== null && !parentTask.IsInteractive && !parentTask.isMainProg())
                shouldBeModal = parentTask.ShouldNonInteractiveChildBeModal();
        }
        return shouldBeModal;
    }
    isTableWithPagination() {
        return this.isTableWithAbsoluteScrollbar();
    }
    isTableWithAbsoluteScrollbar() {
        let tableWithAbsoluteScrollbar = false;
        if (this._tableMgControl !== null)
            tableWithAbsoluteScrollbar = this._tableMgControl.IsTableWithAbsoluteScrollbar();
        return tableWithAbsoluteScrollbar;
    }
    SetRecordsBeforeCurrentView(value) {
        if (this._tableMgControl !== null) {
            Commands.addWithNumber(CommandType.SET_RECORDS_BEFORE_CURRENT_VIEW, this._tableMgControl, value);
        }
    }
    buildFormName() {
        let name = this.UniqueName;
        let task = this.getTask();
        name = (task.getProp(PropInterface.PROP_TYPE_NAME)).GetComputedValue() + '_' + name;
        while (task.IsSubtask) {
            task = task.StudioParentTask;
            name = (task.getProp(PropInterface.PROP_TYPE_NAME)).GetComputedValue() + '_' + name;
        }
        name = name.replace(/ /g, "_");
        name = name.replace(/-/g, "_");
        return name;
    }
    async fetchPrevChunkForCurrentPage() {
        if (this.isTableWithAbsoluteScrollbar() && this.GetDataview().TotalRecordsCount > 0) {
            let recPosOnPage = this.GetDataview().RecordsBeforeCurrentView % this._rowsInPage;
            if (recPosOnPage > 0 && this.DisplayLine === 0) {
                await this.setRowData(-recPosOnPage, false);
            }
        }
    }
    getGuiRowidx(dvRowIdx) {
        return dvRowIdx + this.GetDataview().RecordsBeforeCurrentView;
    }
    getDvRowIdx(guiRowIdx) {
        return guiRowIdx - this.GetDataview().RecordsBeforeCurrentView;
    }
}
MgForm.TIME_LIMIT = 50;
MgForm._createdForms = new CreatedFormVector();
//# sourceMappingURL=data:application/json;base64,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