import { HttpErrorResponse } from "@angular/common/http";
import { NString, Thread, WebException } from "@magic-xpa/mscorelib";
import { HttpClientBase, RequestMethod } from "./HttpClientBase";
import { Logger, Logger_LogLevels, Logger_MessageDirection, Misc } from "@magic-xpa/utils";
import { HttpClientEvents } from "./HttpClientEvents";
export class HttpClientAsync extends HttpClientBase {
    constructor(httpClient) {
        super();
        this.httpClient = httpClient;
    }
    prepareRequest() {
    }
    addHeaders(urlString, useCache, httpHeaders, clientID) {
        if (!urlString.startsWith("/"))
            httpHeaders = httpHeaders.append("MgxpaRIAglobalUniqueSessionID", clientID);
        if (!useCache)
            httpHeaders = httpHeaders.append("Cache-Control", 'no-cache');
        return httpHeaders;
    }
    async sendRequestToServer(httpMethod, urlString, httpHeaders, requestContent, contentFromServer) {
        let httpResponse = await this.httpClient.request(RequestMethod[httpMethod], urlString, { headers: httpHeaders, responseType: "text", observe: "response", body: requestContent }).toPromise();
        contentFromServer.value = httpResponse.body;
        return httpResponse;
    }
    logResponse(httpResponse, clientID, urlString, contentFromServer, timeBeforeRequest) {
        let statusCode = httpResponse['status'];
        let responseTime = Misc.getSystemMilliseconds() - timeBeforeRequest;
        if (Logger.Instance.LogLevel === Logger_LogLevels.Basic)
            Logger.Instance.WriteBasicToLog(Logger_MessageDirection.MessageEntering, HttpClientEvents.GetRuntimeCtxID(), HttpClientEvents.GetSessionCounter(), clientID, HttpClientEvents.ShouldDisplayGenericError() ? "-" : new URL(urlString).host, responseTime, statusCode.toString(), JSON.stringify(httpResponse.headers), contentFromServer.value.length);
    }
    handleHttpErrorException(ex, urlString) {
        if (ex instanceof HttpErrorResponse) {
            let statusCode = ex['status'];
            if (statusCode === 404 || statusCode === 403)
                throw new WebException(new Error("bad URL: " + urlString + " - StatusCode = " + statusCode));
        }
    }
    async shouldRetry(httpCommunicationTimeoutMS, urlString, communicationsFailureHandler, ex, startTimeRef) {
        let currentDelayMS = Math.floor(httpCommunicationTimeoutMS / 10);
        let httpElapsedTimeMS = Misc.getSystemMilliseconds() - startTimeRef.value + currentDelayMS;
        if (httpElapsedTimeMS <= httpCommunicationTimeoutMS) {
            await Thread.Sleep(currentDelayMS);
            Logger.Instance.WriteWarningToLogWithMsg(NString.Format("Retrying {0} : elapsed time {1:N0}ms out of {2:N0}ms", urlString, httpElapsedTimeMS, httpCommunicationTimeoutMS));
            return true;
        }
        Logger.Instance.WriteWarningToLogWithMsg(NString.Format("{0} : http timeout {1:N0}ms expired", urlString, httpCommunicationTimeoutMS));
        if (communicationsFailureHandler != null) {
            communicationsFailureHandler.CommunicationFailed(urlString, ex);
            if (communicationsFailureHandler.ShouldRetryLastRequest) {
                Logger.Instance.WriteServerToLog(NString.Format("Retrying {0}, confirmed by user ...", urlString));
                startTimeRef.value = Misc.getSystemMilliseconds();
                return true;
            }
        }
        return false;
    }
    logAndThrowException(ex) {
        Logger.Instance.WriteWarningToLogWithMsg("Re-throwing ...");
        Logger.Instance.WriteWarningToLog(ex);
        throw new WebException(ex);
    }
    logExecutionAttempts(executionAttempts) {
        if (executionAttempts > 1)
            Logger.Instance.WriteServerToLog(NString.Format("Succeeded after {0} attempts ...", executionAttempts));
    }
}
//# sourceMappingURL=data:application/json;base64,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