import { NString, RefParam, StringBuilder, isNullOrUndefined } from "@magic-xpa/mscorelib";
import { Logger, Logger_LogLevels, Logger_MessageDirection, Misc } from "@magic-xpa/utils";
import { HttpClientEvents } from "./HttpClientEvents";
import { HttpHeaders } from "@angular/common/http";
export var RequestMethod;
(function (RequestMethod) {
    RequestMethod[RequestMethod["Get"] = 0] = "Get";
    RequestMethod[RequestMethod["Post"] = 1] = "Post";
    RequestMethod[RequestMethod["Put"] = 2] = "Put";
    RequestMethod[RequestMethod["Delete"] = 3] = "Delete";
    RequestMethod[RequestMethod["Options"] = 4] = "Options";
    RequestMethod[RequestMethod["Head"] = 5] = "Head";
    RequestMethod[RequestMethod["Patch"] = 6] = "Patch";
})(RequestMethod || (RequestMethod = {}));
export class HttpClientBase {
    constructor() {
        this._HTTPMaxURLLength = 2048;
        this.CommunicationsFailureHandler = null;
        this.CommunicationsFailureHandler = null;
    }
    DecideOnRequestMethod(requestContent, requestURL) {
        let method = RequestMethod.Get;
        if (requestContent === null) {
            method = RequestMethod.Get;
        }
        else {
            if (requestURL.value.length + 1 + requestContent.length <= this._HTTPMaxURLLength) {
                requestURL.value = requestURL.value + "?" + requestContent;
                method = RequestMethod.Get;
            }
            else {
                method = RequestMethod.Post;
            }
        }
        return method;
    }
    async GetContent(requestURL, requestContent, useCache) {
        let contentFromServer = new RefParam(null);
        let requestUrlRef = new RefParam(requestURL);
        let httpMethod = this.DecideOnRequestMethod(requestContent, requestUrlRef);
        requestURL = requestUrlRef.value;
        try {
            let response = await this.ExecuteHttpRequest(requestURL, requestContent, useCache, httpMethod, contentFromServer);
            if (response != null) {
                Logger.Instance.WriteServerToLog("Incoming Headers : " + HttpClientBase.HeadersToString(response.headers, true));
                let nextSessionCounterString = response.headers.get("MgxpaNextSessionCounter".toLowerCase());
                if (!isNullOrUndefined(nextSessionCounterString)) {
                    HttpClientEvents.CheckAndSetSessionCounter(+nextSessionCounterString);
                }
            }
        }
        catch (ex) {
            Logger.Instance.WriteWarningToLog(ex);
            throw ex;
        }
        return contentFromServer.value;
    }
    async ExecuteHttpRequest(urlString, requestContent, useCache, httpMethod, contentFromServer) {
        let httpResponse = null;
        this.prepareRequest();
        let httpCommunicationTimeoutMS = HttpClientEvents.GetHttpCommunicationTimeout();
        let clientID = HttpClientEvents.GetGlobalUniqueSessionID();
        let executionAttempts = 0;
        let startTime = Misc.getSystemMilliseconds();
        while (true) {
            executionAttempts++;
            try {
                let httpHeaders = new HttpHeaders();
                httpHeaders = this.addHeaders(urlString, useCache, httpHeaders, clientID);
                Logger.Instance.WriteServerToLog(NString.Format("Request Timeout set to {0} ms", httpCommunicationTimeoutMS));
                if (Logger.Instance.LogLevel === Logger_LogLevels.Basic) {
                    let contentLength = 0;
                    if (httpMethod === RequestMethod.Get) {
                        let parts = urlString.split('?');
                        if (parts.length === 2)
                            contentLength = parts[1].length;
                    }
                    else
                        contentLength = requestContent.length;
                    Logger.Instance.WriteBasicToLog(Logger_MessageDirection.MessageLeaving, HttpClientEvents.GetRuntimeCtxID(), HttpClientEvents.GetSessionCounter(), clientID, HttpClientEvents.ShouldDisplayGenericError() ? "-" : new URL(urlString).host, 0, '-', JSON.stringify(httpHeaders), contentLength);
                }
                let timeBeforeRequest = Misc.getSystemMilliseconds();
                Logger.Instance.WriteServerToLog(NString.Format("Accessing (method: '{0}'): '{1}'", httpMethod, urlString));
                Logger.Instance.WriteServerToLog("Outgoing Headers : " + HttpClientBase.HeadersToString(httpHeaders, false));
                if (httpMethod === RequestMethod.Post) {
                }
                let requestTime = Misc.getSystemMilliseconds();
                this.LogRequestInfo(!isNullOrUndefined(requestContent) ? requestContent.length : 0, requestTime, false);
                httpResponse = await this.sendRequestToServer(httpMethod, urlString, httpHeaders, requestContent, contentFromServer);
                this.logResponse(httpResponse, clientID, urlString, contentFromServer, timeBeforeRequest);
                this.LogRequestInfo(!isNullOrUndefined(requestContent) ? requestContent.length : 0, Misc.getSystemMilliseconds() - requestTime, true);
                break;
            }
            catch (ex) {
                if (Logger.Instance.LogLevel === Logger_LogLevels.Basic)
                    Logger.Instance.WriteBasicErrorToLog();
                else
                    Logger.Instance.WriteWarningToLog(ex);
                this.handleHttpErrorException(ex, urlString);
                let startTimeRef = new RefParam(startTime);
                if (this.shouldRetry(httpCommunicationTimeoutMS, urlString, this.CommunicationsFailureHandler, ex, startTimeRef)) {
                    startTime = startTimeRef.value;
                    continue;
                }
                this.logAndThrowException(ex);
            }
        }
        this.logExecutionAttempts(executionAttempts);
        return httpResponse;
    }
    LogRequestInfo(contentLength, roundTrip, isLoggingResponse) {
        if (Logger.Instance.LogLevel == Logger_LogLevels.RequestInfo) {
            HttpClientEvents.ComputeAndLogRequestInfo(contentLength, roundTrip, isLoggingResponse);
        }
    }
    static HeadersToString(headers, bFilter) {
        let headersStr = new StringBuilder();
        let headerKeys = headers.keys();
        for (let key in headerKeys) {
            if (!bFilter || headerKeys[key].startsWith("Mg"))
                headersStr.Append(NString.Format("{0}:{1} ", headerKeys[key], headers.get(headerKeys[key])));
        }
        return headersStr;
    }
    async sendRequestToServer(httpMethod, urlString, httpHeaders, requestContent, contentFromServer) {
        return Promise.resolve();
    }
    async shouldRetry(httpCommunicationTimeoutMS, urlString, communicationsFailureHandler, ex, startTimeRef) {
        return Promise.resolve(false);
    }
}
//# sourceMappingURL=data:application/json;base64,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