import { Base64, JSON_Utils, Logger, Logger_LogLevels, Misc, MsgInterface, OSEnvironment, RequestInfo, StrUtil, XMLConstants } from "@magic-xpa/utils";
import { Debug, Encoding, Exception, List, NString, RefParam, StringBuilder, isNullOrUndefined, isUndefined } from "@magic-xpa/mscorelib";
import { Commands, RuntimeContextBase, Styles, UsernamePasswordCredentials } from "@magic-xpa/gui";
import { ExecOperCommand } from '../commands/ClientToServer/ExecOperCommand';
import { CommandsProcessorBase, CommandsProcessorBase_SendingInstruction, CommandsProcessorBase_SessionStage } from "../CommandsProcessorBase";
import { HttpManager } from "../http/HttpManager";
import { ConstInterface } from "../ConstInterface";
import { Scrambler } from "../util/Scrambler";
import { ServerError } from "./ServerError";
import { FlowMonitorQueue } from "../util/FlowMonitorQueue";
import { OpeningTaskDetails } from "../tasks/TaskBase";
import { HttpUtility } from "../http/client/HttpUtility";
import { HttpClientEvents } from "../http/client/HttpClientEvents";
import { ServerConfig } from "../ServerConfig";
import { AccessHelper } from "../AccessHelper";
import { Environment, Event, EventCommand, EventsManager, GUIManager } from "../../";
import { MGDataCollection } from "../tasks/MGDataCollection";
import { timer } from "rxjs";
var RequestStatus;
(function (RequestStatus) {
    RequestStatus[RequestStatus["Handled"] = 0] = "Handled";
    RequestStatus[RequestStatus["Retry"] = 1] = "Retry";
    RequestStatus[RequestStatus["Abort"] = 2] = "Abort";
})(RequestStatus || (RequestStatus = {}));
export class RemoteCommandsProcessor extends CommandsProcessorBase {
    static GetInstance() {
        if (RemoteCommandsProcessor._instance === null) {
            RemoteCommandsProcessor._instance = new RemoteCommandsProcessor();
        }
        return RemoteCommandsProcessor._instance;
    }
    constructor() {
        super();
        this._lastRequestTime = 0;
        this.ServerUrl = null;
        this.DelayCommandExecution = false;
        this._requestInfo = new RequestInfo();
        this.ServerUrl = ServerConfig.Instance.getServerURL();
        HttpManager.GetInstance();
        this.RegisterDelegates();
    }
    CheckAndSetSessionCounter(newSessionCounter) {
        if (newSessionCounter === ConstInterface.SESSION_COUNTER_CLOSE_CTX_INDICATION) {
            Debug.Assert(this.GetSessionCounter() === ConstInterface.SESSION_COUNTER_CLOSE_CTX_INDICATION);
        }
        else {
            Debug.Assert(newSessionCounter === this.GetSessionCounter() + 1);
            this.SetSessionCounter(newSessionCounter);
            Logger.Instance.WriteServerToLog(NString.Format("Session Counter --> {0}", this._sessionCounter));
        }
    }
    ClearSessionCounter() {
        this._sessionCounter = 0;
        Logger.Instance.WriteServerToLog(NString.Format("Session Counter --> {0}", this._sessionCounter));
    }
    SetSessionCounter(newSessionCounter) {
        this._sessionCounter = newSessionCounter;
    }
    get GetLastRequestTime() {
        return this._lastRequestTime;
    }
    async ReStartSession() {
        RemoteCommandsProcessor.IsSessionReInitializing = true;
        await this.StoreSessionReInitializingDataOnLocalStorage();
        let navigateRootRoute = false;
        if (!RuntimeContextBase.Instance.RemovedContextFromServer)
            navigateRootRoute = Environment.Instance.getSpecialRouteToRootProgOnContextRecreation() ? true : false;
        Commands.addRefreshPage(navigateRootRoute, RemoteCommandsProcessor.InitialUrl);
        throw new Exception('');
    }
    async StoreSessionReInitializingDataOnLocalStorage() {
        let mainPrgViewStringForServer = await this.BuildXMLForMainProgramDataView();
        let globalParamsString = AccessHelper.globalParams.mirrorAllToXML();
        let changedEnvVarList = AccessHelper.envParamsTable.mirrorAllToXML();
        let dataStorage = window.localStorage;
        dataStorage.setItem(ConstInterface.IS_SESSION_REINITIALIZING, "true");
        dataStorage.setItem(ConstInterface.MAIN_PROG_VIEW, mainPrgViewStringForServer.toString());
        dataStorage.setItem(ConstInterface.GLOBAL_PARAM_LIST, globalParamsString);
        dataStorage.setItem(ConstInterface.ENV_VAR_LIST, changedEnvVarList);
        dataStorage.setItem(ConstInterface.LAST_EXCEPTION, RemoteCommandsProcessor.lastExceptionMessage);
        if (RuntimeContextBase.Instance.RemovedContextFromServer)
            dataStorage.setItem(ConstInterface.CTX_REMOVED_FROM_SRVR, "1");
        EventsManager.Instance.storeLastSavedRouteEventOnLocalStorage();
    }
    RestoreSessionReInitializingDataFromLocalStorage(key) {
        let dataStorage = window.localStorage;
        let val = dataStorage.getItem(key);
        return val;
    }
    async StartSession() {
        let authenticationCancelled = false;
        let handshakeResponse = null;
        try {
            let lastTypedUserId = null;
            let handshakeInitialTokens = new StringBuilder();
            handshakeInitialTokens.Append(ConstInterface.UTF8TRANS +
                ConstInterface.REQ_APP_NAME + "=" +
                HttpUtility.UrlEncode(ServerConfig.Instance.getAppName(), Encoding.UTF8));
            handshakeInitialTokens.Append("&" + ConstInterface.REQ_ARGS + "=" + ConstInterface.REQ_ARG_ALPHA +
                "<Richclient><Requires EncryptionKey=\"False\"/><RIAProtocolVersion=\"" + RemoteCommandsProcessor.WEB_COMMUNICATION_PROTOCOL_VERSION + "\"/></Richclient>");
            Logger.Instance.WriteDevToLog(NString.Format("Handshake request #1 (not scrambled) : {0}", this.ServerUrl + ConstInterface.REQ_ARG_START +
                ConstInterface.RC_INDICATION_INITIAL + handshakeInitialTokens));
            let handshakeInitialUrl = this.ServerUrl + ConstInterface.REQ_ARG_START +
                ConstInterface.RC_INDICATION_INITIAL +
                ConstInterface.RC_TOKEN_DATA +
                HttpUtility.UrlEncode(Scrambler.Scramble(handshakeInitialTokens.ToString()), Encoding.UTF8);
            Logger.Instance.WriteServerMessagesToLog("");
            Logger.Instance.WriteServerMessagesToLog(NString.Format("Handshake request #1: {0}", handshakeInitialUrl));
            let ctxGroup = ServerConfig.Instance.getCtxGroup();
            while (!authenticationCancelled) {
                let requestStatus = new RefParam(RequestStatus.Handled);
                let responseStr = await this.DispatchRequest(handshakeInitialUrl, null, CommandsProcessorBase_SessionStage.HANDSHAKE, requestStatus);
                if (NString.IsNullOrEmpty(responseStr)) {
                    throw new ServerError("Client failed to initialize a session." + OSEnvironment.EolSeq +
                        "Empty response was received from the web server." + OSEnvironment.EolSeq + OSEnvironment.EolSeq + this.ServerUrl);
                }
                Logger.Instance.WriteServerMessagesToLog(NString.Format("Handshake response #1: {0}", responseStr));
                Logger.Instance.WriteServerMessagesToLog("");
                handshakeResponse = new HandshakeResponse(responseStr);
                RuntimeContextBase.Instance.ContextID = handshakeResponse.ContextId;
                this.SessionId = handshakeResponse.GetSessionId;
                HttpManager.GetInstance().HttpCommunicationTimeoutMS = handshakeResponse.HttpTimeout * 1000;
                if (NString.IsNullOrEmpty(ServerConfig.Instance.getCtxGroup())) {
                    ctxGroup = handshakeResponse.ContextId;
                }
                let credentials = null;
                if (ServerConfig.Instance.getSkipAuthenticationDialog()) {
                    if (ServerConfig.Instance.getUsername() !== null) {
                        credentials = new UsernamePasswordCredentials(ServerConfig.Instance.getUsername(), ServerConfig.Instance.getPassword());
                        ServerConfig.Instance.setUsername("");
                        ServerConfig.Instance.setPassword("");
                    }
                }
                else if (handshakeResponse.InputPassword && handshakeResponse.SystemLogin !== HandshakeResponse.SYSTEM_LOGIN_AD) {
                    if (lastTypedUserId !== null) {
                        let title = AccessHelper.languageData.getConstMessage(MsgInterface.BRKTAB_STR_ERROR);
                        let error = AccessHelper.languageData.getConstMessage(MsgInterface.USRINP_STR_BADPASSW);
                        await GUIManager.Instance.MessageBox(title, error, Styles.MSGBOX_ICON_ERROR | Styles.MSGBOX_BUTTON_OK);
                    }
                }
                if (authenticationCancelled) {
                    this.SetSessionCounter(ConstInterface.SESSION_COUNTER_CLOSE_CTX_INDICATION);
                }
                let handshakeAuthUrl = this.PrepareAuthenticationUrl(handshakeResponse.ContextId, ctxGroup, this.GetSessionCounter());
                try {
                    if (credentials !== null) {
                        let credentialsStr = credentials.Username + ":";
                        if (!NString.IsNullOrEmpty(credentials.Password)) {
                            credentialsStr = credentialsStr + credentials.Password + ":";
                        }
                        credentialsStr += handshakeResponse.ContextId;
                        credentialsStr = Scrambler.Scramble(credentialsStr);
                        credentialsStr = Base64.encode(credentialsStr, true, Encoding.UTF8);
                        handshakeAuthUrl += ("&USERNAME=" + HttpUtility.UrlEncode(credentialsStr, Encoding.UTF8));
                        Logger.Instance.WriteServerMessagesToLog(NString.Format("Handshake request #2: {0}", handshakeAuthUrl));
                        let statusRequest = new RefParam(RequestStatus.Handled);
                        responseStr = await this.DispatchRequest(handshakeAuthUrl, null, CommandsProcessorBase_SessionStage.HANDSHAKE, statusRequest);
                        if (responseStr.indexOf(handshakeResponse.ContextId) === -1) {
                            throw new ServerError(AccessHelper.languageData.getConstMessage(MsgInterface.USRINP_STR_BADPASSW));
                        }
                        ServerConfig.Instance.setUsername(credentials.Username);
                        ServerConfig.Instance.setPassword(credentials.Password);
                    }
                    else {
                        let requestStatus = new RefParam(RequestStatus.Handled);
                        Logger.Instance.WriteServerMessagesToLog(NString.Format("Handshake request #2: {0}", handshakeAuthUrl));
                        responseStr = await this.DispatchRequest(handshakeAuthUrl, null, CommandsProcessorBase_SessionStage.HANDSHAKE, requestStatus);
                    }
                    Logger.Instance.WriteServerMessagesToLog(NString.Format("Handshake response #2: {0}", responseStr));
                    Logger.Instance.WriteServerMessagesToLog("");
                    break;
                }
                catch (ex) {
                    if (ex instanceof ServerError) {
                        let serverError = ex;
                        if (ServerConfig.Instance.getSkipAuthenticationDialog()) {
                            switch (serverError.GetCode()) {
                                case ServerError.ERR_ACCESS_DENIED:
                                case ServerError.ERR_AUTHENTICATION:
                                    ServerConfig.Instance.setSkipAuthenticationDialog(false);
                                    throw serverError;
                            }
                        }
                        if (!handshakeResponse.InputPassword)
                            throw serverError;
                        if (serverError.GetCode() !== ServerError.ERR_AUTHENTICATION &&
                            (!authenticationCancelled || serverError.GetCode() !== ServerError.INF_NO_RESULT)) {
                            throw serverError;
                        }
                    }
                    else
                        throw ex;
                }
            }
            ServerConfig.Instance.setSkipAuthenticationDialog(true);
            ServerConfig.Instance.setCtxGroup(ctxGroup);
            RemoteCommandsProcessor.ShouldScrambleAndUnscrambleMessages = handshakeResponse.ScrambleMessages;
            if (Logger.Instance.LogLevel !== Logger_LogLevels.Basic) {
                let maxInternalLogLevel = handshakeResponse.MaxInternalLogLevel;
                if (maxInternalLogLevel !== null) {
                    let maxLogLevel = this.parseLogLevel(maxInternalLogLevel);
                    if (maxLogLevel < Logger.Instance.LogLevel) {
                        ServerConfig.Instance.setInternalLogLevel(maxInternalLogLevel);
                        Logger.Instance.WriteToLog(NString.Format("Internal log level was restricted to '{0}' by the Magic xpa server.", maxInternalLogLevel), false);
                        Logger.Instance.LogLevel = maxLogLevel;
                    }
                }
            }
            if (!authenticationCancelled) {
                await this.ExecuteInitialRequest();
            }
        }
        catch (ex) {
            if (ex instanceof ServerError)
                throw ex;
            if (isNullOrUndefined(ex.InnerException))
                throw new ServerError(ex.message, new Exception(ex));
            else
                throw new ServerError(ex.message, ex.InnerException);
        }
        return !authenticationCancelled;
    }
    PrepareAuthenticationUrl(contextId, ctxGroup, sessionCount) {
        let handshakeAuthUrl = this.ServerUrl + ConstInterface.REQ_ARG_START +
            ConstInterface.RC_INDICATION + ConstInterface.UTF8TRANS +
            ConstInterface.RC_TOKEN_CTX_ID + contextId;
        if (!isNullOrUndefined(this.SessionId))
            handshakeAuthUrl += (ConstInterface.REQ_ARG_SEPARATOR + ConstInterface.RC_TOKEN_SESSION_ID + this.SessionId);
        handshakeAuthUrl += (ConstInterface.REQ_ARG_SEPARATOR +
            ConstInterface.RC_TOKEN_SESSION_COUNT + sessionCount +
            ConstInterface.REQ_ARG_SEPARATOR +
            ConstInterface.RC_TOKEN_CTX_GROUP + ctxGroup +
            ConstInterface.REQ_ARG_SEPARATOR +
            ConstInterface.RC_AUTHENTICATION_REQUEST);
        return handshakeAuthUrl;
    }
    async ExecuteInitialRequest() {
        await this.Execute_1(CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS, CommandsProcessorBase_SessionStage.INITIAL, null);
    }
    GetTaskTree(task) {
        let taskTree = new List();
        while (!isNullOrUndefined(task)) {
            taskTree.push(task.getTaskInfo());
            task = task.getParent();
        }
        return taskTree;
    }
    async UpdateRequestInfo() {
        let currMGData = AccessHelper.mgDataTable.getCurrMGData();
        let eventsManager = AccessHelper.eventsManager;
        let operExecution = false;
        let cmdsToServer = currMGData.CmdsToServer;
        for (let i = 0; i < cmdsToServer.getSize(); i++) {
            if (cmdsToServer.getCmd(i) instanceof ExecOperCommand) {
                let operCmd = currMGData.CmdsToServer.getCmd(i);
                this._requestInfo.serverCallAt = operCmd.getCommandInfo();
                this._requestInfo.runtimeTaskTree = this.GetTaskTree(operCmd.getTask());
                operExecution = true;
                break;
            }
        }
        this._requestInfo.extraInfo.contextID = RuntimeContextBase.Instance.ContextID;
        this._requestInfo.extraInfo.sessionCounter = this.GetSessionCounter();
        if (!operExecution) {
            if (eventsManager.getLastRtEvent() != null && eventsManager.getLastRtEvent().InternalEvent != 0) {
                let rtEvt = eventsManager.getLastRtEvent();
                this._requestInfo.runtimeTaskTree = this.GetTaskTree(rtEvt.getTask());
                let eventDescription = rtEvt.isNewInternalEvent()
                    ? NString.TrimEnd(rtEvt.getNewInternalEventDescription())
                    : Event.getInternalEvtDescription(rtEvt.InternalEvent);
                let cmd = cmdsToServer.getCmd(cmdsToServer.getSize() - 1);
                if (cmd) {
                    let commandInfo = NString.IsNullOrEmpty(cmd.getCommandInfo())
                        ? NString.Empty
                        : cmd.getCommandInfo();
                    if (cmd instanceof EventCommand && rtEvt.InternalEvent == cmd.MagicEvent)
                        eventDescription = NString.Empty;
                    this._requestInfo.serverCallAt = 'Event ' + commandInfo +
                        (!NString.IsNullOrEmpty(eventDescription)
                            ? NString.Format(' while executing [{0}]', eventDescription)
                            : NString.Empty);
                }
            }
        }
    }
    async BuildXMLForMainProgramDataView() {
        let message = new StringBuilder();
        let firstMgData = MGDataCollection.Instance.getMGData(0);
        let mainPrg = firstMgData.getMainProg(0);
        while (mainPrg != null) {
            await mainPrg.buildXML(message);
            mainPrg = firstMgData.getNextMainProg(mainPrg.getCtlIdx());
        }
        if (RemoteCommandsProcessor.ShouldScrambleAndUnscrambleMessages) {
            let scrambledChanges = Scrambler.Scramble(message.ToString());
            message = new StringBuilder(scrambledChanges);
        }
        message.Insert(0, XMLConstants.MG_TAG_OPEN);
        message.Append(XMLConstants.MG_TAG_XML_END_TAGGED);
        return message.ToString();
    }
    async Execute_1(sendingInstruction, sessionStage, res) {
        let reqBuf;
        let isInitialCall = sessionStage === CommandsProcessorBase_SessionStage.INITIAL;
        let globalParamsString = null;
        let envVarsString = null;
        if (this.DelayCommandExecution)
            return;
        if (Logger.Instance.LogLevel == Logger_LogLevels.RequestInfo && !isInitialCall)
            await this.UpdateRequestInfo();
        if (isInitialCall) {
            let val = this.RestoreSessionReInitializingDataFromLocalStorage(ConstInterface.IS_SESSION_REINITIALIZING);
            if (val === 'true') {
                RemoteCommandsProcessor.IsSessionReInitializing = (val === "true") ? true : false;
                RemoteCommandsProcessor.lastExceptionMessage = this.RestoreSessionReInitializingDataFromLocalStorage(ConstInterface.LAST_EXCEPTION);
                let ctxRemoved = this.RestoreSessionReInitializingDataFromLocalStorage(ConstInterface.CTX_REMOVED_FROM_SRVR);
                if (ctxRemoved !== '1') {
                    Logger.Instance.WriteErrorToLog(RemoteCommandsProcessor.lastExceptionMessage);
                    Logger.Instance.WriteToLog("Session is reinitializing...", true);
                }
            }
        }
        if (sendingInstruction === CommandsProcessorBase_SendingInstruction.NO_TASKS_OR_COMMANDS)
            reqBuf = null;
        else {
            reqBuf = await this.PrepareRequest(sendingInstruction === CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS);
            if (RemoteCommandsProcessor.IsSessionReInitializing || !isInitialCall) {
                let changes = new StringBuilder();
                let buffer = new StringBuilder();
                if (!RemoteCommandsProcessor.IsSessionReInitializing)
                    buffer.Append(reqBuf);
                if (RemoteCommandsProcessor.IsSessionReInitializing) {
                    globalParamsString = this.RestoreSessionReInitializingDataFromLocalStorage(ConstInterface.GLOBAL_PARAM_LIST);
                    envVarsString = this.RestoreSessionReInitializingDataFromLocalStorage(ConstInterface.ENV_VAR_LIST);
                }
                else {
                    globalParamsString = AccessHelper.globalParams.mirrorToXML();
                    envVarsString = AccessHelper.envParamsTable.mirrorToXML();
                }
                changes.Append(globalParamsString);
                changes.Append(envVarsString);
                if (changes.Length > 0) {
                    changes.Insert(0, "<" + ConstInterface.MG_TAG_ENV_CHANGES + ">");
                    changes.Append("</" + ConstInterface.MG_TAG_ENV_CHANGES + ">");
                    if (RemoteCommandsProcessor.ShouldScrambleAndUnscrambleMessages) {
                        let scrambledChanges = Scrambler.Scramble(changes.ToString());
                        changes = new StringBuilder(scrambledChanges);
                    }
                    changes.Insert(0, XMLConstants.MG_TAG_OPEN);
                    changes.Append(XMLConstants.MG_TAG_XML_END_TAGGED);
                    buffer.Append(changes.ToString());
                }
                reqBuf = buffer.ToString();
            }
        }
        let requestStatus = new RefParam(RequestStatus.Handled);
        let respBuf = null;
        if (RemoteCommandsProcessor.IsSessionReInitializing) {
            reqBuf = reqBuf + this.RestoreSessionReInitializingDataFromLocalStorage(ConstInterface.MAIN_PROG_VIEW);
        }
        if (RuntimeContextBase.Instance.RemovedContextFromServer)
            requestStatus.value = RequestStatus.Retry;
        else {
            if (RuntimeContextBase.Instance.RemovingContextFromServer)
                RuntimeContextBase.Instance.RemovedContextFromServer = true;
            respBuf = await this.DispatchRequest(this.ServerUrl, reqBuf, sessionStage, requestStatus);
        }
        if (requestStatus.value == RequestStatus.Retry) {
            await this.ReStartSession();
        }
        if (respBuf == null)
            return;
        if (isInitialCall) {
            RemoteCommandsProcessor.InitialUrl = window.location.href;
            if (RemoteCommandsProcessor.IsSessionReInitializing) {
                let dataStorage = window.localStorage;
                dataStorage.removeItem(ConstInterface.IS_SESSION_REINITIALIZING);
                dataStorage.removeItem(ConstInterface.MAIN_PROG_VIEW);
                dataStorage.removeItem(ConstInterface.GLOBAL_PARAM_LIST);
                dataStorage.removeItem(ConstInterface.ENV_VAR_LIST);
                dataStorage.removeItem(ConstInterface.LAST_EXCEPTION);
            }
        }
        FlowMonitorQueue.Instance.enable(false);
        await this.ProcessResponse(respBuf, AccessHelper.mgDataTable.currMgdID, null, res);
        if (RemoteCommandsProcessor.IsSessionReInitializing) {
            AccessHelper.globalParams.RestoreParams(globalParamsString);
            AccessHelper.envParamsTable.RestoreParams(envVarsString);
            let dataStorage = window.localStorage;
            let ctxRemoved = dataStorage.getItem(ConstInterface.CTX_REMOVED_FROM_SRVR);
            if (ctxRemoved === "1") {
                EventsManager.Instance.restoreLastSavedRouteEventFromLocalStorage();
                dataStorage.removeItem(ConstInterface.CTX_REMOVED_FROM_SRVR);
                dataStorage.removeItem(ConstInterface.LAST_ROUTE_EVENT);
                dataStorage.removeItem(ConstInterface.LAST_ROUTE_EVENT_SRC_TSK);
                dataStorage.removeItem(ConstInterface.LAST_ROUTE_EVENT_ARG_LIST);
            }
            RemoteCommandsProcessor.IsSessionReInitializing = false;
        }
        if (sendingInstruction === CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS) {
            AccessHelper.mgDataTable.startTasksIteration();
            let task;
            while ((task = AccessHelper.mgDataTable.getNextTask()) !== null) {
                let dataViewWasRetrieved = task.DataViewWasRetrieved;
                if (dataViewWasRetrieved) {
                    await task.RefreshDisplay();
                }
            }
        }
        if (isInitialCall)
            ServerConfig.Instance.setGlobalParams(null);
    }
    async DispatchRequest(url, reqBuf, sessionStage, requestStatus) {
        let response = null;
        let encodedBody = null;
        requestStatus.value = RequestStatus.Handled;
        if (url === null) {
            Logger.Instance.WriteExceptionToLogWithMsg("in sendMsgToSrvr() unknown server");
            return response;
        }
        if (sessionStage !== CommandsProcessorBase_SessionStage.HANDSHAKE) {
            let urlSuffix = this.BuildUrlSuffix(reqBuf !== null, sessionStage === CommandsProcessorBase_SessionStage.INITIAL);
            let reqBufEncoded = HttpUtility.UrlEncode(reqBuf, Encoding.UTF8);
            encodedBody = urlSuffix + reqBufEncoded;
        }
        try {
            if (Logger.Instance.ShouldLogExtendedServerRelatedMessages())
                Logger.Instance.WriteServerMessagesToLog(NString.Format("MESSAGE TO SERVER:\n URL: {0}\n BODY: {1}", url, encodedBody));
            response = await this.ExecuteRequest(url, encodedBody);
            if (!response.toUpperCase().startsWith("<HTML")) {
                let startIdx = response.indexOf("<xml id=\"MGDATA\">");
                if (startIdx > 0)
                    response = response.substr(startIdx);
            }
            Logger.Instance.WriteServerMessagesToLog("MESSAGE FROM SERVER: " + response);
            if (sessionStage === CommandsProcessorBase_SessionStage.HANDSHAKE ||
                (response.length > 0 &&
                    RemoteCommandsProcessor.ShouldScrambleAndUnscrambleMessages && response !== "<xml id=\"MGDATA\">\n</xml>")) {
                if (sessionStage === CommandsProcessorBase_SessionStage.HANDSHAKE) {
                    Debug.Assert(response.length > 0 && response[0] !== '<');
                }
                response = CommandsProcessorBase.UnScramble(response);
            }
            Logger.Instance.WriteDevToLog("MESSAGE FROM SERVER: (size = " + response.length + ")" + OSEnvironment.EolSeq + response);
            this._lastRequestTime = Misc.getSystemMilliseconds();
        }
        catch (ex) {
            if (ex instanceof ServerError) {
                requestStatus.value = RequestStatus.Abort;
                if (sessionStage == CommandsProcessorBase_SessionStage.NORMAL) {
                    if (ex.GetCode() == ServerError.ERR_CTX_NOT_FOUND || ex.GetCode() == ServerError.ERR_THREAD_ABORTED) {
                        if (RuntimeContextBase.Instance.RemovedContextFromServer) {
                            requestStatus.value = RequestStatus.Retry;
                        }
                        else {
                            switch (Environment.Instance.getWebClientReinitSession()) {
                                case 'Y':
                                    requestStatus.value = RequestStatus.Retry;
                                    break;
                                case 'A':
                                    let retval = await GUIManager.Instance.MessageBox("Error", "Unknown error occurred at server-side." +
                                        " Would you like to reinitialize the session?", Styles.MSGBOX_BUTTON_YES_NO);
                                    requestStatus.value = (retval == Styles.MSGBOX_RESULT_YES) ? RequestStatus.Retry : RequestStatus.Abort;
                                    break;
                                case 'N':
                                default:
                                    requestStatus.value = RequestStatus.Abort;
                                    break;
                            }
                        }
                        if (requestStatus.value == RequestStatus.Abort) {
                            Environment.Instance.setWebClientReinitSession('N');
                            throw ex;
                        }
                        else {
                            let errForConsole = StrUtil.getConsoleErorString(ex.GetMessage());
                            RemoteCommandsProcessor.lastExceptionMessage = "Exception Occured at server side : " + ex.GetCode().toString() + " : " + errForConsole;
                            console.log(RemoteCommandsProcessor.lastExceptionMessage);
                        }
                        return null;
                    }
                }
            }
            this.clearLocalStorage();
            throw ex;
        }
        return response;
    }
    clearLocalStorage() {
        window.localStorage.removeItem(ConstInterface.LAST_ROUTE_EVENT);
        window.localStorage.removeItem(ConstInterface.LAST_ROUTE_EVENT_SRC_TSK);
        window.localStorage.removeItem(ConstInterface.LAST_ROUTE_EVENT_ARG_LIST);
        window.localStorage.removeItem(ConstInterface.IS_SESSION_REINITIALIZING);
        window.localStorage.removeItem(ConstInterface.MAIN_PROG_VIEW);
        window.localStorage.removeItem(ConstInterface.GLOBAL_PARAM_LIST);
        window.localStorage.removeItem(ConstInterface.ENV_VAR_LIST);
        window.localStorage.removeItem(ConstInterface.LAST_EXCEPTION);
        window.localStorage.removeItem(ConstInterface.CTX_REMOVED_FROM_SRVR);
    }
    HandleErrorResponse(response) {
        try {
            Logger.Instance.WriteServerMessagesToLog("MESSAGE FROM SERVER: " + response);
            response = CommandsProcessorBase.UnScramble(response);
            Logger.Instance.WriteServerMessagesToLog("MESSAGE FROM SERVER: " + StrUtil.getConsoleErorString(response));
        }
        catch (ex) {
        }
        if (response.startsWith("<xmlerr")) {
            let errorMessageXml = new ErrorMessageXml(response, this._lastRequestTime, AccessHelper.environment.getContextInactivityTimeout());
            throw new ServerError(errorMessageXml.GetMessage(), errorMessageXml.GetCode());
        }
        else if (response.toUpperCase().startsWith("<HTML")) {
            throw new ServerError(response);
        }
    }
    async ExecuteRequest(url, encodedBody) {
        return await this.GetContent(url, false, encodedBody);
    }
    async GetContent(requestedURL, useCache, requestContent) {
        if (isUndefined(requestContent))
            requestContent = null;
        let responseStr;
        try {
            if (requestedURL.startsWith("/"))
                requestedURL = ServerConfig.Instance.getProtocol() + "://" + ServerConfig.Instance.getServer() + requestedURL;
            var spinnerTimer = timer(50, 50);
            var spinnerTimerSubscription = spinnerTimer.subscribe(() => {
                AccessHelper.eventsManager.CheckAndShowSpinner(true);
            });
            let isError = new RefParam(false);
            responseStr = await HttpManager.GetInstance().GetContent(requestedURL, requestContent, useCache, isError);
            spinnerTimerSubscription.unsubscribe();
            spinnerTimerSubscription = null;
            if (isError.value) {
                this.HandleErrorResponse(responseStr);
            }
        }
        catch (ex) {
            if (!(ex instanceof ServerError) || Logger.Instance.LogLevel !== Logger_LogLevels.Basic) {
                Logger.Instance.WriteExceptionToLog(ex, NString.Format("requested URL = \"{0}\"", requestedURL));
            }
            if (!(ex instanceof ServerError)) {
                ex = new ServerError(ex.Message, ex.InnerException ? ex.InnerException : ex);
            }
            throw ex;
        }
        finally {
            if (spinnerTimerSubscription != null)
                spinnerTimerSubscription.unsubscribe();
        }
        return responseStr;
    }
    ClientActivated() {
    }
    BuildUrlSuffix(hasContent, isInitialCall) {
        let prefix = ConstInterface.RC_INDICATION + ConstInterface.RC_TOKEN_CTX_ID + RuntimeContextBase.Instance.ContextID;
        if (!isNullOrUndefined(this.SessionId))
            prefix += (ConstInterface.REQ_ARG_SEPARATOR + ConstInterface.RC_TOKEN_SESSION_ID + this.SessionId);
        if (isInitialCall) {
            if (RemoteCommandsProcessor.IsSessionReInitializing)
                prefix += (ConstInterface.REQ_ARG_SEPARATOR + ConstInterface.WEBCLIENT_REINITIALIZE_REQUEST);
            let globalParams = ServerConfig.Instance.getGlobalParams();
            if (globalParams !== null) {
                globalParams = NString.Replace(globalParams, "+", "%2B");
                prefix += ("&" + ConstInterface.MG_TAG_GLOBALPARAMS + "=" + globalParams);
            }
        }
        if (hasContent) {
            prefix += ("&" + ConstInterface.RC_TOKEN_SESSION_COUNT + this.GetSessionCounter() +
                ConstInterface.REQ_ARG_SEPARATOR + ConstInterface.RC_TOKEN_DATA);
        }
        return prefix;
    }
    async SendMonitorOnly() {
        let flowMonitor = FlowMonitorQueue.Instance;
        if (AccessHelper.mgDataTable == null || AccessHelper.mgDataTable.getMGData(0) == null || AccessHelper.mgDataTable.getMGData(0).IsAborting)
            return;
        if (!flowMonitor.isEmpty()) {
            let buffer = this.BuildMonitorMessage();
            let shouldAccumulateClientLog = false;
            if (shouldAccumulateClientLog) {
            }
            else {
                try {
                    let requestStatus = new RefParam(RequestStatus.Handled);
                    await this.DispatchRequest(this.ServerUrl, buffer.ToString(), CommandsProcessorBase_SessionStage.NORMAL, requestStatus);
                }
                catch (err) {
                }
            }
        }
    }
    BuildMonitorMessage() {
        let buffer = new StringBuilder(1000);
        let flowMonitor = FlowMonitorQueue.Instance;
        if (!RemoteCommandsProcessor.ShouldScrambleAndUnscrambleMessages) {
            buffer.Append(XMLConstants.MG_TAG_OPEN);
            flowMonitor.buildXML(buffer);
        }
        else {
            flowMonitor.buildXML(buffer);
            let scrambledOut = Scrambler.Scramble(buffer.ToString());
            buffer = new StringBuilder(1000);
            buffer.Append(XMLConstants.MG_TAG_OPEN + scrambledOut);
        }
        buffer.Append("</" + XMLConstants.MG_TAG_XML + XMLConstants.TAG_CLOSE);
        return buffer;
    }
    RegisterDelegates() {
        HttpClientEvents.GetSessionCounter_Event = this.GetSessionCounter.bind(this);
        HttpClientEvents.CheckAndSetSessionCounter_Event = this.CheckAndSetSessionCounter.bind(this);
        HttpClientEvents.ComputeAndLogRequestInfo_Event = this.ComputeAndLogRequestInfo.bind(this);
        AccessHelper.eventsManager.SpinnerStopped.subscribe(clientBusyTime => this.ClientActivated());
    }
    ComputeAndLogRequestInfo(contentLength, roundTrip, isLoggingResponse) {
        if (!isNullOrUndefined(this._requestInfo.serverCallAt)) {
            this._requestInfo.extraInfo.contentLength = contentLength;
            this._requestInfo.extraInfo.timeStamp = new Date().toString();
            let extraMessageStr = isLoggingResponse ? 'Response ⤵' : 'Request ⤵';
            if (isLoggingResponse) {
                this._requestInfo.roundtripTime = roundTrip;
            }
            Logger.Instance.WriteRequestInfoToLog(this._requestInfo, extraMessageStr);
            if (isLoggingResponse)
                this._requestInfo.clear();
        }
    }
    static set ShouldScrambleAndUnscrambleMessages(value) {
        RemoteCommandsProcessor._shouldScrambleAndUnscrambleMessages = value;
        Logger.Instance.WriteServerMessagesToLog(NString.Format("ShouldScrambleAndUnscrambleMessages.Set: {0}", this._shouldScrambleAndUnscrambleMessages));
    }
    static get ShouldScrambleAndUnscrambleMessages() {
        Logger.Instance.WriteServerMessagesToLog(NString.Format("ShouldScrambleAndUnscrambleMessages.Get: {0}", this._shouldScrambleAndUnscrambleMessages));
        return RemoteCommandsProcessor._shouldScrambleAndUnscrambleMessages;
    }
    parseLogLevel(strLogLevel) {
        let logLevel = Logger_LogLevels.None;
        if (!NString.IsNullOrEmpty(strLogLevel)) {
            if (strLogLevel.toUpperCase().startsWith("SERVER"))
                logLevel = strLogLevel.endsWith("#") ? Logger_LogLevels.ServerMessages : Logger_LogLevels.Server;
            else if (strLogLevel.toUpperCase().startsWith("S") || strLogLevel.toUpperCase().startsWith("Q"))
                logLevel = Logger_LogLevels.Support;
            else if (strLogLevel.toUpperCase().startsWith("G"))
                logLevel = Logger_LogLevels.Gui;
            else if (strLogLevel.toUpperCase().startsWith("D"))
                logLevel = Logger_LogLevels.Development;
            else if (strLogLevel.toUpperCase().startsWith("B"))
                logLevel = Logger_LogLevels.Basic;
            else if (strLogLevel.toUpperCase().startsWith("R"))
                logLevel = Logger_LogLevels.RequestInfo;
        }
        return logLevel;
    }
    async PrepareRequest(serializeTasks) {
        let xmlBuf = new StringBuilder();
        xmlBuf.Append(XMLConstants.MG_TAG_OPEN);
        if (AccessHelper.mgDataTable.getCurrMGData() !== null) {
            let xmlUnscrambled = new StringBuilder();
            await AccessHelper.mgDataTable.buildXML(xmlUnscrambled, serializeTasks);
            if (RemoteCommandsProcessor.ShouldScrambleAndUnscrambleMessages)
                xmlBuf.Append(Scrambler.Scramble(xmlUnscrambled.ToString()));
            else
                xmlBuf.Append(xmlUnscrambled.ToString());
        }
        xmlBuf.Append("</" + XMLConstants.MG_TAG_XML + XMLConstants.TAG_CLOSE);
        return xmlBuf.ToString();
    }
    async ProcessResponse(response, currMgdID, openingTaskDetails, res) {
        Logger.Instance.WriteDevToLog("<-- ProcessResponse started -->");
        if (openingTaskDetails === null)
            openingTaskDetails = new OpeningTaskDetails();
        let systemMilliseconds = Misc.getSystemMilliseconds();
        if (response === null || response.trim().length === 0)
            return;
        AccessHelper.mgDataTable.currMgdID = currMgdID;
        RuntimeContextBase.Instance.Parser.push();
        RuntimeContextBase.Instance.Parser.setXMLdata(response);
        RuntimeContextBase.Instance.Parser.setCurrIndex(0);
        let currMGData = AccessHelper.mgDataTable.getCurrMGData();
        await currMGData.fillData(openingTaskDetails, RuntimeContextBase.Instance.Parser);
        RuntimeContextBase.Instance.Parser.setXMLdata(null);
        RuntimeContextBase.Instance.Parser.pop();
        Logger.Instance.WriteDevToLog("<-- ProcessResponse finished --> (" + (Misc.getSystemMilliseconds() - systemMilliseconds) + ")");
        await currMGData.CmdsToClient.Execute(res);
        await this.ProcessRecovery();
    }
    async ProcessRecovery() {
        AccessHelper.eventsManager.pushNewExecStacks();
        await AccessHelper.mgDataTable.processRecovery();
        AccessHelper.eventsManager.popNewExecStacks();
    }
}
RemoteCommandsProcessor.RC_NO_CONTEXT_ID = '-1';
RemoteCommandsProcessor._instance = null;
RemoteCommandsProcessor.IsSessionReInitializing = false;
RemoteCommandsProcessor.lastExceptionMessage = null;
RemoteCommandsProcessor.InitialUrl = null;
RemoteCommandsProcessor.WEB_COMMUNICATION_PROTOCOL_VERSION = "14002";
RemoteCommandsProcessor._shouldScrambleAndUnscrambleMessages = false;
class HandshakeResponse {
    get ScrambleMessages() {
        return this._scrambleMessages;
    }
    get ContextId() {
        return this._contextId;
    }
    get GetSessionId() {
        return this._privateSessionId;
    }
    get InputPassword() {
        return this._inputPassword;
    }
    get HttpTimeout() {
        return this._httpTimeout;
    }
    get SystemLogin() {
        return this._systemLogin;
    }
    get MaxInternalLogLevel() {
        return this._maxInternalLogLevel;
    }
    constructor(responseXML) {
        this._scrambleMessages = true;
        this._contextId = null;
        this._privateSessionId = null;
        this._inputPassword = false;
        this._httpTimeout = 0;
        this._systemLogin = null;
        this._maxInternalLogLevel = null;
        try {
            JSON_Utils.JSONFromXML(responseXML, this.FillFromJSON.bind(this));
        }
        catch (ex) {
            Logger.Instance.WriteExceptionToLog(ex, responseXML);
        }
    }
    FillFromJSON(error, result) {
        if (error != null) {
            throw error;
        }
        let response = result['Richclientresponse'];
        for (let elementName in response) {
            if (response.hasOwnProperty(elementName)) {
                switch (elementName) {
                    case "ContextID":
                        this._contextId = response[elementName][0];
                        break;
                    case "SessionID":
                        this._privateSessionId = response[elementName][0];
                        break;
                    case "Environment":
                        let envAttributes = response["Environment"][0]['$'];
                        if (!isNullOrUndefined(envAttributes["ScrambleMessages"])) {
                            Debug.Assert(envAttributes["ScrambleMessages"] === "N");
                            this._scrambleMessages = false;
                        }
                        if (!isNullOrUndefined(envAttributes["MaxInternalLogLevel"]))
                            this._maxInternalLogLevel = envAttributes["MaxInternalLogLevel"];
                        if (!isNullOrUndefined(envAttributes["InputPassword"]) && envAttributes["InputPassword"].toLocaleLowerCase() === "y")
                            this._inputPassword = true;
                        if (!isNullOrUndefined(envAttributes["SystemLogin"]))
                            this._systemLogin = envAttributes["SystemLogin"];
                        if (!isNullOrUndefined(envAttributes[ConstInterface.MG_TAG_HTTP_COMMUNICATION_TIMEOUT]))
                            this._httpTimeout = +envAttributes[ConstInterface.MG_TAG_HTTP_COMMUNICATION_TIMEOUT];
                        if (!isNullOrUndefined(envAttributes["ForwardSlash"]))
                            AccessHelper.environment.ForwardSlashUsage = envAttributes["ForwardSlash"];
                        break;
                }
            }
        }
    }
}
HandshakeResponse.SYSTEM_LOGIN_AD = 'D';
class ErrorMessageXml {
    constructor(xmlContent, lastRequestTime, contextInactivityTimeout) {
        this._lastRequestTime = 0;
        this._errorMessage = null;
        this._errorCode = 0;
        this._middlewareAddress = null;
        this._parsingFailed = false;
        this._lastRequestTime = lastRequestTime;
        this.contextInactivityTimeout = contextInactivityTimeout;
        try {
            JSON_Utils.JSONFromXML(xmlContent, this.FillFromJSON.bind(this));
        }
        catch (ex) {
            Logger.Instance.WriteExceptionToLog(ex);
            Misc.WriteStackTrace(ex);
            this._parsingFailed = true;
        }
    }
    FillFromJSON(error, result) {
        if (error != null) {
            throw error;
        }
        let xmlerr = result['xmlerr'];
        for (let elementName in xmlerr) {
            if (xmlerr.hasOwnProperty(elementName)) {
                switch (elementName) {
                    case "errmsg":
                        this._errorMessage = xmlerr[elementName][0];
                        break;
                    case "errcode":
                        this._errorCode = +(xmlerr[elementName]);
                        break;
                    case "server":
                        this._middlewareAddress = xmlerr[elementName][0];
                        break;
                    case "appname":
                    case "prgname":
                    case "arguments":
                    case "username":
                    case "xmlerr":
                        break;
                    default:
                        Logger.Instance.WriteExceptionToLogWithMsg(NString.Format("Unknown element: '{0}'", elementName));
                        break;
                }
            }
        }
    }
    GetCode() {
        return this._errorCode;
    }
    GetMessage() {
        let sb = new StringBuilder();
        if (this._parsingFailed) {
            sb.Append(AccessHelper.languageData.getConstMessage(MsgInterface.RC_STR_F7_UNEXPECTED_ERR));
        }
        else {
            switch (this._errorCode) {
                case ServerError.ERR_CTX_NOT_FOUND:
                    if (this.InactivityTimeoutExpired())
                        sb.Append(NString.Format("{0} ({1} {2})", AccessHelper.languageData.getConstMessage(MsgInterface.STR_ERR_SESSION_CLOSED_INACTIVITY), Math.floor(this.contextInactivityTimeout / 600), AccessHelper.languageData.getConstMessage(MsgInterface.STR_MINUTES)));
                    else
                        sb.Append(NString.Format("{0} ({1}).", AccessHelper.languageData.getConstMessage(MsgInterface.STR_ERR_SESSION_CLOSED), this._errorCode));
                    break;
                case ServerError.ERR_WEBCLIENT_PROGRAM_RELOADED:
                    sb.Append(AccessHelper.languageData.getConstMessage(MsgInterface.STR_ERR_WEBCLIENT_PROGRAM_RELOADED));
                    break;
                case ServerError.ERR_EXECUTED_PROGRAM_CHANGED:
                    sb.Append(AccessHelper.languageData.getConstMessage(MsgInterface.STR_ERR_EXECUTED_PROGRAM_CHANGED));
                    break;
                case ServerError.ERR_AUTHENTICATION:
                    sb.Append(AccessHelper.languageData.getConstMessage(MsgInterface.USRINP_STR_BADPASSW));
                    break;
                case ServerError.ERR_ACCESS_DENIED:
                    sb.Append(AccessHelper.languageData.getConstMessage(MsgInterface.STR_ERR_AUTHORIZATION_FAILURE));
                    break;
                default:
                    sb.Append(this._errorMessage);
                    break;
            }
            sb.Append(OSEnvironment.EolSeq + OSEnvironment.EolSeq);
            if (!ServerConfig.Instance.ShouldDisplayGenericError()) {
                sb.Append(AccessHelper.languageData.getConstMessage(MsgInterface.BROWSER_OPT_INFO_SERVER_STR) + ":   ");
                sb.Append(ServerConfig.Instance.getServer());
                if (!NString.IsNullOrEmpty(this._middlewareAddress)) {
                    sb.Append(" --> ");
                    sb.Append(this._middlewareAddress);
                }
                sb.Append(OSEnvironment.EolSeq);
            }
            sb.Append(AccessHelper.languageData.getConstMessage(MsgInterface.TASKRULE_STR_APPLICATION) + ":   \"");
            sb.Append(ServerConfig.Instance.getAppName());
            sb.Append("\" (\"");
            sb.Append("\")");
        }
        return sb.ToString();
    }
    InactivityTimeoutExpired() {
        let expired = false;
        if (this.contextInactivityTimeout > 0) {
            let currTimeMilli = Misc.getSystemMilliseconds();
            expired = ((currTimeMilli - this._lastRequestTime) > (this.contextInactivityTimeout * 100));
        }
        return expired;
    }
}
//# sourceMappingURL=data:application/json;base64,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