import { ApplicationException, NNumber } from '@magic-xpa/mscorelib';
import { Logger, StorageAttribute, StorageAttributeCheck } from '@magic-xpa/utils';
import { BlobType, ExpVal, FieldDef } from '@magic-xpa/gui';
import { ConstInterface } from '../ConstInterface';
import { FieldBase } from '../data/FieldBase';
export class Argument {
    constructor(fieldOrExpValOrSrcArg) {
        this._exp = null;
        this._fld = null;
        this._skip = false;
        this._type = '\0';
        this._val = null;
        this._valueAttr = StorageAttribute.NONE;
        this._valueIsNull = false;
        if (arguments.length === 0) {
            this.constructor_0();
            return;
        }
        if (arguments.length === 1 && (fieldOrExpValOrSrcArg === null || fieldOrExpValOrSrcArg instanceof FieldBase)) {
            this.constructor_1(fieldOrExpValOrSrcArg);
            return;
        }
        if (arguments.length === 1 && (fieldOrExpValOrSrcArg === null || fieldOrExpValOrSrcArg instanceof ExpVal)) {
            this.constructor_2(fieldOrExpValOrSrcArg);
            return;
        }
        this.constructor_3(fieldOrExpValOrSrcArg);
    }
    constructor_0() {
        this._type = '\0';
        this._fld = null;
        this._exp = null;
        this._skip = false;
    }
    constructor_1(field) {
        this._fld = field;
        this._type = ConstInterface.ARG_TYPE_FIELD;
        this._exp = null;
        this._skip = false;
    }
    constructor_2(expVal) {
        this._type = ConstInterface.ARG_TYPE_VALUE;
        this._fld = null;
        this._exp = null;
        this._skip = false;
        if (expVal.IsNull)
            this._val = null;
        else
            this._val = expVal.ToMgVal();
        this._valueIsNull = (this._val === null);
        this._valueAttr = expVal.Attr;
    }
    constructor_3(srcArg) {
        this._type = ConstInterface.ARG_TYPE_VALUE;
        switch (srcArg._type) {
            case ConstInterface.ARG_TYPE_FIELD:
                this._valueAttr = srcArg._fld.getType();
                break;
            case ConstInterface.ARG_TYPE_EXP:
                break;
            case ConstInterface.ARG_TYPE_SKIP:
                this._skip = true;
                break;
            default:
                throw new ApplicationException("in Argument.Argument(): illegal source Argument type!");
        }
    }
    async Initialize(srcArg) {
        switch (srcArg._type) {
            case ConstInterface.ARG_TYPE_FIELD:
                this._val = srcArg._fld.getValue(true);
                this._valueIsNull = srcArg._fld.isNull();
                break;
            case ConstInterface.ARG_TYPE_EXP:
                let retVal = await srcArg._exp.evaluateWithResLength(255);
                this._val = retVal.mgVal;
                this._valueAttr = retVal.type;
                this._valueIsNull = (this._val == null);
                if (this._valueIsNull)
                    this._val = Argument.getEmptyValue((retVal.type === StorageAttribute.BLOB ||
                        retVal.type === StorageAttribute.BLOB_VECTOR));
                break;
        }
    }
    fillData(arg, srcTask) {
        this._type = arg[0];
        let argElements = arg.substr(2);
        switch (this._type) {
            case ConstInterface.ARG_TYPE_FIELD:
                {
                    let fieldId = argElements.split(',');
                    let parent = NNumber.Parse(fieldId[0]);
                    let fldIdx = NNumber.Parse(fieldId[1]);
                    this._fld = srcTask.getFieldDef(parent, fldIdx);
                }
                break;
            case ConstInterface.ARG_TYPE_EXP:
                let expNum = NNumber.Parse(argElements);
                this._exp = srcTask.getExpById(expNum);
                break;
            case ConstInterface.ARG_TYPE_SKIP:
                this._skip = true;
                break;
            default:
                Logger.Instance.WriteExceptionToLogWithMsg("in Argument.FillData() illegal type: " + arg);
                break;
        }
    }
    buildXML(message) {
        switch (this._type) {
            case ConstInterface.ARG_TYPE_FIELD:
                message.Append("F:" + this._fld.getTask().getTaskTag() + "," + this._fld.getId());
                break;
            case ConstInterface.ARG_TYPE_EXP:
                message.Append("E:" + this._exp.getId());
                break;
            case ConstInterface.ARG_TYPE_SKIP:
                message.Append("X:0");
                break;
            default:
                Logger.Instance.WriteExceptionToLogWithMsg("in Argument.buildXML() illegal type: " + this._type);
                break;
        }
    }
    getType() {
        return this._type;
    }
    getField() {
        if (this._type === ConstInterface.ARG_TYPE_FIELD)
            return this._fld;
        return null;
    }
    getExp() {
        if (this._type === ConstInterface.ARG_TYPE_EXP)
            return this._exp;
        return null;
    }
    async setValueToField(destFld) {
        let val;
        let isNull;
        switch (this._type) {
            case ConstInterface.ARG_TYPE_FIELD:
                val = this._fld.getValue(true);
                val = Argument.convertArgs(val, this._fld.getType(), destFld.getType());
                isNull = this._fld.isNull() || val == null;
                break;
            case ConstInterface.ARG_TYPE_EXP:
                val = await this._exp.evaluateWithResultTypeAndLength(destFld.getType(), destFld.getSize());
                isNull = (val == null);
                if (isNull)
                    val = Argument.getEmptyValue((destFld.getType() === StorageAttribute.BLOB ||
                        destFld.getType() === StorageAttribute.BLOB_VECTOR));
                break;
            case ConstInterface.ARG_TYPE_VALUE:
                val = Argument.convertArgs(this._val, this._valueAttr, destFld.getType());
                isNull = this._valueIsNull || val == null;
                break;
            default:
                return;
        }
        destFld.takeValFromRec();
        await destFld.setValueAndStartRecompute(val, isNull, destFld.getTask().DataViewWasRetrieved, false, true);
    }
    static convertArgs(value, srcAttr, expectedType) {
        let invalidArg = false;
        switch (expectedType) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.UNICODE:
                if (srcAttr === StorageAttribute.BLOB) {
                    if (BlobType.isValidBlob(value))
                        value = BlobType.getString(value);
                }
                else if (!StorageAttributeCheck.IsTypeAlphaOrUnicode(srcAttr))
                    invalidArg = true;
                break;
            case StorageAttribute.NUMERIC:
            case StorageAttribute.DATE:
            case StorageAttribute.TIME:
                if (!StorageAttributeCheck.isTypeNumeric(srcAttr))
                    invalidArg = true;
                break;
            case StorageAttribute.BLOB:
                if (StorageAttributeCheck.IsTypeAlphaOrUnicode(srcAttr)) {
                    let contentType = srcAttr === StorageAttribute.ALPHA
                        ? BlobType.CONTENT_TYPE_ANSI
                        : BlobType.CONTENT_TYPE_UNICODE;
                    value = BlobType.createFromString(value, contentType);
                }
                else if (!StorageAttributeCheck.isTypeBlob(srcAttr))
                    invalidArg = true;
                break;
        }
        if (invalidArg)
            value = FieldDef.getMagicDefaultValue(expectedType);
        return value;
    }
    async getValue(expType, expSize) {
        switch (this._type) {
            case ConstInterface.ARG_TYPE_EXP:
                this._val = await this._exp.evaluateWithResultTypeAndLength(expType, expSize);
                if (this._val == null)
                    this._val = Argument.getEmptyValue(expType === StorageAttribute.BLOB
                        || expType === StorageAttribute.BLOB_VECTOR);
                break;
            case ConstInterface.ARG_TYPE_FIELD:
                this._val = this._fld.getValue(true);
                break;
            case ConstInterface.ARG_TYPE_VALUE:
                break;
            default:
                return null;
        }
        return this._val;
    }
    skipArg() {
        return this._skip;
    }
    fillDataByParams(paramValueAttr, paramValue, paramNull) {
        this._valueAttr = paramValueAttr;
        this._val = paramValue;
        this._valueIsNull = paramNull;
        if (this._valueAttr === StorageAttribute.SKIP) {
            this._type = ConstInterface.ARG_TYPE_SKIP;
            this._skip = true;
        }
        else {
            this._type = ConstInterface.ARG_TYPE_VALUE;
        }
    }
    static getEmptyValue(isBlob) {
        return isBlob ? (BlobType.getEmptyBlobPrefix('\0') + ";") : "";
    }
    FillFromString(argStr) {
        this._type = ConstInterface.ARG_TYPE_VALUE;
        let argType = null;
        if (argStr.length > 2)
            argType = argStr.substr(0, 2);
        switch (argType) {
            case ConstInterface.REQ_ARG_ALPHA:
                this._valueAttr = StorageAttribute.ALPHA;
                break;
            case ConstInterface.REQ_ARG_UNICODE:
                this._valueAttr = StorageAttribute.UNICODE;
                break;
            case ConstInterface.REQ_ARG_NUMERIC:
                this._valueAttr = StorageAttribute.NUMERIC;
                break;
            case ConstInterface.REQ_ARG_DOUBLE:
                this._valueAttr = StorageAttribute.NUMERIC;
                break;
            case ConstInterface.REQ_ARG_LOGICAL:
                this._valueAttr = StorageAttribute.BOOLEAN;
                break;
            case ConstInterface.REQ_ARG_NULL:
                this._valueAttr = StorageAttribute.NONE;
                break;
            default:
                this._valueAttr = StorageAttribute.ALPHA;
                this._val = argStr;
                return;
        }
        this._val = argStr.substr(2);
    }
}
//# sourceMappingURL=data:application/json;base64,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