import { ApplicationException, Debug, Int32, NNumber, NString, Thread } from '@magic-xpa/mscorelib';
import { Constants, Logger, StorageAttribute, StorageAttributeCheck, StrUtil, UtilStrByteMode, XMLConstants, XmlParser } from '@magic-xpa/utils';
import { Commands, NUM_TYPE, PropInterface, Styles, TaskDefinitionIdTableSaxHandler, VectorType } from '@magic-xpa/gui';
import { YesNoExp } from '../exp/YesNoExp';
import { ArgumentsList } from './ArgumentsList';
import { RunTimeEvent } from '../event/RunTimeEvent';
import { CommandFactory } from '../commands/ClientToServer/CommandFactory';
import { FlowMonitorQueue } from '../util/FlowMonitorQueue';
import { ExecOperCommand } from '../commands/ClientToServer/ExecOperCommand';
import { ConstInterface } from '../ConstInterface';
import { FlowMonitorInterface } from '../FlowMonitorInterface';
import { LanguageData } from '../env/LanguageData';
import { LastFocusedManager } from './LastFocusedManager';
import { AccessHelper, ClearEventsOnStopExecution } from '../AccessHelper';
import { GUIManager } from '../GUIManager';
export class Operation {
    constructor() {
        this._condExp = new YesNoExp(true);
        this._retainFocus = new YesNoExp(false);
        this._waitExp = new YesNoExp(false);
        this._syncData = new YesNoExp(false);
        this._argList = null;
        this._blockClose = 0;
        this._blockEnd = 0;
        this._buttons = '\0';
        this._checkByServer = false;
        this._cmdToServer = null;
        this._display = 'S';
        this._errLogAppend = false;
        this._evtHandler = null;
        this._execOnServer = false;
        this._execOnClient = false;
        this._exp = null;
        this._field = null;
        this._image = '\0';
        this._incremental = false;
        this._mode = '\0';
        this._publicName = null;
        this._returnVal = null;
        this._rtEvt = null;
        this._serverId = -1;
        this.calledTaskDefinitionId = null;
        this._subformCtrlName = null;
        this._subtype = '\0';
        this._task = null;
        this._text = null;
        this._title = null;
        this._titleExp = null;
        this._type = 0;
        this._undo = true;
        this._isRoute = false;
        this._routerPath = null;
        this._originalRouterOutletName = null;
        this._routeParams = null;
        this.OperCallMode = 0;
    }
    get CalledTaskDefinitionId() {
        return this.calledTaskDefinitionId;
    }
    set Task(value) {
        this._task = value;
    }
    get Task() {
        return this._task;
    }
    async Immediate() {
        return await this._waitExp.getVal();
    }
    fillData(taskRef, evtHandler, parser) {
        this._task = taskRef;
        this._evtHandler = evtHandler;
        while (this.initInnerObjects(parser, parser.getNextTag(), taskRef)) {
        }
    }
    initInnerObjects(parser, foundTagName, taskRef) {
        if (foundTagName === null)
            return false;
        if (foundTagName === ConstInterface.MG_TAG_EVENT) {
            this._rtEvt = new RunTimeEvent(taskRef);
            this._rtEvt.fillData(parser, taskRef);
        }
        else if (foundTagName === ConstInterface.MG_TAG_OPER) {
            this.fillAttributes(parser, taskRef);
        }
        else if (foundTagName === "/" + ConstInterface.MG_TAG_OPER) {
            parser.setCurrIndex2EndOfTag();
            return false;
        }
        else if (foundTagName === XMLConstants.MG_TAG_TASKDEFINITIONID_ENTRY) {
            this.InitTaskDefinitionID(parser);
        }
        else {
            Logger.Instance.WriteExceptionToLogWithMsg("There is no such tag in Operation. Insert else if to Operation.initInnerObjects for " + foundTagName);
            return false;
        }
        return true;
    }
    fillAttributes(parser, taskRef) {
        let tokensVector;
        let endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, parser.getCurrIndex());
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            let tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(ConstInterface.MG_TAG_OPER) + ConstInterface.MG_TAG_OPER.length);
            tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            this.initElements(tokensVector, taskRef);
            parser.setCurrIndex(endContext + XMLConstants.TAG_CLOSE.length);
            return;
        }
        Logger.Instance.WriteExceptionToLogWithMsg("in Command.FillData() out of string bounds");
    }
    initElements(tokensVector, taskRef) {
        let attribute, valueStr;
        let expId;
        for (let j = 0; j < tokensVector.length; j += 2) {
            attribute = (tokensVector.get_Item(j));
            valueStr = (tokensVector.get_Item(j + 1));
            switch (attribute) {
                case XMLConstants.MG_ATTR_TYPE:
                    this._type = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_FLD:
                    this._field = Operation.InitField(valueStr, this._task);
                    break;
                case XMLConstants.MG_ATTR_EXP:
                    expId = XmlParser.getInt(valueStr);
                    this._exp = taskRef.getExpById(expId);
                    break;
                case ConstInterface.MG_ATTR_TEXT:
                    this._text = XmlParser.unescape(valueStr);
                    break;
                case ConstInterface.MG_ATTR_MODE:
                    this._mode = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_SUBTYPE:
                    this._subtype = valueStr[0];
                    if (this._type === ConstInterface.MG_OPER_BLOCK) {
                        if (this._subtype === 'E')
                            this._type = ConstInterface.MG_OPER_ELSE;
                        else if (this._subtype === 'L')
                            this._type = ConstInterface.MG_OPER_LOOP;
                    }
                    break;
                case ConstInterface.MG_ATTR_CLOSE:
                    this._blockClose = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_END:
                    this._blockEnd = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_HOW:
                    this._incremental = (valueStr === "I");
                    break;
                case ConstInterface.MG_ATTR_UNDO:
                    if (valueStr[0] === 'N')
                        this._undo = false;
                    else if (valueStr[0] === 'Y')
                        this._undo = true;
                    else
                        Logger.Instance.WriteExceptionToLogWithMsg("in Operation.initElements(): No such value to the MG_ATTR_UNDO for " + valueStr);
                    break;
                case ConstInterface.MG_ATTR_DISPLAY:
                    this._display = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_WAIT:
                    this._waitExp.setVal(taskRef, valueStr);
                    break;
                case ConstInterface.MG_ATTR_RETAIN_FOCUS:
                    this._retainFocus.setVal(taskRef, valueStr);
                    break;
                case ConstInterface.MG_ATTR_SYNC_DATA:
                    this._syncData.setVal(taskRef, valueStr);
                    break;
                case ConstInterface.MG_ATTR_SHOW:
                    break;
                case ConstInterface.MG_ATTR_CND:
                    this._condExp.setVal(taskRef, valueStr);
                    break;
                case ConstInterface.MG_ATTR_ARGLIST:
                    this._argList = new ArgumentsList();
                    this._argList.fillList(valueStr, this._task);
                    break;
                case ConstInterface.MG_ATTR_REFRESHON:
                    this._argList.RefreshOnString = valueStr.Trim();
                    break;
                case ConstInterface.MG_ATTR_SERVER_ID:
                    this._serverId = XmlParser.getInt(valueStr);
                    break;
                case ConstInterface.MG_ATTR_OPER_CALLMODE:
                    this.OperCallMode = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_SUBFORM_CTRL:
                    this._subformCtrlName = XmlParser.unescape(valueStr);
                    break;
                case ConstInterface.MG_ATTR_CHECK_BY_SERVER:
                    this._checkByServer = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_PUBLIC:
                    this._publicName = valueStr;
                    break;
                case ConstInterface.MG_ATTR_PRG_DESCRIPTION:
                    break;
                case ConstInterface.MG_ATTR_CPY_GLB_PRMS:
                    break;
                case ConstInterface.MG_ATTR_EXEC_ON_SERVER:
                    this._execOnServer = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_EXEC_ON_CLIENT:
                    this._execOnClient = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_TITLE:
                    this._title = XmlParser.unescape(valueStr);
                    break;
                case ConstInterface.MG_ATTR_TITLE_EXP:
                    expId = XmlParser.getInt(valueStr);
                    this._titleExp = taskRef.getExpById(expId);
                    break;
                case ConstInterface.MG_ATTR_IMAGE:
                    this._image = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_BUTTONS:
                    this._buttons = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_RETURN_VAL:
                    this._returnVal = Operation.InitField(valueStr, this._task);
                    break;
                case ConstInterface.MG_ATTR_ERR_LOG_APPEND:
                    this._errLogAppend = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_OPER_METHODNAME:
                    break;
                case ConstInterface.MG_ATTR_IS_GUI_THREAD_EXECUTION:
                    break;
                case ConstInterface.MG_ATTR_IS_ROUTE:
                    this._isRoute = XmlParser.getBoolean(valueStr);
                    break;
                case XMLConstants.MG_ATTR_ROUTER_PATH:
                    this._routerPath = XmlParser.unescape(valueStr);
                    break;
                default:
                    Logger.Instance.WriteExceptionToLogWithMsg("There is no such tag in Operation class. Insert case to Operation.initElements for " + attribute);
                    break;
            }
        }
        if (this._serverId > -1)
            this._cmdToServer = CommandFactory.CreateExecOperCommand(this._task.getTaskTag(), "" + this._evtHandler.getId(), this._serverId, Int32.MinValue, null);
        if (this._isRoute) {
            this._originalRouterOutletName = this._subformCtrlName !== null ? this._subformCtrlName : "";
            if (this._subformCtrlName == null) {
                let defaultRouterOutlet = this._task.getForm().DefaultRouterOutlet;
                this._subformCtrlName = defaultRouterOutlet !== null ? defaultRouterOutlet.Name : "";
            }
        }
    }
    InitTaskDefinitionID(parser) {
        let xmlBuffer = parser.ReadToEndOfCurrentElement();
        let taskDefinitionIdTableSaxHandler = new TaskDefinitionIdTableSaxHandler(this.SetTaskDefinitionId);
        taskDefinitionIdTableSaxHandler.parse(xmlBuffer);
    }
    SetTaskDefinitionId(taskDefinitionId) {
        this.calledTaskDefinitionId = taskDefinitionId;
    }
    static InitField(valueStr, task) {
        let TaskField = XmlParser.getTokens(valueStr, ',');
        if (TaskField.length === 2) {
            let parent = NNumber.Parse(TaskField.get_Item(0));
            let fldID = NNumber.Parse(TaskField.get_Item(1));
            let field = task.getFieldDef(parent, fldID);
            return field;
        }
        Logger.Instance.WriteExceptionToLogWithMsg(NString.Format("Unknown field: '{0}'", valueStr));
        return null;
    }
    async execute(returnedFromServer) {
        let lastRtEvent = AccessHelper.eventsManager.getLastRtEvent();
        let mprgCreator = null;
        let flowMonitor = FlowMonitorQueue.Instance;
        if (lastRtEvent !== null)
            mprgCreator = lastRtEvent.getMainPrgCreator();
        if (returnedFromServer ||
            (!returnedFromServer && (this._type !== ConstInterface.MG_OPER_SERVER || !this._condExp.isServerExp()))) {
            if (!await this.canExecute()) {
                if (this._type !== ConstInterface.MG_OPER_LOOP)
                    await flowMonitor.addFlowFieldOperation(this, false, this._task.GetTaskDetails());
                return false;
            }
        }
        if (this._type !== ConstInterface.MG_OPER_LOOP && this._type !== ConstInterface.MG_OPER_ENDBLOCK)
            await flowMonitor.addFlowFieldOperation(this, true, this._task.GetTaskDetails());
        try {
            switch (this._type) {
                case ConstInterface.MG_OPER_VERIFY:
                    return await this.operVerify();
                case ConstInterface.MG_OPER_BLOCK:
                    return this.operBlock();
                case ConstInterface.MG_OPER_LOOP:
                    if (!this.getExecOnServer())
                        return this.operBlock();
                    else {
                        await this.operServer(mprgCreator);
                        break;
                    }
                case ConstInterface.MG_OPER_ELSE:
                    return this.operElse();
                case ConstInterface.MG_OPER_EVALUATE:
                    await this.operEvaluate();
                    break;
                case ConstInterface.MG_OPER_UPDATE:
                    await this.operUpdate(mprgCreator);
                    break;
                case ConstInterface.MG_OPER_RAISE_EVENT:
                    await this.operRaiseEvent(mprgCreator, returnedFromServer);
                    break;
                case ConstInterface.MG_OPER_SERVER:
                    await this.operServer(mprgCreator);
                    break;
                case ConstInterface.MG_OPER_CALL:
                    if (this._publicName != null) {
                        this.operCallParallel();
                    }
                    else if (this._subformCtrlName == null)
                        await this.operCall(mprgCreator);
                    else {
                        let subformTask;
                        let isEndProg = true;
                        if (this._checkByServer) {
                            if (!(this._cmdToServer instanceof ExecOperCommand))
                                Debug.Assert(false);
                            let command = this._cmdToServer;
                            command.CheckOnly = true;
                            await this.operCall(mprgCreator);
                            command.CheckOnly = false;
                            if (AccessHelper.eventsManager.getNextOperIdx(this, true) > -1)
                                isEndProg = false;
                        }
                        if (isEndProg) {
                            let destSubForm = this._task.getForm().getSubFormCtrlByName(this._subformCtrlName);
                            if (this._isRoute && (destSubForm === null || !destSubForm.IsRouterOutlet()))
                                return false;
                            if (destSubForm != null) {
                                if (this._isRoute) {
                                    let canRoute = false;
                                    let rtEvnt = AccessHelper.eventsManager.getLastRtEvent();
                                    if (rtEvnt !== null) {
                                        this._routeParams = rtEvnt.getRouteParamList();
                                        if ((await rtEvnt.getArgList().getArg(0).getValue(StorageAttribute.ALPHA, 0)).trim() === this._routerPath &&
                                            ((rtEvnt.getArgList().getArg(1).skipArg() && destSubForm === this._task.getForm().DefaultRouterOutlet) ||
                                                (!rtEvnt.getArgList().getArg(1).skipArg() && (await rtEvnt.getArgList().getArg(1).getValue(StorageAttribute.ALPHA, 0)).trim() === this._originalRouterOutletName)))
                                            canRoute = true;
                                        if (!canRoute)
                                            return false;
                                    }
                                }
                                subformTask = destSubForm.getSubformTask();
                                if (subformTask != null) {
                                    let terminateTaskGracefully = true;
                                    let rtEvnt = AccessHelper.eventsManager.getLastRtEvent();
                                    if (this._isRoute && rtEvnt !== null && rtEvnt.isGuiTriggeredEvent()) {
                                        terminateTaskGracefully = false;
                                        AccessHelper.eventsManager.setStopExecution(true, ClearEventsOnStopExecution.NONE);
                                    }
                                    if (await subformTask.endTask(true, false, !terminateTaskGracefully, true)) {
                                        let parentTask = destSubForm.getForm().getTask();
                                        parentTask.TaskService.RemoveRecomputes(parentTask);
                                    }
                                    else
                                        break;
                                }
                                if (LastFocusedManager.getLastFocusedControl() != null &&
                                    LastFocusedManager.getLastFocusedControl().isDescendentOfControl(destSubForm))
                                    this._task.RetainFocus = false;
                                else
                                    this._task.RetainFocus = await this._retainFocus.getVal();
                            }
                            await this.operCall(mprgCreator);
                        }
                    }
                    return this._isRoute;
                default:
                    Logger.Instance.WriteExceptionToLogWithMsg("There is no such type of operation " + this._type);
                    break;
            }
        }
        catch (ex) {
            throw ex;
        }
        return true;
    }
    operCallParallel() {
        Operation.callParallel();
    }
    static callParallel() {
    }
    async getCondVal() {
        return await this._condExp.getVal();
    }
    getType() {
        return this._type;
    }
    getRouteParams() {
        return this._routeParams;
    }
    getServerId() {
        return this._serverId;
    }
    async operVerify() {
        let isError = (this._mode === ConstInterface.FLW_VERIFY_MODE_ERROR);
        let textToDisplay = this._exp === null ? this._text : await this._exp.evaluateWithResultTypeAndLength(StorageAttribute.UNICODE, 255);
        let titleToDisplay = this._titleExp === null ? this._title : await this._titleExp.evaluateWithResultTypeAndLength(StorageAttribute.UNICODE, 255);
        textToDisplay = textToDisplay === null ? "" : StrUtil.rtrim(textToDisplay);
        titleToDisplay = titleToDisplay === null ? "" : StrUtil.rtrim(titleToDisplay);
        if (this._errLogAppend) {
            let prgDescription = this._task.GetContextTask().GetComputedProperty(PropInterface.PROP_TYPE_NAME).GetComputedValue();
            Logger.Instance.WriteExceptionToLogWithMsg(textToDisplay + ", program : " + prgDescription);
        }
        if (this._display === ConstInterface.DISPLAY_STATUS) {
            if (!NString.IsNullOrEmpty(textToDisplay)) {
                FlowMonitorQueue.Instance.addFlowVerifyInfo(textToDisplay, this._task.GetTaskDetails());
                this._task.GetContextTask().WriteToMessagePane(textToDisplay);
            }
        }
        else {
            if (!NString.IsNullOrEmpty(textToDisplay)) {
                let mlsTransTextToDisplay = LanguageData.Instance.translate(textToDisplay);
                let mlsTransTitleToDisplay = LanguageData.Instance.translate(titleToDisplay);
                let verifyMode = Operation.getButtons(this._buttons);
                verifyMode = verifyMode || Operation.getImage(this._image);
                verifyMode = verifyMode;
                if (UtilStrByteMode.isLocaleDefLangJPN()) {
                    let delimPos = mlsTransTextToDisplay.indexOf('|');
                    if (0 <= delimPos && delimPos < mlsTransTextToDisplay.length) {
                        mlsTransTitleToDisplay = mlsTransTextToDisplay.substr(delimPos + 1);
                        mlsTransTextToDisplay = mlsTransTextToDisplay.substr(0, delimPos);
                    }
                }
                let mgForm = null;
                if (!this._task.GetContextTask().getMGData().IsAborting)
                    mgForm = this._task.GetContextTask().getTopMostForm();
                if (mgForm == null && LastFocusedManager.Instance.getLastFocusedTask() != null)
                    mgForm = LastFocusedManager.Instance.getLastFocusedTask().getTopMostForm();
                if (mgForm != null) {
                    mgForm.RefreshUI();
                    Commands.invoke();
                    await Thread.Sleep(10);
                }
                let returnValue = await GUIManager.Instance.MessageBox(mlsTransTitleToDisplay, mlsTransTextToDisplay, verifyMode);
                await Operation.setoperVerifyReturnValue(returnValue, this._returnVal);
            }
        }
        return isError;
    }
    static getButtons(buttonsID) {
        let tmpbuttons = 0;
        switch (buttonsID) {
            case ConstInterface.BUTTONS_OK:
                tmpbuttons = Styles.MSGBOX_BUTTON_OK;
                break;
            case ConstInterface.BUTTONS_OK_CANCEL:
                tmpbuttons = Styles.MSGBOX_BUTTON_OK_CANCEL;
                break;
            case ConstInterface.BUTTONS_ABORT_RETRY_IGNORE:
                tmpbuttons = Styles.MSGBOX_BUTTON_ABORT_RETRY_IGNORE;
                break;
            case ConstInterface.BUTTONS_YES_NO_CANCEL:
                tmpbuttons = Styles.MSGBOX_BUTTON_YES_NO_CANCEL;
                break;
            case ConstInterface.BUTTONS_YES_NO:
                tmpbuttons = Styles.MSGBOX_BUTTON_YES_NO;
                break;
            case ConstInterface.BUTTONS_RETRY_CANCEL:
                tmpbuttons = Styles.MSGBOX_BUTTON_RETRY_CANCEL;
                break;
            default:
                break;
        }
        return tmpbuttons;
    }
    static getImage(imageID) {
        let tmpImage = 0;
        switch (imageID) {
            case ConstInterface.IMAGE_EXCLAMATION:
                tmpImage = Styles.MSGBOX_ICON_EXCLAMATION;
                break;
            case ConstInterface.IMAGE_CRITICAL:
                tmpImage = Styles.MSGBOX_ICON_ERROR;
                break;
            case ConstInterface.IMAGE_QUESTION:
                tmpImage = Styles.MSGBOX_ICON_QUESTION;
                break;
            case ConstInterface.IMAGE_INFORMATION:
                tmpImage = Styles.MSGBOX_ICON_INFORMATION;
                break;
            default:
                break;
        }
        return tmpImage;
    }
    static async setoperVerifyReturnValue(returnValue, returnVal) {
        if (returnVal !== null) {
            let retValueNum = new NUM_TYPE();
            retValueNum.NUM_4_LONG(returnValue);
            let returnValueStr = retValueNum.toXMLrecord();
            await returnVal.setValueAndStartRecompute(returnValueStr, false, true, true, false);
            await returnVal.updateDisplay();
        }
    }
    operBlock() {
        return true;
    }
    operElse() {
        return true;
    }
    async operEvaluate() {
        let result = '\0';
        if (this._field !== null) {
            result = await this._exp.evaluateWithResultTypeAndLength(this._field.getType(), this._field.getSize());
            await this._field.setValueAndStartRecompute(result, result === null, true, true, false);
            await this._field.updateDisplay();
        }
        else
            result = await this._exp.evaluateWithResultTypeAndLength(StorageAttribute.BOOLEAN, 0);
    }
    async operUpdate(mprgCreator) {
        let result, oldVal, newVal, fieldVal;
        let setRecordUpdated;
        let recompute;
        let nOld, nNew, nResult;
        let flowMonitor = FlowMonitorQueue.Instance;
        flowMonitor.addFlowOperationUpdate(FlowMonitorInterface.FLWMTR_START, this._task.GetTaskDetails());
        if (!this._field.DbModifiable && (this._task.getMode() !== Constants.TASK_MODE_CREATE)) {
            if (!await this.CanModify(mprgCreator))
                return;
        }
        let fieldType = this._field.getType();
        if (this._incremental) {
            if (!this._field.IsLinkField) {
                fieldVal = this._field.getValue(true);
                nResult = (fieldVal != null
                    ? new NUM_TYPE(fieldVal)
                    : null);
                if (this._task.getMode() !== Constants.TASK_MODE_CREATE) {
                    this._task.setEvalOldValues(true);
                    oldVal = (this._field.isNull()
                        ? this._field.getMagicDefaultValue()
                        : await this._exp.evaluateWithResultTypeAndLength(fieldType, this._field.getSize()));
                    this._task.setEvalOldValues(false);
                    nOld = new NUM_TYPE(oldVal);
                    nResult = NUM_TYPE.sub(nResult, nOld);
                }
                if (this._task.getMode() !== Constants.TASK_MODE_DELETE) {
                    newVal = await this._exp.evaluateWithResultTypeAndLength(fieldType, this._field.getSize());
                    if (newVal != null) {
                        nNew = new NUM_TYPE(newVal);
                        nResult = NUM_TYPE.add(nResult, nNew);
                    }
                    else
                        nNew = nResult = null;
                }
                if (nResult != null)
                    result = nResult.toXMLrecord();
                else
                    result = this._field.getMagicDefaultValue();
            }
            else {
                await this.operServer(mprgCreator);
                return;
            }
        }
        else
            result = await this._exp.evaluateWithResultTypeAndLength(fieldType, this._field.getSize());
        if (fieldType === StorageAttribute.BLOB_VECTOR)
            if (result != null)
                result = Operation.operUpdateVectors(this._field, result);
        recompute = (this._field.getTask() === this._task);
        setRecordUpdated = (!recompute || !this._undo);
        await this._field.setValueAndStartRecompute(result, result == null, true, setRecordUpdated, false);
        if (!this._undo)
            this._field.setUpdated();
        this._field.setModified();
        await this._field.updateDisplay();
        flowMonitor.addFlowOperationUpdate(FlowMonitorInterface.FLWMTR_END, this._task.GetTaskDetails());
    }
    async CanModify(mprgCreator) {
        let execOperCommand = ((this._cmdToServer instanceof ExecOperCommand) ? this._cmdToServer : null);
        execOperCommand.Operation = this;
        execOperCommand.MprgCreator = mprgCreator;
        return (await this.Task.DataviewManager.Execute(this._cmdToServer)).Success;
    }
    static operUpdateVectors(field, vec) {
        let result = null;
        if (field.getType() === StorageAttribute.BLOB_VECTOR) {
            let cellsAttr = VectorType.getCellsAttr(vec);
            let cellsType = field.getCellsType();
            if (StorageAttributeCheck.isTheSameType(cellsAttr, cellsType)) {
                let newVec = new VectorType(vec);
                newVec.adjustToFit(field);
                return newVec.toString();
            }
            return result;
        }
        throw new ApplicationException("in operUpdateVectors " + field.getName() + " is not of type vector");
    }
    async operRaiseEvent(mprgCreator, returnedFromServer) {
        let immediate = await this.Immediate();
        let aRtEvt = this._rtEvt.replicate();
        await aRtEvt.setPublicName();
        aRtEvt.setImmediate(immediate);
        aRtEvt.setMainPrgCreator(null);
        if (immediate) {
            aRtEvt.setCtrl(this._task.getLastParkedCtrl());
            aRtEvt.setArgList(this._argList);
            if (aRtEvt.getTask().isMainProg())
                aRtEvt.setMainPrgCreator(mprgCreator);
            AccessHelper.eventsManager.checkAndSaveRouteEventIfCtxRemovedFromSrvr(aRtEvt);
            AccessHelper.eventsManager.pushExecStack(this._task.getTaskTag(), this._evtHandler.getId().toString(), this._serverId);
            await AccessHelper.eventsManager.handleEvent(aRtEvt, returnedFromServer);
            AccessHelper.eventsManager.popExecStack();
        }
        else {
            let arglist = new ArgumentsList();
            await arglist.Initialize(this._argList);
            aRtEvt.setArgList(arglist);
            aRtEvt.setTask(null);
            AccessHelper.eventsManager.addToTail(aRtEvt);
            AccessHelper.eventsManager.checkAndSaveRouteEventIfCtxRemovedFromSrvr(aRtEvt);
        }
    }
    async operCall(mprgCreator) {
        await this.operServer(mprgCreator);
    }
    async operServer(mprgCreator) {
        let command = ((this._cmdToServer instanceof ExecOperCommand) ? this._cmdToServer : null);
        Debug.Assert(command !== null);
        if (this._task.isMainProg())
            command.MprgCreator = mprgCreator;
        else
            command.MprgCreator = null;
        command.SetExecutionStack(AccessHelper.eventsManager.getExecStack());
        command.Operation = this;
        await AccessHelper.mgDataTable.execRequestWithSubformRecordCycle(this._task.getMGData().CmdsToServer, this._cmdToServer, null);
    }
    getBlockEnd() {
        return this._blockEnd;
    }
    setBlockEnd(val) {
        this._blockEnd = val;
    }
    getBlockClose() {
        return this._blockClose;
    }
    setBlockClose(val) {
        this._blockClose = val;
    }
    getTaskTag() {
        return this._task.getTaskTag();
    }
    getHandlerId() {
        return this._evtHandler.getId().toString();
    }
    getEventHandler() {
        return this._evtHandler;
    }
    getExecOnServer() {
        return this._execOnServer;
    }
    setExecOnServer(val) {
        this._execOnServer = val;
    }
    getExecOnClient() {
        return this._execOnClient;
    }
    async canExecute() {
        return await this.getCondVal();
    }
    async AddFlowDescription(buffer) {
        switch (this._type) {
            case ConstInterface.MG_OPER_VERIFY:
                buffer.Append("Verify: ");
                if (this._exp == null)
                    buffer.Append(this._text);
                else
                    buffer.Append("Exp #").Append(this._exp.getId());
                break;
            case ConstInterface.MG_OPER_BLOCK:
                buffer.Append("Block If");
                break;
            case ConstInterface.MG_OPER_LOOP:
                buffer.Append("Block Loop");
                break;
            case ConstInterface.MG_OPER_ELSE:
                buffer.Append("Block Else");
                break;
            case ConstInterface.MG_OPER_EVALUATE:
                buffer.Append("Evaluate Exp #").Append(this._exp.getId());
                break;
            case ConstInterface.MG_OPER_UPDATE:
                buffer.AppendFormat("Update {0} with Exp #{1}", this._field.getVarName(), this._exp.getId());
                break;
            case ConstInterface.MG_OPER_ENDBLOCK:
                buffer.Append("End Block");
                break;
            case ConstInterface.MG_OPER_USR_EXIT:
                buffer.Append("Invoke OS");
                break;
            case ConstInterface.MG_OPER_RAISE_EVENT:
                buffer.Append("Raise Event:");
                this._rtEvt.AppendDescription(buffer);
                buffer.AppendFormat(" (Wait={0})", (await this.Immediate() ? 'Y' : 'N'));
                break;
            case ConstInterface.MG_OPER_SERVER:
                buffer.Append("Run server-side operation");
                break;
            case ConstInterface.MG_OPER_CALL:
                buffer.Append("Call program");
                break;
            default:
                buffer.AppendFormat("<<Unknown Operation Code {0}>>", this._type);
                break;
        }
    }
    GetArgList() {
        return this._argList;
    }
    GetReturnValueField() {
        return this._field;
    }
    GetSubformControlName() {
        return this._subformCtrlName;
    }
    toString() {
        return NString.Format("Executing operation number {0}", this._serverId + 1);
    }
    GetRuntimeEvent() {
        return this._rtEvt;
    }
}
//# sourceMappingURL=data:application/json;base64,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