import { PropInterface, PropTable } from "@magic-xpa/gui";
import { FieldBase } from "../data/FieldBase";
import { List } from "@magic-xpa/mscorelib";
import { CommandsProcessorBase_SendingInstruction } from "../CommandsProcessorBase";
import { Constants, InternalInterface, Logger, XMLConstants, XmlParser } from "@magic-xpa/utils";
import { Record } from "../data/Record";
import { CommandFactory } from "../commands/ClientToServer/CommandFactory";
import { RemoteCommandsProcessor } from "../remote/RemoteCommandsProcessor";
import { DataviewHeaderBase } from "./DataviewHeaderBase";
import { FlowMonitorQueue } from "../util/FlowMonitorQueue";
import { ConstInterface } from "../ConstInterface";
import { Environment } from "../env/Environment";
import { AccessHelper } from "../AccessHelper";
export var Recompute_RcmpBy;
(function (Recompute_RcmpBy) {
    Recompute_RcmpBy["CLIENT"] = "C";
    Recompute_RcmpBy["SERVER_ON_CHANGE"] = "O";
})(Recompute_RcmpBy || (Recompute_RcmpBy = {}));
export class Recompute {
    constructor() {
        this._ctrlProps = null;
        this._formProps = null;
        this._isOwnerFld = true;
        this.OwnerFld = null;
        this._rcmpOrder = null;
        this._subForms = null;
        this._subFormsOnlyRecomp = false;
        this._hasServerLinksRecomputes = false;
        this.RcmpMode = null;
        this.Task = null;
    }
    fillData(dataView, taskRef, parser) {
        this.Task = taskRef;
        this._isOwnerFld = true;
        while (this.initInnerObjects(parser, parser.getNextTag(), dataView)) {
        }
    }
    initInnerObjects(parser, foundTagName, dataView) {
        switch (foundTagName) {
            case XMLConstants.MG_TAG_FLD:
                if (this._isOwnerFld) {
                    this.fillFldField(parser, dataView);
                    this._isOwnerFld = false;
                }
                else
                    this.fillFldField(parser, dataView);
                break;
            case ConstInterface.MG_TAG_LINK:
                this.fillLink(parser, dataView);
                break;
            case XMLConstants.MG_TAG_CONTROL:
                if (this._ctrlProps == null)
                    this._ctrlProps = new PropTable();
                this._ctrlProps.fillDataByExists(this.Task, parser);
                if (this._ctrlProps.getCtrlRef() != null && this._ctrlProps.getCtrlRef().IsRepeatable && this.OwnerFld.IsVirtual &&
                    !(this.OwnerFld.hasInitExp()))
                    this.OwnerFld.causeTableInvalidation(true);
                break;
            case XMLConstants.MG_TAG_FORM_PROPERTIES:
                if (this._formProps == null)
                    this._formProps = new PropTable();
                this._formProps.fillDataByExists(this.Task, parser);
                break;
            case ConstInterface.MG_TAG_FLD_END:
                parser.setCurrIndex2EndOfTag();
                return false;
            default:
                Logger.Instance.WriteExceptionToLogWithMsg("There is no such tag in Recompute. Insert else if to Recompute.initInnerObjects for " + foundTagName);
                return false;
        }
        return true;
    }
    AddRecomputeItem(item) {
        if (this._rcmpOrder === null)
            this._rcmpOrder = new List();
        this._rcmpOrder.push(item);
    }
    fillLink(parser, dataView) {
        let endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_TERM, parser.getCurrIndex());
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            let tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(ConstInterface.MG_TAG_LINK) + ConstInterface.MG_TAG_LINK.length);
            let tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            for (let j = 0; j < tokensVector.length; j += 2) {
                let attribute = (tokensVector.get_Item(j));
                let valueStr = (tokensVector.get_Item(j + 1));
                if (attribute === XMLConstants.MG_ATTR_ID)
                    this.AddRecomputeItem(dataView.getTask().getDataviewHeaders().getDataviewHeaderById(XmlParser.getInt(valueStr)));
                else
                    Logger.Instance.WriteExceptionToLogWithMsg("There is no such tag in Recompute class. Insert case to Recompute.fillLink for " + attribute);
            }
            parser.setCurrIndex(endContext + XMLConstants.TAG_TERM.length);
            return;
        }
        else
            Logger.Instance.WriteExceptionToLogWithMsg("in Recompute.fillLink() out of bounds");
    }
    fillFldField(parser, dataView) {
        let endContext = parser.getXMLdata().indexOf(this._isOwnerFld
            ? XMLConstants.TAG_CLOSE
            : XMLConstants.TAG_TERM, parser.getCurrIndex());
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            let tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(XMLConstants.MG_TAG_FLD) + XMLConstants.MG_TAG_FLD.length);
            let tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            this.initElements(tokensVector, dataView);
            if (this._isOwnerFld) {
                parser.setCurrIndex(++endContext);
                return;
            }
            parser.setCurrIndex(endContext + XMLConstants.TAG_TERM.length);
            return;
        }
        Logger.Instance.WriteExceptionToLogWithMsg("in Command.FillData() out of bounds");
    }
    initElements(tokensVector, dataView) {
        for (let j = 0; j < tokensVector.length; j += 2) {
            let attribute = (tokensVector.get_Item(j));
            let valueStr = (tokensVector.get_Item(j + 1));
            switch (attribute) {
                case XMLConstants.MG_ATTR_ID: {
                    let fldId = XmlParser.getInt(valueStr);
                    let fld = dataView.getField(fldId);
                    if (this._isOwnerFld) {
                        this.OwnerFld = fld;
                        this.OwnerFld.setRecompute(this);
                    }
                    else {
                        fld.setRecomputed();
                        this.AddRecomputeItem(fld);
                    }
                    break;
                }
                case ConstInterface.MG_ATTR_RECOMPUTEBY:
                    this.RcmpMode = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_SUB_FORM_RCMP:
                    this._subFormsOnlyRecomp = true;
                    break;
                case ConstInterface.MG_ATTR_HAS_LINK_RECOMPUTES:
                    this._hasServerLinksRecomputes = true;
                    break;
                case XMLConstants.MG_ATTR_NAME:
                    break;
                default:
                    Logger.Instance.WriteExceptionToLogWithMsg("There is no such tag in Recompute class. Insert case to Recompute.initElements for " + attribute);
                    break;
            }
        }
    }
    async execute(rec) {
        let i;
        let fld;
        let cmdsToServer = this.Task.getMGData().CmdsToServer;
        let cmd;
        try {
            rec.setInRecompute(true);
            this.Task.SetInRecompute(true);
            this.SetSubformsInRecompute(true);
            let allowServerRecompute = this._hasServerLinksRecomputes ||
                (this.Task.getForm().AllowedSubformRecompute && await this.checkRefreshSubForms());
            if (this.RcmpMode !== Recompute_RcmpBy.CLIENT && allowServerRecompute) {
                let inClient = this._subFormsOnlyRecomp;
                this.Task.ExecuteClientSubformRefresh = false;
                if (inClient) {
                    inClient = await this.Task.prepareCache(true);
                    if (inClient)
                        inClient = await this.Task.testAndSet(true);
                }
                if (!inClient) {
                    this.Task.DataView.GetFieldsTab().setServerRcmp(true);
                    cmd = CommandFactory.CreateRecomputeCommand(this.Task.getTaskTag(), this.OwnerFld.getId(), !this.Task.getForm().AllowedSubformRecompute);
                    cmdsToServer.Add(cmd);
                    await RemoteCommandsProcessor.GetInstance().Execute(CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS);
                }
                if (this.Task.ExecuteClientSubformRefresh)
                    await this.RefreshSubforms();
                else {
                    if (await this.recPrefixSubForms())
                        await this.recSuffixSubForms();
                    this.Task.CleanDoSubformPrefixSuffix();
                }
            }
            else {
                try {
                    FlowMonitorQueue.Instance.addRecompute(this.OwnerFld.getVarName(), this.Task.GetTaskDetails());
                    if (this._formProps != null)
                        await this._formProps.RefreshDisplay(false, false);
                    if (this._ctrlProps != null)
                        await this._ctrlProps.RefreshDisplay(false, false);
                    if (this._rcmpOrder != null) {
                        for (i = 0; i < this._rcmpOrder.length; i++) {
                            if (this._rcmpOrder.get_Item(i) instanceof FieldBase) {
                                fld = this._rcmpOrder.get_Item(i);
                                await this.fldRcmp(fld, true);
                            }
                            else if (this._rcmpOrder.get_Item(i) instanceof DataviewHeaderBase) {
                                let curLnk = this._rcmpOrder.get_Item(i);
                                await curLnk.getLinkedRecord(rec);
                                let linkFields = this.Task.DataView.GetFieldsTab().getLinkFields(curLnk.Id);
                                rec.setInCompute(true);
                                let saveInForceUpdate = rec.InForceUpdate;
                                rec.InForceUpdate = false;
                                for (let j = 0; j < linkFields.length; j++) {
                                    await this.fldRcmp(linkFields.get_Item(j), false);
                                    rec.clearFlag((linkFields.get_Item(j)).getId(), Record.FLAG_UPDATED);
                                    rec.clearFlag((linkFields.get_Item(j)).getId(), Record.FLAG_MODIFIED);
                                    rec.clearFlag((linkFields.get_Item(j)).getId(), Record.FLAG_CRSR_MODIFIED);
                                    rec.clearHistoryFlag((linkFields.get_Item(j)).getId());
                                }
                                rec.InForceUpdate = saveInForceUpdate;
                                let retFld = curLnk.ReturnField;
                                if (retFld != null)
                                    await this.fldRcmp(retFld, false);
                                rec.setInCompute(false);
                                rec.setForceSaveOrg(true);
                            }
                        }
                    }
                    await this.RefreshSubforms();
                }
                catch (e) {
                    Logger.Instance.WriteExceptionToLogWithMsg("in Recompute.execute(): " + e.Message);
                }
            }
            this.Task.SetInRecompute(false);
            this.SetSubformsInRecompute(false);
        }
        finally {
            rec.buildLinksPosStr();
            rec.setInRecompute(false);
        }
    }
    async RefreshSubforms() {
        if (this.Task.getForm().AllowedSubformRecompute) {
            let subformsToRefresh = await this.GetSubformsToRefresh();
            for (let i = 0; i < subformsToRefresh.length; i++) {
                let subformTask = subformsToRefresh.get_Item(i);
                await this.Task.SubformRefresh(subformTask, true);
            }
        }
    }
    async SetSubformsInRecompute(inRecompute) {
        let subformsToRefresh = await this.GetSubformsToRefresh();
        for (let i = 0; i < subformsToRefresh.length; i++) {
            let subformTask = subformsToRefresh.get_Item(i);
            subformTask.SetInRecompute(inRecompute);
        }
    }
    async fldRcmp(fld, computeField) {
        if (fld.IsVirtual
            || ((this.Task.getMode() !== Constants.TASK_MODE_QUERY ||
                Environment.Instance.allowUpdateInQueryMode(this.Task.getCompIdx())))) {
            if (!fld.isInEvalProcess()) {
                if (computeField)
                    await fld.compute(true);
                else
                    await fld.setValueAndStartRecompute(fld.getValue(false), fld.isNull(), true, false, false);
                await fld.updateDisplay();
            }
        }
    }
    buildSubFormList() {
        if (this._subForms === null) {
            this._subForms = new List();
            let subTasksTab = this.Task.getSubTasks();
            if (subTasksTab !== null) {
                for (let i = 0; i < subTasksTab.getSize(); i = i + 1) {
                    let subForm = subTasksTab.getTask(i);
                    if (subForm.getForm().getSubFormCtrl() !== null && subForm.refreshesOn(this.OwnerFld.getId())) {
                        this._subForms.push(subForm);
                    }
                }
            }
        }
    }
    RemoveSubform(subformTask) {
        this._subForms.Remove(subformTask);
    }
    AddSubform(subformTask) {
        if (this._subForms === null)
            this._subForms = new List();
        if (!this._subForms.Contains(subformTask))
            this._subForms.push(subformTask);
    }
    async recSuffixSubForms() {
        let i;
        let subForm;
        let successful = true;
        let subformCtrl;
        this.buildSubFormList();
        for (i = 0; successful && i < this._subForms.length; i++) {
            subForm = this._subForms.get_Item(i);
            subformCtrl = subForm.getForm().getSubFormCtrl();
            if (subForm.isStarted() && !subformCtrl.RefreshOnVisible &&
                await subformCtrl.checkProp(PropInterface.PROP_TYPE_AUTO_REFRESH, true) && !subForm.InSelect
                && subForm.DoSubformPrefixSuffix) {
                await AccessHelper.eventsManager.handleInternalEventWithTask(subForm, InternalInterface.MG_ACT_REC_SUFFIX);
                successful = !AccessHelper.eventsManager.GetStopExecutionFlag();
            }
        }
        this.Task.DataView.setPrevCurrRec();
        return (successful);
    }
    async checkRefreshSubForms() {
        let refresh = !this._subFormsOnlyRecomp;
        let subformsToRefresh = await this.GetSubformsToRefresh();
        if (subformsToRefresh.length > 0)
            refresh = true;
        return refresh;
    }
    async GetSubformsToRefresh() {
        let subTasks = new List();
        let subformCtrl;
        this.buildSubFormList();
        for (let i = 0; i < this._subForms.length; i++) {
            let subTask = this._subForms[i];
            subformCtrl = subTask.getForm().getSubFormCtrl();
            if (subTask.isStarted() && await subformCtrl.checkProp(PropInterface.PROP_TYPE_AUTO_REFRESH, true) &&
                !subTask.InSelect && !subTask.InEndTask) {
                await subformCtrl.checkProp(PropInterface.PROP_TYPE_VISIBLE, true);
                if (!subformCtrl.isVisible() && !await subformCtrl.checkProp(PropInterface.PROP_TYPE_REFRESH_WHEN_HIDDEN, false))
                    subformCtrl.RefreshOnVisible = true;
                else
                    subTasks.push(subTask);
            }
        }
        return subTasks;
    }
    async recPrefixSubForms() {
        let i;
        let subForm;
        let successful = true;
        let subformCtrl;
        this.buildSubFormList();
        for (i = 0; successful && i < this._subForms.length; i++) {
            subForm = this._subForms.get_Item(i);
            subformCtrl = subForm.getForm().getSubFormCtrl();
            if (subForm.isStarted() && !subformCtrl.RefreshOnVisible
                && await subformCtrl.checkProp(PropInterface.PROP_TYPE_AUTO_REFRESH, true)
                && !subForm.InSelect && subForm.DoSubformPrefixSuffix) {
                await AccessHelper.eventsManager.handleInternalEventWithTaskAndSubformRefresh(subForm, InternalInterface.MG_ACT_REC_PREFIX, true);
                successful = !AccessHelper.eventsManager.GetStopExecutionFlag();
            }
        }
        return successful;
    }
    notifyServerOnChange() {
        return this.RcmpMode === Recompute_RcmpBy.SERVER_ON_CHANGE;
    }
    isServerRcmp() {
        return this.RcmpMode !== Recompute_RcmpBy.CLIENT;
    }
}
//# sourceMappingURL=data:application/json;base64,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