import { List, NString } from "@magic-xpa/mscorelib";
import { Key } from "../data/Key";
import { FieldsTableExt as FieldsTable } from "../data/FieldsTable";
import { Logger, XMLConstants, XmlParser } from "@magic-xpa/utils";
import { RuntimeContextBase } from "@magic-xpa/gui";
import { Record } from "../data/Record";
import { HeapSort } from "../util/HeapSort";
import { ConstInterface } from "../ConstInterface";
import { TableCacheManager } from "./TableCacheManager";
export class TableCache {
    constructor(tableId) {
        this._keys = null;
        this._records = null;
        this._currSortDir = '\0';
        this._currSortKey = 0;
        this.FldsTab = null;
        this._isLoaded = false;
        this._tableIdent = null;
        this._tableUId = null;
        this._tableUId = tableId;
        this._keys = new List();
        this._records = new List();
        this._isLoaded = false;
    }
    SetTableIdent(ident) {
        if (this._tableIdent === null)
            this._tableIdent = ident;
        else if (this._tableIdent !== ident) {
            Logger.Instance.WriteExceptionToLogWithMsg("in TableCache.setTableIdent() already set and table identifier  does not match");
        }
    }
    FillData() {
        let parser = RuntimeContextBase.Instance.Parser;
        this.FillAttributes(parser);
        while (this.InitInnerObjects(parser, parser.getNextTag())) {
        }
    }
    FillAttributes(parser) {
        let endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, parser.getCurrIndex());
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            let tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(ConstInterface.MG_TAG_CACHED_TABLE) + ConstInterface.MG_TAG_CACHED_TABLE.length);
            let tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            for (let j = 0; j < tokensVector.length; j += 2) {
                let attribute = (tokensVector.get_Item(j));
                let valueStr = (tokensVector.get_Item(j + 1));
                switch (attribute) {
                    case XMLConstants.MG_ATTR_ID:
                        if (this._tableUId.indexOf(valueStr) === -1) {
                            this._tableUId = valueStr;
                            Logger.Instance.WriteExceptionToLogWithMsg("in TableCache.fillAttributes() table unique id does not match");
                        }
                        break;
                    case XMLConstants.MG_ATTR_NAME:
                        break;
                    case ConstInterface.MG_ATTR_IDENT:
                        if (this._tableIdent !== valueStr) {
                            this._tableIdent = valueStr;
                            Logger.Instance.WriteExceptionToLogWithMsg("in TableCache.fillAttributes() table identifier id does not match");
                        }
                        break;
                    default:
                        Logger.Instance.WriteExceptionToLogWithMsg(NString.Format("Unrecognized attribute: '{0}'", attribute));
                        break;
                }
            }
            parser.setCurrIndex(++endContext);
            return;
        }
        Logger.Instance.WriteExceptionToLogWithMsg("in TableCache.fillAttributes() out of string bounds");
    }
    InitInnerObjects(parser, foundTagName) {
        if (foundTagName == null)
            return false;
        switch (foundTagName) {
            case XMLConstants.MG_TAG_FLDH:
                this.FldsTab = new FieldsTable();
                this.FldsTab.fillData(parser);
                break;
            case ConstInterface.MG_ATTR_KEY: {
                let current = new Key(this);
                current.FillData(parser);
                this._keys.push(current);
                break;
            }
            case ConstInterface.MG_TAG_RECORDS:
                parser.setCurrIndex(parser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, parser.getCurrIndex()) + 1);
                break;
            case ConstInterface.MG_TAG_RECORDS_END:
                parser.setCurrIndex2EndOfTag();
                break;
            case ConstInterface.MG_TAG_CACHED_TABLE_END:
                parser.setCurrIndex2EndOfTag();
                return false;
            case ConstInterface.MG_TAG_REC: {
                let current = new Record(this);
                current.Initialize();
                current.fillData(parser);
                this._records.push(current);
                break;
            }
        }
        return true;
    }
    GetTableUniqueId() {
        return this._tableUId;
    }
    GetTableCommonIdentifier() {
        return this._tableIdent;
    }
    GetCurrSortKey() {
        return this._currSortKey;
    }
    GetKeyById(keyId) {
        for (let i = 0; i < this._keys.length; i = i + 1) {
            if (this._keys.get_Item(i).GetKeyId() === keyId)
                return this._keys.get_Item(i);
        }
        return null;
    }
    async Load() {
        if (!this._isLoaded) {
            await TableCacheManager.Instance.LoadTable(this._tableUId);
            this._isLoaded = true;
        }
    }
    async SortTable(sortKeyId, sortDir) {
        if (!this._isLoaded)
            await this.Load();
        if (sortKeyId !== this._currSortKey || sortDir !== this._currSortDir) {
            this._currSortDir = sortDir;
            this._currSortKey = sortKeyId;
            let recordsArr = this._records.ToArray();
            HeapSort.sort(recordsArr);
            if (this._currSortDir === 'D')
                this.ReverseVector(recordsArr);
            for (let i = 0; i < this._records.length; i = i + 1) {
                this._records.set_Item(i, recordsArr[i]);
            }
        }
    }
    Fetch(loc) {
        let checkLoc = loc.length > 0;
        if (this._isLoaded) {
            if (!checkLoc) {
                return ((this._records.length !== 0) ? this._records.get_Item(0) : null);
            }
            for (let i = 0; i < this._records.length; i = i + 1) {
                let currRec = this._records.get_Item(i);
                if (TableCache.validateRec(currRec, loc)) {
                    return currRec;
                }
            }
        }
        return null;
    }
    static validateRec(currRec, rangeCond) {
        let result = true;
        for (let i = 0; i < rangeCond.length; i = i + 1) {
            let currCnd = rangeCond.get_Item(i);
            if (!currCnd.checkRange(currRec.GetFieldValue(currCnd.getCacheTableFldId()), currRec.IsNull(currCnd.getCacheTableFldId()))) {
                result = false;
                break;
            }
        }
        return result;
    }
    ReverseVector(array) {
        let right = array.length - 1;
        for (let left = 0; left < Math.floor((array.length - 1) / 2); left++, right--) {
            let tmp = array[left];
            array[left] = array[right];
            array[right] = tmp;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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