import { ApplicationException, Debug, List } from "@magic-xpa/mscorelib";
import { InternalInterface, XMLConstants } from "@magic-xpa/utils";
import { Commands } from "@magic-xpa/gui";
import { FlowMonitorQueue } from "../util/FlowMonitorQueue";
import { LastFocusedManager } from "../rt/LastFocusedManager";
import { CommandsProcessorBase_SendingInstruction, CommandsProcessorBase_SessionStage } from "../CommandsProcessorBase";
import { CommandsProcessorManager } from "../CommandsProcessorManager";
import { AccessHelper } from "../AccessHelper";
import { ConstInterface } from "../ConstInterface";
export class MGDataCollection {
    constructor() {
        this._mgDataTab = new List();
        this._iteratorMgdIdx = 0;
        this._iteratorTaskIdx = 0;
        this.currMgdID = 0;
        this.StartupMgData = null;
        this._lastRouteSentToServer = "";
    }
    GetTaskByID(id) {
        let task = null;
        for (let i = 0; i < this.getSize() && task == null; i++) {
            let mgd = this.getMGData(i);
            if (mgd == null)
                continue;
            task = mgd.getTask(id);
        }
        return task;
    }
    static get Instance() {
        if (MGDataCollection._instance == null) {
            if (MGDataCollection._instance == null)
                MGDataCollection._instance = new MGDataCollection();
        }
        return MGDataCollection._instance;
    }
    async addMGData(mgd, idx, isStartup) {
        if (idx < 0 || idx > this.getSize())
            throw new ApplicationException("Illegal MGData requested");
        if (isStartup)
            this.StartupMgData = mgd;
        if (idx === this.getSize())
            this._mgDataTab.push(mgd);
        else {
            let oldMgd = this.getMGData(idx);
            if (oldMgd != null && !oldMgd.IsAborting) {
                await oldMgd.getFirstTask().stop();
                oldMgd.abort();
            }
            this._mgDataTab.set_Item(idx, mgd);
        }
    }
    getMGData(idx) {
        let mgd = null;
        if (idx >= 0 && idx < this.getSize()) {
            mgd = this._mgDataTab.get_Item(idx);
        }
        return mgd;
    }
    getAvailableIdx() {
        let i;
        for (i = 0; i < this._mgDataTab.length; i = i + 1) {
            if (this._mgDataTab.get_Item(i) === null)
                break;
        }
        return i;
    }
    getMgDataIdx(mgd) {
        return this._mgDataTab.indexOf(mgd);
    }
    deleteMGDataTree(index) {
        let mgd, childMgd;
        let i;
        if (index < 0 || index >= this.getSize())
            throw new ApplicationException("in deleteMGData() illegal index: " + index);
        mgd = this.getMGData(index);
        if (mgd != null) {
            if (index > 0 && mgd.getParentMGdata() != null) {
                this._mgDataTab.set_Item(index, null);
            }
            LastFocusedManager.Instance.clean(index);
            for (i = 0; i < this.getSize(); i++) {
                childMgd = this.getMGData(i);
                if (childMgd != null && childMgd.getParentMGdata() === mgd)
                    this.deleteMGDataTree(i);
            }
        }
    }
    getCurrMGData() {
        return this.getMGData(this.currMgdID);
    }
    GetMainProgByCtlIdx(contextIDOrCtlIdx, ctlIdx) {
        if (arguments.length === 2)
            return this.GetMainProgByCtlIdx_0(ctlIdx);
        else
            return this.GetMainProgByCtlIdx_1(contextIDOrCtlIdx);
    }
    GetMainProgByCtlIdx_0(ctlIdx) {
        let task = null;
        for (let i = 0; i < this.getSize() && task == null; i++) {
            let mgd = this.getMGData(i);
            if (mgd == null)
                continue;
            task = mgd.getMainProg(ctlIdx);
        }
        return task;
    }
    GetMainProgByCtlIdx_1(ctlIdx) {
        return this.GetMainProgByCtlIdx(-1, ctlIdx);
    }
    startTasksIteration() {
        this._iteratorMgdIdx = 0;
        this._iteratorTaskIdx = 0;
    }
    getNextTask() {
        let task;
        let mgd = this.getMGData(this._iteratorMgdIdx);
        if (mgd == null)
            return null;
        task = mgd.getTask(this._iteratorTaskIdx);
        if (task == null) {
            do {
                this._iteratorMgdIdx++;
            } while (this._iteratorMgdIdx < this.getSize() && this.getMGData(this._iteratorMgdIdx) == null);
            this._iteratorTaskIdx = 0;
            return this.getNextTask();
        }
        this._iteratorTaskIdx++;
        return task;
    }
    async buildXML(message, serializeTasks) {
        let lastRoute = Commands.getLastRoute();
        if (lastRoute !== this._lastRouteSentToServer) {
            message.Append(XMLConstants.START_TAG + ConstInterface.MG_TAG_LAST_ROUTE);
            message.Append(" " + ConstInterface.MG_ATTR_ENV_VALUE + "=\"" + lastRoute + "\"");
            message.Append(XMLConstants.TAG_TERM);
            this._lastRouteSentToServer = lastRoute;
        }
        for (let i = 0; i < this.getSize(); i = i + 1) {
            let mgd = this.getMGData(i);
            if (mgd !== null && !mgd.IsAborting) {
                await mgd.buildXML(message, serializeTasks);
            }
        }
        FlowMonitorQueue.Instance.buildXML(message);
    }
    getSize() {
        return this._mgDataTab.length;
    }
    async processRecovery() {
        for (let i = 0; i < this.getSize(); i = i + 1) {
            let mgd = this.getMGData(i);
            if (mgd !== null) {
                for (let j = 0; j < mgd.getTasksCount(); j = j + 1) {
                    let task = mgd.getTask(j);
                    if (task !== null)
                        await task.DataView.processRecovery();
                }
            }
        }
    }
    getTriggeredTasks(triggeringTask) {
        let list = new List();
        let task;
        let tag = triggeringTask.getTaskTag();
        this.startTasksIteration();
        while ((task = this.getNextTask()) !== null) {
            if (tag === task.PreviouslyActiveTaskId)
                list.push(task);
        }
        return list;
    }
    GetTasks(p) {
        let taskList = new List();
        this._mgDataTab.forEach(mgd => {
            if (mgd !== null && !mgd.IsAborting) {
                for (let i = 0; i < mgd.getTasksCount(); i = i + 1) {
                    let task = mgd.getTask(i);
                    if (task !== null && p(task)) {
                        taskList.push(task);
                    }
                }
            }
        });
        return taskList;
    }
    getTaskIdById(taskId) {
        let task = this.GetTaskByID(taskId);
        let tag = taskId;
        if (task != null)
            tag = task.getTaskTag();
        return tag;
    }
    GetTopMostForms() {
        let task;
        let forms = new List();
        this.startTasksIteration();
        while ((task = this.getNextTask()) != null) {
            let form = task.getForm();
            if (!task.isAborting() && form != null && !form.isSubForm())
                forms.push(form);
        }
        return forms;
    }
    GetMGDataForStartupProgram() {
        Debug.Assert(this._mgDataTab.length > 0, "The main program must be processed before invoking this method.");
        let mainProgByCtlIdx = this.GetMainProgByCtlIdx(0);
        Debug.Assert(mainProgByCtlIdx !== null, "The main program must be processed before invoking this method.");
        Debug.Assert(mainProgByCtlIdx.getMgdID() === 0, "Main program is expected to be on MGData 0. Is this an error?");
        return this._mgDataTab.get_Item(0);
    }
    async execRequestWithSubformRecordCycle(cmdsToServer, cmdToServer, exp) {
        this.cleanDoSubformPrefixSuffix();
        cmdsToServer.Add(cmdToServer);
        if (!(MGDataCollection.Instance.StartupMgData.getFirstTask().InEndTask && MGDataCollection.Instance.StartupMgData.getFirstTask().ClosingFormUI))
            await CommandsProcessorManager.GetCommandsProcessor().Execute(CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS, CommandsProcessorBase_SessionStage.NORMAL, exp);
        await this.execAllSubformRecordCycle();
    }
    cleanDoSubformPrefixSuffix() {
        for (let i = 0; i < this.getSize(); i++) {
            let mgd = this.getMGData(i);
            if (mgd === null || mgd.IsAborting)
                continue;
            let tasksCount = mgd.getTasksCount();
            for (let j = 0; j < tasksCount; j++) {
                let task = mgd.getTask(j);
                task.DoSubformPrefixSuffix = false;
            }
        }
    }
    async execAllSubformRecordCycle() {
        for (let i = 0; i < this.getSize(); i = i + 1) {
            let mgd = this.getMGData(i);
            if (mgd === null || mgd.IsAborting)
                continue;
            let tasksCount = mgd.getTasksCount();
            for (let j = 0; j < tasksCount; j = j + 1) {
                let task = mgd.getTask(j);
                if (task.IsSubForm && task.DoSubformPrefixSuffix) {
                    task.PerformParentRecordPrefix = false;
                    AccessHelper.eventsManager.pushNewExecStacks();
                    await AccessHelper.eventsManager.handleInternalEventWithTaskAndSubformRefresh(task, InternalInterface.MG_ACT_REC_PREFIX, true);
                    if (!AccessHelper.eventsManager.GetStopExecutionFlag())
                        await AccessHelper.eventsManager.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
                    AccessHelper.eventsManager.popNewExecStacks();
                    task.PerformParentRecordPrefix = true;
                }
            }
        }
    }
}
MGDataCollection._instance = null;
//# sourceMappingURL=data:application/json;base64,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