import { Debug, HashUtils, List, NNumber, NString, RefParam, Stack, StringBuilder, isNullOrUndefined } from '@magic-xpa/mscorelib';
import { Constants, InternalInterface, Logger, Logger_LogLevels, StorageAttribute, StrUtil, WindowType, XMLConstants, XmlParser } from '@magic-xpa/utils';
import { Commands, DataModificationTypes, DisplayConvertor, Events, GuiDataCollection, Helps, Manager, NUM_TYPE, PIC, PropInterface, RecordUtils, RuntimeContextBase, Styles, TaskDefinitionId } from '@magic-xpa/gui';
import { DataView } from '../data/DataView';
import { DvCache } from '../data/DvCache';
import { DataViewCommandType } from '../commands/ClientToServer/DataviewCommand';
import { CommandsProcessorBase_SendingInstruction } from '../CommandsProcessorBase';
import { TaskServiceBase } from './TaskServiceBase';
import { TaskTransactionManager } from '../data/TaskTransactionManager';
import { DataviewManager } from '../data/DataviewManager';
import { SortCollection } from './sort/SortCollection';
import { MgForm } from '../gui/MgForm';
import { RemoteTaskService } from './RemoteTaskService';
import { ReturnResult } from '../util/ReturnResult';
import { RunTimeEvent } from '../event/RunTimeEvent';
import { CommandFactory } from '../commands/ClientToServer/CommandFactory';
import { DataviewHeaders } from '../rt/DataviewHeaders';
import { Transaction } from '../rt/Transaction';
import { TasksTable } from './TasksTable';
import { HandlersTable } from '../rt/HandlersTable';
import { RecomputeTable } from '../rt/RecomputeTable';
import { ExpTable } from '../exp/ExpTable';
import { UserEventsTable } from '../event/UserEventsTable';
import { FormsTable } from '../gui/FormsTable';
import { FlowMonitorQueue } from '../util/FlowMonitorQueue';
import { ActionManager } from '../event/ActionManager';
import { ConstInterface } from '../ConstInterface';
import { GUIManager } from '../GUIManager';
import { CommandsProcessorManager } from '../CommandsProcessorManager';
import { EventSubType } from '../enums';
import { OpeningTaskDetails, Task_SubformExecModeEnum, TaskBase } from './TaskBase';
import { LanguageData } from '../env/LanguageData';
import { Environment } from '../env/Environment';
import { LastFocusedManager } from '../rt/LastFocusedManager';
import { GlobalCommandsManager } from '../GlobalCommandsManager';
import { EventsAllowedType } from '../event/IEventsManager';
import { AccessHelper } from '../AccessHelper';
import { MGDataCollection } from './MGDataCollection';
import { RemoteCommandsProcessor } from "../remote/RemoteCommandsProcessor";
import { EventHandlerPosition } from '../event/EventHandlerPosition';
export var Task_Flow;
(function (Task_Flow) {
    Task_Flow[Task_Flow["NONE"] = 0] = "NONE";
    Task_Flow[Task_Flow["STEP"] = 1] = "STEP";
    Task_Flow[Task_Flow["FAST"] = 2] = "FAST";
})(Task_Flow || (Task_Flow = {}));
export var Task_Direction;
(function (Task_Direction) {
    Task_Direction[Task_Direction["BACK"] = -1] = "BACK";
    Task_Direction[Task_Direction["NONE"] = 0] = "NONE";
    Task_Direction[Task_Direction["FORE"] = 1] = "FORE";
})(Task_Direction || (Task_Direction = {}));
export class Task extends TaskBase {
    get ParentTask() {
        return this._parentTask;
    }
    set TryingToCommit(value) {
        this._tryingToCommit = value;
    }
    get TryingToCommit() {
        return this._tryingToCommit;
    }
    get TaskService() {
        if (this._taskService == null)
            this._taskService = new RemoteTaskService();
        return this._taskService;
    }
    static get CommandsProcessor() {
        return CommandsProcessorManager.GetCommandsProcessor();
    }
    get LogicalStudioParentTask() {
        if (this.StudioParentTask === null && !super.isMainProg()) {
            return GuiDataCollection.MGDataTable.GetMainProgByCtlIdx(this.ContextID, this._ctlIdx);
        }
        return this.StudioParentTask;
    }
    get KnownToServer() {
        return this._knownToServer;
    }
    set Transaction(value) {
        this.DataviewManager.CurrentDataviewManager.Transaction = value;
    }
    get Transaction() {
        return this.DataviewManager.CurrentDataviewManager.Transaction;
    }
    get TaskDefinitionId() {
        if (this.taskDefinitionId === null) {
            this.taskDefinitionId = new TaskDefinitionId(this._ctlIdx, this.ProgramIsn, this.TaskIsn, this._isPrg);
        }
        return this.taskDefinitionId;
    }
    static get IsBlockingBatch() {
        return false;
    }
    constructor(parent) {
        super();
        this._dvCache = null;
        this._mgData = null;
        this._aborting = false;
        this._bExecEndTask = false;
        this._isStarted = false;
        this._evalOldValues = false;
        this._inRecordSuffix = false;
        this._cancelWasRaised = false;
        this._counter = 0;
        this._currStartProgLevel = 0;
        this._destinationSubform = false;
        this._enableZoomHandler = false;
        this._firstRecordCycle = false;
        this._originalTaskMode = ' ';
        this._inCreateLine = false;
        this.InSelect = false;
        this.InStartProcess = false;
        this._isDestinationCall = false;
        this._knownToServer = true;
        this._loopStack = null;
        this.DelayCommandExecution = false;
        this._parentTask = null;
        this.ContextTask = null;
        this._triggeringTask = null;
        this.PathParentTask = null;
        this._preventControlChange = false;
        this._preventRecordSuffix = false;
        this._taskPath = null;
        this._tryingToCommit = false;
        this._useLoopStack = false;
        this._inProcessingTopMostEndTaskSaved = false;
        this.InHandleActCancel = false;
        this.RuntimeSorts = null;
        this.VewFirst = 0;
        this.dataViewContent = null;
        this.PerformParentRecordPrefix = false;
        this.ModeAsParent = false;
        this.TaskTransactionManager = null;
        this.ExecuteClientSubformRefresh = false;
        this.InCommonHandlerBeforeRP = false;
        this._taskService = null;
        this.hasLocate = false;
        this.taskDefinitionId = null;
        this.ArgumentsList = null;
        this.ConfirmUpdateNo = false;
        this.DataSynced = false;
        this.UniqueSort = null;
        this.RefreshOnVars = null;
        this.DvPosDescriptor = null;
        this.SubTasks = null;
        this.ExpTab = null;
        this.UserEvtTab = null;
        this.DataviewHeadersTable = null;
        this.HandlersTab = null;
        this._forms = null;
        this.Name = null;
        this.PublicName = null;
        this.TaskSuffixExecuted = false;
        this.RetrunValueExp = 0;
        this.SubformControl = null;
        if (isNullOrUndefined(parent)) {
            this.constructor_0();
            return;
        }
        this.constructor_1(parent);
    }
    constructor_0() {
        this.ActionManager = new ActionManager();
        this.DataView = new DataView(this);
        this._flowMonitor = FlowMonitorQueue.Instance;
        this.ContextTask = this;
        this._dvCache = new DvCache(this);
        this.PerformParentRecordPrefix = true;
        this.DataviewManager = new DataviewManager(this);
        this.allowRemoteCmdExecution = true;
        this.TaskTransactionManager = new TaskTransactionManager(this);
    }
    Init() {
        this.DataView.Init();
    }
    constructor_1(parent) {
        this.constructor_0();
        this.PathParentTask = parent;
        this._parentTask = parent;
        this._triggeringTask = null;
        this.buildTaskPath();
    }
    static CreateTask() {
        let task = new Task();
        task.Init();
        return task;
    }
    isAborting() {
        return this._aborting;
    }
    buildTaskPath() {
        this._taskPath = new List();
        if (this._parentTask._taskPath !== null)
            this._taskPath.AddRange(this._parentTask._taskPath.GetEnumerator());
        this._taskPath.push(this);
    }
    async fillData(mgd, openingTaskDetails) {
        if (!super.isMainProg()) {
            if (this._parentTask === null && openingTaskDetails.CallingTask !== null) {
                this._parentTask = openingTaskDetails.CallingTask;
                this.buildTaskPath();
                openingTaskDetails.CallingTask.addSubTask(this);
            }
            if (this.PathParentTask === null && openingTaskDetails.PathParentTask !== null) {
                this.PathParentTask = openingTaskDetails.PathParentTask;
            }
        }
        if (this._parentTask !== null && this._parentTask.getEnableZoomHandler()) {
            this.setEnableZoomHandler();
        }
        this._mgData = mgd;
        this.fillAttributes();
        this._mgData.addTask(this);
        let parentForm = (this._parentTask !== null) ? this._parentTask.getForm() : null;
        while (await this.initInnerObjects(RuntimeContextBase.Instance.Parser.getNextTag(), parentForm)) {
        }
        this.TaskTransactionManager.PrepareTransactionProperties(this.DataviewManager.RemoteDataviewManager.Transaction, false);
        this.HandleTriggerTask();
        this.HandlePreviouslyActiveTaskId();
        this.CheckAndSetSubForm();
        if (this.IsSubForm) {
            let subFormCtrl = this.getParent().getForm().getSubFormCtrlForTask(this._taskTag);
            if (subFormCtrl !== null) {
                subFormCtrl.setSubformTaskId(this._taskTag);
            }
            if (this.Form !== null) {
                this.Form.setSubFormCtrl(subFormCtrl);
            }
        }
    }
    HandlePreviouslyActiveTaskId() {
        if (this._parentTask !== null && this._parentTask.GetContextTask() !== null && this._mgData !== this._parentTask.getMGData())
            this.PreviouslyActiveTaskId = (this._parentTask.GetContextTask()).getTaskTag();
        else if (this._parentTask !== null)
            this.PreviouslyActiveTaskId = this._parentTask.getTaskTag();
    }
    HandleTriggerTask() {
        this._triggeringTask = this._parentTask;
    }
    FillTaskTables(taskTablesData) {
        if (arguments.length === 1) {
            this.FillTaskTables_0();
            return;
        }
        this.FillTaskTables_1();
    }
    FillTaskTables_0() {
    }
    fillAttributes() {
        let parser = RuntimeContextBase.Instance.Parser;
        let endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, parser.getCurrIndex());
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            super.fillAttributes(parser);
            if (super.isMainProg())
                this._mgData = AccessHelper.mgDataTable.getMGData(0);
        }
        else {
            Logger.Instance.WriteExceptionToLogWithMsg("in Task.fillAttributes() out of string bounds");
        }
    }
    setAttribute(attribute, valueStr) {
        let isTagProcessed;
        isTagProcessed = super.setAttribute(attribute, valueStr);
        if (!isTagProcessed) {
            isTagProcessed = true;
            switch (attribute) {
                case ConstInterface.MG_ATTR_REFRESHON:
                    if (!(valueStr.trim() === ""))
                        this.SetRefreshOnVars(valueStr);
                    break;
                case XMLConstants.MG_ATTR_NAME:
                    this.Name = valueStr;
                    break;
                case ConstInterface.MG_ATTR_DVPOS_DEC:
                    if (!(valueStr.trim() === ""))
                        this.setDescriptor(valueStr);
                    break;
                case ConstInterface.MG_ATTR_HAS_LOCATE:
                    if (+valueStr == 1)
                        this.hasLocate = true;
                    break;
                case ConstInterface.MG_ATTR_AS_PARENT:
                    if (+valueStr == 1)
                        this.ModeAsParent = true;
                    break;
                case ConstInterface.MG_ATTR_TASK_UNIQUE_SORT:
                    this.UniqueSort = valueStr[0];
                    break;
                case ConstInterface.MG_ATTR_TRANS_ID:
                    this.setRemoteTransaction(valueStr);
                    break;
                case ConstInterface.MG_ATTR_PUBLIC:
                    this.PublicName = XmlParser.unescape(valueStr);
                    break;
                case XMLConstants.MG_ATTR_IS_OFFLINE:
                    this.IsOffline = XmlParser.getBoolean(valueStr);
                    break;
                case XMLConstants.MG_ATTR_RETURN_VALUE_EXP:
                    this.RetrunValueExp = XmlParser.getInt(valueStr);
                    break;
                case XMLConstants.MG_ATTR_MENUS_FILE_NAME:
                    break;
                default:
                    isTagProcessed = false;
                    break;
            }
        }
        if (!isTagProcessed)
            Logger.Instance.WriteExceptionToLogWithMsg(NString.Format("Unrecognized attribute: '{0}'", attribute));
        return true;
    }
    async initInnerObjects(foundTagName, parentForm) {
        if (foundTagName == null)
            return false;
        let parser = RuntimeContextBase.Instance.Parser;
        switch (foundTagName) {
            case XMLConstants.MG_TAG_HELPTABLE:
                if (this._helpTab == null)
                    this._helpTab = new Helps();
                if (Events.ShouldLog(Logger_LogLevels.Development))
                    Events.WriteDevToLog(NString.Format("{0} ...", foundTagName));
                this._helpTab.fillData(parser);
                break;
            case XMLConstants.MG_TAG_DVHEADER:
                if (Events.ShouldLog(Logger_LogLevels.Development))
                    Events.WriteDevToLog(NString.Format("{0} ...", foundTagName));
                this.DataView.fillHeaderData(parser);
                break;
            case XMLConstants.MG_TAG_PROP:
                this._propTab.fillData(this, 'T', parser);
                break;
            case XMLConstants.MG_TAG_FORM:
                this.Form = this.FormInitData(parentForm);
                break;
            case XMLConstants.MG_ATTR_MENU_CONTENT:
                break;
            case "assemblies":
                let endContext = parser.getXMLdata().indexOf('/assemblies', parser.getCurrIndex());
                parser.setCurrIndex(endContext);
                parser.setCurrIndex2EndOfTag();
                break;
            case XMLConstants.MG_TAG_FORMS:
                this._forms = new FormsTable(this, parentForm);
                Logger.Instance.WriteDevToLog(NString.Format("{0} ...", foundTagName));
                await this._forms.fillData();
                break;
            case XMLConstants.MG_TAG_TASK:
                if (this.SubTasks == null)
                    this.SubTasks = new TasksTable();
                let subtask = new Task(this);
                subtask.Init();
                this.SubTasks.addTask(subtask);
                let mgd = this._mgData;
                if (this.isMainProg())
                    mgd = AccessHelper.mgDataTable.GetMGDataForStartupProgram();
                await subtask.fillData(mgd, new OpeningTaskDetails());
                break;
            case ConstInterface.MG_TAG_TASKURL:
                await Task.ProcessTaskURL();
                break;
            case ConstInterface.MG_TAG_EXPTABLE:
                if (this.ExpTab == null)
                    this.ExpTab = new ExpTable();
                Logger.Instance.WriteDevToLog("goes to exp");
                this.ExpTab.fillData(this, parser);
                break;
            case ConstInterface.MG_TAG_USER_EVENTS:
                if (this.UserEvtTab == null)
                    this.UserEvtTab = new UserEventsTable();
                Logger.Instance.WriteDevToLog("goes to user event tab");
                this.UserEvtTab.fillData(this, parser);
                break;
            case ConstInterface.MG_TAG_LINKS:
                this.DataviewHeadersTable = new DataviewHeaders(this);
                this.DataviewHeadersTable.fillData(parser);
                break;
            case ConstInterface.MG_TAG_SORTS:
                this.RuntimeSorts = new SortCollection();
                this.RuntimeSorts.fillData(parser);
                break;
            case ConstInterface.MG_TAG_TASK_TABLES:
                this.FillTaskTables();
                break;
            case XMLConstants.MG_TAG_RECOMPUTE:
                Logger.Instance.WriteDevToLog("goes to recompute");
                this.RecomputeFillData(parser);
                break;
            case ConstInterface.MG_TAG_EVENTHANDLERS:
                Logger.Instance.WriteDevToLog("goes to eventhandlers");
                if (this.HandlersTab == null)
                    this.HandlersTab = new HandlersTable();
                this.HandlersTab.fillData(this, parser);
                this.addExpHandlersToMGData();
                break;
            default:
                if (foundTagName === ("/" + XMLConstants.MG_TAG_TASK)) {
                    RuntimeContextBase.Instance.Parser.setCurrIndex2EndOfTag();
                    return false;
                }
                else {
                    Logger.Instance.WriteDevToLog("There is no such tag in Task.initInnerObjects . Enter case for " + foundTagName);
                    return false;
                }
        }
        return true;
    }
    RecomputeFillData(parser) {
        let recomputeTable = new RecomputeTable();
        recomputeTable.fillData_1(this.DataView, this, parser);
    }
    FillTaskTables_1() {
        let parser = RuntimeContextBase.Instance.Parser;
        let endContext = parser.getXMLdata().indexOf(ConstInterface.MG_TAG_TASK_TABLES_END, parser.getCurrIndex());
        endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, endContext) + XMLConstants.TAG_CLOSE.length;
        let taskTablesData = parser.getXMLsubstring(endContext);
        this.FillTaskTables(taskTablesData);
        parser.setCurrIndex(endContext);
    }
    addExpHandlersToMGData() {
        let idx = 0;
        for (let i = this.HandlersTab.getSize() - 1; i >= 0; i--) {
            let handler = this.HandlersTab.getHandler(i);
            let evt = handler.getEvent();
            if (evt.getType() === ConstInterface.EVENT_TYPE_EXPRESSION ||
                (evt.getType() === ConstInterface.EVENT_TYPE_USER && evt.getUserEventType() === ConstInterface.EVENT_TYPE_EXPRESSION)) {
                this._mgData.addExpHandler(handler, idx);
                idx++;
            }
        }
    }
    SetRefreshOnVars(variables) {
        let strTok = StrUtil.tokenize(variables, ",");
        let fldNum;
        let intVals = new List();
        let i;
        for (i = 0; i < strTok.length; i++) {
            fldNum = NNumber.Parse(strTok[i]);
            intVals.push(fldNum);
        }
        this.RefreshOnVars = new Array(intVals.length);
        for (i = 0; i < intVals.length; i++) {
            fldNum = intVals.get_Item(i);
            this.RefreshOnVars[i] = fldNum;
        }
    }
    setDataSynced(synced) {
        this.DataSynced = synced;
    }
    getField(fldName) {
        return this.DataView.getFieldByName(fldName);
    }
    getExpById(id) {
        if (this.ExpTab !== null)
            return this.ExpTab.getExpById(id);
        else
            Logger.Instance.WriteExceptionToLogWithMsg("in Task.GetExpById(): no expression table");
        return null;
    }
    getExpObjectById(id) {
        return this.getExpById(id);
    }
    GetExpressionStorage(expId) {
        let result = null;
        if (expId > 0) {
            let expById = this.getExpById(expId);
            Debug.Assert(expById !== null);
            result = expById.GetStorageAttribute();
        }
        return result;
    }
    async CalculateExpression(expId, resType, length) {
        let result = null;
        if (expId > 0) {
            let expById = this.getExpById(expId);
            Debug.Assert(expById !== null);
            result = await expById.evaluateWithResultTypeAndLength(resType, length);
        }
        return result;
    }
    async insertRecordTable(invalidate, parser) {
        if (await this.isCached()) {
            let firstTime = this.DataView.getFirstDv();
            if (!firstTime && invalidate) {
                if (!this.hasLocate) {
                    if (this.DataView.IncludesFirst()) {
                        if (!this.DataView.getChanged())
                            this._dvCache.putInCache(this.DataView.replicate());
                        else
                            this._dvCache.removeDvFromCache(this.DataView.getDvPosValue(), true);
                    }
                }
                else
                    this.locatePutInCache();
            }
        }
        await this.DataView.fillData(parser);
        if (invalidate)
            this.DataView.setChanged(false);
    }
    getHandlersTab() {
        return this.HandlersTab;
    }
    async buildXML(message) {
        this.IsAfterRetryBeforeBuildXML = this.getAfterRetry();
        if (this.KnownToServer && !this.IsOffline) {
            message.Append(XMLConstants.START_TAG + XMLConstants.MG_TAG_TASK);
            message.Append(" " + XMLConstants.MG_ATTR_TASKID + "=\"" + this.getTaskTag() + "\"");
            message.Append(" " + ConstInterface.MG_ATTR_TASK_MAINLEVEL + "=\"" + this.getMainLevel() + "\"");
            message.Append(XMLConstants.TAG_CLOSE);
            await this.DataView.buildXML(message);
            if (this.Form != null && this.Form.Opened)
                message.Append(this.Form.GetHiddenControlListXML());
            message.Append("\n   </" + XMLConstants.MG_TAG_TASK + XMLConstants.TAG_CLOSE);
        }
    }
    buildXMLForRngs(message, UserRanges, locate) {
        let i;
        let fld;
        let cellAttr = StorageAttribute.SKIP;
        let toBase64 = (Environment.Instance.GetDebugLevel() <= 1);
        message.Append(XMLConstants.TAG_CLOSE);
        for (i = 0; i < UserRanges.length; i++) {
            message.Append(XMLConstants.START_TAG + ConstInterface.USER_RNG);
            let rng = UserRanges.get_Item(i);
            message.Append(" " + XMLConstants.MG_TAG_FLD + "=\"" + rng.veeIdx + "\"");
            fld = this.DataView.getField(rng.veeIdx - 1);
            let fldAttr = fld.getType();
            if (rng.nullMin)
                message.Append(" " + ConstInterface.NULL_MIN_RNG + "=\"1\"");
            if (!rng.discardMin && rng.min != null) {
                let val = RecordUtils.itemValToXML(rng.min, fldAttr, cellAttr, toBase64);
                message.Append(" " + ConstInterface.MIN_RNG + "=\"" + val + "\"");
            }
            if (rng.nullMax)
                message.Append(" " + ConstInterface.NULL_MAX_RNG + "=\"1\"");
            if (!rng.discardMax && rng.max != null) {
                let val = RecordUtils.itemValToXML(rng.max, fldAttr, cellAttr, toBase64);
                message.Append(" " + ConstInterface.MAX_RNG + "=\"" + val + "\"");
            }
            message.Append(XMLConstants.TAG_TERM);
        }
        if (locate)
            message.Append(XMLConstants.END_TAG + ConstInterface.USER_LOCATES + XMLConstants.TAG_CLOSE);
        else
            message.Append(XMLConstants.END_TAG + ConstInterface.USER_RANGES + XMLConstants.TAG_CLOSE);
    }
    buildXMLForSorts(message) {
        if (this.UserSorts != null && this.UserSorts.length > 0) {
            let i;
            for (i = 0; i < this.UserSorts.length; i++) {
                message.Append(XMLConstants.TAG_CLOSE);
                let srt = this.UserSorts.get_Item(i);
                message.Append(XMLConstants.START_TAG + ConstInterface.SORT);
                message.Append(" " + XMLConstants.MG_TAG_FLD + "=\"" + srt.fldIdx + "\"");
                if (srt.dir)
                    message.Append(" " + ConstInterface.MG_ATTR_DIR + "=\"1\"");
                message.Append(XMLConstants.TAG_TERM);
            }
            this.UserSorts.Clear();
            this.UserSorts = null;
            message.Append(XMLConstants.END_TAG + ConstInterface.MG_TAG_SORTS + XMLConstants.TAG_CLOSE);
        }
    }
    getUserEvent(idx) {
        Debug.Assert(this.UserEvtTab !== null);
        let retEvent = null;
        if (this.UserEvtTab !== null)
            retEvent = this.UserEvtTab.getEvent(idx);
        return retEvent;
    }
    SetMoveToFirstControl(moveToFirstControl) {
        if (moveToFirstControl || !this.IsSubForm) {
            if (this.Form !== null) {
                let alreadyMoved = this.Form.alreadyMovedToFirstControl();
                if (!alreadyMoved)
                    this.Form.MovedToFirstControl = true;
            }
        }
    }
    async Start(moveToFirstControl, callByDestSubForm) {
        let nonInteractiveTask = null;
        if (!this._isStarted) {
            this._isStarted = true;
            this.InitializeExecution();
            this.InStartProcess = true;
            let result = await this.Setup();
            if (!result.Success)
                return null;
            this._inProcessingTopMostEndTaskSaved = AccessHelper.eventsManager.getProcessingTopMostEndTask();
            AccessHelper.eventsManager.setProcessingTopMostEndTask(false);
            await this.EnableActions();
            result = await this.InitializeForm(moveToFirstControl);
            if (!result.Success)
                return null;
            if (!this.IsInteractive && !this._isMainPrg)
                nonInteractiveTask = this;
            this.DataViewWasRetrieved = true;
            let subTasks = this.SubTasks;
            if (this.isMainProg())
                this.SubTasks = null;
            if (this.isMainProg())
                this.SubTasks = subTasks;
            if (!result.Success)
                return null;
            if (this._aborting)
                return null;
            let dataViewCommand = CommandFactory.CreateDataViewCommand(this.getTaskTag(), DataViewCommandType.FirstChunk);
            result = await this.DataviewManager.Execute(dataViewCommand);
            if (!result.Success)
                return null;
            this.ResumeSubformLayout();
        }
        nonInteractiveTask = await this.StartSubTasks(moveToFirstControl, nonInteractiveTask, callByDestSubForm);
        this.InStartProcess = false;
        return nonInteractiveTask;
    }
    InitializeExecution() {
        super.setBrkLevel(ConstInterface.BRK_LEVEL_REC_MAIN, -1);
        this._firstRecordCycle = true;
        this._currStartProgLevel = ((this.IsSubForm && this.getParent().getForm().alreadyMovedToFirstControl())
            ? this.getParent()._currStartProgLevel
            : Task.StartProgLevel);
    }
    async InitializeForm(moveToFirstControl) {
        if (this.Form !== null) {
            this.SetMoveToFirstControl(moveToFirstControl);
            AccessHelper.eventsManager.pushNewExecStacks();
            this.routeParams = this.DataView.GetRouteParams();
            await this.InitForm();
            AccessHelper.eventsManager.popNewExecStacks();
            this.getMGData().StartTimers();
            Commands.beginInvoke();
        }
        return new ReturnResult();
    }
    async StartSubTasks(moveToFirstControl, nonInteractiveTask, callByDestSubForm) {
        if (this.hasSubTasks()) {
            for (let i = 0; i < this.SubTasks.getSize(); i = i + 1) {
                let task = this.SubTasks.getTask(i);
                let tmpNonInteractiveTask = await task.Start(moveToFirstControl, callByDestSubForm);
                if (nonInteractiveTask === null)
                    nonInteractiveTask = tmpNonInteractiveTask;
                else if (tmpNonInteractiveTask !== null)
                    Debug.Assert(false, "more than 1 non interactive in task.start");
            }
        }
        return nonInteractiveTask;
    }
    async EnableActions() {
        let actList = ActionManager.actEnabled;
        this.ActionManager.enableList(actList, true, false);
        if (this.IsSubForm)
            this.ActionManager.enable(InternalInterface.MG_ACT_POST_REFRESH_BY_PARENT, true);
        await this.enableModes();
        if (await this.checkProp(PropInterface.PROP_TYPE_SELECTION, false)) {
            this.ActionManager.enable(InternalInterface.MG_ACT_SELECT, true);
        }
        let enablePrintdata = await this.checkProp(PropInterface.PROP_TYPE_PRINT_DATA, false);
        this.ActionManager.enable(InternalInterface.MG_ACT_PRINT_DATA, enablePrintdata);
        if (this.getEnableZoomHandler())
            this.ActionManager.enable(InternalInterface.MG_ACT_ZOOM, true);
    }
    async ShowError(text) {
        let mlsTransText = LanguageData.Instance.translate(text);
        let mlsTransTitle = LanguageData.Instance.translate(ConstInterface.ERROR_STRING);
        await GUIManager.Instance.MessageBox(mlsTransTitle, mlsTransText, Styles.MSGBOX_ICON_ERROR | Styles.MSGBOX_BUTTON_OK);
    }
    async EndTaskOnError(result, displayError) {
        if (!result.Success && !this.isAborting()) {
            if (this.InStartProcess) {
                await this.endTask(false, false, false);
                if (displayError)
                    await this.ShowError(result.ErrorDescription);
                this.InStartProcess = false;
                return true;
            }
            else {
                if (displayError)
                    await this.ShowError(result.ErrorDescription);
                await AccessHelper.eventsManager.handleInternalEventWithTask(this, InternalInterface.MG_ACT_EXIT);
            }
        }
        return false;
    }
    async Setup() {
        let result;
        this.TaskService.InitTaskPrefixExecutedFlag(this);
        this.TaskSuffixExecuted = false;
        TaskServiceBase.CreateFirstRecord(this);
        let dataViewCommand = CommandFactory.CreateDataViewCommand(this.getTaskTag(), DataViewCommandType.Init);
        result = await this.DataviewManager.Execute(dataViewCommand);
        if (!result.Success)
            return result;
        if (result.Success) {
            let commands = GlobalCommandsManager.Instance.getCommandsExecutedAfterTaskStarted();
            for (let cmdIdx = 0; cmdIdx < commands.length; cmdIdx++) {
                let command = (commands.get_Item(cmdIdx));
                if (command.TaskTag === this.getTaskTag()) {
                    await command.Execute(null);
                    GlobalCommandsManager.Instance.removeCommandsExecutedAfterTaskStarted(command);
                }
            }
            dataViewCommand = CommandFactory.CreateDataViewCommand(this.getTaskTag(), DataViewCommandType.Prepare);
            result = await this.DataviewManager.Execute(dataViewCommand);
        }
        if (result.Success) {
            result = this.TaskService.PrepareTask(this);
            if (!result.Success)
                await this.EndTaskOnError(result, true);
        }
        if (result.Success) {
            dataViewCommand = CommandFactory.CreateDataViewCommand(this.getTaskTag(), DataViewCommandType.InitDataControlViews);
            result = await this.DataviewManager.Execute(dataViewCommand);
        }
        return result;
    }
    async doFirstRecordCycle() {
        if (!this._isMainPrg) {
            AccessHelper.eventsManager.pushNewExecStacks();
            this.SubformExecMode = Task_SubformExecModeEnum.FIRST_TIME;
            await AccessHelper.eventsManager.handleInternalEventWithTaskAndSubformRefresh(this, InternalInterface.MG_ACT_REC_PREFIX, this.IsSubForm);
            if (!AccessHelper.eventsManager.GetStopExecutionFlag()) {
                if (this.IsSubForm) {
                    if (this.hasSubTasks())
                        for (let i = 0; i < this.SubTasks.getSize(); i++) {
                            let subTask = this.SubTasks.getTask(i);
                            if (subTask.IsSubForm) {
                                subTask.getForm().IgnoreFirstRecordCycle = true;
                                await subTask.doFirstRecordCycle();
                            }
                        }
                    await AccessHelper.eventsManager.handleInternalEventWithTask(this, InternalInterface.MG_ACT_REC_SUFFIX);
                }
            }
            this.SubformExecMode = Task_SubformExecModeEnum.SET_FOCUS;
            AccessHelper.eventsManager.popNewExecStacks();
        }
    }
    CheckAndSetSubForm() {
        this.IsSubForm = (!super.isMainProg() && this !== this._mgData.getFirstTask());
    }
    IsRoute() {
        return (this.RouterPath != null);
    }
    async stop() {
        let task;
        if (this._aborting)
            return;
        if (this.hasSubTasks()) {
            while ((task = this.SubTasks.getTask(0)) != null) {
                task.setDestinationSubform(this._destinationSubform);
                await task.stop();
            }
        }
        if (!this._isMainPrg)
            this._aborting = true;
        if (this._parentTask != null) {
            this._parentTask.removeSubTask(this);
            if (this.IsSubForm) {
                let oldTimers = new List(), newTimers = new List();
                let mgd = this.getMGData();
                if (mgd.getTimerHandlers() != null)
                    oldTimers = mgd.getTimerHandlers().getTimersVector();
                mgd.removeTimerHandler(this);
                mgd.removeExpressionHandler(this);
                if (mgd.getTimerHandlers() != null)
                    newTimers = mgd.getTimerHandlers().getTimersVector();
                mgd.changeTimers(oldTimers, newTimers);
                mgd.removeTask(this);
                GUIManager.Instance.abort(this.getForm());
            }
        }
        this.AbortTransaction();
        if (this.getClickedControl() != null)
            RuntimeContextBase.Instance.CurrentClickedCtrl = null;
        if (this.Form != null)
            this.Form.removeRefsToCtrls();
        if (this.IsSubForm) {
            let subForm = this.Form.getSubFormCtrl();
            subForm.setSubformTaskId(null);
            subForm.initSubformTask();
            if (LastFocusedManager.Instance.getLastFocusedTask() === this)
                LastFocusedManager.Instance.setLastFocusedTask(this._parentTask);
            if (subForm.SubformLoaded)
                await (subForm.getProp(PropInterface.PROP_TYPE_VISIBLE)).RefreshDisplay(true);
        }
        let clearDataViewCommand = CommandFactory.CreateDataViewCommand(this.getTaskTag(), DataViewCommandType.Clear);
        await this.DataviewManager.Execute(clearDataViewCommand);
    }
    abort() {
        this.getMGData().abort();
    }
    hasSubTasks() {
        return this.SubTasks !== null && this.SubTasks.getSize() > 0;
    }
    getTaskDepth(byParentOrder) {
        if (byParentOrder)
            return this.getTaskDepth_(byParentOrder);
        let depth = this.GetContextTask().getTaskDepth_(byParentOrder);
        if (this.isMainProg() && this.getCtlIdx() !== ConstInterface.TOP_LEVEL_CONTAINER)
            depth = depth + 1;
        return depth;
    }
    getTaskDepth_(byParentOrder) {
        let tmpTask = this.getTriggeringTask();
        if (tmpTask == null || byParentOrder)
            if (this._parentTask == null)
                return 1;
            else {
                if (byParentOrder && this.getCtlIdx() !== this._parentTask.getCtlIdx())
                    return 2 + this._parentTask.getTaskDepth_(byParentOrder);
                return 1 + this._parentTask.getTaskDepth_(byParentOrder);
            }
        return 1 + (tmpTask).getTaskDepth_(byParentOrder);
    }
    GetTaskAncestor(generation) {
        return this.getTaskAncestor_(generation);
    }
    getTaskAncestor_(generation) {
        let retTask = null;
        if (generation === 0)
            retTask = this;
        else {
            let tmpTask = this.getTriggeringTask();
            if (tmpTask === null)
                tmpTask = this._parentTask;
            if (tmpTask !== null)
                retTask = tmpTask.getTaskAncestor_(--generation);
        }
        return retTask;
    }
    GetTaskDepth() {
        return this.getTaskDepth(false);
    }
    pathContains(task) {
        return this._taskPath.Contains(task);
    }
    SetContextTask(context) {
        this.ContextTask = context;
    }
    GetContextTask() {
        let currRec = (this.DataView.getCurrRec());
        if (this.InStartProcess
            || currRec !== null && (currRec.InCompute || currRec.InRecompute)
            || this.Form !== null && this.Form.inRefreshDisplay())
            return this;
        return this.ContextTask;
    }
    getTriggeringTask() {
        if (this._triggeringTask !== null && this._triggeringTask.getMGData().IsAborting)
            this._triggeringTask = this._parentTask;
        return this._triggeringTask;
    }
    setLastParkedCtrl(ctrl) {
        super.setLastParkedCtrl(ctrl);
        LastFocusedManager.setLastFocusedControl(this, ctrl);
    }
    queryTaskPath() {
        let insertedNames = 0;
        let result = new StringBuilder(7936);
        let treeRoute = new Array(this.getTaskDepth(false));
        this.pathToRoot(treeRoute, false);
        for (let tsk = treeRoute.length - 1; tsk >= 0; tsk--) {
            let currTsk = treeRoute[tsk];
            if (currTsk.isMainProg())
                continue;
            let nameProp = currTsk.getProp(PropInterface.PROP_TYPE_NAME);
            if (nameProp == null)
                continue;
            let name = nameProp.GetComputedValue();
            if (!NString.IsNullOrEmpty(name)) {
                if (insertedNames++ !== 0)
                    result.Append(';');
                result.Append(name);
            }
        }
        return result;
    }
    ctl_itm_2_parent_vee(itm) {
        let currTsk;
        let currItm = 0;
        let lastItm = 0;
        let treeRoute = new Array(this.getTaskDepth(true));
        this.pathToRoot(treeRoute, true);
        for (let tsk = treeRoute.length - 1; tsk >= 0; tsk--) {
            currTsk = treeRoute[tsk];
            currItm += currTsk.DataView.GetFieldsTab().getSize();
            if (itm < currItm) {
                itm -= lastItm;
                return currTsk.DataView.getField(itm);
            }
            lastItm = currItm;
        }
        return null;
    }
    GetControlIDFromVarItem(item) {
        let currTask;
        let currItem = 0;
        let lastItem = 0;
        let noOfControlsBeforeCurrTask = 0;
        let treeRoute = new Array(this.getTaskDepth(true));
        this.pathToRoot(treeRoute, true);
        for (let task = treeRoute.length - 1; task >= 0; task--) {
            currTask = treeRoute[task];
            currItem += currTask.DataView.GetFieldsTab().getSize();
            if (item < currItem) {
                item -= lastItem;
                let field = currTask.DataView.getField(item);
                if (field.getCtrl() != null)
                    return noOfControlsBeforeCurrTask = noOfControlsBeforeCurrTask + currTask.Form.getControlIdx(field.getCtrl()) + 1;
                return 0;
            }
            if (currTask.Form != null)
                noOfControlsBeforeCurrTask += currTask.Form.getCtrlCount();
            lastItem = currItem;
        }
        return 0;
    }
    GetControlFromControlID(controlID, parent) {
        let control = null;
        parent.value = -1;
        if (controlID >= 0) {
            let currTask = null;
            let callerTask = null;
            let noOfControlsInCurrtask = 0;
            let noOfControlsBeforeCurrTask = 0;
            let controlIdx = 0;
            let treeRoute = new Array(this.getTaskDepth(true));
            this.pathToRoot(treeRoute, true);
            for (let task = treeRoute.length - 1; task >= 0; task--) {
                currTask = treeRoute[task];
                if (currTask.Form != null) {
                    noOfControlsInCurrtask = currTask.Form.getCtrlCount();
                    if (controlID < (noOfControlsBeforeCurrTask + noOfControlsInCurrtask)) {
                        controlIdx = controlID - noOfControlsBeforeCurrTask;
                        control = currTask.Form.CtrlTab.getCtrl(controlIdx);
                        callerTask = currTask;
                        break;
                    }
                    noOfControlsBeforeCurrTask += noOfControlsInCurrtask;
                }
            }
            if (control != null) {
                parent.value = 0;
                let prevCtlIdx = treeRoute[0].TaskDefinitionId.CtlIndex;
                for (let task = 0; task < treeRoute.length - 1; task++) {
                    currTask = treeRoute[task];
                    if (prevCtlIdx !== currTask.TaskDefinitionId.CtlIndex)
                        parent.value++;
                    if (callerTask.TaskDefinitionId.CtlIndex === currTask.TaskDefinitionId.CtlIndex)
                        break;
                    prevCtlIdx = currTask.TaskDefinitionId.CtlIndex;
                }
            }
        }
        return control;
    }
    ctl_itm_4_parent_vee(parent, vee) {
        let tsk;
        let depth = this.getTaskDepth(true);
        if (parent !== TaskBase.MAIN_PRG_PARENT_ID && depth <= parent)
            return vee;
        let treeRoute = new Array(depth);
        if (vee !== 0) {
            this.pathToRoot(treeRoute, true);
            let indOfParentInTaskTree = this.getIndOfParentInTaskTree(parent, treeRoute);
            let i;
            for (i = treeRoute.length - 1; i > indOfParentInTaskTree; i--) {
                tsk = treeRoute[i];
                vee += tsk.DataView.GetFieldsTab().getSize();
            }
        }
        return vee;
    }
    getIndOfParentInTaskTree(parent, taskTree) {
        let indOfParentInTaskTree = 0;
        if (parent === TaskBase.MAIN_PRG_PARENT_ID) {
            for (let i = 1; i < taskTree.length; i = i + 1) {
                if (taskTree[i].isMainProg() && this._ctlIdx === taskTree[i].getCtlIdx()) {
                    indOfParentInTaskTree = i;
                    break;
                }
            }
        }
        else
            indOfParentInTaskTree = parent;
        return indOfParentInTaskTree;
    }
    getFieldByName(fldName) {
        return (this.GetContextTask()).getFieldByName_(fldName);
    }
    getFieldByName_(fldName) {
        let tmpTask;
        if (fldName == null)
            return null;
        let fld = this.getField(fldName);
        if (fld == null) {
            tmpTask = this.getTriggeringTask() != null ? this.getTriggeringTask() : this._parentTask;
            if (tmpTask != null) {
                if (!this.isMainProg() && this.getCtlIdx() !== tmpTask.getCtlIdx()) {
                    let mainPrg = AccessHelper.mgDataTable.GetMainProgByCtlIdx(-1, this.getCtlIdx());
                    fld = mainPrg.getFieldByName_(fldName);
                }
                if (fld == null)
                    fld = tmpTask.getFieldByName_(fldName);
            }
        }
        return fld;
    }
    getIndexOfFieldByName(fldName) {
        let depth = this.getTaskDepth(true);
        let index = 0;
        let fld = this.getFieldByName(fldName);
        let fldTab;
        if (fld == null)
            return 0;
        let treeRoute = new Array(depth);
        this.pathToRoot(treeRoute, true);
        for (let curr = depth - 1; curr >= 0; curr--) {
            let currTask = treeRoute[curr];
            if (currTask === fld.getTask()) {
                curr = fld.getId();
                index += ++curr;
                break;
            }
            fldTab = currTask.DataView.GetFieldsTab();
            index += fldTab.getSize();
        }
        return index;
    }
    pathToRoot(path, byParentOrder) {
        let task = (byParentOrder ? this : this.GetContextTask());
        for (let i = 0; i < path.length; i++) {
            path[i] = task;
            let ctlIdx = task.getCtlIdx();
            let tmpTask = byParentOrder ? null : task.getTriggeringTask();
            if (tmpTask != null)
                task = tmpTask;
            else {
                task = task.getParent();
                if (task == null) {
                    if (!byParentOrder && this.isMainProg() && this.getCtlIdx() !== ConstInterface.TOP_LEVEL_CONTAINER)
                        path[i + 1] = this;
                    break;
                }
                if (byParentOrder && ctlIdx !== task.getCtlIdx())
                    path[++i] = AccessHelper.mgDataTable.GetMainProgByCtlIdx(-1, ctlIdx);
            }
        }
    }
    shouldBeRefreshed() {
        let dataView = this._parentTask.DataView;
        let result = false;
        if (this._parentTask === null || dataView.isPrevCurrRecNull() ||
            (this.RefreshOnVars !== null && !dataView.currEqualsPrev(this.RefreshOnVars)))
            result = true;
        return result;
    }
    async CheckRefreshSubTasks() {
        let ret = false;
        if (this._enteredRecLevel) {
            let subformsToRefresh = await this.GetSubformsToRefresh();
            if (subformsToRefresh.length > 0)
                ret = true;
        }
        return ret;
    }
    GetSubTasks() {
        let subTasks = new List();
        if (this.hasSubTasks()) {
            for (let i = 0; i < this.SubTasks.getSize(); i = i + 1)
                subTasks.push(this.SubTasks.getTask(i));
        }
        return subTasks;
    }
    async GetSubformsToRefresh() {
        let subTasks = this.GetSubTasks();
        let subTasksToRefresh = new List();
        for (let i = 0; i < subTasks.length; i++) {
            let subTask = subTasks.get_Item(i);
            if (await this.ShouldRefreshSubformTask(subTask))
                subTasksToRefresh.push(subTask);
        }
        return subTasksToRefresh;
    }
    async ShouldRefreshSubformTask(subTask) {
        let refresh = false;
        let subformCtrl = subTask.getForm().getSubFormCtrl();
        subTask.DoSubformPrefixSuffix = false;
        if (subformCtrl != null && !subTask.InSelect && subTask.AfterFirstRecordPrefix) {
            let parentMode = this.getMode();
            let modeProperty = (parentMode === Constants.TASK_MODE_QUERY ? PropInterface.PROP_TYPE_ALLOW_QUERY
                : parentMode === Constants.TASK_MODE_MODIFY
                    ? PropInterface.PROP_TYPE_ALLOW_MODIFY
                    : PropInterface.PROP_TYPE_ALLOW_CREATE);
            if (subTask.ModeAsParent && parentMode !== subTask.getMode() &&
                this.getLevel() === Constants.TASK_LEVEL_RECORD &&
                await (subTask.getProp(PropInterface.PROP_TYPE_ALLOW_OPTION)).getValueBoolean() &&
                await subTask.checkProp(modeProperty, true)) {
                if (!subformCtrl.isVisible() &&
                    !await subformCtrl.checkProp(PropInterface.PROP_TYPE_REFRESH_WHEN_HIDDEN, false)) {
                    subformCtrl.RefreshOnVisible = true;
                    refresh = false;
                }
                else
                    refresh = true;
                await subTask.SetModeAsParent(parentMode);
            }
            else {
                refresh = await subformCtrl.checkProp(PropInterface.PROP_TYPE_AUTO_REFRESH, true) && subTask.shouldBeRefreshed();
                if (refresh) {
                    if (!subformCtrl.isVisible() &&
                        !await subformCtrl.checkProp(PropInterface.PROP_TYPE_REFRESH_WHEN_HIDDEN, false)) {
                        subformCtrl.RefreshOnVisible = true;
                        refresh = false;
                    }
                }
            }
            if (refresh)
                subTask.DoSubformPrefixSuffix = true;
        }
        return refresh;
    }
    async SetModeAsParent(parentMode) {
        await this.enableModes();
        if (parentMode === Constants.TASK_MODE_CREATE) {
            let cmd = CommandFactory.CreateEventCommand(this.getTaskTag(), InternalInterface.MG_ACT_RTO_CREATE);
            await this.DataviewManager.Execute(cmd);
        }
        else if (this.getMode() === Constants.TASK_MODE_CREATE) {
            this.setPreventRecordSuffix(true);
            this.setPreventControlChange(true);
            await AccessHelper.eventsManager.handleInternalEventWithTask(this, InternalInterface.MG_ACT_RT_REFRESH_VIEW);
            this.setPreventRecordSuffix(false);
            this.setPreventControlChange(false);
            let subTaskDv = this.DataView;
            if (!subTaskDv.isEmpty() && (subTaskDv.getCurrRec()).getMode() !== DataModificationTypes.Insert)
                this.setMode(parentMode);
            else
                this.setMode(Constants.TASK_MODE_CREATE);
            await this.getForm().RefreshDisplay(Constants.TASK_REFRESH_CURR_REC);
        }
        else
            this.setMode(parentMode);
        this.setOriginalTaskMode(parentMode);
        await this.setCreateDeleteActsEnableState();
    }
    async doSubformRecPrefixSuffix() {
        let subTasks = this.GetSubTasks();
        for (let i = 0; i < subTasks.length; i++) {
            let subTask = subTasks.get_Item(i);
            if (subTask.DoSubformPrefixSuffix) {
                await AccessHelper.eventsManager.handleInternalEventWithTaskAndSubformRefresh(subTask, InternalInterface.MG_ACT_REC_PREFIX, true);
                if (!AccessHelper.eventsManager.GetStopExecutionFlag())
                    await AccessHelper.eventsManager.handleInternalEventWithTask(subTask, InternalInterface.MG_ACT_REC_SUFFIX);
            }
        }
    }
    CleanDoSubformPrefixSuffix() {
        let subTasks = this.GetSubTasks();
        for (let i = 0; i < subTasks.length; i++) {
            let subTask = subTasks.get_Item(i);
            subTask.DoSubformPrefixSuffix = false;
        }
    }
    mustConfirmInDeleteMode() {
        let prop = this.getProp(PropInterface.PROP_TYPE_CONFIRM_UPDATE);
        return !prop.isExpression();
    }
    addSubTask(subTask) {
        if (this.SubTasks === null)
            this.SubTasks = new TasksTable();
        this.SubTasks.addTask(subTask);
    }
    removeSubTask(subTask) {
        if (this.SubTasks !== null)
            this.SubTasks.removeTask(subTask);
    }
    setTransactionFailed(val) {
        this._transactionFailed = val;
        if (!val)
            this.TransactionErrorHandlingsRetry = null;
    }
    transactionFailed(level) {
        if (level === ConstInterface.TRANS_NONE ||
            (this.Transaction != null && this.Transaction.getLevel() === level))
            return this._transactionFailed;
        else if (level === ConstInterface.TRANS_NONE ||
            (this.TransactionErrorHandlingsRetry != null && this.TransactionErrorHandlingsRetry.getLevel() === level))
            return this._transactionFailed;
        return false;
    }
    setAfterRetry(val) {
        if (this.Transaction != null)
            this.Transaction.setAfterRetry(val);
        else if (this.TransactionErrorHandlingsRetry != null) {
            if (val === ConstInterface.RECOVERY_NONE)
                this.TransactionErrorHandlingsRetry = null;
            else
                this.TransactionErrorHandlingsRetry.setAfterRetry(val);
        }
    }
    getAfterRetry(recovery) {
        if (arguments.length === 0) {
            return this.getAfterRetry_0();
        }
        return this.getAfterRetry_1(recovery);
    }
    getAfterRetry_0() {
        if (this.Transaction != null)
            return this.Transaction.getAfterRetry();
        else if (this.TransactionErrorHandlingsRetry != null)
            return this.TransactionErrorHandlingsRetry.getAfterRetry();
        return false;
    }
    getAfterRetry_1(recovery) {
        if (this.Transaction != null)
            return this.Transaction.getAfterRetry(recovery);
        else if (this.TransactionErrorHandlingsRetry != null)
            return this.TransactionErrorHandlingsRetry.getAfterRetry(recovery);
        return false;
    }
    getExecEndTask() {
        return this._bExecEndTask;
    }
    resetExecEndTask() {
        this._bExecEndTask = false;
    }
    setExecEndTask() {
        this._bExecEndTask = true;
    }
    async evalEndCond(mode) {
        let endTaskCondProp;
        let evalEndCondProp = this.getProp(PropInterface.PROP_TYPE_EVAL_END_CONDITION);
        if (this.InEndTask)
            return false;
        if (this._bExecEndTask)
            return true;
        if (evalEndCondProp != null && (NString.Compare(await evalEndCondProp.getValue(), ConstInterface.END_COND_EVAL_IMMIDIATE, true) === 0 || ((await evalEndCondProp.getValue()).toLowerCase() === mode.toLowerCase()))) {
            endTaskCondProp = this.getProp(PropInterface.PROP_TYPE_END_CONDITION);
            if (endTaskCondProp == null)
                return false;
            let endTask = await endTaskCondProp.getValue();
            if (DisplayConvertor.toBoolean(endTask)) {
                this._bExecEndTask = true;
                return true;
            }
            return false;
        }
        return false;
    }
    setTryingToStop(val) {
        if (this.hasSubTasks()) {
            for (let i = 0; i < this.SubTasks.getSize(); i = i + 1) {
                let task = this.SubTasks.getTask(i);
                task.setTryingToStop(val);
            }
        }
        this.IsTryingToStop = val;
    }
    getInRecordSuffix() {
        return this._inRecordSuffix;
    }
    setInRecordSuffix(bValue) {
        if (bValue !== this._inRecordSuffix) {
            this._inRecordSuffix = bValue;
            return true;
        }
        return false;
    }
    setEvalOldValues(bFlag) {
        if (bFlag !== this._evalOldValues) {
            this._evalOldValues = bFlag;
            return true;
        }
        return false;
    }
    getEvalOldValues() {
        return this._evalOldValues;
    }
    async endTask(reversibleExit, onlyDescs, dueToVerifyError, subformDestination) {
        if (isNullOrUndefined(subformDestination))
            return await this.endTask_1(reversibleExit, onlyDescs, false, dueToVerifyError);
        else
            return await this.endTask_1(reversibleExit, onlyDescs, subformDestination, dueToVerifyError);
    }
    async endTask_1(reversibleExit, onlyDescs, subformDestination, dueToVerifyError) {
        let succeeded = true;
        let closedSubForm = false;
        let subTask;
        let mgdTab = AccessHelper.mgDataTable;
        if ((this.IsSubForm && !subformDestination) && !this._parentTask.InEndTask) {
            succeeded = await this._parentTask.endTask(reversibleExit, onlyDescs, dueToVerifyError);
        }
        else if (!this.isAborting() && !this.InEndTask) {
            try {
                this.InEndTask = true;
                if ((MGDataCollection.Instance.StartupMgData.getFirstTask().InEndTask && MGDataCollection.Instance.StartupMgData.getFirstTask().ClosingFormUI) && AccessHelper.eventsManager.GetStopExecutionFlag()) {
                    let cmd = CommandFactory.CreateUnloadCommand();
                    MGDataCollection.Instance.getMGData(0).CmdsToServer.Add(cmd);
                    await CommandsProcessorManager.GetCommandsProcessor().Execute(CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS);
                    return;
                }
                if (!this.IsTryingToStop)
                    this.setTryingToStop(true);
                let i;
                if (reversibleExit) {
                    if (this.hasSubTasks()) {
                        if (!RemoteCommandsProcessor.GetInstance().DelayCommandExecution) {
                            let pendingTransactions = true;
                            let clientOperExist = await this.CheckIfClientOperationExistsInSubformTasks(this, InternalInterface.MG_ACT_TASK_SUFFIX);
                            if (!clientOperExist) {
                                clientOperExist = await this.CheckIfClientOperationExistsInSubformTasks(this, InternalInterface.MG_ACT_REC_SUFFIX);
                                if (!clientOperExist)
                                    pendingTransactions = this.CheckIfPendingTransactions(this);
                            }
                            if (!clientOperExist && !pendingTransactions) {
                                this.allowRemoteCmdExecution = false;
                                this.SetServerSideTaskSuffix(this);
                                RemoteCommandsProcessor.GetInstance().DelayCommandExecution = true;
                            }
                        }
                        else
                            this.SetServerSideTaskSuffix(this);
                        for (i = 0; i < this.SubTasks.getSize() && succeeded; i++) {
                            subTask = this.SubTasks.getTask(i);
                            closedSubForm = closedSubForm || subTask.IsSubForm;
                            await AccessHelper.eventsManager.handleInternalEventWithTask(subTask, InternalInterface.MG_ACT_EXIT);
                            succeeded = (!AccessHelper.eventsManager.GetStopExecutionFlag() || dueToVerifyError);
                        }
                    }
                    else {
                        if (RemoteCommandsProcessor.GetInstance().DelayCommandExecution)
                            this.SetServerSideTaskSuffix(this);
                    }
                    if (this === mgdTab.StartupMgData.getFirstTask()) {
                        for (i = mgdTab.getSize() - 1; i > 0 && succeeded; i--) {
                            if (mgdTab.getMGData(i) != null) {
                                await AccessHelper.eventsManager.handleInternalEventWithTask(mgdTab.getMGData(i).getFirstTask(), InternalInterface.MG_ACT_EXIT);
                                succeeded = (!AccessHelper.eventsManager.GetStopExecutionFlag() || dueToVerifyError);
                            }
                        }
                    }
                }
                if (succeeded) {
                    if (reversibleExit && !dueToVerifyError) {
                        let forceSuffix = await this.checkProp(PropInterface.PROP_TYPE_FORCE_SUFFIX, false);
                        if (!this.IsInteractive && forceSuffix === true && this.getMode() === Constants.TASK_MODE_CREATE)
                            this.setProp(PropInterface.PROP_TYPE_FORCE_SUFFIX, "0");
                        await AccessHelper.eventsManager.handleInternalEventWithTask(this, InternalInterface.MG_ACT_REC_SUFFIX);
                        if (!this.IsInteractive && forceSuffix === true && this.getMode() === Constants.TASK_MODE_CREATE)
                            this.setProp(PropInterface.PROP_TYPE_FORCE_SUFFIX, "1");
                    }
                    succeeded = (!AccessHelper.eventsManager.GetStopExecutionFlag() || dueToVerifyError);
                    if (succeeded && !onlyDescs)
                        this.HandleTransactionInfo();
                    let preTskSuffix = succeeded;
                    if (succeeded && !onlyDescs && reversibleExit && !dueToVerifyError &&
                        (!this.IsSubForm || subformDestination) &&
                        this.TaskPrefixExecuted && !this.TaskSuffixExecuted)
                        succeeded = await this.handleTaskSuffix(true);
                    if (reversibleExit && this.hasSubTasks())
                        for (i = 0; i < this.SubTasks.getSize(); i++) {
                            subTask = this.SubTasks.getTask(i);
                            if (!subTask.IsSubForm)
                                await AccessHelper.eventsManager.handleNonReversibleEvent(subTask, InternalInterface.MG_ACT_EXIT);
                        }
                    if (reversibleExit)
                        succeeded = (!AccessHelper.eventsManager.GetStopExecutionFlag() || dueToVerifyError);
                    if (preTskSuffix)
                        AccessHelper.eventsManager.setStopExecution(false);
                    this.setAfterRetry(ConstInterface.RECOVERY_NONE);
                    if (!this.isAborting() && succeeded && !onlyDescs) {
                        if (reversibleExit) {
                            this.setTransactionFailed(false);
                            await this.TaskTransactionManager.checkAndCommit(false, ConstInterface.TRANS_TASK_PREFIX, false);
                            succeeded = (!AccessHelper.eventsManager.GetStopExecutionFlag() || dueToVerifyError) && !this.isAborting() && !this.transactionFailed(ConstInterface.TRANS_TASK_PREFIX);
                        }
                        this.setAfterRetry(ConstInterface.RECOVERY_NONE);
                        if (succeeded) {
                            if (!this.allowRemoteCmdExecution) {
                                this.allowRemoteCmdExecution = true;
                                RemoteCommandsProcessor.GetInstance().DelayCommandExecution = false;
                            }
                            else if (this.DelayCommandExecution) {
                                this.DelayCommandExecution = false;
                                RemoteCommandsProcessor.GetInstance().DelayCommandExecution = true;
                            }
                            let baseMgd = AccessHelper.mgDataTable.StartupMgData;
                            if (reversibleExit && baseMgd.getFirstTask().getTaskTag() === this.getTaskTag()) {
                                AccessHelper.eventsManager.setProcessingTopMostEndTask(true);
                                await AccessHelper.eventsManager.handleEvents(baseMgd, 0);
                                AccessHelper.eventsManager.setProcessingTopMostEndTask(false);
                                if (this.isAborting()) {
                                    RemoteCommandsProcessor.GetInstance().DelayCommandExecution = false;
                                    this.DelayCommandExecution = false;
                                    this.allowRemoteCmdExecution = true;
                                    return true;
                                }
                            }
                            await TaskServiceBase.Exit(this, reversibleExit, subformDestination);
                            if (this._parentTask != null && this._parentTask.isMainProg()) {
                                let mgd = AccessHelper.mgDataTable.getMGData(0);
                                let mainPrgTask = mgd.getMainProg(0);
                                let rootTask = AccessHelper.mgDataTable.StartupMgData.getFirstTask();
                                let endMainPrg = await mainPrgTask.checkProp(PropInterface.PROP_TYPE_END_CONDITION, false);
                                if (this !== rootTask && endMainPrg) {
                                    let rtEvt = new RunTimeEvent(rootTask);
                                    rtEvt.setInternal(InternalInterface.MG_ACT_CLOSE);
                                    rtEvt.setCtrl(null);
                                    rtEvt.setArgList(null);
                                    AccessHelper.eventsManager.addToTail(rtEvt);
                                }
                            }
                            MgForm.CreatedForms.remove(this.Form);
                        }
                    }
                }
            }
            finally {
                if (!this.IsSubForm || subformDestination)
                    this.handleFinallyEndTask(succeeded, reversibleExit);
            }
        }
        else if (this.InEndTask)
            succeeded = false;
        AccessHelper.eventsManager.setProcessingTopMostEndTask(this._inProcessingTopMostEndTaskSaved);
        return succeeded;
    }
    SetServerSideTaskSuffix(task) {
        var rtEvt = new RunTimeEvent(task);
        rtEvt.setInternal(InternalInterface.MG_ACT_TASK_SUFFIX);
        let handler = null;
        var evtHanPos = new EventHandlerPosition();
        evtHanPos.init(rtEvt);
        handler = evtHanPos.getNext();
        if (handler != null) {
            for (let i = 0; i < handler.getOperationTab().getSize(); i++) {
                let oper = handler.getOperationTab().getOperation(i);
                oper.setExecOnServer(true);
            }
        }
        if (task.hasSubTasks()) {
            for (let i = 0; i < task.GetSubTasks().length; i++) {
                let subTask = task.SubTasks.getTask(i);
                this.SetServerSideTaskSuffix(subTask);
            }
        }
    }
    async CheckIfClientOperationExistsInSubformTasks(task, eventCode) {
        let clientOperationExist = false;
        var rtEvt = new RunTimeEvent(task);
        rtEvt.setInternal(eventCode);
        let handler = null;
        var evtHanPos = new EventHandlerPosition();
        evtHanPos.init(rtEvt);
        handler = evtHanPos.getNext();
        if (eventCode == InternalInterface.MG_ACT_REC_SUFFIX) {
            let currRecord = task.DataView.getCurrRec();
            if (task.DataView.modifiedRecordsNumber() > 0 || (currRecord != null && currRecord.Modified))
                clientOperationExist = true;
            if (!clientOperationExist) {
                let forceSuffix = await task.checkProp(PropInterface.PROP_TYPE_FORCE_SUFFIX, false);
                if (forceSuffix === true)
                    if (task.IsInteractive || task.getMode() !== Constants.TASK_MODE_CREATE)
                        clientOperationExist = true;
            }
        }
        if (handler != null && !clientOperationExist) {
            let cnt = handler.getOperationTab().getSize();
            for (let i = 0; i < cnt; i++) {
                let oper = handler.getOperationTab().getOperation(i);
                if (oper.getExecOnClient()) {
                    clientOperationExist = true;
                    break;
                }
                else {
                    if (oper.getType() == ConstInterface.MG_OPER_RAISE_EVENT) {
                        if (oper.GetRuntimeEvent().getType() == ConstInterface.EVENT_TYPE_INTERNAL &&
                            oper.GetRuntimeEvent().getInternalCode() === InternalInterface.MG_ACT_RT_REFRESH_VIEW) {
                            clientOperationExist = true;
                            break;
                        }
                    }
                }
            }
        }
        if (!clientOperationExist && task.hasSubTasks()) {
            for (let i = 0; (!clientOperationExist && i < task.getSubTasks().getSize()); i++) {
                let subTask = task.SubTasks.getTask(i);
                clientOperationExist = await this.CheckIfClientOperationExistsInSubformTasks(subTask, eventCode);
            }
        }
        return clientOperationExist;
    }
    CheckIfPendingTransactions(task) {
        let pendingTransactions = false;
        let currRecord = task.DataView.getCurrRec();
        if (task.DataView.modifiedRecordsNumber() > 0 || (currRecord != null && currRecord.Modified))
            pendingTransactions = true;
        if (!pendingTransactions && task.hasSubTasks()) {
            for (let i = 0; (!pendingTransactions && i < task.getSubTasks().getSize()); i++) {
                let subTask = task.SubTasks.getTask(i);
                pendingTransactions = this.CheckIfPendingTransactions(subTask);
            }
        }
        return pendingTransactions;
    }
    HandleTransactionInfo() {
        let transaction = this.DataviewManager.RemoteDataviewManager.Transaction;
        if (transaction !== null && transaction.isOwner(this) && transaction.getLevel() === 'P') {
            this.DataviewManager.RemoteDataviewManager.Transaction = null;
            this.DataView.setTransCleared();
        }
    }
    async Exit(reversibleExit, subformDestination) {
        let cmd = reversibleExit ?
            CommandFactory.CreateBrowserEscEventCommand(this.getTaskTag(), subformDestination) :
            CommandFactory.CreateNonReversibleExitCommand(this.getTaskTag(), subformDestination);
        this._mgData.CmdsToServer.Add(cmd);
        if (!subformDestination) {
            try {
                if (this.getTaskTag() == MGDataCollection.Instance.StartupMgData.getFirstTask().getTaskTag() ||
                    !(MGDataCollection.Instance.StartupMgData.getFirstTask().InEndTask && MGDataCollection.Instance.StartupMgData.getFirstTask().ClosingFormUI))
                    await Task.CommandsProcessor.Execute(CommandsProcessorBase_SendingInstruction.TASKS_AND_COMMANDS);
            }
            catch (err) {
                if (this.isMainProg()) {
                    this._mgData.CmdsToServer.Add(cmd);
                }
                else
                    throw err;
            }
        }
    }
    handleFinallyEndTask(succeeded, reversibleExit) {
        let subTask;
        if (this.hasSubTasks()) {
            for (let i = 0; i < this.SubTasks.getSize(); i++) {
                subTask = this.SubTasks.getTask(i);
                subTask.handleFinallyEndTask(succeeded, reversibleExit);
            }
        }
        this.InEndTask = false;
        if (succeeded)
            this._isStarted = false;
        if (!succeeded && reversibleExit && this.IsTryingToStop)
            this.setTryingToStop(false);
    }
    async handleTaskSuffix(withSubTasks) {
        let succeeded = true;
        let subTask;
        if (withSubTasks) {
            if (this.hasSubTasks()) {
                for (let i = 0; i < this.SubTasks.getSize() && succeeded; i++) {
                    subTask = this.SubTasks.getTask(i);
                    succeeded = await subTask.handleTaskSuffix(withSubTasks);
                }
            }
        }
        if (succeeded)
            await AccessHelper.eventsManager.handleInternalEventWithTask(this, InternalInterface.MG_ACT_TASK_SUFFIX);
        succeeded = !AccessHelper.eventsManager.GetStopExecutionFlag();
        this.TaskSuffixExecuted = true;
        return succeeded;
    }
    setInCreateLine(inCreateLine) {
        this._inCreateLine = inCreateLine;
    }
    getInCreateLine() {
        return this._inCreateLine;
    }
    async handleEventOnSlaveTasks(internalEvtCode) {
        let i;
        let task = null;
        let failureTask = null;
        let mgdTab = AccessHelper.mgDataTable;
        let triggeredTasks = mgdTab.getTriggeredTasks(this);
        let orphans = null;
        let sameTrans;
        let hasSlaves = false;
        let isOwner = this.Transaction != null && this.Transaction.isOwner(this);
        if (isOwner) {
            AccessHelper.eventsManager.setEventScopeTrans();
            orphans = new List();
            mgdTab.startTasksIteration();
            while ((task = mgdTab.getNextTask()) != null) {
                sameTrans = task.Transaction != null && this.Transaction.getTransId() === task.Transaction.getTransId();
                if (task !== this && sameTrans && task.PreviouslyActiveTaskId != null && mgdTab.GetTaskByID(task.PreviouslyActiveTaskId) == null)
                    orphans.push(task);
            }
        }
        let scopeTrans = AccessHelper.eventsManager.isEventScopeTrans();
        for (i = 0; i < triggeredTasks.length && !AccessHelper.eventsManager.GetStopExecutionFlag(); i++) {
            task = triggeredTasks.get_Item(i);
            sameTrans = this.Transaction != null && task.Transaction != null && this.Transaction.getTransId() === task.Transaction.getTransId();
            if (!task.IsSubForm && (scopeTrans && sameTrans) && task.isStarted()) {
                if (task.getForm().FormRefreshed && internalEvtCode === InternalInterface.MG_ACT_REC_PREFIX && task.getLevel() === Constants.TASK_LEVEL_TASK)
                    await task.DataView.currRecCompute(false);
                await AccessHelper.eventsManager.handleInternalEventWithTask(task, internalEvtCode);
                hasSlaves = true;
            }
        }
        for (; isOwner && i < orphans.length && !AccessHelper.eventsManager.GetStopExecutionFlag(); i++) {
            task = orphans.get_Item(i);
            if (task.getForm().FormRefreshed && internalEvtCode === InternalInterface.MG_ACT_REC_PREFIX && task.getLevel() === Constants.TASK_LEVEL_TASK)
                await task.DataView.currRecCompute(false);
            await AccessHelper.eventsManager.handleInternalEventWithTask(task, internalEvtCode);
            hasSlaves = true;
        }
        if (hasSlaves && AccessHelper.eventsManager.GetStopExecutionFlag())
            failureTask = task;
        if (isOwner)
            AccessHelper.eventsManager.clearEventScope();
        return failureTask;
    }
    async enableRecordActions() {
        let dataview = this.DataView;
        await this.setCreateDeleteActsEnableState();
        if (dataview.IncludesFirst() || dataview.IsOneWayKey) {
            if (dataview.isEmptyDataview() || dataview.getCurrRecIdx() === 0) {
                this.ActionManager.enable(InternalInterface.MG_ACT_TBL_PRVLINE, false);
                if (dataview.IsOneWayKey)
                    this.ActionManager.enable(InternalInterface.MG_ACT_TBL_BEGTBL, true);
                else
                    this.ActionManager.enable(InternalInterface.MG_ACT_TBL_BEGTBL, false);
                if (this.Form.isScreenMode())
                    this.ActionManager.enable(InternalInterface.MG_ACT_TBL_PRVPAGE, false);
            }
            else {
                this.ActionManager.enable(InternalInterface.MG_ACT_TBL_PRVLINE, true);
                this.ActionManager.enable(InternalInterface.MG_ACT_TBL_BEGTBL, true);
                if (this.Form.isScreenMode())
                    this.ActionManager.enable(InternalInterface.MG_ACT_TBL_PRVPAGE, true);
            }
            if (!this.Form.isScreenMode())
                if (dataview.isEmptyDataview() || (dataview.getCurrRecIdx() === 0) || (dataview.getTopRecIdx() === 0))
                    this.ActionManager.enable(InternalInterface.MG_ACT_TBL_PRVPAGE, false);
                else
                    this.ActionManager.enable(InternalInterface.MG_ACT_TBL_PRVPAGE, true);
        }
        else {
            this.ActionManager.enable(InternalInterface.MG_ACT_TBL_BEGTBL, true);
            this.ActionManager.enable(InternalInterface.MG_ACT_TBL_PRVPAGE, true);
            this.ActionManager.enable(InternalInterface.MG_ACT_TBL_PRVLINE, true);
        }
        if (dataview.IncludesLast()) {
            if (((this.getMode() === Constants.TASK_MODE_QUERY) || dataview.isEmptyDataview() ||
                ((this.getMode() === Constants.TASK_MODE_MODIFY) &&
                    (!Environment.Instance.allowCreateInModifyMode(this.getCompIdx()) ||
                        !await this.checkProp(PropInterface.PROP_TYPE_ALLOW_CREATE, true)))) && (dataview.getCurrRecIdx() + 1) === dataview.getSize()) {
                this.ActionManager.enable(InternalInterface.MG_ACT_TBL_NXTLINE, false);
                this.ActionManager.enable(InternalInterface.MG_ACT_TBL_ENDTBL, false);
                if (this.Form.isScreenMode())
                    this.ActionManager.enable(InternalInterface.MG_ACT_TBL_NXTPAGE, false);
            }
            else {
                this.ActionManager.enable(InternalInterface.MG_ACT_TBL_NXTLINE, true);
                this.ActionManager.enable(InternalInterface.MG_ACT_TBL_ENDTBL, true);
                if (this.Form.isScreenMode())
                    this.ActionManager.enable(InternalInterface.MG_ACT_TBL_NXTPAGE, true);
            }
            if (!this.Form.isScreenMode()) {
                if (dataview.isEmptyDataview() || ((dataview.getSize() - dataview.getTopRecIdx()) <= this.Form.getRowsInPage()))
                    this.ActionManager.enable(InternalInterface.MG_ACT_TBL_NXTPAGE, false);
                else
                    this.ActionManager.enable(InternalInterface.MG_ACT_TBL_NXTPAGE, true);
            }
        }
        else {
            this.ActionManager.enable(InternalInterface.MG_ACT_TBL_NXTLINE, true);
            this.ActionManager.enable(InternalInterface.MG_ACT_TBL_NXTPAGE, true);
            this.ActionManager.enable(InternalInterface.MG_ACT_TBL_ENDTBL, true);
        }
        if (dataview.isEmptyDataview())
            this.ActionManager.enable(InternalInterface.MG_ACT_CANCEL, false);
        else
            this.ActionManager.enable(InternalInterface.MG_ACT_CANCEL, true);
        this.ActionManager.enable(InternalInterface.MG_ACT_ENTER_ROW_EDITING, true);
        this.ActionManager.enable(InternalInterface.MG_ACT_NO_PROG_EXE_RIGHTS, true);
    }
    isStarted() {
        return this._isStarted;
    }
    getSubTasks() {
        return this.SubTasks;
    }
    refreshesOn(fldId) {
        let result = false;
        for (let i = 0; !result && this.RefreshOnVars !== null && i < this.RefreshOnVars.length; i++) {
            result = (this.RefreshOnVars[i] === fldId);
        }
        return result;
    }
    AbortTransaction() {
        this.DataviewManager.RemoteDataviewManager.Transaction = null;
    }
    setPreventRecordSuffix(val) {
        this._preventRecordSuffix = val;
    }
    getPreventRecordSuffix() {
        return this._preventRecordSuffix;
    }
    setPreventControlChange(val) {
        this._preventControlChange = val;
    }
    getPreventControlChange() {
        return this._preventControlChange;
    }
    resetKnownToServer() {
        this._knownToServer = false;
    }
    evaluateDescriptor() {
        let result = "";
        let res;
        if (this.DvPosDescriptor == null)
            return HashUtils.GetHashCode(result);
        let first = true;
        for (let i = 0; i < this.DvPosDescriptor.length; i++) {
            if (!first)
                result += ";";
            let currentCell = this.DvPosDescriptor.get_Item(i);
            let currentTag = currentCell[0];
            let currentVeeIdx = NNumber.Parse(currentCell[1]);
            let currentTask = AccessHelper.mgDataTable.GetTaskByID(currentTag);
            let currentField = currentTask.DataView.getField(currentVeeIdx);
            let dispVal = currentField.getDispValue();
            if (currentField.isNull())
                dispVal = "";
            let pic;
            switch (currentField.getType()) {
                case StorageAttribute.BOOLEAN:
                case StorageAttribute.ALPHA:
                case StorageAttribute.UNICODE:
                case StorageAttribute.BLOB:
                case StorageAttribute.BLOB_VECTOR:
                    result += dispVal;
                    break;
                case StorageAttribute.DATE:
                case StorageAttribute.TIME:
                    if (dispVal !== "") {
                        pic = new PIC(currentField.getPicture(), currentField.getType(), 0);
                        let conv = DisplayConvertor.Instance;
                        result += conv.mg2disp(dispVal, "", pic, false, this.getCompIdx(), false);
                    }
                    break;
                case StorageAttribute.NUMERIC:
                    if (dispVal !== "") {
                        res = new NUM_TYPE(dispVal);
                        pic = new PIC(currentField.getPicture(), currentField.getType(), 0);
                        result += res.to_a(pic);
                    }
                    break;
            }
            first = false;
        }
        let tmpStr2 = StrUtil.stringToHexaDump(result, 4);
        return HashUtils.GetHashCode(tmpStr2);
    }
    setDescriptor(newDesc) {
        if (!(newDesc.trim() === "")) {
            let desc = StrUtil.tokenize(newDesc, ";");
            this.DvPosDescriptor = new List();
            for (let i = 0; i < desc.length; i = i + 1) {
                let cell = new Array(2);
                let tagVeePair = StrUtil.tokenize(desc[i], ",");
                cell[0] = tagVeePair[0];
                cell[1] = tagVeePair[1];
                this.DvPosDescriptor.push(cell);
            }
        }
    }
    getTaskCache() {
        return this._dvCache;
    }
    async prepareCache(ignoreCurr) {
        let success = true;
        if (!ignoreCurr) {
            if (await this.isCached()) {
                if (this.DataView.getChanged()) {
                    this._dvCache.removeDvFromCache(this.DataView.getDvPosValue(), true);
                    success = false;
                }
            }
            else {
                if (this.IsSubForm && !await this.isCached()) {
                    success = false;
                }
            }
        }
        let curretTask;
        if (this.hasSubTasks()) {
            for (let i = 0; i < this.SubTasks.getSize(); i = i + 1) {
                curretTask = this.SubTasks.getTask(i);
                if (!await curretTask.prepareCache(false)) {
                    success = false;
                }
            }
        }
        return success;
    }
    async testAndSet(ignoreCurr) {
        if (!ignoreCurr && await this.isCached()) {
            let requestedDv = this.evaluateDescriptor();
            if (requestedDv !== this.DataView.getDvPosValue()) {
                let fromCache = this._dvCache.getCachedDataView(requestedDv);
                if (fromCache == null)
                    return false;
                await this.changeDv(fromCache);
            }
            this.DoSubformPrefixSuffix = true;
        }
        if (this.hasSubTasks()) {
            for (let i = 0; i < this.SubTasks.getSize(); i++) {
                if (!await this.SubTasks.getTask(i).testAndSet(false))
                    return false;
            }
        }
        return true;
    }
    async changeDv(fromCache) {
        let dataview = this.DataView;
        if (!this.hasLocate) {
            if (!dataview.getChanged() && dataview.IncludesFirst())
                this._dvCache.putInCache(dataview.replicate());
        }
        else
            this.locatePutInCache();
        await dataview.setSameAs(fromCache);
        dataview.takeFldValsFromCurrRec();
        this.Form.SetTableItemsCount(0, true);
        this.Form.setTableItemsCount(false);
        await this.Form.RefreshDisplay(Constants.TASK_REFRESH_FORM);
        this.setOriginalTaskMode(this.getMode());
    }
    setOriginalTaskMode(originalTaskMode) {
        this._originalTaskMode = originalTaskMode.charAt(0);
    }
    getOriginalTaskMode() {
        return this._originalTaskMode;
    }
    async isCached() {
        return this.IsSubForm && await this.Form.getSubFormCtrl().checkProp(PropInterface.PROP_TYPE_IS_CACHED, false);
    }
    locatePutInCache() {
        let cached = this._dvCache.getCachedDataView(this.DataView.getDvPosValue());
        if (cached != null) {
            if (this.DataView.checkFirst(cached.getLocateFirstRec()) && !this.DataView.getChanged()) {
                this.DataView.setLocateFirstRec(cached.getLocateFirstRec());
                this._dvCache.putInCache(this.DataView.replicate());
            }
            else
                this._dvCache.removeDvFromCache(this.DataView.getDvPosValue(), true);
        }
        else {
            if (!this.DataView.getChanged())
                this._dvCache.putInCache(this.DataView.replicate());
        }
    }
    HasLoacte() {
        return this.hasLocate;
    }
    setRemoteTransaction(transId) {
        let mgdTab = AccessHelper.mgDataTable;
        let transTask;
        let isFind = false;
        mgdTab.startTasksIteration();
        while ((transTask = mgdTab.getNextTask()) != null) {
            if (transTask.DataviewManager.RemoteDataviewManager.Transaction != null &&
                transTask.DataviewManager.RemoteDataviewManager.Transaction.getTransId() === transId) {
                isFind = true;
                break;
            }
        }
        this.DataviewManager.RemoteDataviewManager.Transaction = (isFind ?
            transTask.DataviewManager.RemoteDataviewManager.Transaction :
            new Transaction(this, transId));
    }
    setTransOwnerTask() {
        this.Transaction.setOwnerTask(this);
        this.Transaction.setTransBegin('T');
    }
    getLoopCounter() {
        let result = 0;
        if (this._useLoopStack && this._loopStack !== null && !(this._loopStack.count() === 0)) {
            result = this._loopStack.peek();
        }
        return result;
    }
    enterLoop() {
        if (this._loopStack === null)
            this._loopStack = new Stack();
        this._loopStack.push(0);
    }
    leaveLoop() {
        this.popLoopCounter();
    }
    popLoopCounter() {
        let result = 0;
        if (this._loopStack !== null && !(this._loopStack.count() === 0))
            result = this._loopStack.pop();
        return result;
    }
    increaseLoopCounter() {
        this._loopStack.push(this.popLoopCounter() + 1);
    }
    getLoopStackSize() {
        if (this._loopStack !== null)
            return this._loopStack.count();
        return 0;
    }
    setUseLoopStack(val) {
        this._useLoopStack = val;
    }
    getDataviewHeaders() {
        return this.DataviewHeadersTable;
    }
    isFirstRecordCycle(Value_) {
        if (arguments.length === 0) {
            return this.isFirstRecordCycle_0();
        }
        return this.isFirstRecordCycle_1(Value_);
    }
    isFirstRecordCycle_0() {
        return this._firstRecordCycle;
    }
    isFirstRecordCycle_1(Value_) {
        this._firstRecordCycle = Value_;
    }
    getPublicName() {
        return this.PublicName;
    }
    async isAllowEvents() {
        let prop = this.getProp(PropInterface.PROP_TYPE_TASK_PROPERTIES_ALLOW_EVENTS);
        return await prop.getValueBoolean();
    }
    getExternalTaskId() {
        let taskId = "";
        let prop = this.getProp(PropInterface.PROP_TYPE_TASK_ID);
        if (prop != null)
            taskId = prop.GetComputedValue();
        return taskId;
    }
    isTransactionOwner() {
        let isTransOwner = false;
        if (this.Transaction != null && this.Transaction.isOwner(this))
            isTransOwner = true;
        return isTransOwner;
    }
    isTransactionOnLevel(level) {
        return this.isTransactionOwner() && this.Transaction.getLevel() === level;
    }
    isProgram() {
        return this._isPrg;
    }
    async enableModes() {
        let enable = await this.checkProp(PropInterface.PROP_TYPE_ALLOW_MODIFY, true);
        this.ActionManager.enable(InternalInterface.MG_ACT_RTO_MODIFY, enable);
        enable = await this.checkProp(PropInterface.PROP_TYPE_ALLOW_CREATE, true);
        this.ActionManager.enable(InternalInterface.MG_ACT_RTO_CREATE, enable);
        enable = await this.checkProp(PropInterface.PROP_TYPE_ALLOW_QUERY, true);
        this.ActionManager.enable(InternalInterface.MG_ACT_RTO_QUERY, enable);
        enable = await this.checkProp(PropInterface.PROP_TYPE_ALLOW_RANGE, true);
        this.ActionManager.enable(InternalInterface.MG_ACT_RTO_RANGE, enable);
        enable = await this.checkProp(PropInterface.PROP_TYPE_ALLOW_LOCATE, true);
        this.ActionManager.enable(InternalInterface.MG_ACT_RTO_LOCATE, enable);
        this.ActionManager.enable(InternalInterface.MG_ACT_RTO_SEARCH, enable);
        enable = await this.checkProp(PropInterface.PROP_TYPE_ALLOW_SORT, true);
        this.ActionManager.enable(InternalInterface.MG_ACT_SORT_RECORDS, enable);
        enable = await this.checkProp(PropInterface.PROP_TYPE_TASK_PROPERTIES_ALLOW_INDEX, true);
        this.ActionManager.enable(InternalInterface.MG_ACT_VIEW_BY_KEY, enable);
        this.ActionManager.enable(InternalInterface.MG_ACT_DUMP_ENVIRONMENT, true);
    }
    setEnableZoomHandler() {
        this._enableZoomHandler = true;
    }
    getEnableZoomHandler() {
        return this._enableZoomHandler;
    }
    isDestinationSubform() {
        return this._destinationSubform;
    }
    setDestinationSubform(destinationSubform) {
        this._destinationSubform = destinationSubform;
    }
    getIsDestinationCall() {
        return this._isDestinationCall;
    }
    setIsDestinationCall(isDestinationCall) {
        this._isDestinationCall = isDestinationCall;
    }
    setCancelWasRaised(val) {
        this._cancelWasRaised = val;
    }
    cancelWasRaised() {
        return this._cancelWasRaised;
    }
    enableCreateActs(val) {
        this.ActionManager.enable(InternalInterface.MG_ACT_CRELINE, val);
        this.ActionManager.enable(InternalInterface.MG_ACT_CRELINE_ABOVE, (this.getForm().HasTable()) ? val : false);
    }
    async setCreateDeleteActsEnableState() {
        if (!(this.DataView).isEmptyDataview() && await this.checkProp(PropInterface.PROP_TYPE_ALLOW_DELETE, true) &&
            this.getMode() === Constants.TASK_MODE_MODIFY && this.DataView.HasMainTable)
            this.ActionManager.enable(InternalInterface.MG_ACT_DELLINE, true);
        else
            this.ActionManager.enable(InternalInterface.MG_ACT_DELLINE, false);
        if ((await this.checkProp(PropInterface.PROP_TYPE_ALLOW_CREATE, true)) &&
            (this.DataView.HasMainTable &&
                (this.getMode() === Constants.TASK_MODE_MODIFY && Environment.Instance.allowCreateInModifyMode(this.getCompIdx())) ||
                (this.getMode() === Constants.TASK_MODE_CREATE)))
            this.enableCreateActs(true);
        else
            this.enableCreateActs(false);
    }
    setEmptyDataview(emptyDataview) {
        this.DataView.setEmptyDataview(emptyDataview);
    }
    getName() {
        return this.Name;
    }
    async emptyDataviewOpen(subformRefresh) {
        if (this.getLevel() === Constants.TASK_LEVEL_TASK) {
            this.ActionManager.enable(InternalInterface.MG_ACT_EMPTY_DATAVIEW, true);
            if (this.IsSubForm) {
                let rtEvnt = AccessHelper.eventsManager.getLastRtEvent();
                if ((rtEvnt != null && rtEvnt.getInternalCode() == InternalInterface.MG_ACT_RT_REFRESH_VIEW) || subformRefresh) {
                    let rtEvt = new RunTimeEvent(this);
                    rtEvt.setInternal(InternalInterface.MG_ACT_EMPTY_DATAVIEW);
                    rtEvt.SetEventSubType(EventSubType.Normal);
                    await AccessHelper.eventsManager.handleEvent(rtEvt, false);
                }
            }
            else
                AccessHelper.eventsManager.addInternalEventWithItaskAndCode(this, InternalInterface.MG_ACT_EMPTY_DATAVIEW);
            this.setLevel(Constants.TASK_LEVEL_RECORD);
            if (this.SubformExecMode !== Task_SubformExecModeEnum.FIRST_TIME && subformRefresh && await this.CheckRefreshSubTasks()) {
                await this.DataView.computeSubForms();
                await this.doSubformRecPrefixSuffix();
            }
            await this.enableRecordActions();
            this.ActionManager.enable(InternalInterface.MG_ACT_RT_REFRESH_RECORD, false);
            this.ActionManager.enable(InternalInterface.MG_ACT_RT_REFRESH_SCREEN, false);
        }
    }
    emptyDataviewClose() {
        this.ActionManager.enable(InternalInterface.MG_ACT_EMPTY_DATAVIEW, false);
        this.setLevel(Constants.TASK_LEVEL_TASK);
        this.ActionManager.enable(InternalInterface.MG_ACT_RT_REFRESH_RECORD, true);
        this.ActionManager.enable(InternalInterface.MG_ACT_RT_REFRESH_SCREEN, true);
    }
    async SubformRefresh(subformTask, explicitSubformRefresh) {
        let currTask = LastFocusedManager.Instance.getLastFocusedTask();
        let isInside = await subformTask.ExecuteNestedRS(currTask);
        if (AccessHelper.eventsManager.GetStopExecutionFlag())
            return;
        if (subformTask.getLevel() !== Constants.TASK_LEVEL_TASK && !AccessHelper.eventsManager.GetStopExecutionFlag()) {
            AccessHelper.eventsManager.pushNewExecStacks();
            await AccessHelper.eventsManager.handleInternalEventWithTaskAndSubformRefresh(subformTask, InternalInterface.MG_ACT_REC_SUFFIX, true);
            AccessHelper.eventsManager.popNewExecStacks();
            isInside = true;
            if (AccessHelper.eventsManager.GetStopExecutionFlag())
                return;
        }
        let cmd = CommandFactory.CreateSubformRefreshCommand(this.getTaskTag(), subformTask.getTaskTag(), explicitSubformRefresh);
        await subformTask.DataviewManager.Execute(cmd);
        if (subformTask.isAborting())
            return;
        if (explicitSubformRefresh) {
            AccessHelper.eventsManager.pushNewExecStacks();
            let performParentRecordPrefixOrg = subformTask.PerformParentRecordPrefix;
            subformTask.PerformParentRecordPrefix = false;
            await AccessHelper.eventsManager.handleInternalEventWithTaskAndSubformRefresh(subformTask, InternalInterface.MG_ACT_REC_PREFIX, true);
            subformTask.PerformParentRecordPrefix = performParentRecordPrefixOrg;
            AccessHelper.eventsManager.popNewExecStacks();
            if (AccessHelper.eventsManager.GetStopExecutionFlag())
                return;
        }
        if (isInside) {
        }
        else {
            if (!AccessHelper.eventsManager.GetStopExecutionFlag()) {
                if (explicitSubformRefresh) {
                    await AccessHelper.eventsManager.handleInternalEventWithTask(subformTask, InternalInterface.MG_ACT_REC_SUFFIX);
                    if (currTask != null)
                        currTask.DataView.setPrevCurrRec();
                    subformTask.DoSubformPrefixSuffix = false;
                }
            }
        }
    }
    async ExecuteNestedRS(lastTask) {
        let isExecuteNestedRS = false;
        if (lastTask == null)
            return isExecuteNestedRS;
        if (this.getLevel() !== Constants.TASK_LEVEL_TASK && lastTask.getMgdID() !== this.getMgdID())
            return isExecuteNestedRS;
        for (let task = lastTask; task !== this && task.pathContains(this) && task.getLevel() !== Constants.TASK_LEVEL_TASK; task = task.getParent()) {
            AccessHelper.eventsManager.pushNewExecStacks();
            await AccessHelper.eventsManager.handleInternalEventWithTask(task, InternalInterface.MG_ACT_REC_SUFFIX);
            AccessHelper.eventsManager.popNewExecStacks();
            isExecuteNestedRS = true;
        }
        return isExecuteNestedRS;
    }
    async execSubformRecSuffix(newTask) {
        let subfTask;
        let i, idx = 0;
        for (i = this._taskPath.length; i > 0; i--) {
            subfTask = this._taskPath.get_Item(i - 1);
            if (!newTask._taskPath.Contains(subfTask)) {
                await AccessHelper.eventsManager.handleInternalEventWithTask(subfTask, InternalInterface.MG_ACT_REC_SUFFIX);
                if (AccessHelper.eventsManager.GetStopExecutionFlag())
                    break;
                if (subfTask.getMode() === Constants.TASK_MODE_CREATE && !(subfTask.DataView.getCurrRec()).isNewRec())
                    subfTask.setMode(Constants.TASK_MODE_MODIFY);
            }
            else {
                idx = newTask._taskPath.indexOf(subfTask);
                break;
            }
        }
        Debug.Assert(i > 0);
        return idx;
    }
    getCounter() {
        return this._counter;
    }
    increaseCounter() {
        this._counter = this._counter + 1;
    }
    setCounter(cnt) {
        this._counter = cnt;
    }
    isCurrentStartProgLevel() {
        return this._currStartProgLevel === Task.StartProgLevel;
    }
    static isMainProgramField(fieldStr) {
        let Field = XmlParser.getTokens(fieldStr, ',');
        let mainProgField = false;
        if (Field.length === 2) {
            let parent = NNumber.Parse(Field.get_Item(0));
            if (parent === TaskBase.MAIN_PRG_PARENT_ID)
                mainProgField = true;
        }
        return mainProgField;
    }
    getTaskInfo() {
        let taskName = (this.getProp(PropInterface.PROP_TYPE_NAME)).GetComputedValue();
        return NString.Format('{0} {1}', this.isProgram() ? 'Program' : 'Subtask', taskName);
    }
    toString() {
        return NString.Format("(task {0}-\"{1}\"{2})", this._taskTag, this.Name, super.isMainProg() ? (" ctl " + this._ctlIdx) : "");
    }
    ConstructMgForm(alreadySetParentForm) {
        let form = new MgForm();
        let parentForm = (this.ParentTask != null ? this.ParentTask.getForm() : null);
        while (parentForm != null && !parentForm.getTask().isOpenWin()) {
            parentForm = parentForm.ParentForm;
        }
        form.ParentForm = parentForm;
        alreadySetParentForm.value = true;
        return form;
    }
    getMgdID() {
        return this._mgData.GetId();
    }
    getMGData() {
        return this._mgData;
    }
    async RefreshDisplay() {
        if (this.Form !== null && !this.isAborting() && this.isStarted()) {
            Logger.Instance.WriteDevToLog("Start task.RefreshDisplay()");
            if (this._refreshType !== Constants.TASK_REFRESH_NONE) {
                let form = this.Form;
                let wasRefreshed = await form.RefreshDisplay(Constants.TASK_REFRESH_FORM);
                if (wasRefreshed) {
                    this._refreshType = Constants.TASK_REFRESH_NONE;
                }
            }
            Logger.Instance.WriteDevToLog("End task.RefreshDisplay()");
        }
    }
    getFieldDisplayValue(fieldDef, value, isNull) {
        value.value = fieldDef.getDispValue();
        isNull.value = fieldDef.isNull();
    }
    async UpdateFieldValueAndStartRecompute(fieldDef, value, isNull) {
        if (this.TaskPrefixExecuted) {
            await fieldDef.setValueAndStartRecompute(value, isNull, true, false, false);
        }
    }
    ShouldNonInteractiveChildBeModal() {
        return this.ShouldChildBeModal();
    }
    ShouldChildBeModal() {
        let childShouldBehaveAsModal = false;
        if ((this.Form.ConcreteWindowType === WindowType.Overlay) && this.getForm().Opened)
            childShouldBehaveAsModal = true;
        else if (!this.IsInteractive) {
            childShouldBehaveAsModal = true;
        }
        return childShouldBehaveAsModal;
    }
    getParent() {
        return this._parentTask;
    }
    isDescendentOf(chkTask) {
        if (chkTask === this)
            return true;
        else if (chkTask === null || this._parentTask === null)
            return false;
        return this._parentTask.isDescendentOf(chkTask);
    }
    GetSubformModalParentTask() {
        let currTask = this;
        while (currTask._parentTask != null && currTask._parentTask.getIsDestinationCall()) {
            let subformCtrl = (currTask._parentTask.getForm().getSubFormCtrl());
            if (subformCtrl.IsModal()) {
                return subformCtrl;
            }
            currTask = currTask._parentTask;
        }
        return null;
    }
    ShouldEvaluatePropertyLocally(propId) {
        return this.TaskService.ShouldEvaluatePropertyLocally(propId);
    }
    EnsureValidForm() {
        if (this._isMainPrg && this.Form !== null) {
            if (!this._openWin || !this.Form.isLegalForm) {
                this.Form = null;
            }
        }
    }
    async ComputeMainDisplay() {
        let propMainDisplay = this.getProp(PropInterface.PROP_TYPE_MAIN_DISPLAY);
        let mainDisplayIndex = await propMainDisplay.getValueInt();
        mainDisplayIndex = this.GetRealMainDisplayIndexOnCurrentTask(mainDisplayIndex);
        this._forms.InitFormFromXmlString(mainDisplayIndex);
        this.EnsureValidForm();
    }
    async PrepareTaskForm() {
        await TaskServiceBase.PreparePropOpenTaskWindow(this);
        return await RemoteTaskService.PreparePropMainDisplay(this);
    }
    FormIsLegal() {
        let isFormIsLegal = super.isMainProg();
        return isFormIsLegal || super.getForm().isLegalForm;
    }
    GetRealMainDisplayIndexOnDepth(mainDspIdx) {
        let task = this.LogicalStudioParentTask;
        while (task != null) {
            mainDspIdx += task._forms.Count;
            task = task.LogicalStudioParentTask;
        }
        return mainDspIdx;
    }
    GetRealMainDisplayIndexOnCurrentTask(mainDspIdx) {
        if (mainDspIdx > 1) {
            let task = this.LogicalStudioParentTask;
            while (task != null) {
                mainDspIdx -= task._forms.Count;
                task = task.LogicalStudioParentTask;
            }
        }
        return mainDspIdx;
    }
    GetAncestorTaskByTaskDefinitionId(findTaskDefinitionId) {
        let currentTask = this;
        if (findTaskDefinitionId.IsMainProgram())
            return GuiDataCollection.MGDataTable.GetMainProgByCtlIdx(this.ContextID, findTaskDefinitionId.CtlIndex);
        while (currentTask != null) {
            if (currentTask.TaskDefinitionId.Equals(findTaskDefinitionId))
                return currentTask;
            currentTask = currentTask._parentTask;
        }
        return null;
    }
    SetDataControlValuesReference(controlId, dcValuesId) {
        let ctrl = super.getForm().getCtrl(controlId);
        ctrl.setDcValId(dcValuesId);
    }
    DisplayMessageToStatusBar(text) {
        let taskToVerifyOn = this.GetContextTask();
        if (taskToVerifyOn.getForm() == null)
            taskToVerifyOn = LastFocusedManager.Instance.getLastFocusedTask();
        taskToVerifyOn.WriteToMessagePane(text);
    }
    isTableWithAbsolutesScrollbar() {
        let isTableWithAbsolutesScrollbar = false;
        if (this.getForm() != null)
            isTableWithAbsolutesScrollbar = this.getForm().isTableWithAbsoluteScrollbar();
        return isTableWithAbsolutesScrollbar;
    }
    GetTaskDetails() {
        let names = (this.getProp(PropInterface.PROP_TYPE_NAME)).GetComputedValue();
        let task = this;
        while (task.ParentTask !== null && !task.ParentTask.isMainProg()) {
            task = task.ParentTask;
            names = (task.getProp(PropInterface.PROP_TYPE_NAME)).GetComputedValue() + "." + names;
        }
        return names;
    }
    static async ProcessTaskURL() {
        let xmlData = RuntimeContextBase.Instance.Parser.getXMLdata();
        let endTaskUrlIdx = xmlData.indexOf(XMLConstants.TAG_CLOSE, RuntimeContextBase.Instance.Parser.getCurrIndex());
        Debug.Assert(endTaskUrlIdx !== -1 && endTaskUrlIdx < xmlData.length);
        let tag = RuntimeContextBase.Instance.Parser.getXMLsubstring(endTaskUrlIdx);
        RuntimeContextBase.Instance.Parser.add2CurrIndex(tag.indexOf(ConstInterface.MG_TAG_TASKURL) + ConstInterface.MG_TAG_TASKURL.length);
        let tokensVector = XmlParser.getTokens(RuntimeContextBase.Instance.Parser.getXMLsubstring(endTaskUrlIdx), XMLConstants.XML_ATTR_DELIM);
        let taskCacheURL = tokensVector.get_Item(1);
        let refListStr = tokensVector.get_Item(3);
        let taskContentOriginal = "";
        if (taskCacheURL.trim() != "") {
            if (taskCacheURL.startsWith("./"))
                taskCacheURL = NString.Replace(taskCacheURL, './', './assets/cache/');
            taskContentOriginal = await Task.CommandsProcessor.GetContent(taskCacheURL, true);
        }
        let taskContentFinal = new StringBuilder(xmlData.substr(0, RuntimeContextBase.Instance.Parser.getCurrIndex() - (ConstInterface.MG_TAG_TASKURL.length + 1)), taskContentOriginal.length);
        let reflist = this.getReflist(refListStr, ';', true, -1);
        let refListIdx = 0;
        let currIdx = taskContentOriginal.indexOf("[:");
        let prevIdx = 0;
        while (currIdx !== -1) {
            taskContentFinal.Append(taskContentOriginal.substr(prevIdx, currIdx - prevIdx));
            taskContentFinal.Append(reflist.get_Item(refListIdx++));
            currIdx += 2;
            prevIdx = currIdx;
            currIdx = taskContentOriginal.indexOf("[:", prevIdx);
        }
        taskContentFinal.Append(taskContentOriginal.substr(prevIdx));
        taskContentFinal.Append(xmlData.substr(xmlData.indexOf(XMLConstants.TAG_CLOSE, endTaskUrlIdx + 1) + 1));
        xmlData = taskContentFinal.ToString();
        Logger.Instance.WriteDevToLog(NString.Format("{0} -------> {1}", taskCacheURL, xmlData));
        RuntimeContextBase.Instance.Parser.setXMLdata(xmlData);
        RuntimeContextBase.Instance.Parser.setCurrIndex(RuntimeContextBase.Instance.Parser.getCurrIndex() - (ConstInterface.MG_TAG_TASKURL.length + 1));
    }
    static getReflist(data, delim, trunc, maxTokens) {
        let currStr = new StringBuilder();
        let list = new List();
        let tokensCnt = 0;
        for (let i = 1; i < data.length; i++) {
            let curr = data.charAt(i);
            if (curr !== '\\' && curr !== delim)
                currStr.Append(curr);
            else if (curr === delim) {
                list.push(currStr.ToString());
                currStr = new StringBuilder();
                tokensCnt++;
                if (tokensCnt === maxTokens)
                    break;
            }
            else if (curr === '\\') {
                if (!trunc && data.charAt(i + 1) !== delim)
                    currStr.Append(data.charAt(i));
                i++;
                currStr.Append(data.charAt(i));
            }
        }
        list.push(currStr.ToString());
        return list;
    }
    static async StartProgram(callByDestSubForm, moveToFirstControl) {
        let orgStopExecution = AccessHelper.eventsManager.GetStopExecutionFlag();
        let nonInteractiveTaskAlreadyExecuted = null;
        let allowEvents = AccessHelper.eventsManager.getAllowEvents();
        AccessHelper.eventsManager.setAllowEvents(EventsAllowedType.ALL);
        Logger.Instance.WriteDevToLog("Start \"startProg\"");
        Task.StartProgLevel++;
        AccessHelper.eventsManager.setStopExecution(false);
        nonInteractiveTaskAlreadyExecuted = await this.ExecuteComponentMainPrograms(callByDestSubForm, moveToFirstControl);
        let nonInteractiveTaskAlreadyExecutedRef = new RefParam(nonInteractiveTaskAlreadyExecuted);
        let currentNonInteractiveTaskRef = new RefParam(null);
        await Task.ExecuteMainProgram(0, callByDestSubForm, moveToFirstControl, nonInteractiveTaskAlreadyExecutedRef, currentNonInteractiveTaskRef, AccessHelper.mgDataTable);
        nonInteractiveTaskAlreadyExecuted = nonInteractiveTaskAlreadyExecutedRef.value;
        if (!Task._idleTimerStarted) {
            GUIManager.Instance.startTimer(AccessHelper.mgDataTable.getMGData(0), Environment.Instance.getIdleTime(0), true);
            Task._idleTimerStarted = true;
        }
        await this.OpenForms(callByDestSubForm);
        await this.DoFirstRecordCycle();
        await this.MoveToFirstControls(callByDestSubForm);
        if (Task.StartProgLevel === 1) {
            let resetAllowEvents = false;
            let canExecuteEvents = true;
            if (nonInteractiveTaskAlreadyExecuted !== null && nonInteractiveTaskAlreadyExecuted.isStarted()) {
                resetAllowEvents = true;
                AccessHelper.eventsManager.setNonInteractiveAllowEvents(await nonInteractiveTaskAlreadyExecuted.isAllowEvents());
                canExecuteEvents = await nonInteractiveTaskAlreadyExecuted.isAllowEvents();
            }
            if (LastFocusedManager.Instance.getLastFocusedTask() !== null && canExecuteEvents) {
                AccessHelper.eventsManager.pushNewExecStacks();
                await AccessHelper.eventsManager.handleEvents(AccessHelper.mgDataTable.getCurrMGData(), 0);
                AccessHelper.eventsManager.popNewExecStacks();
            }
            if (resetAllowEvents)
                AccessHelper.eventsManager.setAllowEvents(allowEvents);
        }
        Task.StartProgLevel--;
        Logger.Instance.WriteDevToLog("End \"startProg\"");
        if (orgStopExecution && !AccessHelper.eventsManager.GetStopExecutionFlag())
            if (orgStopExecution && !AccessHelper.eventsManager.GetStopExecutionFlag())
                AccessHelper.eventsManager.setStopExecution(orgStopExecution);
        return nonInteractiveTaskAlreadyExecuted;
    }
    static async ExecuteComponentMainPrograms(callByDestSubForm, moveToFirstControl) {
        let currentNonInteractiveTask = new RefParam(null);
        let nonInteractiveTaskAlreadyExecuted = new RefParam(null);
        let maxCtlIdx = AccessHelper.mgDataTable.getMGData(0).getMaxCtlIdx();
        for (let i = 1; i <= maxCtlIdx; i = i + 1) {
            let componentMainProgram = AccessHelper.mgDataTable.GetMainProgByCtlIdx(-1, i);
            if (componentMainProgram !== null)
                await Task.ExecuteMainProgram(i, callByDestSubForm, moveToFirstControl, currentNonInteractiveTask, nonInteractiveTaskAlreadyExecuted, AccessHelper.mgDataTable);
        }
        return nonInteractiveTaskAlreadyExecuted.value;
    }
    static async ExecuteMainProgram(CtlIndex, callByDestSubForm, moveToFirstControl, currentNonInteractiveTask, nonInteractiveTaskAlreadyExecuted, mgDataTable) {
        let mainProgramTask = mgDataTable.GetMainProgByCtlIdx(-1, CtlIndex);
        Logger.Instance.WriteDevToLog(NString.Format("  ClientManager.startProg: ctlIdx={0}, task={1}", CtlIndex, mainProgramTask));
        currentNonInteractiveTask.value = (await mainProgramTask.Start(moveToFirstControl, callByDestSubForm));
        if (nonInteractiveTaskAlreadyExecuted.value === null)
            nonInteractiveTaskAlreadyExecuted.value = currentNonInteractiveTask.value;
        else if (currentNonInteractiveTask.value !== null) {
            Debug.Assert(false, "more than 1 non interactive tasks in startProg");
        }
    }
    static async OpenForms(callByDestSubForm) {
        let createdForms = MgForm.CreatedForms;
        if (createdForms.Count() > 0) {
            let i = 0;
            while (i < createdForms.Count()) {
                let mgForm = createdForms.get(i);
                let canOpen = false;
                if (mgForm.isSubForm()) {
                    let opened = mgForm.getSubFormCtrl().getForm().Opened;
                    if (opened) {
                        canOpen = true;
                    }
                }
                else {
                    if (mgForm.getTask().isCurrentStartProgLevel()) {
                        canOpen = true;
                    }
                }
                if (canOpen) {
                    Manager.OpenForm(mgForm);
                    await Manager.DoFirstRefreshTable(mgForm);
                }
                if (mgForm.Opened) {
                    createdForms.remove(mgForm);
                }
                else
                    i++;
            }
        }
    }
    static async DoFirstRecordCycle() {
        for (let i = 0; i < AccessHelper.mgDataTable.getSize(); i++) {
            let mgd = AccessHelper.mgDataTable.getMGData(i);
            if (mgd === null || mgd.IsAborting)
                continue;
            for (let j = 0; j < mgd.getTasksCount(); j++) {
                let task = mgd.getTask(j);
                if (task !== null) {
                    let mgForm = task.getForm();
                    if (mgForm !== null && task.isCurrentStartProgLevel() && !mgForm.IgnoreFirstRecordCycle) {
                        mgForm.IgnoreFirstRecordCycle = true;
                        await task.doFirstRecordCycle();
                    }
                }
            }
        }
    }
    static async MoveToFirstControls(callByDestSubForm) {
        for (let i = 0; i < AccessHelper.mgDataTable.getSize(); i++) {
            let mgd = AccessHelper.mgDataTable.getMGData(i);
            if (mgd === null || mgd.IsAborting)
                continue;
            let tasksCount = mgd.getTasksCount();
            for (let j = 0; j < tasksCount; j++) {
                let task = mgd.getTask(j);
                let mgForm = task.getForm();
                if (task.isCurrentStartProgLevel() && mgForm !== null && mgForm.MovedToFirstControl) {
                    mgForm.MovedToFirstControl = false;
                    if (!callByDestSubForm)
                        await AccessHelper.eventsManager.HandleNonParkableControls(task);
                }
            }
        }
    }
}
Task.StartProgLevel = 0;
Task._idleTimerStarted = false;
//# sourceMappingURL=data:application/json;base64,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