import { BlobType, DisplayConvertor, PIC, VectorType } from '@magic-xpa/gui';
import { DateTime, NString, StringBuilder } from '@magic-xpa/mscorelib';
import { Constants, DateTimeUtils, InternalInterface, Logger, Queue, StorageAttribute, XMLConstants, XmlParser } from '@magic-xpa/utils';
import { FlowMonitorInterface } from '../FlowMonitorInterface';
import { ConstInterface } from '../ConstInterface';
export class FlowMonitorQueue {
    constructor() {
        this._queue = new Queue();
        this._enabled = false;
        this._isFlowOperation = false;
        this._isRecompute = false;
        this._isTask = false;
        this._isTaskFlow = false;
        this._isDataView = false;
        this.ShouldSerialize = false;
        this.padRight = (string, length, character = ' ') => {
            let result = "";
            let totalLength = length - string.length;
            for (let i = 0; i < totalLength; i++) {
                result = character + result;
            }
            result = result + string;
            return result;
        };
    }
    static get Instance() {
        if (FlowMonitorQueue._instance === null)
            FlowMonitorQueue._instance = new FlowMonitorQueue();
        return FlowMonitorQueue._instance;
    }
    addTaskCngMode(newTaskMode, taskInfo) {
        let info = "Task Mode Change - ";
        if (this._enabled && this._isTask) {
            let activityItem = new ActivityItem(this, FlowMonitorQueue.ACT_TASK, FlowMonitorInterface.FLWMTR_CHNG_MODE);
            switch (newTaskMode) {
                case Constants.TASK_MODE_MODIFY:
                    info += "Modify";
                    break;
                case Constants.TASK_MODE_CREATE:
                    info += "Create";
                    break;
                case Constants.TASK_MODE_DELETE:
                    info += "Delete";
                    break;
                case Constants.TASK_MODE_QUERY:
                    info += "Query";
                    break;
                default:
                    info = null;
                    break;
            }
            activityItem.setInfo(info);
            this._queue.put(activityItem);
            Logger.Instance.WriteSupportToLog(taskInfo + ": " + info, true);
        }
    }
    fillData(parser) {
        let endContext = parser.getXMLdata().indexOf("/>", parser.getCurrIndex());
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            let tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf("flwmtr_config") + "flwmtr_config".length);
            let tokens = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            Logger.Instance.WriteDevToLog("in FlowMonitorQueue.FillData: " + tokens.toString());
            this.initElements(tokens);
            parser.setCurrIndex(endContext + XMLConstants.TAG_TERM.length);
            return;
        }
        Logger.Instance.WriteExceptionToLogWithMsg("in  FlowMonitorQueue.FillData() out of string bounds");
    }
    initElements(tokensVector) {
        for (let j = 0; j < tokensVector.length; j = j + 2) {
            let attribute = tokensVector.get_Item(j);
            let valueStr = tokensVector.get_Item(j + 1);
            Logger.Instance.WriteDevToLog(attribute + " value: " + valueStr);
            switch (attribute) {
                case ConstInterface.MG_ATTR_TASK:
                    this._isTask = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_TASKFLW:
                    this._isTaskFlow = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_TAG_DATAVIEW:
                    this._isDataView = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_RECOMP:
                    this._isRecompute = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_FLWOP:
                    this._isFlowOperation = XmlParser.getBoolean(valueStr);
                    break;
                case ConstInterface.MG_ATTR_ENABLED:
                    this.enable(XmlParser.getBoolean(valueStr));
                    break;
                default:
                    Logger.Instance.WriteExceptionToLogWithMsg("in FlowMonitorQueue.initElements(): unknown  attribute: " + attribute);
                    break;
            }
        }
    }
    buildXML(message) {
        if (!this._queue.isEmpty() && this.ShouldSerialize) {
            message.Append(XMLConstants.START_TAG + ConstInterface.MG_TAG_FLWMTR_MSG + XMLConstants.TAG_CLOSE);
            while (!this._queue.isEmpty()) {
                let currAct = this._queue.get();
                currAct.buildXML(message);
            }
            message.Append("</" + ConstInterface.MG_TAG_FLWMTR_MSG + XMLConstants.TAG_CLOSE);
        }
    }
    isEmpty() {
        return this._queue.isEmpty();
    }
    enable(value) {
        let wasEnabled = this._enabled;
        this._enabled = value;
        this.ShouldSerialize = this._enabled;
        return wasEnabled;
    }
    addTaskEvent(triggeredBy, state, taskInfo) {
        if (this._enabled && this._isTask) {
            let act = new ActivityItem(this, FlowMonitorQueue.ACT_TASK, FlowMonitorInterface.FLWMTR_EVENT);
            let info;
            switch (state) {
                case FlowMonitorInterface.FLWMTR_START:
                    info = FlowMonitorQueue.S_EVENT_STR1 + triggeredBy + FlowMonitorQueue.S_EVENT_STR2;
                    break;
                case FlowMonitorInterface.FLWMTR_END:
                    info = FlowMonitorQueue.E_EVENT_STR + triggeredBy;
                    break;
                case FlowMonitorInterface.FLWMTR_PROPAGATE:
                    info = FlowMonitorQueue.S_EVENT_PROPAGATED;
                    break;
                default:
                    info = null;
                    break;
            }
            act.setInfo(info);
            this._queue.put(act);
            Logger.Instance.WriteSupportToLog(taskInfo + ": " + info, true);
        }
    }
    appendVirtualsAndParameters(currentValues) {
        let act = new ActivityItem(this, FlowMonitorQueue.ACT_TASK_FLW, InternalInterface.MG_ACT_REC_PREFIX);
        act.setInfo(currentValues);
        this._queue.put(act);
    }
    addDataViewFlow(task) {
        if (this._isDataView) {
            let fldTab = task.DataView.GetFieldsTab();
            let preparedDisplayString = "";
            let finaldisplayData = "\n";
            let fldValue;
            let data = "";
            for (let i = 0; i < fldTab.getSize(); i++) {
                let fldDef = fldTab.getField(i);
                if (!fldDef.IsEventHandlerField && (fldDef.IsVirtual && !fldDef.VirAsReal) || fldDef.isParam()) {
                    fldValue = fldDef.isNull() ? fldDef.getNullValue() : fldDef.getValue(true);
                    data = this.getFieldData(fldDef.getType(), fldValue, fldDef.getPicture(), fldDef.getCellsType(), task);
                    preparedDisplayString = this.PrepareDisplayString(fldDef.isParam(), fldDef.getVarName(), data, true);
                    finaldisplayData = finaldisplayData + preparedDisplayString + "\n";
                }
            }
            this.appendVirtualsAndParameters(finaldisplayData);
        }
    }
    getFieldData(Storagetype, data, picture, vecCellType, task) {
        switch (Storagetype) {
            case StorageAttribute.DOTNET:
                data = FlowMonitorQueue.UNPRINTABLE_STR_LOG;
                break;
            case StorageAttribute.BLOB:
                {
                    if (BlobType.isValidBlob(data)) {
                        let contentType = BlobType.getContentType(data);
                        if (contentType == BlobType.CONTENT_TYPE_ANSI || contentType == BlobType.CONTENT_TYPE_UNICODE)
                            data = data != null ? BlobType.getString(data) : "";
                        else
                            data = FlowMonitorQueue.UNPRINTABLE_STR_LOG;
                    }
                    else
                        data = "";
                }
                break;
            case StorageAttribute.NUMERIC:
            case StorageAttribute.DATE:
            case StorageAttribute.TIME:
                let conv = DisplayConvertor.Instance;
                data = conv.mg2disp(data, " ", new PIC(picture, Storagetype, task.getCompIdx()), false, task.getCompIdx(), false);
                break;
            case StorageAttribute.BLOB_VECTOR:
                let vecOutData;
                if (data == null) {
                    vecOutData = "[]";
                }
                else {
                    let cellAtt = vecCellType;
                    let vector = new VectorType(data);
                    if (cellAtt == StorageAttribute.BLOB && !VectorType.validateBlobContents(data))
                        vecOutData = "[]";
                    else {
                        let vecSize = VectorType.getVecSize(data);
                        let cellPicture = (cellAtt == StorageAttribute.NUMERIC || cellAtt == StorageAttribute.DATE || cellAtt == StorageAttribute.TIME) ? PIC.buildPicture(cellAtt, vector.getVecCell(1), task.getCompIdx(), true).getFormat() : picture;
                        vecOutData = "[";
                        for (let i = 0; i < vecSize; i++) {
                            vecOutData += this.getFieldData(cellAtt, vector.getVecCell(i + 1), cellPicture, vecCellType, task);
                            vecOutData += i < vecSize - 1 ? "," : "]";
                        }
                    }
                }
                data = vecOutData;
                break;
            case StorageAttribute.BOOLEAN:
                data = data == "1" ? "TRUE" : "FALSE";
                break;
        }
        return data;
    }
    PrepareDisplayString(mode, name, valueContent, addDoubleQuotes) {
        let str = "";
        let finalStringToDisplay = "";
        if (addDoubleQuotes)
            finalStringToDisplay = finalStringToDisplay + "\"";
        finalStringToDisplay = finalStringToDisplay + name;
        if (addDoubleQuotes)
            finalStringToDisplay = finalStringToDisplay + "\"";
        let paddedName = this.padRight(name, 34);
        let formatOfStringVirtual = `Virtual`;
        let formatOfStringParameter = `Parameter`;
        str = (mode ? formatOfStringParameter : formatOfStringVirtual) + " : " + `${paddedName}` + " : " + `${valueContent.trim()}`;
        return str;
    }
    addTaskFlowRec(id, state, taskInfo) {
        if (this._enabled && this._isTaskFlow) {
            let info;
            switch (id) {
                case InternalInterface.MG_ACT_REC_PREFIX:
                    info = state === FlowMonitorInterface.FLWMTR_START
                        ? FlowMonitorQueue.S_RECPRF_STR
                        : FlowMonitorQueue.E_RECPRF_STR;
                    id = FlowMonitorInterface.FLWMTR_PREFIX;
                    break;
                case InternalInterface.MG_ACT_REC_SUFFIX:
                    info = state === FlowMonitorInterface.FLWMTR_START
                        ? FlowMonitorQueue.S_RECSUF_STR
                        : FlowMonitorQueue.E_RECSUF_STR;
                    id = FlowMonitorInterface.FLWMTR_SUFFIX;
                    break;
                case InternalInterface.MG_ACT_TASK_PREFIX:
                    info = state === FlowMonitorInterface.FLWMTR_START
                        ? FlowMonitorQueue.S_TASKPRF_STR
                        : FlowMonitorQueue.E_TASKPRF_STR;
                    id = FlowMonitorInterface.FLWMTR_PREFIX;
                    break;
                case InternalInterface.MG_ACT_TASK_SUFFIX:
                    info = state === FlowMonitorInterface.FLWMTR_START
                        ? FlowMonitorQueue.S_TASKSUF_STR
                        : FlowMonitorQueue.E_TASKSUF_STR;
                    id = FlowMonitorInterface.FLWMTR_SUFFIX;
                    break;
                default:
                    info = null;
                    break;
            }
            let act = new ActivityItem(this, FlowMonitorQueue.ACT_TASK_FLW, id);
            act.setInfo(info);
            this._queue.put(act);
            Logger.Instance.WriteSupportToLog(taskInfo + ": " + info, true);
        }
    }
    addTaskFlowFld(id, fldName, state, taskInfo) {
        this.addTaskFlowCtrl(id, fldName, state, taskInfo);
    }
    addTaskFlowCtrl(id, ctrlName, state, taskInfo) {
        if (this._enabled && this._isTaskFlow) {
            let info;
            switch (id) {
                case InternalInterface.MG_ACT_VARIABLE:
                    info = state === FlowMonitorInterface.FLWMTR_START
                        ? FlowMonitorQueue.S_VARIABLE_STR
                        : FlowMonitorQueue.E_VARIABLE_STR;
                    id = FlowMonitorInterface.FLWMTR_VARCHG_VALUE;
                    break;
                case InternalInterface.MG_ACT_CTRL_PREFIX:
                    info = state === FlowMonitorInterface.FLWMTR_START
                        ? FlowMonitorQueue.S_CTRLPRF_STR
                        : FlowMonitorQueue.E_CTRLPRF_STR;
                    id = FlowMonitorInterface.FLWMTR_CTRL_PREFIX;
                    break;
                case InternalInterface.MG_ACT_CTRL_SUFFIX:
                    info = state === FlowMonitorInterface.FLWMTR_START
                        ? FlowMonitorQueue.S_CTRLSUF_STR
                        : FlowMonitorQueue.E_CTRLSUF_STR;
                    id = FlowMonitorInterface.FLWMTR_CTRL_SUFFIX;
                    break;
                case InternalInterface.MG_ACT_CTRL_VERIFICATION:
                    info = state === FlowMonitorInterface.FLWMTR_START
                        ? FlowMonitorQueue.S_CTRLVER_STR
                        : FlowMonitorQueue.E_CTRLVER_STR;
                    id = FlowMonitorInterface.FLWMTR_CTRL_SUFFIX;
                    break;
                default:
                    info = null;
                    break;
            }
            let act = new ActivityItem(this, FlowMonitorQueue.ACT_TASK_FLW, id);
            if (info !== null) {
                info = info + ctrlName;
                if (id === FlowMonitorInterface.FLWMTR_VARCHG_VALUE && state === FlowMonitorInterface.FLWMTR_START)
                    info = info + FlowMonitorQueue.VARIABLE_REASON_STR;
                act.setInfo(info);
                Logger.Instance.WriteSupportToLog(taskInfo + ": " + info, true);
            }
            this._queue.put(act);
        }
    }
    addTaskFlowHandler(handlerId, state, taskInfo) {
        if (this._enabled && this._isTaskFlow) {
            let act = new ActivityItem(this, FlowMonitorQueue.ACT_TASK_FLW, FlowMonitorInterface.FLWMTR_TSK_HANDLER);
            let info;
            switch (state) {
                case FlowMonitorInterface.FLWMTR_START:
                    info = NString.Format(FlowMonitorQueue.S_HANDLER_STR, handlerId);
                    break;
                case FlowMonitorInterface.FLWMTR_END:
                    info = NString.Format(FlowMonitorQueue.E_HANDLER_STR, handlerId);
                    break;
                default:
                    info = null;
                    break;
            }
            if (info !== null) {
                act.setInfo(info);
                Logger.Instance.WriteSupportToLog(taskInfo + ": " + info, true);
            }
            this._queue.put(act);
        }
    }
    addRecompute(triggeredByVarName, taskInfo) {
        if (this._enabled && this._isRecompute) {
            let act = new ActivityItem(this, FlowMonitorQueue.ACT_RECOMPUTE, FlowMonitorInterface.FLWMTR_RECOMP);
            let info = FlowMonitorQueue.RECOMP_STR + triggeredByVarName;
            act.setInfo(info);
            this._queue.put(act);
            Logger.Instance.WriteSupportToLog(taskInfo + ": " + info, true);
        }
    }
    async addFlowFieldOperation(oper, bExecuted, taskInfo) {
        if (this._enabled && this._isFlowOperation) {
            let act = new ActivityItem(this, FlowMonitorQueue.ACT_FLW_OPER, FlowMonitorInterface.FLWMTR_DATA_OPER);
            let buffer = new StringBuilder(FlowMonitorQueue.FLW_PERFIX);
            await oper.AddFlowDescription(buffer);
            buffer.Append(' ');
            if (!bExecuted)
                buffer.Append("[Not Executed]");
            let info = buffer.ToString();
            act.setInfo(info);
            this._queue.put(act);
            Logger.Instance.WriteSupportToLog(taskInfo + ": " + info, true);
        }
    }
    addFlowOperationUpdate(state, taskInfo) {
        if (this._enabled && this._isFlowOperation) {
            let act = new ActivityItem(this, FlowMonitorQueue.ACT_FLW_OPER, FlowMonitorInterface.FLWMTR_DATA_OPER);
            let info;
            if (state === FlowMonitorInterface.FLWMTR_START)
                info = "Starts Update";
            else
                info = "Ends Update";
            act.setInfo(info);
            this._queue.put(act);
            Logger.Instance.WriteSupportToLog(taskInfo + ": " + info, true);
        }
    }
    addFlowVerifyInfo(info, taskInfo) {
        this.addFlowInfo(info, taskInfo);
    }
    addFlowInvokeOsInfo(info, taskInfo) {
        this.addFlowInfo(info, taskInfo);
    }
    addFlowInfo(info, taskInfo) {
        if (!this._enabled)
            return;
        let act = new ActivityItem(this, FlowMonitorQueue.ACT_FLW_OPER, FlowMonitorInterface.FLWMTR_DATA_OPER);
        let buffer = new StringBuilder("");
        if (!(info === "")) {
            buffer.Append(" >> INFORMATION >> ");
            buffer.Append(info);
        }
        let newInfo = buffer.ToString();
        act.setInfo(newInfo);
        this._queue.put(act);
        Logger.Instance.WriteSupportToLog(taskInfo + ": " + newInfo, true);
    }
}
FlowMonitorQueue._instance = null;
FlowMonitorQueue.UNPRINTABLE_STR_LOG = "#UNPRINTABLE#";
FlowMonitorQueue.S_EVENT_STR1 = ">>Starts ";
FlowMonitorQueue.S_EVENT_STR2 = " Event";
FlowMonitorQueue.S_EVENT_PROPAGATED = "Event was propagated";
FlowMonitorQueue.E_EVENT_STR = "<<Ends Event";
FlowMonitorQueue.S_RECPRF_STR = "Starts Record Prefix";
FlowMonitorQueue.E_RECPRF_STR = "Ends Record Prefix";
FlowMonitorQueue.S_RECSUF_STR = "Starts Record Suffix";
FlowMonitorQueue.E_RECSUF_STR = "Ends Record Suffix";
FlowMonitorQueue.S_TASKSUF_STR = "Starts Task Suffix";
FlowMonitorQueue.E_TASKSUF_STR = "Ends Task Suffix";
FlowMonitorQueue.S_TASKPRF_STR = "Starts Task Prefix";
FlowMonitorQueue.E_TASKPRF_STR = "Ends Task Prefix";
FlowMonitorQueue.S_CTRLPRF_STR = "Starts Control Prefix - ";
FlowMonitorQueue.E_CTRLPRF_STR = "Ends Control Prefix - ";
FlowMonitorQueue.S_CTRLSUF_STR = "Starts Control Suffix - ";
FlowMonitorQueue.E_CTRLSUF_STR = "Ends Control Suffix - ";
FlowMonitorQueue.S_HANDLER_STR = "Starts handling event {0}";
FlowMonitorQueue.E_HANDLER_STR = "Ends handling event {0}";
FlowMonitorQueue.S_CTRLVER_STR = "Starts Control Verification for Control - ";
FlowMonitorQueue.E_CTRLVER_STR = "Ends Control Verification for Control - ";
FlowMonitorQueue.RECOMP_STR = "Recomputes - ";
FlowMonitorQueue.S_VARIABLE_STR = "Starts Variable Change - ";
FlowMonitorQueue.E_VARIABLE_STR = "Ends Variable Change - ";
FlowMonitorQueue.VARIABLE_REASON_STR = " - Reason - Previous value";
FlowMonitorQueue.FLW_PERFIX = "Flow - ";
FlowMonitorQueue.ACT_TASK = 'T';
FlowMonitorQueue.ACT_TASK_FLW = 'F';
FlowMonitorQueue.ACT_RECOMPUTE = 'R';
FlowMonitorQueue.ACT_FLW_OPER = 'T';
FlowMonitorQueue.LongTimePattern = "HH:mm:ss:SSS";
class ActivityItem {
    constructor(enclosingInstance, type, id) {
        this._id = 0;
        this._type = null;
        this._info = null;
        this._time = null;
        this._type = type;
        this._id = id;
        this.setTime();
    }
    setTime() {
        this._time = DateTimeUtils.ToString(DateTime.Now, FlowMonitorQueue.LongTimePattern, Logger.Instance);
    }
    setInfo(info_) {
        this._info = ((info_ !== null) ? XmlParser.escape(info_) : info_);
    }
    buildXML(message) {
        message.Append(XMLConstants.START_TAG + ConstInterface.MG_TAG_FLWMTR_ACT);
        message.Append(" " + XMLConstants.MG_ATTR_TYPE + "=\"" + this._type + "\"");
        message.Append(" " + XMLConstants.MG_ATTR_ID + "=\"" + this._id + "\"");
        if (this._info != null)
            message.Append(" " + ConstInterface.MG_ATTR_INFO + "=\"" + this._info + "\"");
        message.Append(" " + ConstInterface.MG_ATTR_TIME + "=\"" + this._time + "\"");
        message.Append(XMLConstants.TAG_TERM);
    }
}
//# sourceMappingURL=data:application/json;base64,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