import { Dictionary, List, StringBuilder } from '@magic-xpa/mscorelib';
import { Logger, XMLConstants, XmlParser } from '@magic-xpa/utils';
import { ConstInterface } from '../ConstInterface';
import { ParamParseResult } from "./ParamParseResult";
export class PrmMap {
    getvalue(s) {
        if (this.values.ContainsKey(s))
            return this.values.get_Item(s);
        else
            return null;
    }
    setValue(s, v) {
        this.values.set_Item(s, v);
    }
    remove(s) {
        this.values.Remove(s);
    }
    constructor() {
        this.values = null;
        this.values = new Dictionary();
    }
}
export class MirrorPrmMap extends PrmMap {
    constructor(type) {
        super();
        this.changes = null;
        this.mirroredID = null;
        this.type = type;
        this.changes = new List();
    }
    new() {
        return new this.type();
    }
    setValue(s, v, addToChanges) {
        if (arguments.length === 3)
            this.setValue_0(s, v, addToChanges);
        else
            this.setValue_1(s, v);
    }
    setValue_0(s, v, addToChanges) {
        if (addToChanges && !this.changes.Contains(s))
            this.changes.push(s);
        super.setValue(s, v);
    }
    setValue_1(s, v) {
        if (!this.changes.Contains(s))
            this.changes.push(s);
        super.setValue(s, v);
    }
    remove(s) {
        if (!this.changes.Contains(s)) {
            this.changes.push(s);
        }
        super.remove(s);
    }
    mirrorAllToXML() {
        let xml = new StringBuilder();
        if (this.values != null) {
            xml.Append("<" + this.mirroredID + ">");
            for (let key of this.values.Keys) {
                if (this.CheckIfSystemReservedParam(key))
                    continue;
                xml.Append("<" + ConstInterface.MG_TAG_PARAM + " " +
                    XMLConstants.MG_ATTR_NAME + "=\"" + XmlParser.escape(key) + "\" ");
                xml.Append(this.values.get_Item(key).mirrorToXML());
                xml.Append(XMLConstants.TAG_TERM);
            }
            xml.Append("</" + this.mirroredID + ">");
        }
        return xml.ToString();
    }
    CheckIfSystemReservedParam(key) {
        if (key == "UTF8TRANS" || key === "REMOTE_HOST" || key === "REMOTE_ADDR" ||
            key === "CTX" || key === "MGARG0" || key === "DATA" || key === "MGCURRENTREQUESTID" ||
            key === "HTTP_MGXPARIAGLOBALUNIQUESESSIONID" || key === "MG_POST_BODY")
            return true;
        else
            return false;
    }
    mirrorToXML() {
        let xml = new StringBuilder();
        if (this.changes.length > 0) {
            xml.Append("<" + this.mirroredID + ">");
            this.changes.forEach(change => {
                xml.Append("<" + ConstInterface.MG_TAG_PARAM + " " +
                    XMLConstants.MG_ATTR_NAME + "=\"" + XmlParser.escape(change) + "\" ");
                if (this.values.ContainsKey(change))
                    xml.Append(this.values.get_Item(change).mirrorToXML());
                else
                    xml.Append("removed=\"Y\"");
                xml.Append(">");
            });
            xml.Append("</" + this.mirroredID + ">");
        }
        this.changes.Clear();
        return xml.ToString();
    }
    fillData(parser) {
        while (this.mirrorFromXML(parser.getNextTag(), parser)) {
        }
    }
    RestoreParams(xml) {
        let parser = new XmlParser(xml);
        while (this.mirrorFromXML(parser.getNextTag(), parser)) {
        }
    }
    mirrorFromXML(foundTagName, xmlParser) {
        if (foundTagName === null)
            return false;
        if (foundTagName === this.mirroredID) {
            xmlParser.setCurrIndex2EndOfTag();
            this.fillDataEntry(xmlParser);
            return true;
        }
        else if (foundTagName === ("/" + this.mirroredID)) {
            this.changes.Clear();
            xmlParser.setCurrIndex2EndOfTag();
            return false;
        }
        else {
            Logger.Instance.WriteExceptionToLogWithMsg("There is no such tag in MirrorPrmMap.mirrorFromXML(): " + foundTagName);
            return false;
        }
    }
    fillDataEntry(xmlParser) {
        let nextTag = xmlParser.getNextTag();
        while (nextTag === ConstInterface.MG_TAG_PARAM) {
            let nameStart = xmlParser.getXMLdata().indexOf(XMLConstants.MG_ATTR_NAME + "=\"", xmlParser.getCurrIndex()) +
                XMLConstants.MG_ATTR_NAME.length + 2;
            xmlParser.setCurrIndex(nameStart);
            let nameEnd = xmlParser.getXMLdata().indexOf("\"", nameStart);
            let name = xmlParser.getXMLsubstring(nameEnd).trim();
            xmlParser.setCurrIndex(nameEnd);
            let newVal = this.new();
            switch (newVal.init(name, xmlParser)) {
                case ParamParseResult.OK:
                    this.values.set_Item(name, newVal);
                    break;
                case ParamParseResult.TOUPPER:
                    this.values.set_Item(name.toUpperCase(), newVal);
                    break;
                case ParamParseResult.DELETE:
                    this.values.Remove(name);
                    break;
            }
            xmlParser.setCurrIndex2EndOfTag();
            nextTag = xmlParser.getNextTag();
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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