import { Logger, StrUtil } from '@magic-xpa/utils';
import { ApplicationException, Debug, Encoding, ISO_8859_1_Encoding, NNumber, NString } from '@magic-xpa/mscorelib';
import { GuiConstants } from '../../GuiConstants';
import { GuiEnvironment } from '../../GuiEnvironment';
export class BlobType {
    static getContentType(blob) {
        try {
            let tokens = StrUtil.tokenize(blob, /,|;/);
            return tokens[4].charAt(0);
        }
        catch (Exception) {
            throw new ApplicationException(" in BlobType.getContentType blob is in invalid format");
        }
    }
    static getEmptyBlobPrefix(vecCellAttr) {
        return NString.Format("0,0,{0},{1},0", '\0', vecCellAttr);
    }
    static getBlobPrefixForContentType(contentType) {
        return NString.Format("0,0,{0},{0},{1};", '\0', contentType);
    }
    static getPrefix(str) {
        let idx = str.indexOf(';');
        return str.substr(0, idx + 1);
    }
    static getEncodingFromContentType(ContentType) {
        let encoding;
        if (ContentType === BlobType.CONTENT_TYPE_UNICODE)
            encoding = Encoding.Unicode;
        else if (ContentType === BlobType.CONTENT_TYPE_ANSI)
            encoding = GuiEnvironment.Environment.GetEncoding();
        else
            encoding = ISO_8859_1_Encoding.ISO_8859_1;
        return encoding;
    }
    static MbToUnicode(BytesInMb) {
        let result = null;
        try {
            let UnicodeString = GuiEnvironment.Environment.GetEncoding().GetString(BytesInMb, 0, BytesInMb.length);
            result = Encoding.Unicode.GetBytes(UnicodeString);
        }
        catch (Exception) {
            result = null;
        }
        return result;
    }
    static UnicodeToMb(BytesInUnicode) {
        let result = null;
        try {
            let UnicodeString = Encoding.Unicode.GetString(BytesInUnicode, 0, BytesInUnicode.length);
            result = GuiEnvironment.Environment.GetEncoding().GetBytes(UnicodeString);
        }
        catch (Exception) {
            result = null;
        }
        return result;
    }
    static getString(blob) {
        let result = null;
        let bytes = null;
        if (BlobType.isValidBlob(blob)) {
            let contentType = BlobType.getContentType(blob);
            if (contentType !== BlobType.CONTENT_TYPE_UNICODE)
                contentType = BlobType.CONTENT_TYPE_ANSI;
            try {
                bytes = BlobType.getBytes(blob);
                let encoding = BlobType.getEncodingFromContentType(contentType);
                result = encoding.GetString(bytes, 0, bytes.length);
                let index = result.indexOf('\0');
                if (index !== -1)
                    result = result.substr(0, index);
            }
            catch (Exception) {
                result = null;
            }
        }
        else
            Debug.Assert(false);
        return result;
    }
    static createFromString(blobStr, contentType) {
        let blob = null;
        let srcContentType = contentType;
        let bytes = null;
        if (contentType !== BlobType.CONTENT_TYPE_UNICODE)
            srcContentType = BlobType.CONTENT_TYPE_ANSI;
        try {
            if (blobStr.constructor === ArrayBuffer) {
                bytes = new Uint8Array(blobStr);
            }
            else {
                let encoding = BlobType.getEncodingFromContentType(srcContentType);
                bytes = encoding.GetBytes(blobStr);
            }
            blob = BlobType.createFromBytes(bytes, contentType);
        }
        catch (Exception) {
            blob = null;
        }
        return blob;
    }
    static getBytes(str) {
        let bytes = null;
        let data = BlobType.removeBlobPrefix(str);
        try {
            let encoding = ISO_8859_1_Encoding.ISO_8859_1;
            bytes = encoding.GetBytes(data);
        }
        catch (Exception) {
            bytes = null;
        }
        return bytes;
    }
    static createFromBytes(bytes, contentType) {
        let blobStr = '';
        let blobPrefix;
        let blobData;
        blobPrefix = BlobType.getBlobPrefixForContentType(contentType);
        try {
            let encoding = ISO_8859_1_Encoding.ISO_8859_1;
            blobData = encoding.GetString(bytes, 0, bytes.length);
        }
        catch (Exception) {
            blobData = null;
        }
        blobStr = blobPrefix + blobData;
        return blobStr;
    }
    static copyBlob(dest, src) {
        let srcBytes;
        let destBytes;
        if (src === null)
            return null;
        else if (dest === null)
            return src;
        srcBytes = BlobType.getBytes(src);
        let destContentType = BlobType.getContentType(dest);
        let srcContentType = BlobType.getContentType(src);
        if (srcContentType === BlobType.CONTENT_TYPE_ANSI && destContentType === BlobType.CONTENT_TYPE_UNICODE)
            destBytes = BlobType.MbToUnicode(srcBytes);
        else if (srcContentType === BlobType.CONTENT_TYPE_UNICODE && destContentType === BlobType.CONTENT_TYPE_ANSI)
            destBytes = BlobType.UnicodeToMb(srcBytes);
        else
            destBytes = srcBytes;
        dest = BlobType.createFromBytes(destBytes, destContentType);
        return dest;
    }
    static removeBlobPrefix(source) {
        let idx;
        if (source !== null) {
            idx = source.indexOf(';');
            if (idx < 0)
                Logger.Instance.WriteToLog("Error: invalid blob prefix", true);
            return source.substr(idx + 1);
        }
        else
            return null;
    }
    static isValidBlob(blob) {
        let isValid = true;
        if (blob === null || blob.indexOf(';') < 0)
            isValid = false;
        if (isValid) {
            let contentType = BlobType.getContentType(blob);
            if (contentType !== BlobType.CONTENT_TYPE_ANSI && contentType !== BlobType.CONTENT_TYPE_BINARY && contentType !== BlobType.CONTENT_TYPE_UNICODE && contentType !== BlobType.CONTENT_TYPE_UNKNOWN)
                isValid = false;
        }
        return isValid;
    }
    static setContentType(str, contentType) {
        let result = "";
        try {
            let prefixLastIndex = str.indexOf(";");
            let prefix = str.substr(0, prefixLastIndex);
            let dataLength = str.length - (prefix.length + 1);
            let data = str.substr(prefixLastIndex + 1, dataLength);
            let prefixTokens = StrUtil.tokenize(prefix, /,|;/);
            for (let i = 0; i < GuiConstants.BLOB_PREFIX_ELEMENTS_COUNT; i = i + 1) {
                if (i === 4)
                    result = result + contentType + ",";
                else
                    result = result + prefixTokens[i] + ",";
            }
            return result.substr(0, result.length - 1) + ";" + data;
        }
        catch (Exception) {
            throw new ApplicationException(" in BlobType.setContentType : invalid format");
        }
    }
    static SetVecCellAttr(str, vecCellAttr) {
        let result = "";
        try {
            let prefixLastIndex = str.indexOf(";");
            let prefix = str.substr(0, prefixLastIndex);
            let data = str.substr(prefixLastIndex + 1);
            let tokens = StrUtil.tokenize(prefix, /,|;/);
            for (let i = 0; i < GuiConstants.BLOB_PREFIX_ELEMENTS_COUNT; i = i + 1) {
                if (i === 3)
                    result = result + vecCellAttr + ",";
                else
                    result = result + tokens[i] + ",";
            }
            return result.substr(0, result.length - 1) + ";" + data;
        }
        catch (Exception) {
            throw new ApplicationException(" in XMLparser.blobPrefixLength invalid format");
        }
    }
    static GetVecCellAttr(blobStr) {
        try {
            let tokens = StrUtil.tokenize(blobStr, /,|;/);
            return tokens[3].charAt(0);
        }
        catch (Exception) {
            throw new ApplicationException(" in BlobType.GetVecCellAttr(): blob is in invalid format");
        }
    }
    static blobPrefixLength(blob) {
        try {
            let prefixLength = blob.indexOf(';') + 1;
            let prefix = blob.substr(0, prefixLength);
            if (prefixLength > 0) {
                let tokens = StrUtil.tokenize(prefix, ",");
                if (tokens.length === GuiConstants.BLOB_PREFIX_ELEMENTS_COUNT)
                    return prefixLength;
            }
        }
        catch (Exception) {
        }
        throw new ApplicationException(" in XMLparser.blobPrefixLength invalid format");
    }
    static getKey(blobStr) {
        let tokens = StrUtil.tokenize(blobStr, /,|;/);
        let key = 0;
        if (tokens.length > 5) {
            try {
                key = NNumber.Parse(tokens[0]);
            }
            catch (Exception) {
            }
        }
        return key;
    }
    static getBlobSize(blob) {
        let size = 0;
        try {
            let tokens = StrUtil.tokenize(blob, /,|;/);
            if (tokens.length > 5) {
                size = blob.length;
                for (let i = 0; i < 5; i = i + 1) {
                    size = size - tokens[i].length;
                    size = size - 1;
                }
            }
        }
        catch (Exception) {
        }
        return size;
    }
    static ParseContentType(contentType) {
        let newContentType = ' ';
        switch (contentType) {
            case 0:
                newContentType = BlobType.CONTENT_TYPE_UNKNOWN;
                break;
            case 1:
                newContentType = BlobType.CONTENT_TYPE_ANSI;
                break;
            case 2:
                newContentType = BlobType.CONTENT_TYPE_UNICODE;
                break;
            case 3:
                newContentType = BlobType.CONTENT_TYPE_BINARY;
        }
        return newContentType;
    }
}
BlobType.CONTENT_TYPE_UNKNOWN = '0';
BlobType.CONTENT_TYPE_ANSI = '1';
BlobType.CONTENT_TYPE_UNICODE = '2';
BlobType.CONTENT_TYPE_BINARY = '3';
//# sourceMappingURL=data:application/json;base64,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