import { BlobType } from "./BlobType";
import { StorageAttribute, StrUtil, XmlParser, Base64, XMLConstants } from "@magic-xpa/utils";
import { NNumber, NString, Int32 } from "@magic-xpa/mscorelib";
import { Events } from "../../Events";
import { DisplayConvertor } from "../gui/DisplayConvertor";
import { VectorType } from "./VectorType";
import { GuiEnvironment } from "../../GuiEnvironment";
export class FieldDef {
    set DbModifiable(value) {
        this._dbModifiable = value;
    }
    get DbModifiable() {
        return this._dbModifiable;
    }
    constructor(id) {
        this.DefaultValue = null;
        this.NullAllowed = false;
        this._nullDefault = false;
        this._nullValue = null;
        this._nullDisplay = null;
        this._dbModifiable = true;
        this._partOfDataview = true;
        this._picture = null;
        this._size = 0;
        this._varName = null;
        this._contentType = BlobType.CONTENT_TYPE_UNKNOWN;
        this._vecCellsContentType = BlobType.CONTENT_TYPE_UNKNOWN;
        this._vecCellsSize = 0;
        this._vecCellsType = StorageAttribute.NONE;
        this._type = StorageAttribute.NONE;
        this.Storage = 0;
        this.VarDisplayName = null;
        this._id = 0;
        this._spaces = null;
        this._id = id;
        this.VarDisplayName = NString.Empty;
    }
    get PartOfDataview() {
        return this._partOfDataview;
    }
    getId() {
        return this._id;
    }
    getType() {
        return this._type;
    }
    static getMagicDefaultNullDisplayValue(type) {
        let val;
        switch (type) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.UNICODE:
            case StorageAttribute.BOOLEAN:
                val = "";
                break;
            case StorageAttribute.NUMERIC:
            case StorageAttribute.DATE:
            case StorageAttribute.TIME:
                val = "";
                break;
        }
        return val;
    }
    static getMagicDefaultValue(type) {
        let val = null;
        switch (type) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.UNICODE:
                val = "";
                break;
            case StorageAttribute.BLOB:
            case StorageAttribute.BLOB_VECTOR:
                val = BlobType.getEmptyBlobPrefix('\0') + ";";
                break;
            case StorageAttribute.NUMERIC:
            case StorageAttribute.TIME:
                val = "FF00000000000000000000000000000000000000";
                break;
            case StorageAttribute.DATE:
                val = FieldDef._default_date;
                break;
            case StorageAttribute.BOOLEAN:
                val = "0";
                break;
        }
        return val;
    }
    getDefaultValue() {
        let val = null;
        if (this._type !== StorageAttribute.BLOB_VECTOR) {
            if (this._nullDefault && this._nullValue !== null)
                val = this._nullValue;
            else if (this.DefaultValue !== null)
                val = this.DefaultValue;
            else
                val = this.getMagicDefaultValue();
        }
        else
            val = this.getMagicDefaultValue();
        return val;
    }
    getCellDefualtValue() {
        let val;
        if (this._type === StorageAttribute.BLOB_VECTOR) {
            if (this._nullDefault && this._nullValue !== null)
                val = this._nullValue;
            else if (this.DefaultValue !== null)
                val = this.DefaultValue;
            else {
                val = FieldDef.getMagicDefaultValue(this._vecCellsType);
                if (this._vecCellsType === StorageAttribute.BLOB)
                    val = BlobType.setContentType(val, this._vecCellsContentType);
            }
        }
        else
            val = this.getDefaultValue();
        return val;
    }
    getMagicDefaultValue() {
        let val = FieldDef.getMagicDefaultValue(this._type);
        if (this._type === StorageAttribute.BLOB)
            val = BlobType.setContentType(val, this._contentType);
        else {
            if (this._type === StorageAttribute.BLOB_VECTOR) {
                val = BlobType.SetVecCellAttr(val, this._vecCellsType);
                if (this._vecCellsType === StorageAttribute.BLOB)
                    val = BlobType.setContentType(val, this._vecCellsContentType);
            }
        }
        return val;
    }
    getContentType() {
        return this._contentType;
    }
    hasNullDisplayValue() {
        return this._nullDisplay !== null;
    }
    getSize() {
        return this._size;
    }
    getVecCellsSize() {
        return (this._type === StorageAttribute.BLOB_VECTOR) ? this._vecCellsSize : this._size;
    }
    setLengths(type, vecCells) {
        let res = this._size;
        switch (type) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.MEMO:
            case StorageAttribute.UNICODE:
                if (vecCells) {
                    res = this._vecCellsSize;
                    this._spaces = new Array(this._vecCellsSize);
                }
                else
                    this._spaces = new Array(this._size);
                for (let i = 0; i < this._size; i = i + 1) {
                    this._spaces[i] = ' ';
                }
                break;
            case StorageAttribute.NUMERIC:
            case StorageAttribute.DATE:
            case StorageAttribute.TIME:
                res = GuiEnvironment.Environment.GetSignificantNumSize() * 2;
                break;
            case StorageAttribute.BOOLEAN:
                res = (!vecCells ? 1 : 4);
                break;
            case StorageAttribute.BLOB:
                res = (!vecCells ? Int32.MaxValue : VectorType.BLOB_TYPE_STRUCT_SIZE);
                break;
            case StorageAttribute.BLOB_VECTOR:
                res = Int32.MaxValue;
                if (!vecCells)
                    this.setLengths(this._vecCellsType, true);
                else
                    res = VectorType.BLOB_TYPE_STRUCT_SIZE;
                break;
        }
        if (vecCells)
            this._vecCellsSize = res;
        else
            this._size = res;
    }
    getCellsType() {
        return (this._type === StorageAttribute.BLOB_VECTOR) ? this._vecCellsType : this._type;
    }
    getVecCellsContentType() {
        return this._vecCellsContentType;
    }
    getNullValue() {
        return this._nullValue;
    }
    isNullDefault() {
        return this._nullDefault;
    }
    getNullDisplay() {
        let val;
        switch (this._type) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.UNICODE:
            case StorageAttribute.BOOLEAN:
                val = "";
                break;
            case StorageAttribute.NUMERIC:
            case StorageAttribute.DATE:
            case StorageAttribute.TIME:
                val = null;
                break;
        }
        return val;
    }
    IsContentUnicode() {
        if (this._type === StorageAttribute.BLOB && this._contentType !== BlobType.CONTENT_TYPE_UNICODE)
            return false;
        return true;
    }
    GetPicture() {
        return this._picture;
    }
    setAttribute(attribute, valueStr) {
        let isTagProcessed = true;
        switch (attribute) {
            case XMLConstants.MG_ATTR_TYPE:
                this._type = valueStr.charAt(0);
                break;
            case XMLConstants.MG_ATTR_SIZE:
                this._size = XmlParser.getInt(valueStr);
                if (this._size <= 0)
                    Events.WriteExceptionToLog("in Field.initElements(): size must be greater than zero");
                break;
            case XMLConstants.MG_ATTR_VAR_NAME:
                this._varName = XmlParser.unescape(valueStr).toString();
                break;
            case XMLConstants.MG_ATTR_VAR_DISP_NAME:
                this.VarDisplayName = XmlParser.unescape(valueStr).toString();
                break;
            case XMLConstants.MG_ATTR_PICTURE:
                this._picture = XmlParser.unescape(valueStr).toString();
                break;
            case XMLConstants.MG_ATTR_VEC_CELLS_SIZE:
                this._vecCellsSize = NNumber.Parse(valueStr);
                break;
            case XMLConstants.MG_ATTR_VEC_CELLS_ATTR:
                this._vecCellsType = valueStr.charAt(0);
                break;
            case XMLConstants.MG_ATTR_VEC_CELLS_CONTENT:
                this._vecCellsContentType = valueStr.charAt(0);
                break;
            case XMLConstants.MG_ATTR_NULLVALUE:
                if (this._type === StorageAttribute.NUMERIC || this._type === StorageAttribute.DATE || this._type === StorageAttribute.TIME) {
                    if (GuiEnvironment.Environment.GetDebugLevel() > 1)
                        this._nullValue = XmlParser.unescape(valueStr).toString();
                    else
                        this._nullValue = Base64.decodeToHex(valueStr);
                }
                else
                    this._nullValue = XmlParser.unescape(valueStr).toString();
                break;
            case XMLConstants.MG_ATTR_NULLDISPLAY:
                this._nullDisplay = XmlParser.unescape(valueStr).toString();
                break;
            case XMLConstants.MG_ATTR_NULLDEFAULT:
                this._nullDefault = DisplayConvertor.toBoolean(valueStr);
                break;
            case XMLConstants.MG_ATTR_DB_MODIFIABLE:
                this.DbModifiable = DisplayConvertor.toBoolean(valueStr);
                break;
            case XMLConstants.MG_ATTR_DEFAULTVALUE:
                this.DefaultValue = valueStr;
                if (this._type === StorageAttribute.ALPHA || this._type === StorageAttribute.UNICODE) {
                    this.DefaultValue = XmlParser.unescape(valueStr).toString();
                    this.DefaultValue = StrUtil.padStr(this.DefaultValue, this._size);
                }
                else if (this._type !== StorageAttribute.BLOB && this._type !== StorageAttribute.BOOLEAN) {
                    if ((this._type === StorageAttribute.BLOB_VECTOR &&
                        (this._vecCellsType === StorageAttribute.NUMERIC || this._vecCellsType === StorageAttribute.DATE || this._vecCellsType === StorageAttribute.TIME)) ||
                        (this._type === StorageAttribute.NUMERIC || this._type === StorageAttribute.DATE || this._type === StorageAttribute.TIME)) {
                        if (GuiEnvironment.Environment.GetDebugLevel() < 1)
                            this.DefaultValue = Base64.decodeToHex(valueStr);
                    }
                }
                else if (this.DefaultValue.length === 0 && this._type !== StorageAttribute.BLOB)
                    this.DefaultValue = null;
                else if (this._type === StorageAttribute.BLOB)
                    this.DefaultValue = BlobType.createFromString(this.DefaultValue, this._contentType);
                break;
            case XMLConstants.MG_ATTR_NULLALLOWED:
                this.NullAllowed = DisplayConvertor.toBoolean(valueStr);
                break;
            case XMLConstants.MG_ATTR_BLOB_CONTENT:
                this._contentType = valueStr.charAt(0);
                break;
            case XMLConstants.MG_ATTR_PART_OF_DATAVIEW:
                this._partOfDataview = DisplayConvertor.toBoolean(valueStr);
                break;
            case XMLConstants.MG_ATTR_STORAGE:
                this.Storage = XmlParser.getInt(valueStr);
                break;
            default:
                isTagProcessed = false;
                break;
        }
        return isTagProcessed;
    }
    fillData(parser) {
        let text = parser.ReadToEndOfCurrentElement();
        text = text.substr(text.indexOf(XMLConstants.MG_TAG_FLDH) + XMLConstants.MG_TAG_FLDH.length);
        let tokens = XmlParser.getTokens(text, XMLConstants.XML_ATTR_DELIM);
        this.initElements(tokens);
    }
    initElements(tokensVector) {
        for (let i = 0; i < tokensVector.length; i = i + 2) {
            let attribute = tokensVector.get_Item(i);
            let valueStr = tokensVector.get_Item(i + 1);
            this.setAttribute(attribute, valueStr);
        }
        this.SetAfterParsing();
    }
    SetAfterParsing() {
        this.setLengths(this._type, false);
        if (this.NullAllowed && this._nullDisplay === null)
            this._nullDisplay = "";
    }
    getVarName() {
        return this._varName || "";
    }
    getName() {
        return this.getVarName();
    }
}
FieldDef._default_date = null;
//# sourceMappingURL=data:application/json;base64,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