import { ApplicationException, Debug, ISO_8859_1_Encoding, NNumber, NumberStyles, StringBuilder } from "@magic-xpa/mscorelib";
import { Base64, StorageAttribute, StorageAttributeCheck, StrUtil, UtilStrByteMode, XmlParser } from "@magic-xpa/utils";
import { GuiEnvironment } from "../../GuiEnvironment";
import { VectorTypeBase } from "./VectorTypeBase";
export class RecordUtils {
    static byteStreamToString(stream) {
        if (arguments.length === 1 && (stream === null || stream instanceof StringBuilder)) {
            return RecordUtils.byteStreamToString_0(stream);
        }
        return RecordUtils.byteStreamToString_1(stream);
    }
    static byteStreamToString_0(stream) {
        let currStr;
        let currChar;
        let result = new StringBuilder(Math.floor(stream.Length / 2));
        for (let indx = 0; indx < stream.Length; indx = indx + 2) {
            currStr = stream.get_Item(indx) + stream.get_Item(indx + 1);
            currChar = String.fromCharCode(NNumber.Parse(currStr, NumberStyles.HexNumber));
            result.Append(currChar);
        }
        return result.ToString();
    }
    static byteStreamToString_1(stream) {
        return RecordUtils.byteStreamToString(new StringBuilder(stream));
    }
    static serializeItemVal(itemVal, itemAttr, cellAttr, toBase64) {
        Debug.Assert(itemVal != null);
        let significantNumSize = GuiEnvironment.Environment.GetSignificantNumSize() * 2;
        let valueSize;
        let j;
        let tmpBuf = new StringBuilder();
        let pos;
        let fldValLen;
        switch (itemAttr) {
            case StorageAttribute.NUMERIC:
            case StorageAttribute.DATE:
            case StorageAttribute.TIME:
                itemVal = !toBase64
                    ? itemVal.substr(0, significantNumSize)
                    : Base64.encode(this.byteStreamToString(itemVal.substr(0, significantNumSize)));
                break;
            case StorageAttribute.ALPHA:
            case StorageAttribute.UNICODE:
                itemVal = StrUtil.rtrim(itemVal);
                pos = 0;
                fldValLen = itemVal.length;
                do {
                    let nullChrPos = itemVal.indexOf(String.fromCharCode(0), pos);
                    if (nullChrPos === -1) {
                        valueSize = (fldValLen - pos).toString(16).toUpperCase();
                        for (j = 0; j < 4 - valueSize.length; j++)
                            tmpBuf.Append('0');
                        tmpBuf.Append(valueSize);
                        if (pos > 0)
                            itemVal = itemVal.substr(pos, (fldValLen) - (pos));
                        pos = fldValLen;
                    }
                    else {
                        valueSize = (nullChrPos - pos + 0x8000).toString(16).toUpperCase();
                        tmpBuf.Append(valueSize);
                        tmpBuf.Append(itemVal.substr(pos, (nullChrPos) - (pos)));
                        for (j = 1; j < fldValLen - nullChrPos && itemVal.charCodeAt(nullChrPos + j) === 0; j++) {
                        }
                        valueSize = "0000" + (j.toString(16)).toUpperCase();
                        tmpBuf.Append(valueSize.substr(valueSize.length - 4));
                        for (pos = nullChrPos; j > 0; j--, pos++) {
                            let tmpStr = "0" + itemVal.charCodeAt(nullChrPos).toString(16);
                            tmpBuf.Append(tmpStr.substr(tmpStr.length - 2));
                        }
                        if (pos >= fldValLen) {
                            tmpBuf.Append("0000");
                            itemVal = "";
                            break;
                        }
                    }
                } while (pos < fldValLen);
                break;
            case StorageAttribute.BLOB:
            case StorageAttribute.BLOB_VECTOR:
            case StorageAttribute.DOTNET:
                pos = 0;
                fldValLen = itemVal.length;
                if (UtilStrByteMode.isLocaleDefLangDBCS() && itemAttr === StorageAttribute.BLOB_VECTOR) {
                    if (cellAttr === StorageAttribute.ALPHA || cellAttr === StorageAttribute.MEMO) {
                        itemVal = VectorTypeBase.adjustAlphaStringsInFlatData(itemVal);
                        do {
                            if (itemVal.length < pos + 0x3FFF) {
                                if (pos > 0)
                                    itemVal = itemVal.substr(pos);
                                valueSize = (UtilStrByteMode.lenB(itemVal).toString(16)).toUpperCase();
                                for (j = 0; j < 4 - valueSize.length; j++)
                                    tmpBuf.Append('0');
                                tmpBuf.Append(valueSize);
                                itemVal = !toBase64
                                    ? StrUtil.stringToHexaDump(itemVal, 4)
                                    : Base64.encode(itemVal, true, GuiEnvironment.Environment.GetEncoding());
                                pos = fldValLen;
                            }
                            else {
                                let strSub = itemVal.substr(pos, 0x3FFF);
                                valueSize = ((UtilStrByteMode.lenB(strSub) + 0x8000).toString(16)).toUpperCase();
                                tmpBuf.Append(valueSize);
                                tmpBuf.Append(!toBase64
                                    ? StrUtil.stringToHexaDump(strSub, 4)
                                    : Base64.encode(strSub, true, GuiEnvironment.Environment.GetEncoding()));
                                tmpBuf.Append("0000");
                                pos += 0x3FFF;
                            }
                        } while (pos < fldValLen);
                        break;
                    }
                }
                do {
                    if (fldValLen < pos + 0x7FFF) {
                        valueSize = ((fldValLen - pos).toString(16)).toUpperCase();
                        for (j = 0; j < 4 - valueSize.length; j++)
                            tmpBuf.Append('0');
                        tmpBuf.Append(valueSize);
                        if (pos > 0)
                            itemVal = itemVal.substr(pos, (fldValLen) - (pos));
                        itemVal = !toBase64
                            ? StrUtil.stringToHexaDump(itemVal, 4)
                            : Base64.encode(itemVal);
                        pos = fldValLen;
                    }
                    else {
                        valueSize = "FFFF";
                        tmpBuf.Append(valueSize);
                        if (!toBase64)
                            tmpBuf.Append(StrUtil.stringToHexaDump(itemVal.substr(pos, 0x7FFF), 4));
                        else
                            tmpBuf.Append(Base64.encode(itemVal.substr(pos, 0x7FFF)));
                        tmpBuf.Append("0000");
                        pos += 0x7FFF;
                    }
                } while (pos < fldValLen);
                break;
        }
        tmpBuf.Append(itemVal);
        return tmpBuf.ToString();
    }
    static deSerializeItemVal(itemVal, itemAttr, itemLen, useHex, cellAttr, parsedLen) {
        let val = null;
        let idx = 0;
        let len, endIdx;
        let suffixBuf = null;
        let tmp = null;
        if (itemAttr === StorageAttribute.ALPHA
            || itemAttr === StorageAttribute.UNICODE
            || itemAttr === StorageAttribute.BLOB
            || itemAttr === StorageAttribute.BLOB_VECTOR
            || itemAttr === StorageAttribute.DOTNET) {
            endIdx = idx + 4;
            tmp = itemVal.substr(idx, (endIdx) - (idx));
            len = NNumber.Parse(tmp, NumberStyles.HexNumber);
            idx = endIdx;
        }
        else if (itemAttr === StorageAttribute.BOOLEAN)
            len = 1;
        else {
            let significantNumSize = GuiEnvironment.Environment.GetSignificantNumSize();
            if (useHex)
                len = significantNumSize * 2;
            else
                len = (Math.floor((significantNumSize + 2) / 3) * 4);
        }
        if ((len & 0x8000) > 0) {
            suffixBuf = new StringBuilder();
            len = (len & 0x7FFF);
            if (itemAttr === StorageAttribute.BLOB || itemAttr === StorageAttribute.BLOB_VECTOR
                || itemAttr === StorageAttribute.DOTNET)
                if (useHex)
                    len *= 2;
                else
                    len = (Math.floor((len + 2) / 3) * 4);
            parsedLen.value = this.getSpannedField(itemVal, len, idx, itemAttr, suffixBuf, useHex);
            val = suffixBuf.ToString();
            endIdx = idx + parsedLen.value;
        }
        else {
            if (itemAttr === StorageAttribute.BLOB
                || itemAttr === StorageAttribute.BLOB_VECTOR
                || itemAttr === StorageAttribute.DOTNET)
                if (useHex)
                    len *= 2;
                else
                    len = (Math.floor((len + 2) / 3) * 4);
            endIdx = idx + len;
            if (endIdx > itemVal.length)
                throw new ApplicationException("in Record.fillFieldsData() data string too short:\n" + itemVal);
            if (UtilStrByteMode.isLocaleDefLangDBCS() && itemAttr === StorageAttribute.BLOB_VECTOR)
                val = this.getString(itemVal.substr(idx, (endIdx) - (idx)), itemAttr, useHex, (cellAttr === StorageAttribute.ALPHA));
            else
                val = this.getString(itemVal.substr(idx, (endIdx) - (idx)), itemAttr, useHex);
        }
        idx = endIdx;
        if (itemAttr === StorageAttribute.ALPHA || itemAttr === StorageAttribute.UNICODE) {
            len = itemLen;
            val = StrUtil.padStr(val, len);
            if (itemAttr === StorageAttribute.ALPHA && UtilStrByteMode.isLocaleDefLangDBCS())
                val = UtilStrByteMode.leftB(val, len);
        }
        parsedLen.value = endIdx;
        return val;
    }
    static getString(str, type, useHex, useEnvCharset) {
        if (arguments.length === 3)
            return RecordUtils.getString_0(str, type, useHex);
        if (arguments.length === 2)
            return RecordUtils.getString_2(str, type);
        return RecordUtils.getString_1(str, type, useHex, useEnvCharset);
    }
    static getString_0(str, type, useHex) {
        return RecordUtils.getString(str, type, useHex, false);
    }
    static getString_1(str, type, useHex, useEnvCharset) {
        let result;
        if (useHex) {
            if (type === StorageAttribute.BLOB || type === StorageAttribute.BLOB_VECTOR
                || type === StorageAttribute.DOTNET)
                result = RecordUtils.byteStreamToString(str);
            else
                result = str;
        }
        else {
            if (type === StorageAttribute.BLOB || type === StorageAttribute.BLOB_VECTOR
                || type === StorageAttribute.DOTNET)
                result = Base64.decode(str, useEnvCharset ? GuiEnvironment.Environment.GetEncoding() : null);
            else
                result = Base64.decodeToHex(str);
        }
        return result;
    }
    static getSpannedField(fldsVal, firstSegLen, idx, type, result, useHex, noOfPackets) {
        if (arguments.length === 6)
            return RecordUtils.getSpannedField_0(fldsVal, firstSegLen, idx, type, result, useHex);
        return RecordUtils.getSpannedField_1(fldsVal, firstSegLen, idx, type, result, useHex, noOfPackets);
    }
    static getSpannedField_0(fldsVal, firstSegLen, idx, type, result, useHex) {
        let endIdx = idx + firstSegLen;
        let len;
        let begin = idx;
        let asciiCode;
        let tmp;
        let suffixBuf = null;
        let parsedLen;
        if (endIdx > fldsVal.length) {
            throw new ApplicationException("in Record.getSpannedField() data string too short:\n" + fldsVal);
        }
        result.Remove(0, result.Length);
        result.Append(RecordUtils.getString(fldsVal.substr(idx, endIdx - idx), type, useHex));
        idx = idx + firstSegLen;
        endIdx = idx + 4;
        tmp = fldsVal.substr(idx, endIdx - idx);
        len = NNumber.Parse(tmp, NumberStyles.HexNumber);
        idx = endIdx;
        if (useHex)
            endIdx = idx + len * 2;
        else
            endIdx = idx + Math.floor((len + 2) / 3) * 4;
        if (endIdx > fldsVal.length)
            throw new ApplicationException("in Record.getSpannedField() data string too short:\n" + fldsVal);
        while (idx < endIdx) {
            tmp = fldsVal.substr(idx, 2);
            asciiCode = String.fromCharCode(NNumber.Parse(tmp, NumberStyles.HexNumber));
            result.Append(asciiCode);
            idx = idx + 2;
        }
        endIdx = idx + 4;
        tmp = fldsVal.substr(idx, endIdx - idx);
        len = NNumber.Parse(tmp, NumberStyles.HexNumber);
        idx = endIdx;
        if ((len & 0x8000) > 0) {
            suffixBuf = new StringBuilder();
            len = (len & 0x7FFF);
            if (type === StorageAttribute.BLOB || type === StorageAttribute.BLOB_VECTOR)
                if (useHex)
                    len = len * 2;
                else
                    len = Math.floor((len + 2) / 3) * 4;
            parsedLen = RecordUtils.getSpannedField(fldsVal, len, idx, type, suffixBuf, useHex);
            result.Append(suffixBuf.ToString());
            idx = idx + parsedLen;
        }
        else {
            if (type === StorageAttribute.BLOB || type === StorageAttribute.BLOB_VECTOR)
                if (useHex)
                    len = len * 2;
                else
                    len = Math.floor((len + 2) / 3) * 4;
            endIdx = idx + len;
            if (endIdx > fldsVal.length)
                throw new ApplicationException("in Record.fillFieldsData() data string too short:\n" + fldsVal);
            result.Append(RecordUtils.getString(fldsVal.substr(idx, endIdx - idx), type, useHex));
            idx = endIdx;
        }
        return idx - begin;
    }
    static getSpannedField_1(fldsVal, firstSegLen, idx, type, result, useHex, noOfPackets) {
        let endIdx = idx + firstSegLen;
        let len;
        let begin = idx;
        let tmp;
        let suffixBuf = null;
        let parsedLen;
        let tmpEnc;
        if (UtilStrByteMode.isLocaleDefLangDBCS())
            tmpEnc = ISO_8859_1_Encoding.ISO_8859_1;
        else
            tmpEnc = GuiEnvironment.Environment.GetEncoding();
        if (endIdx > fldsVal.length)
            throw new ApplicationException("in Record.getSpannedField() data string too short:\n" + fldsVal);
        result.Remove(0, result.Length);
        result.Append(tmpEnc.GetString(fldsVal, idx, firstSegLen));
        noOfPackets = noOfPackets - 1;
        if (noOfPackets <= 0)
            return 0;
        idx = idx + firstSegLen;
        endIdx = idx + 4;
        tmp = tmpEnc.GetString(fldsVal, idx, 4);
        len = NNumber.Parse(tmp, NumberStyles.HexNumber);
        idx = endIdx;
        if (endIdx > fldsVal.length)
            throw new ApplicationException("in Record.getSpannedField() data string too short:\n" + fldsVal);
        endIdx = idx + 4;
        tmp = tmpEnc.GetString(fldsVal, idx, 4);
        len = NNumber.Parse(tmp, NumberStyles.HexNumber);
        idx = endIdx;
        if ((len & 0x8000) > 0) {
            suffixBuf = new StringBuilder();
            parsedLen = RecordUtils.getSpannedField(fldsVal, len, idx, type, suffixBuf, useHex, noOfPackets);
            result.Append(suffixBuf.ToString());
            idx = idx + parsedLen;
        }
        else {
            endIdx = idx + len;
            if (endIdx > fldsVal.length)
                throw new ApplicationException("in Record.fillFieldsData() data string too short:\n" + fldsVal);
            result.Append(tmpEnc.GetString(fldsVal, idx, len));
            idx = endIdx;
        }
        return idx - begin;
    }
    static itemValToXML(itemVal, itemAttr, cellAttr, ToBase64) {
        let tmpBuf = RecordUtils.serializeItemVal(itemVal, itemAttr, cellAttr, ToBase64);
        return XmlParser.escape(tmpBuf).toString();
    }
    static getString_2(str, type) {
        let useHex = GuiEnvironment.Environment.GetDebugLevel() > 1 || type === StorageAttribute.ALPHA
            || type === StorageAttribute.UNICODE || StorageAttributeCheck.isTypeLogical(type);
        let tmp;
        if (type === StorageAttribute.ALPHA || type === StorageAttribute.UNICODE ||
            type === StorageAttribute.BLOB || type === StorageAttribute.BLOB_VECTOR) {
            tmp = str.substr(4);
        }
        else
            tmp = str;
        return RecordUtils.getString(tmp, type, useHex);
    }
}
//# sourceMappingURL=data:application/json;base64,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