import { ApplicationException, Encoding, ISO_8859_1_Encoding, List, NNumber, NString, StringBuilder } from '@magic-xpa/mscorelib';
import { BlobType } from './BlobType';
import { StorageAttribute, StorageAttributeCheck, StrUtil, UtilStrByteMode } from '@magic-xpa/utils';
import { DisplayConvertor } from '../gui/DisplayConvertor';
import { RecordUtils } from './RecordUtils';
import { GuiConstants } from '../../GuiConstants';
import { GuiEnvironment } from '../../GuiEnvironment';
import { VectorTypeBase } from './VectorTypeBase';
export class VectorType extends VectorTypeBase {
    constructor(cellsTypeOrBlobStringOrFld, contentType, defualt, isDefNull, nullAlowed, length) {
        super();
        this._cells = null;
        this._encoding = null;
        this._allowNull = false;
        this._cellContentType = null;
        this._cellSize = 0;
        this._cellsAttr = null;
        this._cellsDefaultVal = null;
        this._cellsIsNullDefault = false;
        this._dataBuf = null;
        this._initialized = false;
        this._nullBuf = null;
        this._originalflatData = null;
        if (arguments.length === 6)
            this.constructor_0(cellsTypeOrBlobStringOrFld, contentType, defualt, isDefNull, nullAlowed, length);
        else if (arguments.length === 1 && (cellsTypeOrBlobStringOrFld === null || cellsTypeOrBlobStringOrFld.constructor === String))
            this.constructor_1(cellsTypeOrBlobStringOrFld);
        else
            this.constructor_2(cellsTypeOrBlobStringOrFld);
    }
    constructor_0(cellsType, contentType, defualt, isDefNull, nullAlowed, length) {
        this._cells = new List();
        this._cellsAttr = cellsType;
        this._cellContentType = contentType;
        this._cellsDefaultVal = defualt;
        this._cellsIsNullDefault = isDefNull;
        this._cellSize = ((this._cellsAttr === StorageAttribute.UNICODE) ? (length * 2) : length);
        this._initialized = true;
        this._allowNull = nullAlowed;
        this._nullBuf = new StringBuilder();
        this._dataBuf = new StringBuilder();
        this._originalflatData = this.toString();
        this._encoding = GuiEnvironment.Environment.GetEncoding();
    }
    constructor_1(blobString) {
        this._cells = new List();
        this._initialized = false;
        this._originalflatData = blobString;
        this._encoding = GuiEnvironment.Environment.GetEncoding();
    }
    constructor_2(fld) {
        this._cells = new List();
        this._cellsAttr = fld.getCellsType();
        this._cellContentType = fld.getVecCellsContentType();
        this._cellsDefaultVal = fld.getCellDefualtValue();
        this._cellsIsNullDefault = fld.isNullDefault();
        this._cellSize = ((this._cellsAttr === StorageAttribute.UNICODE) ? (fld.getVecCellsSize() * 2) : fld.getVecCellsSize());
        this._initialized = true;
        this._allowNull = fld.NullAllowed;
        this._nullBuf = new StringBuilder();
        this._dataBuf = new StringBuilder();
        this._originalflatData = this.toString();
        this._encoding = GuiEnvironment.Environment.GetEncoding();
    }
    init() {
        let tokens = null;
        let tokenIdx = 0;
        let currentToken = null;
        let isFlatDataTranslated = this.isUnicode(this._originalflatData);
        let isDbcsAlpha = false;
        let pos = BlobType.blobPrefixLength(this._originalflatData);
        tokens = StrUtil.tokenize(this._originalflatData.substr(pos), ",");
        let vecSize = 0;
        if (!this._initialized) {
            if (VectorType.validateBlobContents(this._originalflatData)) {
                let blobPrefix = BlobType.getPrefix(this._originalflatData);
                this._cellContentType = BlobType.getContentType(blobPrefix);
                pos = (pos + 4 + 5 + 1);
                tokenIdx++;
                currentToken = tokens[tokenIdx];
                pos = pos + (currentToken.length + 1);
                tokenIdx++;
                currentToken = tokens[tokenIdx];
                pos = pos + (currentToken.length + 1);
                tokenIdx++;
                currentToken = tokens[tokenIdx];
                this._cellsAttr = currentToken.charAt(0);
                pos = pos + 2;
                tokenIdx++;
                currentToken = tokens[tokenIdx];
                this._cellSize = NNumber.Parse(currentToken);
                pos = pos + (currentToken.length + 1);
                if (this._cellsAttr === StorageAttribute.NUMERIC || this._cellsAttr === StorageAttribute.DATE || this._cellsAttr === StorageAttribute.TIME)
                    this._cellsDefaultVal = StrUtil.stringToHexaDump(this._originalflatData.substr(pos, this._cellSize), 2);
                else if (this._cellsAttr === StorageAttribute.ALPHA || this._cellsAttr === StorageAttribute.MEMO) {
                    if (UtilStrByteMode.isLocaleDefLangDBCS()) {
                        isDbcsAlpha = true;
                        isFlatDataTranslated = true;
                        this._cellsDefaultVal = UtilStrByteMode.leftB(this._originalflatData.substr(pos), this._cellSize);
                    }
                    else
                        this._cellsDefaultVal = this._originalflatData.substr(pos, this._cellSize);
                    if (this._encoding !== null && !isFlatDataTranslated) {
                        try {
                            let ba = ISO_8859_1_Encoding.ISO_8859_1.GetBytes(this._cellsDefaultVal);
                            this._cellsDefaultVal = this._encoding.GetString(ba, 0, ba.length);
                        }
                        catch (Exception) {
                        }
                    }
                }
                else if (this._cellsAttr === StorageAttribute.UNICODE) {
                    this._cellsDefaultVal = this._originalflatData.substr(pos, this._cellSize);
                    try {
                        let ba = ISO_8859_1_Encoding.ISO_8859_1.GetBytes(this._cellsDefaultVal);
                        this._cellsDefaultVal = Encoding.Unicode.GetString(ba, 0, ba.length);
                    }
                    catch (Exception) {
                    }
                }
                else
                    this._cellsDefaultVal = this._originalflatData.substr(pos, this._cellSize);
                if (isDbcsAlpha)
                    pos = pos + this._cellsDefaultVal.length + 1;
                else
                    pos = (pos + this._cellSize + 1);
                tokens = StrUtil.tokenize(this._originalflatData.substr(pos), ",");
                tokenIdx = 0;
                currentToken = tokens[tokenIdx];
                this._cellsIsNullDefault = DisplayConvertor.toBoolean(currentToken);
                pos = pos + (currentToken.length + 1);
                tokenIdx++;
                currentToken = tokens[tokenIdx];
                this._allowNull = DisplayConvertor.toBoolean(currentToken);
                pos = pos + (currentToken.length + 1);
                tokenIdx++;
                currentToken = tokens[tokenIdx];
                pos = pos + (currentToken.length + 1);
                tokenIdx++;
                currentToken = tokens[tokenIdx];
                vecSize = NNumber.Parse(currentToken);
                pos = pos + (currentToken.length + 1);
                tokenIdx++;
                currentToken = tokens[tokenIdx];
                pos = pos + (currentToken.length + 1);
                if (this._cellsAttr === StorageAttribute.BLOB || this._cellsAttr === StorageAttribute.BLOB_VECTOR) {
                    tokenIdx++;
                    currentToken = tokens[tokenIdx];
                    pos = pos + (currentToken.length + 1);
                }
                let data;
                if (isDbcsAlpha) {
                    data = UtilStrByteMode.leftB(this._originalflatData.substr(pos), (vecSize * this._cellSize));
                    pos = pos + data.length + 1;
                }
                else {
                    data = this._originalflatData.substr(pos, (vecSize * this._cellSize));
                    pos = (pos + vecSize * this._cellSize + 1);
                }
                this._dataBuf = new StringBuilder(data.length);
                if (this._cellsAttr !== StorageAttribute.ALPHA && this._cellsAttr !== StorageAttribute.MEMO && this._cellsAttr !== StorageAttribute.UNICODE)
                    this._dataBuf.Append(data);
                let nullBuf = this._originalflatData.substr(pos, vecSize);
                pos = (pos + vecSize + 1);
                this._nullBuf = new StringBuilder(nullBuf);
                if (this._cellsAttr !== StorageAttribute.BLOB && this._cellsAttr !== StorageAttribute.BLOB_VECTOR) {
                    for (let i = 0; i < vecSize; i++) {
                        let isNull = nullBuf.charAt(i) > '\0';
                        if (this._cellsAttr === StorageAttribute.NUMERIC || this._cellsAttr === StorageAttribute.DATE || this._cellsAttr === StorageAttribute.TIME)
                            this._cells.push(new CellElement(StrUtil.stringToHexaDump(data.substr((i * this._cellSize), this._cellSize), 2), isNull));
                        else if (this._cellsAttr === StorageAttribute.ALPHA || this._cellsAttr === StorageAttribute.MEMO) {
                            let cellData;
                            if (isDbcsAlpha)
                                cellData = UtilStrByteMode.midB(data, (i * this._cellSize), this._cellSize);
                            else
                                cellData = data.substr((i * this._cellSize), this._cellSize);
                            if (this._encoding !== null && !isFlatDataTranslated) {
                                try {
                                    let ba = ISO_8859_1_Encoding.ISO_8859_1.GetBytes(cellData);
                                    cellData = this._encoding.GetString(ba, 0, ba.length);
                                }
                                catch (SystemException) {
                                }
                            }
                            this._dataBuf.Append(cellData);
                            this._cells.push(new CellElement(cellData, isNull));
                        }
                        else if (this._cellsAttr === StorageAttribute.UNICODE) {
                            let cellData = data.substr((i * this._cellSize), this._cellSize);
                            try {
                                let ba = ISO_8859_1_Encoding.ISO_8859_1.GetBytes(cellData);
                                cellData = Encoding.Unicode.GetString(ba, 0, ba.length);
                            }
                            catch (SystemException) {
                            }
                            this._dataBuf.Append(cellData);
                            this._cells.push(new CellElement(cellData, isNull));
                        }
                        else
                            this._cells.push(new CellElement(data.substr((i * this._cellSize), this._cellSize), isNull));
                    }
                }
                else {
                    for (let i = 0; i < vecSize; i++) {
                        tokens = StrUtil.tokenize(this._originalflatData.substr(pos), ",");
                        tokenIdx = 0;
                        tokenIdx++;
                        currentToken = tokens[tokenIdx];
                        let size = NNumber.Parse(currentToken);
                        pos = pos + (currentToken.length + 1);
                        let blobHeader = "";
                        for (let j = 0; j < GuiConstants.BLOB_PREFIX_ELEMENTS_COUNT; j++) {
                            tokenIdx++;
                            currentToken = tokens[tokenIdx];
                            blobHeader = blobHeader + currentToken;
                            pos = pos + (currentToken.length + 1);
                            if (j !== GuiConstants.BLOB_PREFIX_ELEMENTS_COUNT - 1) {
                                blobHeader = blobHeader + ",";
                            }
                        }
                        let cellData = this._originalflatData.substr(pos, size);
                        let isNull = nullBuf.charAt(i) > '\0';
                        this._cells.push(new CellElement(cellData, isNull, blobHeader));
                        pos = (pos + size + 1);
                    }
                }
                if (this._dataBuf.Length === 0)
                    this._dataBuf = new StringBuilder(data.length);
                this._initialized = true;
            }
            else
                throw new ApplicationException("in VectorType.init wrong vector format");
        }
    }
    toString() {
        let res = "";
        if (this._initialized) {
            res = res + 0 + "," + 0 + "," + "0" + "," + this._cellsAttr + "," + this._cellContentType + ";";
            res = res + this.buildHeadersString();
            if (this._cellsAttr === StorageAttribute.UNICODE) {
                let dataBufCharArry = new Array(this._dataBuf.Length * 2);
                for (let i = 0; i < this._dataBuf.Length; i = i + 1) {
                    dataBufCharArry[i * 2] = String.fromCharCode((this._dataBuf.get_Item(i).charCodeAt(0)) % 256);
                    dataBufCharArry[i * 2 + 1] = String.fromCharCode((this._dataBuf.get_Item(i).charCodeAt(0)) / 256);
                }
                res = res + NString.FromChars(dataBufCharArry) + ",";
            }
            else
                res = res + this._dataBuf + ",";
            res = res + this._nullBuf + ",";
            if (this._cellsAttr === StorageAttribute.BLOB || this._cellsAttr === StorageAttribute.BLOB_VECTOR)
                res = res + this.getBlobsBuf();
            res = res + VectorType.BLOB_TABLE_STR;
        }
        else
            res = this._originalflatData;
        return res;
    }
    getCellsAttr() {
        if (!this._initialized)
            return VectorType.getCellsAttr(this._originalflatData);
        else
            return this._cellsAttr;
    }
    getCellSize() {
        let retVal;
        if (!this._initialized)
            retVal = VectorType.getCellSize(this._originalflatData);
        else
            retVal = this._cellSize;
        if (this.getCellsAttr() === StorageAttribute.UNICODE)
            retVal = Math.floor(retVal / 2);
        return retVal;
    }
    getVecSize() {
        if (!this._initialized)
            return VectorType.getVecSize(this._originalflatData);
        else
            return this._cells.length;
    }
    getVecCell(idx) {
        let retVal = null;
        if (idx > 0) {
            this.init();
            if (idx <= this.getVecSize()) {
                retVal = this._cells.get_Item(idx - 1).data;
                if (this._cellsAttr === StorageAttribute.BLOB || this._cellsAttr === StorageAttribute.BLOB_VECTOR)
                    retVal = this._cells.get_Item(idx - 1).blobFieldPrefix + ";" + retVal;
                if (StorageAttributeCheck.isTypeLogical(this._cellsAttr))
                    retVal = ((retVal.charAt(0) === '\0') ? "0" : "1");
                if (this._cells.get_Item(idx - 1).isNull)
                    retVal = null;
            }
            else if (!this._cellsIsNullDefault)
                retVal = this._cellsDefaultVal;
        }
        return retVal;
    }
    GetCellValues() {
        let retVal = null;
        let cellValues = null;
        this.init();
        if (this.getVecSize() > 0) {
            cellValues = new Array(this.getVecSize());
            for (let idx = 0; idx < this.getVecSize(); idx++) {
                retVal = this._cells.get_Item(idx).data;
                if (this._cellsAttr === StorageAttribute.BLOB || this._cellsAttr === StorageAttribute.BLOB_VECTOR)
                    retVal = this._cells.get_Item(idx).blobFieldPrefix + ";" + retVal;
                if (StorageAttributeCheck.isTypeLogical(this._cellsAttr))
                    retVal = ((retVal.charAt(0) === '\0') ? "0" : "1");
                if (this._cells.get_Item(idx).isNull)
                    retVal = null;
                cellValues[idx] = retVal;
            }
        }
        else if (!this._cellsIsNullDefault) {
            cellValues = [
                this._cellsDefaultVal
            ];
        }
        return cellValues;
    }
    setVecCell(idx, newValue, isNull) {
        let res = false;
        let createBufferValForNumType = true;
        try {
            if (idx > 0) {
                this.init();
                let localCellSize = (this._cellsAttr === StorageAttribute.UNICODE) ? Math.floor(this._cellSize / 2) : this._cellSize;
                if (isNull && !this._allowNull) {
                    isNull = false;
                    newValue = this._cellsDefaultVal;
                }
                if (idx <= this._cells.length) {
                    let curr = this._cells.get_Item(idx - 1);
                    if (!isNull) {
                        if (this._cellsAttr === StorageAttribute.BLOB || this._cellsAttr === StorageAttribute.BLOB_VECTOR) {
                            let blobPrefixLength = BlobType.blobPrefixLength(newValue);
                            curr.blobFieldPrefix = newValue.substr(0, blobPrefixLength - 1);
                            if (newValue.length > blobPrefixLength)
                                curr.data = newValue.substr(blobPrefixLength);
                            else
                                curr.data = "";
                        }
                        else {
                            if (StorageAttributeCheck.isTypeLogical(this._cellsAttr)) {
                                curr.data = (DisplayConvertor.toBoolean(newValue) ? new StringBuilder().Append(1).ToString() : new StringBuilder().Append('\0').ToString());
                                newValue = curr.data;
                            }
                            else
                                curr.data = newValue;
                            let dataBufVal = newValue;
                            if (this._cellsAttr === StorageAttribute.NUMERIC || this._cellsAttr === StorageAttribute.DATE || this._cellsAttr === StorageAttribute.TIME)
                                dataBufVal = RecordUtils.byteStreamToString(dataBufVal);
                            if (UtilStrByteMode.isLocaleDefLangDBCS() && (this._cellsAttr === StorageAttribute.ALPHA || this._cellsAttr === StorageAttribute.MEMO)) {
                                let baDataBuf = this._encoding.GetBytes(this._dataBuf.ToString());
                                let baDataBufVal = this._encoding.GetBytes(dataBufVal);
                                for (let i = 0; i < baDataBufVal.length && i < localCellSize; i++)
                                    baDataBuf[((idx - 1) * localCellSize) + i] = baDataBufVal[i];
                                for (let i = baDataBufVal.length; i < localCellSize; i++)
                                    baDataBuf[((idx - 1) * localCellSize) + i] = ((this._cellsAttr === StorageAttribute.ALPHA) ? 32 : 0);
                                this._dataBuf = new StringBuilder(this._encoding.GetString(baDataBuf, 0, baDataBuf.length));
                                curr.data = this._encoding.GetString(baDataBuf, ((idx - 1) * localCellSize), localCellSize);
                            }
                            else {
                                for (let i = 0; i < dataBufVal.length && i < localCellSize; i++)
                                    this._dataBuf.set_Item(((idx - 1) * localCellSize) + i, dataBufVal.charAt(i));
                                for (let i = dataBufVal.length; i < localCellSize; i++)
                                    this._dataBuf.set_Item(((idx - 1) * localCellSize) + i, (this._cellsAttr === StorageAttribute.ALPHA || this._cellsAttr === StorageAttribute.UNICODE) ? ' ' : '\0');
                                if (this._cellsAttr === StorageAttribute.ALPHA || this._cellsAttr === StorageAttribute.UNICODE || this._cellsAttr === StorageAttribute.MEMO)
                                    curr.data = this._dataBuf.ToString().substr(((idx - 1) * localCellSize), localCellSize);
                            }
                        }
                        this._nullBuf.set_Item(idx - 1, '\0');
                        curr.isNull = false;
                    }
                    else {
                        curr.data = null;
                        if (this._cellsAttr === StorageAttribute.BLOB_VECTOR)
                            curr.blobFieldPrefix = BlobType.getEmptyBlobPrefix('\u0001');
                        else
                            curr.blobFieldPrefix = BlobType.getEmptyBlobPrefix('\0');
                        curr.isNull = true;
                        this._nullBuf.set_Item(idx - 1, '\u0001');
                        for (let i = 0; i < localCellSize; i++)
                            this._dataBuf.set_Item(((idx - 1) * localCellSize) + i, '\0');
                        this._nullBuf.set_Item(idx - 1, '\u0001');
                    }
                    res = true;
                }
                else {
                    let insertVal;
                    if (this._cellsAttr === StorageAttribute.BLOB || this._cellsAttr === StorageAttribute.BLOB_VECTOR) {
                        if (this._cellsAttr === StorageAttribute.BLOB_VECTOR)
                            insertVal = BlobType.getEmptyBlobPrefix('\u0001');
                        else
                            insertVal = BlobType.getEmptyBlobPrefix('\0');
                        if (!this._cellsIsNullDefault)
                            insertVal = insertVal + this._cellsDefaultVal;
                    }
                    else if (this._cellsIsNullDefault)
                        insertVal = null;
                    else
                        insertVal = this._cellsDefaultVal;
                    if (insertVal !== null && StorageAttributeCheck.isTypeLogical(this._cellsAttr))
                        insertVal = (DisplayConvertor.toBoolean(insertVal) ? new StringBuilder().Append('\u0001').ToString() : new StringBuilder().Append('\0').ToString());
                    let dataBufVal = insertVal;
                    while (this._cells.length < idx) {
                        if (this._cellsAttr === StorageAttribute.BLOB || this._cellsAttr === StorageAttribute.BLOB_VECTOR) {
                            this._cells.push(new CellElement(insertVal.substr(VectorType.EMPTY_BLOB_PREFIX_LEN), this._cellsIsNullDefault, insertVal.substr(0, VectorType.EMPTY_BLOB_PREFIX_LEN)));
                            this._dataBuf.Append(this.getNullString(VectorType.BLOB_TYPE_STRUCT_SIZE));
                        }
                        else {
                            this._cells.push(new CellElement(insertVal, this._cellsIsNullDefault));
                            if (insertVal !== null) {
                                if (this._cellsAttr === StorageAttribute.NUMERIC || this._cellsAttr === StorageAttribute.DATE || this._cellsAttr === StorageAttribute.TIME) {
                                    if (createBufferValForNumType) {
                                        createBufferValForNumType = false;
                                        dataBufVal = RecordUtils.byteStreamToString(insertVal);
                                    }
                                }
                                this._dataBuf.Append(dataBufVal);
                                let valLen;
                                if (UtilStrByteMode.isLocaleDefLangDBCS() && (this._cellsAttr === StorageAttribute.ALPHA || this._cellsAttr === StorageAttribute.MEMO))
                                    valLen = UtilStrByteMode.lenB(dataBufVal);
                                else
                                    valLen = dataBufVal.length;
                                for (let i = valLen; i < localCellSize; i++)
                                    this._dataBuf.Append((this._cellsAttr === StorageAttribute.ALPHA || this._cellsAttr === StorageAttribute.UNICODE) ? ' ' : '\0');
                            }
                            else
                                this._dataBuf.Append(this.getNullString(localCellSize));
                        }
                        this._nullBuf.Insert(this._cells.length - 1, this._cellsIsNullDefault ? '\u0001' : '\0');
                    }
                    res = this.setVecCell(idx, newValue, isNull);
                }
            }
        }
        catch (ApplicationException) {
            res = false;
        }
        return res;
    }
    adjustToFit(field) {
        if (field.getType() === StorageAttribute.BLOB_VECTOR) {
            let srcAttr = this.getCellsAttr();
            let dstAttr = field.getCellsType();
            if (StorageAttributeCheck.isTheSameType(srcAttr, dstAttr)) {
                this.init();
                if (StorageAttributeCheck.IsTypeAlphaOrUnicode(srcAttr) && StorageAttributeCheck.IsTypeAlphaOrUnicode(dstAttr)) {
                    let dstSizeInChars = field.getVecCellsSize();
                    let srcSizeInChars = ((this._cellsAttr === StorageAttribute.UNICODE) ? Math.floor(this._cellSize / 2) : this._cellSize);
                    let isByteMode = UtilStrByteMode.isLocaleDefLangDBCS() && StorageAttributeCheck.isTypeAlpha(dstAttr);
                    if (srcSizeInChars !== dstSizeInChars) {
                        let adjData = new StringBuilder();
                        for (let i = 0; i < dstSizeInChars; i = i + 1) {
                            let curr = this._cells.get_Item(i);
                            if (!curr.isNull) {
                                if (srcSizeInChars > dstSizeInChars) {
                                    if (isByteMode)
                                        curr.data = UtilStrByteMode.leftB(curr.data, dstSizeInChars);
                                    else
                                        curr.data = curr.data.substr(0, dstSizeInChars);
                                    adjData.Append(curr.data);
                                }
                                else {
                                    let tmpData = new StringBuilder();
                                    tmpData.Append(curr.data);
                                    let dataLen = isByteMode ? UtilStrByteMode.lenB(curr.data) : curr.data.length;
                                    for (let j = dataLen; j < dstSizeInChars; j = j + 1)
                                        tmpData.Append(' ');
                                    adjData.Append(tmpData.ToString());
                                    curr.data = tmpData.ToString();
                                }
                            }
                            else {
                                let tmpData = new StringBuilder();
                                for (let j = 0; j < dstSizeInChars; j = j + 1)
                                    tmpData.Append('\0');
                                adjData.Append(tmpData.ToString());
                                curr.data = tmpData.ToString();
                            }
                        }
                        this._dataBuf = adjData;
                    }
                }
                let newSize = field.getVecCellsSize();
                this._cellsAttr = field.getCellsType();
                this._cellSize = ((this._cellsAttr === StorageAttribute.UNICODE) ? (newSize * 2) : newSize);
                this._cellsDefaultVal = field.getCellDefualtValue();
                this._cellsIsNullDefault = field.isNullDefault();
                this._allowNull = field.NullAllowed;
                this._originalflatData = this.toString();
                return;
            }
            else
                throw new ApplicationException("in VectorType.adjustToFit vector basic types does not agree");
        }
        else
            throw new ApplicationException("in  VectorType.adjustToFit " + field.getName() + " is not of type vector");
    }
    buildHeadersString() {
        let res = VectorType.BLOB_TABLE_STR + VectorType.VECTOR_STR + "," + VectorType.VERSION + "," + 1 + "," + this._cellsAttr + ",";
        res = res + this._cellSize + ",";
        let def = "";
        if (this._cellsDefaultVal !== null)
            if (this._cellsAttr === StorageAttribute.NUMERIC || this._cellsAttr === StorageAttribute.DATE || this._cellsAttr === StorageAttribute.TIME)
                def = RecordUtils.byteStreamToString(this._cellsDefaultVal);
            else
                def = this._cellsDefaultVal;
        if (UtilStrByteMode.isLocaleDefLangDBCS() && (this._cellsAttr === StorageAttribute.ALPHA || this._cellsAttr === StorageAttribute.MEMO))
            def = def + this.getEmptyString(this._cellSize - UtilStrByteMode.lenB(def));
        else
            def = def + this.getEmptyString(this._cellSize - def.length);
        res = res + def + "," + (this._cellsIsNullDefault ? "1" : "0") + ",";
        res = res + (this._allowNull ? "1" : "0") + "," + this._cellSize + "," + this._cells.length + ",";
        res = res + ((this._cellsAttr === StorageAttribute.BLOB || this._cellsAttr === StorageAttribute.BLOB_VECTOR) ? "1" : "0") + ",";
        if (this._cellsAttr === StorageAttribute.BLOB || this._cellsAttr === StorageAttribute.BLOB_VECTOR)
            res = res + "0,";
        return res;
    }
    getNullString(size) {
        let res = new StringBuilder();
        for (let i = 0; i < size; i = i + 1)
            res.Append('\0');
        return res.ToString();
    }
    getEmptyString(size) {
        let res = new StringBuilder();
        for (let i = 0; i < size; i = i + 1)
            res.Append(' ');
        return res.ToString();
    }
    getBlobsBuf() {
        let res = new StringBuilder();
        for (let i = 0; i < this._cells.length; i = i + 1) {
            let data = "";
            let blobSize = 0;
            let curr = this._cells.get_Item(i);
            if (curr.data !== null) {
                data = curr.data;
                blobSize = data.length;
            }
            res.Append(blobSize);
            res.Append("," + curr.blobFieldPrefix + "," + data + ";");
        }
        return (res + ",");
    }
    static getCellsAttr(blob) {
        if (VectorType.validateBlobContents(blob)) {
            let tokens = StrUtil.tokenize(blob.substr(BlobType.blobPrefixLength(blob)), ",");
            return tokens[3].charAt(0);
        }
        else
            throw new ApplicationException("in static getCellsAttr the blob is in the wrong format");
    }
    static getCellSize(blob) {
        let cellsType = VectorType.getCellsAttr(blob);
        if (cellsType !== StorageAttribute.BLOB && cellsType !== StorageAttribute.BLOB_VECTOR) {
            let tokens = StrUtil.tokenize(blob.substr(BlobType.blobPrefixLength(blob)), ",");
            return NNumber.Parse(tokens[4]);
        }
        else {
            return 2147483647;
        }
    }
    static getVecSize(blob) {
        if (VectorType.validateBlobContents(blob)) {
            let pos = BlobType.blobPrefixLength(blob);
            let tokens = StrUtil.tokenize(blob.substr(pos), ",");
            pos = pos + (tokens[0].length + 1);
            pos = pos + (tokens[1].length + 1);
            pos = pos + (tokens[2].length + 1);
            pos = pos + (tokens[3].length + 1);
            let cellsSize = tokens[4];
            pos = pos + (cellsSize.length + 1);
            pos = pos + (NNumber.Parse(cellsSize) + 1);
            tokens = StrUtil.tokenize(blob.substr(pos), ",");
            return NNumber.Parse(tokens[3]);
        }
        throw new ApplicationException("in static getVecSize the blob is in the wrong format");
    }
    isUnicode(flatData) {
        let start = BlobType.blobPrefixLength(flatData);
        let catcher = flatData.substr(start, VectorType.BLOB_TABLE_STR_LEN);
        return (catcher === VectorType.BLOB_TABLE_STR);
    }
}
VectorType.VERSION = 5;
VectorType.EMPTY_BLOB_PREFIX_LEN = 7;
VectorType.BLOB_TYPE_STRUCT_SIZE = 28;
export class CellElement {
    constructor(val, is_null, ctrlData) {
        this.blobFieldPrefix = null;
        this.data = null;
        this.isNull = false;
        if (arguments.length === 2)
            this.constructor_0(val, is_null);
        this.constructor_1(val, is_null, ctrlData);
    }
    constructor_0(val, is_null) {
        this.data = val;
        this.isNull = is_null;
    }
    constructor_1(val, is_null, ctrlData) {
        this.data = val;
        this.isNull = is_null;
        this.blobFieldPrefix = ctrlData;
    }
}
//# sourceMappingURL=data:application/json;base64,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