import { ApplicationException, NString } from '@magic-xpa/mscorelib';
import { Rtf, StorageAttribute, StorageAttributeCheck, StrUtil } from '@magic-xpa/utils';
import { Manager } from '../../Manager';
import { NUM_TYPE } from '../data/NUM_TYPE';
import { BlobType } from '../data/BlobType';
import { DisplayConvertor } from '../gui/DisplayConvertor';
export class GuiExpressionEvaluator {
    constructor() {
        this.ExpTask = null;
        this.Events = null;
    }
    eval_op_statusbar_set_text(resVal, statusBarText) {
        resVal.Attr = StorageAttribute.UNICODE;
        resVal.StrVal = Manager.GetCurrentRuntimeContext().DefaultStatusMsg;
        if (Manager.GetCurrentRuntimeContext().DefaultStatusMsg === null) {
            resVal.StrVal = '';
        }
        else {
            resVal.StrVal = Manager.GetCurrentRuntimeContext().DefaultStatusMsg;
        }
        resVal.IsNull = false;
        if (!(statusBarText.Attr !== StorageAttribute.ALPHA && statusBarText.Attr !== StorageAttribute.UNICODE)) {
            if (!statusBarText.IsNull) {
                let text = statusBarText.StrVal;
                let task = this.ExpTask.GetContextTask();
                if (task.getForm() === null && Manager.GetCurrentRuntimeContext().FrameForm !== null) {
                    task = Manager.GetCurrentRuntimeContext().FrameForm.getTask();
                }
                task.WriteToMessagePane(text);
                Manager.GetCurrentRuntimeContext().DefaultStatusMsg = text;
            }
        }
    }
    IsParallel() {
        return false;
    }
    eval_op_ctrl_name(resVal) {
        let lastClickedCtrlName = Manager.GetCurrentRuntimeContext().LastClickedCtrlName;
        resVal.StrVal = (NString.IsNullOrEmpty(lastClickedCtrlName) ? '' : lastClickedCtrlName);
        resVal.Attr = StorageAttribute.ALPHA;
    }
    eval_op_gotoCtrl(ctrlName, rowNum, generation, retVal) {
        retVal.Attr = StorageAttribute.BOOLEAN;
        retVal.BoolVal = false;
        let task = this.GetContextTask(generation.MgNumVal.NUM_2_LONG());
        if (task === null || task.getForm() === null)
            return;
        let iRowNum = rowNum.MgNumVal.NUM_2_LONG();
        let sCtrlName = StrUtil.rtrim(ctrlName.StrVal);
        let ctrl = task.getForm().GetCtrl(sCtrlName);
        retVal.BoolVal = this.HandleControlGoto(task, ctrl, iRowNum);
    }
    static GetContextTask(currTask, generation) {
        let task = null;
        let taskBase = currTask.GetContextTask();
        if (generation === GuiExpressionEvaluator.TRIGGER_TASK) {
            task = taskBase;
        }
        else {
            if (generation < taskBase.GetTaskDepth()) {
                task = taskBase.GetTaskAncestor(generation);
            }
        }
        return task;
    }
    GetContextTask(generation) {
        return GuiExpressionEvaluator.GetContextTask(this.ExpTask, generation);
    }
    ConstructMagicNum(resVal, i, attr) {
        resVal.MgNumVal = new NUM_TYPE();
        resVal.MgNumVal.NUM_4_LONG(i);
        resVal.Attr = attr;
    }
    SetNULL(resVal, attr) {
        resVal.IsNull = true;
        resVal.Attr = attr;
        switch (attr) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.UNICODE:
                resVal.StrVal = null;
                break;
            case StorageAttribute.TIME:
            case StorageAttribute.DATE:
            case StorageAttribute.NUMERIC:
                resVal.MgNumVal = null;
                break;
            case StorageAttribute.BOOLEAN:
                resVal.BoolVal = false;
                break;
            default:
                break;
        }
    }
    GetValidatedValue(currCtrl, oldValue, newValue) {
        let vd = currCtrl.buildCopyPicture(oldValue, newValue);
        vd.evaluate();
        let validationFailed = vd.ValidationFailed;
        let ctrlValue;
        if (validationFailed) {
            let field = currCtrl.getField();
            ctrlValue = ((field.getType() === StorageAttribute.BLOB_VECTOR) ? field.getCellDefualtValue() : field.getDefaultValue());
        }
        else {
            ctrlValue = vd.getDispValue();
        }
        return ctrlValue;
    }
    SetVal(resVal, type, val, pic) {
        switch (type) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.BLOB:
            case StorageAttribute.BLOB_VECTOR:
            case StorageAttribute.UNICODE:
                resVal.Attr = type;
                resVal.StrVal = val;
                if (type === StorageAttribute.BLOB || type === StorageAttribute.BLOB_VECTOR)
                    resVal.IncludeBlobPrefix = true;
                break;
            case StorageAttribute.NUMERIC:
            case StorageAttribute.DATE:
            case StorageAttribute.TIME:
                resVal.Attr = type;
                if (val == null)
                    resVal.MgNumVal = null;
                else if (pic == null)
                    resVal.MgNumVal = new NUM_TYPE(val);
                else
                    resVal.MgNumVal = new NUM_TYPE(val, pic, (this.ExpTask).getCompIdx());
                break;
            case StorageAttribute.BOOLEAN:
                resVal.Attr = type;
                resVal.BoolVal = DisplayConvertor.toBoolean(val);
                break;
            default:
                this.SetNULL(resVal, type);
                this.Events.WriteExceptionToLog('ExpressionEvaluator.SetVal() there is no such type : ' + type);
                break;
        }
    }
    static exp_build_string(val) {
        let name = '';
        if (val.StrVal !== null) {
            let len = val.StrVal.length;
            if (len > 0) {
                name = StrUtil.ZstringMake(val.StrVal, len);
            }
        }
        return name;
    }
    static isExpVal(obj) {
        return obj instanceof ExpVal;
    }
    BlobStringConversion(val, expectedType) {
        if (StorageAttributeCheck.IsTypeAlphaOrUnicode(expectedType)) {
            if (val.Attr === StorageAttribute.BLOB) {
                let includeBlobPrefix = val.IncludeBlobPrefix;
                if (includeBlobPrefix) {
                    let contentType = BlobType.getContentType(val.StrVal);
                    val.StrVal = BlobType.getString(val.StrVal);
                    val.IncludeBlobPrefix = false;
                    if (contentType !== BlobType.CONTENT_TYPE_BINARY && Rtf.isRtf(val.StrVal)) {
                        val.StrVal = StrUtil.GetPlainTextfromRtf(val.StrVal);
                    }
                }
                val.Attr = expectedType;
            }
            else {
                if (val.Attr === StorageAttribute.BLOB_VECTOR) {
                    let includeBlobPrefix2 = val.IncludeBlobPrefix;
                    if (includeBlobPrefix2) {
                        val.StrVal = BlobType.removeBlobPrefix(val.StrVal);
                        val.IncludeBlobPrefix = false;
                    }
                    val.Attr = expectedType;
                }
            }
        }
        else {
            if (expectedType === StorageAttribute.BLOB) {
                if (StorageAttributeCheck.IsTypeAlphaOrUnicode(val.Attr)) {
                    let contentType = (val.Attr === StorageAttribute.ALPHA) ? BlobType.CONTENT_TYPE_ANSI : BlobType.CONTENT_TYPE_UNICODE;
                    val.StrVal = BlobType.createFromString(val.StrVal, contentType);
                    val.IncludeBlobPrefix = true;
                    val.Attr = expectedType;
                }
            }
        }
    }
    eval_op_zimeread(resVal) {
        resVal.Attr = StorageAttribute.ALPHA;
        resVal.StrVal = '';
        let utilImeJpn = Manager.UtilImeJpn;
        if (utilImeJpn !== null) {
            let strImeRead = utilImeJpn.StrImeRead;
            if (!NString.IsNullOrEmpty(utilImeJpn.StrImeRead)) {
                resVal.StrVal = strImeRead;
            }
        }
    }
    ConvertExpVal(val, expectedType) {
        if (StorageAttributeCheck.StorageFldAlphaUnicodeOrBlob(val.Attr, expectedType)) {
            this.BlobStringConversion(val, expectedType);
        }
    }
}
GuiExpressionEvaluator.TRIGGER_TASK = 500000;
export class ExpVal {
    constructor(attr, isNull, mgVal) {
        this.Attr = StorageAttribute.NONE;
        this.IsNull = false;
        this.MgNumVal = null;
        this.StrVal = null;
        this.BoolVal = false;
        this.IncludeBlobPrefix = false;
        this.OriginalNull = false;
        this.VectorField = null;
        if (arguments.length === 0) {
            this.constructor_0();
            return;
        }
        this.constructor_1(attr, isNull, mgVal);
    }
    constructor_0() {
        this.Attr = StorageAttribute.NONE;
    }
    constructor_1(attr, isNull, mgVal) {
        this.Init(attr, isNull, mgVal);
    }
    Copy(src) {
        this.Attr = src.Attr;
        this.BoolVal = src.BoolVal;
        this.IncludeBlobPrefix = src.IncludeBlobPrefix;
        this.IsNull = src.IsNull;
        this.MgNumVal = src.MgNumVal;
        this.StrVal = src.StrVal;
        this.VectorField = src.VectorField;
        this.OriginalNull = src.OriginalNull;
    }
    Nullify() {
        this.Init(StorageAttribute.NONE, true, null);
    }
    Init(attr, isNull, mgVal) {
        this.Attr = attr;
        this.IsNull = isNull;
        switch (this.Attr) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.UNICODE:
                this.StrVal = mgVal;
                break;
            case StorageAttribute.BLOB:
            case StorageAttribute.BLOB_VECTOR:
                this.StrVal = mgVal;
                this.IncludeBlobPrefix = true;
                break;
            case StorageAttribute.NUMERIC:
            case StorageAttribute.DATE:
            case StorageAttribute.TIME:
                this.MgNumVal = ((mgVal !== null) ? new NUM_TYPE(mgVal) : null);
                break;
            case StorageAttribute.BOOLEAN:
                this.BoolVal = (mgVal !== null && mgVal === '1');
                break;
            case StorageAttribute.NONE:
                this.BoolVal = false;
                this.StrVal = null;
                this.MgNumVal = null;
                this.OriginalNull = true;
                this.VectorField = null;
                this.IncludeBlobPrefix = false;
                break;
            case StorageAttribute.DOTNET:
                this.StrVal = mgVal;
                break;
            default:
                throw new ApplicationException('in ExpVal.ExpVal() illegal attribute: \'' + this.Attr + '\'');
        }
    }
    ToMgVal() {
        let str;
        switch (this.Attr) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.BLOB:
            case StorageAttribute.BLOB_VECTOR:
            case StorageAttribute.UNICODE:
                str = this.StrVal;
                break;
            case StorageAttribute.NUMERIC:
            case StorageAttribute.DATE:
            case StorageAttribute.TIME:
                str = this.MgNumVal.toXMLrecord();
                break;
            case StorageAttribute.BOOLEAN:
                str = this.BoolVal
                    ? '1'
                    : '0';
                break;
            default:
                str = '[illegal attribute: ' + this.Attr + ']';
                break;
        }
        return str;
    }
    isEmptyString() {
        return (this.Attr === StorageAttribute.ALPHA || this.Attr === StorageAttribute.UNICODE) && this.StrVal === '';
    }
}
//# sourceMappingURL=data:application/json;base64,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