import { PIC } from './PIC';
import { DateTime, NString, RefParam, StringBuilder } from '@magic-xpa/mscorelib';
import { DateUtil, HTML_2_STR, MsgInterface, PICInterface, SEQ_2_HTML, StorageAttribute, StrUtil, UtilDateJpn, UtilStrByteMode } from '@magic-xpa/utils';
import { BlobType } from '../data/BlobType';
import { ValidationDetailsBase } from './ValidationDetailsBase';
import { NUM_TYPE } from '../data/NUM_TYPE';
import { HebrewDate } from './HebrewDate';
import { Events } from '../../Events';
import { GuiEnvironment } from '../../GuiEnvironment';
import { MemoryUtil } from '../../util/MemoryUtil';
export class DisplayConvertor {
    static get Instance() {
        if (DisplayConvertor._instance === null) {
            DisplayConvertor._instance = new DisplayConvertor();
        }
        return DisplayConvertor._instance;
    }
    constructor() {
        this._environment = null;
        this.DATECHAR = 0;
        this._environment = GuiEnvironment.Environment;
    }
    getNewDateBreakParams() {
        return new DateBreakParams();
    }
    getNewTimeBreakParams() {
        return new TimeBreakParams();
    }
    mg2disp(mgValue, rangeStr, pic, useNative, compIdxOrAlwaysFill, time_date_pic_Z_edtOrCompIdx, time_date_pic_Z_edtOrConvertCase, time_date_num_pic_Z_edt) {
        if (arguments.length === 6)
            return this.mg2disp_0(mgValue, rangeStr, pic, useNative, compIdxOrAlwaysFill, time_date_pic_Z_edtOrCompIdx);
        else if (arguments.length === 7)
            return this.mg2disp_1(mgValue, rangeStr, pic, useNative, compIdxOrAlwaysFill, time_date_pic_Z_edtOrCompIdx, time_date_pic_Z_edtOrConvertCase);
        else
            return this.mg2disp_2(mgValue, rangeStr, pic, useNative, compIdxOrAlwaysFill, time_date_pic_Z_edtOrCompIdx, time_date_pic_Z_edtOrConvertCase, time_date_num_pic_Z_edt);
    }
    mg2disp_0(mgValue, rangeStr, pic, useNative, compIdx, time_date_pic_Z_edt) {
        return this.mg2disp(mgValue, rangeStr, pic, useNative, false, compIdx, time_date_pic_Z_edt);
    }
    mg2disp_1(mgValue, rangeStr, pic, useNative, alwaysFill, compIdx, time_date_pic_Z_edt) {
        return this.mg2disp(mgValue, rangeStr, pic, useNative, alwaysFill, compIdx, false, time_date_pic_Z_edt);
    }
    mg2disp_2(mgValue, rangeStr, pic, useNative, alwaysFill, compIdx, convertCase, time_date_num_pic_Z_edt) {
        let str = "";
        let tmpRange = (rangeStr === null) ? "" : rangeStr;
        switch (pic.getAttr()) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.UNICODE:
                str = this.fromAlpha(mgValue, pic, tmpRange, alwaysFill, convertCase);
                break;
            case StorageAttribute.NUMERIC: {
                let newPic = pic.getUnformattedNumericPic();
                let dec = GuiEnvironment.Environment.GetDecimal();
                let thou = GuiEnvironment.Environment.GetThousands();
                GuiEnvironment.Environment.setDecimalSeparator('.');
                GuiEnvironment.Environment.setThousandsSeparator(',');
                str = this.fromNum(mgValue, newPic);
                GuiEnvironment.Environment.setDecimalSeparator(dec);
                GuiEnvironment.Environment.setThousandsSeparator(thou);
                if (newPic.getMaskChars() === 0 || !time_date_num_pic_Z_edt)
                    str = StrUtil.ltrim(str);
                if (useNative) {
                    if (str == "")
                        return null;
                    let dbl = Number(str);
                    return dbl;
                }
                else
                    return str;
            }
            case StorageAttribute.DATE: {
                if (useNative) {
                    let dt;
                    dt = this.fromMgDateToNativeDate(mgValue, pic, compIdx);
                    return dt;
                }
                else {
                    str = this.fromDate(mgValue, pic, compIdx, time_date_num_pic_Z_edt);
                    break;
                }
            }
            case StorageAttribute.TIME: {
                if (useNative) {
                    let dt;
                    dt = this.fromMgTimeToNativeTime(mgValue, pic);
                    return dt;
                }
                else {
                    str = this.fromTime(mgValue, pic, time_date_num_pic_Z_edt);
                    break;
                }
            }
            case StorageAttribute.BOOLEAN: {
                let logVal;
                logVal = this.fromMgLogicalToNativeLogical(mgValue);
                return logVal;
            }
            case StorageAttribute.BLOB:
                str = BlobType.getString(mgValue);
                break;
            case StorageAttribute.BLOB_VECTOR:
                str = mgValue;
                break;
        }
        return str;
    }
    disp2mg(dispValue, pic, compIdx, blobContentType) {
        let str = "";
        switch (pic.getAttr()) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.UNICODE:
                str = this.toAlpha(dispValue, pic);
                break;
            case StorageAttribute.NUMERIC:
                {
                    dispValue = String(dispValue);
                    let len = dispValue.length;
                    while (len > 0) {
                        if (!pic.picIsMask(len - 1) && dispValue[len - 1] !== (' '))
                            break;
                        len--;
                    }
                    if (dispValue.length > len)
                        dispValue = NString.Remove(dispValue, len, (dispValue.length - len));
                    if (pic.isNegative() && dispValue.trim().startsWith("-"))
                        dispValue = StrUtil.ltrim(dispValue);
                    let dec = GuiEnvironment.Environment.GetDecimal();
                    let thou = GuiEnvironment.Environment.GetThousands();
                    GuiEnvironment.Environment.setDecimalSeparator('.');
                    GuiEnvironment.Environment.setThousandsSeparator(',');
                    let newPic = pic.getUnformattedNumericPic();
                    str = this.toNum(dispValue, newPic, compIdx);
                    GuiEnvironment.Environment.setDecimalSeparator(dec);
                    GuiEnvironment.Environment.setThousandsSeparator(thou);
                }
                break;
            case StorageAttribute.DATE:
                {
                    let dt;
                    if (typeof dispValue === 'string')
                        dt = new Date(dispValue);
                    else
                        dt = dispValue;
                    str = this.fromNativeDateToMgDate(dt, pic);
                }
                break;
            case StorageAttribute.TIME:
                {
                    let dt;
                    if (typeof dispValue === 'string') {
                        dt = new Date();
                        if (dispValue !== null) {
                            dt.setHours(+String(dispValue).substr(0, 2));
                            dt.setMinutes(+String(dispValue).substr(0, 2));
                            if (String(dispValue).length > 5)
                                dt.setSeconds(+String(dispValue).substr(0, 2));
                        }
                    }
                    else
                        dt = dispValue;
                    str = this.fromNativeTimeToMgTime(dt);
                }
                break;
            case StorageAttribute.BOOLEAN:
                str = this.fromNativeLogicalToMgLogical(dispValue);
                break;
            case StorageAttribute.BLOB:
                str = BlobType.createFromString(dispValue, blobContentType);
                break;
        }
        return str;
    }
    fromNum(mgValue, pic) {
        let mgNum = new NUM_TYPE(mgValue);
        return mgNum.toDisplayValue(pic);
    }
    fromAlpha(dataStr, pic, rangeStr, alwaysFill, convertCase) {
        let len = pic.getMaskSize();
        let min_len = len;
        let maskStr = pic.getMask();
        let resStr;
        if (pic.getMaskChars() > 0) {
            resStr = this.win_data_cpy(NString.ToCharArray(maskStr), len, NString.ToCharArray(dataStr), pic.isAttrAlpha(), convertCase);
        }
        else {
            if (UtilStrByteMode.isLocaleDefLangDBCS() && pic.isAttrAlpha()) {
                let intDataStrLenB = UtilStrByteMode.lenB(dataStr);
                if (min_len > intDataStrLenB) {
                    min_len = intDataStrLenB;
                }
                resStr = UtilStrByteMode.leftB(dataStr, min_len);
            }
            else if (dataStr !== null) {
                if (min_len > dataStr.length) {
                    min_len = dataStr.length;
                }
                if (convertCase) {
                    resStr = this.win_data_cpy(NString.ToCharArray(maskStr), len, NString.ToCharArray(dataStr), pic.isAttrAlpha(), convertCase);
                }
                else {
                    resStr = dataStr.substr(0, min_len);
                }
            }
        }
        if (alwaysFill || (rangeStr !== null && rangeStr.length > 0 && resStr.length < len)) {
            let vd = new ValidationDetailsBase(rangeStr);
            let continuousRangeValues = vd.getContinuousRangeValues();
            let discreteRangeValues = vd.getDiscreteRangeValues();
            if (continuousRangeValues !== null || discreteRangeValues !== null) {
                resStr = this.fillAlphaByRange(continuousRangeValues, discreteRangeValues, resStr);
            }
        }
        return resStr;
    }
    caseConvertedChar(data, mask) {
        let ch;
        if (mask.charCodeAt(0) === PICInterface.PIC_U) {
            ch = data.toUpperCase();
        }
        else {
            if (mask.charCodeAt(0) === PICInterface.PIC_L) {
                ch = data.toLowerCase();
            }
            else {
                ch = data;
            }
        }
        return ch;
    }
    win_data_cpy(maskStr, maskLen, dataStr, isAttrAlpha, convertCase) {
        if (arguments.length === 4)
            return this.win_data_cpy_0(maskStr, maskLen, dataStr, isAttrAlpha);
        return this.win_data_cpy_1(maskStr, maskLen, dataStr, isAttrAlpha, convertCase);
    }
    win_data_cpy_0(maskStr, maskLen, dataStr, isAttrAlpha) {
        return this.win_data_cpy(maskStr, maskLen, dataStr, isAttrAlpha, false);
    }
    win_data_cpy_1(maskStr, maskLen, dataStr, isAttrAlpha, convertCase) {
        let maskIdx = 0;
        let dataIdx = 0;
        let resStr = new Array(maskLen);
        if (isAttrAlpha && UtilStrByteMode.isLocaleDefLangDBCS()) {
            let num2 = 0;
            while (maskIdx < maskLen && num2 < maskLen) {
                if (maskStr[maskIdx].charCodeAt(0) <= PICInterface.PIC_MAX_OP) {
                    if (dataIdx < dataStr.length) {
                        let strVal = NString.FromChars(dataStr, dataIdx, 1);
                        if (UtilStrByteMode.lenB(strVal) === 2) {
                            if (maskIdx + 1 < maskLen) {
                                if (maskStr[maskIdx + 1].charCodeAt(0) <= PICInterface.PIC_MAX_OP) {
                                    if (convertCase) {
                                        resStr[num2++] = this.caseConvertedChar(dataStr[dataIdx++], maskStr[maskIdx + 1]);
                                    }
                                    else {
                                        resStr[num2++] = dataStr[dataIdx++];
                                    }
                                    maskIdx = maskIdx + 1;
                                }
                                else {
                                    resStr[num2++] = ' ';
                                }
                            }
                            else {
                                resStr[num2++] = ' ';
                            }
                        }
                        else {
                            if (convertCase) {
                                resStr[num2++] = this.caseConvertedChar(dataStr[dataIdx++], maskStr[maskIdx + 1]);
                            }
                            else {
                                resStr[num2++] = dataStr[dataIdx++];
                            }
                        }
                    }
                    else {
                        resStr[num2++] = ' ';
                    }
                }
                else {
                    resStr[num2++] = maskStr[maskIdx];
                }
                maskIdx = maskIdx + 1;
            }
            return NString.FromChars(resStr).substr(0, num2);
        }
        while (maskIdx < maskLen) {
            if (maskStr[maskIdx].charCodeAt(0) <= PICInterface.PIC_MAX_OP) {
                if (dataIdx < dataStr.length) {
                    resStr[maskIdx] = dataStr[dataIdx];
                    if (convertCase) {
                        resStr[maskIdx] = this.caseConvertedChar(dataStr[dataIdx], maskStr[maskIdx]);
                    }
                    else {
                        resStr[maskIdx] = dataStr[dataIdx];
                    }
                    dataIdx = dataIdx + 1;
                }
                else {
                    resStr[maskIdx] = ' ';
                }
            }
            else {
                resStr[maskIdx] = maskStr[maskIdx];
            }
            maskIdx = maskIdx + 1;
        }
        return NString.FromChars(resStr);
    }
    fromDate(dataStr, pic, compIdx, time_date_pic_Z_edt) {
        let mask = pic.getMask();
        let maskSize = pic.getMaskSize();
        let date = new NUM_TYPE(dataStr).NUM_2_ULONG();
        return this.to_a_pic_datemode(mask, maskSize, date, pic, time_date_pic_Z_edt, false, compIdx);
    }
    fromTime(dataStr, pic, time_date_pic_Z_edt) {
        let mask = pic.getMask();
        let maskSize = pic.getMaskSize();
        let time = new NUM_TYPE(dataStr).NUM_2_ULONG();
        return this.time_2_a_pic(mask, maskSize, time, pic, time_date_pic_Z_edt, false);
    }
    fromMgDateToNativeDate(dataStr, pic, compIdx) {
        let date = new NUM_TYPE(dataStr).NUM_2_ULONG();
        let breakParams;
        breakParams = new DateBreakParams();
        this.date_break_datemode(breakParams, date, false, compIdx);
        if (breakParams.month > 0)
            breakParams.month = breakParams.month - 1;
        if (breakParams.year == 0 && breakParams.month == 0 && breakParams.day == 0)
            return null;
        let maskStr = pic.getMask();
        let outVal = NString.ToCharArray(maskStr);
        let len = pic.getMaskSize();
        let year = breakParams.year;
        let outIdx = 0;
        while (len > 0) {
            switch (outVal[outIdx].charCodeAt(0)) {
                case PICInterface.PIC_YY:
                    {
                        this.int_2_a(outVal, outIdx, 2, breakParams.year, '0');
                        let yearStr = NString.FromChars(outVal, outIdx, 2);
                        year = +yearStr;
                        outIdx += 2;
                        len -= 2;
                    }
                    break;
                case PICInterface.PIC_YYYY:
                    this.int_2_a(outVal, outIdx, 4, breakParams.year, '0');
                    let yearStr = NString.FromChars(outVal, outIdx, 4);
                    year = +yearStr;
                    outIdx += 4;
                    len -= 4;
                    break;
                default:
                    outIdx++;
                    len--;
                    break;
            }
        }
        let nativeDt = new Date(year, breakParams.month, breakParams.day);
        return nativeDt;
    }
    fromNativeDateToMgDateNumber(nativeDate, pic) {
        let numVal;
        let dateParams;
        let millenium = 0;
        let year = nativeDate.getFullYear();
        let day = nativeDate.getDate();
        let month = nativeDate.getMonth() + 1;
        if (pic.isHebrew()) {
            if (year > 0) {
                if (millenium === 0)
                    millenium = 5;
                year += millenium * 1000;
            }
            numVal = (HebrewDate.dateheb_2_d(year, month, day));
        }
        else {
            if (year === 0) {
                dateParams = this.date_sys();
                year = dateParams.year;
            }
            numVal = this.date_4_calender(year, month, day, 0, false);
        }
        return numVal;
    }
    fromNativeDateToMgDate(nativeDate, pic) {
        let mgNum = new NUM_TYPE();
        let numVal = nativeDate == null ? 0 : this.fromNativeDateToMgDateNumber(nativeDate, pic);
        mgNum.NUM_4_LONG(numVal);
        return mgNum.toXMLrecord();
    }
    to_a_pic_datemode(outStr, out_len, date, pic, date_pic_Z_edt, ignore_dt_fmt, compIdx) {
        let len;
        let day = 0;
        let doy = 0;
        let dow = 0;
        let year = 0;
        let month = 0;
        let hday = 0;
        let hyear = 0;
        let hmonth = 0;
        let leap = false;
        let i;
        let p;
        let outVal = NString.ToCharArray(outStr);
        let breakParams;
        let dateNameParams;
        let outIdx = 0;
        let strOut = null;
        if (pic.getMaskSize() > out_len) {
            for (; i < out_len; i++)
                outVal[i] = '*';
            return NString.FromChars(outVal);
        }
        out_len = len = pic.getMaskSize();
        if (pic.zeroFill() && date === 0 && !date_pic_Z_edt) {
            for (; i < out_len; i++)
                outVal[i] = pic.getZeroPad();
            return NString.FromChars(outVal);
        }
        if (pic.isHebrew() && (date === 0 || date === 1000000000)) {
            for (; i < out_len; i++)
                outVal[i] = ' ';
            return NString.FromChars(outVal);
        }
        breakParams = new DateBreakParams(year, month, day, doy, dow);
        this.date_break_datemode(breakParams, date, ignore_dt_fmt, compIdx);
        if (pic.isHebrew()) {
            let refhday = new RefParam(hday);
            let refhmonth = new RefParam(hmonth);
            let refhyear = new RefParam(hyear);
            leap = HebrewDate.dateheb_4_d(date, refhday, refhmonth, refhyear);
            hday = refhday.value;
            hmonth = refhmonth.value;
            hyear = refhyear.value;
        }
        year = breakParams.year;
        month = breakParams.month;
        day = breakParams.day;
        doy = breakParams.doy;
        dow = breakParams.dow;
        breakParams = null;
        while (len > 0) {
            let refOutLen = new RefParam(out_len);
            switch (outVal[outIdx].charCodeAt(0)) {
                case PICInterface.PIC_YY:
                    if (pic.zeroFill() && date === 0 && date_pic_Z_edt)
                        MemoryUtil.char_memset(outVal, outIdx, pic.getZeroPad(), 2);
                    else
                        this.int_2_a(outVal, outIdx, 2, year, '0');
                    outIdx += 2;
                    len -= 2;
                    break;
                case PICInterface.PIC_HYYYYY:
                    i = HebrewDate.dateheb_2_str(outVal, outIdx, len, refOutLen, 5, hyear, pic.isTrimed(), true);
                    out_len = refOutLen.value;
                    outIdx += i;
                    len -= i;
                    break;
                case PICInterface.PIC_HL:
                    i = HebrewDate.dateheb_i_2_h((hyear / 1000) % 10, outVal, outIdx, false, false);
                    outIdx++;
                    len--;
                    break;
                case PICInterface.PIC_HDD:
                    i = HebrewDate.dateheb_2_str(outVal, outIdx, len, refOutLen, 2, hday, pic.isTrimed(), false);
                    out_len = refOutLen.value;
                    outIdx += i;
                    len -= i;
                    break;
                case PICInterface.PIC_YYYY:
                    if (pic.zeroFill() && date === 0 && date_pic_Z_edt)
                        MemoryUtil.char_memset(outVal, outIdx, pic.getZeroPad(), 4);
                    else
                        this.int_2_a(outVal, outIdx, 4, year, '0');
                    outIdx += 4;
                    len -= 4;
                    break;
                case PICInterface.PIC_MMD:
                    if (pic.zeroFill() && date === 0 && date_pic_Z_edt)
                        MemoryUtil.char_memset(outVal, outIdx, pic.getZeroPad(), 2);
                    else
                        this.int_2_a(outVal, outIdx, 2, month, '0');
                    outIdx += 2;
                    len -= 2;
                    break;
                case PICInterface.PIC_MMM:
                    if (pic.isHebrew()) {
                        if (leap && hmonth === 6)
                            hmonth = 14;
                        let monthNames = HebrewDate.GetLocalMonths();
                        p = NString.ToCharArray(monthNames[hmonth]);
                    }
                    else {
                        let monthStr = Events.GetMessageString(MsgInterface.MONTHS_PTR);
                        let monthNames = DateUtil.getLocalMonths(monthStr);
                        p = NString.ToCharArray(monthNames[month]);
                    }
                    dateNameParams = new DateNameParams(outVal, outIdx, len);
                    out_len -= this.date_i_2_nm(dateNameParams, p, pic.isTrimed());
                    outIdx = dateNameParams.outIdx;
                    len = dateNameParams.len;
                    dateNameParams = null;
                    break;
                case PICInterface.PIC_DD:
                    let isZero = pic.zeroFill();
                    if (isZero && date === 0 && date_pic_Z_edt)
                        MemoryUtil.char_memset(outVal, outIdx, pic.getZeroPad(), 2);
                    else
                        this.int_2_a(outVal, outIdx, 2, day, '0');
                    outIdx += 2;
                    len -= 2;
                    break;
                case PICInterface.PIC_DDD:
                    if (pic.zeroFill() && date === 0 && date_pic_Z_edt)
                        MemoryUtil.char_memset(outVal, outIdx, pic.getZeroPad(), 3);
                    else
                        this.int_2_a(outVal, outIdx, 3, doy, '0');
                    len -= 3;
                    outIdx += 3;
                    break;
                case PICInterface.PIC_DDDD:
                    i = this.date_2_DDDD(day, outVal, outIdx, len, pic.isTrimed());
                    outIdx += i;
                    out_len += i - 4;
                    len -= 4;
                    break;
                case PICInterface.PIC_W:
                    if (pic.zeroFill() && date === 0 && date_pic_Z_edt)
                        MemoryUtil.char_memset(outVal, outIdx, pic.getZeroPad(), 1);
                    else
                        this.int_2_a(outVal, outIdx, 1, dow, '0');
                    len--;
                    outIdx++;
                    break;
                case PICInterface.PIC_WWW:
                    if (pic.isHebrew()) {
                        let dowsNames = HebrewDate.GetLocalDows();
                        p = NString.ToCharArray(dowsNames[dow]);
                    }
                    else {
                        let dayStr = Events.GetMessageString(MsgInterface.DAYS_PTR);
                        let dayNames = DateUtil.getLocalDays(dayStr);
                        p = NString.ToCharArray(dayNames[dow]);
                    }
                    dateNameParams = new DateNameParams(outVal, outIdx, len);
                    out_len -= this.date_i_2_nm(dateNameParams, p, pic.isTrimed());
                    outIdx = dateNameParams.outIdx;
                    len = dateNameParams.len;
                    dateNameParams = null;
                    break;
                case PICInterface.PIC_BB:
                    p = NString.ToCharArray(UtilDateJpn.getStrDow(dow));
                    strOut = null;
                    dateNameParams = new DateNameParams(outVal, outIdx, len);
                    strOut = this.date_i_2_nm_bytemode(strOut, dateNameParams, NString.FromChars(p), out_len);
                    outVal = new Array(strOut.length);
                    NString.CopyTo(strOut, 0, outVal, 0, strOut.length);
                    outIdx = dateNameParams.outIdx;
                    len = dateNameParams.len;
                    strOut = null;
                    dateNameParams = null;
                    break;
                case PICInterface.PIC_JY1:
                case PICInterface.PIC_JY2:
                case PICInterface.PIC_JY4:
                    let strNum = null;
                    if (outVal[outIdx].charCodeAt(0) === PICInterface.PIC_JY1)
                        strNum = UtilDateJpn.getInstance().date_jpn_yr_2_a(year, doy, false);
                    else
                        strNum = UtilDateJpn.getInstance().date_jpn_yr_2_a(year, doy, true);
                    strOut = null;
                    dateNameParams = new DateNameParams(outVal, outIdx, len);
                    strOut = this.date_i_2_nm_bytemode(strOut, dateNameParams, strNum, out_len);
                    outVal = null;
                    outVal = new Array(strOut.length);
                    NString.CopyTo(strOut, 0, outVal, 0, strOut.length);
                    outIdx = dateNameParams.outIdx;
                    len = dateNameParams.len;
                    strOut = null;
                    dateNameParams = null;
                    break;
                case PICInterface.PIC_YJ:
                    if (pic.zeroFill() && date === 0 && date_pic_Z_edt)
                        MemoryUtil.char_memset(outVal, outIdx, pic.getZeroPad(), 2);
                    else
                        this.int_2_a(outVal, outIdx, 2, UtilDateJpn.getInstance().date_jpn_year_ofs(year, doy), '0');
                    outIdx += 2;
                    len -= 2;
                    break;
                default:
                    outIdx++;
                    len--;
                    break;
            }
        }
        if (out_len < outVal.length)
            MemoryUtil.char_memset(outVal, out_len, ' ', outVal.length - out_len);
        return NString.FromChars(outVal);
    }
    date_break_datemode(breakParams, date, ignore_dt_fmt, compIdx) {
        let year = breakParams.year;
        let month = breakParams.month;
        let day = breakParams.day;
        let doy = breakParams.doy;
        let dow = breakParams.dow;
        if (date <= 0) {
            breakParams.day = 0;
            breakParams.doy = 0;
            breakParams.dow = 0;
            breakParams.year = 0;
            breakParams.month = 0;
        }
        else {
            dow = date % 7 + 1;
            date = date - 1;
            let cent4 = Math.floor(date / PICInterface.DAYSINFOURCENT);
            date = date - cent4 * PICInterface.DAYSINFOURCENT;
            let cent = Math.floor(date / PICInterface.DAYSINCENTURY);
            if (cent > 3) {
                cent = 3;
            }
            date = date - cent * PICInterface.DAYSINCENTURY;
            let year4 = Math.floor(date / PICInterface.DAYSINFOURYEAR);
            date = date - year4 * PICInterface.DAYSINFOURYEAR;
            year = Math.floor(date / PICInterface.DAYSINYEAR);
            if (year > 3) {
                year = 3;
            }
            date = date - year * PICInterface.DAYSINYEAR;
            year = cent4 * 400 + cent * 100 + year4 * 4 + year + 1;
            let leapyear = 0;
            if (year % 4 === 0) {
                if (year % 100 !== 0 || year % 400 === 0) {
                    leapyear = 1;
                }
            }
            if (!ignore_dt_fmt && this._environment.GetDateMode(compIdx) === 'B') {
                year = year + PICInterface.DATE_BUDDHIST_GAP;
            }
            month = (Math.floor(date / PICInterface.DAYSINMONTH) + 1);
            day = PICInterface.date_day_tab[month];
            if (leapyear > 0) {
                if (month > 1) {
                    day = day + 1;
                }
            }
            if (date >= day) {
                month = month + 1;
            }
            else {
                day = PICInterface.date_day_tab[month - 1];
                if (leapyear > 0) {
                    if (month > 2) {
                        day = day + 1;
                    }
                }
            }
            day = date - day + 1;
            doy = date + 1;
            breakParams.year = year;
            breakParams.month = month;
            breakParams.day = day;
            breakParams.doy = doy;
            breakParams.dow = dow;
        }
    }
    int_2_a(str, strPos, len, n, lead) {
        let neg = n < 0;
        if (len <= 0)
            return;
        n = Math.abs(n);
        let pos = len;
        do {
            str[strPos + (pos = pos - 1)] = String.fromCharCode(48 + n % 10);
            n = Math.floor(n / 10);
        } while (pos > 0 && n > 0);
        if (neg && pos > 0) {
            str[strPos + (pos = pos - 1)] = '-';
        }
        return this.lib_a_fill(str, strPos, len, pos, lead.charCodeAt(0));
    }
    lib_a_fill(str, strPos, len, pos, lead) {
        if (lead === 0) {
            len = len - pos;
            if (len > 0 && pos > 0) {
                MemoryUtil.char_memcpy(str, strPos, str, pos, len);
                MemoryUtil.char_memset(str, strPos + len, ' ', pos);
            }
        }
        else {
            if (pos > 0) {
                MemoryUtil.char_memset(str, strPos, String.fromCharCode(lead), pos);
            }
        }
        return len;
    }
    date_i_2_nm(dateNameParams, nm, trim) {
        let n = this.date_msk_cnt(dateNameParams.outVal, dateNameParams.outIdx, dateNameParams.len);
        let l;
        if (trim) {
            l = MemoryUtil.mem_trim(nm, 0, n);
        }
        else {
            l = n;
        }
        l = Math.min(l, dateNameParams.len);
        MemoryUtil.char_memcpy(dateNameParams.outVal, dateNameParams.outIdx, nm, 0, l);
        dateNameParams.outIdx = dateNameParams.outIdx + l;
        dateNameParams.len = dateNameParams.len - n;
        n = n - l;
        if (n > 0) {
            MemoryUtil.char_memcpy(dateNameParams.outVal, dateNameParams.outIdx, dateNameParams.outVal, dateNameParams.outIdx + n, dateNameParams.len);
        }
        return n;
    }
    date_i_2_nm_bytemode(strOut, dateNameParams, strNm, intOutLen) {
        let intMskCnt = this.date_msk_cnt(dateNameParams.outVal, dateNameParams.outIdx, dateNameParams.len);
        let strWork = NString.FromChars(dateNameParams.outVal);
        strWork = strWork.substr(0, dateNameParams.outIdx);
        let intOfs = UtilStrByteMode.lenB(strWork);
        strNm = UtilStrByteMode.leftB(strNm, intMskCnt);
        strOut = UtilStrByteMode.repB(NString.FromChars(dateNameParams.outVal), strNm, intOfs + 1, intMskCnt);
        dateNameParams.outIdx = dateNameParams.outIdx + strNm.length;
        dateNameParams.len = dateNameParams.len - intMskCnt;
        intOutLen = intOutLen - intMskCnt;
        return strOut;
    }
    date_msk_cnt(msk, mskPos, len) {
        let c = msk[mskPos++];
        let i = 1;
        while (i < len) {
            if (msk[mskPos] !== c) {
                break;
            }
            mskPos = mskPos + 1;
            i = i + 1;
        }
        return i;
    }
    date_2_DDDD(day, outVal, outIdx, len, trim) {
        let ret = 4;
        let ext;
        if (day > 9) {
            this.int_2_a(outVal, outIdx, 2, day, 0..toString());
            outIdx += 2;
        }
        else {
            if (trim) {
                MemoryUtil.char_memcpy(outVal, outIdx, outVal, outIdx + 1, len - 1);
                ret--;
            }
            else {
                outVal[outIdx++] = ' ';
            }
            this.int_2_a(outVal, outIdx, 1, day, 0..toString());
            outIdx++;
        }
        switch (day) {
            case 1:
            case 21:
            case 31:
                ext = NString.ToCharArray("st");
                break;
            case 2:
            case 22:
                ext = NString.ToCharArray("nd");
                break;
            case 3:
            case 23:
                ext = NString.ToCharArray("rd");
                break;
            default:
                ext = NString.ToCharArray("th");
                break;
        }
        outVal[outIdx++] = ext[0];
        outVal[outIdx] = ext[1];
        return (ret);
    }
    fromMgTimeToNativeTime(dataStr, pic) {
        let time = new NUM_TYPE(dataStr).NUM_2_ULONG();
        let breakParams = new TimeBreakParams();
        DisplayConvertor.time_break(breakParams, time);
        let nativeDt = new Date(0);
        let maskStr = pic.getMask();
        let outVal = NString.ToCharArray(maskStr);
        let len = pic.getMaskSize();
        let outIdx = 0;
        while (len > 0) {
            switch (outVal[outIdx].charCodeAt(0)) {
                case (PICInterface.PIC_HH):
                    nativeDt.setHours(breakParams.hour);
                    outIdx += 2;
                    len -= 2;
                    break;
                case (PICInterface.PIC_MMT):
                    nativeDt.setMinutes(breakParams.minute);
                    outIdx += 2;
                    len -= 2;
                    break;
                case (PICInterface.PIC_SS):
                    nativeDt.setSeconds(breakParams.second);
                    outIdx += 2;
                    len -= 2;
                    break;
                default:
                    outIdx++;
                    len--;
            }
        }
        return nativeDt;
    }
    time_2_a_pic(AsciiStr, AsciiL, time, pic, time_pic_Z_edt, milliSeconds) {
        let hour = 0;
        let minute = 0;
        let second = 0;
        let millisec = 0;
        let Ascii = NString.ToCharArray(AsciiStr);
        let breakParams = new TimeBreakParams(hour, minute, second);
        let timeSec;
        if (milliSeconds) {
            timeSec = Math.floor(time / 1000);
            millisec = time - timeSec * 1000;
        }
        else {
            timeSec = time;
        }
        DisplayConvertor.time_break(breakParams, timeSec);
        hour = breakParams.hour;
        minute = breakParams.minute;
        second = breakParams.second;
        let result;
        if (pic.getMaskSize() > AsciiL) {
            MemoryUtil.char_memset(Ascii, 0, '*', AsciiL);
            result = NString.FromChars(Ascii);
        }
        else {
            if (pic.zeroFill()) {
                if (timeSec === 0 && (!milliSeconds || millisec === 0)) {
                    if (time_pic_Z_edt) {
                        for (let intx = 0; intx < pic.getMaskSize(); intx = intx + 1) {
                            if (Ascii[intx] <= '\u001f') {
                                Ascii[intx] = pic.getZeroPad();
                            }
                        }
                        result = NString.FromChars(Ascii);
                        return result;
                    }
                    MemoryUtil.char_memset(Ascii, 0, pic.getZeroPad(), pic.getMaskSize());
                    result = NString.FromChars(Ascii);
                    return result;
                }
            }
            let I = 0;
            let am = 0;
            let len = pic.getMaskSize();
            let HourI = -1;
            while (I < len) {
                switch (Ascii[I].charCodeAt(0)) {
                    case (PICInterface.PIC_HH):
                        HourI = I;
                        I += 2;
                        break;
                    case (PICInterface.PIC_MMT):
                        this.int_2_a(Ascii, I, 2, minute, '0');
                        I += 2;
                        break;
                    case (PICInterface.PIC_SS):
                        this.int_2_a(Ascii, I, 2, second, '0');
                        I += 2;
                        break;
                    case (PICInterface.PIC_MS):
                        this.int_2_a(Ascii, I, 3, millisec, '0');
                        I += 3;
                        break;
                    case (PICInterface.PIC_PM):
                        am = 1;
                        let tmpHour = hour % 24;
                        if (tmpHour < 12 || tmpHour === 0)
                            MemoryUtil.char_memcpy(Ascii, I, NString.ToCharArray("am"), 0, 2);
                        else
                            MemoryUtil.char_memcpy(Ascii, I, NString.ToCharArray("pm"), 0, 2);
                        I += 2;
                        break;
                    default:
                        I++;
                        break;
                }
            }
            if (HourI >= 0) {
                if (am > 0) {
                    hour = hour % 24;
                    if (hour === 0) {
                        hour = 12;
                    }
                    else {
                        hour = hour % 12;
                        if (hour === 0) {
                            hour = 12;
                        }
                    }
                }
                this.int_2_a(Ascii, HourI, 2, hour, (am !== 0 && hour < 10) ? ' ' : '0');
            }
            result = NString.FromChars(Ascii);
        }
        return result;
    }
    static time_break(breakParams, time) {
        if (time <= 0) {
            breakParams.second = 0;
            breakParams.minute = 0;
            breakParams.hour = 0;
        }
        else {
            breakParams.hour = Math.floor(time / 3600);
            time = time - breakParams.hour * 3600;
            breakParams.minute = Math.floor(time / 60);
            breakParams.second = time - breakParams.minute * 60;
        }
    }
    fromMgLogicalToNativeLogical(dataStr) {
        let nativeVal;
        nativeVal = (dataStr.charAt(0) === '1') ? true : false;
        return nativeVal;
    }
    fromNativeLogicalToMgLogical(dispVal) {
        let mgVal;
        mgVal = (dispVal === true) ? "1" : "0";
        return mgVal;
    }
    win_rng_bool(rng) {
        if (rng === null || rng.length === 0) {
            rng = "True, False";
        }
        return rng;
    }
    toNum(dispValue, pic, compIdx) {
        let mgNum = new NUM_TYPE(dispValue, pic, compIdx);
        return mgNum.toXMLrecord();
    }
    a_2_date_pic_datemode(str, str_len, pic, mask, compIdx) {
        let pos = 0;
        let year_in_pic = false;
        let month_in_pic = false;
        let day_in_pic = false;
        let inp = false;
        let bYearIsZero = false;
        let era_year = 0;
        let intRef = new IntRef(0);
        let intPicIdxOfs = 0;
        let millenium = 0;
        let dateParams;
        let len = Math.min(str.length, pic.getMaskSize());
        if (str.length < mask.length)
            str.length = mask.length;
        let year = 0;
        let month = 1;
        let day = 1;
        let doy = 1;
        let usedoy = false;
        let i;
        for (i = 0; i < str_len; i = i + 1) {
            if (str[i] !== '0' && str[i] !== ' ') {
                break;
            }
        }
        if (i === str_len)
            return 0;
        while (pos < str_len && pos + intPicIdxOfs < len) {
            switch (mask[pos + intPicIdxOfs].charCodeAt(0)) {
                case PICInterface.PIC_YY:
                    year_in_pic = true;
                    inp = true;
                    year = this.date_a_2_i(str, 2, pos);
                    pos += 2;
                    if (year === 0)
                        bYearIsZero = true;
                    let century_year = this._environment.GetCentury(compIdx) % 100;
                    if (year < century_year)
                        year += 100;
                    year += this._environment.GetCentury(compIdx) - century_year;
                    if (this._environment.GetDateMode(compIdx) === 'B')
                        year -= PICInterface.DATE_BUDDHIST_GAP;
                    break;
                case (PICInterface.PIC_YYYY):
                    year_in_pic = true;
                    inp = true;
                    let YearDigitsNo;
                    if (UtilStrByteMode.isLocaleDefLangJPN()) {
                        for (YearDigitsNo = 0; YearDigitsNo < 4 && pos + YearDigitsNo < str.length && UtilStrByteMode.isDigit(str[pos + YearDigitsNo]); YearDigitsNo++) {
                        }
                        i = YearDigitsNo;
                    }
                    else {
                        for (i = 0; pos + i < str.length && str[pos + i] !== this.DATECHAR.toString() && i < 4; i++) {
                        }
                        for (YearDigitsNo = 0; pos + YearDigitsNo < str.length && UtilStrByteMode.isDigit(str[pos + YearDigitsNo]) &&
                            YearDigitsNo < 4; YearDigitsNo++) {
                        }
                    }
                    if (UtilStrByteMode.isLocaleDefLangJPN()) {
                        if (i < 4 && str.length < mask.length)
                            this.move_date(str, pos, 4 - i, str.length);
                    }
                    else {
                        if (i < 4 && len < mask.length) {
                            this.move_date(str, pos, 4 - i, len);
                            len = len + 4 - i;
                        }
                    }
                    year = this.date_a_2_i(str, 4, pos);
                    pos += 4;
                    if (year === 0)
                        bYearIsZero = true;
                    if (YearDigitsNo <= 2) {
                        century_year = this._environment.GetCentury(compIdx) % 100;
                        if (year < century_year)
                            year += 100;
                        year += this._environment.GetCentury(compIdx) - century_year;
                    }
                    if (this._environment.GetDateMode(compIdx) === 'B')
                        year -= PICInterface.DATE_BUDDHIST_GAP;
                    break;
                case PICInterface.PIC_HYYYYY:
                    inp = true;
                    let quotes = 0;
                    if (len > pos + 5 && mask[pos + 5].charCodeAt(0) === (PICInterface.PIC_HYYYYY))
                        quotes = 1;
                    year = HebrewDate.dateheb_h_2_i(str, 5 + quotes, pos);
                    pos += 5 + quotes;
                    if (year === 0)
                        bYearIsZero = true;
                    break;
                case PICInterface.PIC_HL:
                    intPicIdxOfs = HebrewDate.dateheb_h_2_i(str, 1, pos);
                    pos++;
                    break;
                case PICInterface.PIC_HDD:
                    inp = true;
                    quotes = 0;
                    if (len > pos + 2 && mask[pos + 2].charCodeAt(0) === PICInterface.PIC_HDD)
                        quotes = 1;
                    day = HebrewDate.dateheb_h_2_i(str, 2 + quotes, pos);
                    pos += 2 + quotes;
                    break;
                case PICInterface.PIC_MMD:
                    inp = true;
                    month_in_pic = true;
                    if (UtilStrByteMode.isLocaleDefLangJPN()) {
                        if ((str.length === pos + 1 || str[pos + 1].charCodeAt(0) === this.DATECHAR || !UtilStrByteMode.isDigit(str[pos + 1])) && str.length < mask.length)
                            this.move_date(str, pos, 1, str.length);
                    }
                    else {
                        if (str[pos + 1].charCodeAt(0) === this.DATECHAR && len < mask.length) {
                            this.move_date(str, pos, 1, len);
                            len += 1;
                        }
                    }
                    month = this.date_a_2_i(str, 2, pos);
                    pos += 2;
                    break;
                case PICInterface.PIC_MMM:
                    inp = true;
                    month_in_pic = true;
                    intRef.val = pos;
                    month = this.date_MMM_2_m(str, mask, intRef, len, pic.isHebrew());
                    pos = intRef.val;
                    if (pic.isHebrew() && month === 14)
                        month = 6;
                    break;
                case (PICInterface.PIC_DD):
                    inp = true;
                    day_in_pic = true;
                    if (UtilStrByteMode.isLocaleDefLangJPN()) {
                        if ((str.length === pos + 1 || str[pos + 1].charCodeAt(0) === this.DATECHAR || !UtilStrByteMode.isDigit(str[pos + 1])) && str.length < mask.length)
                            this.move_date(str, pos, 1, str.length);
                    }
                    else {
                        if (str[pos + 1].charCodeAt(0) === this.DATECHAR && len < mask.length) {
                            this.move_date(str, pos, 1, len);
                            len += 1;
                        }
                    }
                    day = this.date_a_2_i(str, 2, pos);
                    pos += 2;
                    break;
                case (PICInterface.PIC_DDD):
                    inp = true;
                    usedoy = true;
                    day_in_pic = true;
                    doy = this.date_a_2_i(str, 3, pos);
                    pos += 3;
                    break;
                case (PICInterface.PIC_DDDD):
                    inp = true;
                    day_in_pic = true;
                    day = this.date_a_2_i(str, 2, pos);
                    pos += 2 + 2;
                    break;
                case (PICInterface.PIC_YJ):
                    year_in_pic = true;
                    inp = true;
                    if (UtilStrByteMode.isLocaleDefLangJPN()) {
                        if ((str.length === pos + 1 || str[pos + 1].charCodeAt(0) === this.DATECHAR || !UtilStrByteMode.isDigit(str[pos + 1])) && str.length < mask.length)
                            this.move_date(str, pos, 1, str.length);
                    }
                    year = this.date_a_2_i(str, 2, pos);
                    pos += 2;
                    let str1 = NString.FromChars(str);
                    era_year = UtilDateJpn.getInstance().getStartYearOfEra(str1, pic.getMask());
                    if (era_year === 0)
                        return 0;
                    year += era_year - 1;
                    if (year === 0)
                        bYearIsZero = true;
                    break;
                default:
                    if (UtilStrByteMode.isLocaleDefLangDBCS()) {
                        if (UtilStrByteMode.isHalfWidth(str[pos]) === false &&
                            UtilStrByteMode.isHalfWidth(mask[pos + intPicIdxOfs]))
                            intPicIdxOfs++;
                    }
                    pos += 1;
                    break;
            }
        }
        if (bYearIsZero) {
            if (!month_in_pic)
                month = 0;
            if (!day_in_pic && month === 0)
                day = 0;
        }
        if (day === 0 && month === 0 && (bYearIsZero || year === 0))
            return (0);
        if (!inp)
            return (0);
        if (pic.isHebrew()) {
            if (year > 0) {
                if (millenium === 0)
                    millenium = 5;
                year += millenium * 1000;
            }
            return (HebrewDate.dateheb_2_d(year, month, day));
        }
        if ((year === 0) && !year_in_pic) {
            dateParams = this.date_sys();
            year = dateParams.year;
        }
        return (this.date_4_calender(year, month, day, doy, usedoy));
    }
    date_a_2_i(s, len, pos) {
        if (arguments.length === 3 && (s === null || s instanceof Array) && (len === null || len.constructor === Number) && (pos === null || pos.constructor === Number)) {
            return this.date_a_2_i_0(s, len, pos);
        }
        return this.date_a_2_i_1(s, len, pos);
    }
    date_a_2_i_0(s, len, pos) {
        let array = new Array(len);
        let mgNum = new NUM_TYPE();
        for (let i = 0; i < len; i = i + 1) {
            array[i] = s[pos + i];
        }
        return mgNum.a_2_long(NString.FromChars(array));
    }
    move_date(str, pos, moveby, len) {
        if (arguments.length === 4 && (str === null || str instanceof Array) && (pos === null || pos.constructor === Number) && (moveby === null || moveby.constructor === Number) && (len === null || len.constructor === Number)) {
            this.move_date_0(str, pos, moveby, len);
            return;
        }
        this.move_date_1(str, pos, moveby, len);
    }
    move_date_0(str, pos, moveby, len) {
        MemoryUtil.char_memmove(str, pos + moveby, str, pos, len - (pos + moveby) + 1);
        MemoryUtil.char_memset(str, pos, '0', moveby);
    }
    date_MMM_2_m(str, mask, pos, len, isHebrew) {
        let l;
        if (UtilStrByteMode.isLocaleDefLangJPN()) {
            let maskPos = UtilStrByteMode.convPos(NString.FromChars(str), NString.FromChars(mask), pos.val, false);
            l = this.date_msk_cnt(mask, maskPos, (len - maskPos));
        }
        else {
            l = this.date_msk_cnt(mask, pos.val, (len - pos.val));
        }
        let strPos = pos.val;
        pos.val = pos.val + l;
        len = MemoryUtil.mem_trim(str, strPos, l);
        if (len === 0) {
            return 0;
        }
        else {
            let strB = NString.FromChars(str, strPos, len);
            let monthNames;
            if (isHebrew) {
                monthNames = HebrewDate.GetLocalMonths();
            }
            else {
                let months = Events.GetMessageString(MsgInterface.MONTHS_PTR);
                monthNames = DateUtil.getLocalMonths(months);
            }
            for (let i = 1; i < monthNames.length; i = i + 1) {
                if (NString.Compare(monthNames[i].substr(0, len), strB, true) === 0)
                    return i;
            }
        }
        return 0;
    }
    date_sys() {
        let date = new DateBreakParams();
        let CurrDate = DateTime.Now;
        date.year = CurrDate.Year;
        date.month = CurrDate.Month;
        date.day = CurrDate.Day;
        return date;
    }
    date_magic(utcDate) {
        let dateTime;
        if (utcDate) {
            dateTime = DateTime.UtcNow;
        }
        else {
            dateTime = DateTime.Now;
        }
        let year = dateTime.Year;
        let month = dateTime.Month;
        let day = dateTime.Day;
        return this.date_4_calender(year, month, day, 1, false);
    }
    time_magic(utcTime) {
        return DateTime.GetTotalSecondsFromMidnight(utcTime, new Date());
    }
    mtime_magic(utcTime) {
        return DateTime.GetTotalMilliSecondsFromMidnight(utcTime, new Date());
    }
    date_4_calender(year, month, day, doy, usedoy) {
        let LeapYear = year % 4 === 0 && (year % 100 !== 0 || year % 400 === 0);
        if (usedoy) {
            if (doy === 0) {
                return 0;
            }
            if (doy > 366 || (!LeapYear && doy > 365)) {
                return (1000000000);
            }
        }
        else {
            if (day === 0 && month === 0 && year === 0) {
                return 0;
            }
            if (day === 0 || month === 0 || year === 0 || month > 12) {
                return 1000000000;
            }
        }
        year = year - 1;
        let num = Math.floor(year / 400);
        year = year - num * 400;
        let num2 = Math.floor(year / 100);
        year = year - num2 * 100;
        let num3 = Math.floor(year / 4);
        year = year - num3 * 4;
        if (!usedoy) {
            let num4 = PICInterface.date_day_tab[month - 1];
            let num5 = PICInterface.date_day_tab[month];
            if (LeapYear) {
                if (month > 1) {
                    num5 = num5 + 1;
                    if (month > 2) {
                        num4 = num4 + 1;
                    }
                }
            }
            if (day > num5 - num4) {
                return 1000000000;
            }
            doy = num4 + day;
        }
        return num * 146097 + num2 * 36524 + num3 * 1461 + year * 365 + doy;
    }
    fromNativeTimeToMgTime(dispValue) {
        let mgNum = new NUM_TYPE();
        let numVal = this.fromNativeTimeToMgTimeNumber(dispValue);
        mgNum.NUM_4_LONG(numVal);
        return mgNum.toXMLrecord();
    }
    fromNativeTimeToMgTimeNumber(dispValue) {
        let numVal = this.time_2_int(dispValue.getHours(), dispValue.getMinutes(), dispValue.getSeconds());
        return numVal;
    }
    a_2_time(Ascii, picOrAsciiL, milliSecondsOrMask, milliSeconds) {
        if (arguments.length === 3)
            return this.a_2_time_0(Ascii, picOrAsciiL, milliSecondsOrMask);
        return this.a_2_time_1(Ascii, picOrAsciiL, milliSecondsOrMask, milliSeconds);
    }
    a_2_time_0(Ascii, pic, milliSeconds) {
        return this.a_2_time(Ascii, Ascii.length, NString.ToCharArray(pic.getMask()), milliSeconds);
    }
    a_2_time_1(Ascii, AsciiL, mask, milliSeconds) {
        let mgNum = new NUM_TYPE();
        let I = 0;
        let len = mask.length;
        if (len > AsciiL) {
            len = AsciiL;
        }
        let ampm = 0;
        let hour = 0;
        let minute = 0;
        let second = 0;
        let millisecond = 0;
        while (I < len) {
            let maskLength;
            let maxMaskLen;
            switch (mask[I].charCodeAt(0)) {
                case (PICInterface.PIC_HH):
                case (PICInterface.PIC_MMT):
                case (PICInterface.PIC_SS):
                case (PICInterface.PIC_PM):
                    maxMaskLen = 2;
                    break;
                case (PICInterface.PIC_MS):
                    maxMaskLen = 3;
                    break;
                default:
                    maxMaskLen = 1;
                    break;
            }
            maskLength = Math.min(maxMaskLen, len - I);
            switch (mask[I].charCodeAt(0)) {
                case (PICInterface.PIC_HH):
                    {
                        hour = mgNum.a_2_long(Ascii.substr(I, maskLength));
                    }
                    break;
                case (PICInterface.PIC_MMT):
                    {
                        minute = mgNum.a_2_long(Ascii.substr(I, maskLength));
                    }
                    break;
                case (PICInterface.PIC_SS):
                    {
                        second = mgNum.a_2_long(Ascii.substr(I, maskLength));
                    }
                    break;
                case (PICInterface.PIC_MS):
                    {
                        millisecond = mgNum.a_2_long(Ascii.substr(I, maskLength));
                    }
                    break;
                case (PICInterface.PIC_PM):
                    ampm = 0;
                    let c0 = Ascii[I];
                    let c1;
                    c0 = c0.toUpperCase();
                    if (I + 1 < len) {
                        c1 = Ascii[I + 1];
                        c1 = c1.toUpperCase();
                        if (c1 === 'M') {
                            if (c0 === 'A')
                                ampm = -1;
                            if (c0 === 'P')
                                ampm = 1;
                        }
                    }
                    break;
                default:
                    break;
            }
            I += maskLength;
        }
        if (second > 59 || minute > 59)
            return (0);
        if (ampm !== 0)
            if (ampm === -1) {
                if (hour === 12)
                    hour = 0;
            }
            else if (hour < 12)
                hour += 12;
        if (milliSeconds)
            return this.time_2_int(hour, minute, second) * 1000 + millisecond;
        else
            return this.time_2_int(hour, minute, second);
    }
    time_2_int(hour, minute, second) {
        return hour * 3600 + minute * 60 + second;
    }
    static toBoolean(boolStr) {
        return boolStr !== null && boolStr === "1";
    }
    a_2_date(str, Format, compIdx) {
        return this.a_2_date_datemode(str, Format, compIdx);
    }
    a_2_date_pic(str, pic, mask, compIdx) {
        if (str === null) {
            str = "";
        }
        return this.a_2_date_pic_datemode(NString.ToCharArray(str), str.length, pic, NString.ToCharArray(mask), compIdx);
    }
    to_a(outVal, out_len, date, Format, compIdx) {
        return this.to_a_datemode(outVal, out_len, date, Format, false, compIdx);
    }
    a_2_date_datemode(str, Format, compIdx) {
        let pIC = new PIC(Format, StorageAttribute.DATE, compIdx);
        if (str === null) {
            str = "";
        }
        return this.a_2_date_pic_datemode(NString.ToCharArray(str), str.length, pIC, NString.ToCharArray(pIC.getMask()), compIdx);
    }
    to_a_datemode(outVal, out_len, date, Format, ignore_dt_fmt, compIdx) {
        let pic = new PIC(Format, StorageAttribute.DATE, compIdx);
        if (outVal === null) {
            outVal = pic.getMask();
        }
        return this.to_a_pic_datemode(outVal, out_len, date, pic, true, ignore_dt_fmt, compIdx);
    }
    date_a_2_i_1(s, len, pos) {
        let mgNum = new NUM_TYPE();
        let str;
        str = s.substr(pos, len);
        return mgNum.a_2_long(str);
    }
    move_date_1(str, pos, moveby, len) {
        str = StrUtil.memmove(str, pos + moveby, str, pos, len - pos);
        str = StrUtil.memset(str, pos, '0', moveby);
        return str;
    }
    time_2_a(Ascii, AsciiL, time, Format, compIdx, milliSeconds) {
        if (StrUtil.rtrim(Format).length === 0) {
            Format = "HH:MM:SS";
        }
        let pic = new PIC(Format, StorageAttribute.TIME, compIdx);
        if (Ascii === null) {
            Ascii = pic.getMask();
        }
        return this.time_2_a_pic(Ascii, AsciiL, time, pic, false, milliSeconds);
    }
    toLogical(dispValue, pic, rangeStr) {
        dispValue = dispValue.trim();
        if (dispValue === null || dispValue.length === 0) {
            dispValue = "False";
        }
        return this.disp_2_logical(dispValue, pic.getMask(), rangeStr);
    }
    disp_2_logical(dispValue, mask, rangeStr) {
        let strippedValue = new StringBuilder(dispValue.length);
        rangeStr = this.win_rng_bool(rangeStr);
        rangeStr = StrUtil.makePrintableTokens(rangeStr, SEQ_2_HTML);
        let i = rangeStr.indexOf(',');
        let trueStr = rangeStr.substr(0, i);
        trueStr = StrUtil.makePrintableTokens(trueStr, HTML_2_STR);
        i = 0;
        for (i = 0; i < dispValue.length && i < mask.length; i++) {
            if (mask.charCodeAt(i) === PICInterface.PIC_X) {
                strippedValue.Append(dispValue.charAt(i));
            }
        }
        let val = strippedValue.ToString();
        if (trueStr.length < val.length) {
            val = val.substr(0, trueStr.length);
        }
        else {
            if (val.length < trueStr.length) {
                trueStr = trueStr.substr(0, val.length);
            }
        }
        return (val.toUpperCase() === trueStr.toUpperCase()) ? "1" : "0";
    }
    toAlpha(dispValue, pic) {
        let picture = pic.getMask();
        let buffer = new StringBuilder(pic.getMaskSize());
        let IsAlphaDBCS = UtilStrByteMode.isLocaleDefLangDBCS() && pic.isAttrAlpha();
        let currPicture;
        let i = currPicture = 0;
        for (currPicture = i = 0; i < dispValue.length && currPicture < picture.length; i++, currPicture++) {
            if (DisplayConvertor.isAlphaPositionalDirective(picture.charAt(currPicture))) {
                buffer.Append(dispValue.charAt(i));
                if (IsAlphaDBCS) {
                    let strDspValOneChar = dispValue.substr(i, 1);
                    if (UtilStrByteMode.lenB(strDspValOneChar) === 2) {
                        currPicture = currPicture + 1;
                    }
                }
            }
        }
        return buffer.ToString();
    }
    static isAlphaPositionalDirective(toCheck) {
        let ascii = toCheck.charCodeAt(0);
        return (ascii === PICInterface.PIC_X || ascii === PICInterface.PIC_U || ascii === PICInterface.PIC_L ||
            ascii === PICInterface.PIC_N || ascii === PICInterface.PIC_J || ascii === PICInterface.PIC_G || ascii === PICInterface.PIC_S || ascii === PICInterface.PIC_T);
    }
    fillAlphaByRange(ContinuousRangeValues, discreteRangeValues, newValue) {
        let tmpBuffer = this.fillAlphaByDiscreteRangeValues(discreteRangeValues, newValue);
        if (tmpBuffer !== null) {
            return tmpBuffer;
        }
        else {
            tmpBuffer = this.fillAlphaByContinuousRangeValues(ContinuousRangeValues, newValue);
            if (tmpBuffer !== null) {
                return tmpBuffer;
            }
            else {
                if (discreteRangeValues !== null) {
                    tmpBuffer = this.completeAlphaByRange(discreteRangeValues, newValue);
                    if (tmpBuffer !== null) {
                        return tmpBuffer;
                    }
                }
            }
        }
        return newValue;
    }
    fillAlphaByDiscreteRangeValues(discreteRangeValues, newValue) {
        if (discreteRangeValues !== null) {
            for (let i = 0; i < discreteRangeValues.length; i = i + 1) {
                let discreteValue = discreteRangeValues.get_Item(i);
                let truncatedValue = discreteValue.substr(0, Math.min(newValue.length, discreteValue.length));
                if (newValue.toUpperCase() === truncatedValue.toUpperCase()) {
                    return discreteValue;
                }
            }
        }
        return null;
    }
    fillAlphaByContinuousRangeValues(ContinuousRangeValues, newValue) {
        newValue = StrUtil.rtrim(newValue);
        if (ContinuousRangeValues !== null) {
            for (let i = 0; i < ContinuousRangeValues.length; i = i + 1) {
                let from = ContinuousRangeValues.get_Item(i);
                let to = ContinuousRangeValues.get_Item(i = i + 1);
                if (NString.CompareOrdinal(newValue, from) >= 0 && NString.CompareOrdinal(newValue, to) <= 0) {
                    return newValue;
                }
            }
        }
        return null;
    }
    completeAlphaByRange(discreteRangeValues, newValue) {
        let maxCoincide = new Array(2);
        for (let _ai = 0; _ai < maxCoincide.length; ++_ai)
            maxCoincide[_ai] = 0;
        let bestItem = new Array(2);
        for (let _ai = 0; _ai < bestItem.length; ++_ai)
            bestItem[_ai] = null;
        let CHECK_CASE = 0;
        let IGNORE_CASE = 1;
        let caseLetters = CHECK_CASE;
        while (caseLetters === CHECK_CASE || caseLetters === IGNORE_CASE) {
            let lowerValue;
            if (caseLetters === CHECK_CASE) {
                lowerValue = newValue;
            }
            else {
                lowerValue = newValue.toLowerCase();
            }
            lowerValue = StrUtil.rtrim(lowerValue);
            for (let i = 0; i < discreteRangeValues.length; i = i + 1) {
                let wrongLetter = false;
                let currCoincide = 0;
                let rangeItem;
                if (caseLetters === CHECK_CASE) {
                    rangeItem = discreteRangeValues.get_Item(i);
                }
                else {
                    rangeItem = discreteRangeValues.get_Item(i).toLowerCase();
                }
                let lowLength = lowerValue.length;
                let rangeLength = rangeItem.length;
                if (lowLength < rangeLength) {
                    for (let j = 0; j < lowLength; j = j + 1) {
                        if (!(lowerValue.charAt(j) === rangeItem.charAt(j))) {
                            wrongLetter = true;
                            break;
                        }
                        currCoincide = currCoincide + 1;
                    }
                }
                else {
                    wrongLetter = true;
                }
                if (currCoincide > maxCoincide[caseLetters] && !wrongLetter) {
                    bestItem[caseLetters] = discreteRangeValues.get_Item(i);
                    maxCoincide[caseLetters] = currCoincide;
                }
            }
            caseLetters = caseLetters + 1;
        }
        if (bestItem[CHECK_CASE] !== null)
            return bestItem[CHECK_CASE];
        else {
            if (bestItem[IGNORE_CASE] !== null)
                return bestItem[IGNORE_CASE];
        }
        return null;
    }
    setDateChar(dateChar) {
        this.DATECHAR = dateChar;
    }
    getDateChar() {
        return this.DATECHAR;
    }
    static StringValueToMgValue(value, storageAttribute, filler, length) {
        switch (storageAttribute) {
            case StorageAttribute.ALPHA:
            case StorageAttribute.UNICODE: {
                let newVal = value;
                newVal = NString.PadRight(newVal, length, filler);
                return newVal;
            }
            case StorageAttribute.NUMERIC:
            case StorageAttribute.DATE:
            case StorageAttribute.TIME:
                return new NUM_TYPE(value);
            default:
                return value;
        }
    }
}
DisplayConvertor._instance = null;
export class TimeBreakParams {
    constructor(hour_, minute_, second_) {
        this.hour = 0;
        this.minute = 0;
        this.second = 0;
        if (arguments.length === 0) {
            this.constructor_0();
            return;
        }
        this.constructor_1(hour_, minute_, second_);
    }
    constructor_0() {
    }
    constructor_1(hour_, minute_, second_) {
        this.hour = hour_;
        this.minute = minute_;
        this.second = second_;
    }
}
class IntRef {
    constructor(val_) {
        this.val = 0;
        this.val = val_;
    }
}
class DateNameParams {
    constructor(out_, outIdx_, len_) {
        this.outVal = null;
        this.len = 0;
        this.outIdx = 0;
        this.outVal = out_;
        this.outIdx = outIdx_;
        this.len = len_;
    }
}
export class DateBreakParams {
    constructor(year_, month_, day_, doy_, dow_) {
        this.day = 0;
        this.dow = 0;
        this.doy = 0;
        this.month = 0;
        this.year = 0;
        if (arguments.length === 0) {
            this.constructor_0();
            return;
        }
        this.constructor_1(year_, month_, day_, doy_, dow_);
    }
    constructor_0() {
    }
    constructor_1(year_, month_, day_, doy_, dow_) {
        this.year = year_;
        this.month = month_;
        this.day = day_;
        this.doy = doy_;
        this.dow = dow_;
    }
}
//# sourceMappingURL=data:application/json;base64,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