import { ApplicationException, Debug, Exception, List, NNumber, NString, RefParam, StringBuilder, isNullOrUndefined } from '@magic-xpa/mscorelib';
import { GuiFieldBase } from '../data/GuiFieldBase';
import { ChoiceUtils, Constants, InternalInterface, MagicProperties, MgControlType, Misc, SEQ_2_STR, StorageAttribute, StorageAttributeCheck, StrUtil, UtilStrByteMode, XMLConstants, XmlParser } from '@magic-xpa/utils';
import { PIC } from './PIC';
import { PropTable } from './PropTable';
import { ValidationDetails } from './ValidationDetails';
import { FieldDef } from '../data/FieldDef';
import { Property } from './Property';
import { Manager } from '../../Manager';
import { CommandType, HtmlProperties } from '../../GuiEnums';
import { Commands } from '../../Commands';
import { NUM_TYPE } from '../data/NUM_TYPE';
import { PropInterface } from './PropInterface';
import { Events } from '../../Events';
import { EMPTY_DCREF } from '../data/DcValues';
import { DisplayConvertor } from './DisplayConvertor';
import { BlobType } from '../data/BlobType';
import { VectorType } from '../data/VectorType';
import { ObjectReference } from '../../util/ObjectReference';
import { GuiConstants } from '../../GuiConstants';
import { GuiControlPropertyAdapter } from './GuiControlPropertyAdapter';
import { FocusManager } from '../../FocusManager';
export class MgControlBase extends GuiControlPropertyAdapter {
    set Form(value) {
        this._form = value;
        this.GuiMgForm = value;
    }
    get Form() {
        return this._form;
    }
    get ControlIsn() {
        return this._controlIsn;
    }
    constructor(type, parentMgForm, parentControl) {
        super();
        this._ditIdx = 0;
        this._linkedControls = null;
        this._field = null;
        this._picStr = null;
        this._choiceDisps = null;
        this._choiceLayerList = null;
        this._choiceLinks = null;
        this._choiceNums = null;
        this._containerDitIdx = -1;
        this._currReadOnly = null;
        this._dataCtrl = false;
        this._dcTableRefs = null;
        this._firstRefreshProperties = true;
        this._form = null;
        this._hasValidItmAndDispVal = false;
        this._controlIsn = -1;
        this._linkedParentDitIdx = -1;
        this._orgChoiceDisps = null;
        this._parentTable = null;
        this._pic = null;
        this._picExpExists = false;
        this._prevIsNulls = null;
        this._prevPicExpResult = null;
        this._prevValues = null;
        this._propTab = null;
        this._range = null;
        this._rangeChanged = false;
        this._valExpId = 0;
        this._vd = null;
        this._dcValId = -2;
        this.parent = 0;
        this.veeIndx = 0;
        this.SourceTableReference = null;
        this.ClipBoardDataExists = false;
        this.DataType = StorageAttribute.NONE;
        this.KeyStrokeOn = false;
        this.ModifiedByUser = false;
        this.Value = null;
        this.IsNull = false;
        this.forceRefresh = false;
        this.RefreshOnVisible = false;
        this.InControl = false;
        this.ValidateControl = false;
        this.PromptHelp = null;
        if (arguments.length === 0) {
            this.constructor_0();
            return;
        }
        this.constructor_1(type, parentMgForm, parentControl);
    }
    constructor_0() {
        this._linkedControls = new List();
        this.DataType = StorageAttribute.NONE;
    }
    constructor_1(type, parentMgForm, parentControl) {
        this.constructor_0();
        this.initReferences(parentMgForm);
        this.Type = type;
        this._containerDitIdx = parentControl;
        this._linkedParentDitIdx = parentControl;
        this._propTab = new PropTable(this);
        this.createArrays(false);
    }
    get UniqueName() {
        return this.Name;
    }
    get TaskTag() {
        return this.Form.getTask().getTaskTag();
    }
    checkIfExistProp(propId) {
        let exist = false;
        if (this._propTab !== null) {
            let propById = this._propTab.getPropById(propId);
            exist = (propById !== null);
        }
        return exist;
    }
    getProp(propId) {
        let prop = null;
        if (this._propTab !== null) {
            prop = this._propTab.getPropById(propId);
            if (prop === null) {
                prop = Property.getDefaultProp(propId, 'C', this);
                if (prop !== null) {
                    this._propTab.addProp(prop);
                }
            }
        }
        return prop;
    }
    getForm() {
        return this.Form;
    }
    getCompIdx() {
        return this.getForm().getTask().getCompIdx();
    }
    IsFirstRefreshOfProps() {
        return this._firstRefreshProperties;
    }
    async EvaluateExpression(expId, resType, length, contentTypeUnicode, resCellType, alwaysEvaluate, wasEvaluated) {
        return await this.getForm().getTask().EvaluateExpression(expId, resType, length, contentTypeUnicode, resCellType, alwaysEvaluate, wasEvaluated);
    }
    GetExpressionStorage(expId) {
        return this.getForm().getTask().GetExpressionStorage(expId);
    }
    GetComputedProperty(propId) {
        let prop = null;
        if (this._propTab !== null)
            prop = this._propTab.getPropById(propId);
        return prop;
    }
    initReferences(mgForm) {
        this.Form = mgForm;
    }
    createArrays(forceCreateArrays) {
        if (this._firstRefreshProperties || forceCreateArrays) {
            if (this.SupportsDataSource()) {
                this._dcTableRefs = new List();
            }
            this._prevValues = new List();
            if (!forceCreateArrays) {
                this._currReadOnly = new List();
            }
            this._choiceDisps = new List();
            this._orgChoiceDisps = new List();
            this._choiceLinks = new List();
            this._choiceNums = new List();
            this._prevIsNulls = new List();
            this._choiceLayerList = new List();
            if (!this.IsRepeatable && !super.isTableControl()) {
                this.updateArrays(1);
            }
        }
    }
    async SetFocus(ctrl, line, activateForm) {
        await FocusManager.SetFocus(ctrl.getForm().getTask(), ctrl, line, activateForm);
    }
    InsertArrays(idx, count) {
        this._prevValues.InsertRange(new Array(count));
        this._prevIsNulls.InsertRange(new Array(count));
        this._currReadOnly.InsertRange(new Array(count));
        this._choiceDisps.InsertRange(new Array(count));
        this._orgChoiceDisps.InsertRange(new Array(count));
        this._choiceLinks.InsertRange(new Array(count));
        this._choiceLayerList.InsertRange(new Array(count));
        this._choiceNums.InsertRange(new Array(count));
        if (this.SupportsDataSource())
            this._dcTableRefs.InsertRange(new Array(count));
        this._propTab.InsertPrevValueArray(count);
    }
    updateArrays(newSize) {
        this._prevValues.SetSize(newSize);
        this._prevIsNulls.SetSize(newSize);
        this._currReadOnly.SetSize(newSize);
        this._choiceDisps.SetSize(newSize);
        this._orgChoiceDisps.SetSize(newSize);
        this._choiceLinks.SetSize(newSize);
        this._choiceLayerList.SetSize(newSize);
        this._choiceNums.SetSize(newSize);
        if (this.SupportsDataSource()) {
            this._dcTableRefs.SetSize(newSize);
        }
        this._propTab.updatePrevValueArray(newSize);
    }
    updatePrevValArrays(newSize) {
        this._propTab.updatePrevValueArray(newSize);
    }
    fillData(mgForm, ditIdx, parser) {
        if (this.Form === null) {
            this.initReferences(mgForm);
        }
        this._ditIdx = ditIdx;
        while (this.initInnerObjects(parser.getNextTag(), parser)) {
        }
        if (this.DataType === StorageAttribute.BLOB && this._picStr === null) {
            this._picStr = "";
        }
        if (this._picStr !== null && typeof this.DataType !== "undefined") {
            this.setPIC(this._picStr);
        }
        else {
            switch (this.Type) {
                case MgControlType.CTRL_TYPE_IMAGE:
                case MgControlType.CTRL_TYPE_BROWSER:
                case MgControlType.CTRL_TYPE_BUTTON:
                    if (this.DataType === StorageAttribute.NONE)
                        this.DataType = StorageAttribute.ALPHA;
                    break;
                case MgControlType.CTRL_TYPE_TABLE:
                case MgControlType.CTRL_TYPE_COLUMN:
                case MgControlType.CTRL_TYPE_SUBFORM:
                case MgControlType.CTRL_TYPE_GROUP:
                case MgControlType.CTRL_TYPE_FRAME_SET:
                case MgControlType.CTRL_TYPE_FRAME_FORM:
                case MgControlType.CTRL_TYPE_CONTAINER:
                case MgControlType.CTRL_TYPE_LINE:
                    break;
                default:
                    Events.WriteExceptionToLog(NString.Format("in Control.fillData(): missing datatype or picture string for control: {0}", this.Name));
                    break;
            }
        }
        if (super.isTableControl()) {
            this.Form.setTableCtrl(this);
        }
        this.createArrays(false);
    }
    initInnerObjects(foundTagName, parser) {
        if (foundTagName === null)
            return false;
        if (foundTagName === XMLConstants.MG_TAG_PROP) {
            if (this._propTab === null)
                this._propTab = new PropTable(this);
            this._propTab.fillData(this, 'C', parser);
        }
        else if (foundTagName === XMLConstants.MG_TAG_CONTROL)
            this.parseAttributes(parser);
        else if (foundTagName === "SourceTable")
            this.ParseSourceTable(parser);
        else if (foundTagName === "/" + XMLConstants.MG_TAG_CONTROL) {
            parser.setCurrIndex2EndOfTag();
            return false;
        }
        else {
            Events.WriteExceptionToLog(NString.Format("There is no such tag in Control. Insert else if to Control.initInnerObjects for {0}", foundTagName));
            return false;
        }
        return true;
    }
    ParseSourceTable(parser) {
        parser.setCurrIndex2EndOfTag();
        let objRefStr = parser.ReadToEndOfCurrentElement();
        let sourceTableReference = ObjectReference.FromXML(objRefStr.trim());
        this.SourceTableReference = sourceTableReference;
        parser.setCurrIndex2EndOfTag();
    }
    parseAttributes(parser) {
        let endContext = parser.getXMLdata().indexOf(">", parser.getCurrIndex());
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            let tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(XMLConstants.MG_TAG_CONTROL) + XMLConstants.MG_TAG_CONTROL.length);
            let tokens = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            for (let i = 0; i < tokens.length; i = i + 2) {
                let attribute = tokens.get_Item(i);
                let valueStr = tokens.get_Item(i + 1);
                this.SetAttribute(attribute, valueStr);
            }
            parser.setCurrIndex(endContext + 1);
        }
        else {
            Events.WriteExceptionToLog("in Control.FillName() out of string bounds");
        }
    }
    SetAttribute(attribute, valueStr) {
        let isTagProcessed = true;
        switch (attribute) {
            case XMLConstants.MG_ATTR_TYPE:
                this.Type = valueStr[0];
                break;
            case XMLConstants.MG_ATTR_DATA_CTRL:
                this._dataCtrl = XmlParser.getBoolean(valueStr);
                break;
            case XMLConstants.MG_ATTR_LINKED_PARENT:
                this.setLinkedParentIdx(XmlParser.getInt(valueStr));
                break;
            case XMLConstants.MG_ATTR_CONTAINER:
                this.setContainer(XmlParser.getInt(valueStr));
                break;
            case XMLConstants.MG_ATTR_CONTROL_ISN:
                this._controlIsn = XmlParser.getInt(valueStr);
                break;
            case XMLConstants.MG_ATTR_IS_MODAL:
                this._isModal = XmlParser.getBoolean(valueStr);
                break;
            case XMLConstants.MG_ATTR_CONTROL_Z_ORDER:
                break;
            case XMLConstants.MG_HOR_ALIGMENT_IS_INHERITED:
                break;
            default:
                isTagProcessed = false;
                break;
        }
        return isTagProcessed;
    }
    setImage() {
        let newVal = this.Value;
        if (newVal === null || this.IsNull) {
            newVal = "";
        }
        if (this.DataType === StorageAttribute.BLOB) {
        }
        else {
            this.setImageWithFileName(newVal);
        }
    }
    setImageWithFileName(fileName) {
        if (fileName === null) {
            fileName = "";
        }
        fileName = Events.TranslateLogicalName(fileName);
        if (!Misc.IsWebUrl(fileName))
            fileName = "./assets/images/" + fileName;
        if (this.IsImageButton()) {
            Commands.addOperationWithLine(CommandType.SET_STYLE, this, 0, MagicProperties.ImageFile, fileName);
        }
        else {
            Commands.addOperationWithLine(CommandType.SET_PROPERTY, this, this.getDisplayLine(false), HtmlProperties.Image, fileName);
            Commands.addValueWithLine(CommandType.SET_VALUE, this, this.getDisplayLine(false), fileName);
        }
    }
    setContainer(containerIdx) {
        this._containerDitIdx = containerIdx;
        let container = null;
        if (this._containerDitIdx !== -1) {
            container = this.Form.getCtrl(this._containerDitIdx);
        }
        if (container !== null && container.Type === MgControlType.CTRL_TYPE_TABLE) {
            this._parentTable = container;
        }
        this.IsRepeatable = (this._parentTable !== null && !super.isColumnControl());
    }
    getName() {
        return this.Name;
    }
    getLinkedControls() {
        return this._linkedControls;
    }
    isRepeatable() {
        return this.IsRepeatable;
    }
    IsModal() {
        return this._isModal;
    }
    getDisplayLine(useLineForItems) {
        let line = 0;
        if (this.IsRepeatable || (useLineForItems && super.isTableControl())) {
            line = this.Form.DisplayLine;
        }
        return line;
    }
    isVisible() {
        let result = this.GetComputedBooleanProperty(PropInterface.PROP_TYPE_VISIBLE, true);
        if (result) {
            result = this.isParentPropValue(PropInterface.PROP_TYPE_VISIBLE);
        }
        return result;
    }
    isEnabled() {
        return this.GetComputedBooleanProperty(PropInterface.PROP_TYPE_ENABLED, true);
    }
    isModifiable() {
        let modifiable = this.GetComputedBooleanProperty(PropInterface.PROP_TYPE_MODIFIABLE, true);
        let taskInQuery = this.getForm().getTask().getMode() === Constants.TASK_MODE_QUERY;
        let modifyInQuery = this.GetComputedBooleanProperty(PropInterface.PROP_TYPE_MODIFY_IN_QUERY, false);
        let result = modifiable && (modifyInQuery || !taskInQuery);
        if (result && this._field != null) {
            result = (this._field.DbModifiable || this.getForm().getTask().getMode() === Constants.TASK_MODE_CREATE);
        }
        return result;
    }
    getDcVals() {
        let dataView = this.getForm().getTask().DataView;
        let dcv = null;
        if (this._dcValId === EMPTY_DCREF) {
            dcv = ((this.DataType === StorageAttribute.BLOB_VECTOR) ? dataView.getEmptyChoiceForVectors() : dataView.getEmptyChoice());
        }
        else {
            if (this._dcValId > -1) {
                dcv = dataView.getDcValues(this._dcValId);
            }
        }
        return dcv;
    }
    getPrevIsNull() {
        let line = this.getDisplayLine(true);
        let currObj = this._prevIsNulls.get_Item(line);
        let curr;
        if (currObj === null) {
            curr = false;
            this._prevIsNulls.set_Item(line, curr);
        }
        else {
            curr = currObj;
        }
        return curr;
    }
    setPrevIsNull(newValue) {
        this._prevIsNulls.set_Item(this.getDisplayLine(true), newValue);
    }
    setPIC(picStr) {
        if (super.isButton()) {
            this._picStr = (super.IsImageButton() ? "" + XMLConstants.FILE_NAME_SIZE.toString() : Events.Translate(picStr));
        }
        else {
            this._picStr = picStr;
        }
        this._pic = new PIC(this._picStr, this.DataType, this.getForm().getTask().getCompIdx());
        if (this._dataCtrl && this._picStr.indexOf('H') > -1) {
            this._pic.setHebrew();
        }
    }
    getPIC() {
        if (this._picExpExists)
            this._pic.setFormatExp(this._picExpExists);
        return this._pic;
    }
    getParent() {
        let retParent = this.Form;
        if (this._containerDitIdx !== -1) {
            retParent = this.Form.getCtrl(this._containerDitIdx);
        }
        else {
            if (this.Form.isSubForm()) {
                retParent = this.Form.getSubFormCtrl();
            }
        }
        return retParent;
    }
    getField() {
        return this._field;
    }
    setField(fieldOrFieldStrID) {
        if (fieldOrFieldStrID === null || fieldOrFieldStrID instanceof GuiFieldBase) {
            this.setField_0(fieldOrFieldStrID);
            return;
        }
        this.setField_1(fieldOrFieldStrID);
    }
    setField_0(field) {
        this._field = field;
        if (field !== null) {
            this._field.SetControl(this);
        }
    }
    setField_1(fieldStrID) {
        let returnField = this.getFieldByValueStr(fieldStrID);
        if (returnField !== null) {
            this._field = returnField;
            this._field.SetControl(this);
            return;
        }
        throw new ApplicationException("in Control.setField(): illegal field identifier: " + fieldStrID);
    }
    setPicStr(format, expId) {
        this._picStr = format;
        if (expId > 0) {
            this._picExpExists = true;
        }
    }
    setRange(newRange) {
        this._range = newRange;
        this._vd = null;
    }
    setValExp(expId) {
        this._valExpId = expId;
    }
    buildCopyPicture(oldVal, newVal) {
        if (super.isTableControl() || super.isColumnControl()) {
            return null;
        }
        else {
            let copyVD = this.getCopyOfVD();
            copyVD.setValue(newVal);
            copyVD.setOldValue(oldVal);
            return copyVD;
        }
    }
    getCopyOfVD() {
        let result;
        if (this._vd === null) {
            result = new ValidationDetails(this.Value, this.Value, this._range, this._pic, this);
        }
        else {
            result = new ValidationDetails(this._vd);
        }
        return result;
    }
    getRanges() {
        return this._range;
    }
    getRangedValue(newVal) {
        let vd;
        vd = this.buildPicture(this.Value, newVal);
        vd.evaluate();
        this.ModifiedByUser = true;
        let val = vd.getDispValue();
        return (val.toString());
    }
    validate(newVal) {
        let vd;
        let modified = this.ModifiedByUser;
        this.ModifiedByUser = true;
        vd = this.buildPicture(this.Value, newVal);
        vd.evaluate();
        this.ModifiedByUser = modified;
        if (vd.ValidationFailed)
            return vd.ErrorMessage;
        else
            return null;
    }
    getFieldByValueStr(valueStr) {
        let refParent = new RefParam(this.parent);
        let refVeeIndx = new RefParam(this.veeIndx);
        let ret = this.getForm().getTask().getFieldByValueStr(valueStr, refParent, refVeeIndx);
        this.parent = refParent.value;
        this.veeIndx = refVeeIndx.value;
        return ret;
    }
    hasContainer() {
        return this._containerDitIdx !== -1;
    }
    getLayer() {
        return this.Layer;
    }
    computePIC(picExpResult) {
        let picResult = (picExpResult !== null && super.isButton()) ? Events.Translate(picExpResult) : picExpResult;
        if (this._prevPicExpResult === null || !(this._prevPicExpResult === picResult)) {
            this._pic = new PIC(picResult, this.DataType, this.getForm().getTask().getCompIdx());
            this._prevPicExpResult = picResult;
        }
        return this._pic;
    }
    getCurrentIndexOfChoice() {
        let selectedIndice = [-1];
        let val = this.Value;
        if (!NString.IsNullOrEmpty(val)) {
            let array2 = val.split(',');
            selectedIndice = new Array(array2.length);
            for (let iCtr = 0; iCtr < array2.length; iCtr = iCtr + 1) {
                let num = NNumber.Parse(array2[iCtr]);
                selectedIndice[iCtr] = ((num >= 0) ? num : -1);
            }
        }
        return selectedIndice;
    }
    IsMultipleSelectionListBox() {
        return super.isListBox();
    }
    IsDateTimePicture() {
        return this._pic.getAttr() === StorageAttribute.DATE || this._pic.getAttr() === StorageAttribute.TIME;
    }
    SupportsDataSource() {
        return super.isSelectionCtrl() || super.isTabControl() || super.isRadio() || this.HasAutocomplete();
    }
    HasAutocomplete() {
        if (super.isTextControl())
            return this.getProp(PropInterface.PROP_TYPE_AUTO_COMPLETE_MODE_WC).GetComputedValueBoolean();
        return false;
    }
    async getIndexOfChoice(mgVal, line, isNull) {
        let splitCommaSeperatedVals = this.IsMultipleSelectionListBox();
        await this.computeChoice(line);
        let isVectorValue = this.getField() !== null && this.getField().getType() === StorageAttribute.BLOB_VECTOR;
        return this.getDcVals().getIndexOf(mgVal, isVectorValue, isNull, this._choiceLinks.get_Item(line), this._choiceNums.get_Item(line), splitCommaSeperatedVals);
    }
    async getDispVals(line, execComputeChoice) {
        if (execComputeChoice) {
            await this.computeChoice(line);
        }
        if (super.isTabControl()) {
            return this._orgChoiceDisps.get_Item(line);
        }
        else {
            return this._choiceDisps.get_Item(line);
        }
    }
    emptyChoice(line) {
        let value = new Array(0);
        this._choiceDisps.set_Item(line, value);
        this._orgChoiceDisps.set_Item(line, value);
        this._choiceLinks.set_Item(line, value);
        this._choiceLayerList.set_Item(line, value);
    }
    async computeChoice(line) {
        let fromHelp = [
            "\\\\", "\\-", "\\,"
        ];
        let toHelp = [
            "XX", "XX", "XX"
        ];
        let task = this.getForm().getTask();
        let dataView = task.DataView;
        let dcv = dataView.getDcValues(this._dcValId);
        let currDcId = this.getDcRef();
        let choiceDispStr = null;
        let choiceLinkStr = "";
        let optionsValid = true;
        let trimToLength = -1;
        let dataType = this.DataType;
        let isItemsListTreatedAsDisplayList = false;
        try {
            if (this._choiceLinks.get_Item(line) === null || (this.isDataCtrl() && currDcId !== this._dcValId)) {
                if (dataType === StorageAttribute.BLOB_VECTOR) {
                    dataType = this.getField().getCellsType();
                }
                let dispVals = dcv.getDispVals();
                let dispProp = this.getProp(PropInterface.PROP_TYPE_DISPLAY_LIST);
                let linkProp = this.getProp(PropInterface.PROP_TYPE_LABEL);
                if (dispProp !== null) {
                    choiceDispStr = await dispProp.getValue();
                }
                if (linkProp !== null) {
                    choiceLinkStr = await linkProp.getValue();
                }
                if (choiceLinkStr === null || StrUtil.rtrim(choiceLinkStr).length === 0) {
                    choiceLinkStr = "";
                }
                if (choiceDispStr === null || StrUtil.rtrim(choiceDispStr).length === 0) {
                    if (!this.isDataCtrl() || dispVals === null || dispVals.length === 0) {
                        choiceDispStr = choiceLinkStr;
                    }
                    else {
                        choiceDispStr = (choiceLinkStr = "");
                    }
                    isItemsListTreatedAsDisplayList = true;
                }
                choiceDispStr = Events.Translate(choiceDispStr);
                if (dataType === StorageAttribute.NUMERIC) {
                    choiceLinkStr = StrUtil.searchAndReplace(choiceLinkStr, "\\-", "-");
                }
                let helpStr = StrUtil.searchAndReplace(choiceLinkStr, fromHelp, toHelp);
                let sTok = StrUtil.tokenize(helpStr, ",");
                let linkSize = (helpStr !== "") ? sTok.length : 0;
                let helpStrDisp = StrUtil.searchAndReplace(choiceDispStr, fromHelp, toHelp);
                sTok = StrUtil.tokenize(helpStrDisp, ",");
                let displaySize = (helpStrDisp !== "") ? sTok.length : 0;
                if (linkSize !== displaySize && displaySize > 0) {
                    choiceDispStr = choiceLinkStr;
                    isItemsListTreatedAsDisplayList = true;
                }
                let size = linkSize;
                let choiceLink = new Array(size);
                this._choiceLinks.set_Item(line, choiceLink);
                if (dataType === StorageAttribute.NUMERIC || dataType === StorageAttribute.DATE || dataType === StorageAttribute.TIME) {
                    this._choiceNums.set_Item(line, new Array(size));
                }
                let orgChoiceDisp = ChoiceUtils.GetDisplayListFromString(choiceDispStr, false, true, !isItemsListTreatedAsDisplayList);
                this._orgChoiceDisps.set_Item(line, orgChoiceDisp);
                let choiceDisp = ChoiceUtils.GetDisplayListFromString(choiceDispStr, super.isSelectionCtrl() || super.isTabControl(), true, !isItemsListTreatedAsDisplayList);
                this._choiceDisps.set_Item(line, choiceDisp);
                if (this.getField() !== null) {
                    trimToLength = this.getField().getSize();
                }
                let token = null;
                let helpToken = null;
                let tokenBuffer;
                for (let i = 0, currPos = 0, nextPos = 0; i < size && optionsValid; i++) {
                    nextPos = currPos;
                    nextPos = helpStr.indexOf(',', nextPos);
                    if (nextPos === currPos)
                        token = helpToken = "";
                    else if (nextPos === -1) {
                        token = choiceLinkStr.substr(currPos);
                        helpToken = helpStr.substr(currPos);
                    }
                    else {
                        token = choiceLinkStr.substr(currPos, (nextPos) - (currPos));
                        helpToken = helpStr.substr(currPos, (nextPos) - (currPos));
                    }
                    currPos = nextPos + 1;
                    switch (dataType) {
                        case StorageAttribute.ALPHA:
                        case StorageAttribute.MEMO:
                        case StorageAttribute.UNICODE:
                        case StorageAttribute.BLOB_VECTOR:
                            token = StrUtil.ltrim(token);
                            helpToken = StrUtil.ltrim(helpToken);
                            if (helpToken.indexOf('\\') >= 0) {
                                tokenBuffer = new StringBuilder();
                                for (let tokenPos = 0; tokenPos < helpToken.length; tokenPos++)
                                    if (helpToken[tokenPos] !== '\\')
                                        tokenBuffer.Append(token[tokenPos]);
                                    else if (tokenPos === helpToken.length - 1)
                                        tokenBuffer.Append(' ');
                                token = tokenBuffer.ToString();
                            }
                            token = StrUtil.makePrintableTokens(token, SEQ_2_STR);
                            if (this.isSelectionCtrl() || this.isTabControl())
                                token = ChoiceUtils.RemoveAcclCharFromOptions(new StringBuilder(token));
                            if (UtilStrByteMode.isLocaleDefLangDBCS() && dataType === StorageAttribute.ALPHA) {
                                if (trimToLength !== -1 && UtilStrByteMode.lenB(token) > trimToLength)
                                    token = UtilStrByteMode.leftB(token, trimToLength);
                            }
                            else {
                                if (trimToLength !== -1 && token.length > trimToLength)
                                    token = token.substr(0, trimToLength);
                            }
                            choiceLink[i] = token;
                            break;
                        case StorageAttribute.NUMERIC:
                        case StorageAttribute.DATE:
                        case StorageAttribute.TIME:
                            let picture = PIC.buildPicture(dataType, token, task.getCompIdx(), false);
                            optionsValid = this.optionIsValid(token);
                            choiceLink[i] = DisplayConvertor.Instance.disp2mg(token.trim(), picture, task.getCompIdx(), BlobType.CONTENT_TYPE_UNKNOWN);
                            ((this._choiceNums.get_Item(line)))[i] = new NUM_TYPE(choiceLink[i]);
                            break;
                        case StorageAttribute.BOOLEAN:
                            choiceLink[i] = (1 - i).toString();
                            break;
                        default:
                            break;
                    }
                }
                if (!optionsValid)
                    this.emptyChoice(line);
            }
        }
        catch (ex) {
            if (ex instanceof Exception) {
                optionsValid = false;
                this.emptyChoice(line);
                Events.WriteExceptionToLog(NString.Format("{0} : {1}", ex.GetType(), ex.Message));
            }
            else
                throw ex;
        }
        this._hasValidItmAndDispVal = optionsValid;
    }
    getTopMostForm() {
        return this.getForm().getTopMostForm();
    }
    setUrl() {
        Commands.addValueWithLine(CommandType.SET_VALUE, this, 0, this.Value);
    }
    async getMgValue(dispVal) {
        let mgVal = null;
        if (super.isCheckBox()) {
            mgVal = dispVal ? "1" : "0";
        }
        else {
            if (super.isSelectionCtrl() || super.isTabControl() || super.isRadio()) {
                let line = this.getDisplayLine(true);
                if (line < 0)
                    line = 0;
                if (typeof dispVal === "number")
                    dispVal = dispVal.toString();
                if (Array.isArray(dispVal))
                    dispVal = dispVal.join(',').toString();
                if (dispVal === "") {
                    mgVal = dispVal;
                }
                else {
                    mgVal = await this.getLinkValue(dispVal, line);
                }
                if (mgVal === null) {
                    let refMgVal = new RefParam(mgVal);
                    if (this._field !== null) {
                        mgVal = refMgVal.value;
                    }
                }
            }
            else {
                let blobContentType = (this._field !== null) ? this._field.getContentType() : BlobType.CONTENT_TYPE_UNKNOWN;
                mgVal = DisplayConvertor.Instance.disp2mg(dispVal, this._pic, this.getForm().getTask().getCompIdx(), blobContentType);
            }
        }
        return mgVal;
    }
    async getLinkValue(selectedIndice, line) {
        let size = 0;
        let result = NString.Empty;
        let temp = new List();
        await this.computeChoice(line);
        let choiceLink = this._choiceLinks.get_Item(line);
        if (choiceLink !== null) {
            size = choiceLink.length;
        }
        let linkIndice = this.getLinkIdxFromLayer(Misc.GetIntArray(selectedIndice));
        let array2 = linkIndice;
        for (let i = 0; i < array2.length; i = i + 1) {
            let idx = array2[i];
            if (idx >= size) {
                let dcVals = this.getDcVals();
                temp.push(dcVals.getLinkValue(idx - size));
            }
            else {
                if (idx >= 0 && idx < choiceLink.length) {
                    temp.push(choiceLink[idx]);
                }
                else {
                    temp.push(null);
                }
            }
        }
        if (this.DataType === StorageAttribute.BLOB_VECTOR) {
            let vectorType = new VectorType(this.getField());
            for (let indx = 0; indx < temp.length; indx = indx + 1) {
                vectorType.setVecCell(indx + 1, temp.get_Item(indx), temp.get_Item(indx) === null);
            }
            result = vectorType.toString();
        }
        else {
            if (temp.length > 1) {
                result = temp.join(",");
            }
            else {
                result = temp.get_Item(0);
            }
        }
        return result;
    }
    setText() {
        let displayLine = this.getDisplayLine(true);
        let mlsTranslatedValue = this.Value;
        if (super.isButton()) {
            mlsTranslatedValue = Events.Translate(this.Value);
        }
        if (this.isTextControl() || this.isButton())
            Commands.addValueWithLine(CommandType.SET_VALUE, this, displayLine, mlsTranslatedValue);
        else
            Commands.addOperationWithLine(CommandType.SET_PROPERTY, this, displayLine, HtmlProperties.Text, mlsTranslatedValue);
    }
    setRadioChecked(index) {
        Commands.addValueWithLine(CommandType.SET_VALUE, this, this.getDisplayLine(false), index);
        Commands.beginInvoke();
        if (index !== GuiConstants.DEFAULT_VALUE_INT) {
            this.setControlToFocus();
        }
    }
    setControlToFocus() {
        if (this.getField() !== null) {
            this.getField().ControlToFocus = this;
        }
    }
    setCheckBoxValue(line, mgVal) {
        let checkState = DisplayConvertor.toBoolean(mgVal);
        Commands.addValueWithLine(CommandType.SET_VALUE, this, line, checkState);
    }
    async refreshAndSetItemListByDataSource(line, valueChanged) {
        if (this.SupportsDataSource()) {
            let currDcId = this.getDcRef();
            if (this._rangeChanged) {
                valueChanged = true;
            }
            this._rangeChanged = false;
            if (this.isDataCtrl() && currDcId !== this._dcValId) {
                valueChanged = true;
                await this.refreshAndSetItemsList(line, true);
                this.setDcRef(this._dcValId);
            }
        }
        return valueChanged;
    }
    setDcRef(dcId) {
        this._dcTableRefs.set_Item(this.getDcLineNum(), dcId);
    }
    setImageList(url) {
    }
    async updatePropertyLogicNesting(propId, commandType, val, updateThis) {
        if (updateThis) {
            Commands.addOperationWithLine(CommandType.SET_PROPERTY, this, this.getDisplayLine(false), commandType === CommandType.PROP_SET_VISIBLE ? HtmlProperties.Visible : HtmlProperties.Enabled, val);
        }
        await this.updateChildrenPropValue(propId, commandType, val);
    }
    SetEnabled(val) {
        Commands.addOperationWithLine(CommandType.SET_PROPERTY, this, this.getDisplayLine(false), HtmlProperties.Enabled, val);
    }
    async updateChildrenPropValue(propId, commandType, val) {
        if (!super.isTableControl()) {
            for (let i = 0; i < this._linkedControls.length; i = i + 1) {
                let child = this._linkedControls.get_Item(i);
                if (child.IsFirstRefreshOfProps())
                    continue;
                let childValue = (child.GetComputedBooleanProperty(propId, true));
                childValue = childValue && val;
                let childOnCurrentLayer = this.isChildOnCurrentLayer(child);
                if (commandType === CommandType.PROP_SET_VISIBLE) {
                    if (childValue) {
                        childValue = childOnCurrentLayer;
                    }
                    await child.updateChildrenPropValue(propId, commandType, childValue);
                }
            }
            if (super.isSubform()) {
                await this.updateSubformChildrenPropValue(propId, commandType, val);
            }
        }
    }
    isDescendentOfControl(control) {
        let isContained = false;
        if (control !== null) {
            let parent = this.getParent();
            if (parent instanceof MgControlBase) {
                if (parent === control) {
                    isContained = true;
                }
                else {
                    let parentControl = ((parent instanceof MgControlBase) ? parent : null);
                    if (control.Type === MgControlType.CTRL_TYPE_COLUMN && parentControl !== null && parentControl.Type === MgControlType.CTRL_TYPE_TABLE && control.Layer === this.Layer)
                        isContained = true;
                    else
                        isContained = (parent.isDescendentOfControl(control));
                }
            }
        }
        return isContained;
    }
    updateSubformChildrenPropValue(propId, commandType, val) {
    }
    isChildOnCurrentLayer(child) {
        let ret = true;
        if (super.isChoiceControl()) {
            let parentLayer = 0;
            if (this.Value !== null) {
                parentLayer = this.getCurrentLinkIdx() + 1;
            }
            let layer = child.getLayer();
            if (parentLayer !== layer && layer > 0) {
                ret = false;
            }
        }
        return ret;
    }
    getCurrentLinkIdx() {
        let currLinkIdx = 0;
        if (super.isChoiceControl() || this.HasAutocomplete()) {
            let currentLayers = this.getCurrentIndexOfChoice();
            let currLinkIndice = this.getLinkIdxFromLayer(currentLayers);
            currLinkIdx = currLinkIndice[0];
        }
        return currLinkIdx;
    }
    getLayerFromLinkIdx(indice) {
        let layers = new Array(indice.length);
        for (let iCtr = 0; iCtr < indice.length; iCtr = iCtr + 1) {
            let idx = indice[iCtr];
            layers[iCtr] = idx;
            if (super.isTabControl() && idx >= 0) {
                let line = this.getDisplayLine(false);
                if (line < 0) {
                    line = 0;
                }
                let layerList = this._choiceLayerList.get_Item(line);
                if (layerList !== null && layerList.length > 0) {
                    layers[iCtr] = layerList.indexOf((idx + 1).toString());
                }
            }
        }
        return layers;
    }
    getLinkIdxFromLayer(layers) {
        let linkIndice = new Array(layers.length);
        for (let iCtr = 0; iCtr < layers.length; iCtr = iCtr + 1) {
            let layer = layers[iCtr];
            linkIndice[iCtr] = layer;
            if (super.isTabControl()) {
                if (layer < 0) {
                    linkIndice[iCtr] = 0;
                }
                else {
                    let line = this.getDisplayLine(false);
                    if (line < 0) {
                        line = 0;
                    }
                    let layerList = this._choiceLayerList.get_Item(line);
                    if (layerList !== null && typeof layerList !== "undefined" && layerList.length !== 0 && layer < layerList.length) {
                        linkIndice[iCtr] = NNumber.Parse(layerList[layer]) - 1;
                    }
                }
            }
        }
        return linkIndice;
    }
    isParentPropValue(propId) {
        let column = this.Form.getControlColumn(this);
        let parent = this.getLinkedParent(true);
        let result;
        if (parent === null) {
            result = true;
        }
        else {
            if (!parent.GetComputedBooleanProperty(propId, true)) {
                result = false;
            }
            else {
                if (!parent.isChildOnCurrentLayer(this)) {
                    result = false;
                }
                else {
                    result = (!(column !== null && !column.GetComputedBooleanProperty(propId, true)) && parent.isParentPropValue(propId));
                }
            }
        }
        return result;
    }
    getLinkedParent(checkSubformFather) {
        if (this._linkedParentDitIdx !== -1) {
            return this.Form.getCtrl(this._linkedParentDitIdx);
        }
        else {
            if (checkSubformFather && this.Form.isSubForm()) {
                return this.Form.getSubFormCtrl();
            }
            else {
                return null;
            }
        }
    }
    SetCurrReadOnly(newVal) {
        this._currReadOnly.set_Item(this.getDisplayLine(true), newVal);
    }
    GetCurrReadOnly() {
        let line = super.isTableControl() ? 0 : this.getDisplayLine(true);
        let currObj = this._currReadOnly.get_Item(line);
        let curr;
        if (currObj === null) {
            curr = false;
            this._currReadOnly.set_Item(line, curr);
        }
        else {
            curr = currObj;
        }
        return curr;
    }
    isDataCtrl() {
        return this._dataCtrl;
    }
    getDcRef() {
        let line = this.getDcLineNum();
        let obj = this._dcTableRefs.get_Item(this.getDcLineNum());
        let dcId;
        if (obj === null) {
            dcId = -2;
            this._dcTableRefs.set_Item(line, dcId);
        }
        else {
            dcId = obj;
        }
        return dcId;
    }
    getDcLineNum() {
        return this.IsRepeatable ? this.Form.DisplayLine : 0;
    }
    optionIsValid(option) {
        let isValid = true;
        if (this.DataType === StorageAttribute.NUMERIC && option.length > 0 && option.indexOf('0') === -1) {
            let convertNum = new NUM_TYPE();
            convertNum.num_4_a_std(option);
            if (convertNum.num_is_zero()) {
                isValid = false;
            }
        }
        return isValid;
    }
    async getOrderedDispList(choiceDispList, line) {
        let choiceLayerStr = "";
        let dispArryList = new List();
        let layerArryList = new List();
        let layerProp = this.getProp(PropInterface.PROP_TYPE_VISIBLE_LAYERS_LIST);
        if (layerProp !== null) {
            choiceLayerStr = await layerProp.getValue();
        }
        if (choiceLayerStr === null || StrUtil.rtrim(choiceLayerStr).length === 0) {
            choiceLayerStr = "";
        }
        if (choiceLayerStr.length > 0 && choiceDispList.length > 0) {
            let layerList = choiceLayerStr.split(",");
            let posCopyIndex = -1;
            for (let index = 0; index < layerList.length; index = index + 1) {
                let currLayerListVal = layerList[index].trim();
                if (!(layerArryList.indexOf(currLayerListVal) !== -1)) {
                    let refPosCopyIndex = new RefParam(posCopyIndex);
                    let isNumber = NNumber.TryParse(currLayerListVal, refPosCopyIndex);
                    posCopyIndex = refPosCopyIndex.value;
                    if (isNumber && posCopyIndex > 0 && posCopyIndex <= choiceDispList.length) {
                        dispArryList.push((choiceDispList[posCopyIndex - 1]).trim());
                        layerArryList.push(currLayerListVal);
                    }
                }
            }
        }
        let result;
        if (dispArryList.length === 0) {
            result = choiceDispList;
            this._choiceLayerList.set_Item(line, new Array(0));
        }
        else {
            result = new Array(dispArryList.length);
            for (let _ai = 0; _ai < result.length; ++_ai)
                result[_ai] = null;
            result = dispArryList.ToArray();
            this._choiceLayerList.set_Item(line, layerArryList.ToArray());
        }
        return result;
    }
    resetPrevVal() {
        let line = this.getDisplayLine(true);
        if (line < 0) {
            line = 0;
        }
        this._prevValues.set_Item(line, null);
        this.setPrevIsNull_ToNull();
    }
    setPrevIsNull_ToNull() {
        this._prevIsNulls.set_Item(this.getDisplayLine(true), null);
    }
    expressionSetAsData() {
        return this._valExpId > 0;
    }
    clearRange(line) {
        this._orgChoiceDisps.set_Item(line, null);
        this._choiceDisps.set_Item(line, null);
        this._choiceLinks.set_Item(line, null);
        this._choiceLayerList.set_Item(line, null);
    }
    haveToCheckParentValue() {
        let ret = false;
        if (this._linkedParentDitIdx !== -1) {
            if (!this.isContainedInLinkedParent()) {
                ret = true;
            }
            else {
                if (this.Form.getCtrl(this._linkedParentDitIdx).isTabControl() || this.Form.getCtrl(this._linkedParentDitIdx).isGroup()) {
                    ret = true;
                }
            }
        }
        else {
            if (this.Form.isSubForm()) {
                ret = true;
            }
        }
        return ret;
    }
    isContainedInLinkedParent() {
        return this._containerDitIdx !== -1 && this._containerDitIdx === this._linkedParentDitIdx;
    }
    static combineStringArrays(firstStrings, lastStrings) {
        let result = new Array();
        if (firstStrings !== null)
            result = result.concat(firstStrings);
        if (lastStrings !== null)
            result = result.concat(lastStrings);
        return result;
    }
    GetComputedBooleanProperty(propId, defaultRetVal, line) {
        if (arguments.length === 2 && (propId === null || propId.constructor === Number) && (defaultRetVal === null || defaultRetVal.constructor === Boolean)) {
            return this.GetComputedBooleanProperty_0(propId, defaultRetVal);
        }
        return this.GetComputedBooleanProperty_1(propId, defaultRetVal, line);
    }
    GetComputedBooleanProperty_0(propId, defaultRetVal) {
        let result = defaultRetVal;
        let prop = this.GetComputedProperty(propId);
        if (prop !== null) {
            result = prop.GetComputedValueBoolean();
        }
        return result;
    }
    GetComputedBooleanProperty_1(propId, defaultRetVal, line) {
        let result = defaultRetVal;
        if (this.isPropertyRepeatable(propId) && this.Form.DisplayLine !== line) {
            let prop = this.GetComputedProperty(propId);
            if (prop !== null) {
                let val = prop.getPrevValue(line);
                if (val !== null) {
                    result = DisplayConvertor.toBoolean(val);
                }
            }
        }
        else {
            result = this.GetComputedBooleanProperty(propId, defaultRetVal);
        }
        return result;
    }
    async checkPropWithLine(propId, defaultRetVal, line) {
        let result = defaultRetVal;
        if (this.isPropertyRepeatable(propId) && this.Form.DisplayLine !== line) {
            let prop = this.getProp(propId);
            if (prop !== null) {
                let val = prop.getPrevValue(line);
                if (val !== null) {
                    result = DisplayConvertor.toBoolean(val);
                }
            }
        }
        else {
            result = await this.checkProp(propId, defaultRetVal);
        }
        return result;
    }
    async checkProp(propId, defaultRetVal) {
        let result = defaultRetVal;
        let prop = this.getProp(propId);
        if (prop !== null) {
            result = await prop.getValueBoolean();
        }
        return result;
    }
    isPropertyRepeatable(propId) {
        let isRepeatable = this.IsRepeatable;
        if (isRepeatable) {
            return true;
        }
        else {
            return super.isTableControl() && Property.isRepeatableInTable();
        }
    }
    async SetAndRefreshDisplayValue(mgVal, isNull) {
        let dcRangeIsEmpty = true;
        if (mgVal === null) {
            this.Value = "";
            if (!isNull)
                mgVal = FieldDef.getMagicDefaultValue(this.DataType);
        }
        else {
            if (super.isChoiceControl()) {
                let line = this.getDisplayLine(true);
                if (line < 0) {
                    line = 0;
                }
                let layers = this.getLayerFromLinkIdx(await this.getIndexOfChoice(mgVal, line, isNull));
                if (layers[0] < 0 && super.isTabControl()) {
                    layers[0] = 0;
                    mgVal = await this.getLinkValue(layers[0].toString(), line);
                    isNull = false;
                    if (this._field !== null && mgVal !== null) {
                        await this._field.getTask().UpdateFieldValueAndStartRecompute(this._field, mgVal, isNull);
                    }
                }
                this.Value = Misc.GetCommaSeperatedString(layers);
                dcRangeIsEmpty = false;
            }
        }
        if (dcRangeIsEmpty) {
            if (!super.isCheckBox() && (this._field === null || !isNull || !this._field.hasNullDisplayValue())) {
                try {
                    if ((super.isImageControl() && this.DataType === StorageAttribute.BLOB) || super.IsImageButton()) {
                        this.Value = mgVal;
                    }
                    else {
                        if ((this._field === null && isNull) && (this.DataType === StorageAttribute.NUMERIC || this.DataType === StorageAttribute.DATE || this.DataType === StorageAttribute.TIME || this.DataType === StorageAttribute.BOOLEAN)) {
                            this.Value = mgVal;
                        }
                        else {
                            let inControl = this.InControl;
                            this.Value = DisplayConvertor.Instance.mg2disp(mgVal, this._range, this._pic, true, super.isSelectionCtrl(), this.getForm().getTask().getCompIdx(), inControl);
                        }
                    }
                    if (typeof this.Value === 'string' && (!this._pic.isAttrNumeric() || this._pic.isLeft())) {
                        this.rtrimValue();
                    }
                }
                catch (innerException) {
                    if (innerException instanceof ApplicationException) {
                        Events.WriteExceptionToLog(new ApplicationException(NString.Format("Control: '{0}', mgval: {1}", this.Name, mgVal), innerException));
                        this.Value = "";
                        mgVal = FieldDef.getMagicDefaultValue(this.DataType);
                    }
                    else
                        throw innerException;
                }
            }
            else {
                if (this.isCheckBox())
                    this.Value = DisplayConvertor.toBoolean(mgVal);
                else
                    this.Value = mgVal;
            }
        }
        if ((super.isImageControl() && this.DataType !== StorageAttribute.BLOB) || super.IsImageButton()) {
            mgVal = Events.TranslateLogicalName(mgVal);
        }
        this.IsNull = isNull;
        await this.RefreshDisplayValue(mgVal);
    }
    UpdateModifiedByUser(newValue) {
        if (!(super.isButton() || super.isImageControl())) {
            let ctrlCurrIsNull = this.IsNull;
            let valChanged = (this.isDifferentValue(newValue, false, false));
            if (!valChanged) {
                if (this.CanGetNullFromControlValue()) {
                    valChanged = (ctrlCurrIsNull !== this.isNullValue(newValue));
                }
            }
            if (valChanged || this.KeyStrokeOn) {
                this.ModifiedByUser = true;
            }
        }
    }
    rtrimValue() {
        if ((super.isImageControl() && this.DataType === StorageAttribute.BLOB) || super.IsImageButton()) {
            this.Value = NString.TrimEnd(this.Value);
        }
        else {
            let minimumValueLength = this.getMinimumValueLength();
            if (this.Value.length > minimumValueLength) {
                let str = this.Value.substr(minimumValueLength);
                this.Value = this.Value.substr(0, minimumValueLength);
                this.Value = this.Value + StrUtil.rtrimWithNull(str, true);
            }
            if (super.isTextControl() && this.DataType === StorageAttribute.NUMERIC) {
                this.Value = NString.TrimEnd(this.Value);
            }
        }
    }
    getMinimumValueLength() {
        let minLength = this._pic.getMaskLength();
        while (minLength > 0 && !this._pic.picIsMask(minLength - 1)) {
            minLength = minLength - 1;
        }
        return minLength;
    }
    isNullValue(str) {
        let retIsNullControl = false;
        if (this.CanGetNullFromControlValue() && str === "") {
            retIsNullControl = true;
        }
        return retIsNullControl;
    }
    CanGetNullFromControlValue() {
        return false;
    }
    isChoiceNull(idx) {
        let result;
        if (idx >= 0) {
            let dcVals = this.getDcVals();
            return dcVals.isNull(idx);
        }
        else {
            return result;
        }
    }
    async ComputeAndRefreshDisplayValue(forceRefresh) {
        if (forceRefresh) {
            this.resetPrevVal();
        }
        if (this._valExpId > 0) {
            let wasEvaluated;
            let refWasEvaluated = new RefParam(wasEvaluated);
            let retExp = await this.EvaluateExpression(this._valExpId, this.DataType, this._pic.getSize(), true, StorageAttribute.SKIP, false, refWasEvaluated);
            wasEvaluated = refWasEvaluated.value;
            if (wasEvaluated) {
                await this.SetAndRefreshDisplayValue(retExp, retExp === null);
            }
        }
        else {
            if (this._field !== null) {
                let value = NString.Empty;
                let isNull = false;
                let refValue = new RefParam(value);
                let refIsNull = new RefParam(isNull);
                this._field.getTask().getFieldDisplayValue(this._field, refValue, refIsNull);
                value = refValue.value;
                isNull = refIsNull.value;
                await this.SetAndRefreshDisplayValue(value, isNull);
            }
            else {
                if (!super.isTableControl() && !super.isColumnControl()) {
                    await this.RefreshDisplayValue(this.Value);
                }
            }
        }
    }
    ShouldRefreshOnControlEnter() {
        return this.Type === MgControlType.CTRL_TYPE_TEXT && this.getField() !== null && (this.getField().getType() === StorageAttribute.DATE ||
            this.getField().getType() === StorageAttribute.TIME || this.getField().getType() === StorageAttribute.NUMERIC);
    }
    async RefreshDisplayValue(mgVal) {
        let line = this.getDisplayLine(true);
        let valueChanged = true;
        let saveValueChanged = valueChanged;
        if (line < 0) {
            line = 0;
        }
        try {
            let prevValue = this._prevValues.get_Item(line);
            let prevNull = this.getPrevIsNull();
            if (StorageAttributeCheck.IsTypeAlphaOrUnicode(this.DataType)) {
                if (StrUtil.rtrim(mgVal) === StrUtil.rtrim(prevValue) ||
                    (mgVal !== null && StrUtil.rtrim(mgVal) === StrUtil.rtrim(prevValue))) {
                    valueChanged = false;
                }
                if (prevNull !== this.IsNull) {
                    valueChanged = true;
                }
            }
            else {
                ;
                if (mgVal === prevValue || (mgVal !== null && mgVal === prevValue))
                    valueChanged = false;
                if (prevNull !== this.IsNull)
                    valueChanged = true;
            }
            this._prevValues.set_Item(line, mgVal);
            this.setPrevIsNull(this.IsNull);
            switch (this.Type) {
                case MgControlType.CTRL_TYPE_TABLE:
                case MgControlType.CTRL_TYPE_COLUMN:
                case MgControlType.CTRL_TYPE_LABEL:
                case MgControlType.CTRL_TYPE_SUBFORM:
                case MgControlType.CTRL_TYPE_GROUP:
                case MgControlType.CTRL_TYPE_STATUS_BAR:
                case MgControlType.CTRL_TYPE_FRAME_SET:
                case MgControlType.CTRL_TYPE_CONTAINER:
                case MgControlType.CTRL_TYPE_FRAME_FORM:
                case MgControlType.CTRL_TYPE_SB_LABEL:
                case MgControlType.CTRL_TYPE_LINE:
                    return;
                case MgControlType.CTRL_TYPE_BROWSER:
                    if (valueChanged && this.Value != null)
                        this.setUrl();
                    return;
                case MgControlType.CTRL_TYPE_IMAGE:
                case MgControlType.CTRL_TYPE_SB_IMAGE:
                    if (valueChanged && this.Value != null)
                        this.setImage();
                    return;
                case MgControlType.CTRL_TYPE_CHECKBOX:
                    if (valueChanged || this.forceRefresh)
                        this.setCheckBoxValue(line, mgVal);
                    return;
                case MgControlType.CTRL_TYPE_TAB:
                case MgControlType.CTRL_TYPE_COMBO:
                case MgControlType.CTRL_TYPE_LIST:
                    saveValueChanged = valueChanged;
                    valueChanged = await this.refreshAndSetItemListByDataSource(line, valueChanged);
                    if (valueChanged) {
                        let prevDisplayValue = null;
                        if (saveValueChanged === valueChanged) {
                            if (prevValue !== null) {
                                prevDisplayValue = this.getLayerFromLinkIdx(await this.getIndexOfChoice(prevValue, line, prevNull));
                                for (let i = 0; i < prevDisplayValue.length; i = i + 1) {
                                    if (prevDisplayValue[i] === GuiConstants.DEFAULT_VALUE_INT) {
                                        prevDisplayValue[i] = GuiConstants.DEFAULT_LIST_VALUE;
                                    }
                                }
                            }
                        }
                        let indice = Misc.GetIntArray(this.Value);
                        if (this.isTabControl()) {
                            let currentLayer = this.getCurrentLinkIdx();
                            Commands.addOperationWithLine(CommandType.SET_PROPERTY, this, line, HtmlProperties.SelectedValue, indice[0].toString());
                            Commands.addOperationWithLine(CommandType.SET_PROPERTY, this, line, HtmlProperties.SelectedLayer, currentLayer.toString());
                        }
                        if (this.IsMultipleSelectionListBox()) {
                            Commands.addValueWithLine(CommandType.SET_VALUE, this, line, indice);
                        }
                        else
                            Commands.addValueWithLine(CommandType.SET_VALUE, this, line, +this.Value);
                    }
                    break;
                case MgControlType.CTRL_TYPE_BUTTON:
                case MgControlType.CTRL_TYPE_TEXT:
                    if (this.HasAutocomplete())
                        await this.refreshAndSetItemListByDataSource(line, valueChanged);
                    if (valueChanged) {
                        if (this.IsImageButton()) {
                            this.setImageWithFileName(this.Value);
                        }
                        else
                            this.setText();
                    }
                    else
                        return;
                    break;
                case MgControlType.CTRL_TYPE_RADIO:
                    valueChanged = await this.refreshAndSetItemListByDataSource(line, valueChanged);
                    if (!valueChanged)
                        return;
                    this.setRadioChecked(+this.Value);
                    break;
                default:
                    Events.WriteExceptionToLog(NString.Format("in Control.RefreshDisplayValue() unknown type: {0}", this.Type));
                    return;
            }
            if (super.isChoiceControl() && valueChanged)
                await this.updatePropertyLogicNesting(PropInterface.PROP_TYPE_VISIBLE, CommandType.PROP_SET_VISIBLE, await this.checkProp(PropInterface.PROP_TYPE_VISIBLE, true), false);
        }
        catch (ex_360) {
            if (ex_360 instanceof Exception) {
                Events.WriteExceptionToLog(NString.Format("in Control.RefreshDisplayValue() for control: {0}", this.Name));
            }
            else
                throw ex_360;
        }
    }
    async RefreshDisplay(onlyRepeatableProps) {
        if (isNullOrUndefined(onlyRepeatableProps))
            onlyRepeatableProps = false;
        await this.RefreshDisplay_1(onlyRepeatableProps);
    }
    async RefreshDisplay_1(onlyRepeatableProps) {
        if (!(super.isSubform() && this.GetSubformMgForm() !== null && this.GetSubformMgForm().inRefreshDisplay())) {
            await this.refreshProperties(onlyRepeatableProps);
            if (this.ShouldComputeAndRefreshOnClosedForm()) {
                await this.ComputeAndRefreshDisplayValue(false);
            }
            let isParkableCtrl = !(await this.isParkable(false)) || !await this.isModifiable();
            if (this.GetCurrReadOnly() !== isParkableCtrl) {
                this.SetCurrReadOnly(isParkableCtrl);
                await Manager.SetReadOnlyControl(this, isParkableCtrl);
            }
            let mustInputProperty = this.GetComputedProperty(PropInterface.PROP_TYPE_MUST_INPUT);
            if (mustInputProperty != null && mustInputProperty.isExpression()) {
                let prevMustInput = this.GetComputedBooleanProperty(PropInterface.PROP_TYPE_MUST_INPUT, false);
                if (prevMustInput !== await this.checkProp(PropInterface.PROP_TYPE_MUST_INPUT, false))
                    Commands.addOperationWithLine(CommandType.SET_PROPERTY, this, this.getDisplayLine(false), HtmlProperties.MustInput, !prevMustInput);
            }
            else if (mustInputProperty != null) {
                let mustInputVal = this.GetComputedBooleanProperty(PropInterface.PROP_TYPE_MUST_INPUT, false);
                if (mustInputVal)
                    Commands.addOperationWithLine(CommandType.SET_PROPERTY, this, this.getDisplayLine(false), HtmlProperties.MustInput, mustInputVal);
            }
            if (!this.GetComputedBooleanProperty(PropInterface.PROP_TYPE_TAB_IN, true))
                Commands.addOperationWithLine(CommandType.SET_PROPERTY, this, this.getDisplayLine(false), HtmlProperties.TabIndex, -1);
        }
    }
    ShouldComputeAndRefreshOnClosedForm() {
        return true;
    }
    async refreshProperties(onlyRepeatableProps) {
        if (this._field !== null && this.DataType !== StorageAttribute.NONE && this._firstRefreshProperties)
            Commands.addWithNumber(CommandType.SET_ATTRIBUTE, this, this.DataType.charCodeAt(0));
        if (this._propTab !== null) {
            try {
                if (this._firstRefreshProperties) {
                    this.createDefaultProps();
                    await this.ZoomButtonInitialization();
                }
                if (!(await this._propTab.RefreshDisplay(false, onlyRepeatableProps))) {
                    Events.WriteExceptionToLog(NString.Format("Control '{0}': Not all properties could be set", this.getName()));
                }
                if (this._firstRefreshProperties) {
                    if (!onlyRepeatableProps) {
                        this._firstRefreshProperties = false;
                    }
                }
            }
            catch (ex) {
                if (ex instanceof Exception) {
                    let msg = new StringBuilder(NString.Format("Control '{0}': {1}", this.getName(), ex.Message));
                    Events.WriteExceptionToLog(msg.ToString());
                }
                else
                    throw ex;
            }
        }
    }
    async refreshTabForLayerList(line) {
        let currentLinkValue = null;
        if (this.getForm().Opened) {
            let indice = this.getCurrentIndexOfChoice();
            let currentLayer = indice[0];
            currentLinkValue = await this.getLinkValue(currentLayer.toString(), line);
        }
        await this.refreshAndSetItemsList(line, false);
        if (this.getForm().Opened) {
            await this.SetAndRefreshDisplayValue(currentLinkValue, this.IsNull);
        }
    }
    async refreshAndSetItemsList(line, execComputeChoice) {
        if (execComputeChoice) {
            this.clearRange(line);
        }
        let optionsStrings = await this.refreshDispRange(execComputeChoice);
        if (super.isChoiceControl() || this.HasAutocomplete()) {
            let options = optionsStrings.map((displayValue, index) => {
                return { index, displayValue };
            });
            Commands.addOperationWithLine(CommandType.SET_PROPERTY, this, line, HtmlProperties.ItemsList, options);
        }
    }
    async refreshDispRange(execComputeChoice) {
        let selectCmd = new Array(0);
        let line = this.getDisplayLine(true);
        if (line < 0) {
            line = 0;
        }
        if (this.SupportsDataSource()) {
            let dataView = this.getForm().getTask().DataView;
            let dcv = dataView.getDcValues(this._dcValId);
            let dispVals = await this.getDispVals(line, execComputeChoice);
            if (this._hasValidItmAndDispVal) {
                selectCmd = MgControlBase.combineStringArrays(dispVals, dcv.getDispVals());
            }
        }
        else {
            selectCmd = await this.getDispVals(line, execComputeChoice);
        }
        this._rangeChanged = true;
        return await this.getOrderedDispList(selectCmd, line);
    }
    async refreshItmRange(execComputeChoice) {
        Debug.Assert(this.SupportsDataSource());
        let orderedItmList = new Array(0);
        let line = this.getDisplayLine(false);
        if (line < 0) {
            line = 0;
        }
        if (execComputeChoice) {
            await this.computeChoice(line);
        }
        let linkVals = this._choiceLinks.get_Item(line);
        let dataView = this.getForm().getTask().DataView;
        let dcv = dataView.getDcValues(this._dcValId);
        if (this._hasValidItmAndDispVal) {
            orderedItmList = MgControlBase.combineStringArrays(linkVals, dcv.GetLinkVals());
        }
        return await this.getOrderedDispList(orderedItmList, line);
    }
    async GetItemsRange() {
        return await this.refreshItmRange(false);
    }
    async GetDisplayRange() {
        return await this.refreshDispRange(false);
    }
    createDefaultProps() {
        this.getProp(PropInterface.PROP_TYPE_VISIBLE);
        this.getProp(PropInterface.PROP_TYPE_ENABLED);
        if (super.isTableControl()) {
            this.getProp(PropInterface.PROP_TYPE_SCROLL_BAR);
        }
    }
    setLinkedParentIdx(linkedParentDitIdx) {
        this._linkedParentDitIdx = linkedParentDitIdx;
    }
    buildPicture(oldVal, val) {
        if (super.isTableControl() || super.isColumnControl())
            return null;
        if (this._picExpExists) {
            this._vd = null;
        }
        if (this._vd === null) {
            this._vd = new ValidationDetails(oldVal, val, this._range, this._pic, this);
        }
        else {
            this._vd.setValue(val);
            this._vd.setOldValue(oldVal);
        }
        return this._vd;
    }
    isDifferentValue(newValue, isNull, checkNullValue) {
        let ctrlCurrValue = this.Value;
        let valChanged = false;
        if (newValue instanceof Date) {
            if (this.DataType === StorageAttribute.DATE)
                valChanged = !this.CompareDate(newValue, ctrlCurrValue);
            else
                valChanged = !this.CompareTime(newValue, ctrlCurrValue);
        }
        if (typeof newValue === 'boolean' || typeof newValue === 'number') {
            if (this.isSelectionCtrl() && typeof ctrlCurrValue === 'string') {
                let newValuestr = newValue.toString();
                valChanged = (newValuestr !== ctrlCurrValue);
            }
            else {
                valChanged = (newValue !== ctrlCurrValue);
            }
        }
        else {
            if (this.isSelectionCtrl() && typeof ctrlCurrValue === 'string' && Array.isArray(newValue)) {
                let newValueStr = newValue.join(',').toString();
                valChanged = !(StrUtil.rtrim(newValueStr) === StrUtil.rtrim(ctrlCurrValue));
            }
            else {
                let ctrlCurrIsNull = this.IsNull;
                valChanged = !(StrUtil.rtrim(newValue) === StrUtil.rtrim(ctrlCurrValue));
                if (!valChanged && checkNullValue) {
                    valChanged = (isNull !== ctrlCurrIsNull);
                }
            }
        }
        return valChanged;
    }
    CompareDate(d1, d2) {
        let retval;
        if (d1 === null && d2 === null)
            retval = true;
        else if ((d1 === null && d2 !== null) || (d1 !== null && d2 === null))
            retval = false;
        else if (d1.getFullYear() === d2.getFullYear()
            && d1.getDate() === d2.getDate()
            && d1.getMonth() === d2.getMonth())
            retval = true;
        return retval;
    }
    CompareTime(d1, d2) {
        let retval;
        if (d1 === null && d2 === null)
            retval = true;
        else if ((d1 === null && d2 !== null) || (d1 !== null && d2 === null))
            retval = false;
        else if (d1.getHours() === d2.getHours()
            && d1.getMinutes() === d2.getMinutes()
            && d1.getSeconds() === d2.getSeconds())
            retval = true;
        return retval;
    }
    isColumnSortable() {
        let isSortable = false;
        if (this.Type === MgControlType.CTRL_TYPE_COLUMN && this._propTab.propExists(PropInterface.PROP_TYPE_SORT_COLUMN)) {
            isSortable = this.GetComputedProperty(PropInterface.PROP_TYPE_SORT_COLUMN).GetComputedValueBoolean();
        }
        return isSortable;
    }
    getColumnChildControl() {
        if (this.Type !== MgControlType.CTRL_TYPE_COLUMN)
            return null;
        let linkedControls = this._parentTable.getLinkedControls();
        return linkedControls.find((control) => (control.getLayer() === this.getLayer() && control.Type !== MgControlType.CTRL_TYPE_COLUMN && !control.IsTableHeaderChild));
    }
    getControlNameForHandlerSearch() {
        let childControl = this.getColumnChildControl();
        if (childControl !== null) {
            return childControl.Name;
        }
        else
            return this.Name;
    }
    linkCtrl(ctrl) {
        this._linkedControls.push(ctrl);
    }
    removeRefFromField() {
        if (this._field !== null) {
            this._field.RemoveControl(this);
        }
    }
    getDitIdx() {
        return this._ditIdx;
    }
    setDcValId(dcValId) {
        this._dcValId = dcValId;
    }
    SetKeyboardLanguage() {
        Commands.beginInvoke();
    }
    Init() {
    }
    async IsParkable() {
        return await this.isParkable(true);
    }
    HasExpOrParentTaskField() {
        let task = this.getForm().getTask();
        return (this.getField() === null || this.getField().getTask() !== task);
    }
    async isParkable(checkEnabledAndVisible) {
        let result;
        result = !super.isTableControl() && !super.isColumnControl();
        let task = this.getForm().getTask();
        if (result) {
            result = task.IsInteractive;
        }
        if (result && task.DataView.isEmptyDataview() && this._field !== null && this._field.PartOfDataview) {
            result = false;
        }
        if (result) {
            if (!super.isSubform()) {
                result = ((this._field !== null && this._field.getTask() === task) || this.isButton());
            }
            if (result && checkEnabledAndVisible) {
                result = (this.isEnabled() && this.isVisible());
            }
            if ((result && this._parentTable !== null) && checkEnabledAndVisible) {
                result = (this._parentTable.GetComputedBooleanProperty(PropInterface.PROP_TYPE_ENABLED, true) && this._parentTable.GetComputedBooleanProperty(PropInterface.PROP_TYPE_VISIBLE, true));
            }
        }
        return result;
    }
    RaiseControlHitOnLeftClickOfMouseDown() {
        let raiseCtrlHit = false;
        if (!super.IsHyperTextButton() && !super.isRadio() && !super.isTabControl()) {
            raiseCtrlHit = true;
        }
        return raiseCtrlHit;
    }
    GetVarIndex() {
        return 0;
    }
    RaiseControlHitOnMouseDown(leftClickWasPressed) {
        let raiseCtrlHit = false;
        if (leftClickWasPressed) {
            if (this.RaiseControlHitOnLeftClickOfMouseDown()) {
                raiseCtrlHit = true;
            }
        }
        return raiseCtrlHit;
    }
    OnSubformClick() {
        Manager.EventsManager.addGuiTriggeredEventWithTaskAndCode(this.getForm().getTask(), InternalInterface.MG_ACT_HIT);
    }
    GetSubformMgForm() {
        return null;
    }
    PropertyExists(id) {
        return this._propTab.propExists(id);
    }
    toString() {
        return "{" + this.constructor.name + ": " + MgControlType[this.Type] + "}";
    }
    isDateZero(val) {
        let zeroDate = false;
        if (val instanceof Date) {
            if (val.getMonth() === 0 && val.getFullYear() === 0 && val.getDay() === 0)
                zeroDate = true;
        }
        return zeroDate;
    }
    IsDefaultRouterOutlet() {
        return (this.Type === MgControlType.CTRL_TYPE_SUBFORM && this.GetComputedBooleanProperty_0(PropInterface.PROP_TYPE_DEFAULT_OUTLET, false));
    }
    IsRouterOutlet() {
        return (this.Type === MgControlType.CTRL_TYPE_SUBFORM && this.GetComputedBooleanProperty_0(PropInterface.PROP_TYPE_IS_ROUTER_OUTLET, false));
    }
    async ZoomButtonInitialization() {
        if (this.isTextControl() && this.checkIfExistProp(PropInterface.PROP_TYPE_CONNECTED_ZOOM_CONTROL)) {
            let connectedButton = this.Form.CtrlTab.GetControlByIsn((await (await this.getProp(PropInterface.PROP_TYPE_CONNECTED_ZOOM_CONTROL)).getValueInt()));
            if (connectedButton) {
                let imageFileName = null;
                if (connectedButton.IsImageButton() && connectedButton.PropertyExists(PropInterface.PROP_TYPE_IMAGE_FILENAME)) {
                    imageFileName = await (await connectedButton.getProp(PropInterface.PROP_TYPE_IMAGE_FILENAME)).getValue();
                    if (!Misc.IsWebUrl(imageFileName))
                        imageFileName = "./assets/images/" + imageFileName;
                    Commands.addOperationWithLine(CommandType.SET_STYLE, this, 0, MagicProperties.ImageFile, imageFileName);
                }
                this.ConnectedControl = connectedButton;
                connectedButton.ConnectedControl = this;
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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