import { ApplicationException, Hashtable, List, NNumber, NString } from "@magic-xpa/mscorelib";
import { Property } from "./Property";
import { XMLConstants, XmlParser } from "@magic-xpa/utils";
import { Events } from "../../Events";
import { PropInterface } from "./PropInterface";
import { GuiFormPropertyAdapter } from "./GuiFormPropertyAdapter";
import { GuiControlPropertyAdapter } from "./GuiControlPropertyAdapter";
export class PropTable {
    constructor(parent_) {
        this._hashTab = null;
        this._props = null;
        this._parent = null;
        if (arguments.length === 0) {
            this.constructor_0();
            return;
        }
        this.constructor_1(parent_);
    }
    constructor_0() {
        this._props = new List();
        this._hashTab = new Hashtable(20);
    }
    constructor_1(parent_) {
        this.constructor_0();
        this._parent = parent_;
    }
    fillData(parentObj, parType, parser) {
        while (this.initInnerObjects(parser.getNextTag(), parentObj, parType, parser)) {
        }
    }
    fillDataByExists(task, parser) {
        while (this.fillExistInnerObjects(parser.getNextTag(), task, parser)) {
        }
    }
    initInnerObjects(foundTagName, parentObj, parType, parser) {
        if (foundTagName !== null && foundTagName === XMLConstants.MG_TAG_PROP) {
            let property = new Property();
            property.fillData(parentObj, parType, parser);
            this.addProp(property);
        }
        else
            return false;
        return true;
    }
    addProp(prop) {
        this._hashTab.set_Item(prop.getID(), prop);
        this._props.push(prop);
    }
    delPropById(id) {
        if (this._props === null)
            return;
        let existingProp = this._hashTab.get_Item(id);
        if (existingProp !== null) {
            this._props.Remove(existingProp);
            this._hashTab.Remove(id);
        }
    }
    fillExistInnerObjects(nameOfFound, task, parser) {
        let tokensVector = new List();
        let endContext = -1;
        if (nameOfFound == null)
            return false;
        if (nameOfFound === XMLConstants.MG_TAG_CONTROL)
            endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_CLOSE, parser.getCurrIndex());
        else if (nameOfFound === XMLConstants.MG_TAG_PROP)
            endContext = parser.getXMLdata().indexOf(XMLConstants.TAG_TERM, parser.getCurrIndex());
        else if (nameOfFound === ('/' + XMLConstants.MG_TAG_CONTROL) ||
            GuiFormPropertyAdapter.IsEndFormTag(nameOfFound)) {
            parser.setCurrIndex2EndOfTag();
            return false;
        }
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            let tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(nameOfFound) + nameOfFound.length);
            tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            if (nameOfFound === XMLConstants.MG_TAG_CONTROL) {
                this._parent = task.getCtrl(NNumber.Parse(this.fillName(tokensVector)));
                parser.setCurrIndex(++endContext);
                return true;
            }
            else if (nameOfFound === XMLConstants.MG_TAG_PROP) {
                if (this._parent != null) {
                    let strPropId = this.fillName(tokensVector);
                    let propId = NNumber.Parse(strPropId);
                    let prop = null;
                    if (this._parent != null)
                        prop = this._parent.getProp(propId);
                    if (prop == null)
                        Events.WriteExceptionToLog(NString.Format("in PropTable.fillExistInnerObjects() no property with id={0}", strPropId));
                    else
                        this.addProp(prop);
                    parser.setCurrIndex(endContext + XMLConstants.TAG_TERM.length);
                    return true;
                }
                else
                    Events.WriteExceptionToLog("in PropTable.fillExistInnerObjects() missing control");
            }
            else {
                Events.WriteExceptionToLog(NString.Format("in PropTable.fillExistInnerObjects() illegal tag name: {0}", nameOfFound));
                parser.setCurrIndex(++endContext);
                return true;
            }
        }
        if (nameOfFound === XMLConstants.MG_TAG_CONTROL) {
            parser.setCurrIndex(++endContext);
            return true;
        }
        else if (GuiFormPropertyAdapter.IsFormTag(nameOfFound)) {
            this._parent = task.getForm();
            this._parent.fillName(nameOfFound);
            return true;
        }
        else if (nameOfFound === ('/' + XMLConstants.MG_TAG_FLD))
            return false;
        parser.setCurrIndex(endContext + XMLConstants.TAG_TERM.length);
        return true;
    }
    fillName(tokensVector) {
        let attribute = null;
        let valueStr;
        for (let j = 0; j < tokensVector.length; j = j + 2) {
            attribute = tokensVector.get_Item(j);
            valueStr = tokensVector.get_Item(j + 1);
            if (attribute === XMLConstants.MG_ATTR_DITIDX || attribute === XMLConstants.MG_ATTR_ID)
                return valueStr;
        }
        Events.WriteExceptionToLog(NString.Format("Unrecognized attribute: '{0}'", attribute));
        return null;
    }
    getSize() {
        return this._props.length;
    }
    getProp(idx) {
        if (idx < 0 || idx >= this._props.length)
            return null;
        return this._props.get_Item(idx);
    }
    getPropById(id) {
        return this._hashTab.get_Item(id);
    }
    propExists(id) {
        return this.getPropById(id) !== null;
    }
    setProp(propId, val, parent, parentType) {
        let prop = this.getPropById(propId);
        if (prop === null) {
            prop = new Property(propId, parent, parentType);
            this.addProp(prop);
        }
        prop.setValue(val);
        prop.setOrgValue();
    }
    async RefreshDisplay(forceRefresh, onlyRepeatableProps) {
        let allPropsRefreshed = true;
        let prop;
        let i = 0;
        let form = null;
        let visibleProps = new List();
        if (this._parent != null && this._parent instanceof GuiControlPropertyAdapter)
            form = this._parent.getForm();
        if (form != null)
            form.checkAndCreateRow(form.DisplayLine);
        for (i = 0; i < this.getSize(); i++) {
            prop = this.getProp(i);
            try {
                if (onlyRepeatableProps && !Property.isRepeatableInTable())
                    continue;
                if (prop.getID() === PropInterface.PROP_TYPE_VISIBLE)
                    visibleProps.push(prop);
                else
                    await prop.RefreshDisplay(forceRefresh);
            }
            catch (ex) {
                if (ex instanceof ApplicationException) {
                    Events.WriteExceptionToLog(ex);
                    allPropsRefreshed = false;
                }
                else
                    throw ex;
            }
        }
        for (let i = 0; i < visibleProps.length; i++) {
            let visibleProp = visibleProps[i];
            await visibleProp.RefreshDisplay(forceRefresh);
        }
        return allPropsRefreshed;
    }
    updatePrevValueArray(newSize) {
        for (let i = 0; i < this._props.length; i = i + 1) {
            let property = this._props.get_Item(i);
            property.updatePrevValueArray(newSize);
        }
    }
    resetPrevValueArray() {
        for (let i = 0; i < this._props.length; i = i + 1) {
            let property = this._props.get_Item(i);
            property.ResetPrevValueArray();
        }
    }
    resetPrevValAt(index) {
        for (let i = 0; i < this._props.length; i = i + 1) {
            let property = this._props.get_Item(i);
            property.resetPrevValAt(index);
        }
    }
    clearLabelPrevValueArray() {
        let labelProperty = this.getPropById(45);
        if (labelProperty !== null)
            labelProperty.clearPrevValueArray();
    }
    getCtrlRef() {
        return ((this._parent instanceof GuiControlPropertyAdapter) ? this._parent : null);
    }
    InsertPrevValueArray(count) {
        for (let i = 0; i < this._props.length; i++) {
            let prop = this._props[i];
            prop.InsertPrevValueArray(count);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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