import { ApplicationException, Debug, Int32, List, NNumber, NString, RefParam, StringBuilder } from '@magic-xpa/mscorelib';
import { Constants, Logger_LogLevels, MgControlType, StorageAttribute, StrUtil, UtilImeJpn, UtilStrByteMode, XMLConstants, XmlParser } from '@magic-xpa/utils';
import { TaskDefinitionIdTableSaxHandler } from '../tasks/TaskDefinitionIdTableSaxHandler';
import { PIC } from './PIC';
import { Events } from '../../Events';
import { GuiConstants } from '../../GuiConstants';
import { PropInterface } from './PropInterface';
import { GuiControlPropertyAdapter } from './GuiControlPropertyAdapter';
import { DisplayConvertor } from './DisplayConvertor';
import { Commands } from '../../Commands';
import { CommandType, HtmlProperties } from '../../GuiEnums';
import { PropDefaults } from './PropDefaults';
export class CustomValidatorProp {
}
;
export class CustomValidator {
    constructor(valid, errMessage) {
        this._valid = valid;
        this._errorMessage = errMessage;
    }
    isValid() { return this._valid; }
    getErrMessage() { return this._errorMessage; }
}
export class Property {
    get TaskDefinitionId() {
        return this._taskDefinitionId;
    }
    constructor(cId, cParentObj, parType, val) {
        this._id = Int32.MinValue;
        this._dataType = StorageAttribute.NONE;
        this._val = null;
        this._pic = null;
        this._expId = 0;
        this._parentObj = null;
        this._parentType = null;
        this._prevValues = null;
        this._orgValue = null;
        this._taskDefinitionId = null;
        this.StudioValue = null;
        this._expAlreadyComputedOnce = false;
        this.customProperties = new Map();
        this.customPropertiesValues = new Map();
        this.customValidatorProperties = new Map();
        this.customValidatorValues = new Map();
        if (arguments.length === 0)
            this.constructor_0();
        else if (arguments.length === 3)
            this.constructor_1(cId, cParentObj, parType);
        else
            this.constructor_2(cId, cParentObj, parType, val);
    }
    constructor_0() {
        this._prevValues = new List();
    }
    constructor_1(cId, cParentObj, parType) {
        this.constructor_0();
        this.setId(cId);
        this._parentObj = cParentObj;
        this._parentType = parType;
        this.setDataType();
    }
    constructor_2(cId, cParentObj, parType, val) {
        this.constructor_1(cId, cParentObj, parType);
        this.setValue(val);
        this.setOrgValue();
    }
    setId(cId) {
        this._id = cId;
    }
    fillData(parentRef, parType, parser) {
        let endContext = parser.getXMLdata().indexOf(">", parser.getCurrIndex());
        if (this._parentObj === null && parentRef !== null) {
            this._parentObj = parentRef;
            this._parentType = parType;
        }
        if (endContext !== -1 && endContext < parser.getXMLdata().length) {
            let tag = parser.getXMLsubstring(endContext);
            parser.add2CurrIndex(tag.indexOf(XMLConstants.MG_TAG_PROP) + XMLConstants.MG_TAG_PROP.length);
            let tokensVector = XmlParser.getTokens(parser.getXMLsubstring(endContext), XMLConstants.XML_ATTR_DELIM);
            if (Events.ShouldLog(Logger_LogLevels.Development)) {
                Events.WriteDevToLog(NString.Format("In Prop.FillData(): {0}", tokensVector.toString()));
            }
            this.initElements(tokensVector);
            parser.setCurrIndex(endContext + XMLConstants.TAG_CLOSE.length);
            this.InitInnerObjects(parser);
            if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL)
                this.setControlAttributes();
            else if (this._parentType === GuiConstants.PARENT_TYPE_FORM)
                this._prevValues.SetSize(1);
        }
        else
            Events.WriteExceptionToLog("In Property.FillData(): Out of string bounds");
    }
    InitInnerObjects(xmlParser) {
        let nextTag = xmlParser.getNextTag();
        if (nextTag === XMLConstants.MG_TAG_TASKDEFINITIONID_ENTRY) {
            let xmlBuffer = xmlParser.ReadToEndOfCurrentElement();
            this.InitTaskDefinitionId(xmlBuffer);
            let endContext = xmlParser.getXMLdata().indexOf(XMLConstants.MG_TAG_PROP + XMLConstants.TAG_CLOSE, xmlParser.getCurrIndex());
            xmlParser.setCurrIndex(endContext + (XMLConstants.MG_TAG_PROP + XMLConstants.TAG_CLOSE).length);
        }
    }
    initElements(tokensVector) {
        for (let j = 0; j < tokensVector.length; j += 2) {
            let attribute = (tokensVector.get_Item(j));
            let valueStr = (tokensVector.get_Item(j + 1));
            switch (attribute) {
                case XMLConstants.MG_ATTR_ID:
                    this.setId(XmlParser.getInt(valueStr));
                    this.setDataType();
                    break;
                case XMLConstants.MG_ATTR_STUDIO_VALUE:
                    this.StudioValue = XmlParser.unescape(valueStr);
                    break;
                case XMLConstants.MG_ATTR_VALUE:
                    if (this._id === PropInterface.PROP_TYPE_CUSTOM_PROPERTIES) {
                        this.InitCustomProperties(XmlParser.unescape(valueStr));
                    }
                    if (this._id === PropInterface.PROP_TYPE_CUSTOM_VALIDATORS) {
                        this.InitCustomValidationProperties(XmlParser.unescape(valueStr));
                    }
                    else {
                        this._val = XmlParser.unescape(valueStr);
                        this._orgValue = this._val;
                    }
                    break;
                case XMLConstants.MG_ATTR_EXP:
                    this._expId = XmlParser.getInt(valueStr);
                    if (this._expId > 0 && this._id === PropInterface.PROP_TYPE_TOOLTIP)
                        this.setDataType();
                    break;
                case XMLConstants.MG_ATTR_NAME:
                    break;
                case XMLConstants.MG_ATTR_CTL_IDX:
                    this._taskDefinitionId.CtlIndex = XmlParser.getInt(valueStr);
                    break;
                case XMLConstants.MG_ATTR_PROGRAM_ISN:
                    this._taskDefinitionId.ProgramIsn = XmlParser.getInt(valueStr);
                    break;
                default:
                    Events.WriteExceptionToLog(NString.Format("There is no such tag in Property class. Insert case to Property.initElements for {0}", attribute));
                    break;
            }
            if (this.IsComputedOnceOnServer())
                this._expAlreadyComputedOnce = true;
        }
    }
    InitCustomProperties(customProprtiesStr) {
        this.customProperties = new Map(JSON.parse(customProprtiesStr));
        if (this._parentObj.IsFirstRefreshOfProps())
            this.SetFirstValueForCustomProperties();
    }
    InitCustomValidator(parser) {
        let text = parser.ReadToEndOfCurrentElement();
        text = text.substr(text.indexOf(XMLConstants.MG_TAG_CUSTOM_VALIDATOR) + XMLConstants.MG_TAG_CUSTOM_VALIDATOR.length);
        let tokens = XmlParser.getTokens(text, XMLConstants.XML_ATTR_DELIM);
        let c = new CustomValidatorProp();
        let key;
        for (let j = 0; j < tokens.length; j += 2) {
            let attribute = tokens.get_Item(j);
            let valueStr = tokens.get_Item(j + 1);
            switch (attribute) {
                case XMLConstants.MG_ATTR_EXP:
                    c._exp = XmlParser.getInt(valueStr);
                    break;
                case XMLConstants.MG_ATTR_NAME:
                    key = XmlParser.unescape(valueStr);
                    break;
                case XMLConstants.MG_ATTR_ERROR_EXP:
                    c._errorMessageExp = XmlParser.getInt(valueStr);
                    break;
                case XMLConstants.MG_ATTR_ERROR:
                    c._errorMessage = XmlParser.unescape(valueStr);
                    break;
            }
        }
        this.customValidatorProperties.set(key, c);
    }
    InitCustomValidationProperties(customProprtiesStr) {
        let parser = new XmlParser(customProprtiesStr);
        let currTagName = parser.getNextTag();
        while (currTagName != null) {
            switch (currTagName) {
                case XMLConstants.MG_TAG_CUSTOM_VALIDATORS:
                    parser.setCurrIndex2EndOfTag();
                    this.customValidatorProperties = new Map();
                    break;
                case XMLConstants.MG_TAG_CUSTOM_VALIDATOR:
                    this.InitCustomValidator(parser);
                    break;
                default:
                    parser.setCurrIndex2EndOfTag();
                    break;
            }
            currTagName = parser.getNextTag();
        }
        if (this._parentObj.IsFirstRefreshOfProps())
            this.SetFirstValueForCustomValidationProperties();
    }
    SetFirstValueForCustomValidationProperties() {
        for (let customProperty of this.customValidatorProperties.keys()) {
            this.customValidatorValues.set(customProperty, new CustomValidator(true, ""));
        }
        this._val = this.GetArrayValuesForCustomValidationProperties();
        this._orgValue = this._val;
    }
    GetArrayValuesForCustomValidationProperties() {
        let values = Array.from(this.customValidatorValues.values());
        let valuesStr = JSON.stringify(values);
        return valuesStr;
    }
    GetArrayValuesForCustomProperties() {
        let values = Array.from(this.customPropertiesValues.values());
        let valuesStr = JSON.stringify(values);
        return valuesStr;
    }
    SetFirstValueForCustomProperties() {
        for (let customProperty of this.customProperties.keys())
            this.customPropertiesValues.set(customProperty, "");
        this._val = this.GetArrayValuesForCustomProperties();
        this._orgValue = this._val;
    }
    InitTaskDefinitionId(xmlBuffer) {
        let taskDefinitionIdTableSaxHandler = new TaskDefinitionIdTableSaxHandler(this.SetTaskDefinitionId);
        taskDefinitionIdTableSaxHandler.parse(xmlBuffer);
    }
    SetTaskDefinitionId(taskDefinitionId) {
        this._taskDefinitionId = taskDefinitionId;
    }
    setDataType() {
        let compIdx = this._parentObj.getCompIdx();
        this._pic = null;
        if (this._id === Int32.MinValue) {
            Events.WriteExceptionToLog(NString.Format("To fill dataType member in Property.FillDataType must id!={0}", Int32.MinValue));
            return;
        }
        switch (this._id) {
            case PropInterface.PROP_TYPE_TITLE_BAR:
            case PropInterface.PROP_TYPE_DISPLAY_MENU:
            case PropInterface.PROP_TYPE_BORDER:
            case PropInterface.PROP_TYPE_HEBREW:
            case PropInterface.PROP_TYPE_MODIFY_IN_QUERY:
            case PropInterface.PROP_TYPE_SYSTEM_MENU:
            case PropInterface.PROP_TYPE_END_CONDITION:
            case PropInterface.PROP_TYPE_SELECTION:
            case PropInterface.PROP_TYPE_ALLOW_MODIFY:
            case PropInterface.PROP_TYPE_ALLOW_CREATE:
            case PropInterface.PROP_TYPE_ALLOW_DELETE:
            case PropInterface.PROP_TYPE_ALLOW_QUERY:
            case PropInterface.PROP_TYPE_ALLOW_RANGE:
            case PropInterface.PROP_TYPE_ALLOW_LOCATE:
            case PropInterface.PROP_TYPE_ALLOW_SORT:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_ALLOW_INDEX:
            case PropInterface.PROP_TYPE_ALLOW_LOCATE_IN_QUERY:
            case PropInterface.PROP_TYPE_CONFIRM_UPDATE:
            case PropInterface.PROP_TYPE_FORCE_SUFFIX:
            case PropInterface.PROP_TYPE_FORCE_DELETE:
            case PropInterface.PROP_TYPE_MUST_INPUT:
            case PropInterface.PROP_TYPE_MODIFIABLE:
            case PropInterface.PROP_TYPE_VISIBLE:
            case PropInterface.PROP_TYPE_ENABLED:
            case PropInterface.PROP_TYPE_CONFIRM_CANCEL:
            case PropInterface.PROP_TYPE_REPEATABLE:
            case PropInterface.PROP_TYPE_HIGHLIGHTING:
            case PropInterface.PROP_TYPE_PASSWORD:
            case PropInterface.PROP_TYPE_MULTILINE:
            case PropInterface.PROP_TYPE_IS_CACHED:
            case PropInterface.PROP_TYPE_PRELOAD_VIEW:
            case PropInterface.PROP_TYPE_MULTILINE_VERTICAL_SCROLL:
            case PropInterface.PROP_TYPE_MULTILINE_ALLOW_CR:
            case PropInterface.PROP_TYPE_ALLOW_COL_RESIZE:
            case PropInterface.PROP_TYPE_ALLOW_REORDER:
            case PropInterface.PROP_TYPE_SHOW_LINES:
            case PropInterface.PROP_TYPE_SORT_COLUMN:
            case PropInterface.PROP_TYPE_COL_ALLOW_FILTERING:
            case PropInterface.PROP_TYPE_DISPLAY_STATUS_BAR:
            case PropInterface.PROP_TYPE_AUTO_REFRESH:
            case PropInterface.PROP_TYPE_PARK_ON_CLICK:
            case PropInterface.PROP_TYPE_SHOW_FULL_ROW:
            case PropInterface.PROP_TYPE_TRACK_SELECTION:
            case PropInterface.PROP_TYPE_SHOW_BUTTONS:
            case PropInterface.PROP_TYPE_LINES_AT_ROOT:
            case PropInterface.PROP_TYPE_ALLOW_EMPTY_DATAVIEW:
            case PropInterface.PROP_TYPE_SCROLL_BAR:
            case PropInterface.PROP_TYPE_COLUMN_DIVIDER:
            case PropInterface.PROP_TYPE_LINE_DIVIDER:
            case PropInterface.PROP_TYPE_THREE_STATES:
            case PropInterface.PROP_TYPE_REFRESH_WHEN_HIDDEN:
            case PropInterface.PROP_TYPE_ALLOW_OPTION:
            case PropInterface.PROP_TYPE_RETAIN_FOCUS:
            case PropInterface.PROP_TYPE_PRINT_DATA:
            case PropInterface.PROP_TYPE_CLOSE_TASKS_BY_MDI_MENU:
            case PropInterface.PROP_TYPE_DATAVIEWCONTROL:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_OPEN_TASK_WINDOW:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_ALLOW_EVENTS:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_INDEX_OPTIMIZATION:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_RANGE:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_LOCATE:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_POSITION:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_SQL_RANGE:
            case PropInterface.PROP_TYPE_BEFORE_900_VERSION:
            case PropInterface.PROP_TYPE_FILL_WIDTH:
            case PropInterface.PROP_TYPE_MULTI_COLUMN_DISPLAY:
            case PropInterface.PROP_TYPE_IS_ROUTER_OUTLET:
            case PropInterface.PROP_TYPE_DEFAULT_OUTLET:
            case PropInterface.PROP_TYPE_SERVER_READ_AHEAD:
            case PropInterface.PROP_TYPE_AUTO_COMPLETE_MODE_WC:
                this._dataType = StorageAttribute.BOOLEAN;
                break;
            case PropInterface.PROP_TYPE_IMAGE_FILENAME:
            case PropInterface.PROP_TYPE_EVAL_END_CONDITION:
            case PropInterface.PROP_TYPE_FRAME_NAME:
            case PropInterface.PROP_TYPE_NAME:
            case PropInterface.PROP_TYPE_TRASACTION_BEGIN:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_TRANSACTION_MODE:
            case PropInterface.PROP_TYPE_WALLPAPER:
            case PropInterface.PROP_TYPE_ATTRIBUTE:
            case PropInterface.PROP_TYPE_TRIGGER:
            case PropInterface.PROP_TYPE_SELECT_MODE:
            case PropInterface.PROP_TYPE_TASK_ID:
            case PropInterface.PROP_TYPE_TABBING_CYCLE:
            case PropInterface.PROP_TYPE_PARAMETERS:
            case PropInterface.PROP_TYPE_IMAGE_LIST_INDEXES:
            case PropInterface.PROP_TYPE_VISIBLE_LAYERS_LIST:
            case PropInterface.PROP_TYPE_OBJECT_TYPE:
            case PropInterface.PROP_TYPE_REAL_OBJECT_TYPE:
            case PropInterface.PROP_TYPE_DATAVIEWCONTROL_FIELDS:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_RANGE_ORDER:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_LOCATE_ORDER:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_POSITION_USAGE:
            case PropInterface.PROP_TYPE_TASK_MODE:
            case PropInterface.PROP_TYPE_CUSTOM_PROPERTIES:
            case PropInterface.PROP_TYPE_CUSTOM_VALIDATORS:
                this._dataType = StorageAttribute.ALPHA;
                break;
            case PropInterface.PROP_TYPE_COLUMN_TITLE:
            case PropInterface.PROP_TYPE_DISPLAY_LIST:
            case PropInterface.PROP_TYPE_LABEL:
            case PropInterface.PROP_TYPE_FORMAT:
            case PropInterface.PROP_TYPE_RANGE:
            case PropInterface.PROP_TYPE_FORM_NAME:
            case PropInterface.PROP_TYPE_ADDITIONAL_INFORMATION:
            case PropInterface.PROP_TYPE_TEXT:
            case PropInterface.PROP_TYPE_HINT:
                this._dataType = StorageAttribute.UNICODE;
                break;
            case PropInterface.PROP_TYPE_UOM:
            case PropInterface.PROP_TYPE_HOR_FAC:
            case PropInterface.PROP_TYPE_VER_FAC:
            case PropInterface.PROP_TYPE_TAB_ORDER:
            case PropInterface.PROP_TYPE_WALLPAPER_STYLE:
            case PropInterface.PROP_TYPE_INDEX:
            case PropInterface.PROP_TYPE_LINK_FIELD:
            case PropInterface.PROP_TYPE_DISPLAY_FIELD:
            case PropInterface.PROP_TYPE_PULLDOWN_MENU:
            case PropInterface.PROP_TYPE_WINDOW_TYPE:
            case PropInterface.PROP_TYPE_IMAGE_STYLE:
            case PropInterface.PROP_TYPE_AUTO_FIT:
            case PropInterface.PROP_TYPE_RAISE_AT:
            case PropInterface.PROP_TYPE_EXPAND_WINDOW:
            case PropInterface.PROP_TYPE_VISIBLE_LINES:
            case PropInterface.PROP_TYPE_CHOICE_COLUMNS:
            case PropInterface.PROP_TYPE_STYLE_3D:
            case PropInterface.PROP_TYPE_FONT:
            case PropInterface.PROP_TYPE_LAYER:
            case PropInterface.PROP_TYPE_WIDTH:
            case PropInterface.PROP_TYPE_TITLE_HEIGHT:
            case PropInterface.PROP_TYPE_COLOR:
            case PropInterface.PROP_TYPE_LEFT:
            case PropInterface.PROP_TYPE_TOP:
            case PropInterface.PROP_TYPE_TOOLTIP:
            case PropInterface.PROP_TYPE_HELP_SCR:
            case PropInterface.PROP_TYPE_LINES_IN_TABLE:
            case PropInterface.PROP_TYPE_SELECTION_ROWS:
            case PropInterface.PROP_TYPE_HORIZONTAL_ALIGNMENT:
            case PropInterface.PROP_TYPE_MULTILINE_WORDWRAP_SCROLL:
            case PropInterface.PROP_TYPE_ROW_HEIGHT:
            case PropInterface.PROP_TYPE_WINDOW_WIDTH:
            case PropInterface.PROP_TYPE_MAIN_DISPLAY:
            case PropInterface.PROP_TYPE_TRANSLATOR:
            case PropInterface.PROP_TYPE_FRAMESET_STYLE:
            case PropInterface.PROP_TYPE_FRAME_TYPE:
            case PropInterface.PROP_TYPE_ALLOWED_DIRECTION:
            case PropInterface.PROP_TYPE_ROW_HIGHLIGHT_STYLE:
            case PropInterface.PROP_TYPE_VERTICAL_ALIGNMENT:
            case PropInterface.PROP_TYPE_TABBING_ORDER:
            case PropInterface.PROP_TYPE_STATIC_TYPE:
            case PropInterface.PROP_TYPE_BUTTON_STYLE:
            case PropInterface.PROP_TYPE_LOAD_IMAGE_FROM:
            case PropInterface.PROP_TYPE_SUBFORM_TYPE:
            case PropInterface.PROP_TYPE_BOTTOM_POSITION_INTERVAL:
            case PropInterface.PROP_TYPE_SELECTION_MODE:
            case PropInterface.PROP_TYPE_PRGTSK_NUM:
            case PropInterface.PROP_TYPE_PERSISTENT_FORM_STATE_VERSION:
            case PropInterface.PROP_TYPE_SCROLL_BAR_THUMB:
            case PropInterface.PROP_TYPE_BINDING_LEVEL:
            case PropInterface.PROP_TYPE_CONNECTED_ZOOM_CONTROL:
            case PropInterface.PROP_TYPE_SELECT_PROGRAM:
                if (this._id === PropInterface.PROP_TYPE_TOOLTIP && this._expId > 0)
                    this._dataType = StorageAttribute.UNICODE;
                else {
                    this._dataType = StorageAttribute.NUMERIC;
                    if (Property._numericPropertyPic == null)
                        Property._numericPropertyPic = new PIC("N6", StorageAttribute.NUMERIC, compIdx);
                    this._pic = Property._numericPropertyPic;
                }
                break;
            case PropInterface.PROP_TYPE_RETURN_ACTION:
                this._dataType = StorageAttribute.NONE;
                break;
            case PropInterface.PROP_TYPE_DATA:
                this._dataType = this._parentObj.DataType;
                break;
            case PropInterface.PROP_TYPE_POP_UP:
            case PropInterface.PROP_TYPE_ORIENTATION_LOCK:
            case PropInterface.PROP_TYPE_ENTER_ANIMATION:
            case PropInterface.PROP_TYPE_EXIT_ANIMATION:
            case PropInterface.PROP_TYPE_NAVIGATION_DRAWER_MENU:
            case PropInterface.PROP_TYPE_ACTION_BAR_MENU:
            case PropInterface.PROP_TYPE_OPEN_PICKER:
            case PropInterface.PROP_TYPE_OPEN_EDIT_DIALOG:
            case PropInterface.PROP_TYPE_DEFAULT_ALIGNMENT:
            case PropInterface.PROP_TYPE_KEYBOARD_TYPE:
            case PropInterface.PROP_TYPE_KEYBOARD_RETURN_KEY:
            case PropInterface.PROP_TYPE_ALLOW_SUGGESTIONS:
            case PropInterface.PROP_TYPE_MOBILE_IMAGE_LIST_FILE_NAME:
            case PropInterface.PROP_TYPE_SWIPE_REFRESH:
                break;
            default:
                Events.WriteExceptionToLog(NString.Format("in Property.setDataType() no case for: {0}", this._id));
                break;
        }
    }
    setControlAttributes() {
        if (this._parentObj == null || this._parentType !== GuiConstants.PARENT_TYPE_CONTROL) {
            Events.WriteExceptionToLog("in Property.setControlAttributes() there is no parent or the parent is not a control");
            return;
        }
        let parentControl = this._parentObj;
        switch (this._id) {
            case PropInterface.PROP_TYPE_FRAME_NAME:
            case PropInterface.PROP_TYPE_NAME:
                parentControl.Name = this._val;
                break;
            case PropInterface.PROP_TYPE_ATTRIBUTE:
                parentControl.DataType = this._val[0];
                break;
            case PropInterface.PROP_TYPE_FORMAT:
                parentControl.setPicStr(this._val, this._expId);
                break;
            case PropInterface.PROP_TYPE_RANGE:
                parentControl.setRange(this._val);
                break;
            case PropInterface.PROP_TYPE_DATA:
                if (this._expId > 0)
                    parentControl.setValExp(this._expId);
                else if (this._val != null)
                    parentControl.setField(this._val);
                break;
            case PropInterface.PROP_TYPE_BUTTON_STYLE:
                if (this._val != null)
                    parentControl.ButtonStyle = this.GetComputedValueInteger();
                break;
            case PropInterface.PROP_TYPE_LAYER:
                if (this._val != null)
                    parentControl.Layer = this.GetComputedValueInteger();
                break;
            default:
                break;
        }
    }
    getID() {
        return this._id;
    }
    setValue(mgVal) {
        let compIdx = this._parentObj.getCompIdx();
        if (this._id === PropInterface.PROP_TYPE_DATA || this._id === PropInterface.PROP_TYPE_NODE_ID || this._id === PropInterface.PROP_TYPE_NODE_PARENTID)
            return;
        if (this._dataType === StorageAttribute.NONE)
            this.setDataType();
        switch (this._dataType) {
            case StorageAttribute.NUMERIC:
                if (mgVal == null)
                    mgVal = "FF00000000000000000000000000000000000000";
                this._val = DisplayConvertor.Instance.mg2disp(mgVal, "", this._pic, false, compIdx, false).trim();
                break;
            case StorageAttribute.ALPHA:
            case StorageAttribute.BLOB:
            case StorageAttribute.BLOB_VECTOR:
            case StorageAttribute.BOOLEAN:
            case StorageAttribute.UNICODE:
                this._val = mgVal;
                break;
            default:
                throw new ApplicationException("in Property.setValue() illegal data type: " + this._dataType);
        }
    }
    SetValue(val) {
        this._val = String(val);
    }
    setOrgValue() {
        this._orgValue = this._val;
    }
    GetTaskByParentObject() {
        let task = null;
        if (this._parentType === GuiConstants.PARENT_TYPE_FORM)
            task = this._parentObj.getTask();
        else if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL)
            task = this._parentObj.getForm().getTask();
        else if (this._parentType === GuiConstants.PARENT_TYPE_TASK)
            task = this._parentObj;
        return task;
    }
    async RefreshDisplay(forceRefresh) {
        await this.RefreshDisplayWithCurrentLine(forceRefresh, Int32.MinValue);
    }
    ShouldSkipRefreshControl() {
        let skip = false;
        let mgControl = this._parentObj;
        switch (this._id) {
            case PropInterface.PROP_TYPE_HEBREW:
                if (mgControl.isTableControl())
                    skip = true;
                break;
            case PropInterface.PROP_TYPE_NODE_ID:
            case PropInterface.PROP_TYPE_NODE_PARENTID:
            case PropInterface.PROP_TYPE_DATA:
            case PropInterface.PROP_TYPE_HIGHLIGHTING:
            case PropInterface.PROP_TYPE_MUST_INPUT:
            case PropInterface.PROP_TYPE_HELP_SCR:
            case PropInterface.PROP_TYPE_SELECT_PROGRAM:
            case PropInterface.PROP_TYPE_SELECT_MODE:
            case PropInterface.PROP_TYPE_ALLOWED_DIRECTION:
            case PropInterface.PROP_TYPE_RETURN_ACTION:
            case PropInterface.PROP_TYPE_ATTRIBUTE:
            case PropInterface.PROP_TYPE_REPEATABLE:
            case PropInterface.PROP_TYPE_FRAME_NAME:
            case PropInterface.PROP_TYPE_TRIGGER:
            case PropInterface.PROP_TYPE_TAB_ORDER:
            case PropInterface.PROP_TYPE_IS_CACHED:
            case PropInterface.PROP_TYPE_LAYER:
            case PropInterface.PROP_TYPE_AUTO_FIT:
            case PropInterface.PROP_TYPE_RAISE_AT:
            case PropInterface.PROP_TYPE_FRAMESET_STYLE:
            case PropInterface.PROP_TYPE_FRAME_TYPE:
            case PropInterface.PROP_TYPE_RANGE:
            case PropInterface.PROP_TYPE_WINDOW_WIDTH:
            case PropInterface.PROP_TYPE_TABBING_ORDER:
            case PropInterface.PROP_TYPE_AUTO_REFRESH:
            case PropInterface.PROP_TYPE_PARAMETERS:
            case PropInterface.PROP_TYPE_OBJECT_TYPE:
            case PropInterface.PROP_TYPE_RETAIN_FOCUS:
            case PropInterface.PROP_TYPE_SUBFORM_TYPE:
            case PropInterface.PROP_TYPE_REAL_OBJECT_TYPE:
            case PropInterface.PROP_TYPE_ADDITIONAL_INFORMATION:
            case PropInterface.PROP_TYPE_DATAVIEWCONTROL:
            case PropInterface.PROP_TYPE_DATAVIEWCONTROL_FIELDS:
            case PropInterface.PROP_TYPE_PRGTSK_NUM:
            case PropInterface.PROP_TYPE_INDEX:
            case PropInterface.PROP_TYPE_DISPLAY_FIELD:
            case PropInterface.PROP_TYPE_LINK_FIELD:
            case PropInterface.PROP_TYPE_BEFORE_900_VERSION:
            case PropInterface.PROP_TYPE_REFRESH_WHEN_HIDDEN:
            case PropInterface.PROP_TYPE_SERVER_READ_AHEAD:
                skip = true;
                break;
            case PropInterface.PROP_TYPE_BUTTON_STYLE:
                if (!this._parentObj.isRadio())
                    skip = true;
                break;
            default:
                break;
        }
        return skip;
    }
    ShouldSkipRefreshForm() {
        Debug.Assert(this._parentType === GuiConstants.PARENT_TYPE_FORM);
        let skip = false;
        let mgFormBase = this._parentObj;
        switch (this._id) {
            case PropInterface.PROP_TYPE_WINDOW_TYPE:
                if (!this._parentObj.IsFirstRefreshOfProps())
                    skip = true;
                break;
            case PropInterface.PROP_TYPE_LINES_IN_TABLE:
            case PropInterface.PROP_TYPE_HELP_SCR:
            case PropInterface.PROP_TYPE_TABBING_ORDER:
            case PropInterface.PROP_TYPE_PERSISTENT_FORM_STATE_VERSION:
                skip = true;
                break;
            default:
                break;
        }
        if (!skip && mgFormBase.isSubForm()) {
            skip = true;
            switch (this._id) {
                case PropInterface.PROP_TYPE_CUSTOM_PROPERTIES:
                case PropInterface.PROP_TYPE_CUSTOM_VALIDATORS:
                    skip = false;
                    break;
            }
        }
        return skip;
    }
    ShouldSkipRefreshTask() {
        let skip = false;
        switch (this._id) {
            case PropInterface.PROP_TYPE_ALLOW_CREATE:
            case PropInterface.PROP_TYPE_ALLOW_DELETE:
            case PropInterface.PROP_TYPE_ALLOW_MODIFY:
            case PropInterface.PROP_TYPE_ALLOW_QUERY:
            case PropInterface.PROP_TYPE_ALLOW_RANGE:
            case PropInterface.PROP_TYPE_ALLOW_LOCATE:
            case PropInterface.PROP_TYPE_ALLOW_SORT:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_ALLOW_INDEX:
            case PropInterface.PROP_TYPE_TABBING_CYCLE:
            case PropInterface.PROP_TYPE_CONFIRM_CANCEL:
            case PropInterface.PROP_TYPE_CONFIRM_UPDATE:
            case PropInterface.PROP_TYPE_END_CONDITION:
            case PropInterface.PROP_TYPE_EVAL_END_CONDITION:
            case PropInterface.PROP_TYPE_FORCE_SUFFIX:
            case PropInterface.PROP_TYPE_FORCE_DELETE:
            case PropInterface.PROP_TYPE_TASK_MODE:
            case PropInterface.PROP_TYPE_SELECTION:
            case PropInterface.PROP_TYPE_TRASACTION_BEGIN:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_TRANSACTION_MODE:
            case PropInterface.PROP_TYPE_PRINT_DATA:
            case PropInterface.PROP_TYPE_CLOSE_TASKS_BY_MDI_MENU:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_OPEN_TASK_WINDOW:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_ALLOW_EVENTS:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_CHUNK_SIZE:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_INDEX_OPTIMIZATION:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_RANGE:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_RANGE_ORDER:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_LOCATE:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_LOCATE_ORDER:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_POSITION:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_POSITION_USAGE:
            case PropInterface.PROP_TYPE_TASK_PROPERTIES_SQL_RANGE:
                skip = true;
                break;
            default:
                break;
        }
        return skip;
    }
    ShouldSkipRefresh() {
        let skip = false;
        switch (this._parentType) {
            case GuiConstants.PARENT_TYPE_CONTROL:
                skip = this.ShouldSkipRefreshControl();
                break;
            case GuiConstants.PARENT_TYPE_FORM:
                skip = this.ShouldSkipRefreshForm();
                break;
            case GuiConstants.PARENT_TYPE_TASK:
                skip = this.ShouldSkipRefreshTask();
                break;
            default:
                break;
        }
        switch (this._id) {
            case PropInterface.PROP_TYPE_POP_UP:
            case PropInterface.PROP_TYPE_ORIENTATION_LOCK:
            case PropInterface.PROP_TYPE_ENTER_ANIMATION:
            case PropInterface.PROP_TYPE_EXIT_ANIMATION:
            case PropInterface.PROP_TYPE_NAVIGATION_DRAWER_MENU:
            case PropInterface.PROP_TYPE_ACTION_BAR_MENU:
            case PropInterface.PROP_TYPE_OPEN_PICKER:
            case PropInterface.PROP_TYPE_OPEN_EDIT_DIALOG:
            case PropInterface.PROP_TYPE_DEFAULT_ALIGNMENT:
            case PropInterface.PROP_TYPE_KEYBOARD_TYPE:
            case PropInterface.PROP_TYPE_KEYBOARD_RETURN_KEY:
            case PropInterface.PROP_TYPE_ALLOW_SUGGESTIONS:
            case PropInterface.PROP_TYPE_MOBILE_IMAGE_LIST_FILE_NAME:
                skip = true;
                break;
        }
        return skip;
    }
    async RefreshDisplayWithCurrentLine(forceRefresh, currLine) {
        await this.RefreshDisplayWithCurrLineAndCheckSkipRefresh(forceRefresh, currLine, true);
    }
    SkipWhenNoExpression() {
        let skipWhenNoExpression = true;
        let ctrl = ((this._parentObj instanceof GuiControlPropertyAdapter) ? this._parentObj : null);
        if ((ctrl != null && ctrl.isButton() && this._id === PropInterface.PROP_TYPE_FORMAT) ||
            (ctrl != null && ctrl.isComboBox() && this._id === PropInterface.PROP_TYPE_LABEL) ||
            (ctrl != null && ctrl.isListBox() && this._id === PropInterface.PROP_TYPE_LABEL) ||
            (ctrl != null && ctrl.isRadio() && this._id === PropInterface.PROP_TYPE_LABEL) ||
            (ctrl != null && ctrl.isCheckBox() && this._id === PropInterface.PROP_TYPE_LABEL) ||
            (ctrl != null && ctrl.isTabControl() && this._id === PropInterface.PROP_TYPE_LABEL) ||
            (ctrl != null && ctrl.isImageControl() && this._id === PropInterface.PROP_TYPE_IMAGE_FILENAME) ||
            (ctrl != null && ctrl.IsImageButton() && this._id === PropInterface.PROP_TYPE_IMAGE_FILENAME) ||
            (ctrl != null && ctrl.isColumnControl() && this._id === PropInterface.PROP_TYPE_COLUMN_TITLE) ||
            (this._id === PropInterface.PROP_TYPE_DISPLAY_LIST) ||
            (this._id === PropInterface.PROP_TYPE_TEXT) ||
            (this._id === PropInterface.PROP_TYPE_TOOLTIP) ||
            (this._id === PropInterface.PROP_TYPE_HINT) ||
            (this._id === PropInterface.PROP_TYPE_PASSWORD) ||
            (this._id === PropInterface.PROP_TYPE_CUSTOM_PROPERTIES) ||
            (this._id === PropInterface.PROP_TYPE_CUSTOM_VALIDATORS) ||
            (this._id === PropInterface.PROP_TYPE_VISIBLE) ||
            (this._id === PropInterface.PROP_TYPE_ENABLED) ||
            (this._id === PropInterface.PROP_TYPE_BINDING_LEVEL))
            skipWhenNoExpression = false;
        return skipWhenNoExpression;
    }
    async RefreshDisplayWithCurrLineAndCheckSkipRefresh(forceRefresh, currLine, checkSkipRefresh) {
        let ctrl = null;
        let line = 0;
        let prevValue;
        let valChanged = true;
        if (this._val === null && this._expId === 0) {
            Events.WriteExceptionToLog("in Property.RefreshDisplay() null value and expression");
            return;
        }
        if (checkSkipRefresh && this.ShouldSkipRefresh())
            return;
        if (this._expId === 0 && this.SkipWhenNoExpression())
            return;
        if (this._prevValues.length === 0)
            this.setPrevArraySize();
        await this.ComputeValue();
        switch (this._parentType) {
            case GuiConstants.PARENT_TYPE_CONTROL:
                ctrl = this._parentObj;
                if (this._id === PropInterface.PROP_TYPE_FORMAT && this._expId > 0)
                    ctrl.computePIC(this._val);
                if (ctrl.IsRepeatable && !ctrl.getForm().isRefreshRepeatableAllowed())
                    return;
                if (currLine === Int32.MinValue)
                    line = this.getLine();
                else
                    line = currLine;
                prevValue = this._prevValues.get_Item(line);
                if (!forceRefresh && (this._val === prevValue)
                    && this._id !== PropInterface.PROP_TYPE_MODIFIABLE
                    && this._id !== PropInterface.PROP_TYPE_VISIBLE
                    && this._id !== PropInterface.PROP_TYPE_ENABLED)
                    return;
                if (ctrl.Type === MgControlType.CTRL_TYPE_COLUMN
                    && this._id === PropInterface.PROP_TYPE_VISIBLE
                    && this._val === prevValue)
                    return;
                if (this._val === prevValue)
                    valChanged = false;
                this._prevValues.set_Item(line, this._val);
                break;
            case GuiConstants.PARENT_TYPE_FORM:
                prevValue = this._prevValues.get_Item(0);
                if (!forceRefresh && (this._val === prevValue))
                    return;
                this._prevValues.set_Item(0, this._val);
                break;
            case GuiConstants.PARENT_TYPE_TASK:
                Events.WriteExceptionToLog(NString.Format("Property.RefreshDisplay(): task property {0} wasn't handled", this._id));
                return;
            default:
                Events.WriteExceptionToLog(NString.Format("Property.RefreshDisplay(): parentType unknown, property {0} wasn't handled", this._id));
                return;
        }
        switch (this._id) {
            case PropInterface.PROP_TYPE_IMAGE_FILENAME:
                this.onImageFileName();
                break;
            case PropInterface.PROP_TYPE_ENABLED:
                this.onEnable(valChanged);
                break;
            case PropInterface.PROP_TYPE_VISIBLE:
                await this.onVisible(valChanged);
                break;
            case PropInterface.PROP_TYPE_NAME:
                this.OnControlName();
                break;
            case PropInterface.PROP_TYPE_FORM_NAME:
                this.onFormName();
                break;
            case PropInterface.PROP_TYPE_COLUMN_TITLE:
                this.onText(0);
                break;
            case PropInterface.PROP_TYPE_TEXT:
                this.onText(line);
                break;
            case PropInterface.PROP_TYPE_HINT:
                this.onHint();
                break;
            case PropInterface.PROP_TYPE_MODIFIABLE:
                await this.onModifiable();
                break;
            case PropInterface.PROP_TYPE_TOOLTIP:
                this.onTooltip();
                break;
            case PropInterface.PROP_TYPE_DISPLAY_LIST:
                await this.onDisplayList(line);
                break;
            case PropInterface.PROP_TYPE_AUTO_COMPLETE_MODE_WC:
                this.onAutoCompleteMode();
                break;
            case PropInterface.PROP_TYPE_VISIBLE_LAYERS_LIST:
                await this.onVisibleLayerList(line);
                break;
            case PropInterface.PROP_TYPE_FORMAT:
                this.onFormat();
                break;
            case PropInterface.PROP_TYPE_LABEL:
                await this.onLabel(line);
                break;
            case PropInterface.PROP_TYPE_PASSWORD:
                this.onPassword();
                break;
            case PropInterface.PROP_TYPE_MODIFY_IN_QUERY:
            case PropInterface.PROP_TYPE_TAB_IN:
            case PropInterface.PROP_TYPE_PARK_ON_CLICK:
            case PropInterface.PROP_TYPE_LOAD_IMAGE_FROM:
                break;
            case PropInterface.PROP_TYPE_CUSTOM_PROPERTIES:
                this.onCustomProperties();
                break;
            case PropInterface.PROP_TYPE_CUSTOM_VALIDATORS:
                this.onCustomValidators();
                break;
            case PropInterface.PROP_TYPE_BINDING_LEVEL:
                this.onBindingLevel();
                break;
            default:
                Events.WriteExceptionToLog(NString.Format("Property.RefreshDisplay(): Property {0} wasn't handled", this._id));
                break;
        }
    }
    ShouldBeComputedOnce() {
        if (this._parentType === GuiConstants.PARENT_TYPE_TASK) {
            switch (this._id) {
                case PropInterface.PROP_TYPE_TASK_MODE:
                case PropInterface.PROP_TYPE_PRELOAD_VIEW:
                case PropInterface.PROP_TYPE_TABBING_CYCLE:
                case PropInterface.PROP_TYPE_TASK_PROPERTIES_OPEN_TASK_WINDOW:
                case PropInterface.PROP_TYPE_TASK_PROPERTIES_ALLOW_EVENTS:
                case PropInterface.PROP_TYPE_ALLOW_LOCATE_IN_QUERY:
                case PropInterface.PROP_TYPE_PRINT_DATA:
                case PropInterface.PROP_TYPE_ALLOW_RANGE:
                case PropInterface.PROP_TYPE_ALLOW_LOCATE:
                case PropInterface.PROP_TYPE_ALLOW_SORT:
                case PropInterface.PROP_TYPE_TASK_PROPERTIES_ALLOW_INDEX:
                case PropInterface.PROP_TYPE_MAIN_DISPLAY:
                    return true;
            }
        }
        else if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
            if (this._id === PropInterface.PROP_TYPE_IS_CACHED || this._id === PropInterface.PROP_TYPE_MULTILINE)
                return true;
        }
        return false;
    }
    IsComputedOnceOnServer() {
        if (this.ShouldBeComputedOnce()) {
            let task = this.GetTaskByParentObject();
            if (task != null)
                return !task.ShouldEvaluatePropertyLocally(this._id);
        }
        return false;
    }
    async ComputeValue() {
        let result = null;
        let len = 255;
        let wasEvaluated = new RefParam(false);
        if (this._id === PropInterface.PROP_TYPE_CUSTOM_PROPERTIES) {
            await this.computeCustomPropertiesValues();
            result = this.GetArrayValuesForCustomProperties();
            this.setValue(result);
        }
        if (this._id === PropInterface.PROP_TYPE_CUSTOM_VALIDATORS) {
            let evaluated = await this.computeCustomValidators();
            if (evaluated) {
                result = this.GetArrayValuesForCustomValidationProperties();
                this.setValue(result);
            }
        }
        else if (this._expId > 0) {
            if (this.ShouldBeComputedOnce() && this._expAlreadyComputedOnce)
                return;
            this._expAlreadyComputedOnce = true;
            let alwaysEvaluate = false;
            if (this._parentType === GuiConstants.PARENT_TYPE_TASK)
                alwaysEvaluate = true;
            if (this._id === PropInterface.PROP_TYPE_FORMAT) {
                len = 65535;
                result = StrUtil.rtrim(await this._parentObj.EvaluateExpression(this._expId, this._dataType, len, true, StorageAttribute.SKIP, alwaysEvaluate, wasEvaluated));
            }
            else {
                try {
                    result = await this._parentObj.EvaluateExpression(this._expId, this._dataType, len, true, StorageAttribute.SKIP, alwaysEvaluate, wasEvaluated);
                }
                catch (e) {
                    let warningMsg = new StringBuilder("Exception: " + e.Message);
                    if (this._parentObj instanceof GuiControlPropertyAdapter)
                        warningMsg.Append(" Control: " + this._parentObj.Name);
                    Events.WriteWarningToLog(warningMsg.ToString());
                }
            }
            if (wasEvaluated.value) {
                result = this.updateResult(result);
                this.setValue(result);
            }
            else if (this._val == null && this._parentType !== GuiConstants.PARENT_TYPE_CONTROL) {
                result = PropDefaults.getDefaultValue(this._id, this._parentType, this._parentObj);
                this.setValue(result);
            }
        }
    }
    updateResult(result) {
        switch (this._id) {
            case PropInterface.PROP_TYPE_FORMAT:
                if (result == null)
                    result = NString.Empty;
                break;
            case PropInterface.PROP_TYPE_TASK_MODE:
                result = Property.updateTaskModeResult(result);
                break;
            default:
                return result;
        }
        return result;
    }
    static updateTaskModeResult(result) {
        let code = ' ';
        if (!NString.IsNullOrEmpty(result)) {
            code = result[0].toUpperCase();
            switch (code) {
                case 'Q':
                    code = 'E';
                    break;
                case 'F':
                    code = 'O';
                    break;
                case 'O':
                    code = 'N';
                    break;
            }
        }
        return code;
    }
    async getValue() {
        await this.ComputeValue();
        return this._val;
    }
    async getValueInt() {
        let result = 0;
        await this.ComputeValue();
        if (this._val !== null)
            result = NNumber.Parse(this._val);
        return result;
    }
    GetComputedValueInteger() {
        return NNumber.Parse(this._val);
    }
    async getValueBoolean() {
        if (this._dataType === StorageAttribute.BOOLEAN) {
            await this.getValue();
            return DisplayConvertor.toBoolean(this._val);
        }
        else {
            Events.WriteExceptionToLog(NString.Format("Property.getValueBoolean() was called for non boolean type property: {0}", this._id));
        }
    }
    GetComputedValue() {
        return this._val;
    }
    GetComputedValueBoolean() {
        if (this._dataType === StorageAttribute.BOOLEAN) {
            return DisplayConvertor.toBoolean(this._val);
        }
        else {
            Events.WriteExceptionToLog(NString.Format("Property.getValueBoolean() was called for non boolean type property: {0}", this._id));
        }
    }
    getPrevValue(line) {
        if (line < this._prevValues.length)
            return this._prevValues.get_Item(line);
        return null;
    }
    isExpression() {
        return this._expId > 0;
    }
    GetExpressionId() {
        return this._expId;
    }
    onImageFileName() {
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
            let mgControlBase = this._parentObj;
            if (mgControlBase.isTabControl()) {
                mgControlBase.setImageList(this.GetComputedValue());
            }
            else if (mgControlBase.isImageControl() || mgControlBase.isRadio() || mgControlBase.isCheckBox() || mgControlBase.IsImageButton()) {
                mgControlBase.setImageWithFileName(this.GetComputedValue());
            }
        }
        else
            throw new ApplicationException("in onImageFileName.onEnable()");
    }
    onEnable(valChanged) {
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
            let ctrl = this._parentObj;
            let currentEditingControl = ctrl.getForm().getTask().CurrentEditingControl;
            if (this._val !== null && this._val === "0" && valChanged && currentEditingControl !== null &&
                (ctrl === currentEditingControl || currentEditingControl.isDescendentOfControl(ctrl) || ctrl === currentEditingControl.getLinkedParent(false))) {
                let parkedControl = ctrl.getForm().getTask().getLastParkedCtrl();
                if (parkedControl !== null && parkedControl.InControl && (ctrl === parkedControl || parkedControl.isDescendentOfControl(ctrl))) {
                    Events.OnNonParkableLastParkedCtrl(ctrl);
                    ctrl.getForm().getTask().CurrentEditingControl = null;
                }
            }
            ctrl.SetEnabled(this.GetComputedValueBoolean());
        }
        else
            throw new ApplicationException("in Property.onEnable()");
    }
    async onVisible(valChanged) {
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
            let ctrl = this._parentObj;
            let currentEditingControl = ctrl.getForm().getTask().CurrentEditingControl;
            if (this._val !== null && this._val === "0" && valChanged && currentEditingControl !== null &&
                (ctrl === currentEditingControl || currentEditingControl.isDescendentOfControl(ctrl) || ctrl === currentEditingControl.getLinkedParent(false))) {
                let parkedControl = ctrl.getForm().getTask().getLastParkedCtrl();
                if (parkedControl !== null && parkedControl.InControl && (ctrl === parkedControl || parkedControl.isDescendentOfControl(ctrl))) {
                    Events.OnNonParkableLastParkedCtrl(ctrl);
                    ctrl.getForm().getTask().CurrentEditingControl = null;
                }
            }
            await ctrl.updatePropertyLogicNesting(PropInterface.PROP_TYPE_VISIBLE, CommandType.PROP_SET_VISIBLE, this.GetComputedValueBoolean(), true);
        }
    }
    OnControlName() {
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
        }
        else
            throw new ApplicationException("in Property.OnControlName()");
    }
    onHint() {
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
            let control = this.getObjectByParentObj();
            let translatedString = Events.Translate(this.GetComputedValue());
            if (!control.IsDateTimePicture() && (!control.IsTableChild || control.IsTableHeaderChild)) {
                Commands.addOperationWithLine(CommandType.SET_PROPERTY, this.getObjectByParentObj(), this.getLine(), HtmlProperties.PlaceHolder, translatedString);
            }
        }
        else
            throw new ApplicationException("inProperty.onHint()");
    }
    onFormName() {
        if (this._parentType === GuiConstants.PARENT_TYPE_FORM)
            this.addCommandTypeText(0);
        else
            throw new ApplicationException("in Property.onFormName()");
    }
    addCommandTypeText(line) {
        let mlsTransValue = Events.Translate(this.GetComputedValue());
        if (this._parentType === GuiConstants.PARENT_TYPE_FORM ||
            (this._parentType === GuiConstants.PARENT_TYPE_CONTROL && this._parentObj.isColumnControl())) {
            mlsTransValue = StrUtil.rtrim(mlsTransValue);
            if (this._parentType === GuiConstants.PARENT_TYPE_FORM && mlsTransValue === NString.Empty) {
                mlsTransValue = " ";
            }
        }
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL && (this._parentObj.isTextControl()
            || (this._parentObj.isButton())))
            Commands.addValueWithLine(CommandType.SET_VALUE, this._parentObj, line, mlsTransValue);
        else
            Commands.addOperationWithLine(CommandType.SET_PROPERTY, this._parentObj, line, HtmlProperties.Text, mlsTransValue);
    }
    onText(line) {
        this.addCommandTypeText(line);
    }
    async onModifiable() {
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
            let ctrl = this._parentObj;
            if (ctrl.isTextControl() || ctrl.isChoiceControl() || ctrl.isCheckBox()) {
                let task = this.GetTaskByParentObject();
                let readOnlyValue = !this.GetComputedValueBoolean();
                if ((readOnlyValue || task.getMode() !== Constants.TASK_MODE_QUERY) && ctrl.GetCurrReadOnly() !== readOnlyValue) {
                    ctrl.SetCurrReadOnly(readOnlyValue);
                    if (UtilStrByteMode.isLocaleDefLangDBCS()) {
                        if (await ctrl.getForm().getTask().checkProp(PropInterface.PROP_TYPE_ALLOW_LOCATE_IN_QUERY, false)) {
                            return;
                        }
                    }
                    Commands.addOperationWithLine(CommandType.SET_PROPERTY, this.getObjectByParentObj(), this.getLine(), HtmlProperties.ReadOnly, readOnlyValue);
                }
            }
        }
        else
            throw new ApplicationException("in Property.onModifiable()");
    }
    onTooltip() {
        let toolTip;
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
            let ctrl = this._parentObj;
            if (this._expId === 0)
                toolTip = ctrl.getForm().getTask().getHelpItem(NNumber.Parse(this._val)).tooltipHelpText;
            else
                toolTip = this._val;
            toolTip = StrUtil.rtrim(toolTip);
            if (ctrl.isButton() && ctrl.ConnectedControl)
                Commands.addOperationWithLine(CommandType.SET_PROPERTY, ctrl.ConnectedControl, this.getLine(), HtmlProperties.ZoomButtonTooltip, toolTip);
            else
                Commands.addOperationWithLine(CommandType.SET_PROPERTY, ctrl, this.getLine(), HtmlProperties.Tooltip, toolTip);
        }
        else
            throw new ApplicationException("in Property.onTooltip()");
    }
    async onDisplayList(currLine) {
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
            let ctrl = this._parentObj;
            if (ctrl.isSelectionCtrl() || ctrl.isTabControl() || ctrl.isRadio() || ctrl.HasAutocomplete()) {
                let line = (currLine === Int32.MinValue) ? ctrl.getDisplayLine(false) : currLine;
                await ctrl.refreshAndSetItemsList(line, true);
            }
        }
        else
            throw new ApplicationException("in Property.onDisplayList()");
    }
    async onVisibleLayerList(currLine) {
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
            let ctrl = this._parentObj;
            let line = 0;
            if (currLine === Int32.MinValue)
                line = ctrl.getDisplayLine(false);
            else
                line = currLine;
            await ctrl.refreshTabForLayerList(line);
        }
        else
            throw new ApplicationException("in Property.onVisibleLayerList()");
    }
    onFormat() {
        let ctrl = null;
        let ctrlPic;
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
            ctrl = this._parentObj;
            ctrlPic = ctrl.getPIC();
            ctrl.resetPrevVal();
            switch (ctrl.Type) {
                case MgControlType.CTRL_TYPE_BUTTON:
                    if (ctrl.getField() == null && !ctrl.expressionSetAsData() || ctrl.IsImageButton() || (this._parentObj.IsFirstRefreshOfProps()))
                        this.addCommandTypeText(this.getLine());
                    break;
                case MgControlType.CTRL_TYPE_TEXT:
                    if (ctrl.Type === MgControlType.CTRL_TYPE_TEXT) {
                        if (ctrlPic.getAttr() !== StorageAttribute.NUMERIC) {
                        }
                    }
                    break;
            }
        }
        else
            throw new ApplicationException("in Property.onFormat()");
    }
    async onLabel(currLine) {
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
            let ctrl = this._parentObj;
            let line;
            if (currLine === Int32.MinValue)
                line = ctrl.getDisplayLine(false);
            else
                line = currLine;
            if (ctrl.Type === MgControlType.CTRL_TYPE_CHECKBOX)
                this.addCommandTypeText(line);
            else if (ctrl.isSelectionCtrl() || ctrl.isTabControl() || ctrl.isRadio()) {
                ctrl.setRange(await this.getValue());
                ctrl.clearRange(line);
                await ctrl.refreshAndSetItemsList(line, true);
                if (this._expId > 0)
                    await ctrl.ComputeAndRefreshDisplayValue(false);
            }
            else if (ctrl.HasAutocomplete())
                await ctrl.refreshAndSetItemsList(line, true);
            else
                throw new ApplicationException("Property.onLabel(), not support control");
        }
        else
            throw new ApplicationException("in Property.onLabel()");
    }
    getObjectByParentObj() {
        let result = this._parentObj;
        if (this._parentType === GuiConstants.PARENT_TYPE_FORM) {
            let form = this._parentObj;
            if (form.isSubForm())
                result = form.getSubFormCtrl();
        }
        return result;
    }
    static isRepeatableInTable() {
        return false;
    }
    getLine() {
        let line = 0;
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
            let ctrl = this._parentObj;
            let displayLineForTreeOrTable = ctrl.isPropertyRepeatable(this._id);
            line = ctrl.getDisplayLine(displayLineForTreeOrTable);
        }
        return line;
    }
    getOrgValue() {
        return this._orgValue;
    }
    updatePrevValueArray(newSize) {
        if (!this.ShouldSkipRefresh())
            this._prevValues.SetSize(newSize);
    }
    clearPrevValueArray() {
        this._prevValues.Clear();
    }
    ResetPrevValueArray() {
        let count = this._prevValues.length;
        this._prevValues.Clear();
        this._prevValues.SetSize(count);
    }
    setPrevArraySize() {
        if (this._parentType === GuiConstants.PARENT_TYPE_FORM || this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
            let size = 1;
            if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL) {
                let mgControlBase = this._parentObj;
                if (mgControlBase.IsRepeatable)
                    size = mgControlBase.getForm().getTableItemsCount();
                else if (mgControlBase.isTableControl() && Property.isRepeatableInTable())
                    size = mgControlBase.getForm().getTableItemsCount();
            }
            this._prevValues.SetSize(size);
        }
    }
    onPassword() {
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL)
            Commands.addOperationWithLine(CommandType.SET_PROPERTY, this.getObjectByParentObj(), this.getLine(), HtmlProperties.Password, this.GetComputedValueBoolean());
        else
            throw new ApplicationException("in Property.onPassword()");
    }
    async computeCustomPropertiesValues() {
        for (let [customPropertyKey, customPropertyValue] of this.customProperties) {
            let expression = customPropertyValue;
            let retExpressionValue = await this.GetRetrunValueForCustomPropertyExpression(expression);
            this.customPropertiesValues.set(customPropertyKey, retExpressionValue);
        }
    }
    async computeCustomValidators() {
        let mgControl = this._parentObj;
        let evaluated = true;
        if (mgControl.ModifiedByUser || mgControl.ValidateControl || this.GetTaskByParentObject().IsInRecompute()) {
            for (let [customPropertyKey, customValidatorProperty] of this.customValidatorProperties) {
                let expression = customValidatorProperty._exp;
                let isInValid = await this.GetRetrunValueForCustomPropertyExpression(expression);
                let errString = "";
                let InValid = (isInValid === "1") ? true : false;
                if (InValid) {
                    expression = customValidatorProperty._errorMessageExp;
                    if (expression > 0)
                        errString = await this.GetRetrunValueForCustomPropertyExpression(expression);
                    else
                        errString = customValidatorProperty._errorMessage;
                }
                evaluated = true;
                this.customValidatorValues.set(customPropertyKey, new CustomValidator(!InValid, errString));
            }
        }
        else {
            if (mgControl.isRepeatable()) {
                this.customValidatorValues.clear();
                evaluated = false;
            }
        }
        return evaluated;
    }
    onCustomProperties() {
        for (let [customPropertyKey, customPropertyValue] of this.customPropertiesValues) {
            Commands.addOperationWithLine(CommandType.PROP_SET_USER_PROPERTY, this._parentObj, this.getLine(), customPropertyKey, customPropertyValue);
        }
    }
    onCustomValidators() {
        for (let [customPropertyKey, customPropertyValue] of this.customValidatorValues) {
            Commands.addOperationWithLine(CommandType.PROP_SET_CUSTOM_VALIDATOR, this._parentObj, this.getLine(), customPropertyKey, customPropertyValue);
        }
    }
    onBindingLevel() {
        if (this._parentType === GuiConstants.PARENT_TYPE_CONTROL)
            Commands.addOperationWithLine(CommandType.SET_PROPERTY, this.getObjectByParentObj(), this.getLine(), HtmlProperties.BindingLevel, this.GetComputedValueInteger());
        else
            throw new ApplicationException("in Property.BindingLevel()");
    }
    onAutoCompleteMode() {
    }
    async GetRetrunValueForCustomPropertyExpression(expression) {
        let len = 255;
        let expStorageAttribute = this._parentObj.GetExpressionStorage(expression);
        let wasEvaluated = new RefParam(false);
        let returnValue = await this._parentObj.EvaluateExpression(expression, expStorageAttribute, len, true, StorageAttribute.SKIP, false, wasEvaluated);
        expStorageAttribute = this._parentObj.GetExpressionStorage(expression);
        let compIdx = this._parentObj.getCompIdx();
        switch (expStorageAttribute) {
            case StorageAttribute.NUMERIC:
                let tempPic = new PIC("N6.2", StorageAttribute.NUMERIC, this._parentObj.getCompIdx());
                if (returnValue == null)
                    returnValue = "FF00000000000000000000000000000000000000";
                returnValue = DisplayConvertor.Instance.mg2disp(returnValue, "", tempPic, false, compIdx, false).trim();
                break;
            case StorageAttribute.DATE:
                tempPic = new PIC("##/##/####", expStorageAttribute, compIdx);
                returnValue = DisplayConvertor.Instance.mg2disp(returnValue, "", tempPic, true, false, compIdx, true);
                break;
            case StorageAttribute.TIME:
                tempPic = new PIC("##:##/####", expStorageAttribute, compIdx);
                returnValue = DisplayConvertor.Instance.mg2disp(returnValue, "", tempPic, true, false, compIdx, true);
                break;
            case StorageAttribute.ALPHA:
            case StorageAttribute.BOOLEAN:
            case StorageAttribute.BLOB:
            case StorageAttribute.BLOB_VECTOR:
            case StorageAttribute.UNICODE:
            case StorageAttribute.NONE:
                break;
            default:
                throw new ApplicationException("in Property.GetRetrunValueForCustomPropertyExpression() illegal data type: " + expStorageAttribute);
        }
        return returnValue;
    }
    RemovePrevValIndexAt(idx) {
        if (this._prevValues.length > idx) {
            this._prevValues.RemoveAt(idx);
        }
    }
    InsertPrevValAt(idx) {
        this._prevValues.Insert(idx, null);
    }
    resetPrevValAt(idx) {
        if (idx >= 0 && idx < this._prevValues.length)
            this._prevValues.set_Item(idx, null);
    }
    toString() {
        return NString.Format("{0} = {1} (exp={2}, studio value={3})", [
            PropDefaults.GetPropertyName(this._id), (this._val === null) ? "is null" : this._val, this._expId, this.StudioValue
        ]);
    }
    static getDefaultProp(propId, parentType, propParent) {
        let prop = null;
        let mgVal = PropDefaults.getDefaultValue(propId, parentType, propParent);
        if (mgVal !== null)
            prop = new Property(propId, propParent, parentType, mgVal);
        return prop;
    }
    InsertPrevValueArray(count) {
        if (!this.ShouldSkipRefresh())
            this._prevValues.InsertRange(new Array(count));
    }
}
Property._numericPropertyPic = null;
Property.UtilImeJpn = (UtilStrByteMode.isLocaleDefLangJPN() ? new UtilImeJpn() : null);
//# sourceMappingURL=data:application/json;base64,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