import { TextDecoder, TextEncoder } from "text-encoding";
export class Encoding {
    constructor(label) {
        this.label = label;
        this.textEncoder = null;
        this.textDecoder = null;
        this.textEncoder = new TextEncoder(label, { NONSTANDARD_allowLegacyEncoding: true });
        this.textDecoder = new TextDecoder(label, { NONSTANDARD_allowLegacyEncoding: true });
    }
    GetBytes(str) {
        if (str === null)
            throw new Error("Argument is null");
        let bytes = null;
        try {
            bytes = this.textEncoder.encode(str);
        }
        catch (ex) {
            bytes = new Uint8Array(str.length * 3);
            let bytesCount = 0;
            let tmpBytes;
            for (let i = 0; i < str.length; i++) {
                try {
                    tmpBytes = this.textEncoder.encode(str[i]);
                    bytes.set(tmpBytes, bytesCount);
                    bytesCount += tmpBytes.length;
                }
                catch (ex) {
                    bytes[bytesCount++] = "?".charCodeAt(0);
                }
            }
            bytes = bytes.subarray(0, 0 + bytesCount);
        }
        return bytes;
    }
    GetByteCount(str) {
        return this.GetBytes(str).length;
    }
    GetString(bytes, index, count) {
        let str = null;
        if (bytes === null)
            throw new Error("Argument is null");
        if (index < 0 || count < 0 || (index + count) > bytes.length)
            throw new Error("Argument out of range");
        bytes = bytes.subarray(index, index + count);
        try {
            str = this.textDecoder.decode(bytes);
        }
        catch (ex) {
        }
        return str;
    }
    static GetEncoding(codepage) {
        let label = Encoding.CodePageToEncodingMap[codepage];
        if (label == null)
            throw new Error("Invalid codepage.");
        return new Encoding(label);
    }
    static PopulateCodePageToEncodingMap() {
        let hashTable = new Object;
        hashTable[20106] = "DIN_66003";
        hashTable[20107] = "SEN_850200_B";
        hashTable[50932] = "_autodetect";
        hashTable[20108] = "NS_4551-1";
        hashTable[50949] = "_autodetect_kr";
        hashTable[950] = "big5";
        hashTable[50221] = "csISO2022JP";
        hashTable[51949] = "euc-kr";
        hashTable[936] = "gb2312";
        hashTable[52936] = "hz-gb-2312";
        hashTable[852] = "ibm852";
        hashTable[866] = "ibm866";
        hashTable[20105] = "irv";
        hashTable[50220] = "iso-2022-jp";
        hashTable[50222] = "iso-2022-jp";
        hashTable[50225] = "iso-2022-kr";
        hashTable[28591] = "iso-8859-1";
        hashTable[28592] = "iso-8859-2";
        hashTable[28593] = "iso-8859-3";
        hashTable[28594] = "iso-8859-4";
        hashTable[28595] = "iso-8859-5";
        hashTable[28596] = "iso-8859-6";
        hashTable[28597] = "iso-8859-7";
        hashTable[28598] = "iso-8859-8";
        hashTable[20866] = "koi8-r";
        hashTable[949] = "ks_c_5601";
        hashTable[932] = "shift-jis";
        hashTable[1200] = "unicode";
        hashTable[1201] = "unicodeFEFF";
        hashTable[65000] = "utf-7";
        hashTable[65001] = "utf-8";
        hashTable[1250] = "windows-1250";
        hashTable[1251] = "windows-1251";
        hashTable[1252] = "windows-1252";
        hashTable[1253] = "windows-1253";
        hashTable[1254] = "windows-1254";
        hashTable[1255] = "windows-1255";
        hashTable[1256] = "windows-1256";
        hashTable[1257] = "windows-1257";
        hashTable[1258] = "windows-1258";
        hashTable[874] = "windows-874";
        hashTable[20127] = "US-ASCII";
        hashTable[51932] = "x-euc";
        return hashTable;
    }
    toString() {
        return this.label;
    }
}
Encoding.ASCII = new Encoding("ascii");
Encoding.UTF8 = new Encoding("utf-8");
Encoding.Unicode = new Encoding("utf-16le");
Encoding.CodePageToEncodingMap = Encoding.PopulateCodePageToEncodingMap();
//# sourceMappingURL=data:application/json;base64,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