import { Array_Enumerator } from "./ArrayEnumerator";
import { Exception } from "./Exception";
import { Encoding } from "./Encoding";
export class HashUtils {
    static GetHashCode(str) {
        let bytes = null;
        bytes = Encoding.UTF8.GetBytes(str);
        return HashUtils.getHashCode(bytes);
    }
    static getHashCode(byteArray) {
        let hashval;
        let hash1 = 5381;
        let hash2 = hash1;
        let c;
        for (let i = 0; i < byteArray.length; i += 2) {
            c = byteArray[i];
            hash1 = ((hash1 << 5) + hash1) ^ c;
            if ((i + 1) === byteArray.length)
                break;
            c = byteArray[i + 1];
            hash2 = ((hash2 << 5) + hash2) ^ c;
            hash1 = hash1 & 0xffff;
            hash2 = hash2 & 0xffff;
        }
        let hash3 = (hash2 * 13577);
        hashval = (hash1 + hash3);
        return (hashval);
    }
}
const HashTableLoadFactor = 0.75;
export class Hashtable {
    constructor(bucketCount = 8, loadFactor = HashTableLoadFactor) {
        this._buckets = [];
        this._elementsCount = 0;
        this._bucketCount = 0;
        this._loadFactor = 0;
        this._bucketCount = bucketCount;
        this._loadFactor = loadFactor;
        if (this._bucketCount % 2 !== 0) {
            throw new Exception('Bucket count must be a positive number and be multiple of 2.');
        }
    }
    HashFunction(key) {
        if (typeof key.GetHashCode === 'function') {
            return key.GetHashCode();
        }
        else if (key.constructor === String) {
            return HashUtils.GetHashCode(key.toString());
        }
        else if (key.constructor === Number) {
            return +key;
        }
        return 0;
    }
    get Count() {
        return this._elementsCount;
    }
    get Values() {
        let array = [];
        this._buckets.forEach(b => b.forEach(item => {
            array.push(item.value);
        }));
        return new Array_Enumerator(array);
    }
    get Keys() {
        let array = [];
        this._buckets.forEach(b => b.forEach(item => {
            array.push(item.key);
        }));
        return new Array_Enumerator(array);
    }
    Add(key, value) {
        this.Insert(key, value, true);
    }
    set_Item(key, value) {
        this.Insert(key, value, false);
    }
    Insert(key, value, add) {
        let bucketIndex = this.GetBucketIndex(key);
        if (typeof this._buckets[bucketIndex] === "undefined") {
            this._buckets[bucketIndex] = [];
        }
        if (this._buckets[bucketIndex].find(x => x.key === key)) {
            if (add) {
                throw new Exception('Item with provided key already exists!');
            }
            else {
                this.Remove(key);
                return;
            }
        }
        this._buckets[bucketIndex].push({ key: key, value: value });
        this._elementsCount++;
        if (this._elementsCount > this._bucketCount * this._loadFactor) {
            this.Resize(this._bucketCount * 2);
        }
    }
    get_Item(key) {
        let bucketIndex = this.GetBucketIndex(key);
        let bucket = this._buckets[bucketIndex];
        if (!bucket) {
            return null;
        }
        let item = bucket.find(x => x.key === key);
        if (item) {
            return item.value;
        }
        return null;
    }
    ContainsKey(key) {
        let bucketIndex = this.GetBucketIndex(key);
        let bucket = this._buckets[bucketIndex];
        if (!bucket) {
            return false;
        }
        let itemIndex = bucket.findIndex(x => x.key === key);
        return (itemIndex > -1);
    }
    Remove(key) {
        let bucketIndex = this.GetBucketIndex(key);
        let bucket = this._buckets[bucketIndex];
        if (!bucket) {
            return;
        }
        let itemIndex = bucket.findIndex(x => x.key === key);
        if (itemIndex > -1) {
            bucket.splice(itemIndex, 1);
            this._elementsCount--;
            if (this._elementsCount <= this._bucketCount * (1 - this._loadFactor)) {
                this.Resize(this._bucketCount / 2);
            }
        }
    }
    Resize(newBucketCount) {
        let _oldBuckets = this._buckets;
        this._elementsCount = 0;
        this._buckets = [];
        this._bucketCount = newBucketCount;
        _oldBuckets.forEach(b => b.forEach(item => this.Add(item.key, item.value)));
    }
    GetBucketIndex(key) {
        let hash = this.HashFunction(key);
        if (hash % 1 !== 0) {
            throw new Exception('Key\'s hash must be an integer!');
        }
        let index = Math.abs(hash) % this._bucketCount;
        if (index < 0 || index >= this._bucketCount) {
            throw new Exception('Index exceeds bucket boundaries');
        }
        return index;
    }
    Clear() {
        this._elementsCount = 0;
        this._buckets = [];
    }
}
//# sourceMappingURL=data:application/json;base64,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