import moment from 'moment';
import { TextEncoder, TextDecoder } from 'text-encoding';
import * as StackTrace$1 from 'stacktrace-js';
import { __awaiter } from 'tslib';
import { interval } from 'rxjs';

function isNullOrUndefined(object) {
    return object == undefined || object == null;
}
function isUndefined(object) {
    return object == undefined;
}

class Exception {
    constructor(messageOrError) {
        this.name = "Exception";
        this.message = "";
        this.stack = "";
        this.errorLevel = 1;
        if (!isNullOrUndefined(messageOrError)) {
            if (messageOrError instanceof Error) {
                this.stack = messageOrError.stack;
                this.message = messageOrError.message;
            }
            else {
                this.message = messageOrError;
                this.stack = new Error().stack;
            }
        }
    }
    get Message() {
        return this.message;
    }
    get StackTrace() {
        let stackTrace = this.stack || '';
        let stackLines = stackTrace.split("\n").map(function (line) { return line.trim(); });
        stackTrace = stackLines.splice(stackLines[0] === 'Error' ? this.errorLevel + 1 : this.errorLevel).join("\n");
        return "\n" + stackTrace;
    }
    GetType() {
        return this.name;
    }
}

class ApplicationException extends Exception {
    constructor(message = "", innerException) {
        super(message);
        this.InnerException = null;
        this.name = 'ApplicationException';
        this.InnerException = (isUndefined(innerException) ? null : innerException);
        this.errorLevel = 2;
    }
}

class Array_Enumerator {
    constructor(array) {
        this.index = -1;
        this.array = array;
    }
    MoveNext() {
        this.index++;
        return this.index < this.array.length;
    }
    get Current() {
        return this.array[this.index];
    }
    Dispose() {
    }
}

class BitArray {
    get Length() {
        return this.array.length;
    }
    constructor(length) {
        this.array = null;
        this.array = new Array(length).fill(false);
    }
    Get(index) {
        return this.array[index];
    }
    Set(index, value) {
        this.array[index] = value;
    }
}

class Char {
}
Char.MinValue = '\0';
Char.MaxValue = String.fromCharCode(0xFFFF);

class CultureInfo {
}
CultureInfo.InvariantCulture = new CultureInfo();

var DateTimeKind;
(function (DateTimeKind) {
    DateTimeKind[DateTimeKind["Local"] = 0] = "Local";
    DateTimeKind[DateTimeKind["Utc"] = 1] = "Utc";
})(DateTimeKind || (DateTimeKind = {}));
class DateTime {
    get Ticks() {
        return ((this.dt.getTime() * this.ticksPerMillisecond) + this.epochTicks) - (this.dt.getTimezoneOffset() * 600000000);
    }
    get Year() {
        return this.kind === DateTimeKind.Utc ? this.dt.getUTCFullYear() : this.dt.getFullYear();
    }
    get Month() {
        return this.kind === DateTimeKind.Utc ? this.dt.getUTCMonth() + 1 : this.dt.getMonth() + 1;
    }
    get Day() {
        return this.kind === DateTimeKind.Utc ? this.dt.getUTCDate() : this.dt.getDate();
    }
    get Hour() {
        return this.kind === DateTimeKind.Utc ? this.dt.getUTCHours() : this.dt.getHours();
    }
    get Minute() {
        return this.kind === DateTimeKind.Utc ? this.dt.getUTCMinutes() : this.dt.getMinutes();
    }
    get Second() {
        return this.kind === DateTimeKind.Utc ? this.dt.getUTCSeconds() : this.dt.getSeconds();
    }
    get Millisecond() {
        return this.kind === DateTimeKind.Utc ? this.dt.getUTCMilliseconds() : this.dt.getMilliseconds();
    }
    constructor(yearOrDate, month, day, hour, minute, second) {
        this.dt = null;
        this.kind = null;
        this.epochTicks = 621355968000000000;
        this.ticksPerMillisecond = 10000;
        if (arguments.length === 1) {
            this.dt = yearOrDate;
        }
        if (arguments.length === 3) {
            this.dt = new Date(yearOrDate, month - 1, day);
        }
        if (arguments.length === 6) {
            this.dt = new Date(yearOrDate, month - 1, day, hour, minute, second);
        }
        this.kind = DateTimeKind.Local;
    }
    Format(formatString) {
        return moment(this.dt).format(formatString);
    }
    static get UtcNow() {
        let d = new DateTime(new Date());
        d.kind = DateTimeKind.Utc;
        return d;
    }
    static get Now() {
        let d = new DateTime(new Date());
        d.kind = DateTimeKind.Local;
        return d;
    }
    static GetTotalSecondsFromMidnight(utcTime, date) {
        let hh = utcTime ? date.getUTCHours() : date.getHours();
        let mm = utcTime ? date.getUTCMinutes() : date.getMinutes();
        let ss = utcTime ? date.getUTCSeconds() : date.getSeconds();
        return hh * 60 * 60 + mm * 60 + ss;
    }
    static GetTotalMilliSecondsFromMidnight(utcTime, date) {
        let hh = utcTime ? date.getUTCHours() : date.getHours();
        let mm = utcTime ? date.getUTCMinutes() : date.getMinutes();
        let ss = utcTime ? date.getUTCSeconds() : date.getSeconds();
        let ms = utcTime ? date.getUTCSeconds() : date.getMilliseconds();
        return (hh * 60 * 60 + mm * 60 + ss) * 1000 + ms;
    }
}

class Debug {
    static WriteLine(text) {
        console.log(text);
    }
    static Assert(assertCondtion, message) {
        if (!assertCondtion)
            alert(message);
    }
}

class Dictionary {
    constructor() {
        this.values = new Array();
    }
    get_Item(key) {
        if (this.values.hasOwnProperty(key))
            return this.values[key];
        else
            return null;
    }
    set_Item(key, value) {
        this.values[key] = value;
    }
    Add(key, value) {
        if (this.values.hasOwnProperty(key))
            throw new Exception("An item with the same key has already been added");
        else
            this.values[key] = value;
    }
    ContainsKey(key) {
        return this.values.hasOwnProperty(key);
    }
    TryGetValue(key, pvalue) {
        if (this.values.hasOwnProperty(key)) {
            pvalue.value = this.values[key];
            return true;
        }
        else {
            pvalue.value = null;
            return false;
        }
    }
    Remove(key) {
        delete this.values[key];
    }
    get Keys() {
        return Object.keys(this.values);
    }
}

class Encoding {
    constructor(label) {
        this.label = label;
        this.textEncoder = null;
        this.textDecoder = null;
        this.textEncoder = new TextEncoder(label, { NONSTANDARD_allowLegacyEncoding: true });
        this.textDecoder = new TextDecoder(label, { NONSTANDARD_allowLegacyEncoding: true });
    }
    GetBytes(str) {
        if (str === null)
            throw new Error("Argument is null");
        let bytes = null;
        try {
            bytes = this.textEncoder.encode(str);
        }
        catch (ex) {
            bytes = new Uint8Array(str.length * 3);
            let bytesCount = 0;
            let tmpBytes;
            for (let i = 0; i < str.length; i++) {
                try {
                    tmpBytes = this.textEncoder.encode(str[i]);
                    bytes.set(tmpBytes, bytesCount);
                    bytesCount += tmpBytes.length;
                }
                catch (ex) {
                    bytes[bytesCount++] = "?".charCodeAt(0);
                }
            }
            bytes = bytes.subarray(0, 0 + bytesCount);
        }
        return bytes;
    }
    GetByteCount(str) {
        return this.GetBytes(str).length;
    }
    GetString(bytes, index, count) {
        let str = null;
        if (bytes === null)
            throw new Error("Argument is null");
        if (index < 0 || count < 0 || (index + count) > bytes.length)
            throw new Error("Argument out of range");
        bytes = bytes.subarray(index, index + count);
        try {
            str = this.textDecoder.decode(bytes);
        }
        catch (ex) {
        }
        return str;
    }
    static GetEncoding(codepage) {
        let label = Encoding.CodePageToEncodingMap[codepage];
        if (label == null)
            throw new Error("Invalid codepage.");
        return new Encoding(label);
    }
    static PopulateCodePageToEncodingMap() {
        let hashTable = new Object;
        hashTable[20106] = "DIN_66003";
        hashTable[20107] = "SEN_850200_B";
        hashTable[50932] = "_autodetect";
        hashTable[20108] = "NS_4551-1";
        hashTable[50949] = "_autodetect_kr";
        hashTable[950] = "big5";
        hashTable[50221] = "csISO2022JP";
        hashTable[51949] = "euc-kr";
        hashTable[936] = "gb2312";
        hashTable[52936] = "hz-gb-2312";
        hashTable[852] = "ibm852";
        hashTable[866] = "ibm866";
        hashTable[20105] = "irv";
        hashTable[50220] = "iso-2022-jp";
        hashTable[50222] = "iso-2022-jp";
        hashTable[50225] = "iso-2022-kr";
        hashTable[28591] = "iso-8859-1";
        hashTable[28592] = "iso-8859-2";
        hashTable[28593] = "iso-8859-3";
        hashTable[28594] = "iso-8859-4";
        hashTable[28595] = "iso-8859-5";
        hashTable[28596] = "iso-8859-6";
        hashTable[28597] = "iso-8859-7";
        hashTable[28598] = "iso-8859-8";
        hashTable[20866] = "koi8-r";
        hashTable[949] = "ks_c_5601";
        hashTable[932] = "shift-jis";
        hashTable[1200] = "unicode";
        hashTable[1201] = "unicodeFEFF";
        hashTable[65000] = "utf-7";
        hashTable[65001] = "utf-8";
        hashTable[1250] = "windows-1250";
        hashTable[1251] = "windows-1251";
        hashTable[1252] = "windows-1252";
        hashTable[1253] = "windows-1253";
        hashTable[1254] = "windows-1254";
        hashTable[1255] = "windows-1255";
        hashTable[1256] = "windows-1256";
        hashTable[1257] = "windows-1257";
        hashTable[1258] = "windows-1258";
        hashTable[874] = "windows-874";
        hashTable[20127] = "US-ASCII";
        hashTable[51932] = "x-euc";
        return hashTable;
    }
    toString() {
        return this.label;
    }
}
Encoding.ASCII = new Encoding("ascii");
Encoding.UTF8 = new Encoding("utf-8");
Encoding.Unicode = new Encoding("utf-16le");
Encoding.CodePageToEncodingMap = Encoding.PopulateCodePageToEncodingMap();

class HashUtils {
    static GetHashCode(str) {
        let bytes = null;
        bytes = Encoding.UTF8.GetBytes(str);
        return HashUtils.getHashCode(bytes);
    }
    static getHashCode(byteArray) {
        let hashval;
        let hash1 = 5381;
        let hash2 = hash1;
        let c;
        for (let i = 0; i < byteArray.length; i += 2) {
            c = byteArray[i];
            hash1 = ((hash1 << 5) + hash1) ^ c;
            if ((i + 1) === byteArray.length)
                break;
            c = byteArray[i + 1];
            hash2 = ((hash2 << 5) + hash2) ^ c;
            hash1 = hash1 & 0xffff;
            hash2 = hash2 & 0xffff;
        }
        let hash3 = (hash2 * 13577);
        hashval = (hash1 + hash3);
        return (hashval);
    }
}
const HashTableLoadFactor = 0.75;
class Hashtable {
    constructor(bucketCount = 8, loadFactor = HashTableLoadFactor) {
        this._buckets = [];
        this._elementsCount = 0;
        this._bucketCount = 0;
        this._loadFactor = 0;
        this._bucketCount = bucketCount;
        this._loadFactor = loadFactor;
        if (this._bucketCount % 2 !== 0) {
            throw new Exception('Bucket count must be a positive number and be multiple of 2.');
        }
    }
    HashFunction(key) {
        if (typeof key.GetHashCode === 'function') {
            return key.GetHashCode();
        }
        else if (key.constructor === String) {
            return HashUtils.GetHashCode(key.toString());
        }
        else if (key.constructor === Number) {
            return +key;
        }
        return 0;
    }
    get Count() {
        return this._elementsCount;
    }
    get Values() {
        let array = [];
        this._buckets.forEach(b => b.forEach(item => {
            array.push(item.value);
        }));
        return new Array_Enumerator(array);
    }
    get Keys() {
        let array = [];
        this._buckets.forEach(b => b.forEach(item => {
            array.push(item.key);
        }));
        return new Array_Enumerator(array);
    }
    Add(key, value) {
        this.Insert(key, value, true);
    }
    set_Item(key, value) {
        this.Insert(key, value, false);
    }
    Insert(key, value, add) {
        let bucketIndex = this.GetBucketIndex(key);
        if (typeof this._buckets[bucketIndex] === "undefined") {
            this._buckets[bucketIndex] = [];
        }
        if (this._buckets[bucketIndex].find(x => x.key === key)) {
            if (add) {
                throw new Exception('Item with provided key already exists!');
            }
            else {
                this.Remove(key);
                return;
            }
        }
        this._buckets[bucketIndex].push({ key: key, value: value });
        this._elementsCount++;
        if (this._elementsCount > this._bucketCount * this._loadFactor) {
            this.Resize(this._bucketCount * 2);
        }
    }
    get_Item(key) {
        let bucketIndex = this.GetBucketIndex(key);
        let bucket = this._buckets[bucketIndex];
        if (!bucket) {
            return null;
        }
        let item = bucket.find(x => x.key === key);
        if (item) {
            return item.value;
        }
        return null;
    }
    ContainsKey(key) {
        let bucketIndex = this.GetBucketIndex(key);
        let bucket = this._buckets[bucketIndex];
        if (!bucket) {
            return false;
        }
        let itemIndex = bucket.findIndex(x => x.key === key);
        return (itemIndex > -1);
    }
    Remove(key) {
        let bucketIndex = this.GetBucketIndex(key);
        let bucket = this._buckets[bucketIndex];
        if (!bucket) {
            return;
        }
        let itemIndex = bucket.findIndex(x => x.key === key);
        if (itemIndex > -1) {
            bucket.splice(itemIndex, 1);
            this._elementsCount--;
            if (this._elementsCount <= this._bucketCount * (1 - this._loadFactor)) {
                this.Resize(this._bucketCount / 2);
            }
        }
    }
    Resize(newBucketCount) {
        let _oldBuckets = this._buckets;
        this._elementsCount = 0;
        this._buckets = [];
        this._bucketCount = newBucketCount;
        _oldBuckets.forEach(b => b.forEach(item => this.Add(item.key, item.value)));
    }
    GetBucketIndex(key) {
        let hash = this.HashFunction(key);
        if (hash % 1 !== 0) {
            throw new Exception('Key\'s hash must be an integer!');
        }
        let index = Math.abs(hash) % this._bucketCount;
        if (index < 0 || index >= this._bucketCount) {
            throw new Exception('Index exceeds bucket boundaries');
        }
        return index;
    }
    Clear() {
        this._elementsCount = 0;
        this._buckets = [];
    }
}

class Int32 {
}
Int32.MinValue = -2147483648;
Int32.MaxValue = 2147483647;

class Int64 {
}
Int64.MinValue = -9223372036854775808;

class ISO_8859_1_Encoding extends Encoding {
    constructor(label) {
        super(label);
    }
    GetBytes(str) {
        if (str === null)
            throw new Error("Argument is null");
        let bytes = null;
        bytes = new Uint8Array(str.length * 3);
        let bytesCount = 0;
        for (let i = 0; i < str.length; i++) {
            bytes[bytesCount++] = str[i].charCodeAt(0);
        }
        bytes = bytes.subarray(0, 0 + bytesCount);
        return bytes;
    }
    GetString(bytes, index, count) {
        let str = null;
        if (bytes === null)
            throw new Error("Argument is null");
        if (index < 0 || count < 0 || (index + count) > bytes.length)
            throw new Error("Argument out of range");
        bytes = bytes.subarray(index, index + count);
        try {
            str = '';
            for (let i = 0; i < bytes.length; i++) {
                str += String.fromCharCode(bytes[i]);
            }
            ;
        }
        catch (ex) {
        }
        return str;
    }
}
ISO_8859_1_Encoding.ISO_8859_1 = new ISO_8859_1_Encoding("iso_8859-1");

class List extends Array {
    constructor(arrayEnumerator) {
        super();
        Object.setPrototypeOf(this, List.prototype);
        if (arguments.length === 1 && arrayEnumerator.constructor === Array_Enumerator) {
            this.AddRange(arrayEnumerator);
        }
    }
    AddRange(arrayEnumeratorOrArray) {
        if (arrayEnumeratorOrArray.constructor === Array) {
            arrayEnumeratorOrArray.forEach((item) => {
                this.push(item);
            });
        }
        else {
            let e = arrayEnumeratorOrArray;
            while (e.MoveNext()) {
                this.push(e.Current);
            }
        }
    }
    get_Item(index) {
        if (isUndefined(this[index]))
            this[index] = null;
        return this[index];
    }
    set_Item(index, value) {
        if (index >= 0 && index < this.length)
            this[index] = value;
        else
            throw new Error("index out of bounds");
    }
    GetEnumerator() {
        return new Array_Enumerator(this);
    }
    Remove(object) {
        let index = this.indexOf(object);
        if (index > -1)
            this.RemoveAt(index);
    }
    RemoveAt(index) {
        this.RemoveRange(index, 1);
    }
    RemoveRange(index, count) {
        this.splice(index, count);
    }
    Insert(index, item) {
        if (index >= 0 && index < this.length)
            this.splice(index, 0, item);
        else if (index === this.length)
            this.push(item);
        else
            throw new Error("index out of bounds");
    }
    InsertRange(items) {
        let idx = 0;
        while (idx < items.length) {
            this.Insert(idx, items[idx]);
            idx++;
        }
    }
    Clear() {
        this.splice(0, this.length);
    }
    ToArray() {
        return this.slice(0);
    }
    Contains(object) {
        return this.indexOf(object) > -1;
    }
    find(predicate) {
        let foundItem = super.find(predicate);
        if (isUndefined(foundItem))
            foundItem = null;
        return foundItem;
    }
    SetSize(size) {
        if (this.length > size) {
            this.RemoveRange(size, this.length - size);
        }
        else
            this.length = size;
    }
}

const $EOF = 0;
const $TAB = 9;
const $LF = 10;
const $VTAB = 11;
const $FF = 12;
const $CR = 13;
const $SPACE = 32;
const $BANG = 33;
const $DQ = 34;
const $HASH = 35;
const $$ = 36;
const $PERCENT = 37;
const $AMPERSAND = 38;
const $SQ = 39;
const $LPAREN = 40;
const $RPAREN = 41;
const $STAR = 42;
const $PLUS = 43;
const $COMMA = 44;
const $MINUS = 45;
const $PERIOD = 46;
const $SLASH = 47;
const $COLON = 58;
const $SEMICOLON = 59;
const $LT = 60;
const $EQ = 61;
const $GT = 62;
const $QUESTION = 63;
const $0 = 48;
const $9 = 57;
const $A = 65;
const $E = 69;
const $F = 70;
const $X = 88;
const $Z = 90;
const $LBRACKET = 91;
const $BACKSLASH = 92;
const $RBRACKET = 93;
const $CARET = 94;
const $_ = 95;
const $a = 97;
const $e = 101;
const $f = 102;
const $n = 110;
const $r = 114;
const $t = 116;
const $u = 117;
const $v = 118;
const $x = 120;
const $z = 122;
const $LBRACE = 123;
const $BAR = 124;
const $RBRACE = 125;
const $NBSP = 160;
const $PIPE = 124;
const $TILDA = 126;
const $AT = 64;
const $BT = 96;
function isWhitespace(code) {
    return (code >= $TAB && code <= $SPACE) || code === $NBSP;
}
function isDigit(code) {
    return $0 <= code && code <= $9;
}
function isAsciiLetter(code) {
    return (code >= $a && code <= $z) || (code >= $A && code <= $Z);
}
function isAsciiHexDigit(code) {
    return (code >= $a && code <= $f) || (code >= $A && code <= $F) || isDigit(code);
}
function isLowerCase(str) {
    return str === str.toLowerCase();
}
function isUpperCase(str) {
    return str === str.toUpperCase();
}

class NChar {
    static IsWhiteSpace(str) {
        let ch = str.charCodeAt(0);
        return isWhitespace(ch);
    }
    static IsLetterOrDigit(str) {
        let ch = str.charCodeAt(0);
        return isDigit(ch) || isAsciiLetter(ch);
    }
    static IsDigit(str) {
        let ch = str.charCodeAt(0);
        return isDigit(ch);
    }
    static IsLetter(ltr) {
        return isAsciiLetter(ltr.charCodeAt(0));
    }
    static IsLower(ch) {
        return isLowerCase(ch);
    }
    static IsUpper(ch) {
        return isUpperCase(ch);
    }
}

class NotImplementedException extends Exception {
    constructor(message = "Not implemented") {
        super(message);
    }
}

var NumberStyles;
(function (NumberStyles) {
    NumberStyles[NumberStyles["HexNumber"] = 0] = "HexNumber";
})(NumberStyles || (NumberStyles = {}));
class NNumber {
    static Parse(text, style) {
        if (arguments.length === 2) {
            if (style === NumberStyles.HexNumber)
                return parseInt(text, 16);
            else
                throw new NotImplementedException();
        }
        return +text;
    }
    static TryParse(str, pvalue) {
        pvalue.value = +str;
        return !isNaN(pvalue.value);
    }
    static ToString(num, format) {
        if (format === 'X2') {
            let res = num.toString(16);
            return res.length === 1 ? '0' + res : res;
        }
        throw new NotImplementedException();
    }
}

class NString {
    static IndexOf(str, searchStr, startIndex, count) {
        let index = str.substr(startIndex, count).indexOf(searchStr);
        return index > -1 ? startIndex + index : index;
    }
    static IndexOfAny(str, subs, startIndex, count) {
        for (let i = 0; i < count; ++i) {
            let c = str.charAt(startIndex + i);
            for (let j = 0; j < subs.length; ++j) {
                if (c === subs[j])
                    return (startIndex + i);
            }
        }
        return -1;
    }
    static CopyTo(source, sourceIndex, destination, destinationIndex, count) {
        for (let i = 0; i < count; i++) {
            destination[destinationIndex + i] = source[sourceIndex + i];
        }
    }
    static Compare(strA, strB, ignoreCase, indexA, indexB, length) {
        if (arguments.length = 6) {
            strA = strA.substr(indexA, length);
            strB = strB.substr(indexB, length);
        }
        if (ignoreCase) {
            strA = strA.toLowerCase();
            strB = strB.toLowerCase();
        }
        return strA.localeCompare(strB);
    }
    static Equals(strA, strB, ignoreCase) {
        if (!NString.IsNullOrEmpty(strA) && !NString.IsNullOrEmpty(strB)) {
            if (ignoreCase) {
                strA = strA.toLowerCase();
                strB = strB.toLowerCase();
            }
            return strA === strB;
        }
        return false;
    }
    static GetHashCode(str) {
        var hash = 0, i, l, ch;
        if (str.length === 0)
            return hash;
        for (i = 0, l = str.length; i < l; i++) {
            ch = str.charCodeAt(i);
            hash = ((hash << 5) - hash) + ch;
            hash |= 0;
        }
        return hash;
    }
    static Remove(str, startIndex, length) {
        return str.substr(0, startIndex - 1) + str.substr(startIndex + length - 1);
    }
    static TrimStart(str) {
        return str.replace(/^\s\s*/, '');
    }
    static TrimEnd(string, trimChars) {
        if (arguments.length === 1)
            return string.replace(/\s\s*$/, '');
        for (let j = 0; j < trimChars.length; j++) {
            let trimChar = trimChars[j];
            let i = string.length - 1;
            while (!isNullOrUndefined(string[i]) && string[i].endsWith(trimChar)) {
                string = NString.Remove(string, string.length, 1);
                i--;
            }
        }
        return string;
    }
    static Format(format, arg0, arg1, arg2, arg3, arg4, arg5) {
        if (!isNullOrUndefined(arg0) && arg0.constructor === Array) {
            var s = format, i = arg0.length;
            while (i--) {
                s = s.replace(new RegExp('\\{' + i + '\\}', 'gm'), arg0[i]);
            }
            return s;
        }
        else {
            var args = [arg0, arg1, arg2, arg3, arg4, arg5];
            return NString.Format(format, args);
        }
    }
    static IsNullOrEmpty(str1) {
        return !str1;
    }
    static FromChars(chOrChars, startIndex, len) {
        let str = chOrChars.join('');
        if (arguments.length === 3)
            return str.substr(startIndex, len);
        return str;
    }
    static FromChar(ch, count) {
        return (new Array(count).fill(ch)).join('');
    }
    static ToCharArray(str) {
        return Array.from(str);
    }
    static CompareOrdinal(strA, strB) {
        let lenA = strA.length;
        let lenB = strB.length;
        let len = Math.min(lenA, lenB);
        for (let i = 0; i < len; i++) {
            if (strA[i].charCodeAt(0) !== strB[i].charCodeAt(0)) {
                return strA[i].charCodeAt(0) - strB[i].charCodeAt(0);
            }
        }
        if (lenA !== lenB) {
            return lenA - lenB;
        }
        return 0;
    }
    static PadRight(source, maxLength, fillString) {
        if (source.length >= maxLength)
            return source;
        let fillLen = maxLength - source.length;
        let timesToRepeat = Math.ceil(fillLen / fillString.length);
        let truncatedStringFiller = fillString.repeat(timesToRepeat).slice(0, fillLen);
        return source + truncatedStringFiller;
    }
    static Insert(str, index, ch) {
        return str.substr(0, index) + ch + str.substr(index, str.length - index);
    }
    static Replace(str, orgSubStr, newSubStr) {
        let resultStr = '';
        orgSubStr = orgSubStr.replace(new RegExp("\\\\", 'g'), "\\\\");
        orgSubStr = orgSubStr.replace(new RegExp("\\*", 'g'), "\\\*");
        orgSubStr = orgSubStr.replace(new RegExp("\\$", 'g'), "\\\$");
        orgSubStr = orgSubStr.replace(new RegExp("\\^", 'g'), "\\\^");
        orgSubStr = orgSubStr.replace(new RegExp("\\.", 'g'), "\\\.");
        orgSubStr = orgSubStr.replace(new RegExp("\\?", 'g'), "\\\?");
        orgSubStr = orgSubStr.replace(new RegExp("\\+", 'g'), "\\\+");
        orgSubStr = orgSubStr.replace(new RegExp("\\,", 'g'), "\\\,");
        orgSubStr = orgSubStr.replace(new RegExp("\\[", 'g'), "\\\[");
        orgSubStr = orgSubStr.replace(new RegExp("\\|", 'g'), "\\\|");
        try {
            resultStr = str.replace(new RegExp(orgSubStr, 'g'), newSubStr);
        }
        catch (e) {
            throw new Exception(e.message);
        }
        return resultStr;
    }
}
NString.Empty = "";

class NumberFormatInfo {
    static GetLocaleDecimalSeperator() {
        return (1.1).toLocaleString()[1];
    }
}
NumberFormatInfo.NegativeSign = '-';
NumberFormatInfo.NumberDecimalSeparator = NumberFormatInfo.GetLocaleDecimalSeperator();

class RefParam {
    constructor(value) {
        this.value = value;
    }
}

class Stack {
    constructor() {
        this._array = new Array();
    }
    count() {
        return this._array.length;
    }
    isEmpty() {
        return this._array.length === 0;
    }
    push(value) {
        this._array.push(value);
    }
    pop() {
        return this._array.pop();
    }
    peek() {
        if (this._array.length > 0) {
            return this._array[this._array.length - 1];
        }
        else {
            return null;
        }
    }
    Clear() {
        this._array.splice(0, this._array.length);
    }
    Clone() {
        let clone = new Stack();
        clone._array = this._array.slice();
        return clone;
    }
}

class StackFrame {
    constructor(skipFrames) {
        this.stackFrame = null;
        let stackFrames = StackTrace$1.getSync();
        if (skipFrames < 0 || skipFrames >= stackFrames.length)
            throw new Error("Argument out of range");
        this.stackFrame = stackFrames[skipFrames];
    }
    GetFileName() {
        return this.stackFrame.fileName;
    }
}

class StackTrace {
    GetFrames() {
        return StackTrace$1.getSync();
    }
}

class StringBuilder {
    constructor(valueOrLength, length) {
        this.part = "";
        if (arguments.length > 0) {
            if (valueOrLength != null && valueOrLength.constructor === Number) {
            }
            else if (valueOrLength != null && valueOrLength.constructor === String) {
                this.part = valueOrLength.toString();
            }
        }
    }
    Append(textOrNum, startIndexOrNumberOfCharacters, charCount) {
        if (textOrNum === null)
            return this;
        if (textOrNum.constructor === String) {
            if (arguments.length === 1)
                this.AppendString(textOrNum.toString());
            else if (arguments.length === 2) {
                Debug.Assert(textOrNum.length === 1, "character is expected not string");
                this.AppendString(textOrNum.charAt(0).repeat(startIndexOrNumberOfCharacters));
            }
            else
                this.AppendString(textOrNum.toString(), startIndexOrNumberOfCharacters, charCount);
        }
        else
            this.AppendNumber(textOrNum, startIndexOrNumberOfCharacters);
        return this;
    }
    AppendString(text, startIndex = 0, charCount = text.length) {
        this.part = this.part + text.substr(startIndex, charCount);
    }
    AppendNumber(num, numberOfCharacters = 1) {
        if (numberOfCharacters <= 0)
            throw new Error("numberOfCharacters cannot be less than or equal to zero");
        this.part = this.part + num.toString().repeat(numberOfCharacters);
    }
    AppendLine(text = null) {
        if (text !== null)
            this.part = this.part + text;
        this.part = this.part + '\n';
    }
    AppendFormat(format, arg0, arg1, arg2) {
        this.part = this.part + NString.Format(format, arg0, arg1, arg2);
        return this;
    }
    ToString(startIndex, length) {
        if (arguments.length === 2) {
            if (startIndex < 0 || length < 0 || (startIndex + length) > this.part.length)
                throw new Error("Argument out of range");
            return this.part.substr(startIndex, length);
        }
        return this.part;
    }
    get Length() {
        return this.part.length;
    }
    get_Item(index) {
        if (index < 0 || index >= this.part.length)
            throw new Error("Index out of range");
        return this.part.charAt(index);
    }
    set_Item(index, value) {
        if (index < 0 || index >= this.part.length)
            throw new Error("Index out of range");
        Debug.Assert(value.length <= 1, "Length of string cannot be more than 1. Only character is expected");
        let leftPart = this.part.substring(0, index);
        let rightPart = this.part.substring(index + 1, this.part.length);
        this.part = leftPart + value.charAt(0) + rightPart;
    }
    Insert(index, value) {
        if (index < 0 || index > this.part.length)
            throw new Error("Argument out of range");
        let leftPart = this.part.substring(0, index);
        let rightPart = this.part.substring(index, this.part.length);
        this.part = leftPart + value + rightPart;
        return this;
    }
    Remove(startIndex, length) {
        if (startIndex < 0 || length < 0 || (startIndex + length) > this.part.length)
            throw new Error("Argument out of range");
        let leftPart = this.part.substring(0, startIndex);
        let rightPart = this.part.substring(startIndex + length, this.part.length);
        this.part = leftPart + rightPart;
        return this;
    }
    Replace(oldValue, newValue, startIndex, count) {
        if (oldValue === null)
            throw new Error("oldValue cannot be null");
        else if (oldValue.length === 0)
            throw new Error("Length of oldValue cannot be 0");
        if (arguments.length === 4) {
            if (startIndex < 0 || count < 0 || (startIndex + count) > this.part.length)
                throw new Error("Argument out of range");
        }
        if (arguments.length === 2)
            this.part = this.part.replace(new RegExp(oldValue, 'g'), newValue);
        else if (arguments.length === 4) {
            let substr = this.part.substring(startIndex, startIndex + count);
            substr = substr.replace(new RegExp(oldValue, 'g'), newValue);
            let leftPart = this.part.substring(0, startIndex);
            let rightPart = this.part.substring(startIndex + count, this.part.length);
            this.part = leftPart + substr + rightPart;
        }
        return this;
    }
    toString() {
        return this.ToString();
    }
}

class Thread {
    constructor() {
        this.ManagedThreadId = 0;
        this.ManagedThreadId = Thread.nextId++;
    }
    static Sleep(millisecondsTimeout) {
        return __awaiter(this, void 0, void 0, function* () {
            yield new Promise((resolve) => {
                let timer = interval(millisecondsTimeout);
                let subscription = timer.subscribe(() => { subscription.unsubscribe(); resolve(); });
            }).then();
        });
    }
}
Thread.nextId = 1;
Thread.CurrentThread = new Thread();

class WebException extends Exception {
    constructor(error) {
        super();
        this.name = "WebException";
        this.stack = error.stack;
        this.message = error.message;
    }
}

class XmlConvert {
    static EncodeName(name) {
        throw new NotImplementedException();
    }
}

export { $$, $0, $9, $A, $AMPERSAND, $AT, $BACKSLASH, $BANG, $BAR, $BT, $CARET, $COLON, $COMMA, $CR, $DQ, $E, $EOF, $EQ, $F, $FF, $GT, $HASH, $LBRACE, $LBRACKET, $LF, $LPAREN, $LT, $MINUS, $NBSP, $PERCENT, $PERIOD, $PIPE, $PLUS, $QUESTION, $RBRACE, $RBRACKET, $RPAREN, $SEMICOLON, $SLASH, $SPACE, $SQ, $STAR, $TAB, $TILDA, $VTAB, $X, $Z, $_, $a, $e, $f, $n, $r, $t, $u, $v, $x, $z, ApplicationException, Array_Enumerator, BitArray, Char, CultureInfo, DateTime, DateTimeKind, Debug, Dictionary, Encoding, Exception, HashUtils, Hashtable, ISO_8859_1_Encoding, Int32, Int64, List, NChar, NNumber, NString, NotImplementedException, NumberFormatInfo, NumberStyles, RefParam, Stack, StackFrame, StackTrace, StringBuilder, Thread, WebException, XmlConvert, isAsciiHexDigit, isAsciiLetter, isDigit, isLowerCase, isNullOrUndefined, isUndefined, isUpperCase, isWhitespace };
//# sourceMappingURL=magic-xpa-mscorelib.mjs.map
