import { ApplicationException, ISO_8859_1_Encoding } from "@magic-xpa/mscorelib";
import { StrUtil } from "./StrUtil";
import { Misc } from "./Misc";
export class Base64 {
    static encode(str, isUseEnvCharset, encoding) {
        if (arguments.length === 1)
            return Base64.encode_1(str, false, null);
        else
            return Base64.encode_1(str, isUseEnvCharset, encoding);
    }
    static encode_1(str, isUseEnvCharset, encoding) {
        let result;
        if (str === null) {
            result = null;
        }
        else {
            if (str === "") {
                result = str;
            }
            else {
                try {
                    if (isUseEnvCharset === false) {
                        result = btoa(str);
                    }
                    else {
                        let instance = ISO_8859_1_Encoding.ISO_8859_1;
                        let encoding2 = isUseEnvCharset ? encoding : instance;
                        let ba = Base64.encodeBytes(encoding2.GetBytes(str));
                        result = instance.GetString(ba, 0, ba.length);
                    }
                }
                catch (ex) {
                    throw new ApplicationException(ex.Message);
                }
            }
        }
        return result;
    }
    static encodeBytes(data) {
        if (data === null)
            return null;
        let dest = new Uint8Array(Math.floor((data.length + 2) / 3) * 4);
        let sidx = 0;
        let didx = 0;
        while (sidx < data.length - 2) {
            dest[didx++] = Base64._base64EncMap[Misc.URShift(data[sidx], 2) & 63];
            dest[didx++] = Base64._base64EncMap[(Misc.URShift(data[sidx + 1], 4) & 15) | (data[sidx] << 4 & 63)];
            dest[didx++] = Base64._base64EncMap[(Misc.URShift(data[sidx + 2], 6) & 3) | (data[sidx + 1] << 2 & 63)];
            dest[didx++] = Base64._base64EncMap[(data[sidx + 2] & 63)];
            sidx = sidx + 3;
        }
        if (sidx < data.length) {
            dest[didx++] = Base64._base64EncMap[Misc.URShift(data[sidx], 2) & 63];
            if (sidx < data.length - 1) {
                dest[didx++] = Base64._base64EncMap[(Misc.URShift(data[sidx + 1], 4) & 15) | (data[sidx] << 4 & 63)];
                dest[didx++] = Base64._base64EncMap[data[sidx + 1] << 2 & 63];
            }
            else {
                dest[didx++] = Base64._base64EncMap[data[sidx] << 4 & 63];
            }
        }
        while (didx < dest.length) {
            dest[didx] = 61;
            didx = didx + 1;
        }
        return dest;
    }
    static decode(str, encoding) {
        if (arguments.length === 1)
            encoding = null;
        return Base64.decode_1(str, encoding);
    }
    static decode_1(str, encoding) {
        let result;
        if (str === null) {
            result = null;
        }
        else {
            if (str === "") {
                result = str;
            }
            else {
                try {
                    if (encoding === null || encoding === undefined) {
                        result = atob(str);
                    }
                    else {
                        let instance = ISO_8859_1_Encoding.ISO_8859_1;
                        let array = Base64.decodeBytes(instance.GetBytes(str));
                        let encoding2 = (encoding !== null) ? encoding : instance;
                        result = encoding2.GetString(array, 0, array.length);
                    }
                }
                catch (ex) {
                    throw new ApplicationException(ex.Message);
                }
            }
        }
        return result;
    }
    static decodeBytes(data) {
        if (data === null)
            return null;
        let tail = data.length;
        while (data[tail - 1] === '='.charCodeAt(0)) {
            tail = tail - 1;
        }
        let dest = new Uint8Array(tail - Math.floor(data.length / 4));
        for (let idx = 0; idx < data.length; idx = idx + 1) {
            data[idx] = Base64._base64DecMap[data[idx]];
        }
        let sidx = 0;
        let didx;
        for (didx = 0; didx < dest.length - 2; didx = didx + 3) {
            dest[didx] = (((data[sidx] << 2) & 255) | (Misc.URShift(data[sidx + 1], 4) & 3));
            dest[didx + 1] = (((data[sidx + 1] << 4) & 255) | (Misc.URShift(data[sidx + 2], 2) & 15));
            dest[didx + 2] = (((data[sidx + 2] << 6) & 255) | (data[sidx + 3] & 63));
            sidx = sidx + 4;
        }
        if (didx < dest.length)
            dest[didx] = (((data[sidx] << 2) & 255) | (Misc.URShift(data[sidx + 1], 4) & 3));
        if ((didx = didx + 1) < dest.length)
            dest[didx] = (((data[sidx + 1] << 4) & 255) | (Misc.URShift(data[sidx + 2], 2) & 15));
        return dest;
    }
    static decodeToHex(str) {
        if (str === null)
            return null;
        if (str === "")
            return str;
        return StrUtil.stringToHexaDump(Base64.decode(str), 2);
    }
    static decodeToByte(str) {
        if (str === null)
            return null;
        if (str === "" || str === " ")
            return new Uint8Array(0);
        try {
            let instance = ISO_8859_1_Encoding.ISO_8859_1;
            return Base64.decodeBytes(instance.GetBytes(str));
        }
        catch (ex) {
            throw new ApplicationException(ex.Message);
        }
    }
    static initializeEncMap() {
        return new Uint8Array([
            65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83,
            84, 85, 86, 87, 88, 89, 90, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109,
            110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 48, 49, 50, 51, 52, 53, 54,
            55, 56, 57, 43, 47
        ]);
    }
    static initializeDecMap() {
        let decMap = new Uint8Array(128);
        for (let i = 0; i < Base64._base64EncMap.length; i = i + 1) {
            decMap[Base64._base64EncMap[i]] = i;
        }
        return decMap;
    }
}
Base64._base64EncMap = Base64.initializeEncMap();
Base64._base64DecMap = Base64.initializeDecMap();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiQmFzZTY0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vcHJvamVjdHMvdXRpbHMvc3JjL0Jhc2U2NC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQSxPQUFPLEVBQVcsb0JBQW9CLEVBQUUsbUJBQW1CLEVBQUMsTUFBTSxzQkFBc0IsQ0FBQztBQUN6RixPQUFPLEVBQUMsT0FBTyxFQUFDLE1BQU0sV0FBVyxDQUFDO0FBQ2xDLE9BQU8sRUFBQyxJQUFJLEVBQUMsTUFBTSxRQUFRLENBQUM7QUFFNUIsTUFBTSxPQUFPLE1BQU07SUFPakIsTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFXLEVBQUUsZUFBcUIsRUFBRSxRQUFtQjtRQUNuRSxJQUFJLFNBQVMsQ0FBQyxNQUFNLEtBQUssQ0FBQztZQUN4QixPQUFPLE1BQU0sQ0FBQyxRQUFRLENBQUMsR0FBRyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQzs7WUFFekMsT0FBTyxNQUFNLENBQUMsUUFBUSxDQUFDLEdBQUcsRUFBRSxlQUFlLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDM0QsQ0FBQztJQVVPLE1BQU0sQ0FBQyxRQUFRLENBQUMsR0FBVyxFQUFFLGVBQXdCLEVBQUUsUUFBa0I7UUFDL0UsSUFBSSxNQUFjLENBQUM7UUFDbkIsSUFBSSxHQUFHLEtBQUssSUFBSSxFQUFFO1lBQ2hCLE1BQU0sR0FBRyxJQUFJLENBQUM7U0FDZjthQUNJO1lBQ0gsSUFBSSxHQUFHLEtBQUssRUFBRSxFQUFFO2dCQUNkLE1BQU0sR0FBRyxHQUFHLENBQUM7YUFDZDtpQkFDSTtnQkFDSCxJQUFJO29CQUNGLElBQUksZUFBZSxLQUFLLEtBQUssRUFBQzt3QkFDNUIsTUFBTSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztxQkFDcEI7eUJBQ0k7d0JBQ0gsSUFBSSxRQUFRLEdBQWEsbUJBQW1CLENBQUMsVUFBVSxDQUFDO3dCQUN4RCxJQUFJLFNBQVMsR0FBYSxlQUFlLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDO3dCQUNoRSxJQUFJLEVBQUUsR0FBZSxNQUFNLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQzt3QkFDakUsTUFBTSxHQUFHLFFBQVEsQ0FBQyxTQUFTLENBQUMsRUFBRSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsTUFBTSxDQUFDLENBQUM7cUJBQy9DO2lCQUNGO2dCQUNELE9BQU8sRUFBRSxFQUFFO29CQUNULE1BQU0sSUFBSSxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLENBQUM7aUJBQzVDO2FBQ0Y7U0FDRjtRQUNELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUM7SUFPTyxNQUFNLENBQUMsV0FBVyxDQUFDLElBQWdCO1FBRXpDLElBQUksSUFBSSxLQUFLLElBQUk7WUFDZixPQUFPLElBQUksQ0FBQztRQUVkLElBQUksSUFBSSxHQUFlLElBQUksVUFBVSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQzdFLElBQUksSUFBSSxHQUFXLENBQUMsQ0FBQztRQUNyQixJQUFJLElBQUksR0FBVyxDQUFDLENBQUM7UUFHckIsT0FBTyxJQUFJLEdBQUcsSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDN0IsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDLEdBQUcsTUFBTSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFTLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztZQUM5RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUMsR0FBRyxNQUFNLENBQUMsYUFBYSxDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBUyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxHQUFHLENBQVMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ3JILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFTLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEdBQUcsQ0FBUyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ3hILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxhQUFhLENBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDbkUsSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLENBQUM7U0FDakI7UUFFRCxJQUFJLElBQUksR0FBRyxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ3RCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBUyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7WUFDOUUsSUFBSSxJQUFJLEdBQUcsSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQzFCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFTLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEdBQUcsRUFBRSxDQUFDLEdBQUcsQ0FBUyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUM7Z0JBQ3JILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxhQUFhLENBQVMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7YUFDdkU7aUJBQ0k7Z0JBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDLEdBQUcsTUFBTSxDQUFDLGFBQWEsQ0FBUyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO2FBQ25FO1NBQ0Y7UUFHRCxPQUFPLElBQUksR0FBRyxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ3pCLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUM7WUFDaEIsSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLENBQUM7U0FDakI7UUFFRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFJRCxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQVcsRUFBRSxRQUFtQjtRQUM1QyxJQUFJLFNBQVMsQ0FBQyxNQUFNLEtBQUssQ0FBQztZQUN4QixRQUFRLEdBQUcsSUFBSSxDQUFDO1FBRWxCLE9BQU8sTUFBTSxDQUFDLFFBQVEsQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDeEMsQ0FBQztJQVFPLE1BQU0sQ0FBQyxRQUFRLENBQUMsR0FBVyxFQUFFLFFBQWtCO1FBRXJELElBQUksTUFBYyxDQUFDO1FBQ25CLElBQUksR0FBRyxLQUFLLElBQUksRUFBRTtZQUNoQixNQUFNLEdBQUcsSUFBSSxDQUFDO1NBQ2Y7YUFDSTtZQUNILElBQUksR0FBRyxLQUFLLEVBQUUsRUFBRTtnQkFDZCxNQUFNLEdBQUcsR0FBRyxDQUFDO2FBQ2Q7aUJBQ0k7Z0JBQ0gsSUFBSTtvQkFDRixJQUFJLFFBQVEsS0FBSyxJQUFJLElBQUksUUFBUSxLQUFLLFNBQVMsRUFBQzt3QkFDOUMsTUFBTSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztxQkFDcEI7eUJBQ0k7d0JBQ0gsSUFBSSxRQUFRLEdBQXdCLG1CQUFtQixDQUFDLFVBQVUsQ0FBQzt3QkFDbkUsSUFBSSxLQUFLLEdBQWUsTUFBTSxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7d0JBQ25FLElBQUksU0FBUyxHQUFhLENBQUMsUUFBUSxLQUFLLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQzt3QkFDcEUsTUFBTSxHQUFHLFNBQVMsQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLENBQUMsRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7cUJBQ3REO2lCQUNGO2dCQUNELE9BQU8sRUFBRSxFQUFFO29CQUNULE1BQU0sSUFBSSxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLENBQUM7aUJBQzVDO2FBQ0Y7U0FDRjtRQUNELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUM7SUFPTyxNQUFNLENBQUMsV0FBVyxDQUFDLElBQWdCO1FBQ3pDLElBQUksSUFBSSxLQUFLLElBQUk7WUFDZixPQUFPLElBQUksQ0FBQztRQUVkLElBQUksSUFBSSxHQUFXLElBQUksQ0FBQyxNQUFNLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxLQUFLLEdBQUcsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDM0MsSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLENBQUM7U0FDakI7UUFDRCxJQUFJLElBQUksR0FBZSxJQUFJLFVBQVUsQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFHMUUsS0FBSyxJQUFJLEdBQUcsR0FBVyxDQUFDLEVBQUUsR0FBRyxHQUFHLElBQUksQ0FBQyxNQUFNLEVBQUUsR0FBRyxHQUFHLEdBQUcsR0FBRyxDQUFDLEVBQUU7WUFDMUQsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7U0FDN0M7UUFHRCxJQUFJLElBQUksR0FBVyxDQUFDLENBQUM7UUFDckIsSUFBSSxJQUFZLENBQUM7UUFDakIsS0FBSyxJQUFJLEdBQUcsQ0FBQyxFQUFFLElBQUksR0FBRyxJQUFJLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsRUFBRTtZQUN0RCxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDakYsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDMUYsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ3pFLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1NBQ2pCO1FBRUQsSUFBSSxJQUFJLEdBQUcsSUFBSSxDQUFDLE1BQU07WUFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRyxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ25GLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsQ0FBQyxHQUFHLElBQUksQ0FBQyxNQUFNO1lBQ2pDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFFeEYsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBR0QsTUFBTSxDQUFDLFdBQVcsQ0FBQyxHQUFXO1FBQzVCLElBQUksR0FBRyxLQUFLLElBQUk7WUFDZCxPQUFPLElBQUksQ0FBQztRQUNkLElBQUksR0FBRyxLQUFLLEVBQUU7WUFDWixPQUFPLEdBQUcsQ0FBQztRQUViLE9BQU8sT0FBTyxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDekQsQ0FBQztJQUdELE1BQU0sQ0FBQyxZQUFZLENBQUMsR0FBVztRQUM3QixJQUFJLEdBQUcsS0FBSyxJQUFJO1lBQ2QsT0FBTyxJQUFJLENBQUM7UUFLZCxJQUFJLEdBQUcsS0FBSyxFQUFFLElBQUksR0FBRyxLQUFLLEdBQUc7WUFDM0IsT0FBTyxJQUFJLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUUzQixJQUFJO1lBQ0YsSUFBSSxRQUFRLEdBQWEsbUJBQW1CLENBQUMsVUFBVSxDQUFDO1lBQ3hELE9BQU8sTUFBTSxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7U0FDbkQ7UUFDRCxPQUFPLEVBQUUsRUFBRTtZQUNULE1BQU0sSUFBSSxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLENBQUM7U0FDNUM7SUFDSCxDQUFDO0lBRU8sTUFBTSxDQUFDLGdCQUFnQjtRQUM3QixPQUFPLElBQUksVUFBVSxDQUFDO1lBQ3BCLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUU7WUFDMUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUc7WUFDeEYsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUU7WUFDM0YsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUU7U0FDbkIsQ0FBQyxDQUFDO0lBRUwsQ0FBQztJQUVPLE1BQU0sQ0FBQyxnQkFBZ0I7UUFDN0IsSUFBSSxNQUFNLEdBQWUsSUFBSSxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFN0MsS0FBSyxJQUFJLENBQUMsR0FBVyxDQUFDLEVBQUUsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFO1lBQ2xFLE1BQU0sQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ3JDO1FBRUQsT0FBTyxNQUFNLENBQUM7SUFDaEIsQ0FBQzs7QUFoT2Msb0JBQWEsR0FBZSxNQUFNLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQztBQUN0RCxvQkFBYSxHQUFlLE1BQU0sQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHtFbmNvZGluZywgQXBwbGljYXRpb25FeGNlcHRpb24sIElTT184ODU5XzFfRW5jb2Rpbmd9IGZyb20gXCJAbWFnaWMteHBhL21zY29yZWxpYlwiO1xyXG5pbXBvcnQge1N0clV0aWx9IGZyb20gXCIuL1N0clV0aWxcIjtcclxuaW1wb3J0IHtNaXNjfSBmcm9tIFwiLi9NaXNjXCI7XHJcblxyXG5leHBvcnQgY2xhc3MgQmFzZTY0e1xyXG4gIHByaXZhdGUgc3RhdGljIF9iYXNlNjRFbmNNYXA6IFVpbnQ4QXJyYXkgPSBCYXNlNjQuaW5pdGlhbGl6ZUVuY01hcCgpO1xyXG4gIHByaXZhdGUgc3RhdGljIF9iYXNlNjREZWNNYXA6IFVpbnQ4QXJyYXkgPSBCYXNlNjQuaW5pdGlhbGl6ZURlY01hcCgpO1xyXG5cclxuXHJcbiAgc3RhdGljIGVuY29kZShzdHI6IHN0cmluZyk6IHN0cmluZztcclxuICBzdGF0aWMgZW5jb2RlKHN0cjogc3RyaW5nLCBpc1VzZUVudkNoYXJzZXQ6IGJvb2xlYW4sIGVuY29kaW5nOiBFbmNvZGluZyk6IHN0cmluZztcclxuICBzdGF0aWMgZW5jb2RlKHN0cjogc3RyaW5nLCBpc1VzZUVudkNoYXJzZXQ/OiBhbnksIGVuY29kaW5nPzogRW5jb2RpbmcpOiBhbnkge1xyXG4gICAgaWYgKGFyZ3VtZW50cy5sZW5ndGggPT09IDEpXHJcbiAgICAgIHJldHVybiBCYXNlNjQuZW5jb2RlXzEoc3RyLCBmYWxzZSwgbnVsbCk7XHJcbiAgICBlbHNlXHJcbiAgICAgIHJldHVybiBCYXNlNjQuZW5jb2RlXzEoc3RyLCBpc1VzZUVudkNoYXJzZXQsIGVuY29kaW5nKTtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT4gRW5jb2RlcyBzdHJpbmcgdXNpbmcgdGhlIGJhc2U2NC1lbmNvZGluZy5cclxuICAvLy8gSWYgaXNVc2VFbnZDaGFyc2V0IGlzIHRydWUsIHVzZSB0aGUgc3BlY2lmaWMgY2hhcnNldCB3aGVuIGNvbnZlcnRpbmdcclxuICAvLy8gc3RyaW5nIHRvIGJ5dGUgYXJyYXkuIChEQkNTIHN1cHBvcnQpXHJcbiAgLy8vIDwvc3VtbWFyeT5cclxuICAvLy8gPHBhcmFtIG5hbWU9XCJzdHJcIj50aGUgc3RyaW5nIDwvcGFyYW0+XHJcbiAgLy8vIDxwYXJhbSBuYW1lPVwiaXNVc2VFbnZDaGFyc2V0XCI+IDwvcGFyYW0+XHJcbiAgLy8vIDxwYXJhbSBuYW1lPVwiZW5jb2RpbmdcIj4gRW52aXJvbm1lbnQuRW5jb2RpbmcgPC9wYXJhbT5cclxuICAvLy8gPHJldHVybnM+IHRoZSBiYXNlNjQtZW5jb2RlZCBzdHIgPC9yZXR1cm5zPlxyXG4gIHByaXZhdGUgc3RhdGljIGVuY29kZV8xKHN0cjogc3RyaW5nLCBpc1VzZUVudkNoYXJzZXQ6IGJvb2xlYW4sIGVuY29kaW5nOiBFbmNvZGluZyk6IHN0cmluZyB7XHJcbiAgICBsZXQgcmVzdWx0OiBzdHJpbmc7XHJcbiAgICBpZiAoc3RyID09PSBudWxsKSB7XHJcbiAgICAgIHJlc3VsdCA9IG51bGw7XHJcbiAgICB9XHJcbiAgICBlbHNlIHtcclxuICAgICAgaWYgKHN0ciA9PT0gXCJcIikge1xyXG4gICAgICAgIHJlc3VsdCA9IHN0cjtcclxuICAgICAgfVxyXG4gICAgICBlbHNlIHtcclxuICAgICAgICB0cnkge1xyXG4gICAgICAgICAgaWYgKGlzVXNlRW52Q2hhcnNldCA9PT0gZmFsc2Upe1xyXG4gICAgICAgICAgICByZXN1bHQgPSBidG9hKHN0cik7XHJcbiAgICAgICAgICB9XHJcbiAgICAgICAgICBlbHNlIHtcclxuICAgICAgICAgICAgbGV0IGluc3RhbmNlOiBFbmNvZGluZyA9IElTT184ODU5XzFfRW5jb2RpbmcuSVNPXzg4NTlfMTtcclxuICAgICAgICAgICAgbGV0IGVuY29kaW5nMjogRW5jb2RpbmcgPSBpc1VzZUVudkNoYXJzZXQgPyBlbmNvZGluZyA6IGluc3RhbmNlO1xyXG4gICAgICAgICAgICBsZXQgYmE6IFVpbnQ4QXJyYXkgPSBCYXNlNjQuZW5jb2RlQnl0ZXMoZW5jb2RpbmcyLkdldEJ5dGVzKHN0cikpO1xyXG4gICAgICAgICAgICByZXN1bHQgPSBpbnN0YW5jZS5HZXRTdHJpbmcoYmEsIDAsIGJhLmxlbmd0aCk7XHJcbiAgICAgICAgICB9ICBcclxuICAgICAgICB9XHJcbiAgICAgICAgY2F0Y2ggKGV4KSB7XHJcbiAgICAgICAgICB0aHJvdyBuZXcgQXBwbGljYXRpb25FeGNlcHRpb24oZXguTWVzc2FnZSk7XHJcbiAgICAgICAgfVxyXG4gICAgICB9XHJcbiAgICB9XHJcbiAgICByZXR1cm4gcmVzdWx0O1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PiBUaGlzIG1ldGhvZCBlbmNvZGVzIHRoZSBnaXZlbiBieXRlW10gdXNpbmcgdGhlIGJhc2U2NC1lbmNvZGluZ1xyXG4gIC8vLyBzcGVjaWZpZWQgaW4gUkZDLTIwNDUgKFNlY3Rpb24gNi44KS5cclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIC8vLyA8cGFyYW0gbmFtZT1cImRhdGFcIj50aGUgZGF0YSA8L3BhcmFtPlxyXG4gIC8vLyA8cmV0dXJucz4gdGhlIGJhc2U2NC1lbmNvZGVkIGRhdGEgPC9yZXR1cm5zPlxyXG4gIHByaXZhdGUgc3RhdGljIGVuY29kZUJ5dGVzKGRhdGE6IFVpbnQ4QXJyYXkpOiBVaW50OEFycmF5IHtcclxuXHJcbiAgICBpZiAoZGF0YSA9PT0gbnVsbClcclxuICAgICAgcmV0dXJuIG51bGw7XHJcblxyXG4gICAgbGV0IGRlc3Q6IFVpbnQ4QXJyYXkgPSBuZXcgVWludDhBcnJheShNYXRoLmZsb29yKChkYXRhLmxlbmd0aCArIDIpIC8gMykgKiA0KTtcclxuICAgIGxldCBzaWR4OiBudW1iZXIgPSAwO1xyXG4gICAgbGV0IGRpZHg6IG51bWJlciA9IDA7XHJcblxyXG4gICAgLy8gMy1ieXRlIHRvIDQtYnl0ZSBjb252ZXJzaW9uICsgMC02MyB0byBBU0NJSSBwcmludGFibGUgY29udmVyc2lvblxyXG4gICAgd2hpbGUgKHNpZHggPCBkYXRhLmxlbmd0aCAtIDIpIHtcclxuICAgICAgZGVzdFtkaWR4KytdID0gQmFzZTY0Ll9iYXNlNjRFbmNNYXBbTWlzYy5VUlNoaWZ0KDxudW1iZXI+ZGF0YVtzaWR4XSwgMikgJiA2M107XHJcbiAgICAgIGRlc3RbZGlkeCsrXSA9IEJhc2U2NC5fYmFzZTY0RW5jTWFwWyhNaXNjLlVSU2hpZnQoPG51bWJlcj5kYXRhW3NpZHggKyAxXSwgNCkgJiAxNSkgfCAoPG51bWJlcj5kYXRhW3NpZHhdIDw8IDQgJiA2MyldO1xyXG4gICAgICBkZXN0W2RpZHgrK10gPSBCYXNlNjQuX2Jhc2U2NEVuY01hcFsoTWlzYy5VUlNoaWZ0KDxudW1iZXI+ZGF0YVtzaWR4ICsgMl0sIDYpICYgMykgfCAoPG51bWJlcj5kYXRhW3NpZHggKyAxXSA8PCAyICYgNjMpXTtcclxuICAgICAgZGVzdFtkaWR4KytdID0gQmFzZTY0Ll9iYXNlNjRFbmNNYXBbPG51bWJlcj4oZGF0YVtzaWR4ICsgMl0gJiA2MyldO1xyXG4gICAgICBzaWR4ID0gc2lkeCArIDM7XHJcbiAgICB9XHJcblxyXG4gICAgaWYgKHNpZHggPCBkYXRhLmxlbmd0aCkge1xyXG4gICAgICBkZXN0W2RpZHgrK10gPSBCYXNlNjQuX2Jhc2U2NEVuY01hcFtNaXNjLlVSU2hpZnQoPG51bWJlcj5kYXRhW3NpZHhdLCAyKSAmIDYzXTtcclxuICAgICAgaWYgKHNpZHggPCBkYXRhLmxlbmd0aCAtIDEpIHtcclxuICAgICAgICBkZXN0W2RpZHgrK10gPSBCYXNlNjQuX2Jhc2U2NEVuY01hcFsoTWlzYy5VUlNoaWZ0KDxudW1iZXI+ZGF0YVtzaWR4ICsgMV0sIDQpICYgMTUpIHwgKDxudW1iZXI+ZGF0YVtzaWR4XSA8PCA0ICYgNjMpXTtcclxuICAgICAgICBkZXN0W2RpZHgrK10gPSBCYXNlNjQuX2Jhc2U2NEVuY01hcFs8bnVtYmVyPmRhdGFbc2lkeCArIDFdIDw8IDIgJiA2M107XHJcbiAgICAgIH1cclxuICAgICAgZWxzZSB7XHJcbiAgICAgICAgZGVzdFtkaWR4KytdID0gQmFzZTY0Ll9iYXNlNjRFbmNNYXBbPG51bWJlcj5kYXRhW3NpZHhdIDw8IDQgJiA2M107XHJcbiAgICAgIH1cclxuICAgIH1cclxuXHJcbiAgICAvLyBhZGQgcGFkZGluZ1xyXG4gICAgd2hpbGUgKGRpZHggPCBkZXN0Lmxlbmd0aCkge1xyXG4gICAgICBkZXN0W2RpZHhdID0gNjE7XHJcbiAgICAgIGRpZHggPSBkaWR4ICsgMTtcclxuICAgIH1cclxuXHJcbiAgICByZXR1cm4gZGVzdDtcclxuICB9XHJcblxyXG4gIHN0YXRpYyBkZWNvZGUoc3RyOiBzdHJpbmcpOiBzdHJpbmc7XHJcbiAgc3RhdGljIGRlY29kZShzdHI6IHN0cmluZywgZW5jb2Rpbmc6IEVuY29kaW5nKTogc3RyaW5nO1xyXG4gIHN0YXRpYyBkZWNvZGUoc3RyOiBzdHJpbmcsIGVuY29kaW5nPzogRW5jb2RpbmcpOiBhbnkge1xyXG4gICAgaWYgKGFyZ3VtZW50cy5sZW5ndGggPT09IDEpXHJcbiAgICAgIGVuY29kaW5nID0gbnVsbDtcclxuXHJcbiAgICByZXR1cm4gQmFzZTY0LmRlY29kZV8xKHN0ciwgZW5jb2RpbmcpO1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PiBUaGlzIG1ldGhvZCBkZWNvZGVzIHRoZSBnaXZlbiBzdHJpbmcgdXNpbmcgdGhlIGJhc2U2NC1lbmNvZGluZ1xyXG4gIC8vLyBzcGVjaWZpZWQgaW4gUkZDLTIwNDUgKFNlY3Rpb24gNi44KS5cclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIC8vLyA8cGFyYW0gbmFtZT1cInN0clwiPnRoZSBiYXNlNjQtZW5jb2RlZCBzdHJpbmcuIDwvcGFyYW0+XHJcbiAgLy8vIDxwYXJhbSBuYW1lPVwiZW5jb2RpbmdcIj5FbnZpcm9ubWVudC5FbmNvZGluZyBvciBudWxsLjwvcGFyYW0+XHJcbiAgLy8vIDxyZXR1cm5zPiB0aGUgZGVjb2RlZCBzdHIuPC9yZXR1cm5zPlxyXG4gIHByaXZhdGUgc3RhdGljIGRlY29kZV8xKHN0cjogc3RyaW5nLCBlbmNvZGluZzogRW5jb2RpbmcpOiBzdHJpbmcge1xyXG5cclxuICAgIGxldCByZXN1bHQ6IHN0cmluZztcclxuICAgIGlmIChzdHIgPT09IG51bGwpIHtcclxuICAgICAgcmVzdWx0ID0gbnVsbDtcclxuICAgIH1cclxuICAgIGVsc2Uge1xyXG4gICAgICBpZiAoc3RyID09PSBcIlwiKSB7XHJcbiAgICAgICAgcmVzdWx0ID0gc3RyO1xyXG4gICAgICB9XHJcbiAgICAgIGVsc2Uge1xyXG4gICAgICAgIHRyeSB7XHJcbiAgICAgICAgICBpZiAoZW5jb2RpbmcgPT09IG51bGwgfHwgZW5jb2RpbmcgPT09IHVuZGVmaW5lZCl7XHJcbiAgICAgICAgICAgIHJlc3VsdCA9IGF0b2Ioc3RyKTtcclxuICAgICAgICAgIH1cclxuICAgICAgICAgIGVsc2Uge1xyXG4gICAgICAgICAgICBsZXQgaW5zdGFuY2U6IElTT184ODU5XzFfRW5jb2RpbmcgPSBJU09fODg1OV8xX0VuY29kaW5nLklTT184ODU5XzE7XHJcbiAgICAgICAgICAgIGxldCBhcnJheTogVWludDhBcnJheSA9IEJhc2U2NC5kZWNvZGVCeXRlcyhpbnN0YW5jZS5HZXRCeXRlcyhzdHIpKTtcclxuICAgICAgICAgICAgbGV0IGVuY29kaW5nMjogRW5jb2RpbmcgPSAoZW5jb2RpbmcgIT09IG51bGwpID8gZW5jb2RpbmcgOiBpbnN0YW5jZTtcclxuICAgICAgICAgICAgcmVzdWx0ID0gZW5jb2RpbmcyLkdldFN0cmluZyhhcnJheSwgMCwgYXJyYXkubGVuZ3RoKTtcclxuICAgICAgICAgIH0gICBcclxuICAgICAgICB9XHJcbiAgICAgICAgY2F0Y2ggKGV4KSB7XHJcbiAgICAgICAgICB0aHJvdyBuZXcgQXBwbGljYXRpb25FeGNlcHRpb24oZXguTWVzc2FnZSk7XHJcbiAgICAgICAgfVxyXG4gICAgICB9XHJcbiAgICB9XHJcbiAgICByZXR1cm4gcmVzdWx0O1xyXG4gIH1cclxuXHJcbiAgLy8vIDxzdW1tYXJ5PiBUaGlzIG1ldGhvZCBkZWNvZGVzIHRoZSBnaXZlbiBieXRlW10gdXNpbmcgdGhlIGJhc2U2NC1lbmNvZGluZ1xyXG4gIC8vLyBzcGVjaWZpZWQgaW4gUkZDLTIwNDUgKFNlY3Rpb24gNi44KS5cclxuICAvLy8gPC9zdW1tYXJ5PlxyXG4gIC8vLyA8cGFyYW0gbmFtZT1cImRhdGFcIj50aGUgYmFzZTY0LWVuY29kZWQgZGF0YS48L3BhcmFtPlxyXG4gIC8vLyA8cmV0dXJucz4gdGhlIGRlY29kZWQgPHZhcj5kYXRhPC92YVxyXG4gIHByaXZhdGUgc3RhdGljIGRlY29kZUJ5dGVzKGRhdGE6IFVpbnQ4QXJyYXkpOiBVaW50OEFycmF5IHtcclxuICAgIGlmIChkYXRhID09PSBudWxsKVxyXG4gICAgICByZXR1cm4gbnVsbDtcclxuXHJcbiAgICBsZXQgdGFpbDogbnVtYmVyID0gZGF0YS5sZW5ndGg7XHJcbiAgICB3aGlsZSAoZGF0YVt0YWlsIC0gMV0gPT09ICc9Jy5jaGFyQ29kZUF0KDApKSB7XHJcbiAgICAgIHRhaWwgPSB0YWlsIC0gMTtcclxuICAgIH1cclxuICAgIGxldCBkZXN0OiBVaW50OEFycmF5ID0gbmV3IFVpbnQ4QXJyYXkodGFpbCAtIE1hdGguZmxvb3IoZGF0YS5sZW5ndGggLyA0KSk7XHJcblxyXG4gICAgLy8gQVNDSUkgcHJpbnRhYmxlIHRvIDAtNjMgY29udmVyc2lvblxyXG4gICAgZm9yIChsZXQgaWR4OiBudW1iZXIgPSAwOyBpZHggPCBkYXRhLmxlbmd0aDsgaWR4ID0gaWR4ICsgMSkge1xyXG4gICAgICBkYXRhW2lkeF0gPSBCYXNlNjQuX2Jhc2U2NERlY01hcFtkYXRhW2lkeF1dO1xyXG4gICAgfVxyXG5cclxuICAgIC8vIDQtYnl0ZSB0byAzLWJ5dGUgY29udmVyc2lvblxyXG4gICAgbGV0IHNpZHg6IG51bWJlciA9IDA7XHJcbiAgICBsZXQgZGlkeDogbnVtYmVyO1xyXG4gICAgZm9yIChkaWR4ID0gMDsgZGlkeCA8IGRlc3QubGVuZ3RoIC0gMjsgZGlkeCA9IGRpZHggKyAzKSB7XHJcbiAgICAgIGRlc3RbZGlkeF0gPSAoKChkYXRhW3NpZHhdIDw8IDIpICYgMjU1KSB8IChNaXNjLlVSU2hpZnQoZGF0YVtzaWR4ICsgMV0sIDQpICYgMykpO1xyXG4gICAgICBkZXN0W2RpZHggKyAxXSA9ICgoKGRhdGFbc2lkeCArIDFdIDw8IDQpICYgMjU1KSB8IChNaXNjLlVSU2hpZnQoZGF0YVtzaWR4ICsgMl0sIDIpICYgMTUpKTtcclxuICAgICAgZGVzdFtkaWR4ICsgMl0gPSAoKChkYXRhW3NpZHggKyAyXSA8PCA2KSAmIDI1NSkgfCAoZGF0YVtzaWR4ICsgM10gJiA2MykpO1xyXG4gICAgICBzaWR4ID0gc2lkeCArIDQ7XHJcbiAgICB9XHJcblxyXG4gICAgaWYgKGRpZHggPCBkZXN0Lmxlbmd0aClcclxuICAgICAgZGVzdFtkaWR4XSA9ICgoKGRhdGFbc2lkeF0gPDwgMikgJiAyNTUpIHwgKE1pc2MuVVJTaGlmdChkYXRhW3NpZHggKyAxXSwgNCkgJiAzKSk7XHJcbiAgICBpZiAoKGRpZHggPSBkaWR4ICsgMSkgPCBkZXN0Lmxlbmd0aClcclxuICAgICAgZGVzdFtkaWR4XSA9ICgoKGRhdGFbc2lkeCArIDFdIDw8IDQpICYgMjU1KSB8IChNaXNjLlVSU2hpZnQoZGF0YVtzaWR4ICsgMl0sIDIpICYgMTUpKTtcclxuXHJcbiAgICByZXR1cm4gZGVzdDtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT4gZGVjb2RlZCBhbmQgcmV0dXJuIGFuIGhleCByZXByZXNlbnRhdGlvbiBvZiB0aGUgZGF0YTwvc3VtbWFyeT5cclxuICBzdGF0aWMgZGVjb2RlVG9IZXgoc3RyOiBzdHJpbmcpOiBzdHJpbmcge1xyXG4gICAgaWYgKHN0ciA9PT0gbnVsbClcclxuICAgICAgcmV0dXJuIG51bGw7XHJcbiAgICBpZiAoc3RyID09PSBcIlwiKVxyXG4gICAgICByZXR1cm4gc3RyO1xyXG5cclxuICAgIHJldHVybiBTdHJVdGlsLnN0cmluZ1RvSGV4YUR1bXAoQmFzZTY0LmRlY29kZShzdHIpLCAyKTtcclxuICB9XHJcblxyXG4gIC8vLyA8c3VtbWFyeT4gZGVjb2RlcyBhIHN0cmluZyB0byBieXRlIGFycmF5PC9zdW1tYXJ5PlxyXG4gIHN0YXRpYyBkZWNvZGVUb0J5dGUoc3RyOiBzdHJpbmcpOiBVaW50OEFycmF5IHtcclxuICAgIGlmIChzdHIgPT09IG51bGwpXHJcbiAgICAgIHJldHVybiBudWxsO1xyXG5cclxuICAgIC8vIFFDUiA3NDA5MTggaWYgd2UgaGF2ZSBhbmQgZW1wdHkgZXhwcmVzc2lvbiBpdCBpcyBzZW50IGZyb20gdGhlIHNlcnZlciBhcyBlbXB0eSBzdHJpbmdcclxuICAgIC8vIGFuZCBjaGFuZ2VkIGxvY2FsbHkgdG8gYSBzdHJpbmcgd2l0aCBvbmUgYmxhbmsgZWl0aGVyIHdheSB0aGV5IGFyZSBub3QgdmFsaWQgYmFzZTY0IGVuY29kZWRcclxuICAgIC8vIHN0cmluZyBhbmQgc2hvdWxkIG5vdCBiZSBkZWNvZGVkLlxyXG4gICAgaWYgKHN0ciA9PT0gXCJcIiB8fCBzdHIgPT09IFwiIFwiKVxyXG4gICAgICByZXR1cm4gbmV3IFVpbnQ4QXJyYXkoMCk7XHJcblxyXG4gICAgdHJ5IHtcclxuICAgICAgbGV0IGluc3RhbmNlOiBFbmNvZGluZyA9IElTT184ODU5XzFfRW5jb2RpbmcuSVNPXzg4NTlfMTtcclxuICAgICAgcmV0dXJuIEJhc2U2NC5kZWNvZGVCeXRlcyhpbnN0YW5jZS5HZXRCeXRlcyhzdHIpKTtcclxuICAgIH1cclxuICAgIGNhdGNoIChleCkge1xyXG4gICAgICB0aHJvdyBuZXcgQXBwbGljYXRpb25FeGNlcHRpb24oZXguTWVzc2FnZSk7XHJcbiAgICB9XHJcbiAgfVxyXG5cclxuICBwcml2YXRlIHN0YXRpYyBpbml0aWFsaXplRW5jTWFwKCkge1xyXG4gICAgcmV0dXJuIG5ldyBVaW50OEFycmF5KFtcclxuICAgICAgNjUsIDY2LCA2NywgNjgsIDY5LCA3MCwgNzEsIDcyLCA3MywgNzQsIDc1LCA3NiwgNzcsIDc4LCA3OSwgODAsIDgxLCA4MiwgODMsXHJcbiAgICAgIDg0LCA4NSwgODYsIDg3LCA4OCwgODksIDkwLCA5NywgOTgsIDk5LCAxMDAsIDEwMSwgMTAyLCAxMDMsIDEwNCwgMTA1LCAxMDYsIDEwNywgMTA4LCAxMDksXHJcbiAgICAgIDExMCwgMTExLCAxMTIsIDExMywgMTE0LCAxMTUsIDExNiwgMTE3LCAxMTgsIDExOSwgMTIwLCAxMjEsIDEyMiwgNDgsIDQ5LCA1MCwgNTEsIDUyLCA1MywgNTQsXHJcbiAgICAgIDU1LCA1NiwgNTcsIDQzLCA0N1xyXG4gICAgXSk7XHJcblxyXG4gIH1cclxuXHJcbiAgcHJpdmF0ZSBzdGF0aWMgaW5pdGlhbGl6ZURlY01hcCgpIHtcclxuICAgIGxldCBkZWNNYXA6IFVpbnQ4QXJyYXkgPSBuZXcgVWludDhBcnJheSgxMjgpO1xyXG5cclxuICAgIGZvciAobGV0IGk6IG51bWJlciA9IDA7IGkgPCBCYXNlNjQuX2Jhc2U2NEVuY01hcC5sZW5ndGg7IGkgPSBpICsgMSkge1xyXG4gICAgICBkZWNNYXBbQmFzZTY0Ll9iYXNlNjRFbmNNYXBbaV1dID0gaTtcclxuICAgIH1cclxuXHJcbiAgICByZXR1cm4gZGVjTWFwO1xyXG4gIH1cclxufVxyXG4iXX0=