import { Logger_LogLevels, Logger_MessageDirection } from "./enums";
import { DateTime, Debug, Exception, NString, StringBuilder, Thread, isNullOrUndefined } from "@magic-xpa/mscorelib";
import { OSEnvironment } from "./PlatformUtils";
import { DateTimeUtils } from "./DateTimeUtils";
import { XMLConstants } from "./XMLConstants";
import { RequestInfo } from "./RequestInfo";
import { StrUtil } from "./StrUtil";
export var LogType;
(function (LogType) {
    LogType[LogType["info"] = 1] = "info";
    LogType[LogType["warning"] = 2] = "warning";
    LogType[LogType["error"] = 3] = "error";
})(LogType || (LogType = {}));
export class Logger {
    static set Instance(value) {
        Logger.instance = value;
    }
    static get Instance() {
        if (Logger.instance === null) {
            Logger.instance = new Logger();
        }
        return Logger.instance;
    }
    Initialize(logLevel, shouldBeep) {
        try {
            this.LogLevel = logLevel;
            this.ShouldBeep = shouldBeep;
        }
        catch (e) {
            this.WriteDevToLog("ClientManager.init(): " + e.Message);
        }
    }
    ShouldLog(logLevel) {
        if (arguments.length === 1)
            return this.ShouldLog_0(logLevel);
        else
            return this.ShouldLog_1();
    }
    ShouldLog_0(logLevel) {
        return this.LogLevel === logLevel;
    }
    ShouldLog_1() {
        return this.LogLevel > Logger_LogLevels.None;
    }
    ShouldLogServerRelatedMessages() {
        return (this.ShouldLogExtendedServerRelatedMessages() || Logger.Instance.ShouldLog(Logger_LogLevels.Server)) && this.LogLevel !== Logger_LogLevels.Basic;
    }
    ShouldLogExtendedServerRelatedMessages() {
        return (Logger.Instance.ShouldLog(Logger_LogLevels.ServerMessages) || Logger.Instance.ShouldLog(Logger_LogLevels.Support) || Logger.Instance.ShouldLog(Logger_LogLevels.Development)) && this.LogLevel !== Logger_LogLevels.Basic;
    }
    WriteToLog(msg, openIfNecessary, logType = LogType.info) {
        if (this.LogLevel > Logger_LogLevels.None || openIfNecessary) {
            msg = NString.Format("{0} {1}", (this.LogLevel === Logger_LogLevels.Basic) ? new Date().toISOString() : DateTimeUtils.ToString(DateTime.Now, XMLConstants.ERROR_LOG_TIME_FORMAT, this), msg);
            switch (logType) {
                case LogType.error:
                    console.error(msg);
                    break;
                case LogType.warning:
                    console.warn(msg);
                    break;
                default:
                    console.log(msg);
            }
        }
    }
    WriteServerToLog(msg) {
        if (this.ShouldLogServerRelatedMessages()) {
            this.WriteToLog(NString.Format("Server, Thread={0}: ", Thread.CurrentThread.ManagedThreadId) + msg, false, LogType.info);
        }
    }
    WriteServerMessagesToLog(msg) {
        if (this.ShouldLogExtendedServerRelatedMessages()) {
            this.WriteToLog("Server#: " + msg, false, LogType.info);
        }
    }
    WriteSupportToLog(msg, skipLine) {
        if (this.LogLevel >= Logger_LogLevels.Support && this.LogLevel !== Logger_LogLevels.Basic) {
            if (skipLine) {
                this.WriteToLog("SUPPORT: " + msg, false, LogType.info);
            }
            else {
                this.WriteToLog("SUPPORT: " + msg + OSEnvironment.EolSeq + "-----------------------------------------------------------------------------------------------------------", false, LogType.info);
            }
        }
    }
    WriteGuiToLog(msg) {
        if (this.LogLevel >= Logger_LogLevels.Gui && this.LogLevel !== Logger_LogLevels.Basic) {
            this.WriteToLog(msg, false, LogType.info);
        }
    }
    WriteDevToLog(msg) {
        if (this.LogLevel >= Logger_LogLevels.Development && this.LogLevel !== Logger_LogLevels.Basic) {
            this.WriteToLog("DEV: " + msg, false, LogType.info);
        }
    }
    WriteBasicToLog(messageDirection, contextID, sessionCounter, clientID, serverID, responseTime, statusCode, httpHeaders, contentLength) {
        if (this.LogLevel === Logger_LogLevels.Basic) {
            let text = httpHeaders;
            text = text.trim();
            text = NString.Replace(text, "\r\n", "|");
            let arg_E4_0 = "RIA,{0}_{1},{2},{3},{4},{5},-,{6},{7},{8},{9},{10},{11}";
            let expr_3E = new Array(12);
            expr_3E[1] = Thread.CurrentThread.ManagedThreadId;
            expr_3E[2] = new Date().toISOString();
            expr_3E[3] = ((messageDirection === Logger_MessageDirection.MessageLeaving) ? "MSGL" : "MSGE");
            expr_3E[4] = contextID;
            expr_3E[5] = sessionCounter;
            expr_3E[6] = clientID;
            expr_3E[7] = serverID;
            expr_3E[8] = ((responseTime !== 0) ? responseTime.toString() : "-");
            let arg_D3_1 = 9;
            let arg_D3_2;
            arg_D3_2 = statusCode;
            expr_3E[arg_D3_1] = arg_D3_2;
            expr_3E[10] = text;
            expr_3E[11] = contentLength;
            let value = NString.Format(arg_E4_0, expr_3E);
            console.log(value);
        }
    }
    WriteRequestInfoToLog(requestInfo, extraMessageStr) {
        if (this.LogLevel === Logger_LogLevels.RequestInfo) {
            console.log(extraMessageStr, RequestInfo.clone(requestInfo));
        }
    }
    WriteBasicErrorToLog() {
        Debug.Assert(this.LogLevel === Logger_LogLevels.Basic);
    }
    WriteErrorToLog(msg) {
        this.WriteToLog("MagicWeb [ERROR]: " + StrUtil.getConsoleErorString(msg), true, LogType.error);
    }
    WriteExceptionToLog(msgOrEx, msg) {
        if (msgOrEx instanceof Error) {
            this.WriteExceptionToLog_2(msgOrEx, msg);
            return;
        }
        if (arguments.length === 1 && (msgOrEx !== null || msgOrEx instanceof Exception)) {
            this.WriteExceptionToLog_1(msgOrEx);
            return;
        }
        this.WriteExceptionToLog_3(msgOrEx, msg);
    }
    WriteExceptionToLogWithMsg(msg) {
        this.WriteToLog("MagicWeb [ERROR]: " + StrUtil.getConsoleErorString(msg), true, LogType.error);
    }
    WriteExceptionToLog_1(ex) {
        this.WriteExceptionToLogWithMsg(NString.Format("{0} : {1}{2}{3}{4}", [
            ex.GetType(), OSEnvironment.EolSeq, ex.StackTrace, OSEnvironment.EolSeq, ex.Message
        ]));
    }
    WriteExceptionToLog_2(ex, message) {
        if (isNullOrUndefined(message))
            this.WriteExceptionToLogWithMsg(NString.Format("{0}{1}{2}", [ex.stack, OSEnvironment.EolSeq, ex.message]));
        else
            this.WriteExceptionToLogWithMsg(NString.Format("{0}{1}{2}{4}{5}", [message, OSEnvironment.EolSeq, ex.stack, OSEnvironment.EolSeq, ex.message]));
    }
    WriteExceptionToLog_3(ex, msg) {
        this.WriteExceptionToLogWithMsg(NString.Format("{0}, {1} : {2}{3}{4}{5}", [
            ex.GetType(), msg, OSEnvironment.EolSeq, ex.StackTrace, OSEnvironment.EolSeq, ex.Message
        ]));
    }
    WriteWarningToLog(msgOrEx, msg) {
        if (arguments.length === 1 && msgOrEx !== null) {
            if (msgOrEx instanceof Exception)
                this.WriteWarningToLog_1(msgOrEx);
            else if (msgOrEx instanceof Error)
                this.WriteWarningToLog_2(msgOrEx);
        }
        else
            this.WriteWarningToLog_3(msgOrEx, msg);
    }
    WriteWarningToLogWithMsg(msg) {
        if (this.LogLevel !== Logger_LogLevels.Basic) {
            this.WriteToLog("MagicWeb [WARNING]: " + msg, true, LogType.warning);
        }
    }
    WriteWarningToLog_1(ex) {
        this.WriteWarningToLogWithMsg(ex.GetType() + " : " + OSEnvironment.EolSeq + ex.StackTrace + OSEnvironment.EolSeq + ex.Message);
    }
    WriteWarningToLog_2(ex) {
        this.WriteWarningToLogWithMsg(NString.Format("{0}{1}{2}", [
            ex.stack, OSEnvironment.EolSeq, ex.message
        ]));
    }
    WriteWarningToLog_3(ex, msg) {
        this.WriteWarningToLogWithMsg(NString.Format("{0}, {1} : {2}{3}{4}{5}", [
            ex.GetType(), msg, OSEnvironment.EolSeq, ex.StackTrace, OSEnvironment.EolSeq, ex.Message
        ]));
    }
    WriteStackTrace(stackTrace, framesToPrint, traceTitle) {
        if (traceTitle === null) {
            traceTitle = "Stack trace:";
        }
        let stringBuilder = new StringBuilder(traceTitle + OSEnvironment.EolSeq);
        let frames = stackTrace.GetFrames();
        let array = frames;
        for (let i = 0; i < array.length; i = i + 1) {
            let stackFrame = array[i];
            framesToPrint = framesToPrint - 1;
            stringBuilder.Append(stackFrame.toString());
            if (framesToPrint === 0) {
                stringBuilder.Append("\t... more stack frames ...\n");
                break;
            }
        }
        this.WriteToLog(stringBuilder.ToString(), true);
    }
    Flush() {
    }
    constructor() {
        this.LogLevel = 0;
        this.ShouldBeep = false;
    }
}
Logger.instance = null;
//# sourceMappingURL=data:application/json;base64,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