import { NString, List, ApplicationException, StringBuilder, NChar, NNumber } from "@magic-xpa/mscorelib";
import { XMLConstants } from "../XMLConstants";
export class XmlParser {
    constructor(data = NString.Empty) {
        this._currIndex = 0;
        this._xmLdata = "";
        this._history = new List();
        this.setXMLdata(data);
        this.setCurrIndex(0);
    }
    static getTokens(str, delimiter, isMagicXML = true) {
        let tokensVec = new List();
        let token = null;
        if (isMagicXML) {
            str = str.trim();
        }
        let strTok = str.split(delimiter.charAt(0));
        for (let i = 0; i < strTok.length; i = i + 1) {
            if (isMagicXML && i === strTok.length - 1 && strTok.length % 2 === 1) {
                break;
            }
            token = strTok[i];
            if (isMagicXML) {
                if (i % 2 === 0) {
                    token = token.trim();
                    if (token.endsWith("=")) {
                        token = token.substr(0, token.length - 1);
                    }
                }
                else if (token === "")
                    token = " ";
            }
            if (token === null)
                throw new ApplicationException("in ClientManager.Instance.XMLParser.getTokens() null token value");
            tokensVec.push(token);
        }
        return tokensVec;
    }
    static unescape(str) {
        let unescapedString = new StringBuilder(str.length);
        for (let i = 0; i < str.length; i++) {
            if (str[i] !== '\\') {
                unescapedString.Append(str[i]);
                continue;
            }
            switch (str[++i]) {
                case 'q':
                    unescapedString.Append('\"');
                    break;
                case 'o':
                    unescapedString.Append('\'');
                    break;
                case 'l':
                    unescapedString.Append('<');
                    break;
                case 'g':
                    unescapedString.Append('>');
                    break;
                case 'e':
                    unescapedString.Append('=');
                    break;
                case 'r':
                    unescapedString.Append('\r');
                    break;
                case 'n':
                    unescapedString.Append('\n');
                    break;
                default:
                    unescapedString.Append(str[i]);
                    break;
            }
        }
        return (unescapedString.ToString());
    }
    static escape(str) {
        let escapedString = new StringBuilder(str.length * 2);
        for (let i = 0; i < str.length; i++) {
            switch (str[i]) {
                case '\\':
                    escapedString.Append("\\\\");
                    break;
                case '"':
                    escapedString.Append("\\q");
                    break;
                case '\'':
                    escapedString.Append("\\o");
                    break;
                case '<':
                    escapedString.Append("\\l");
                    break;
                case '>':
                    escapedString.Append("\\g");
                    break;
                case '=':
                    escapedString.Append("\\e");
                    break;
                case '\r':
                    escapedString.Append("\r");
                    break;
                case '\n':
                    escapedString.Append("\n");
                    break;
                default:
                    escapedString.Append(str[i]);
                    break;
            }
        }
        return (escapedString.ToString());
    }
    static escapeUrl(str) {
        return NString.Replace(str, "&", "&amp;");
    }
    getNextTag() {
        if (this._xmLdata.length - this._currIndex <= 1) {
            return null;
        }
        for (let tmpIndx = this._currIndex + 1; tmpIndx < this._xmLdata.length; tmpIndx++) {
            let tmpChar = this._xmLdata[tmpIndx];
            if (NChar.IsLetter(tmpChar) || tmpChar === '/') {
                let endOfTag = NString.IndexOfAny(this._xmLdata, XmlParser.endOfNameChar, tmpIndx, this._xmLdata.length - tmpIndx);
                if (endOfTag === -1)
                    return null;
                else
                    return this._xmLdata.substr(tmpIndx, endOfTag - tmpIndx);
            }
        }
        return null;
    }
    getXMLsubstring(endContext) {
        return this._xmLdata.substr(this._currIndex, endContext - this._currIndex);
    }
    GetCurrentElementValue() {
        this.setCurrIndex2EndOfTag();
        let endContext = this.getXMLdata().indexOf(XMLConstants.TAG_OPEN, this.getCurrIndex());
        let value = this.getXMLsubstring(endContext);
        this.setCurrIndex2EndOfTag();
        return value;
    }
    setCurrIndex2EndOfTag() {
        this._currIndex = this._xmLdata.indexOf(XMLConstants.TAG_CLOSE, this._currIndex) + 1;
    }
    static getInt(valueStr) {
        return NNumber.Parse(valueStr.trim());
    }
    static getBoolean(valueStr) {
        return valueStr[0] === '1';
    }
    getCurrIndex() {
        return this._currIndex;
    }
    getXMLdata() {
        return this._xmLdata;
    }
    add2CurrIndex(add) {
        this._currIndex = this._currIndex + add;
    }
    setCurrIndex(index) {
        this._currIndex = index;
    }
    setXMLdata(data) {
        if (data !== null)
            this._xmLdata = data.trim();
        else {
            this._xmLdata = null;
            this.setCurrIndex(0);
        }
    }
    PrepareFormReadString(newXmlString) {
        this.setXMLdata(newXmlString);
        this.setCurrIndex(0);
    }
    push() {
        this._history.push(this._currIndex);
        this._history.push(this._xmLdata);
    }
    pop() {
        let count = this._history.length;
        this._xmLdata = this._history.get_Item(count - 1);
        this._currIndex = this._history.get_Item(count - 2);
        this._history.SetSize(count - 2);
    }
    loadTableCacheData(data) {
        this.setXMLdata(data);
        this.setCurrIndex(0);
    }
    ReadToEndOfCurrentElement() {
        let currentTag = this.getNextTag();
        let currentTagIndex = this._xmLdata.indexOf(XMLConstants.TAG_OPEN + currentTag, this.getCurrIndex());
        let nextOpenTagIndex = this._xmLdata.indexOf(XMLConstants.TAG_OPEN, currentTagIndex + 1);
        if (nextOpenTagIndex === -1)
            nextOpenTagIndex = this._xmLdata.length;
        let elementEndIndex = NString.IndexOf(this._xmLdata, XMLConstants.TAG_TERM, this.getCurrIndex(), nextOpenTagIndex - this.getCurrIndex());
        if (elementEndIndex === -1)
            elementEndIndex = this._xmLdata.indexOf("/" + currentTag, this.getCurrIndex()) + currentTag.length + XMLConstants.TAG_TERM.length;
        else
            elementEndIndex = elementEndIndex + XMLConstants.TAG_TERM.length;
        let elementBlock = this.getXMLsubstring(elementEndIndex);
        this.setCurrIndex(elementEndIndex);
        return elementBlock;
    }
    ReadContentOfCurrentElement() {
        let currentTag = this.getNextTag();
        let elementEndIndex = this._xmLdata.indexOf("</" + currentTag + ">", this.getCurrIndex());
        if (elementEndIndex === -1)
            return NString.Empty;
        this.setCurrIndex2EndOfTag();
        let elementBlock = this.getXMLsubstring(elementEndIndex);
        this.setCurrIndex(elementEndIndex);
        this.setCurrIndex2EndOfTag();
        return elementBlock;
    }
    toString(headCharCount, tailCharCount) {
        if (arguments.length === 0) {
            return this.ToString_0();
        }
        return this.ToString_1(headCharCount, tailCharCount);
    }
    ToString_0() {
        return this.toString(20, 50);
    }
    ToString_1(headCharCount, tailCharCount) {
        let markerPosition = Math.min(this._currIndex, this._xmLdata.length);
        let segmentStartIndex = Math.max(0, markerPosition - headCharCount);
        let segmentEndIndex = Math.min(this._xmLdata.length, markerPosition + tailCharCount);
        let headLength = markerPosition - segmentStartIndex;
        let tailLength = segmentEndIndex - markerPosition;
        let segment = new StringBuilder();
        if (segmentStartIndex > 0)
            segment.Append("...");
        if (headLength > 0)
            segment.Append(this._xmLdata, segmentStartIndex, headLength);
        segment.Append("|-{").Append(this._currIndex).Append("}-|");
        if (tailLength > 0)
            segment.Append(this._xmLdata, this._currIndex, tailLength);
        if (segmentEndIndex < this._xmLdata.length)
            segment.Append("...");
        return segment.ToString();
    }
    SkipXMLElement() {
        let endContext = this.getXMLdata().indexOf(XMLConstants.TAG_TERM, this.getCurrIndex());
        if (endContext !== -1 && endContext < this.getXMLdata().length) {
            this.setCurrIndex2EndOfTag();
        }
    }
}
XmlParser.endOfNameChar = [' ', '>'];
//# sourceMappingURL=data:application/json;base64,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